#include <kdisplay.h>
#include <kdisplay.moc>

#include <stdio.h>
#include <math.h>

#include <qlayout.h>
#include <qpainter.h>
#include <qcursor.h>
#include <qpixmap.h>
#include <qevent.h>
#include <qpointarray.h>
#include <qpoint.h>

#include <xpm/toolbox.xpm>
#include <xpm/maxmarker.xpm>
#include <xpm/redmarker.xpm>
#include <xpm/yellowmarker.xpm>

/* constants */
#define	BLACK	QColor(0,0,0)
#define	GREEN	QColor(0,255,0)
#define	LTGRAY	QColor(128,128,128)

const int idxL = 70;
const int idxR = 130;
const int idyT = 30;
const int idyB = 30;

QPoint ptMaxMarker(0, 100);
QPoint ptMarker1(0, 140);
QPoint ptMarker2(0, 180);
QPoint ptDiffMarker(0, 220);

QRect rMaxMarker(0, 120, 120, 20);
QRect rMarker1(0, 160, 120, 20);
QRect rMarker2(0, 200, 120, 20);
QRect rDiffMarker(0, 240, 120, 20);

/****************************************************************/
/*																*/
/*							KScale								*/
/*																*/
/****************************************************************/

KScale::KScale(QWidget *parent, const char *name)
	: QFrame(parent, name)
{
	strcpy(chXFmt, "%3.f");
	strcpy(chYFmt, "%3.f");
	strcpy(chFmt, "%3.f %3.f");
	iStep = 100;
	fXScale = fYScale = 1.0f;
	bAdaptiveZeroLine = TRUE;
	strcpy(chMaxMarkerVal, "");
	strcpy(chMarker1Val, "");
	strcpy(chMarker2Val, "");
	strcpy(chDiffMarkerVal, "");
}

KScale::~KScale()
{
}

void KScale::paintEvent(QPaintEvent* ev)
{
	int i, z, iwidth, iheight;
	char label[20];
	
	QFrame::paintEvent(ev);

	QPainter paint(this);
	
	iwidth = width();
	for(i = idxL + iStep; i < iwidth - idxR; i = i + iStep)
	{
		sprintf(label, chXFmt, (i - idxL)*fXScale);
		paint.drawText(i - iStep/2, 10, iStep, 10, AlignCenter, label);
		paint.drawLine(i, idyT -5, i, idyT);
	}
	
	iheight = height();
	z = 0;
	if(bAdaptiveZeroLine)	iZeroLine = height()/2;
	else z = idyB;
	for(i = iZeroLine + iStep + z; i < iheight - idyB; i = i + iStep)
	{
		sprintf(label, chYFmt, (iZeroLine - i + z)*fYScale);
		paint.drawText(0, i -5, idxL, 10, AlignCenter, label);
		paint.drawLine(idxL -5, i, idxL, i);
	}
	for(i = iZeroLine; i > idyT; i = i - iStep)
	{
		sprintf(label, chYFmt, (iZeroLine - i)*fYScale);
		paint.drawText(0, i -5, idxL, 10, AlignCenter, label);
		paint.drawLine(idxL -5, i, idxL, i);
	}
	rMaxMarker.moveBottomLeft(QPoint(width() - idxR + 5, rMaxMarker.bottom()));
	rMarker1.moveBottomLeft(QPoint(width() - idxR + 5, rMarker1.bottom()));
	rMarker2.moveBottomLeft(QPoint(width() - idxR + 5, rMarker2.bottom()));
	rDiffMarker.moveBottomLeft(QPoint(width() - idxR + 5, rDiffMarker.bottom()));

	if(ptMaxMarkerVal.flag) paint.drawText(rMaxMarker, AlignRight, chMaxMarkerVal);
	if(ptMarker1Val.flag) paint.drawText(rMarker1, AlignRight, chMarker1Val);
	if(ptMarker2Val.flag) paint.drawText(rMarker2, AlignRight, chMarker2Val);
	if(ptDiffMarkerVal.flag) paint.drawText(rDiffMarker, AlignRight, chDiffMarkerVal);

	ptMaxMarker.setX(width() - idxR + 5);
	ptMarker1.setX(width() - idxR + 5);
	ptMarker2.setX(width() - idxR + 5);

	paint.drawPixmap(ptMaxMarker, QPixmap((const char **)maxmarker_xpm));
	paint.drawPixmap(ptMarker1, QPixmap((const char **)redmarker_xpm));
	paint.drawPixmap(ptMarker2, QPixmap((const char **)yellowmarker_xpm));
}

void KScale::updateMarkers(ptdata* ptmaxmarker, ptdata* ptmarker1, ptdata* ptmarker2)
{
	ptdata ptdiffmarker;
	QPainter paint(this);
	DrawMarkerValues(&paint, ptmaxmarker, &ptMaxMarkerVal, rMaxMarker, chMaxMarkerVal);
	DrawMarkerValues(&paint, ptmarker1, &ptMarker1Val, rMarker1, chMarker1Val);
	DrawMarkerValues(&paint, ptmarker2, &ptMarker2Val, rMarker2, chMarker2Val);
	if(ptmarker1->flag && ptmarker2->flag)
	{
		ptdiffmarker.flag = drawing;
		ptdiffmarker.iXVal = ptmarker1->iXVal - ptmarker2->iXVal;
		ptdiffmarker.iYVal = ptmarker1->iYVal - ptmarker2->iYVal;
		ptdiffmarker.fYVal = ptmarker1->fYVal - ptmarker2->fYVal;
		DrawMarkerValues(&paint, &ptdiffmarker, &ptDiffMarkerVal, rDiffMarker, chDiffMarkerVal);
	}
	else
		DrawMarkerValues(&paint, &ptdiffmarker, &ptDiffMarkerVal, rDiffMarker, chDiffMarkerVal);
}

void KScale::DrawMarkerValues(QPainter* p, ptdata* ptmarker, ptdata* ptMarkerVal, QRect r, char* chMarkerVal)
{
	if(ptmarker->flag & drawing)
	{
		if(ptmarker->iXVal != ptMarkerVal->iXVal
					|| ptmarker->iYVal != ptMarkerVal->iYVal)
		{
			ptMarkerVal->iXVal = ptmarker->iXVal;
			ptMarkerVal->iYVal = ptmarker->iYVal;
			ptMarkerVal->fXVal = ptmarker->iXVal * fXScale;
			ptMarkerVal->fYVal = ptmarker->fYVal;
			
			p->setPen(p->backgroundColor());	
			p->drawText(r, AlignRight, chMarkerVal);
			
			p->setPen(BLACK);	
			sprintf(chMarkerVal, chFmt, ptMarkerVal->fYVal, ptMarkerVal->fXVal);
			p->drawText(r, AlignRight, chMarkerVal);
			
			ptMarkerVal->flag = ptmarker->flag;
		}
	}
	else if(ptMarkerVal->flag)
	{
		ptMarkerVal->flag = ptMarkerVal->iXVal = ptMarkerVal->iYVal = 0;
			
		p->setPen(p->backgroundColor());	
		p->drawText(r, AlignRight, chMarkerVal);
		
		ptMarkerVal->flag = disabled;
	}
	
}

void KScale::setFixedZeroLine(bool flag, int zeroline)
{
	if(flag)
	{
		iZeroLine = zeroline;
		bAdaptiveZeroLine = FALSE;
	}
	else
	{
		bAdaptiveZeroLine = TRUE;
		if(zeroline == -1)
			iZeroLine = height()/2;
		else
			iZeroLine = zeroline;
	}
}

/****************************************************************/
/*																*/
/*							KDrawArea							*/
/*																*/
/****************************************************************/

static QPointArray data(POINTS);

KDrawArea::KDrawArea(QWidget *parent, const char *name)
	: QWidget(parent, name)
{
	iStep = 100;
	bAdaptiveZeroLine = TRUE;
	iDrawWidth = -1;
	fXScale = fYScale = 1.0f;

	iActiveMarker = 0;

	ptMaxMarkerVal.flag = disabled;
	ptMarker1Val.flag = disabled;
	ptMarker2Val.flag = disabled;
}

KDrawArea::~KDrawArea()
{
}

void KDrawArea::mousePressEvent(QMouseEvent* ev)
{
	if(iActiveMarker == MARKER1)
	{
		if(ptMarker1Val.flag)
		{
			QPainter paint(this);
			DeleteMarker1(&paint);
		}
		ptMarker1Val.flag = drawing;
		ptMarker1Val.iXVal = ev->pos().x();
//		setCursor(arrowCursor);
	}
	else if(iActiveMarker == MARKER2)
	{
		if(ptMarker1Val.flag)
		{
			QPainter paint(this);
			DeleteMarker2(&paint);
		}
		ptMarker2Val.flag = drawing;
		ptMarker2Val.iXVal = ev->pos().x();
//		setCursor(arrowCursor);
	}
}

void KDrawArea::resizeEvent(QResizeEvent* ev)
{
	int i, iheight, iwidth;
	QPoint pt;
	QSize  oldsize;

	iheight = height();
	iwidth = width();
	oldsize = ev->oldSize();

	if(oldsize.width() !=iwidth)
	{
		iwidth = width();
		if(bAdaptiveZeroLine)
		{
			iZeroLine = iheight/2;
			for( i = 0; i < iwidth; i++)
			{
				pt = data.point(i);
				data.setPoint(i, QPoint(i, iZeroLine));
			}
		}
	}
	else
	{
		if(bAdaptiveZeroLine)
		{
			for( i = 0; i < iwidth; i++)
			{
				pt = data.point(i);
				data.setPoint(i, QPoint(i, pt.y() - iZeroLine + iheight/2));
			}
			iZeroLine = iheight/2;
		}
	}
}

void KDrawArea::paintEvent(QPaintEvent* ev)
{
	int iwidth;

	if(iDrawWidth >0)
		iwidth = iDrawWidth;
	else
		iwidth = width();
	
	QPainter paint(this);
	paint.setPen(GREEN);
	paint.drawPolyline(data, 0, iwidth);
	DrawGrid(&paint);
	DrawMarkers(&paint);
}

void KDrawArea::setFixedZeroLine(bool flag, int zeroline)
{
	if(flag)
	{
		iZeroLine = zeroline;
		bAdaptiveZeroLine = FALSE;
	}
	else
	{
		bAdaptiveZeroLine = TRUE;
		if(zeroline == -1)
			iZeroLine = height()/2;
		else
			iZeroLine = zeroline;
	}
}

void KDrawArea::DrawGrid(QPainter* p)
{
	int i, iwidth, iheight;

	iheight = height();
	iwidth = width();

	p->setPen(LTGRAY);
	for(i = iStep; i < iwidth; i = i + iStep)
	{
		p->drawLine(i, 0, i, iheight);
	}
	
	if(bAdaptiveZeroLine)	iZeroLine = iheight/2;
	for(i = iZeroLine + iStep; i < iheight; i = i + iStep)
	{
		p->drawLine(0, i, iwidth, i);
	}
	for(i = iZeroLine; i > 0; i = i - iStep)
	{
		p->drawLine(0, i, iwidth, i);
	}

}

void KDrawArea::DrawData(double* buf, int idx)
{
	int i, dx, dx0;
	int value;
	double fvalue;
	
	dx0 = width();
	if(dx0 > POINTS/2) dx0 = POINTS/2;
	dx = idx;
	
	QPainter paint(this);
	DeleteMarkers(&paint);

	paint.setPen(BLACK);
	paint.drawPolyline(data, 0, dx0);

	ptMaxMarkerVal.fYVal = buf[dx];
	ptMaxMarkerVal.iYVal = iZeroLine - (int) fYScale * ptMaxMarkerVal.fYVal;
	ptMaxMarkerVal.iXVal = 0;
	for( i = 0; i < dx0; i++)
	{
		fvalue = buf[dx];
		value = iZeroLine - (int) fYScale * fvalue;
		if(ptMaxMarkerVal.iYVal > value)
		{
			ptMaxMarkerVal.fYVal = fvalue;
			ptMaxMarkerVal.iYVal = value;
			ptMaxMarkerVal.iXVal = i;
		}
		if(i == ptMarker1Val.iXVal)
		{
			ptMarker1Val.fYVal = fvalue;
			ptMarker1Val.iYVal = value;
		}
		if(i == ptMarker2Val.iXVal)
		{
			ptMarker2Val.fYVal = fvalue;
			ptMarker2Val.iYVal = value;
		}
			
		data.setPoint(i, QPoint(i, value));
		dx = (dx + 1) % BUFFERSIZE;	
	}
	paint.setPen(GREEN);

	if(iDrawWidth >0)
		dx0 = iDrawWidth;

	paint.drawPolyline(data, 0, dx0);
	DrawGrid(&paint);
	DrawMarkers(&paint);
}

void KDrawArea::DrawData(short int* buf, int idx)
{
	int i, dx, dx0;
	int value, ivalue;
	
	dx0 = width();
	dx = idx;
	
	QPainter paint(this);
	DeleteMarkers(&paint);

	paint.setPen(BLACK);
	paint.drawPolyline(data, 0, dx0);

	ptMaxMarkerVal.fYVal = (double) buf[idx];
	ptMaxMarkerVal.iYVal = iZeroLine - fYScale * ptMaxMarkerVal.fYVal/256.0;
	ptMaxMarkerVal.iXVal = 0;
	for( i = 0; i < dx0; i++)
	{
		ivalue = buf[dx];
		value = iZeroLine - fYScale*ivalue/256.0;
		if(ptMaxMarkerVal.iYVal > value)
		{
			ptMaxMarkerVal.iYVal = value;
			ptMaxMarkerVal.fYVal = (double)ivalue;
			ptMaxMarkerVal.iXVal = i;
		}
		if(i == ptMarker1Val.iXVal)
		{
			ptMarker1Val.iYVal = value;
			ptMarker1Val.fYVal = (double)ivalue;
		}
		if(i == ptMarker2Val.iXVal)
		{
			ptMarker2Val.iYVal = value;
			ptMarker2Val.fYVal = (double)ivalue;
		}

		data.setPoint(i, QPoint(i, value));
		dx = (dx + 1) % BUFFERSIZE;	
	}
	paint.setPen(GREEN);

	if(iDrawWidth >0)
		dx0 = iDrawWidth;

	paint.drawPolyline(data, 0, dx0);
	DrawGrid(&paint);
	DrawMarkers(&paint);
}

void KDrawArea::DeleteMarker1(QPainter* p)
{
		p->eraseRect(ptMarker1Val.iXVal - 8, ptMarker1Val.iYVal - 14,
							16, 16);
}

void KDrawArea::DeleteMarker2(QPainter* p)
{
		p->eraseRect(ptMarker2Val.iXVal - 8, ptMarker2Val.iYVal - 14,
							16, 16);
}

void KDrawArea::DeleteMarkers(QPainter* p)
{
	if(ptMaxMarkerVal.flag)
	{
		p->eraseRect(ptMaxMarkerVal.iXVal - 8, ptMaxMarkerVal.iYVal - 14,
							16, 16);
		if(ptMaxMarkerVal.flag == deleting)
			ptMaxMarkerVal.flag = disabled;
	}
	if(ptMarker1Val.flag)
	{
		DeleteMarker1(p);
		if(ptMarker1Val.flag == deleting)
			ptMarker1Val.flag = disabled;
	}
	if(ptMarker2Val.flag)
	{
		DeleteMarker2(p);
		if(ptMarker2Val.flag == deleting)
			ptMarker2Val.flag = disabled;
	}
}

void KDrawArea::DrawMarkers(QPainter* p)
{
	if(ptMaxMarkerVal.flag & drawing)
		p->drawPixmap(ptMaxMarkerVal.iXVal - 8, ptMaxMarkerVal.iYVal - 14,
							QPixmap((const char **)maxmarker_xpm));
	if(ptMarker1Val.flag & drawing)
		p->drawPixmap(ptMarker1Val.iXVal - 8, ptMarker1Val.iYVal - 14,
							QPixmap((const char **)redmarker_xpm));
	if(ptMarker2Val.flag & drawing)
		p->drawPixmap(ptMarker2Val.iXVal - 8, ptMarker2Val.iYVal - 14,
							QPixmap((const char **)yellowmarker_xpm));
}

void KDrawArea::DeleteContents(void)
{
	int i;
	for( i = 0; i < POINTS; i++)
	{
		data.setPoint(i, QPoint(i, iZeroLine));
	}
}

void KDrawArea::mouseMoveEvent(QMouseEvent* ev)
{
	static int x = 0,y = 0;
	QPainter paint(this);
	
	return;
	paint.setRasterOp(XorROP);
	if(x)
		paint.drawPixmap(x,y,QPixmap((const char **)toolbox_xpm));
	x = ev->x() +10;
	y = ev->y() +10;
	paint.drawPixmap(x,y,QPixmap((const char **)toolbox_xpm));

}

int KDrawArea::marker(int id, bool down)
{
	switch(id)
	{
		default:
			break;
		case MARKER1:
			{
				if(down)
				{
					iActiveMarker = id;
					setCursor(QCursor(upArrowCursor));
				}
				else
				{
					ptMarker1Val.flag = deleting;
					if(ptMarker2Val.flag & drawing)
					{
						iActiveMarker = MARKER2;
					}
					else
					{
						setCursor(crossCursor);
						iActiveMarker = 0;
					}
				}
			}
			break;
		case MARKER2:
			{
				if(down)
				{
					iActiveMarker = id;
					setCursor(QCursor(upArrowCursor));
				}
				else
				{
					ptMarker2Val.flag = deleting;
					if(ptMarker1Val.flag & drawing)
					{
						iActiveMarker = MARKER1;
					}
					else
					{
						setCursor(crossCursor);
						iActiveMarker = 0;
					}
				}
			}
			break;
		case EXMARKER:
			{
				int ex;
				ex = ptMarker1Val.iXVal;
				ptMarker1Val.iXVal = ptMarker2Val.iXVal;
				ptMarker2Val.iXVal = ex;
				ex = ptMarker1Val.iYVal;
				ptMarker1Val.iYVal = ptMarker2Val.iYVal;
				ptMarker2Val.iYVal = ex;
			}
			break;
		case MAXMARKER:
			{
				if(down)
				{
					ptMaxMarkerVal.flag = drawing;
				}
				else
				{
					ptMaxMarkerVal.flag = deleting;
				}
			}
			break;
		case KILLMARKER:
			{
					ptMaxMarkerVal.flag = deleting;
					ptMarker1Val.flag = deleting;
					ptMarker2Val.flag = deleting;
					if(iActiveMarker)
					{
						setCursor(crossCursor);
						iActiveMarker = 0;
					}
			}
			break;
	}
	return 0;
}

/****************************************************************/
/*																*/
/*							KDisplay							*/
/*																*/
/****************************************************************/

double Fact(int x)
{
	int i;
	double fact = x;
	for(i = x - 1; i > 0; --i)
	{
		fact = fact * i;
	}
	return fact;
}

double Bessel(double x, int accuracy)
{
	int i;
	double out, x2;
	
	out = 0.0; x2 = x/2;
	for(i = 1; i < accuracy; ++i)
		out = out + pow(( pow(x2, i) / Fact(i)), 2);
	out = out + 1.0;
	return out;
}

KDisplay::KDisplay(QWidget *parent, const char *name)
	: QWidget(parent, name)
{
	int i;
	
	QBoxLayout* layout = new QVBoxLayout( this);
	
	pScale = new KScale(this);
	pScale->setFrameStyle(QFrame::Panel | QFrame::Raised);

	layout->addWidget(pScale);

	pDrawArea = new KDrawArea(pScale);

	pDrawArea->setBackgroundColor(BLACK);
	pDrawArea->setCursor(crossCursor);

	p = rfftw_create_plan(POINTS, FFTW_REAL_TO_COMPLEX, FFTW_ESTIMATE);
	for(i = 0; i < POINTS; i++)
	{
		win[i] = Bessel( 2* 5 * sqrt( i/(double)(POINTS - 1) - 
								pow(i /(double)(POINTS - 1), 2)), 10) /
					Bessel((double) 5, 10);
	}
}

KDisplay::~KDisplay()
{
	rfftw_destroy_plan(p);
}

void KDisplay::paintEvent(QPaintEvent*)
{
}

void KDisplay::resizeEvent(QResizeEvent* resizeEv)
{
	int w = width() - idxL - idxR;
	int h = height() - idyT - idyB;
	pDrawArea->setGeometry(idxL, idyT, w, h);
}

void KDisplay::updateScales(void)
{
	if(WaveInfo.iDomain == TIMEDOMAIN)
	{
		pScale->setXFmtString("%5.1f ms");
		pScale->setXScale(1000.0/(float)WaveInfo.iSampleFrequency);
		pScale->setFixedZeroLine(FALSE);
		pScale->setYScale(256.0);
		pScale->setYFmtString("%7.1f");
		pDrawArea->setFixedZeroLine(FALSE);
		pDrawArea->setDrawWidth();
		pDrawArea->setYScale(1.0);
	}
	else if(WaveInfo.iDomain == FREQUENCYDOMAIN)
	{
		pScale->setXFmtString("%5.1f kHz");
		pScale->setXScale((float)WaveInfo.iSampleFrequency/((float)POINTS * 1000.0));
		pScale->setFixedZeroLine();
		pScale->setYScale(0.25);
		pScale->setYFmtString("%7.1f dB");
		pDrawArea->setFixedZeroLine();
		pDrawArea->setDrawWidth(POINTS/2);
		pDrawArea->setYScale(4.0);
	}
	else pScale->setXFmtString("");

	pDrawArea->DeleteContents();

	pScale->update();
	pDrawArea->update();
}

int KDisplay::open(KDevice* pDev)
{
	int intervall;

	iOldBufferIdx = iBufferIdx = 0;
	iTriggerPoint = 0;
	iTriggerPosition = 0;
	iSlope = 5;
	bTriggerPointFound = FALSE;
	bTrigger = TRUE;

	updateScales();

	pDevice = pDev;
	if(pDevice->open() < 0) return -1;
	pDeviceBuffer = pDevice->getbufadr();

	intervall = (int)(1024000.0/(float)WaveInfo.iSampleFrequency);
	startTimer(intervall);
	return 0;
}

int KDisplay::close(void)
{
	killTimers();
	pDevice->close();
	return 0;
}

void KDisplay::timerEvent(QTimerEvent*)
{
	int idx, newidx;
	
	idx = pDevice->read();
	
	if(WaveInfo.iDomain == FREQUENCYDOMAIN)
		newidx = FrequencyDomain(idx);
	else if(WaveInfo.iDomain == TIMEDOMAIN)
		newidx = TimeDomain(idx);
	else newidx = -1;

	if(newidx > -1)
	{
		if(WaveInfo.iDomain == FREQUENCYDOMAIN)
			pDrawArea->DrawData((double*)&power_spectrum, 0);
		else if(WaveInfo.iDomain == TIMEDOMAIN)
			pDrawArea->DrawData(pDeviceBuffer, newidx);
	

		pScale->updateMarkers(&pDrawArea->ptMaxMarkerVal,
								&pDrawArea->ptMarker1Val,
								&pDrawArea->ptMarker2Val);	
	}
}

int KDisplay::FrequencyDomain(int end)
{
	int i, dx, max,xmax;

	dx  = end - iOldBufferIdx;
	if(dx < 0) dx = dx + BUFFERSIZE;
	if(dx < POINTS) return -1;

	for(i = 0; i < POINTS; i++)
	{
		in[i] = win[i]* pDeviceBuffer[iOldBufferIdx]/32767.0;
		iOldBufferIdx = (iOldBufferIdx + 1) % BUFFERSIZE;
	}
	rfftw_one(p, in, out);
	power_spectrum[0] = 20* log10(sqrt(out[0]*out[0])*2/POINTS);
	for(i = 1, max = power_spectrum[0]; i < (POINTS+1)/2;++i)
	{
		power_spectrum[i] = 20*log10(sqrt(out[i]*out[i] + out[POINTS-i]*out[POINTS-i])*2/POINTS);
		if(max < power_spectrum[i])
		{
			max = power_spectrum[i];
			xmax = i;
		}
	}
	if(POINTS%2 == 0)
		power_spectrum[POINTS/2] = 20* log10(sqrt(out[POINTS/2]*out[POINTS/2])*2/POINTS);
		
	iOldBufferIdx = end;
	
	return end;
}

int KDisplay::TimeDomain(int end)
{
	int j, dx, dslopeval, dy0, dy1;
	short int value;

	if(!bTrigger)
	{
		dx = end - iOldBufferIdx;
		if(dx < 0) dx = dx + BUFFERSIZE;
		if(dx >= pDrawArea->width())
		{
			iOldBufferIdx = end;
			return iOldBufferIdx;
		}
	}
	
	for( ; iOldBufferIdx!= end; iOldBufferIdx = (iOldBufferIdx + 1) % BUFFERSIZE)
	{
		if(bTriggerPointFound)
		{
			dx = end - iTriggerPosition;
			if(dx < 0) dx = dx + BUFFERSIZE;
			
			if(dx >= pDrawArea->width())
			{
				bTriggerPointFound = FALSE;
				iOldBufferIdx = end;
				return iTriggerPosition;
			}
		}
		else
		{
			value =  pDeviceBuffer[iOldBufferIdx];
			if((value > (iTriggerPoint -20)) && (value < (iTriggerPoint +20)) )
			{
				j = iOldBufferIdx - iSlope;
				if(j < 0) j = j + BUFFERSIZE;
				
				dy0 = pDeviceBuffer[j];
				j = (j + 1) % BUFFERSIZE;
				
				for(dslopeval = 0; j != iOldBufferIdx;)
				{
					dy1 = pDeviceBuffer[j];
					dslopeval = dslopeval + dy1 - dy0;
					dy0 = dy1;
					j = (j + 1) % BUFFERSIZE;
				}

				if(dslopeval > 0)
				{
					bTriggerPointFound = TRUE;
					iTriggerPosition = iOldBufferIdx;
				}
			}
		} 
	}
	return -1;
}

int KDisplay::marker(int id, bool down)
{
	
	return pDrawArea->marker(id, down);
}
