/*
   Siag, Scheme In A Grid
   Copyright (C) 1996  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
 */

/*
 * matrix.c
 *
 * This module hides the details of stringpool and matrix handling.
 */

/* 971225 Use strftime for date and time (Dag Nygren, dag@newtech.fi) */

#include <stdio.h>
#include <string.h>
#include "../common/cmalloc.h"
#include <time.h>

#include "calc.h"
#include "types.h"
#include "../common/fonts.h"

static char fmt_integer[80] = "%d";
static char fmt_scientific[80] = "%g";
static char fmt_fixed[80] = "%.2f";

static char fmt_date[80] = "%m/%d/%Y";
static char fmt_time[80] = "%H:%M:%S";

static char fmt_timediff[80] = "%ld:%ld:%ld";
static char fmt_percent[80] = "%.2f %%";
static char fmt_hex[80] = "%lX";
static char fmt_currency[80] = "$%.2f";
static char fmt_user1[80] = "%.2f";
static char fmt_user2[80] = "%.2f";
static char fmt_user3[80] = "%.2f";
static char fmt_user4[80] = "%.2f";
static char fmt_user5[80] = "%.2f";

char *fmt_put(char *fmt, int mode)
{
	switch (mode) {
	case FMT_INTEGER:
		return strncpy(fmt_integer, fmt, 79);
	case FMT_SCIENTIFIC:
		return strncpy(fmt_scientific, fmt, 79);
	case FMT_FIXED:
		return strncpy(fmt_fixed, fmt, 79);
	case FMT_DATE:
		return strncpy(fmt_date, fmt, 79);
	case FMT_TIME:
		return strncpy(fmt_time, fmt, 79);
	case FMT_TIMEDIFF:
		return strncpy(fmt_timediff, fmt, 79);
	case FMT_PERCENT:
		return strncpy(fmt_percent, fmt, 79);
	case FMT_HEX:
		return strncpy(fmt_hex, fmt, 79);
	case FMT_CURRENCY:
		return strncpy(fmt_currency, fmt, 79);
	case FMT_USER1:
		return strncpy(fmt_user1, fmt, 79);
	case FMT_USER2:
		return strncpy(fmt_user2, fmt, 79);
	case FMT_USER3:
		return strncpy(fmt_user3, fmt, 79);
	case FMT_USER4:
		return strncpy(fmt_user4, fmt, 79);
	case FMT_USER5:
		return strncpy(fmt_user5, fmt, 79);
	default:
		return NULL;
	}
}

char *fmt_get(char *fmt, int mode)
{
	char buf[80];

	if (fmt == NULL) fmt = buf;

	switch (mode) {
	case FMT_INTEGER:
		return strncpy(fmt, fmt_integer, 79);
	case FMT_SCIENTIFIC:
		return strncpy(fmt, fmt_scientific, 79);
	case FMT_FIXED:
		return strncpy(fmt, fmt_fixed, 79);
	case FMT_DATE:
		return strncpy(fmt, fmt_date, 79);
	case FMT_TIME:
		return strncpy(fmt, fmt_time, 79);
	case FMT_TIMEDIFF:
		return strncpy(fmt, fmt_timediff, 79);
	case FMT_PERCENT:
		return strncpy(fmt, fmt_percent, 79);
	case FMT_HEX:
		return strncpy(fmt, fmt_hex, 79);
	case FMT_CURRENCY:
		return strncpy(fmt, fmt_currency, 79);
	case FMT_USER1:
		return strncpy(fmt, fmt_user1, 79);
	case FMT_USER2:
		return strncpy(fmt, fmt_user2, 79);
	case FMT_USER3:
		return strncpy(fmt, fmt_user3, 79);
	case FMT_USER4:
		return strncpy(fmt, fmt_user4, 79);
	case FMT_USER5:
		return strncpy(fmt, fmt_user5, 79);
	default:
		return NULL;
	}
}

/*X
   void free_matrix(spread **matrix)
   Frees the memory used by the matrix.
   X */
void free_matrix(spread ** matrix)
{

	cfree((char *) matrix);
}


static spread empty_cell = {
	NULL, 0, {0}, EMPTY, HELVETICA | SIZE_10
};

/* make sure the cell is allocated */
static void alloc_cell(buffer *b, int row, int col)
{
	int i;

	empty_cell.format = b->sf;
	if (b->matrix == NULL) {
		b->matrix = (spread **)cmalloc((row+1)*sizeof(spread *));
		b->alloc_cols = (long *)cmalloc((row+1)*sizeof(long));
		for (i = 0; i <= row; i++) {
			b->matrix[i] = NULL;
			b->alloc_cols[i] = 0;
		}
		b->alloc_lines = row;
	} else if (row > b->alloc_lines) {
		b->matrix = (spread **)crealloc(b->matrix, (row+1)*sizeof(spread *));
		b->alloc_cols = (long *)crealloc(b->alloc_cols, (row+1)*sizeof(long));
		for (i = b->alloc_lines+1; i <= row; i++) {
			b->matrix[i] = NULL;
			b->alloc_cols[i] = 0;
		}
		b->alloc_lines = row;
	}
	if (b->matrix[row] == NULL) {
		b->matrix[row] = (spread *)cmalloc((col+1)*sizeof(spread));
		for (i = 0; i <= col; i++)
			b->matrix[row][i] = empty_cell;
		b->alloc_cols[row] = col;
	} else if (col > b->alloc_cols[row]) {
		b->matrix[row] = (spread *)crealloc(b->matrix[row],
						(col+1)*sizeof(spread));
		for (i = b->alloc_cols[row]+1; i <= col; i++)
			b->matrix[row][i] = empty_cell;
		b->alloc_cols[row] = col;
	}
}

void swap_cells(buffer *buf, long r1, long c1, long r2, long c2)
{
	spread sp;

	alloc_cell(buf, r1, c1);
        alloc_cell(buf, r2, c2);
        sp = buf->matrix[r1][c1];
        buf->matrix[r1][c1] = buf->matrix[r2][c2];
        buf->matrix[r2][c2] = sp;
}

int ins_data(buffer *b, int interpreter, char *texti, cval v, short t,
	     int row, int col)
{
	int change = FALSE;
	char *p = NULL;
	register spread *mrowcol;	/* faster than matrix[row][col] */

	if (texti == NULL) texti = "";
	if (strlen(texti) == 0) t = EMPTY;

	/* these limits are now very large */
	if (row > BUFFER_ROWS || col > BUFFER_COLS) return FALSE;

	alloc_cell(b, row, col);
	mrowcol = b->matrix[row] + col;
	if (mrowcol->text != texti || mrowcol->type != t)
		change = TRUE;
	mrowcol->interpreter = interpreter;
	if (mrowcol->text != texti) {	/* we could be restoring the same string */
		if (mrowcol->text) cfree(mrowcol->text);
		mrowcol->text = cstrdup(texti);
	}


	if (t == STRING) {
		if (v.text) p = cstrdup(v.text);
		else p = cstrdup("");
	}

	if (mrowcol->type == STRING)	/* free the old value */
		cfree(mrowcol->value.text);
	
	if (t == STRING)
		mrowcol->value.text = p;
	else
		mrowcol->value.number = v.number;
	mrowcol->type = t;
	return change;
}				/* ins_data */

/* check if a cell has been allocated */
static int is_alloc(buffer *b, int row, int col)
{
	if (row < 1 || row > BUFFER_ROWS ||
		col < 1 || col > BUFFER_COLS) return 0;
	if (b->matrix == NULL) return 0;
	if (row > b->alloc_lines) return 0;
	if (b->matrix[row] == NULL) return 0;
	if (col > b->alloc_cols[row]) return 0;
	return 1;
}

int ins_format(buffer *b, int row, int col, int format)
{
	int retval;

	if (row > BUFFER_ROWS || col > BUFFER_COLS) return FALSE;

	if ((format & BORDER_MASK) || is_alloc(b, row, col))
		retval = 1;	/* must change */
	else
		retval = 2;	/* don't want to */

	alloc_cell(b, row, col);

	b->matrix[row][col].format = format;
	return 1;
}

char *ret_text(buffer *b, int row, int col)
{
	if (!is_alloc(b, row, col)) return NULL;
	return b->matrix[row][col].text;
}	/* ret_text */

int ret_interpreter(buffer *b, int row, int col)
{
	if (!is_alloc(b, row, col)) return 0;
	return b->matrix[row][col].interpreter;
}

cval ret_val(buffer *b, int row, int col)
{
	register spread *mrow;	/* faster than matrix[row] */

	if (!is_alloc(b, row, col)) {
		cval value;
		value.number = 0;
		return value;
	}

	mrow = b->matrix[row];
	return mrow[col].value;
}				/* ret_val */

char *ret_string(buffer *b, int row, int col)
{
	register spread *mrow;
	char *v;
	if (!is_alloc(b, row, col)) return NULL;
	mrow = b->matrix[row];
	if (mrow[col].type == STRING) {
		v = mrow[col].value.text;
		return v;
	}
	return NULL;
}

short ret_type(buffer *b, int row, int col)
{
	register spread *mrow;	/* faster than matrix[row] */
	if (!is_alloc(b, row, col)) return EMPTY;
	mrow = b->matrix[row];
	return mrow[col].type;
}				/* ret_type */

int ret_format(buffer *b, int row, int col)
{
	if (!is_alloc(b, row, col)) return b->sf;
	return b->matrix[row][col].format;
}

int ret_font(buffer *b, int row, int col)
{
	return ret_format(b, row, col) & 127;
}

int ret_color(buffer *b, int row, int col)
{
	if (!is_alloc(b, row, col)) return 0;
	return (ret_format(b, row, col) & COLOR_MASK) >> COLOR_SHIFT;
}

void set_color(buffer *b, int row, int col, int color)
{
	int oldfmt;
	if (row > BUFFER_ROWS || col > BUFFER_COLS) return;
	oldfmt = ret_format(b, row, col) & ~COLOR_MASK;
	ins_format(b, row, col, oldfmt | (color << COLOR_SHIFT));
}

#ifndef ABS
#define ABS(x) ((x)>0?(x):-(x))
#endif

char *ret_pvalue(char *buf, buffer *b, int row, int col, int mode)
{
	static char p[1024];
	int type;
	double f;
	long d;
	time_t ttime;
	struct tm *ltime;
	long hour, min, sec;

	f = 0.0;

	if (buf == NULL) buf = p;
	if (row > BUFFER_ROWS || col > BUFFER_COLS) return NULL;
	type = ret_type(b, row, col);

	if (mode < 0) mode = ret_format(b, row, col);
	mode &= FMT_MASK;

	if (type == EXPRESSION) {
		f = ret_val(b, row, col).number;
		/* default means we get to guess */
		if (mode == FMT_DEFAULT) {
			d = (long) f;
			if (ABS(d-f) < .001) mode = FMT_INTEGER;
			else if (f > 10000000) mode = FMT_SCIENTIFIC;
			else mode = FMT_FIXED;
		}
	}
	buf[0] = '\0';
	switch (type) {
	case EMPTY:
		break;
	case ERROR:
		if (mode != FMT_INVISIBLE)
			strcpy(buf, "ERROR");
		break;
	case LABEL:
	case EMBED:
		if (mode != FMT_INVISIBLE)
			strncpy(buf, ret_text(b, row, col), 1000);
		break;
	case STRING:
		if (mode != FMT_INVISIBLE)
			strncpy(buf, ret_string(b, row, col), 1000);
		break;
	case EXPRESSION:
		switch (mode & FMT_MASK) {
		case FMT_DEFAULT:
		case FMT_INVISIBLE:
		default:
			buf[0] = '\0';
			break;
		case FMT_INTEGER:
			d = (long) f;
			sprintf(buf, fmt_integer, d);
			break;
		case FMT_SCIENTIFIC:
			sprintf(buf, fmt_scientific, f);
			break;
		case FMT_FIXED:
			sprintf(buf, fmt_fixed, f);
			break;
		/* date, time and comma are NYI */
		case FMT_DATE:
			ttime = (time_t) f;
			ltime = localtime(&ttime);

			strftime(buf, sizeof(p), fmt_date, ltime);

			break;
		case FMT_TIME:
			ttime = (time_t) f;
			ltime = localtime(&ttime);

			strftime(buf, sizeof(p), fmt_time, ltime);

			break;
		case FMT_TIMEDIFF:
			d = (long)f;
			sec = d % 60;
			d /= 60;
			min = d % 60;
			hour = d / 60;
			sprintf(buf, fmt_timediff, hour, min, sec);
			break;
		case FMT_PERCENT:
			sprintf(buf, fmt_percent, 100*f);
			break;
		case FMT_HEX:
			d = (long) f;
			sprintf(buf, fmt_hex, d);
			break;
		case FMT_CURRENCY:
			sprintf(buf, fmt_currency, f);
			break;
		case FMT_USER1:
			sprintf(buf, fmt_user1, f);
			break;
		case FMT_USER2:
			sprintf(buf, fmt_user2, f);
			break;
		case FMT_USER3:
			sprintf(buf, fmt_user3, f);
			break;
		case FMT_USER4:
			sprintf(buf, fmt_user4, f);
			break;
		case FMT_USER5:
			sprintf(buf, fmt_user5, f);
			break;
		}
		break;
	}
	return buf;
}

int line_last_used(buffer *b)
{
	register int r;
	for (r = b->alloc_lines; r > 1 && b->matrix[r] == NULL; r--);
	return r;
}

int col_last_used(buffer *b, int row)
{
	register int i;
	if (row < 1 || row > b->alloc_lines) return 0;
	for (i = b->alloc_cols[row];
	     i > 1 && ret_type(b, row, i) == EMPTY; i--);
	return i;
}

void downshift_matrix(buffer *b, int row)
{
	int i;

	/* free the last line, if there is anything in it */
	if (b->alloc_lines > BUFFER_ROWS && b->matrix[BUFFER_ROWS])
		cfree(b->matrix[BUFFER_ROWS]);

	/* extend the line allocation in the matrix */
	if (b->alloc_lines >= row && b->alloc_lines < BUFFER_ROWS) {
		b->alloc_lines++;
		b->matrix = (spread **)crealloc(b->matrix, (b->alloc_lines+1)*sizeof(spread *));
		b->alloc_cols = (long *)crealloc(b->alloc_cols, (b->alloc_lines+1)*sizeof(long));
	}

	/* extend the height table */
	if (b->used_lines >= row && b->used_lines < BUFFER_ROWS) {
		b->used_lines++;
		b->height = (int *)crealloc(b->height, (b->used_lines+1)*sizeof(int));
	}

	/* move all the stuff down one line */
	for (i = b->alloc_lines; i > row; i--) {
		b->matrix[i] = b->matrix[i - 1];
		b->alloc_cols[i] = b->alloc_cols[i - 1];
	}
	for (i = b->used_lines; i > row; i--)
		b->height[i] = b->height[i - 1];

	/* make the first line point to nothing */
	if (b->alloc_lines >= row) {
		b->matrix[row] = NULL;
		b->alloc_cols[row] = 0;
	}
	if (b->used_lines >= row)
		b->height[row] = 20;

	update_all_references(b, row, 1, BUFFER_ROWS, BUFFER_COLS, 1, 0);

	b->recalc = b->change = TRUE;

	for (i = 0; i < b->nplugin; i++) {
		if (row <= b->plugin[i].row) b->plugin[i].row++;
	}
}

void upshift_matrix(buffer *b, int row)
{
	int i;

	/* free the first line, if there is anything in it */
	if (b->alloc_lines > row && b->matrix[row])
		cfree(b->matrix[row]);

	/* we don't bother to shrink the matrix or height table */

	/* move everything up one line */
	for (i = row; i < b->alloc_lines; i++) {
		b->matrix[i] = b->matrix[i + 1];
		b->alloc_cols[i] = b->alloc_cols[i + 1];
	}
	for (i = row; i < b->used_lines; i++)
		b->height[i] = b->height[i + 1];

	/* make the last line point to nothing */
	if (b->alloc_lines > row) {
		b->matrix[b->alloc_lines] = NULL;
		b->alloc_cols[b->alloc_lines] = 0;
	}
	if (b->used_lines > row)
		b->height[b->used_lines] = 20;

	update_all_references(b, row, 1, BUFFER_ROWS, BUFFER_COLS, -1, 0);

	b->recalc = b->change = TRUE;

	for (i = 0; i < b->nplugin; i++) {
		if (row <= b->plugin[i].row) b->plugin[i].row--;
	}
}

void rightshift_matrix(buffer *b, int col)
{
	int i, j;
	empty_cell.format = b->sf;
	/* extend the width table if necessary */
	if (b->used_cols >= col && b->used_cols < BUFFER_COLS) {
		b->used_cols++;
		b->width = (int *)crealloc(b->width, (b->used_cols+1)*sizeof(int));
		for (i = b->used_cols; i > col; i--)
			b->width[i] = b->width[i-1];
		b->width[col] = 80;
	}

	/* process all the allocated lines */
	for (i = 1; i <= b->alloc_lines; i++) {
		if (b->matrix[i]) {
			/* check if it needs to grow */
			if (b->alloc_cols[i] >= col) {
				b->alloc_cols[i]++;
				b->matrix[i] = (spread *)crealloc(b->matrix[i],
						(b->alloc_cols[i]+1)*sizeof(spread));
			}

			/* shift to the right */
			for (j = b->alloc_cols[i]; j > col; j--) {
				b->matrix[i][j] = b->matrix[i][j - 1];
			}

			/* and make the leftmost column empty */
			if (b->alloc_cols[i] > col) {
				b->matrix[i][col] = empty_cell;
			}
		}
	}

	update_all_references(b, 1, col, BUFFER_ROWS, BUFFER_COLS, 0, 1);

	b->recalc = b->change = TRUE;

	for (i = 0; i < b->nplugin; i++) {
		if (col <= b->plugin[i].col) b->plugin[i].col++;
	}
}

void leftshift_matrix(buffer *b, int col)
{
	int i, j;

	empty_cell.format = b->sf;
	/* do not shrink the width table */
	for (j = col; j < b->used_cols; j++)
		b->width[j] = b->width[j + 1];
	if (b->used_cols >= col)
		b->width[col] = 80;

	for (i = 1; i <= b->alloc_lines; i++) {
		if (b->matrix[i]) {
			/* do not free unused columns */

			/* shift to the left */
			for (j = col; j < b->alloc_cols[i]; j++) {
				b->matrix[i][j] = b->matrix[i][j + 1];
			}

			/* and make the rightmost empty */
			if (b->alloc_cols[i] >= col)
				b->matrix[i][b->alloc_cols[i]] = empty_cell;
		}
	}

	update_all_references(b, 1, col, BUFFER_ROWS, BUFFER_COLS, 0, -1);

	b->recalc = b->change = TRUE;

	for (i = 0; i < b->nplugin; i++) {
		if (col <= b->plugin[i].col) b->plugin[i].col--;
	}
}

/* Allocate a block of memory of sufficient size and encode the contents
   of a specified rectangular area into it. Caller must free.
*/
char *pack_area(buffer *b, long r1, long c1, long r2, long c2, long *size)
{
	long r, c;
	int intp;
	char *p, *data;

	/* use fixed size buffer for now */
	*size = 32000;
	p = cmalloc(*size);
	data = p;
	sprintf(data, "%ld %ld", r2-r1+1, c2-c1+1);
	data += strlen(data)+1;

	for (r = r1; r <= r2 && r <= line_last_used(b); r++) {
		for (c = c1; c <= c2 && c <= col_last_used(b, r); c++) {
			switch (ret_type(b, r, c)) {
			case EMPTY:
				sprintf(data, "#");
				break;
			case LABEL:
				sprintf(data, "\"%s",
					ret_text(b, r, c));
				break;
			case EMBED:
				sprintf(data, "m%s",
					ret_text(b, r, c));
				break;
			default:
				intp = ret_interpreter(b, r, c);
				sprintf(data, "+%s,%s",
					interpreter2name(intp),
					ret_text(b, r, c));
				break;
			}
			data += strlen(data)+1;
			if (data-p > 31000) {
				cfree(p);
				return NULL;
			}
		}
		/* use newline to mark end of line */
		sprintf(data, "\n");
		data += strlen(data)+1;
	}
	/* use null character to mark end of area */
	*data = '\0';
	return p;
}

/* Unpack a previously encoded memory block into the position given.
*/
void unpack_area(buffer *b, char *data, long r1, long c1)
{
	long r2, c2, r, c, rows, cols;
	int intp;
	char *comma, *texti;
	cval value;
	char name[256];
	int len;

	if (!data) return;
	value.number = 0;
	sscanf(data, "%ld %ld", &rows, &cols);

	data += strlen(data)+1;
	r2 = r1+rows-1;
	c2 = c1+cols-1;
	/* blank out all cells in the target */
	for (r = r1; r <= r2 && r <= line_last_used(b); r++) {
		for (c = c1; c <= c2 && c <= col_last_used(b, r); c++) {
			ins_data(b, siod_interpreter,
				NULL, value, EMPTY, r, c);
		}
	}
	r = r1;
	c = c1;
	while (*data) {
		switch (*data++) {
		case '"':
			texti = data;
			ins_data(b, siod_interpreter,
				texti, value, LABEL, r, c++);
			break;
		case '+':
			comma = strchr(data, ',');
			if (!comma) {
				fprintf(stderr,
					"unpack_area: no interpreter\n");
				return;
			}
			len = comma-data;
			strncpy(name, data, len);
			name[len] = '\0';
			intp = name2interpreter(name);
			if (intp < 0) intp = siod_interpreter;
			data = comma+1;
			ins_data(b, intp, data, value,
				EXPRESSION, r, c++);
			break;
		case '\n':
			r++;
			c = c1;
			break;
		case '#':
			c++;
			break;
		default:
			fprintf(stderr, "unpack_area: bogus data\n");
			fprintf(stderr, "\t%s\n", data-1);
			return;
		}
		data += strlen(data)+1;
	}
}

static struct {
	char *data;
	long r1, c1;
} undo_buffer;

/* Saves a rectangular area of the buffer. Called before any command
   that changes the buffer and should be undoable. Returns 0 for
   success and 1 for failure. */
int undo_save(buffer *b, long r1, long c1, long r2, long c2)
{
	long size;

	cfree(undo_buffer.data);
	undo_buffer.r1 = r1;
	undo_buffer.c1 = c1;
	undo_buffer.data = pack_area(b, r1, c1, r2, c2, &size);
	return (undo_buffer.data == NULL);
}

/* Puts back whatever was last saved by undo_save. Returns 0 for
   success and 1 for failure. */
int undo_restore(buffer *b)
{
	if (undo_buffer.data == NULL) return 1;
	unpack_area(b, undo_buffer.data, undo_buffer.r1, undo_buffer.c1);
	undo_buffer.data = NULL;
	return 0;
}

