/* -*- mode: C++; c-file-style: "Stroustrup" -*-
 *
 * $Id: qkbrowser.C,v 1.7 1999/01/29 15:24:04 kuepper Exp $
 */


#include <qstring.h>
#include <qpixmap.h>
#include <qkeycode.h>
#include <qpainter.h>
#include <qapp.h>
#include <qdrawutl.h>

#include <stdio.h>
#include <math.h>

#include "qkbrowser.h"


QKBrowseBox::QKBrowseBox( int x, int y, QWidget* parent, const char* name, WFlags f )
    : QTableView( parent, name, WType_Popup ),
      _firstrelease( false ), _mousepressed( false )
{
  setNumCols( x );
  setNumRows( y );
  setCellWidth( width()/x );
  setCellHeight( height()/y );
  setTableFlags( Tbl_clipCellPainting );

  _texts = new QString[x*y];
  _pixmaps = new QPixmap[x*y];

  _activecell.setX( -1 );
  _activecell.setY( -1 );
  setMouseTracking( true );
  if( style() == MotifStyle )
      setFrameStyle( QFrame::Panel | QFrame::Sunken );
  else
      setFrameStyle( QFrame::WinPanel | QFrame::Sunken );
}



QKBrowseBox::~QKBrowseBox()
{
    delete [] _texts;
    delete [] _pixmaps;
}


int QKBrowseBox::coordsToIndex( int x, int y )
{
  if( x > numCols() || y > numRows() )
	fprintf( stderr, "coordsToIndex: invalid coords (%d, %d)\n", x, y  );

  return y * numCols() + x;
}

  
void QKBrowseBox::insertItem( QString text, int x, int y )
{
  _texts[ coordsToIndex( x, y ) ] = text.copy();
}

  
void QKBrowseBox::insertItem( QPixmap pixmap, int x, int y )
{
  _pixmaps[ coordsToIndex( x, y ) ] = pixmap;
}

  
void QKBrowseBox::removeItem( int x, int y )
{
  _texts[ coordsToIndex( x, y ) ] = "";
  _pixmaps[ coordsToIndex( x, y ) ].resize( 0, 0 );
}

  
void QKBrowseBox::clear()
{
  for( int x = 0; x < numCols(); x++ )
	for( int y = 0; y < numRows(); y++ )
	  removeItem( x, y );
}

  
QString QKBrowseBox::text( int x, int y )
{
  if( x >= numCols() || y >= numRows() )
	return "";
  return _texts[ coordsToIndex( x, y ) ];
}

  
QPixmap QKBrowseBox::pixmap( int x, int y )
{
  static QPixmap empty;

  if( x >= numCols() || y >= numRows() )
	return empty;
  return _pixmaps[ coordsToIndex( x, y ) ];
}

  
int QKBrowseBox::exec( const QPoint& pos )
{
    return exec( pos.x(), pos.y() );
}

  
int QKBrowseBox::exec( int x, int y )
{
  _firstrelease = true;
  move( x, y );
  show();
  repaint();
  qApp->enter_loop();
  hide();
  if( _activecell.x() != -1 && _activecell.y() != -1 )
	return _activecell.x() + _activecell.y() * numCols();
  else
	return -1;
}

  
int QKBrowseBox::exec( const QWidget* trigger )
{
  QPoint globalpos = trigger->parentWidget()->mapToGlobal( trigger->pos() );

  // is there enough space to put the box below the trigger?
  if( globalpos.y() + trigger->height() + height() + 1 < 
	  QApplication::desktop()->height() )
	{
	  // is there enough space to set the box left-justified with the trigger
	  if( globalpos.x() + width() < QApplication::desktop()->width() )
		// put the box left-justified below the trigger
		return exec( globalpos.x(), globalpos.y()+trigger->height() + 1 );
	  else
		// put the box below the trigger, extending to the left
		return exec( globalpos.x() - width() - 1, 
					 globalpos.y()+trigger->height() + 1 );
	}
  else
	{
	  // not enough space below: put the box above the trigger
	  // is there enough space to set the box left-justified with the trigger
	  if( globalpos.x() + width() < QApplication::desktop()->width() )
		// put the box left-justified above the trigger
		return exec( globalpos.x(), globalpos.y() - height() - 1 );
	  else
		// put the box above the trigger, extending to the left
		return exec( globalpos.x() - width() - 1, 
					 globalpos.y() - height() + 1 );
	}
}


void QKBrowseBox::keyPressEvent( QKeyEvent* e )
{
    if( e->key() == Key_Escape ) {
	qApp->exit_loop();
    }
}


void QKBrowseBox::paintCell( class QPainter * painter, int y, int x )
{
    if( ! _pixmaps[ coordsToIndex( x, y ) ].isNull() )
	painter->drawPixmap( 0, 0, _pixmaps[ coordsToIndex( x, y ) ] );
    bool bActive = ( ( _activecell.x() == x ) && ( _activecell.y() == y ) );
    if( style() == MotifStyle ) {
	if( bActive ) {
	    if( _mousepressed )
		qDrawShadePanel( painter, 0, 0, cellWidth(), cellHeight(), colorGroup(), true, 2 );
	    else
		qDrawShadePanel( painter, 0, 0, cellWidth(), cellHeight(), colorGroup(), false, 2 );
	} else {
	    qDrawPlainRect( painter, 0, 0, cellWidth(), cellHeight(), colorGroup().background(), 2 );
	}
    } else if( bActive ) {
	if( _mousepressed )
	    qDrawShadePanel( painter, 0, 0, cellWidth(), cellHeight(), colorGroup(), true, 1 );
	else
	    qDrawShadePanel( painter, 0, 0, cellWidth(), cellHeight(), colorGroup(), false, 1 );
    } else {
	qDrawPlainRect( painter, 0, 0, cellWidth(), cellHeight(), colorGroup().background(), 1 );
    }
    if( ! _texts[ coordsToIndex( x, y ) ].isEmpty() ) {
	painter->drawText( 0, 0, cellWidth(), cellHeight(), AlignLeft, _texts[ coordsToIndex( x, y ) ] );
    }
}


void QKBrowseBox::resizeEvent( QResizeEvent* e )
{
  setCellWidth( width()/numCols() );
  setCellHeight( height()/numRows() );
}



void QKBrowseBox::mousePressEvent( QMouseEvent* e )
{
    _mousepressed = true;
    update();
}



void QKBrowseBox::mouseMoveEvent( QMouseEvent* e )
{
  int x = e->pos().x();
  int y = e->pos().y();

  int cellx, celly;
  if( x < 0 || y < 0 || x > width() || y > height() ) // outside the box
	{
	  cellx = -1;
	  celly = -1;
	}
  else
	{
	  cellx = (int)floor( ((double)x) / ((double)cellWidth()) );
	  celly = (int)floor( ((double)y) / ((double)cellHeight()) );
	}

  if( (_activecell.x() != cellx) || (_activecell.y() != celly) )
	 {
	   // mouse has been moved to another cell
	   int oldactivecellx = _activecell.x();
	   int oldactivecelly = _activecell.y();
	   _activecell.setX( cellx );
	   _activecell.setY( celly );
	   updateCell( oldactivecelly, oldactivecellx ); // remove old highlighting
	   updateCell( _activecell.y(), _activecell.x() ); // set new highlighting
	 }
}


void QKBrowseBox::mouseReleaseEvent( QMouseEvent* e )
{
    if( _firstrelease )
	_firstrelease = false;
    else {
	_mousepressed = false;
	qApp->exit_loop();
    }
}

