/* -*- mode: C++; c-file-style: "Stroustrup" -*-
 * This file is part of
 * ======================================================
 *
 *           LyX, the High Level Word Processor
 * 	
 *	    Copyright (C) 1995 Matthias Ettrich,
 *           1995 - 1997 The LyX Team.
 *
 * ======================================================
 *             Now its also part of KLyX :-)
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <qprinter.h>
#if QT_VERSION >= 140
#include <qprintdialog.h> //sven
#endif

#include <kfiledialog.h>

#include "klyx.h"

#include <config.h>

#include <kapp.h>
#include <klocale.h>

#include <ctype.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <X11/cursorfont.h>

#include "LString.h"
#include "lyx_main.h"
#include "spellchecker.h"
#include "lyx_cb.h"
#include "credits.h"
#include "insetref.h"
#include "insetquotes.h"
#include "insetlatex.h"
#include "insetlabel.h"
#include "insetinfo.h"
#include "insetlatexdel.h"
#include "insetspecialchar.h"
#include "figinset.h"
#include "lyxfunc.h"
#include "lyxfont.h"
#include "minibuffer.h"
#include "bufferlist.h"
#include "filetools.h"
#include "pathstack.h"
#include "lyx_gui_misc.h"
#include "LyXView.h" // only because of form_main
#include "lastfiles.h"
#include "FileInfo.h"
#include "lyxscreen.h"
#include "error.h"
#include "syscall.h"
#include "lyxlib.h"
#include "lyxserver.h"
#include "FontLoader.h"
#include "lyxrc.h"
#include "lyxtext.h"
#include "qkstring.h"

#include <ktopwidget.h>
#include <kapp.h>
#include <klocale.h>
#include <ktreelist.h>

#include "ParagraphDialog.h"
#include "InsertTableDialog.h"
#include "InsertFigureDialog.h"
#include "DocumentLayoutDialog.h"
#include "CharacterStyleDialog.h"
#include "ScreenOptionsTabDialog.h"
#include "ScreenFontsDialog.h"
#include "ScreenToolbarsDialog.h"
#include "SpellcheckerOptionsDialog.h"
#include "MathSpacingDialog.h"
#include "MathMatrixDialog.h"
#include "MathBrowsers.h"
#include "MathDelimiterDialog.h"
#include "SendFaxDialog.h"
#include "TableOfContentsDialog.h"
#include "ExportCustomDialog.h"
#include "TableOfContentsDialog.h"
#include "FindReplaceDialog.h"
#include "ViewLogfileDialog.h"
#include "BulletShapeDialog.h"
#include "InsertLabelDialog.h"
#include "InsertIndexEntryDialog.h"
#include "InsertCrossReferenceDialog.h"
#include "InsertCitationReferenceDialog.h"
#include "LaTeXOptionsDialog.h"
#include "LaTeXPreambleDialog.h"
#include "QuotesDialog.h"
#include "PaperFormatDialog.h"
#include "InsertBibitemDialog.h"
#include "InsertIncludeDialog.h"
#include "CreditsDialog.h"
#include "InsertErrorDialog.h"
#include "InsertInfoDialog.h"
#include "VCLogDialog.h"

// 	$Id: lyx_cb.C,v 1.82 1999/01/29 12:57:23 kuepper Exp $	

#if !defined(lint) && !defined(WITH_WARNINGS)
static char vcid[] = "$Id: lyx_cb.C,v 1.82 1999/01/29 12:57:23 kuepper Exp $";
#endif /* lint */

extern BufferList bufferlist;
extern void show_symbols_form(void);

extern BufferView *current_view; // called too many times in this file...

extern void DeleteSimpleCutBuffer(); /* for the cleanup when exiting */
extern void FigureOKCB();

extern void sigchldhandler(int sig);
extern int send_fax(LString const &fname, LString const &sendcmd);

extern LyXServer *lyxserver;

extern Window selection_communication_window;
extern BufferView* selection_communication_bufferview;

// global toolbar settings
extern ScreenToolbarsDialogData global_toolbar_visibilities;

// this should be static, but I need it in buffer.C
bool quitting;	// flag, that we are quitting the program
extern bool finished; // all cleanup done just let it run through now.

bool scrolling = false;

char updatetimer = 0;

/* whether the work area should get callbacks */
bool input_prohibited = false;

/* the selection possible is needed, that only motion events are
* used, where the bottom press event was on the drawing area too */
bool selection_possible = false;

void InsertCorrectQuote();

void TableOKCB();

/*
   This is the inset locking stuff needed for mathed --------------------

   an inset can simple call LockInset in it's edit call and *ONLY* in it's
   edit call.
   Inset::Edit() can only be called by the main lyx module.

   Then the inset may modify the menu's and/or iconbars.

   Unlocking is either done by LyX or the inset itself with a UnlockInset-call

   During the lock, all button and keyboard events will be modified
   and send to the inset through the following inset-features. Note that
   Inset::InsetUnlock will be called from inside UnlockInset. It is meant
   to contain the code for restoring the menus and things like this.


   virtual void InsetButtonPress(int x, int y, int button);
   virtual void InsetButtonRelease(int x, int y, int button);
   virtual void InsetKeyPress(XKeyEvent *ev);
   virtual void InsetMotionNotify(int x, int y, int state);
   virtual void InsetUnlock();

   If a inset wishes any redraw and/or update it just has to call
   UpdateInset(this).
   It's is completly irrelevant, where the inset is. UpdateInset will
   find it in any paragraph in any buffer.
   Of course the_locking_inset and the insets in the current paragraph/buffer
   are checked first, so no performance problem should occur.

   Hope that's ok for the beginning, Alejandro,
   sorry that I needed so much time,

                  Matthias
   */

void UpdateInset(Inset* inset, bool mark_dirty = true);
/* these functions return 1 if an error occured,
   otherwise 0 */
// Now they work only for updatable insets. [Alejandro 080596]
int LockInset(UpdatableInset* inset);
void ToggleLockedInsetCursor(long x, long y, int asc, int desc);
void FitLockedInsetCursor(long x, long y, int asc, int desc);
int UnlockInset(UpdatableInset* inset);
void LockedInsetStoreUndo(Undo::undo_kind kind);

/* this is for asyncron updating. UpdateInsetUpdateList will be called
   automatically from LyX. Just insert the Inset into the Updatelist */
void UpdateInsetUpdateList();
void PutInsetIntoInsetUpdateList(Inset* inset);

InsetUpdateStruct *InsetUpdateList = NULL;


/*
  -----------------------------------------------------------------------
 */

/* some function prototypes */

void GotoNote();
void OpenStuff();
void ToggleFloat();
void MenuUndo();
void MenuRedo();
void HyphenationPoint();
void HFill();
void Newline();
void ProtectedBlank();
void CopyCB();
int RunLinuxDoc(int, LString const &);
bool MenuWrite(Buffer* buf);
bool MenuWriteAs(Buffer *buffer);
void MenuReload(Buffer *buf);
void MenuLayoutSave();
void UpdateLayoutDocument( BufferParams const &params );

unsigned char GetCurrentTextClass()
	// Who are we asking?
	// Shouldn't this question be directed to the buffer?
	// Indeed it should. Asger.
{
	return current_view->currentBuffer()->params.textclass;
}


// How should this actually work? Should it prohibit input in all BufferViews,
// or just in the current one? If "just the current one", then it should be
// placed in BufferView. If "all BufferViews" then LyXGUI (I think) should
// run "ProhibitInput" on all LyXViews which will run prohibitInput on all
// BufferViews. Or is it perhaps just the (input in) BufferViews in the
// current LyxView that should be prohibited (Lgb) (This applies to
// "AllowInput" as well.)
void ProhibitInput()
{
	input_prohibited = true;
	if (current_view->getScreen())
		current_view->getScreen()->HideCursor();

	/* set the cursor to the watch for all forms and the canvas */
	QApplication::setOverrideCursor( waitCursor );

	QApplication::flushX();
}


// Should find a way to move this into BufferView.C
void SetXtermCursor(Window win)
{
	static Cursor cursor;
	static char cursor_undefined = 1;
	if (cursor_undefined){
		cursor = XCreateFontCursor(qt_display, XC_xterm);
		QApplication::flushX();
		cursor_undefined = 0;
	}
	XDefineCursor(qt_display, win, cursor);
	QApplication::flushX();
}


void AllowInput()
{
	input_prohibited = false;

	/* reset the cursor from the watch for all forms and the canvas */
	QApplication::restoreOverrideCursor();

	QApplication::flushX();
}





void SmallUpdate()
{
	current_view->getScreen()->SmallUpdate();
	if (current_view->getScreen()->TopCursorVisible()
	    != current_view->getScreen()->first){
		current_view->currentBuffer()->updateFull();
		return;
	}

	current_view->fitCursor();
	current_view->updateScrollbar();

	if (!current_view->currentBuffer()->text->selection)
		current_view->currentBuffer()->text->sel_cursor =
			current_view->currentBuffer()->text->cursor;

	if (current_view->currentBuffer()->isLyxClean()) {
	  current_view->currentBuffer()->markDirty();
	  current_view->getMiniBuffer()->startTimer(1000*4);
	}
	else {
	  current_view->currentBuffer()->markDirty();
	}
}


//
// Menu callbacks
//

//
// File menu
//

// should be moved to lyxfunc.C
bool MenuWrite(Buffer* buf)
{
	QApplication::flushX();
	if (buf->getFileName() == i18n("---unnamed---")) {
	    return MenuWriteAs(buf);
	}
	
	bool result = bufferlist.write(buf);
	if (!result) {
		LString fname = buf->getFileName();
		LString s = MakeAbsPath(fname);
		if (AskQuestion(i18n("Save failed. Rename and try again?"),
				 MakeDisplayPath(s,50),
				 i18n("(If not, document is not saved.)"))) {
			return MenuWriteAs(buf);
		}
	} else {
		lastfiles->newFile(buf->getFileName());
	}
	return result;
}


// should be moved to BufferView.C
bool MenuWriteAs(Buffer *buffer)
{
	if (!buffer->text) return false;

	LString fname = buffer->getFileName();
	LString oldname = fname;

	if (!IsLyXFilename(fname))
		fname += ".lyx";

	fname = KFileDialog::getSaveFileName( OnlyPath( fname ).c_str(), 
					      i18n("*.lyx|Klyx Documents (*.lyx)\n"
						   "*|All Files"),
					      k_tlw,
					      i18n("Enter Filename to Save Document as") );

	if (fname.empty()) {
	    current_view->getMiniBuffer()->Set(i18n("Canceled."));
	    return false;
	}

	// Make sure the absolute filename ends with appropriate suffix
	LString s= MakeAbsPath(fname);
	if (!IsLyXFilename(s))
		s += ".lyx";

	// Same name as we have already?
	if (s == oldname) {
		if (!AskQuestion(i18n("Same name as document already has:"),
				 MakeDisplayPath(s,50),
				 i18n("Save anyway?")))
			return false;
		// Falls through to name change and save
	}
	// No, but do we have another file with this name open?
	else if (bufferlist.exists(s)) {
		if (AskQuestion(i18n("Another document with same name open!"),
				MakeDisplayPath(s,50),
				i18n("Replace with current document?")))
			{
				bufferlist.close(bufferlist.getBuffer(s));

				// Ok, change the name of the buffer, but don't save!
				buffer->setFileName(s);
				buffer->markDirty();

				current_view->getMiniBuffer()->Set(i18n("Document renamed to '"),
						MakeDisplayPath(s),
						i18n("', but not saved..."));
			}
		return false;
	} // Check whether the file exists
	else {
		FileInfo myfile(s);
		if (myfile.isOK() && !AskQuestion(i18n("Document already exists:"),
						  MakeDisplayPath(s,50),
						  i18n("Replace file?")))
			return false;
	}

	// Ok, change the name of the buffer
	buffer->setFileName(s);
	buffer->markDirty();
	// And save
	// Small bug: If the save fails, we have irreversible changed the name
	// of the document.
	return MenuWrite(buffer);
}


extern bool gsworking();

int MenuRunLaTeX(Buffer *buffer)
{
	int ret;

#warning Matthias believes this is not a problem
// 	if (gsworking()) {
// 		WriteAlert(i18n("Sorry, can't do this while pictures are being rendered."),
// 			   i18n("Please wait a few seconds for this to finish and try again."),
// 			   i18n("(or kill runaway gs processes by hand and try again.)"));
// 		return 1;
// 	}

	if (buffer->isLinuxDoc())
		ret = RunLinuxDoc(1, buffer->getFileName());
	else
		ret = buffer->runLaTeX();

	if (ret > 0) {
		LString s;
		LString t;
		if (ret == 1) {
			s = i18n("One error detected");
			t = i18n("You should try to fix it.");
		} else {
		        s += ret;
			s += i18n(" errors detected.");
			t = i18n("You should try to fix them.");
		}
		WriteAlert(i18n("There where errors during the LaTeX run."), s, t);
	}
	return ret;
}


int MenuRunChktex(Buffer *buffer)
{
	int ret;

	if (buffer->isLinuxDoc()) {
		WriteAlert(i18n("Chktex is no fun for LinuxDoc."));
		return 0;
	} else
		ret = buffer->runChktex();

	if (ret >= 0) {
		LString s;
		LString t;
		if (ret == 0) {
			s = i18n("No warnings found.");
		} else if (ret == 1) {
			s = i18n("One warning found.");
			t = i18n("Use 'Edit->Go to Error' to find it.");
		} else {
			s += ret;
			s += i18n(" warnings found.");
			t = i18n("Use 'Edit->Go to Error' to find them.");
		}
		WriteAlert(i18n("Chktex run successfully"), s, t);
	} else {
		WriteAlert(i18n("Error!"),i18n("It seems chktex does not work."));
	}
	return ret;
}


int MakeDVIOutput(Buffer *buffer)
{
	if (!(buffer->text))
		return 1;

	int ret = 0;

	LString path = OnlyPath(buffer->getFileName());
	if (lyxrc->use_tempdir || (IsDirWriteable(path) < 1)) {
		path = buffer->tmppath;
	}
	if (!buffer->isDviClean()) {
		PathPush(path);
		ret = MenuRunLaTeX(buffer);
		PathPop();
	}
	return ret;
}


/* wait == false means don't wait for termination */
/* wait == true means wait for termination       */
// The bool should be placed last on the argument line. (Lgb)
// Returns false if we fail.
bool RunScript(Buffer *buffer, bool wait,
	       LString const & command, LString const & orgname = LString(),
	       bool need_shell=true)
{
	LString path;
	LString cmd;
	LString name= orgname;
	
	if (MakeDVIOutput(buffer) > 0)
		return false;
	/* get DVI-Filename */
	if (name.empty())
		name = ChangeExtension(buffer->getFileName(),
				       ".dvi", true);

	path = OnlyPath(name);
	if (lyxrc->use_tempdir || (IsDirWriteable(path) < 1)) {
		path = buffer->tmppath;
	}
	PathPush(path);

	cmd = command + ' ' + SpaceLess(name);
	if (need_shell) {
		if (!wait)
			cmd += " &";
		
		current_view->getMiniBuffer()->Set(i18n("Executing command:"), cmd);
		Systemcalls one(Systemcalls::System, cmd);
	} else {
		current_view->getMiniBuffer()->Set(i18n("Executing command:"), cmd);
		Systemcalls one(wait ? Systemcalls::Wait
                                     : Systemcalls::DontWait, cmd);
	}
	PathPop();
	return true;
}


// Returns false if we fail
bool MenuRunDvips(Buffer *buffer, bool wait=false)
{
	if (!buffer->text)
		return false;

	ProhibitInput();

	// Generate dvi file
        if (MakeDVIOutput(buffer) > 0) {
            	AllowInput();
		return false;
        }
	// Generate postscript file
	LString ps = ChangeExtension (buffer->getFileName(),
				      ".ps_tmp", true);

	LString paper;
	
	char real_papersize = buffer->params.papersize;
	if (real_papersize == PAPER_DEFAULT)
	  real_papersize = lyxrc->default_papersize;

	switch (real_papersize) {
	case PAPER_USLETTER:
		paper = "letter";
		break;
	case PAPER_A3PAPER:
		paper = "a3";
		break;
	case PAPER_A4PAPER:
		paper = "a4";
		break;
	case PAPER_A5PAPER:
		paper = "a5";
		break;
	case PAPER_B5PAPER:
		paper = "b5";
		break;
	case PAPER_EXECUTIVEPAPER:
		paper = "foolscap";
		break;
	case PAPER_LEGALPAPER:
		paper = "legal";
		break;
	default: /* If nothing else fits... */
	        paper = "a4";
		break;
	}

	// Make postscript file.
	LString command = "dvips -o " + SpaceLess(ps);
	// dvips won't accept -t letter -t landscape.  In all other
	// cases, include the paper size explicitly.
	if (real_papersize != PAPER_USLETTER ||
	    buffer->params.orientation == ORIENTATION_PORTRAIT)
	  command += " -t " + paper;
	if (buffer->params.orientation == ORIENTATION_LANDSCAPE)
		command += " -t landscape";
	// push directorypath, if necessary
        LString path = OnlyPath(buffer->getFileName());
        if (lyxrc->use_tempdir || (IsDirWriteable(path) < 1)){
		path = buffer->tmppath;
        }
        PathPush(path);
	bool ret = RunScript(buffer, wait, command);
	AllowInput();
        PathPop();
	return ret;
}


// Returns false if we fail
bool MenuPreviewPS(Buffer *buffer)
{
	if (!buffer->text)
		return false;

	// Generate postscript file
	if (!MenuRunDvips(buffer, true)) {
		return false;
	}

	// Start postscript viewer
	ProhibitInput();
	LString ps = ChangeExtension (buffer->getFileName(),
				      ".ps_tmp", true);
	// push directorypath, if necessary
        LString path = OnlyPath(buffer->getFileName());
        if (lyxrc->use_tempdir || (IsDirWriteable(path) < 1)){
		path = buffer->tmppath;
        }
        PathPush(path);
	bool ret = RunScript(buffer, false, lyxrc->view_ps_command, ps);
        PathPop();
	AllowInput();
	return ret;
}


void MenuFax(Buffer *buffer)
{
	if (!buffer->text)
		return;

	// Generate postscript file
	if (!MenuRunDvips(buffer, true)) {
		return;
	}

	// Send fax
	LString ps = ChangeExtension (buffer->getFileName(), ".ps_tmp", true);
	LString path = OnlyPath (buffer->getFileName());
	if (lyxrc->use_tempdir || (IsDirWriteable(path) < 1)) {
		path = buffer->tmppath;
	}
	PathPush(path);
	if (!lyxrc->fax_program.empty()) {
		LString help2;
		char *ptr1, *ptr2;
		char *help = lyxrc->fax_program.copy();
		if ((ptr1=strstr(help,"$$FName"))) {
			ptr2=strchr(ptr1+1,'\'');
			*ptr1 = 0;
			help2 = help + ' ' + ps +' ' + (ptr2) + " &";
			Systemcalls one(Systemcalls::System, help2);
		} else {
			help2 = lyxrc->fax_program +" &";
			Systemcalls one(Systemcalls::System, help2);
		}
		delete [] help;
	} else
		send_fax(ps,lyxrc->fax_command);
	PathPop();
}


// Returns false if we fail
bool MenuPreview(Buffer *buffer)
{
	if (!buffer->text)
		return false;

	LString paper;
	
	char real_papersize = buffer->params.papersize;
	if (real_papersize == PAPER_DEFAULT)
	  real_papersize = lyxrc->default_papersize;

	switch (real_papersize) {
	case PAPER_USLETTER:
		paper = "us";
		break;
	case PAPER_A3PAPER:
		paper = "a3";
		break;
	case PAPER_A4PAPER:
		paper = "a4";
		break;
	case PAPER_A5PAPER:
		paper = "a5";
		break;
	case PAPER_B5PAPER:
		paper = "b5";
		break;
	case PAPER_EXECUTIVEPAPER:
		paper = "foolscap";
		break;
	case PAPER_LEGALPAPER:
		paper = "legal";
		break;
	default: /* If nothing else fits... */
	        paper = "a4";
		break;
	}

	LString rotate;

	if (buffer->params.orientation == ORIENTATION_LANDSCAPE)
		rotate ='r';

	// push directorypath, if necessary
        LString path = OnlyPath(buffer->getFileName());
        if (lyxrc->use_tempdir || (IsDirWriteable(path) < 1)){
		path = buffer->tmppath;
        }
        PathPush(path);
	// Run dvi-viewer
//	LString command = lyxrc->view_dvi_command + " -paper " + paper + rotate;
	LString command = lyxrc->view_dvi_command;
	bool ret = RunScript(buffer, false, command);
        PathPop();
	return ret;
}


void MenuMakeLaTeX(Buffer *buffer)
{
	if (buffer->text) {
		// Get LaTeX-Filename
		LString s = SpaceLess(ChangeExtension(
						buffer->getFileName(),
						".tex", false));
		
		KFileDialog* filedlg = new KFileDialog( 0, 
							i18n("*.tex|LaTex Files (*.tex)\n"
							     "*|All Files"),
							k_tlw, "", true );
		filedlg->setCaption( i18n( "Export LaTeX File" ) );
		filedlg->setSelection( s.c_str() );
		
		if( filedlg->exec() == QDialog::Accepted )
		  s = filedlg->selectedFile().data();
		else
		  return;

		if (buffer->isLinuxDoc())
			RunLinuxDoc(0, buffer->getFileName());
		else
			buffer->makeLaTeXFile(s, "", true);
		current_view->getMiniBuffer()->Set(i18n("Nice LaTeX file saved as"),
				MakeDisplayPath(s));
		buffer->markDviDirty();
	}
}


void MenuMakeLinuxDoc(Buffer *buffer)
{
	if (buffer->text) {

		if (!buffer->isLinuxDoc()) {
			WriteAlert(i18n("Error!"), i18n("Document class must be linuxdoc."));
			return;
		}


		// Get LinuxDoc-Filename
		LString s = ChangeExtension (buffer->getFileName(),
					     ".sgml", false);

		KFileDialog* filedlg = new KFileDialog( 0, 
							i18n("*.sgml|LinuxDoc SGML file (*.sgml)"),
							k_tlw, "", true );
		filedlg->setSelection( s.c_str() );
		if( filedlg->exec() == QDialog::Accepted )
		    s = filedlg->selectedFile().data();
		else
		    return;

		current_view->getMiniBuffer()->Set(i18n("Building LinuxDoc SGML file `"),
				MakeDisplayPath(s), "'...");

		buffer->makeLinuxDocFile(s, 65);
		current_view->getMiniBuffer()->Set(i18n("LinuxDoc SGML file"),
						   MakeDisplayPath(s), i18n("built"));
	}
}


void MenuMakeAscii(Buffer *buffer)
{
	if (buffer->text) {
		/* get LaTeX-Filename */
		LString s = ChangeExtension (buffer->getFileName(),
					     ".txt", false);

		KFileDialog* filedlg = new KFileDialog( 0, i18n("*.txt|Text Files"), k_tlw, "", true );
		filedlg->setCaption( i18n( "Export Text File" ) );
		filedlg->setSelection( s.c_str() );
		if( filedlg->exec() == QDialog::Accepted )
		  s = filedlg->selectedFile().data();
		else
		  return;


		buffer->writeFileAscii(s, lyxrc->ascii_linelen);

		current_view->getMiniBuffer()->Set(i18n("Ascii file saved as"), MakeDisplayPath(s));
	}
}




void QuitLyX()
{
	lyxerr.debug("Running QuitLyX.");

	if (!bufferlist.QwriteAll())
		return;

	lastfiles->writeFile(lyxrc->lastfiles);

	// Set a flag that we do quitting from the program,
	// so no refreshes are necessary.
	quitting = true;

	// close buffers first
	bufferlist.closeAll();

	// do any other cleanup procedures now
	lyxerr.debug("Deleting tmp dir " + system_tempdir);


	DestroyLyXTmpDir(system_tempdir);
	finished = true;

	// delete all toplevel widgets (will destroy lyxviews as well)
	deleteAllToplevelWidgets();
	
	kapp->quit();
}



void AutoSave()
	// should probably be moved into BufferList (Lgb)
	// Perfect target for a thread...
{
	if (!current_view->getScreen() || !current_view->available())
		return;
	
	if (current_view->currentBuffer()->isBakClean()
		|| current_view->currentBuffer()->isReadonly())
		return;
	
	LString fname;
	
	current_view->getMiniBuffer()->Set(i18n("Autosaving current document..."));
	
	// create autosave filename
	fname =	OnlyPath(current_view->currentBuffer()->getFileName());
	fname += "#";
	fname += OnlyFilename(current_view->currentBuffer()->getFileName());
	fname += "#";
	
	// tmp_ret will be located (usually) in /tmp
	// will that we a problem?
	LString tmp_ret = tmpnam(NULL);
	
	signal(SIGCHLD, sigchldhandler);
	
	pid_t pid = fork(); // If you want to debug the autosave
	// you should set pid to -1, and comment out the
	// fork.
	if (pid == 0 || pid == -1) {
		// pid = -1 signifies that lyx was unable
		// to fork. But we will do the save
		// anyway.
		bool failed = false;
		if (!tmp_ret.empty()) {
			current_view->currentBuffer()->writeFile(tmp_ret, 1);
			// assume successful write of tmp_ret
			if (rename(tmp_ret.c_str(), fname.c_str()) == -1) {
				failed = true;
				// most likely couldn't move between filesystems
				// unless write of tmp_ret failed
				// so remove tmp file (if it exists)
				remove(tmp_ret.c_str());
			}
		} else {
			failed = true;
		}
		
		if (failed) {
			// failed to write/rename tmp_ret so try writing direct
			if (!current_view->currentBuffer()->writeFile(fname,
								      1)) {
				current_view->getMiniBuffer()->Set(i18n("Autosave Failed!"));
			}
		}
		if (pid == 0){
			_exit(0);
		}
	}
	
	current_view->currentBuffer()->markBakClean();
// 	current_view->getOwner()->resetAutosaveTimer();
}


//
// (c) CHT Software Service GmbH
// Uwe C. Schroeder
//
// create new file with template
// SERVERCMD !
//
void NewLyxFile(LString const & filename)
{
	// Split argument by :
	LString tmpname = filename;
	LString name;
	tmpname.split(name, ':');

	lyxerr.print("Arg is " + filename);
	lyxerr.print("Name is " + name);
	lyxerr.print("Template is " + tmpname);

	// find a free buffer
	bufferlist.newFile(name,tmpname);
	lastfiles->newFile(current_view->currentBuffer()->getFileName());
}


// Insert ascii file
void InsertAsciiFile(bool asParagraph)
{
	LString fname;
	LyXParagraph *tmppar;

	if (!current_view->getScreen()) return;

	fname = KFileDialog::getSaveFileName( current_view->getOwner()->currentBuffer()->filepath.c_str(),
					      i18n("*|All Files"), k_tlw,
					      i18n("File to Insert") );
	
	if (fname.empty()) return;
	
	FileInfo fi(fname);
 	FilePtr myfile(fname, FilePtr::read);
	
 	if (!fi.exist() || !fi.readable() || !myfile()) {
	    WriteFSAlert(i18n("Error! Cannot open specified file:"),
			 MakeDisplayPath(fname,50));
	    return;
	}
	
	tmppar = new LyXParagraph();
	tmppar->readSimpleWholeFile(myfile);
	
	// set the end of the string
	tmppar->InsertChar(tmppar->last-1,'\0');

	// insert the string
	current_view->getScreen()->HideCursor(true);

	if (!asParagraph)
		current_view->currentBuffer()->text->InsertStringA(tmppar->text);
	else
		current_view->currentBuffer()->text->InsertStringB(tmppar->text);
	delete tmppar;
	current_view->currentBuffer()->updateFull();
}


void MenuShowTableOfContents()
{
  if( !klyxdialogs->toc )
    klyxdialogs->toc =
      new TableOfContentsDialog(current_view->getOwner());

  klyxdialogs->toc->updateToc();

  if( klyxdialogs->toc->isVisible() )
	klyxdialogs->toc->raise();
  else
	klyxdialogs->toc->show();
}


void MenuInsertLabel()
{
    if( ! current_view->available() )
	return;
    if( !klyxdialogs->insertlabel )
	klyxdialogs->insertlabel = new InsertLabelDialog( current_view->getOwner()->toplevelWidget() );
    QString prefix( i18n( "sec: " ) );
    if( current_view->currentBuffer()->text->cursor.par->footnoteflag==LyXParagraph::OPEN_FOOTNOTE ) {
	prefix = i18n( "fig: " );
	if( current_view->currentBuffer()->text->cursor.par->footnotekind==LyXParagraph::FIG ){
	} else if( current_view->currentBuffer()->text->cursor.par->footnotekind==LyXParagraph::TAB ) {
	    prefix = i18n( "tab: " );
	}
    }
    klyxdialogs->insertlabel->setLabel( prefix );
    klyxdialogs->insertlabel->setFocus();
    if( QDialog::Accepted == klyxdialogs->insertlabel->exec() ) {
	LString label = klyxdialogs->insertlabel->getLabel();
	label.strip();
	label.frontStrip();
	if( ! label.empty() ) {
	    InsetLabel *new_inset = new InsetLabel;
	    new_inset->setContents( label );
	    current_view->currentBuffer()->insertInset( new_inset );
	}
    }
}


void MenuInsertRef()
{
  if( !klyxdialogs->insertxref )
	klyxdialogs->insertxref = new InsertCrossReferenceDialog( current_view->getOwner() );

  klyxdialogs->insertxref->updateXRefs();

  if( klyxdialogs->insertxref->isVisible() )
	klyxdialogs->insertxref->raise();
  else
	klyxdialogs->insertxref->show();
}


void FootCB()
{
	if (!current_view->available())
		return;
	
	current_view->getMiniBuffer()->Set(i18n("Inserting Footnote..."));
	current_view->getScreen()->HideCursor();
	current_view->currentBuffer()->updateFull(false);
	current_view->currentBuffer()->text->InsertFootnoteEnvironment(LyXParagraph::FOOTNOTE);
	current_view->currentBuffer()->updateFull();
}


void LayoutsCB(int sel, void *)
{
fprintf( stderr, "LayoutsCB\n" );
	LString tmp;
	tmp += sel;
	current_view->getOwner()->getLyXFunc()->Dispatch(LFUN_LAYOUTNO,
							 tmp.c_str());
}


/*
 * SGML support:
 * (flag == -1) import SGML file
 * (flag == 0) make TeX output
 * (flag == 1) make dvi output
 */
int RunLinuxDoc(int flag, LString const & filename)
{
	LString name;
	LString s2;
	LString path;
	LString add_flags;

	int errorcode = 0;

	/* generate a path-less extension name */
	name = ChangeExtension (filename, ".sgml", true);
	path = OnlyPath (filename);
	if (lyxrc->use_tempdir || (IsDirWriteable(path) < 1)) {
		path = current_view->currentBuffer()->tmppath;
	}
	PathPush (path);
	
	if (flag != -1) {
		if (!current_view->available())
			return 0;
		current_view->currentBuffer()->makeLinuxDocFile(name,0);
		LYX_PAPER_SIZE ps = (LYX_PAPER_SIZE) current_view->currentBuffer()->params.papersize;
		switch (ps) {
		case PAPER_A4PAPER:  add_flags = "-p a4";     break;
		case PAPER_USLETTER: add_flags = "-p letter"; break;
		default: /* nothing to be done yet ;-) */     break;
		}
	}
	
	ProhibitInput();
	
	Systemcalls one;
	switch (flag) {
	case -1: /* Import file */
		current_view->getMiniBuffer()->Set(i18n("Importing LinuxDoc SGML file `"),
				MakeDisplayPath(filename), "'...");
		s2 = "sgml2lyx " + lyxrc->sgml_extra_options + ' '
			+ name;
		if (one.Startscript(Systemcalls::System, s2))
			errorcode = 1;
		break;
	case 0: /* TeX output asked */
		current_view->getMiniBuffer()->Set(i18n("Converting LinuxDoc SGML to TeX file..."));
 		s2 = "sgml2latex " + add_flags + " -o tex "
			+ lyxrc->sgml_extra_options + ' ' + name;
		if (one.Startscript(Systemcalls::System, s2))
			errorcode = 1;
		break;
	case 1: /* dvi output asked */
		current_view->getMiniBuffer()->Set(i18n("Converting LinuxDoc SGML to dvi file..."));
 		s2 = "sgml2latex " + add_flags + " -o dvi "
			+ lyxrc->sgml_extra_options + ' ' + name;
		if (one.Startscript(Systemcalls::System, s2)) {
			errorcode = 1;
		} else
			current_view->currentBuffer()->markDviClean();
		break;
	default: /* unknown output */
		break;
	}
	
	PathPop();
	AllowInput();
	return errorcode;
}


void AllFloats(char flag, char figmar)
{
	if (!current_view->available())
		return;

	LyXCursor cursor = current_view->currentBuffer()->text->cursor;

	if (!flag && cursor.par->footnoteflag != LyXParagraph::NO_FOOTNOTE
	    && ((figmar
		 && cursor.par->footnotekind != LyXParagraph::FOOTNOTE
		 && cursor.par->footnotekind != LyXParagraph::MARGIN)
		|| (!figmar
		    && cursor.par->footnotekind != LyXParagraph::FIG
		    && cursor.par->footnotekind != LyXParagraph::TAB
 		    && cursor.par->footnotekind != LyXParagraph::WIDE_FIG
 		    && cursor.par->footnotekind != LyXParagraph::WIDE_TAB
		    && cursor.par->footnotekind != LyXParagraph::ALGORITHM)))
		ToggleFloat();
	else
	  current_view->getScreen()->HideCursor(true);

	LyXCursor tmpcursor = cursor;
	cursor.par = tmpcursor.par->ParFromPos(tmpcursor.pos);
	cursor.pos = tmpcursor.par->PositionInParFromPos(tmpcursor.pos);

	LyXParagraph *par = current_view->currentBuffer()->paragraph;
	while (par) {
		if (flag) {
			if (par->footnoteflag == LyXParagraph::CLOSED_FOOTNOTE
			    && (
				    (figmar
				     &&
				     par->footnotekind != LyXParagraph::FOOTNOTE
				     &&
				     par->footnotekind !=  LyXParagraph::MARGIN
					    )
				    ||
				    (!figmar
				     &&
				     par->footnotekind != LyXParagraph::FIG
				     &&
				     par->footnotekind != LyXParagraph::TAB
				     &&
 				     par->footnotekind != LyXParagraph::WIDE_FIG
 				     &&
 				     par->footnotekind != LyXParagraph::WIDE_TAB
 				     &&
				     par->footnotekind != LyXParagraph::ALGORITHM
					    )
				    )
				){
				if (par->previous
				    && par->previous->footnoteflag !=
				    LyXParagraph::CLOSED_FOOTNOTE){ /* should be */
					current_view->currentBuffer()->text->SetCursorIntern(par->previous,
								      0);
					current_view->currentBuffer()->text->OpenFootnote();
				}
			}
		}
		else  {
			if (par->footnoteflag == LyXParagraph::OPEN_FOOTNOTE
			    && (
				    (figmar
				     &&
				     par->footnotekind != LyXParagraph::FOOTNOTE
				     &&
				     par->footnotekind !=  LyXParagraph::MARGIN
					    )
				    ||
				    (!figmar
				     &&
				     par->footnotekind != LyXParagraph::FIG
				     &&
				     par->footnotekind != LyXParagraph::TAB
				     &&
 				     par->footnotekind != LyXParagraph::WIDE_FIG
 				     &&
 				     par->footnotekind != LyXParagraph::WIDE_TAB
 				     &&
				     par->footnotekind != LyXParagraph::ALGORITHM
					    )
				    )
				){
				current_view->currentBuffer()->text->SetCursorIntern(par, 0);
				current_view->currentBuffer()->text->CloseFootnote();
			}
		}
		par = par->next;
	}

	current_view->currentBuffer()->text->SetCursorIntern(cursor.par, cursor.pos);
	current_view->redraw();
	current_view->fitCursor();
	current_view->updateScrollbar();
}


void MenuLayoutCharacter() // ported to KDE, KD 97-12-14
{
  if( klyxdialogs->character == 0 )
	  // it would be nicer if the parent was k_tlw, but this won't work
	  // with modeless dialogs (Kalle, 14.12.97)
	  klyxdialogs->character = new CharacterStyleDialog;

  if( klyxdialogs->character->isVisible() )
	klyxdialogs->character->raise();
  else
	klyxdialogs->character->show();
}


void MenuLayoutParagraph() // ported to KDE, KD 97-11-27
{
  // was: XForms form fd_form_paragraph

  if (!current_view->getScreen() || !current_view->available())
    return;

  Buffer * buf = current_view->currentBuffer();

  if( klyxdialogs->paragraph == 0 )
	{
	  // it would be nicer if the parent was k_tlw, but this won't work
	  // with modeless dialogs (Kalle, 27.11.97)
	  klyxdialogs->paragraph = new ParagraphDialog;

	  klyxdialogs->paragraphreceiver = new ParagraphDialogReceiver( klyxdialogs->paragraph );
	}

  ParagraphDialogData* dlg_data = klyxdialogs->paragraph->data();

  int align = buf->text->cursor.par->GetAlign();
  if (align == LYX_ALIGN_LAYOUT)
    align =	lyxstyle.Style(buf->params.textclass,
			       buf->text->cursor.par->GetLayout())->align;

  switch( align )
    {
    case LYX_ALIGN_RIGHT:
      dlg_data->alignment = ParagraphDialogData::Right;
      break;
    case LYX_ALIGN_LEFT:
      dlg_data->alignment = ParagraphDialogData::Left;
      break;
    case LYX_ALIGN_CENTER:
      dlg_data->alignment = ParagraphDialogData::Center;
      break;
    default:
      dlg_data->alignment = ParagraphDialogData::Block;
    };

  dlg_data->lines_above = buf->text->cursor.par->FirstPhysicalPar()->line_top;
  dlg_data->lines_below = buf->text->cursor.par->FirstPhysicalPar()->line_bottom;
  dlg_data->pagebreaks_above = buf->text->cursor.par->FirstPhysicalPar()->pagebreak_top;
  dlg_data->pagebreaks_below = buf->text->cursor.par->FirstPhysicalPar()->pagebreak_bottom;

  dlg_data->space_above = "";
  switch (buf->text->cursor.par->FirstPhysicalPar()->added_space_top.kind())
    {
    case VSpace::NONE:
      dlg_data->vertical_spaces_above = ParagraphDialogData::VS_None;
      break;
    case VSpace::DEFSKIP:
      dlg_data->vertical_spaces_above = ParagraphDialogData::VS_Defskip;
      break;
    case VSpace::SMALLSKIP:
      dlg_data->vertical_spaces_above = ParagraphDialogData::VS_Smallskip;
      break;
    case VSpace::MEDSKIP:
      dlg_data->vertical_spaces_above = ParagraphDialogData::VS_Medskip;
      break;
    case VSpace::BIGSKIP:
      dlg_data->vertical_spaces_above = ParagraphDialogData::VS_Bigskip;
      break;
    case VSpace::VFILL:
      dlg_data->vertical_spaces_above = ParagraphDialogData::VS_Vfill;
      break;
    case VSpace::LENGTH:
      dlg_data->vertical_spaces_above = ParagraphDialogData::VS_Length;
      dlg_data->space_above = buf->text->cursor.par->FirstPhysicalPar()->added_space_top.length().asString().c_str();
      break;
    }
  dlg_data->space_keep_above = buf->text->cursor.par->FirstPhysicalPar()->added_space_top.keep();

  dlg_data->space_below = "";
  switch (buf->text->cursor.par->FirstPhysicalPar()->added_space_bottom.kind())
    {
    case VSpace::NONE:
      dlg_data->vertical_spaces_below = ParagraphDialogData::VS_None;
      break;
    case VSpace::DEFSKIP:
      dlg_data->vertical_spaces_below = ParagraphDialogData::VS_Defskip;
      break;
    case VSpace::SMALLSKIP:
      dlg_data->vertical_spaces_below = ParagraphDialogData::VS_Smallskip;
      break;
    case VSpace::MEDSKIP:
      dlg_data->vertical_spaces_below = ParagraphDialogData::VS_Medskip;
      break;
    case VSpace::BIGSKIP:
      dlg_data->vertical_spaces_below = ParagraphDialogData::VS_Bigskip;
      break;
    case VSpace::VFILL:
      dlg_data->vertical_spaces_below = ParagraphDialogData::VS_Vfill;
      break;
    case VSpace::LENGTH:
      dlg_data->vertical_spaces_below = ParagraphDialogData::VS_Length;
      dlg_data->space_below = buf->text->cursor.par->FirstPhysicalPar()->added_space_bottom.length().asString().c_str();
      break;
    }
  dlg_data->space_keep_below = buf->text->cursor.par->FirstPhysicalPar()->added_space_bottom.keep();

  dlg_data->noindent = buf->text->cursor.par->FirstPhysicalPar()->noindent;
  dlg_data->label_width = buf->text->cursor.par->GetLabelWidthString().c_str();

  dlg_data->spacing_length = "";
  switch (buf->text->cursor.par->FirstPhysicalPar()->spacing.getSpace()) {
  case Spacing::LayoutDefault:
    dlg_data->spacing = ParagraphDialogData::LayoutDefault;
    break;
  case Spacing::Single:
    dlg_data->spacing = ParagraphDialogData::Single;
    break;
  case Spacing::Onehalf:
    dlg_data->spacing = ParagraphDialogData::Onehalf;
    break;
  case Spacing::Double:
    dlg_data->spacing = ParagraphDialogData::Double;
    break;
  case Spacing::Other:
    dlg_data->spacing = ParagraphDialogData::Other;
    dlg_data->spacing_length.setNum(buf->text->cursor.par->FirstPhysicalPar()->spacing.getValue());
    break;
  }

  klyxdialogs->paragraph->setData( dlg_data );

  if( klyxdialogs->paragraph->isVisible() )
    klyxdialogs->paragraph->raise();
  else
    klyxdialogs->paragraph->show();
}


inline
void DeactivateDocumentButtons () // ported to KDE, KD 97-12-05
{
  klyxdialogs->document->deactivateButtons();
}


inline
void ActivateDocumentButtons () // ported to KDE, KD 97-12-05
{
  klyxdialogs->document->activateButtons();
}

inline
void DisableDocumentLayout () // ported to KDE, KD 97-12-05
{
  DeactivateDocumentButtons ();
}

inline
void EnableDocumentLayout () // ported to KDE, KD 97-12-05
{
  ActivateDocumentButtons ();
}

void MenuLayoutDocument() // ported to KDE, KD 97-12-05
{
  if (!current_view->getScreen() || !current_view->available())
	return;

  if( klyxdialogs->document == 0 )
	{
	  // it would be nicer if the parent was k_tlw, but this won't work
	  // with modeless dialogs (Kalle, 27.11.97)
	  klyxdialogs->document = new DocumentLayoutDialog(current_view->getOwner()->toplevelWidget() );
	  klyxdialogs->documentreceiver = new DocumentLayoutDialogReceiver( klyxdialogs->document );
	}

  EnableDocumentLayout();
  BufferParams const& params = current_view->currentBuffer()->params;

  UpdateLayoutDocument( params );

	if( klyxdialogs->document->isVisible() )
	  klyxdialogs->document->raise();
	else
	  klyxdialogs->document->show();
}


void UpdateLayoutDocument( BufferParams const &params )
{
  LyXTextClass *tclass = lyxstyle.TextClass(params.textclass);

  DocumentLayoutDialogData* dlg_data = klyxdialogs->document->data();
  dlg_data->_class =
	klyxdialogs->document->stringToClass( lyxstyle.DescOfClass(
															  params.textclass).c_str() );
  dlg_data->language =
	klyxdialogs->document->stringToLanguage( params.language.c_str() );
  dlg_data->fonts =
	klyxdialogs->document->stringToFonts( params.fonts.c_str() );
  dlg_data->encoding =
	klyxdialogs->document->stringToEnc( params.inputenc.c_str() );
  dlg_data->ps_driver =
	klyxdialogs->document->stringToPSDriver( params.graphicsDriver.c_str() );

        // ale970405+lasgoutt970513
  klyxdialogs->document->fontsizeCombo()->clear();
  klyxdialogs->document->fontsizeCombo()->insertItem( "default" );
  QkString tempstr = tclass->opt_fontsize.c_str();
  for( unsigned int i = 0; i < tempstr.tokenCount( '|' ); i++ )
	klyxdialogs->document->fontsizeCombo()->insertItem( tempstr.getToken( i, '|' ) );
  dlg_data->font_size = static_cast< DocumentLayoutDialogData::FontSize >( tclass->opt_fontsize.tokenPos( '|', params.fontsize ) + 1 );

        // ale970405+lasgoutt970513
  klyxdialogs->document->pagestyleCombo()->clear();
  klyxdialogs->document->pagestyleCombo()->insertItem( "default" );
  tempstr = tclass->opt_pagestyle.c_str();
  for( unsigned int i = 0; i < tempstr.tokenCount( '|' ); i++ )
	klyxdialogs->document->pagestyleCombo()->insertItem( tempstr.getToken( i, '|' ) );
  dlg_data->pagestyle = static_cast< DocumentLayoutDialogData::Pagestyle >( tclass->opt_pagestyle.tokenPos('|', params.pagestyle) + 1 );

  // FIXME: This button does not exist?! (at least it is not visible)
  //  fl_set_button(fd_form_document->check_use_amsmath, params.use_amsmath);

	if (params.paragraph_separation == LYX_PARSEP_INDENT)
	  dlg_data->separation = DocumentLayoutDialogData::Indent;
	else
	  dlg_data->separation = DocumentLayoutDialogData::Skip;

	dlg_data->amsmath = params.use_amsmath;

	switch (params.getDefSkip().kind()) {
	case VSpace::SMALLSKIP:
	  dlg_data->default_skip = DocumentLayoutDialogData::Smallskip;
	  break;
	case VSpace::MEDSKIP:
	  dlg_data->default_skip = DocumentLayoutDialogData::Medskip;
	  break;
	case VSpace::BIGSKIP:
	  dlg_data->default_skip = DocumentLayoutDialogData::Bigskip;
	  break;
	case VSpace::LENGTH:
	  dlg_data->default_skip = DocumentLayoutDialogData::Length;
	  dlg_data->default_skip_length = params.getDefSkip().asLyXCommand().c_str();
	  break;
	default:
	  dlg_data->default_skip = DocumentLayoutDialogData::Medskip;
	  break;
	}
	
	if (params.sides == 2)
	  dlg_data->sides = DocumentLayoutDialogData::TwoSides;
	else
	  dlg_data->sides = DocumentLayoutDialogData::OneSide;

	if (params.columns == 2)
	  dlg_data->columns = DocumentLayoutDialogData::TwoColumns;
	else
	  dlg_data->columns = DocumentLayoutDialogData::OneColumn;

	dlg_data->spacing_length = "";
	switch (params.spacing.getSpace())
	  {
	  case Spacing::Single:
		{
		  // \singlespacing
		  dlg_data->spacing = DocumentLayoutDialogData::Single;
		  break;
		}
	  case Spacing::Onehalf:
		{
		  // \onehalfspacing
		  dlg_data->spacing = DocumentLayoutDialogData::Onehalf;
		  break;
		}
	  case Spacing::Double:
		{
		  // \ doublespacing
		  dlg_data->spacing = DocumentLayoutDialogData::Double;
		  break;
		}
	  case Spacing::Other:
		{
		  dlg_data->spacing = DocumentLayoutDialogData::Other;
		  dlg_data->spacing_length.setNum(params.spacing.getValue());
		  break;
		}
	  default:
	    break;
	  }
	
	dlg_data->section_number_depth = params.secnumdepth;
	dlg_data->table_of_contents_depth = params.tocdepth;

	if (!params.float_placement.empty())
	  { // buffer local (Lgb)
		dlg_data->float_placement = params.float_placement.c_str();
	  }
	else
	  {
		dlg_data->float_placement = "";
	  }

	if (!params.options.empty())
	  dlg_data->extra_options = params.options.c_str();
	else
	  dlg_data->extra_options = "";

	if (current_view->currentBuffer()->isLinuxDoc())
	  {
		// bullets not used in LinuxDoc
		klyxdialogs->document->getBulletsButton()->setEnabled( false );
	  }
	else
	  {
		klyxdialogs->document->getBulletsButton()->setEnabled( true );
	  }

	if (current_view->currentBuffer()->isReadonly()) {
	  DisableDocumentLayout();
	  WriteAlert(i18n("Any changes will be ignored"),
				 i18n("The document is read-only:"),
				 current_view->currentBuffer()->getFileName());
	}
	
	klyxdialogs->document->setData( dlg_data );
}

void MenuLayoutQuotes()
{
	if (!current_view->getScreen() || !current_view->available())
		return;

	if( !klyxdialogs->quotes )
	  klyxdialogs->quotes = new QuotesDialog( current_view->getOwner() );

	klyxdialogs->quotes->setQuotes( current_view->currentBuffer()->params.quotes_language );
	
	if( current_view->currentBuffer()->params.quotes_times ==
		InsetQuotes::SingleQ )
	  klyxdialogs->quotes->setSingle();
	else
	  klyxdialogs->quotes->setDouble();

	(void)klyxdialogs->quotes->exec();
}


void MenuLayoutPreamble()
{
	if (!current_view->getScreen() || ! current_view->available())
	    return;

	if( !klyxdialogs->latexpreamble )
	  klyxdialogs->latexpreamble = new LaTeXPreambleDialog( current_view->getOwner () );


	klyxdialogs->latexpreamble->setPreamble( current_view->currentBuffer()->params.preamble.c_str() );

	if (current_view->currentBuffer()->isReadonly()) {
	  klyxdialogs->latexpreamble->deactivate();
	}
	else {
	  klyxdialogs->latexpreamble->activate();
	}
	
	(void)klyxdialogs->latexpreamble->exec();
}

void MenuLayoutSave()
{
	if (!current_view->getScreen() || ! current_view->available())
	    return;

	if (AskQuestion(i18n("Do you want to save the current settings\nfor Character, Document, Paper and Quotes\nas default for new documents?")))
		current_view->currentBuffer()->saveParamsAsDefaults();
}

/* callbacks added for LinuxDoc support (Pascal Andr) :
 *   - marks and cross references TeX macros
 *   - url display macros
 *   - htmlurl macros
 */

/* inserts an inset (used for LinuxDoc SGML related callbacks) */
void InsertLatexDelInset(LString const & macro)
{
	InsetLatexDel *new_inset = new InsetLatexDel(macro);
	current_view->currentBuffer()->insertInset(new_inset);
	//current_view->currentBuffer()->updateFull();
}


/* inserts an url skeleton */
void UrlCB()
{
	InsertLatexDelInset("\\url{");
	InsertLatexDelInset("}{");
	InsertLatexDelInset("}");
	current_view->currentBuffer()->text->InsertChar(' ');
}


/* inserts an html url skeleton */
void HtmlUrlCB()
{
	InsertLatexDelInset("\\htmlurl{");
	InsertLatexDelInset("}{");
	InsertLatexDelInset("}");
	current_view->currentBuffer()->text->InsertChar(' ');
}

/* end of LinuxDoc SGML special CB */

void NoteCB()
{
	LString string;
	LString userName(getUserName());
	string = userName + ' ' + date();
	InsetInfo *new_inset = new InsetInfo(string);
	current_view->currentBuffer()->insertInset(new_inset);
	//current_view->currentBuffer()->updateFull();
}


void OpenStuff()
{
	if (current_view->available()) {
		current_view->getMiniBuffer()->Set(i18n("Open/Close..."));
		current_view->getScreen()->HideCursor(true);
		current_view->currentBuffer()->updateFull(false);
		current_view->currentBuffer()->text->OpenStuff();
		current_view->currentBuffer()->updateFull(false);
	}
}

void ToggleFloat()
{
	if (current_view->available()) {
		current_view->getMiniBuffer()->Set(i18n("Open/Close..."));
		current_view->getScreen()->HideCursor(true);
		current_view->currentBuffer()->updateFull(false);
		current_view->currentBuffer()->text->ToggleFootnote();
		current_view->currentBuffer()->updateFull(false);
	}
}


void MenuUndo()
{
/*	if (current_view->currentBuffer()->the_locking_inset) {
		current_view->getMiniBuffer()->Set(i18n("Undo not yet supported in math mode"));
		return;
	}*/

	if (current_view->available()) {
		current_view->getMiniBuffer()->Set(i18n("Undo"));
		current_view->getScreen()->HideCursor(true);
		current_view->currentBuffer()->updateFull(false);
		if (!current_view->currentBuffer()->text->TextUndo())
			current_view->getMiniBuffer()->Set(i18n("No further undo information"));
		else
			current_view->currentBuffer()->updateFull();
	}
}


void MenuRedo()
{
	if (current_view->currentBuffer()->the_locking_inset) {
		current_view->getMiniBuffer()->Set(i18n("Redo not yet supported in math mode"));
		return;
	}

	if (current_view->available()) {
		current_view->getMiniBuffer()->Set(i18n("Redo"));
		current_view->getScreen()->HideCursor(true);
		current_view->currentBuffer()->updateFull(false);
		if (!current_view->currentBuffer()->text->TextRedo())
			current_view->getMiniBuffer()->Set(i18n("No further redo information"));
		else
			current_view->currentBuffer()->updateFull();
	}
}


void HyphenationPoint()
{
	if (current_view->available())  {
		current_view->getScreen()->HideCursor();
		current_view->currentBuffer()->updateFull(false);
		InsetSpecialChar *new_inset =
			new InsetSpecialChar(InsetSpecialChar::HYPHENATION);
		current_view->currentBuffer()->insertInset(new_inset);
		//current_view->currentBuffer()->updateFull();
	}
}


void Ldots()
{
	if (current_view->available())  {
		current_view->getScreen()->HideCursor();
		current_view->currentBuffer()->updateFull(false);
		InsetSpecialChar *new_inset =
			new InsetSpecialChar(InsetSpecialChar::LDOTS);
		current_view->currentBuffer()->insertInset(new_inset);
	}
}


void EndOfSentenceDot()
{
	if (current_view->available())  {
		current_view->getScreen()->HideCursor();
		current_view->currentBuffer()->updateFull(false);
		InsetSpecialChar *new_inset =
			new InsetSpecialChar(InsetSpecialChar::END_OF_SENTENCE);
		current_view->currentBuffer()->insertInset(new_inset);
	}
}


void Newline()
{
	if (current_view->available())  {
		current_view->getScreen()->HideCursor();
		current_view->currentBuffer()->updateFull(false);
		current_view->currentBuffer()->text->InsertChar(LYX_META_NEWLINE);
		current_view->currentBuffer()->updateFull();
	}
}


void ProtectedBlank()
{
	if (current_view->available())  {
		current_view->getScreen()->HideCursor();
		current_view->currentBuffer()->updateFull(false);
		current_view->currentBuffer()->text->InsertChar(LYX_META_PROTECTED_SEPARATOR);
		current_view->currentBuffer()->updateFull();
	}
}


void HFill()
{
	if (current_view->available())  {
		current_view->getScreen()->HideCursor();
		current_view->currentBuffer()->updateFull(false);
		current_view->currentBuffer()->text->InsertChar(LYX_META_HFILL);
		current_view->currentBuffer()->updateFull();
	}
}


/* -------> These CB's use ToggleFree() as the (one and only?) font-changer.
			They also show the current font state. */


void ToggleAndShow(LyXFont const &);


void FontSizeCB(LString const & size)
{
	LyXFont font(LyXFont::ALL_IGNORE);
	font.setGUISize(size);
	ToggleAndShow(font);
}


void EmphCB()
{
	LyXFont font(LyXFont::ALL_IGNORE);
	font.setEmph(LyXFont::ON);
	ToggleAndShow(font);
}


void NounCB()
{
	LyXFont font(LyXFont::ALL_IGNORE);
	font.setNoun(LyXFont::ON);
	ToggleAndShow( font);
}


void BoldCB()
{
	LyXFont font(LyXFont::ALL_IGNORE);
	font.setSeries(LyXFont::BOLD_SERIES);
	ToggleAndShow( font);
}


void UnderlineCB()
{
	LyXFont font(LyXFont::ALL_IGNORE);
	font.setUnderbar(LyXFont::ON);
	ToggleAndShow( font);
}


void CodeCB()
{
	LyXFont font(LyXFont::ALL_IGNORE);
	font.setFamily(LyXFont::TYPEWRITER_FAMILY); // no good
	ToggleAndShow( font);
}


void SansCB()
{
	LyXFont font(LyXFont::ALL_IGNORE);
	font.setFamily(LyXFont::SANS_FAMILY);
	ToggleAndShow( font);
}


void RomanCB()
{
	LyXFont font(LyXFont::ALL_IGNORE);
	font.setFamily(LyXFont::ROMAN_FAMILY);
	ToggleAndShow( font);
}


void TexCB()
{
	LyXFont font(LyXFont::ALL_IGNORE);
	font.setLatex (LyXFont::ON);
	ToggleAndShow( font);
}


void StyleResetCB()
{
	LyXFont font(LyXFont::ALL_INHERIT);
	ToggleAndShow( font);
}


/* -------> Returns the current font and depth by printing a message. In the
 * future perhaps we could try to implement a callback to the button-bar.
 * That is, `light' the bold button when the font is currently bold, etc.
 */
LString CurrentState()
{
	LString state;
	if (current_view->available()) {
		// I think we should only show changes from the default
		// font. (Asger)
		Buffer * buffer = current_view->currentBuffer();
		LyXFont font = buffer->text->real_current_font;
		LyXFont defaultfont = lyxstyle.TextClass(buffer->
							 params.textclass)->defaultfont;
		font.reduce(defaultfont);
		state = i18n("Font: ") + font.stateText();

		int depth = buffer->text->GetDepth();
		if (depth>0)
			state += LString(i18n(", Depth: ")) + depth;
	}
	return state;
}


/* -------> Does the actual toggle job of the XxxCB() calls above.
 * Also shows the current font state.
 */
void ToggleAndShow(LyXFont const & font)
{
	if (current_view->available()) {
		current_view->getScreen()->HideCursor();
		current_view->currentBuffer()->updateFull(false);
 		current_view->currentBuffer()->text->ToggleFree(font);
		current_view->currentBuffer()->updateFull();
	}
 	// removed since it overrides the ToggleFree Message about the style
 	// Since Styles are more "High Level" than raw fonts I think the user
 	// prefers it like this               Matthias
 	// FontStateShowCB( 0, 0 );
}


void MarginCB()
{
	if (current_view->available()) {
		current_view->getMiniBuffer()->Set(i18n("Inserting margin note..."));
		current_view->getScreen()->HideCursor();
		current_view->currentBuffer()->updateFull(false);
		current_view->currentBuffer()->text->InsertFootnoteEnvironment(LyXParagraph::MARGIN);
		current_view->currentBuffer()->updateFull();
	}
}


void FigureCB() // ported to KDE, KD 97-12-04
{
  if( !klyxdialogs->insertfigure )
	klyxdialogs->insertfigure = new InsertFigureDialog( k_tlw );
  int ret = klyxdialogs->insertfigure->exec();
  if( ret == QDialog::Accepted )
	FigureOKCB();
}


void TableCB() // ported to KDE, KD 97-12-04
{
  if( !klyxdialogs->inserttable )
	klyxdialogs->inserttable = new InsertTableDialog( k_tlw );
  int ret = klyxdialogs->inserttable->exec();
  if( ret == QDialog::Accepted )
	TableOKCB();
}


void CopyEnvironmentCB()
{
	if (current_view->available()) {
		current_view->currentBuffer()->text->copyEnvironmentType();
		current_view->getMiniBuffer()->Set(i18n("Paragraph environment type copied"));
	}
}


void PasteEnvironmentCB()
{
	if (current_view->available()) {
		current_view->currentBuffer()->text->pasteEnvironmentType();
		current_view->getMiniBuffer()->Set(i18n("Paragraph environment type set"));
		current_view->currentBuffer()->updateFull();
	}
}


void CopyCB()
{
	if (current_view->available()) {
		current_view->currentBuffer()->text->CopySelection();
		current_view->getMiniBuffer()->Set(i18n("Copy"));
	}
}


void CutCB()
{
	if (current_view->available()) {
		current_view->getScreen()->HideCursor();
		current_view->currentBuffer()->updateFull(false);
		current_view->currentBuffer()->text->CutSelection();
		current_view->currentBuffer()->updateFull();
		current_view->getMiniBuffer()->Set(i18n("Cut"));
	}
}


void PasteCB()
{
	if (current_view->available()) {
		current_view->getMiniBuffer()->Set(i18n("Paste"));
		current_view->getScreen()->HideCursor();
		/* clear the selection */
		current_view->getScreen()->ToggleSelection();
		current_view->currentBuffer()->text->ClearSelection();
		current_view->currentBuffer()->updateFull(false);

		/* paste */
		current_view->currentBuffer()->text->PasteSelection();
		current_view->currentBuffer()->updateFull();

		/* clear the selection */
		current_view->getScreen()->ToggleSelection();
		current_view->currentBuffer()->text->ClearSelection();
		current_view->currentBuffer()->updateFull(false);
	}
}


void MeltCB()
{
	if (current_view->available()) {
		current_view->getMiniBuffer()->Set(i18n("Melt"));
		current_view->getScreen()->HideCursor(true);
		current_view->currentBuffer()->updateFull(false);
		current_view->currentBuffer()->text->MeltFootnoteEnvironment();
		current_view->currentBuffer()->updateFull();
	}
}


// Change environment depth.
// if decInc == -1, decrement depth
// other  increment depth
void DepthCB(long decInc)
{
	if (current_view->available()) {
		current_view->getScreen()->HideCursor();
		current_view->currentBuffer()->updateFull(false);
		if( decInc == -1 )
		  current_view->currentBuffer()->text->DecDepth();
		else
		  current_view->currentBuffer()->text->IncDepth();

		current_view->currentBuffer()->updateFull();
		current_view->getMiniBuffer()->Set(i18n("Changed environment depth"
				  " (in possible range, maybe not)"));
	}
}





void FreeCB()
{
	if (current_view->available()) {
	  if( klyxdialogs->character == 0 )
	    MenuLayoutCharacter();
	  else{
	    // If the mask is completely neutral, tell user
	    if (klyxdialogs->character->data()->font() == LyXFont(LyXFont::ALL_IGNORE)){
	      // Could only happen with user style
	      current_view->getMiniBuffer()->Set(i18n("No font change defined. Use Character under"
						 " the Layout menu to define font change."));
	      return;
	    }
	
	    ToggleAndShow( klyxdialogs->character->data()->font() );
	  }
	}
}



/* callbacks for form ParagraphDialog */
inline void DeactivateParagraphButtons () // ported to KDE, KD 97-11-27
{
  klyxdialogs->paragraph->deactivate();
}

inline void ActivateParagraphButtons () // ported to KDE, KD 97-11-27
{
  klyxdialogs->paragraph->activate();
}


  // "Synchronize" the comboboxes and edit fields, making it
  // impossible to commit senseless data.
void ParagraphVSpaceCB_SpaceAboveCombo( int ) // ported to KDE, KD 97-11-27
{
  ParagraphDialogData* dlg_data = klyxdialogs->paragraph->data();
  if( dlg_data->vertical_spaces_above != ParagraphDialogData::VS_Length )
	{
	  dlg_data->space_above = "";
	  klyxdialogs->paragraph->setData( dlg_data );
	  ActivateParagraphButtons();
	}
}


void ParagraphVSpaceCB_SpaceBelowCombo( int ) // ported to KDE, KD 97-11-27
{
  ParagraphDialogData* dlg_data = klyxdialogs->paragraph->data();
  if( dlg_data->vertical_spaces_below != ParagraphDialogData::VS_Length )
	{
	  dlg_data->space_below = "";
	  klyxdialogs->paragraph->setData( dlg_data );
	  ActivateParagraphButtons();
	}
}


void ParagraphVSpaceCB_SpaceAboveEdit( const char* text ) // ported to KDE, KD 97-11-27
{
  if( *text == 0 ) // empty string
	{
	  ParagraphDialogData* dlg_data = klyxdialogs->paragraph->data();
	  dlg_data->vertical_spaces_above = ParagraphDialogData::VS_None;
	  klyxdialogs->paragraph->setData( dlg_data );
	  ActivateParagraphButtons();
	}
  else if( isValidGlueLength( text ) )
	{
	  ParagraphDialogData* dlg_data = klyxdialogs->paragraph->data();
	  dlg_data->vertical_spaces_above = ParagraphDialogData::VS_Length;
	  klyxdialogs->paragraph->setData( dlg_data );
	  ActivateParagraphButtons();
	}
  else
	{
	  ParagraphDialogData* dlg_data = klyxdialogs->paragraph->data();
	  dlg_data->vertical_spaces_above = ParagraphDialogData::VS_Length;
	  klyxdialogs->paragraph->setData( dlg_data );
	  DeactivateParagraphButtons();
	}
}

void ParagraphVSpaceCB_SpaceBelowEdit( const char* text ) // ported to KDE, KD 97-11-27
{
  if( *text == 0 ) // empty string
	{
	  ParagraphDialogData* dlg_data = klyxdialogs->paragraph->data();
	  dlg_data->vertical_spaces_below = ParagraphDialogData::VS_None;
	  klyxdialogs->paragraph->setData( dlg_data );
	  ActivateParagraphButtons();
	}
  else if( isValidGlueLength( text ) )
	{
	  ParagraphDialogData* dlg_data = klyxdialogs->paragraph->data();
	  dlg_data->vertical_spaces_below = ParagraphDialogData::VS_Length;
	  klyxdialogs->paragraph->setData( dlg_data );
	  ActivateParagraphButtons();
	}
  else
	{
	  ParagraphDialogData* dlg_data = klyxdialogs->paragraph->data();
	  dlg_data->vertical_spaces_below = ParagraphDialogData::VS_Length;
	  klyxdialogs->paragraph->setData( dlg_data );
	  DeactivateParagraphButtons();
	}
}


void ParagraphApplyCB() // ported to KDE, KD 97-11-27
{
  if (!current_view->available())
	return;

  VSpace space_top, space_bottom;
  char align;

  // If a vspace kind is "Length" but there's no text in
  // the input field, reset the kind to "None".
  ParagraphDialogData* dlg_data = klyxdialogs->paragraph->data();
  if( dlg_data->vertical_spaces_above == ParagraphDialogData::VS_Length &&
	  dlg_data->space_above.isEmpty() )
	{
	  dlg_data->vertical_spaces_above = ParagraphDialogData::VS_None;
	  klyxdialogs->paragraph->setData( dlg_data );
	}
  if( dlg_data->vertical_spaces_below == ParagraphDialogData::VS_Length &&
	  dlg_data->space_below.isEmpty() )
	{
	  dlg_data->vertical_spaces_below = ParagraphDialogData::VS_None;
	  klyxdialogs->paragraph->setData( dlg_data );
	}

	switch ( dlg_data->vertical_spaces_above )
	  {
	  case ParagraphDialogData::VS_None: space_top = VSpace::NONE; break;
	  case ParagraphDialogData::VS_Defskip: space_top = VSpace::DEFSKIP; break;
	  case ParagraphDialogData::VS_Smallskip: space_top = VSpace::SMALLSKIP; break;
	  case ParagraphDialogData::VS_Medskip: space_top = VSpace::MEDSKIP; break;
	  case ParagraphDialogData::VS_Bigskip: space_top = VSpace::BIGSKIP; break;
	  case ParagraphDialogData::VS_Vfill: space_top = VSpace::VFILL; break;
	  case ParagraphDialogData::VS_Length: space_top = LyXGlueLength ( dlg_data->space_above.data() ); break;
	}
	if ( dlg_data->space_keep_above )
	  space_top.setKeep (true);

	switch ( dlg_data->vertical_spaces_below )
	  {
	  case ParagraphDialogData::VS_None: space_bottom = VSpace::NONE; break;
	  case ParagraphDialogData::VS_Defskip: space_bottom = VSpace::DEFSKIP; break;
	  case ParagraphDialogData::VS_Smallskip: space_bottom = VSpace::SMALLSKIP; break;
	  case ParagraphDialogData::VS_Medskip: space_bottom = VSpace::MEDSKIP; break;
	  case ParagraphDialogData::VS_Bigskip: space_bottom = VSpace::BIGSKIP; break;
	  case ParagraphDialogData::VS_Vfill: space_bottom = VSpace::VFILL; break;
	  case ParagraphDialogData::VS_Length: space_bottom = LyXGlueLength ( dlg_data->space_below.data() ); break;
	}
	if ( dlg_data->space_keep_below )
	  space_bottom.setKeep (true);

	switch( dlg_data->alignment )
	  {
	  case ParagraphDialogData::Left:
		align = LYX_ALIGN_LEFT;
		break;
	  case ParagraphDialogData::Right:
		align = LYX_ALIGN_RIGHT;
		break;
	  case ParagraphDialogData::Center:
		align = LYX_ALIGN_CENTER;
		break;
	  default:
		align = LYX_ALIGN_BLOCK;
	  }

	Spacing spacing(Spacing::LayoutDefault);
	if (dlg_data->spacing != ParagraphDialogData::LayoutDefault){
	  switch (dlg_data->spacing){
	  case ParagraphDialogData::Single:
	    spacing.set(Spacing::Single);
	    break;
	  case ParagraphDialogData::Onehalf:
	    spacing.set(Spacing::Onehalf);
	    break;
	  case ParagraphDialogData::Double:
	    spacing.set(Spacing::Double);
	    break;
	  case ParagraphDialogData::Other:
	    if (dlg_data->spacing_length.toFloat()>0)
	      spacing.set(Spacing::Other, dlg_data->spacing_length.toFloat());
	    break;
	  default:
	    break;
	  }
	}
	

	current_view->currentBuffer()->text->
	  SetParagraph( dlg_data->lines_above,
			dlg_data->lines_below,
			dlg_data->pagebreaks_above,
			dlg_data->pagebreaks_below,
			space_top,
			space_bottom,
			spacing,
			align,
			dlg_data->label_width.data(),
			dlg_data->noindent );
	current_view->currentBuffer()->updateFull();
	current_view->getMiniBuffer()->Set( i18n( "Paragraph layout set" ) );
}



/* callbacks for form form_document */

void UpdateDocumentButtons(BufferParams const &params) // ported to KDE; KD 09-12-97
{
  DocumentLayoutDialogData* dlg_data = klyxdialogs->document->data();

  dlg_data->pagestyle = DocumentLayoutDialogData::PageStyleDefault;
	
  if (params.sides == 2)
	dlg_data->sides = DocumentLayoutDialogData::TwoSides;
  else
	dlg_data->sides = DocumentLayoutDialogData::OneSide;
	
  if (params.columns == 2)
	dlg_data->columns = DocumentLayoutDialogData::TwoColumns;
  else
	dlg_data->columns = DocumentLayoutDialogData::OneColumn;
	
  dlg_data->extra_options = params.options.c_str();
  dlg_data->section_number_depth = params.secnumdepth;
  dlg_data->table_of_contents_depth = params.tocdepth;

  klyxdialogs->document->setData( dlg_data );
}



void DocumentDefskipCB_comboDefaultSkip( int selection ) // ported to KDE; KD 09-12-97
{
  if( selection != DocumentLayoutDialogData::Length )
	{
	  DocumentLayoutDialogData* dlg_data = klyxdialogs->document->data();
	  dlg_data->default_skip_length = "";
	  klyxdialogs->document->setData( dlg_data );
	  ActivateDocumentButtons();
	}
}


void DocumentDefskipCB_editDefaultSkip( const char* text ) // ported to KDE; KD 09-12-97
{
  DocumentLayoutDialogData* dlg_data = klyxdialogs->document->data();
  if( !*text )
	{
	  dlg_data->default_skip = DocumentLayoutDialogData::Medskip;
	  ActivateDocumentButtons();
	}
  else if( isValidGlueLength( text ) )
	{
	  dlg_data->default_skip = DocumentLayoutDialogData::Length;
	  ActivateDocumentButtons();
	}
  else
	{
	  dlg_data->default_skip = DocumentLayoutDialogData::Length;
	  DeactivateDocumentButtons();
	}
}


void DocumentApplyCB() // ported to KDE; KD 09-12-97
{
  bool redo = false;
  BufferParams *params = &(current_view->currentBuffer()->params);

  // If default skip is a "Length" but there's no text in the
  // input field, reset the kind to "Medskip", which is the default.
  DocumentLayoutDialogData* dlg_data = klyxdialogs->document->data();
  if( dlg_data->default_skip == DocumentLayoutDialogData::Length &&
	  !*dlg_data->default_skip_length.data() )
	{
	  dlg_data->default_skip = DocumentLayoutDialogData::Medskip;
	  klyxdialogs->document->setData( dlg_data );
	}

  /* this shouldn't be done automatically IMO. For example I write german
   * documents with an american keyboard very often. Matthias */

  params->language = klyxdialogs->document->numToLanguageString( dlg_data->language );

  /* ChangeKeymap(buffer->parameters.language, TRUE, false,
//	 fl_get_choice(fd_form_document->choice_language)); */
  params->fonts = klyxdialogs->document->numToFontString( dlg_data->fonts );
  params->inputenc = klyxdialogs->document->numToInputencString( dlg_data->encoding );
  params->fontsize = klyxdialogs->document->numToFontSizeString( dlg_data->font_size );
  params->pagestyle = klyxdialogs->document->numToPageStyleString( dlg_data->pagestyle );
  params->graphicsDriver = klyxdialogs->document->numToPostScriptDriverString( dlg_data->ps_driver );

  // FIXME: Button does not exist ?!
  //	params->use_amsmath =
//fl_get_button(fd_form_document->check_use_amsmath);

  if (!current_view->available())
	return;

  char new_class = dlg_data->_class;
  if (params->textclass != new_class) {
	redo = true;
	current_view->getMiniBuffer()->Set(i18n("Converting document to new document class..."));
	int ret = current_view->currentBuffer()->
	  text->
	  SwitchLayoutsBetweenClasses(current_view->currentBuffer()->
								  params.textclass,
								  new_class,
								  current_view->currentBuffer()->
								  paragraph);

	if (ret){
	  LString s;
	  if (ret==1)
		s= i18n("One paragraph couldn't be converted");
	  else {
		s += ret;
		s += i18n(" paragraphs couldn't be converted");
	  }
	  WriteAlert(i18n("Conversion Errors!"),s,
				 i18n("into chosen document class"));
	}
		
	params->textclass = new_class;
  }
	
  char tmpsep = params->paragraph_separation;
  if( dlg_data->separation == DocumentLayoutDialogData::Indent )
	params->paragraph_separation = LYX_PARSEP_INDENT;
  else
	params->paragraph_separation = LYX_PARSEP_SKIP;
  if (tmpsep != params->paragraph_separation)
	redo = true;

  params->use_amsmath = dlg_data->amsmath;

  VSpace tmpdefskip = params->getDefSkip();
  switch ( dlg_data->default_skip + 1 ) {
  case 1: params->setDefSkip(VSpace::SMALLSKIP); break;
  case 2: params->setDefSkip(VSpace::MEDSKIP); break;
  case 3: params->setDefSkip(VSpace::BIGSKIP); break;
  case 4: params->setDefSkip(
							 VSpace (LyXGlueLength ( dlg_data->default_skip_length.data() ) ) );
  break;
  // DocumentDefskipCB assures that this never happens
  default: params->setDefSkip(VSpace::MEDSKIP); break;
  }
  if (!(tmpdefskip == params->getDefSkip()))
	redo = true;

  if ( dlg_data->columns == DocumentLayoutDialogData::TwoColumns )
	params->columns = 2;
  else
	params->columns = 1;
  if ( dlg_data->sides == DocumentLayoutDialogData::TwoSides )
	params->sides = 2;
  else
	params->sides = 1;

  Spacing tmpSpacing = params->spacing;
  switch( dlg_data->spacing)
	{
	case DocumentLayoutDialogData::Single:
	  params->spacing.set(Spacing::Single);
	  break;
	case DocumentLayoutDialogData::Onehalf:
	  params->spacing.set(Spacing::Onehalf);
	  break;
	case DocumentLayoutDialogData::Double:
	  params->spacing.set(Spacing::Double);
	  break;
	case DocumentLayoutDialogData::Other:
	  params->spacing.set(Spacing::Other, dlg_data->spacing_length.toFloat());
	  break;
	}
  if (tmpSpacing != params->spacing)
	redo = true;

  signed char tmpchar = (signed char) dlg_data->section_number_depth;
  if (params->secnumdepth != tmpchar)
	redo = true;
  params->secnumdepth = tmpchar;

  params->tocdepth = (signed char) dlg_data->table_of_contents_depth;

  params->float_placement = dlg_data->float_placement;

  // More checking should be done to ensure the string doesn't have
  // spaces or illegal placement characters in it. (thornley)

  if (redo)
	current_view->redoCurrentBuffer();

  current_view->getMiniBuffer()->Set(i18n("Document layout set"));
  current_view->currentBuffer()->markDirty();

  params->options = dlg_data->extra_options;
}


void DocumentCancelCB() // ported to KDE; KD 09-12-97
{
  klyxdialogs->document->hide();
}


void DocumentOKCB() // ported to KDE; KD 09-12-97
{
	DocumentCancelCB();
	DocumentApplyCB();
}


void GotoNote()
{
	if (!current_view->getScreen())
		return;

	current_view->getScreen()->HideCursor(true);
	current_view->currentBuffer()->updateFull(false);
	LyXCursor tmp;

	if (!current_view->currentBuffer()->text->GotoNextNote()) {
		if (current_view->currentBuffer()->text->cursor.pos
		    || current_view->currentBuffer()->text->cursor.par !=
		    current_view->currentBuffer()->text->FirstParagraph())
			{
				tmp = current_view->currentBuffer()->text->cursor;
				current_view->currentBuffer()->text->cursor.par =
					current_view->currentBuffer()->text->FirstParagraph();
				current_view->currentBuffer()->text->cursor.pos = 0;
				if (!current_view->currentBuffer()->text->GotoNextNote()) {
					current_view->currentBuffer()->text->cursor = tmp;
					current_view->getMiniBuffer()->Set(i18n("No more notes"));
					LyXBell();
				}
			} else {
				current_view->getMiniBuffer()->Set(i18n("No more notes"));
				LyXBell();
			}
	}
	current_view->currentBuffer()->updateFull(false);
	current_view->currentBuffer()->text->sel_cursor =
		current_view->currentBuffer()->text->cursor;
}


void InsertCorrectQuote()
{
	Buffer *cbuffer = current_view->currentBuffer();
	char c;

	if  (cbuffer->text->cursor.pos )
		c = cbuffer->text->cursor.par->GetChar(cbuffer->text->cursor.pos - 1);
	else
		c = ' ';

	cbuffer->insertInset(new InsetQuotes(c, cbuffer->params));
}


void TableOKCB() // ported to KDE, KD 97-12-04
{
	int xsize,ysize;
	if (!current_view->getScreen())
		return;

	// check for tables in tables
	if (current_view->currentBuffer()->text->cursor.par->table){
		WriteAlert(i18n("Impossible Operation!"),
			   i18n("Cannot insert table in table."),
			   i18n("Sorry."));
		return;
	}

	current_view->getMiniBuffer()->Set(i18n("Inserting table..."));

	InsertTableDialogData* dlg_data = klyxdialogs->inserttable->data();
	ysize = dlg_data->columns;
	xsize = dlg_data->rows;


	current_view->getScreen()->HideCursor(true);
	current_view->currentBuffer()->updateFull(false);

	current_view->currentBuffer()->text->SetCursorParUndo();
	current_view->currentBuffer()->text->FreezeUndo();

	current_view->currentBuffer()->text->BreakParagraph();
	current_view->currentBuffer()->updateFull();

	if (current_view->currentBuffer()->text->cursor.par->Last()) {
		current_view->currentBuffer()->text->CursorLeft();

		current_view->currentBuffer()->text->BreakParagraph();
		current_view->currentBuffer()->updateFull();
	}

	current_view->currentBuffer()->text->current_font.setLatex(LyXFont::OFF);
	current_view->currentBuffer()->text->SetLayout(0); // standard layout
	if (current_view->currentBuffer()->text->cursor.par->footnoteflag ==
	    LyXParagraph::NO_FOOTNOTE) {
		current_view->currentBuffer()->
			text->SetParagraph(0, 0,
					   0, 0,
					   VSpace (0.3 * current_view->currentBuffer()->
						   params.spacing.getValue(),
						   LyXLength::CM),
					   VSpace (0.3 * current_view->currentBuffer()->
						   params.spacing.getValue(),
						   LyXLength::CM),
					   Spacing(Spacing::LayoutDefault),
					   LYX_ALIGN_CENTER,
					   "",
					   0);
	}
	else
		current_view->currentBuffer()->text->SetParagraph(0, 0,
					   0, 0,
					   VSpace::NONE, VSpace::NONE,
					   Spacing(Spacing::LayoutDefault),
					   LYX_ALIGN_CENTER,
					   "",
					   0);

	current_view->currentBuffer()->text->cursor.par->table = new LyXTable(xsize, ysize);
	int i;
	for (i=0; i<xsize * ysize - 1; i++)
		current_view->currentBuffer()->text->cursor.par->InsertChar(0,LYX_META_NEWLINE);
	current_view->currentBuffer()->text->RedoParagraph();

	current_view->currentBuffer()->text->UnFreezeUndo();

	current_view->currentBuffer()->updateFull();
	current_view->getMiniBuffer()->Set(i18n("Table inserted"));
}


void PrintApplyCB()
{
    if ( !current_view->available() )
	return;

  Buffer *buffer = current_view->currentBuffer();
  LString path = OnlyPath(buffer->getFileName());

  LString pageflag;

  // FIXME: QPrinter currently does not support even/odd printing
  // KD, 10-12-97
//   if (fl_get_button(fd_form_print->radio_even_pages))
// 	pageflag = lyxrc->print_evenpage_flag;
//   else if (fl_get_button(fd_form_print->radio_odd_pages))
// 	pageflag = lyxrc->print_oddpage_flag;

	LString reverseflag;
	
	// FIXME: QPrinter currently does not support reverse printing
	// KD, 10-12-97
	//	if (fl_get_button(fd_form_print->radio_order_reverse))
	//	reverseflag = lyxrc->print_reverse_flag;

	LString orientationflag;
	if (buffer->params.orientation == ORIENTATION_LANDSCAPE)
	  {
		orientationflag = lyxrc->print_landscape_flag;
		klyxprinter->setOrientation( QPrinter::Landscape );
	  }

	LString ps_file = klyxprinter->outputFileName();
	LString printer = klyxprinter->printerName();

	//Matthias
	QString tmp = klyxprinter->printProgram();
	if (!tmp.isEmpty()){
	  lyxrc->print_spool_command = tmp.data();
	}

	LString command = lyxrc->print_command + ' '
		+ pageflag + ' '
		+ reverseflag + ' '
		+ orientationflag + ' '
		+ lyxrc->print_extra_options + ' ';

	if( klyxprinter->outputToFile() )
		command += lyxrc->print_to_file + " \""
			+ MakeAbsPath(ps_file, path)
			+ '\"';
	else if (!lyxrc->print_spool_command.empty())
		command += lyxrc->print_to_file + " \""
			+ ps_file
			+ '\"';
	else if (!printer.empty())
		command += lyxrc->print_to_printer + '\"'
			+ printer + '\"';

	// push directorypath, if necessary
        if (lyxrc->use_tempdir || (IsDirWriteable(path) < 1)){
		path = buffer->tmppath;
        }
        PathPush(path);
	if (!lyxrc->print_spool_command.empty() &&
		!klyxprinter->outputToFile() ) {
 	    	LString command2;
  		if (!printer.empty())
 			command2 = lyxrc->print_spool_command + ' '
  				+ lyxrc->print_to_printer
  				+ '\"' + printer +  '\"' ;
                  else
 			command2 = lyxrc->print_spool_command;
 		// First run dvips and, if succesful, then spool command
 		if (RunScript(buffer, true, command)) {
 			RunScript(buffer, false, command2, ps_file);
 		}
        } else
		RunScript(buffer, false, command);
        PathPop();
	return;
}



void MenuPrint( Buffer *buffer )
{
    LString input_file;
    KConfig *config = kapp->getConfig();
    config->setGroup( "Printer" );
    if( ! buffer->text )
	return;
    input_file = ChangeExtension( buffer->getFileName(),
				  lyxrc->print_file_extension, true );
    klyxprinter->setOutputFileName( input_file.c_str() );
    klyxprinter->setOutputToFile( config->readBoolEntry( "FileOutput", false ) );
    klyxprinter->setPrintProgram( config->readEntry( "PrintProgram", "lpr" ) );
    klyxprinter->setPrinterName( config->readEntry( "Printer", "lp" ) );
    klyxprinter->setOrientation( static_cast< QPrinter::Orientation >( config->readNumEntry( "Orientation", QPrinter::Portrait ) ) );
    klyxprinter->setPageSize( static_cast< QPrinter::PageSize >( config->readNumEntry( "PageSize", QPrinter::A4 ) ) );
#if QT_VERSION >= 140
    if( QPrintDialog::getPrinterSetup( klyxprinter ) ) {
#else
    if( klyxprinter->setup() ) {
#endif
	config->setGroup( "Printer" );
	config->writeEntry( "FileOutput", klyxprinter->outputToFile() );
	config->writeEntry( "PrintProgram",  (const char*) klyxprinter->printProgram() );
	config->writeEntry( "Printer",  (const char*) klyxprinter->printerName() );
	config->writeEntry( "Orientation",  (int) klyxprinter->orientation() );
	config->writeEntry( "PageSize",  (int) klyxprinter->pageSize() );
	PrintApplyCB();
	}

    return;
}


/* callbacks for form form_figure */

void FigureOKCB() // ported to KDE, KD 97-12-04
{
	if (!current_view->available())
		return;

	Buffer * buffer = current_view->currentBuffer();
	if(buffer->isReadonly()) // paranoia
		return;
	
	current_view->getMiniBuffer()->Set(i18n("Inserting figure..."));
	InsertFigureDialogData* dlg_data = klyxdialogs->insertfigure->data();
	if( dlg_data->type == InsertFigureDialogData::Inlined
	    || buffer->text->cursor.par->table)
	  {
		buffer->insertInset(new InsetFig(100, 20, buffer));
		current_view->getMiniBuffer()->Set(i18n("Figure inserted"));
		return;
	  }
	
	current_view->getScreen()->HideCursor(true);
	buffer->updateFull(false);

	buffer->text->SetCursorParUndo();
	buffer->text->FreezeUndo();

	buffer->text->BreakParagraph();
	buffer->updateFull();

	if (buffer->text->cursor.par->Last()) {
		buffer->text->CursorLeft();
	
		buffer->text->BreakParagraph();
		buffer->updateFull();
	}

	// The standard layout should always be numer 0;
	buffer->text->SetLayout(0);
	
	if (buffer->text->cursor.par->footnoteflag ==
	    LyXParagraph::NO_FOOTNOTE) {
		buffer->text->SetParagraph(0, 0,
					   0, 0,
					   VSpace (0.3 * buffer->params.spacing.getValue(),
						   LyXLength::CM),
					   VSpace (0.3 *
						   buffer->params.spacing.getValue(),
						   LyXLength::CM),
					   Spacing(Spacing::LayoutDefault),
					   LYX_ALIGN_CENTER, "", 0);
	} else
		buffer->text->SetParagraph(0, 0,
					   0, 0,
					   VSpace::NONE, VSpace::NONE,
					   Spacing(Spacing::LayoutDefault),
					   LYX_ALIGN_CENTER,
					   "",
					   0);
	
	buffer->updateFull();

	Inset *new_inset = NULL;
	
	new_inset = new InsetFig(100, 100, buffer);
	buffer->insertInset(new_inset);
	
	buffer->updateFull(false);
	current_view->getMiniBuffer()->Set(i18n("Figure inserted"));
	buffer->text->UnFreezeUndo();
}


void ScreenApplyCB()
{
  KConfig* config = kapp->getConfig();
  QString oldgroup = config->group();
  config->setGroup( "ScreenOptions" );

  ScreenOptionsTabDialogData* dlg_data = klyxdialogs->screenoptions->data();
  config->writeEntry( "RomanFontName",
					  (const char*) dlg_data->fontsdata->roman_font );
  config->writeEntry( "SansFontName",
					   (const char*)dlg_data->fontsdata->sans_serif_font );
  config->writeEntry( "TypewriterFontName",
					   (const char*)dlg_data->fontsdata->typewriter_font );
  config->writeEntry( "FontNorm",
					   (const char*)dlg_data->fontsdata->font_norm );
  // 	// TODO
// 	lyxrc->zoom = atoi(fl_get_input(fd_form_screen->intinput_size));
  config->writeEntry( "FontZoom",
					  atoi( dlg_data->fontsdata->font_zoom ) );

  config->writeEntry( "GeneralToolbar", static_cast< int >( dlg_data->toolbarsdata->general_toolbar ) );
  config->writeEntry( "MathToolbar", static_cast< int >( dlg_data->toolbarsdata->math_toolbar ) );
  config->writeEntry( "TablesToolbar", static_cast< int >( dlg_data->toolbarsdata->tables_toolbar ) );
  config->writeEntry( "CharacterToolbar", static_cast< int >( dlg_data->toolbarsdata->character_toolbar) );

  current_view->getFontloader()->update(atoi(dlg_data->fontsdata->font_zoom));


  current_view->getMiniBuffer()->Set(i18n("Screen options set"));

  config->setGroup( oldgroup );

  global_toolbar_visibilities = *dlg_data->toolbarsdata;

  // 	// All buffers might need a resize
  printf("SCREEN APPLY CALLBACK........new zoom %s, %d %d...\n",
	 dlg_data->fontsdata->font_zoom.data(),
	 atoi(dlg_data->fontsdata->font_zoom),
	 atoi(dlg_data->fontsdata->font_zoom.data()));
  bufferlist.screenSettingsChanged();
}


void LaTeXOptions()
{
    if ( ! current_view->available() )
	return;
    if( ! klyxdialogs->latexoptions )
	klyxdialogs->latexoptions = new LaTeXOptionsDialog( current_view->getOwner() );
    klyxdialogs->latexoptions->setAllowAccents( current_view->currentBuffer()->params.allowAccents );
    klyxdialogs->latexoptions->exec();
#warning Is this setting stored in the *.lyx file ? It does not seem like (Jochen).
    return;
}


// This function runs "configure" and then rereads lyx.defaults to
// reconfigure the automatic settings.
void Reconfigure()
{
	current_view->getMiniBuffer()->Set(i18n("Running configure..."));

	// Run configure in user lyx directory
	PathPush(user_lyxdir);
	Systemcalls one(Systemcalls::System,
                          AddName(system_lyxdir,"configure"));
	PathPop();
	current_view->getMiniBuffer()->Set(i18n("Reloading configuration..."));
	lyxrc->Read(LibFileSearch("", "lyxrc.defaults"));
	WriteAlert(i18n("The system has been reconfigured."),
		   i18n("You need to restart LyX to make use of any"),
		   i18n("updated document class specifications."));
}


/* these functions are for the spellchecker */
char* NextWord(float &value)
{
	if (!current_view->available()){
		value = 1;
		return NULL;
	}

	char* string =  current_view->currentBuffer()->text->SelectNextWord(value);

	return string;
}


void SelectLastWord()
{
	if (!current_view->available())
		return;

	current_view->getScreen()->HideCursor(true);
	current_view->currentBuffer()->text->SelectSelectedWord();
	current_view->getScreen()->ToggleSelection(false);
	current_view->currentBuffer()->updateFull(false);
}


void EndOfSpellCheck()
{
	if (!current_view->available())
		return;

	current_view->getScreen()->HideCursor(true);
	current_view->currentBuffer()->text->SelectSelectedWord();
	current_view->currentBuffer()->text->ClearSelection();
	current_view->currentBuffer()->updateFull(false);
}


void ReplaceWord(LString const & replacestring)
{
	if (!current_view->getScreen())
		return;

	current_view->getScreen()->HideCursor();
	current_view->currentBuffer()->updateFull(false);

	/* clear the selection (if there is any) */
	current_view->getScreen()->ToggleSelection(false);
	current_view->currentBuffer()->text->
		ReplaceSelectionWithString(replacestring.c_str());

	current_view->currentBuffer()->text->SetSelectionOverString(replacestring.c_str());

	// Go back so that replacement string is also spellchecked
	for (int i=-1; i< replacestring.length(); i++) {
		current_view->currentBuffer()->text->CursorLeftIntern();
	}
	current_view->currentBuffer()->updateFull();
}
// End of spellchecker stuff


void UpdateInset(Inset* inset, bool mark_dirty)
{
	if (!inset)
		return;

	/* very first check for locking insets*/
	if (current_view->currentBuffer()->the_locking_inset == inset){
		if (current_view->currentBuffer()->text->UpdateInset(inset)){
			current_view->update();
			if (mark_dirty){
				if (current_view->currentBuffer()->isLyxClean())
					current_view->getMiniBuffer()->startTimer(1000*4);
				current_view->currentBuffer()->markDirty();
			}
			current_view->updateScrollbar();
			return;
		}
	}

	/* first check the current buffer */
	if (current_view->available()){
		current_view->getScreen()->HideCursor();
		current_view->currentBuffer()->updateCursorMove();
		if (current_view->currentBuffer()->text->UpdateInset(inset)){
		  if (mark_dirty)
		    current_view->currentBuffer()->updateFull(mark_dirty);
		  else
		    current_view->currentBuffer()->updateCursorMove();
		  return;
		}
	}

	// check all buffers
	bufferlist.updateInset(inset, mark_dirty);

}


/* these functions return 1 if an error occured,
   otherwise 0 */
int LockInset(UpdatableInset* inset)
{
	if (!current_view->currentBuffer()->the_locking_inset && inset){
		current_view->currentBuffer()->the_locking_inset = inset;
		current_view->getOwner()->updateToolbars();
		return 0;
	}
	return 1;
}


void ShowLockedInsetCursor(long x, long y, int asc, int desc)
{
	if (current_view->currentBuffer()->the_locking_inset &&
	    current_view->getScreen()){
		y += current_view->currentBuffer()->text->cursor.y;
		current_view->getScreen()->ShowManualCursor(x, y,
							    asc, desc);
	}
}


void HideLockedInsetCursor(long x, long y, int asc, int desc)
{
	if (current_view->currentBuffer()->the_locking_inset &&
	    current_view->getScreen()){
		y += current_view->currentBuffer()->text->cursor.y;
		current_view->getScreen()->HideManualCursor(x, y,
							    asc, desc);
	}
}


void FitLockedInsetCursor(long x, long y, int asc, int desc)
{
	if (current_view->currentBuffer()->the_locking_inset &&
	    current_view->getScreen()){
		y += current_view->currentBuffer()->text->cursor.y;
		if (current_view->getScreen()->FitManualCursor(x, y, asc, desc))
			current_view->updateScrollbar();
	}
}


int UnlockInset(UpdatableInset* inset)
{
	if (inset &&
	    current_view->currentBuffer()->the_locking_inset == inset){
		inset->InsetUnlock();
		current_view->currentBuffer()->the_locking_inset = NULL;
		current_view->currentBuffer()->text->FinishUndo();
		return 0;
	}
	return bufferlist.unlockInset(inset);
}


void LockedInsetStoreUndo(Undo::undo_kind kind)
{
	if (!current_view->currentBuffer()->the_locking_inset)
		return; // shouldn't happen
	if (kind == Undo::EDIT) // in this case insets would not be stored!
		kind = Undo::FINISH;
	current_view->currentBuffer()->text->SetUndo(kind,
			      current_view->currentBuffer()->text->cursor.par->
			      ParFromPos(current_view->currentBuffer()->text->cursor.pos)->previous,
			      current_view->currentBuffer()->text->cursor.par->
			      ParFromPos(current_view->currentBuffer()->text->cursor.pos)->next);
}


void PutInsetIntoInsetUpdateList(Inset* inset)
{
	if (inset) {
		InsetUpdateStruct* tmp = new InsetUpdateStruct();
		tmp->inset = inset;
		tmp->next = InsetUpdateList;
		InsetUpdateList = tmp;
	}
}


void UpdateInsetUpdateList()
{
	InsetUpdateStruct *tmp = InsetUpdateList;
	while (tmp) {
		UpdateInset(tmp->inset, false); // "false" because no document change
		tmp = tmp->next;
	}

	/* delete the update list */
	while (InsetUpdateList) {
		tmp = InsetUpdateList;
		InsetUpdateList = InsetUpdateList->next;
		delete tmp;
	}
	InsetUpdateList = NULL;
}


