/* This file is part of
* ======================================================
*
*           LyX, the High Level Word Processor
*
*           Copyright (C) 1995 1996 Matthias Ettrich
*           and the LyX Team.
*
*======================================================*/

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include "klyx.h"

#include "BufferView.h"
#include "LyXView.h"
#include "lyxfunc.h"
#include "bufferlist.h"
#include "LyXView.h"
#include "minibuffer.h"
#include "lyxscreen.h"
#include "error.h"
#include "lyxdraw.h"
#include "lyx_gui_misc.h"
#include "BackStack.h"
#include "lyxtext.h"
#include "lyx_cb.h"
#include "FontLoader.h"
#include "KLyXMenu.h"

#include <qscrbar.h>
#include <qapp.h>
#include <qpaintd.h>
#include <qclipbrd.h>

#include <X11/Xatom.h>

#include <stdlib.h>

#if !defined(lint) && !defined(WITH_WARNINGS)
static char vcid[] = "$Id: BufferView.C,v 1.28 1999/04/13 09:16:54 kuepper Exp $";
#endif /* lint */

extern BufferList bufferlist;
extern BufferView *current_view;

extern void SetXtermCursor(Window win);
extern bool input_prohibited;
extern bool selection_possible;
extern void BeforeChange();
extern int UnlockInset(UpdatableInset* inset);
extern void ToggleFloat();
extern void MenuLayoutTable(int flag);
extern InsetUpdateStruct *InsetUpdateList;
extern void UpdateInsetUpdateList();

extern void checkchildren();


BufferView::BufferView(LyXView *o, QWidget* parent)
	: _owner(o)
{
	_buffer = 0;
	
	screen = 0;
	current_scrollbar_value = 0;
// 	create_view(xpos, ypos, width, height);
	// Activate the timer for the cursor
// 	fl_set_timer(timer_cursor, 0.4);
// 	fl_set_focus_object(_owner->getForm(), work_area);
	work_area_focus = true;
	lyx_focus = false;
        backstack = new BackStack(100);

        k_canvas = new K_Canvas(this, parent);

}



BufferView::~BufferView()
{
    debug("BufferView destroyed");
	if (screen)
	  delete screen;
	screen = 0;
	if (_buffer)
	  _buffer->delUser(this);
	delete backstack;
	delete k_canvas;
}

void BufferView::updateTimerTimeout(){

  if (_buffer == 0 || screen == 0) return;
  setBufferInternal(_buffer);

  screen->HideCursor();
  _buffer->updateCursorMove();
  // we will also use this routine for clean-up in case some children
  // are still waiting to be terminated
  checkchildren();
}

void BufferView::setBufferInternal(Buffer *b, bool do_resize, long first_arg){
  long first = first_arg;

  current_view = this;
  _owner->setCurrentView( current_view );

  if (b && _buffer == b && b->getUser() == this){
    if (do_resize){
      if (screen)
	screen->expose();
      else
	k_canvas->repaint();

      updateScrollbar();
    }
    return;
  }


  // If we are closing the buffer, use the first buffer as current
  if (!b) {
    b = bufferlist.first();
  }

  if (b && b->getUser() != this)
    do_resize = true;


  if (screen && b ==_buffer)
    first = screen->first;

  if (b && b->getUser() != this){
    if (_buffer)
      _buffer->delUser(this);
    b->addUser(this);
    _owner->getMenuReceiver()->showMenus();
  }

  // Set current buffer
  _buffer = b;

  if (_buffer){
    lyxerr.debug(LString("  Buffer addr: ") + int(_buffer));

    // If we don't have a text object for this, we make one
    if (_buffer->text == 0 || _buffer->text->paperwidth !=
	k_canvas->work_area->width()
	|| _buffer->text->zoom != getFontloader()->getZoom())
      resizeCurrentBuffer();
    else
      updateScreen();

    if (first < 0)
      screen->first = screen->TopCursorVisible();
    else {
      if (first > _buffer->text->height -
	  k_canvas->work_area->height()*3/4
	  && first > k_canvas->work_area->height()
	  )
	first = _buffer->text->height
	  -k_canvas->work_area->height()*3/4;
      screen->first = first;
    }
  } else {
    if (screen)
      delete screen;
    screen = 0;
    _owner->getMenuReceiver()->hideMenus();
  }


  if (do_resize){
    if (screen)
      screen->Redraw();
    else
      k_canvas->repaint();

    updateScrollbar();
  }

}

void BufferView::setBuffer(Buffer *b)
{
	lyxerr.debug("Setting buffer in BufferView");

	if (bufferlist.getState() == BufferList::CLOSING){
	  _buffer = 0;
	  return;
	}

	
	setBufferInternal(b, true);
	_owner->updateToolbars();
	_owner->getMiniBuffer()->Init();
	_owner->updateWindowTitle();

}




void BufferView::switchBuffer(Buffer *buff)
{
	if (_buffer) {
	    _buffer->InsetSleep();
	}

	if (!buff) return;

        setBuffer(buff);
        buff->InsetWakeup();
}



void BufferView::updateScreen()
{
	// Regenerated the screen.
	if (screen)
		delete screen;
	screen = new LyXScreen(k_canvas->winId(),
			       k_canvas->work_area->width(),
			       k_canvas->work_area->height(),
			       _buffer->text);
}


void BufferView::resize()
{
  int first = -1;
  if (screen){
    if (screen->width() == k_canvas->work_area->width()){
      if (screen->height() < k_canvas->work_area->height())
	first = screen->first - (k_canvas->work_area->height() - screen->height());
      else
	first = screen->first + (screen->height() - k_canvas->work_area->height() );
    }
  }
  Buffer* tmp = _buffer;
  _buffer = 0;
  setBufferInternal(tmp, true, first);
}

void BufferView::screenSettingsChanged()
{
  getFontloader()->update(getFontloader()->getZoom());
  _owner->updateToolbars();
  redoCurrentBuffer();
}


void BufferView::redraw()
{

  lyxerr.debug("BufferView::redraw()");

  setBufferInternal(_buffer, true);

}


void BufferView::fitCursor()
{
	setBufferInternal(_buffer );
	if (screen) screen->FitCursor();
}


void BufferView::update()
{
	setBufferInternal(_buffer );
	if (screen) screen->Update();
}


void BufferView::updateScrollbar()
{
	/* If the text is smaller than the working area, the scrollbar
	 * maximum must be the working area height. No scrolling will
	 * be possible */

	if (!_buffer) {
	  k_canvas->scrollbar->setRange(0,0);
	  k_canvas->scrollbar->setSteps(16,
						k_canvas->work_area->height());
	  return;
	}
	
	setBufferInternal(_buffer );

	long cbth = 0;
	long cbsf = 0;

	if (_buffer->text)
		cbth = _buffer->text->height;
	if (screen)
		cbsf = screen->first;

	current_scrollbar_value = cbsf;

	if (cbth <= k_canvas->work_area->height()) {
	  // text is smaller than screen
	        k_canvas->scrollbar->setRange(0,0);
		k_canvas->scrollbar->setSteps(_buffer->text->DefaultHeight(),
					      k_canvas->work_area->height());
// 		fl_set_slider_size(scrollbar, scrollbar->h);
		return;
	}
	
	long maximum_height = -k_canvas->work_area->height()/4 + cbth;
	long value = cbsf;

	k_canvas->scrollbar->setRange(0, maximum_height);
	k_canvas->scrollbar->setSteps(_buffer->text->DefaultHeight(),
				      k_canvas->work_area->height());
	k_canvas->scrollbar->setValue(value);

}


void BufferView::redoCurrentBuffer()
{
	lyxerr.debug("BufferView::redoCurrentBuffer");
	if (_buffer && _buffer->text) {
	  delete _buffer->text;
	  _buffer->text = 0;
	  if (screen){
	    delete screen;
	    screen = 0;
	  }
	  Buffer* tmp = _buffer;
	  _buffer = 0;
	  setBufferInternal(tmp, true);
	}
}


int BufferView::resizeCurrentBuffer()
{
	lyxerr.debug("resizeCurrentBuffer");
	
	LyXParagraph *par = 0;
	LyXParagraph *selstartpar = 0;
	LyXParagraph *selendpar = 0;
	int pos = 0;
	int selstartpos = 0;
	int selendpos = 0;
	int selection = 0;
	int mark_set = 0;

	ProhibitInput();

	_owner->getMiniBuffer()->Set(i18n("Formatting document..."));

	if (_buffer->text) {
		par = _buffer->text->cursor.par;
		pos = _buffer->text->cursor.pos;
		selstartpar = _buffer->text->sel_start_cursor.par;
		selstartpos = _buffer->text->sel_start_cursor.pos;
		selendpar = _buffer->text->sel_end_cursor.par;
		selendpos = _buffer->text->sel_end_cursor.pos;
		selection = _buffer->text->selection;
		mark_set = _buffer->text->mark_set;
		delete _buffer->text;
	}
	//_buffer->text = new LyXText(work_area->w, _buffer);
	_buffer->text = new LyXText(k_canvas->work_area->width(),
				    getFontloader()->getZoom(),
				    _buffer);

	updateScreen();

	if (par) {
		_buffer->text->mark_set = mark_set;
		if (selection) {
			_buffer->text->selection = true;
			_buffer->text->SetCursor(selstartpar,
							       selstartpos);
			_buffer->text->sel_cursor = _buffer->text->cursor;
			_buffer->text->SetCursor(selendpar,
							       selendpos);
			_buffer->text->SetSelection();
			_buffer->text->SetCursor(par, pos);
		} else {
			_buffer->text->SetCursorIntern(par, pos);
			_buffer->text->sel_cursor = _buffer->text->cursor;
			_buffer->text->selection = false;
		}
	}
	
	
	_owner->getMiniBuffer()->Init();
	AllowInput();

	return 0;
}


void BufferView::gotoError()
{
	if (_buffer == 0) return;
	setBufferInternal(_buffer);
	if (!screen)
		return;

	screen->HideCursor(true);
	_buffer->updateFull(false);
	LyXCursor tmp;

	if (!_buffer->text->GotoNextError()) {
		if (_buffer->text->cursor.pos
		    || _buffer->text->cursor.par !=
		    _buffer->text->FirstParagraph()) {
			tmp = _buffer->text->cursor;
			_buffer->text->cursor.par =
				_buffer->text->FirstParagraph();
			_buffer->text->cursor.pos = 0;
			if (!_buffer->text->GotoNextError()) {
				_buffer->text->cursor = tmp;
				_owner->getMiniBuffer()->Set(i18n("No more errors"));
				LyXBell();
			}
		} else {
			_owner->getMiniBuffer()->Set(i18n("No more errors"));
			LyXBell();
		}
	}
	_buffer->updateFull(false);
	_buffer->text->sel_cursor =
		_buffer->text->cursor;
}


// this is a hand-crafted XSync, since some X-Servers seem to optimize XSync in a strange mannor.
static
void waitForX()
{
	static Window w = 0;
	static Atom a = 0;
	if (!a)
		a = XInternAtom(qt_display, "WAIT_FOR_X", False);
	int mask;
	XSetWindowAttributes attr;
	if (w == 0) {
		mask = CWOverrideRedirect;
		attr.override_redirect = 1;
		w = XCreateWindow(qt_display, QApplication::desktop()->handle(),
				  0, 0, 1, 1, 0, CopyFromParent,
				  InputOnly, CopyFromParent, mask, &attr);
		XSelectInput(qt_display, w,
			     PropertyChangeMask);
	}
	XEvent ev;
	XChangeProperty(qt_display, w, a, a, 8,
			PropModeAppend, (unsigned char *)"", 0);
	XWindowEvent(qt_display, w, PropertyChangeMask, &ev);
}



void BufferView::kScrollCB(int v){

	if (_buffer == 0) return;
	setBufferInternal(_buffer);

	current_scrollbar_value = (long)v;
	if (current_scrollbar_value < 0)
		current_scrollbar_value = 0;

	if (!screen)
		return;

	screen->Draw(current_scrollbar_value);

 	waitForX();
}





int BufferView::kMouseMove(int x, int y){
	if (_buffer == 0) return 0;
	if (!screen)
		return 0;

	/* check for inset locking */
	if (_buffer->the_locking_inset){
		LyXCursor cursor = _buffer->text->cursor;
		LyXFont font = _buffer->
			text->GetFont(cursor.par,
				      cursor.pos);
		_buffer->the_locking_inset->
			InsetMotionNotify(x - cursor.x,
					  y -
					  (cursor.y -
					   _buffer->
					   the_locking_inset->Ascent(font)),
					  Button1MotionMask);
		return 0;
	}

// 	/* only use motion with button 1 */
// 	if (!ev->xmotion.state & Button1MotionMask)
// 		return 0;


	/* the selection possible is needed, that only motion events are
	 * used, where the bottom press event was on the drawing area too */
	if (selection_possible) {
	
		screen->HideCursor();
		/* screen->ToggleSelection(); */

		_buffer->text->
			SetCursorFromCoordinates(x, y + screen->first);

		_buffer->updateCursorMove();       /* maybe an empty line was deleted  */

		_buffer->text->SetSelection();
		screen->ToggleToggle();
		if (screen->FitCursor())
			updateScrollbar();
		screen->ShowCursor();
	}
	return 0;
}


void BufferView::PasteSelection(bool asParagraph) {
  /* clear the selection */
  screen->ToggleSelection();
  _buffer->text->ClearSelection();
  _buffer->text->FullRebreak();
  screen->Update();
  updateScrollbar();

  /* get the clipboard contents */
  const char* text = QApplication::clipboard()->text();
  if (!text)
    return;
  if (!asParagraph)
    _buffer->text->InsertStringA(text);
  else
    _buffer->text->InsertStringB(text);

  _buffer->updateFull();
}



extern int bibitemMaxWidth(const class LyXFont &);

int BufferView::kButtonPress(int x, int y, int b){
	if (_buffer == 0) return 0;
	if (!screen)
		return 0 ;
	setBufferInternal(_buffer);

	if (_buffer->the_locking_inset){
	  int x2 = x;
	  int y2 = y;
	  /* we are in inset locking mode. */
	
	  /* check whether the inset was hit. If not reset mode,
	     otherwise give the event to the inset */
	  if (checkInsetHit(x2, y2)){
	    _buffer->the_locking_inset->
	      InsetButtonPress(x2, y2, b);
	    return 0;
	  }
	  else {
	    UnlockInset(_buffer->the_locking_inset);
	  }
	}

	selection_possible = true;
	screen->HideCursor();

	// Right button mouse click on a table
	if (b== 3){
	  if (!_buffer->text->selection){
	    screen->ToggleSelection();
	    _buffer->text->ClearSelection();
	    _buffer->text->FullRebreak();
	    screen->Update();
	    updateScrollbar();
	  }
	  selection_possible = false;
	
	  // check for special insets
	  {
	    LyXCursor tmpcursor = _buffer->text->cursor;
	    _buffer->text->SetCursorFromCoordinates(x, y + screen->first, false);
	    LyXCursor tmpcursor2 = _buffer->text->cursor;
	    int x2 = x;
	    int y2 = y;
	    // Did we hit an inset?
	    Inset* tmpinset = 0;
	    if (checkInsetHit(x2, y2)){
	      tmpinset = _buffer->text->cursor.par->
		GetInset(_buffer->text->cursor.pos);
	    }
	    _buffer->text->cursor = tmpcursor;
	
	    if (tmpinset){
	      if (tmpinset->Editable() &&
		  tmpinset->Editable() != Inset::HIGHLY_EDITABLE){
		QPopupMenu* popup = new QPopupMenu;
		popup->insertItem(tmpinset->EditCommand(), 1 );
		popup->insertSeparator();
		popup->insertItem(i18n( "&Delete" ), 2 );
		popup->move(getKCanvas()->mapToGlobal(QPoint(x,y)));
		int ret = popup->exec();
		delete popup;
		switch (ret) {
		case 1:
		  _buffer->text->SetCursor(tmpcursor2.par, tmpcursor2.pos);
		  _buffer->text->sel_cursor = _buffer->text->cursor;
		  _buffer->text->cursor.x_fix = _buffer->text->cursor.x;
		  _buffer->text->SetCursorParUndo();
		  _owner->getMiniBuffer()->Set(tmpinset->EditMessage());
		  tmpinset->Edit(x,y);
		  break;
		case 2:
		  _buffer->text->SetCursor(tmpcursor2.par, tmpcursor2.pos);
		  _buffer->text->sel_cursor = _buffer->text->cursor;
		  _buffer->text->cursor.x_fix = _buffer->text->cursor.x;
		  _buffer->text->SetCursorParUndo();
		  _buffer->text->cursor.par->Erase(_buffer->text->cursor.pos);
		  _buffer->text->RedoParagraph();
		  _buffer->updateFull();
		  break;
		}
		return 0;
	      }
	    }
	  }

// 	  // set the cursor to the press-position
// 	  _buffer->text->SetCursorFromCoordinates(x, y + screen->first);
	  if (_buffer->text->cursor.par->table ||
	      _buffer->text->MouseHitInTable(x, y+screen->first)) {
		bool doit = true;
		// only show the table popup if the hit is on the table, too
		if (!_buffer->text->
		    HitInTable(_buffer->text->cursor.row,
			       x))
			doit = false;

		// hit above or below the table ?
		if (doit){
			long y_tmp = y + screen->first;
			Row*  row =  _buffer->text->GetRowNearY(y_tmp);
			if (row->par != _buffer->text->cursor.par)
				doit = false;
		}

		if (doit){

		  QPopupMenu* popup = new QPopupMenu;
 		  _owner->getMenuReceiver()->connect(popup,
 						     SIGNAL(activated(int)),
 						     SLOT(m_editTableActivated(int)));
		  _owner->getMenuReceiver()->createTableMenu(popup);
		  popup->move(getKCanvas()->mapToGlobal(QPoint(x,y)));
		  popup->exec();
		  delete popup;
		  return 0;
		}
	  }

	  // TODO more context popups. Float labels, Figures etc.

	  {
	    QPopupMenu* popup = new QPopupMenu;
	    _buffer->updateFull(false);
// 	    _owner->getMenuReceiver()->connect(popup,
// 					       SIGNAL(activated(int)),
// 					       SLOT(activated(int)));
	    _owner->getMenuReceiver()->createEditMenu(popup);
	    popup->move(getKCanvas()->mapToGlobal(QPoint(x,y)));
	    int result = popup->exec();
	    if (result > -1)
	      _owner->getLyXFunc()->Dispatch(result);
	    delete popup;
	    return 0;
	  }
	  return 0;
	}


	// Middle button press pastes if we have a selection
	bool paste_internally = false;
	if (b == 2 && !_buffer->the_locking_inset
	    && _buffer->text->selection){
		_owner->getLyXFunc()->Dispatch(LFUN_COPY);
		paste_internally = true;
	}

	/* clear the selection */
	screen->ToggleSelection();
	_buffer->text->ClearSelection();
	_buffer->text->FullRebreak();
	screen->Update();
	updateScrollbar();


	_buffer->text->SetCursorFromCoordinates(x, y + screen->first);
	screen->Update();
	_buffer->text->FinishUndo();
	_buffer->text->sel_cursor = _buffer->text->cursor;
	_buffer->text->cursor.x_fix = _buffer->text->cursor.x;

	/* Single left click */
	if (b == 1){
		int x2 = x;
		int y2 = y;
		// Did we hit an inset?
		if (checkInsetHit(x2, y2)){
			if (_buffer->text->cursor.par->
			    GetInset(_buffer->text->cursor.pos)->
			    Editable() == Inset::HIGHLY_EDITABLE)
				{
					// Highly editable inset, like math
					selection_possible = false;
					_owner->updateToolbars();
					_owner->getMiniBuffer()->
                                                Set(_buffer->text->cursor.par->
                                                    GetInset(_buffer->text->cursor.pos)->
                                                    EditMessage());
					_buffer->text->cursor.par->
						GetInset(_buffer->text->cursor.pos)->
						Edit(x2, y2);
					return 0;
				} else {
					// Other inset like error, notes
					// and figures
					selection_possible = false;
					if (_buffer->text->cursor.par->
					    GetInset(_buffer->text->cursor.pos)->
					    Editable() != Inset::EDITABLE_NO_UNDO)
					  _buffer->text->
					    SetCursorParUndo();
					_owner->getMiniBuffer()->
						Set(_buffer->
						    text->cursor.par->
						    GetInset(_buffer->
							     text->
							     cursor.pos)->
						    EditMessage());
					_buffer->
						text->cursor.par->
						GetInset(_buffer->
							 text->cursor.pos)->
						Edit(x, y);
					return 0;
				}
		}

		// check whether we want to open a float
		if (_buffer->text &&
		    _buffer->text->cursor.pos <
		    _buffer->text->cursor.par->Last()) {
			char c = _buffer->text->cursor.par->
				GetChar(_buffer->text->cursor.pos);
			if (c == LYX_META_FOOTNOTE || c == LYX_META_MARGIN
			    || c == LYX_META_FIG || c == LYX_META_TAB
			    || c == LYX_META_WIDE_FIG || c == LYX_META_WIDE_TAB
                            || c == LYX_META_ALGORITHM){
				ToggleFloat();
				selection_possible = false;
				return 0;
			}
		}

		// do we want to close a float (click on the float-label)
		if (_buffer->text->cursor.row->par->footnoteflag ==
		    LyXParagraph::OPEN_FOOTNOTE
		    && _buffer->text->cursor.pos == 0
		    && _buffer->text->cursor.row->previous &&
		    _buffer->text->cursor.row->previous->par->
		    footnoteflag != LyXParagraph::OPEN_FOOTNOTE){
			LyXFont font (LyXFont::ALL_SANE);
			font.setSize(LyXFont::SIZE_SMALL);

			int box_x = 20; // LYX_PAPER_MARGIN;
			box_x += font.textWidth("Mwide-figM", 10);

			if (x < box_x
			    && y + screen->first
			    > _buffer->text->cursor.y -
			    _buffer->text->cursor.row->baseline
			    && y + screen->first
			    < _buffer->text->cursor.y -
			    _buffer->text->cursor.row->baseline
			    + font.maxAscent()*1.2 +
			    font.maxDescent()*1.2){
				ToggleFloat();
				selection_possible = false;
				return 0;
			}
		}
	
	        // Maybe we want to edit a bibitem ale970302
	        if (_buffer->text->cursor.par->bibkey &&
		    x2 < 20+bibitemMaxWidth(lyxstyle.TextClass(_buffer->
					    params.textclass)->defaultfont)) {
			_buffer->text->cursor.par->bibkey->Edit(0, 0);
		}
	}

	_owner->updateToolbars();
	if (screen->FitCursor()){
		updateScrollbar();
		selection_possible = false;
	}

	// insert primary selection with middle mouse
	// if there is a local selection in the current buffer, insert this
	if (b == 2 && !_buffer->the_locking_inset){
		if (paste_internally)
			_owner->getLyXFunc()->Dispatch(LFUN_PASTE);
		else
			_owner->getLyXFunc()->Dispatch(LFUN_PASTESELECTION,
						       "line");
		selection_possible = false;
	}
	return 0;
}


int BufferView::kButtonDoubleClick(int /* x */ , int /* y */ , int b){
  if (_buffer == 0) return 0;
  setBufferInternal(_buffer);
  if (!screen)
    return 0 ;
  if (_buffer &&
      !_buffer->the_locking_inset) {
    if (screen &&
	b == 1) {
      screen->HideCursor();
      screen->
	ToggleSelection();
      _buffer->text->SelectWord();
      screen->
	ToggleSelection(false);
      _buffer->updateFull(false); /* this will fit the cursor on the screen
				 * if necessary */
    }
  }
  return 0;
}


// Callback for cursor timer
void BufferView::kCursorToggleCB(){
	
  /* quite a nice place for asyncron Inset updating, isn't it? */
  // actually no! This is run even if no buffer exist... so (Lgb)
  if (!_buffer)
    return;

  if (InsetUpdateList)
    UpdateInsetUpdateList();

  if (!screen)
    return;

  if (!_buffer->the_locking_inset){
    screen->CursorToggle();
  } else {
    _buffer->the_locking_inset->
      ToggleInsetCursor();
  }
}

void BufferView::kHideCursor(){
  if (!_buffer || !screen)
    return;
  if (!_buffer->the_locking_inset){
    if (screen->cursor_visible)
      screen->HideCursor();
  } else {
    if (_buffer->the_locking_inset->isCursorVisible())
      _buffer->the_locking_inset->ToggleInsetCursor();
  }
}

void BufferView::kKeyPress(XEvent *ev){
  if (_buffer == 0) return;
  setBufferInternal(_buffer);
  getOwner()->getLyXFunc()->processKeyEvent(ev);
}


FontLoader* BufferView::getFontloader(){
  return getOwner()->getFontloader();
}

MiniBuffer* BufferView::getMiniBuffer(){
  return getOwner()->getMiniBuffer();
}


int BufferView::getHeight(){
  return k_canvas->work_area->height();
}


void BufferView::startUpdateTimer(float timer){
  getKCanvas()->startUpdateTimer(timer);
}

void BufferView::stopUpdateTimer(){
  getKCanvas()->stopUpdateTimer();
}


int BufferView::kButtonRelease(int x, int y, int b){
	if (_buffer == 0) return 0;
	setBufferInternal(_buffer);

	if (screen && _buffer->the_locking_inset){

		/* we are in inset locking mode. */
		/* LyX does a kind of work-area grabbing for insets.
		   only a ButtonPress Event outside the inset will
		   force a InsetUnlock */
		checkInsetHit(x, y); /* just to adept the coordinates */
		_buffer->the_locking_inset->
			InsetButtonRelease(x, y, b);
		return 0;
	}

	selection_possible = false;
        if (_buffer->text->cursor.par->table) {
                int cell = _buffer->text->
                        NumberOfCell(_buffer->text->cursor.par,
                                     _buffer->text->cursor.pos);
                if (_buffer->text->cursor.par->table->IsContRow(cell) &&
                    _buffer->text->cursor.par->table->
                    CellHasContRow(_buffer->text->cursor.par->table->
                                   GetCellAbove(cell))<0) {
                        _buffer->text->CursorUp();
                }
        }
	return 0;
}


bool BufferView::checkInsetHit(int &x, int &y)
	/* returns 1 if inset was hit. Coordinates are set relativly
	   to the inset. Otherwise coordinates doesn't change their
	   values and 0 is returned.
	   */
{
	if (!getScreen())
		return false;
	setBufferInternal(_buffer);

	int y_tmp = y + getScreen()->first;

	LyXCursor cursor = _buffer->text->cursor;
	if (cursor.pos
	    < cursor.par->Last()
	    && cursor.par->GetChar(cursor.pos) == LYX_META_INSET
	    && cursor.par->GetInset(cursor.pos)
	    && cursor.par->GetInset(cursor.pos)->Editable()) {

		/* check wether the inset was really hit */
		Inset* tmpinset = cursor.par->GetInset(cursor.pos);
		LyXFont font = _buffer->text->GetFont(cursor.par, cursor.pos);
		if (x > cursor.x
		    && x < cursor.x + tmpinset->Width(font)
		    && y_tmp > cursor.y - tmpinset->Ascent(font)
		    && y_tmp < cursor.y + tmpinset->Descent(font)){
			x = x - cursor.x;
			y = y_tmp - (cursor.y); // now the origin of an inset
			// is on the baseline (for mathed) (Matthias)
			return true;
		}
	}
	return false;
}


void BufferView::cursorPrevious()
{
	if (!currentBuffer()->text->cursor.row->previous) return;
	setBufferInternal(_buffer);
	
	long y = getScreen()->first;
	Row* cursorrow = currentBuffer()->text->cursor.row;
	currentBuffer()->text->
	  SetCursorFromCoordinates(currentBuffer()->text->
				   cursor.x_fix,
				   y);
	currentBuffer()->text->FinishUndo();
	/* this is to allow jumping over large insets */
	if ((cursorrow == currentBuffer()->text->cursor.row))
	  currentBuffer()->text->CursorUp();
	
  	if (currentBuffer()->text->cursor.row->height < k_canvas->work_area->height())
	  getScreen()->Draw(currentBuffer()->text->cursor.y
  			    - currentBuffer()->text->cursor.row->baseline
  			    + currentBuffer()->text->cursor.row->height
  			    - k_canvas->work_area->height() +1 );
}


void BufferView::cursorNext()
{
	if (!currentBuffer()->text->cursor.row->next) return;
	setBufferInternal(_buffer);
	
	long y = getScreen()->first;
	currentBuffer()->text->GetRowNearY(y);
	Row* cursorrow = currentBuffer()->text->cursor.row;
	currentBuffer()->text->
		SetCursorFromCoordinates(currentBuffer()->text->
					 cursor.x_fix,
					 y + k_canvas->work_area->height());
	currentBuffer()->text->FinishUndo();
	/* this is to allow jumping over large insets */
	if ((cursorrow == currentBuffer()->text->cursor.row))
	  currentBuffer()->text->CursorDown();
	
 	if (currentBuffer()->text->cursor.row->height < k_canvas->work_area->height())
 	  getScreen()->Draw(currentBuffer()->text->cursor.y
 			    - currentBuffer()->text->cursor.row->baseline);
}


bool BufferView::available() const
{
	if (_buffer && _buffer->text) return true;
	return false;
}


void BufferView::savePosition()
{
	backstack->push(currentBuffer()->getFileName(),
			currentBuffer()->text->cursor.x,
			currentBuffer()->text->cursor.y);
}


void BufferView::restorePosition()
{
	int  x, y;

	if (backstack->isEmpty())
	  return;

	setBufferInternal(_buffer);

	LString fname = backstack->pop(&x, &y);
	
	screen->HideCursor(true);
 	Buffer *b = (bufferlist.exists(fname)) ? bufferlist.getBuffer(fname):
 		bufferlist.loadLyXFile(fname); // don't ask, just load it
	if (b) {
	  setBufferInternal(b);
	
	  currentBuffer()->text->SetCursorFromCoordinates(x, y);
	  currentBuffer()->updateFull(false);
	}
}


// Local Variables:
// mode: C++
// c-file-style: "Stroustrup"
// End:
