//-< CURSOR.H >------------------------------------------------------*--------*
// GigaBASE                  Version 1.0         (c) 1999  GARRET    *     ?  *
// (Post Relational Database Management System)                      *   /\|  *
//                                                                   *  /  \  *
//                          Created:     20-Nov-98    K.A. Knizhnik  * / [] \ *
//                          Last update: 10-Dec-98    K.A. Knizhnik  * GARRET *
//-------------------------------------------------------------------*--------*
// Table cursor
//-------------------------------------------------------------------*--------*

#ifndef __CURSOR_H__
#define __CURSOR_H__

class dbOrderByNode;

class dbSelection { 
  public:
    enum { quantum = 1024 };
    class segment { 
      public:
	segment* prev;
	segment* next;
	size_t   nRows;
	oid_t    rows[quantum];

	segment(segment* after) { 
	    prev = after;
	    next = NULL;
	    nRows = 0;
	}	
    };
    segment*  first;
    segment*  last;
    segment*  curr;
    size_t    nRows;
    size_t    pos;

    void add(oid_t oid) { 
	if (last == NULL) { 
	    first = last = new segment(NULL);
	} else if (last->nRows == quantum) { 
	    last = last->next = new segment(last);
	}
	last->rows[last->nRows++] = oid;
	nRows += 1;
    }
    
    void sort(dbDatabase* db, dbOrderByNode* order);
    static int compare(dbRecord* a, dbRecord* b, dbOrderByNode* order);

    static int exactKeyCmp(void const* a, void const* b);

    dbSelection() { 
	nRows = 0;
	pos = 0;
	first = curr = last = NULL;
    }
    void reverse();
    void reset();
};

enum dbCursorType { 
    dbCursorViewOnly,
    dbCursorForUpdate
};

class dbL2List { 
  public:
    dbL2List* next; 
    dbL2List* prev; 

    void link(dbL2List* elem) { 
	elem->prev = this;
	elem->next = next;
	next = next->prev = elem;
    }
    void unlink() { 
#ifdef __INSURE__
        if (((void*) next == (void*) prev) && 
        ((void*) next == (void*) this)) return;
#endif	
	next->prev = prev;
	prev->next = next;
        next = prev = this;
    }
    bool isEmpty() { 
	return next == this;
    }
    dbL2List() { 
	next = prev = this;
    }
    ~dbL2List() { 
	unlink();
    }
};


class dbAnyCursor : public dbL2List { 
    friend class dbDatabase;
    friend class dbHashTable;
    friend class dbBtreePage;
    friend class dbBtreeLeafPage;
    friend class dbSubSql;
  public:
    int getNumberOfRecords() { return selection.nRows; }

    void remove();
    
    bool isEmpty() { return currId == 0; }

    bool isLimitReached() { return selection.nRows >= limit; }

    int select(dbQuery& query, dbCursorType aType) {
	type = aType;
	reset();
	table.db->select(this, query); 
	if (gotoFirst() && prefetch) { 
	    fetch();
	}
	return selection.nRows;
    } 
 
    int select(dbQuery& query) { 
	return select(query, defaultType);
    }

    int select(char const* condition, dbCursorType aType) { 
	type = aType;
	dbQuery query(condition);
	return select(query);
    } 

    int select(char const* condition) { 
	return select(condition, defaultType);
    }

    int select(dbCursorType aType) { 
	type = aType;
	reset();
	table.db->select(this); 
	if (gotoFirst() && prefetch) { 
	    fetch();
	}
	return selection.nRows;
    } 

    int select() {
	return select(defaultType);
    }

    void update() { 
	assert(type == dbCursorForUpdate && currId != 0);
	table.db->update(currId, &table, record);
    }

    void removeAll() {
	table.db->deleteTable(&table);
	reset();
    }

    void removeAllSelected();

    void setSelectionLimit(size_t lim) { limit = lim; }
    
    void unsetSelectionLimit() { limit = dbDefaultSelectionLimit; }

    void setPrefetchMode(bool mode) { prefetch = mode; }

    void reset();

  protected: 
    dbTableDescriptor& table;
    dbCursorType       type;
    dbCursorType       defaultType;
    dbSelection        selection;
    bool               allRecords;
    oid_t              firstId;
    oid_t              lastId;
    oid_t              currId;
    byte*              record;
    size_t             limit;
    dbGetTie           tie;

    int4*              bitmap; // bitmap to avoid duplicates
    size_t             bitmapSize;
    bool               eliminateDuplicates;
    bool               prefetch;
    
    void checkForDuplicates() { 
	if (!eliminateDuplicates && limit > 1) { 
	    eliminateDuplicates = true;
	    size_t size = (table.db->currIndexSize + 31) / 32;
	    if (size > bitmapSize) { 
		delete[] bitmap;
		bitmap = new int4[size];
		bitmapSize = size;
	    }
	    memset(bitmap, 0, size*4);
	}
    }

    bool isMarked(oid_t oid) { 
	return bitmap != NULL && (bitmap[oid >> 5] & (1 << (oid & 31))) != 0;
    }

    void mark(oid_t oid) { 
	if (bitmap != NULL) { 
	    bitmap[oid >> 5] |= 1 << (oid & 31);
	}
    }

    bool add(oid_t oid) { 
	if (selection.nRows < limit) { 
	    if (eliminateDuplicates) { 
		if (bitmap[oid >> 5] & (1 << (oid & 31))) { 
		    return true;
		}
		bitmap[oid >> 5] |= 1 << (oid & 31);
	    } 
	    selection.add(oid);
	    return selection.nRows < limit;
	} 
	return false;
    }

    bool gotoNext();
    bool gotoPrev(); 
    bool gotoFirst();
    bool gotoLast();
    
    void setCurrent(dbAnyReference const& ref); 

    void fetch() { 
	table.colons->fetchRecordFields(record, 
					(byte*)table.db->getRow(tie, currId));
    }

    dbAnyCursor(dbTableDescriptor& aTable, dbCursorType aType, byte* rec)
    : table(aTable),defaultType(aType),allRecords(false),currId(0),record(rec)
    {
	limit = dbDefaultSelectionLimit;
	prefetch = true;
	bitmap = NULL; 
	bitmapSize = 0;
	eliminateDuplicates = false;
    }
    ~dbAnyCursor();
};

template<class T>
class dbCursor : public dbAnyCursor { 
  protected:
    T record;
    
  public:
    dbCursor(dbCursorType type = dbCursorViewOnly) 
        : dbAnyCursor(T::dbDescriptor, type, (byte*)&record) {}

    T* get() { 
	return currId == 0 ? (T*)NULL : &record; 
    }
    T* next() { 
	if (gotoNext()) { 
	    fetch();
	    return &record;
	}
	return NULL;
    }
    T* prev() {	
	if (gotoPrev()) { 
	    fetch();
	    return &record;
	}
	return NULL;
    }
    T* first() { 
	if (gotoFirst()) {
	    fetch();
	    return &record;
	}
	return NULL;
    }
    T* last() { 
	if (gotoLast()) {
	    fetch();
	    return &record;
	}
	return NULL;
    }    
    T* operator ->() { 
	assert(currId != 0);
	return &record;
    }
    T* at(dbReference<T> const& ref) { 
	setCurrent(ref);
	return &record;
    }
    dbReference<T> currentId() { 
	return dbReference<T>(currId);
    }
};

class dbParallelQueryContext { 
  public:
    dbDatabase* const      db;
    dbCompiledQuery* const query;
    oid_t                  firstRow;
    dbTableDescriptor*     table;
    dbSelection            selection[dbMaxParallelSearchThreads];

    void search(int i); 

    dbParallelQueryContext(dbDatabase* aDb, dbTableDescriptor* desc, 
			   dbCompiledQuery* aQuery)
      : db(aDb), query(aQuery), firstRow(desc->firstRow), table(desc) {}
};


extern char* strupper(char* s);

extern char* strlower(char* s);

#endif
