/* 
    xWinTV using Bt848 frame grabber driver

    Copyright (C) 1998 Moritz Wenk (wenk@mathematik.uni-kl.de)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <wintv.h>

#include <wintvchannel.h>

#include <kwm.h>
#include <kiconloader.h>
#include <klocale.h>

#define BID_ADD     0
#define BID_EDIT    1
#define BID_DELETE  2
#define BID_SL      3
#define BID_SC      4
#define BID_MUP     5
#define BID_MDOWN   6 

#ifdef myDEBUG
#define chanDEBUG
#endif

/* ------------------------------------------------------------------------------*/
/* ------------------------------------------------------------------------------*/

#define PID_MUTE 0
#define PID_SPLIT 1

winSlider::winSlider( int _min, int _max, int _val, bool _stereo, bool _muteable, 
		      QPixmap * pix, QWidget * parent, const char * name )
  : QWidget( parent, name ), min(_min), max(_max), lval(_val), 
    rval(_val), stereo(_stereo), muteable(_muteable)
{
  sleft = new QSlider( this );
  sleft->setMinimumSize( 10, 10 );
  sleft->setMaximumSize( 32767, 32767 );
  sleft->setRange( min, max );
  sleft->setSteps( 1, 10 );
  sleft->setValue( max-lval );
  sleft->setTracking( TRUE );
  sleft->setTickmarks( QSlider::Left );
  sleft->setTickInterval( 10 );

  sright = new QSlider( this );
  sright->setMinimumSize( 10, 10 );
  sright->setMaximumSize( 32767, 32767 );
  sright->setRange( min, max );
  sright->setSteps( 1, 10 );
  sright->setValue( max-rval );
  sright->setTracking( TRUE );
  sright->setTickmarks( QSlider::Right );
  sright->setTickInterval( 10 );

  icon= 0;
  if ( pix ) {
    icon = new QLabel( this );
    icon->setGeometry(0,0,20,20);
    icon->setPixmap( *pix );
  }
  
  connect(sright,SIGNAL(valueChanged(int)),this,SLOT(valueRChangedI(int)));
  connect(sleft,SIGNAL(valueChanged(int)),this,SLOT(valueLChangedI(int)));

  sPopup=new QPopupMenu;
  if ( muteable ) sid[PID_MUTE]=sPopup->insertItem(i18n("&Mute"));
  if ( stereo ) sid[PID_SPLIT]=sPopup->insertItem(i18n("Un&split"));
  connect(sPopup,SIGNAL(activated(int)),this,SLOT(popupSLOT(int)));

  sright->installEventFilter( this );
  sleft->installEventFilter( this );

  setMinimumSize( 20 , 60 );

  splited= true;

  if ( !stereo ) {
    stereo=true;
    split(false);
    stereo=false;
  }
  
  muted=false;
  mm=true;
}

winSlider::~winSlider ()
{
  delete sleft;
  delete sright;
}

void winSlider::popupSLOT( int id )
{
#ifdef chanDEBUG
  debug("winSlider: popup %d",id);
#endif 
  if ( id == sid[PID_MUTE] ) mute(!muted); 
  if ( id == sid[PID_SPLIT] )split(!splited);
}

void winSlider::setLValue( int v )
{
#ifdef chanDEBUG
  debug("winSlider: setLValue %d",lval);
#endif
  lval=v;
  sleft->setValue( max-v );
  if ( !splited ) {
    sright->setValue( max-v );
  }
}

void winSlider::setRValue( int v )
{
#ifdef chanDEBUG
  debug("winSlider: setRValue %d",lval);
#endif
  rval=v;
  sright->setValue( max-v );
}

void winSlider::valueLChangedI( int v )
{
  lval=max-v;
  if ( !splited ) {
    rval=max-v;
    sright->setValue( v );
  }
  emit valueChanged( rval, lval );
}

void winSlider::valueRChangedI( int v )
{
  rval=max-v;
  emit valueChanged( rval, lval );
}

void winSlider::split( bool sp )
{
  if ( !stereo ) return;
  if ( splited != sp ) {
    splited= sp;
    if ( !splited ) {
#ifdef chanDEBUG
  debug("winSlider: split %d",sp);
#endif 
      sPopup->changeItem(i18n("&Split"),sid[PID_SPLIT]);
      //sleft->setTickmarks( QSlider::Both );
      sright->hide();
      int val= (lval+rval)/2;
      sleft->setValue( max-val);
      sright->setValue( max-val );
    } else {
#ifdef chanDEBUG
  debug("winSlider: split %d",sp);
#endif 
      sPopup->changeItem(i18n("Un&split"),sid[PID_SPLIT]);
      sleft->setTickmarks( QSlider::Left );
      sright->setValue( max-rval );
      sright->show();
    }
    resizeEvent(NULL);
    emit splitChanged( sp );
  }
}

void winSlider::mute( bool on )
{
  if ( !muteable ) return;
#ifdef chanDEBUG
  debug("winSlider: mute %d",on);
#endif 
  muted= on;
  if ( muted ) {
    sleft->setBackgroundColor( black ); 
    sright->setBackgroundColor( black ); 
    sPopup->changeItem(i18n("&Unmute"),sid[PID_MUTE]);
  } else {
    sleft->setBackgroundColor( colorGroup().mid() ); 
    sright->setBackgroundColor( colorGroup().mid() ); 
    sPopup->changeItem(i18n("&Mute"),sid[PID_MUTE]);
  }
  emit muteChanged( on );
#ifdef chanDEBUG
  debug("winSlider: mute");
#endif
}
void winSlider::setEnabled( bool on )
{
  mm=on;
  sleft->setEnabled(on);
  sright->setEnabled(on);
}

void winSlider::resizeEvent ( QResizeEvent *  )
{
  if ( icon ) {
    icon->setGeometry( width()/2 - 10, 0, 20, 20 );
    yoff= 25;
  } else {
    yoff= 0;
  }
  if (splited) {
    sleft->setGeometry( 0, yoff, width()/2-1, height()-yoff );
    sright->setGeometry( sleft->width()+2, yoff, width()/2-1, height()-yoff );
  } else {
    sleft->setGeometry( 0, yoff, width(), height()-yoff );
  }
}

bool winSlider::eventFilter( QObject *o, QEvent *e )
{
  // Lets see, if we have a "Right mouse button press"
  if (e->type() == Event_MouseButtonPress) {
    QMouseEvent *qme = (QMouseEvent*)e;
    if (qme->button() == RightButton) {
      sPopup->popup(QCursor::pos()/*mapToGlobal(qme->pos())*/);
      return true;
    }
  }
  return false;
}

/* ------------------------------------------------------------------------------*/
/* ------------------------------------------------------------------------------*/

winMixer::winMixer( rcfile * _rc,
		    QWidget * parent, const char * name )
  : rc(_rc)
{
#ifdef chanDEBUG
  debug("winMixer: init begin");
#endif 
  KIconLoader *loader = kapp->getIconLoader();
  QPixmap pm = loader->loadIcon("kwintv_16x16.xpm");
  setIcon(pm);
  setIconText("Mixer");
  setCaption("Mixer");

  // init sound mixer
  mvol= new mixer((const char *)rc->mixerDeviceName,(unsigned int)rc->mixerDevice);
  rc->mixerDevice=(int)mvol->getDevice();
  mtreble= new mixer((const char *)rc->mixerDeviceName,SOUND_MIXER_TREBLE,false);
  mbass= new mixer((const char *)rc->mixerDeviceName,SOUND_MIXER_BASS,false);

  pm = loader->loadIcon("mix_volume.xpm");
  volmix= new winSlider(0,MIXER_MAX_VALUE,0,mvol->isStereoDevice(),true,&pm,this);
  if ( !rc->sound_on ) volmix->mute( true ); // sound_on == false -> mute sound

  pm = loader->loadIcon("mix_treble.xpm");
  treblemix= new winSlider(0,MIXER_MAX_VALUE,0,mtreble->isStereoDevice(),true,&pm,this);

  pm = loader->loadIcon("mix_bass.xpm");
  bassmix= new winSlider(0,MIXER_MAX_VALUE,0,mbass->isStereoDevice(),true,&pm,this);

  logopm = loader->loadIcon("kwintv_48x48.xpm");
  logo = new QPushButton( this );
  logo->setPixmap( logopm );
  logo->resize(logopm.width(),logopm.height());

  logowidth= logo->width();
  logoheight= logo->height();

  setMinimumSize( 25*3+30+logo->width()+15+6 , 20+10+100 );
  resize( 25*3+30+logo->width()+15+6 , 20+10+100 );

  volmix->split( rc->mix_vol_split );
  treblemix->split( rc->mix_treble_split );
  bassmix->split( rc->mix_bass_split );

  if ( rc->use_master_volume ) {
    volmix->setRValue(mvol->getVolume(VOL_RIGHT));
    volmix->setLValue(mvol->getVolume(VOL_LEFT));
  } else {
    volmix->setRValue(rc->volume_right);
    volmix->setLValue(rc->volume_left);
  }

  bassmix->setLValue(mbass->getVolume(VOL_LEFT));
  bassmix->setRValue(mbass->getVolume(VOL_RIGHT));

  treblemix->setLValue(mtreble->getVolume(VOL_LEFT));
  treblemix->setRValue(mtreble->getVolume(VOL_RIGHT));
  
  if ( mvol->getDevice() == NO_MDEV ) volmix->setEnabled(false);
  if ( (mtreble->getDevice() == NO_MDEV) ) treblemix->setEnabled(false);
  if ( (mbass->getDevice() == NO_MDEV) ) bassmix->setEnabled(false);

  connect(volmix,SIGNAL(valueChanged(int,int)),this,SLOT(vol(int,int)));
  connect(volmix,SIGNAL(muteChanged(bool)),this,SLOT(volMuteI(bool)));
  connect(volmix,SIGNAL(splitChanged(bool)),this,SLOT(changeSize(bool)));

  connect(treblemix,SIGNAL(valueChanged(int,int)),this,SLOT(treble(int,int)));
  connect(treblemix,SIGNAL(splitChanged(bool)),this,SLOT(changeSize(bool)));

  connect(bassmix,SIGNAL(valueChanged(int,int)),this,SLOT(bass(int,int)));
  connect(bassmix,SIGNAL(splitChanged(bool)),this,SLOT(changeSize(bool)));

  connect(logo,SIGNAL(released()),this,SLOT(showMixer()));

  QToolTip::add(volmix,i18n("Volume"));
  QToolTip::add(treblemix,i18n("Bass"));
  QToolTip::add(bassmix,i18n("Treble"));
  QToolTip::add(logo,i18n("Close Mixer"));

  changeSize(false);

#ifdef chanDEBUG
  debug("winMixer: init end");
#endif 
}

winMixer::~winMixer()
{
  rc->mix_bass_split= bassmix->isSplited();
  rc->mix_treble_split= treblemix->isSplited();
  rc->mix_vol_split= volmix->isSplited();

  delete mvol;
  delete mtreble;
  delete mbass;
  
  delete volmix;
  delete treblemix;
  delete bassmix;
}

void winMixer::setVolume( int rval, int lval )
{
  if ( rc->volume_right==rval && rc->volume_left == lval) return;
#ifdef chanDEBUG
  debug("winMixer: setVolume %d,%d -> %d,%d",rc->volume_right,rc->volume_left,rval,lval);
#endif 
  rc->volume_right=rval;
  rc->volume_left=lval;
  volmix->setRValue(rval);
  volmix->setLValue(lval);
}
void winMixer::setTreble( int rval, int lval )
{
  treblemix->setRValue(rval);
  treblemix->setLValue(lval);
}
void winMixer::setBass( int rval, int lval )
{
  bassmix->setRValue(rval);
  bassmix->setLValue(lval);
}
void winMixer::showMixer()
{
  isVisible()?hide():show();
}

void winMixer::vol( int rval, int lval )
{
  mvol->setStereoVolume(rval,lval);
  rc->volume_right=rval;
  rc->volume_left=lval;
  emit volChanged(rval,lval);
}
void winMixer::volMute( bool on ) // on==true -> mute ( -> rc->sound_on=false )
{
#ifdef chanDEBUG
  debug("winMixer: volMute %d",on);
#endif
  //if ( on == !rc->sound_on ) return;
  rc->sound_on=!on;
  volmix->mute(on);
}
void winMixer::volMuteI( bool on ) // on==true -> mute ( -> rc->sound_on=false )
{
  if ( on == !rc->sound_on ) return;
#ifdef chanDEBUG
  debug("winMixer: volMuteI %d",on);
#endif
  rc->sound_on= !on;
  emit volMuteChanged( rc->sound_on );
}
void winMixer::bass( int rval, int lval )
{
  mbass->setStereoVolume(rval,lval);
}
void winMixer::treble( int rval, int lval )
{
  mtreble->setStereoVolume(rval,lval);
}
void winMixer::changeSize( bool )
{
#ifdef chanDEBUG
  debug("winMixer: changeSize");
#endif 
  xoff=logowidth+40+6
    +(volmix->isSplited()?50:25)+(treblemix->isSplited()?50:25)+(bassmix->isSplited()?50:25);
  resize(xoff,height());
  setFixedSize(xoff,height());
}

void winMixer::resizeEvent ( QResizeEvent * )
{
#ifdef chanDEBUG
  debug("winMixer: resizeEvent");
#endif 
  volmix->setGeometry(5,5,volmix->isSplited()?50:25,height()-10);

  int ww= (volmix->isSplited()?50:25);
  treblemix->setGeometry(ww+15,5,treblemix->isSplited()?50:25,height()-10);

  ww+= (treblemix->isSplited()?50:25);
  bassmix->setGeometry(ww+25,5,bassmix->isSplited()?50:25,height()-10);

  ww+= (bassmix->isSplited()?50:25);
  logo->setGeometry(ww+35,(height()-logoheight+6)/2,logowidth+6,logoheight+6);
}

/* ------------------------------------------------------------------------------*/
/* ------------------------------------------------------------------------------*/

winTVChannelWidget::winTVChannelWidget(QWidget* parent, const char* name, 
				       rcfile *_rc)
  : KTopLevelWidget( name ),rc(_rc)
{
#ifdef chanDEBUG
  debug("channel: begin of init");
#endif

  KWM::setIcon(winId(), kapp->getIcon());
  KWM::setMiniIcon(winId(), kapp->getMiniIcon());

  setCaption((const char*)(QString("program box")));
  // set icon

  KIconLoader *loader = kapp->getIconLoader();

  pm = loader->loadIcon("kwintv_logo.xpm");
  setIcon(pm);
  setIconText("Program Box");

  // toolbar
  toolbar = new KToolBar(this,"kwintv -- ToolBar --",26);
  toolbar->setBarPos(KToolBar::Top);
  addToolBar(toolbar);

  ch_add_xpm     = loader->loadIcon("filenew.xpm");
  ch_edit_xpm    = loader->loadIcon("editpaste.xpm");
  ch_delete_xpm  = loader->loadIcon("editcut.xpm");
  mup_xpm        = loader->loadIcon("up.xpm");
  mdown_xpm      = loader->loadIcon("down.xpm");
  cs_xpm         = loader->loadIcon("delete.xpm");
  
  toolbar->insertButton(ch_add_xpm,BID_ADD,SIGNAL(released(int)),this,
			SLOT(buttonsSLOT(int)),true,i18n("add new program"));
  toolbar->insertButton(ch_edit_xpm,BID_EDIT,SIGNAL(released(int)),this,
			SLOT(buttonsSLOT(int)),true,i18n("edit this program"));
  toolbar->insertButton(ch_delete_xpm,BID_DELETE,SIGNAL(released(int)),this,
			SLOT(buttonsSLOT(int)),true,i18n("delete this program"));
#if 1
  toolbar->insertSeparator();
  
  toolbar->insertButton(mup_xpm,BID_MUP,SIGNAL(released(int)),this,
			SLOT(buttonsSLOT(int)),true,i18n("move up"));
  toolbar->insertButton(mdown_xpm,BID_MDOWN,SIGNAL(released(int)),this,
			SLOT(buttonsSLOT(int)),true,i18n("move down"));
#endif
  toolbar->insertSeparator();

  toolbar->insertButton(cs_xpm,BID_SC,SIGNAL(released(int)),this,
			SLOT(buttonsSLOT(int)),true,i18n("hide program box"));

  channelbox= new winWidget(this);
  setView(channelbox);

  connect(channelbox,SIGNAL(highlighted(int)),this,SLOT(channelSelectedEmitSLOT(int)));
  connect(channelbox,SIGNAL(channelEditSelectSIGNAL()),this,SLOT(channelEditEmitSLOT()));
  connect(channelbox,SIGNAL(moveItemSIGNAL( int, int )),this,SLOT(moveItemSLOT( int, int )));

  setMinimumSize( 180,100 );
  //  resize( 200,230 );
  resize( 210,260 );
}

/* ------------------------------------------------------------------------------*/

winTVChannelWidget::~winTVChannelWidget()
{
  delete toolbar;
  delete channelbox;
}

/* ------------------------------------------------------------------------------*/

void winTVChannelWidget::newChannelListSLOT( QList<channel> *clist )
{
  if ( clist->isEmpty() ) return;

  clistp= clist;

  int id= clist->at();

  inserting=TRUE;
  channelbox->clear();
  
  channel *c= clist->first();
  while ( c ) {
    c->id=clist->at();
    channelbox->insertItem( c, inserting );
    c= clist->next();
  }
  inserting=FALSE;

  clist->at(id);

#ifdef chanDEBUG
  debug("channel: channel list insertion done %d, %d",clist->count(),clist->at());
#endif
}

/* ------------------------------------------------------------------------------*/

void winTVChannelWidget::moveItemSLOT( int from, int to )
{
  if ( from == to || from == -1 || to == -1 || from == clistp->count() || to == clistp->count() )
    return;

#ifdef chanDEBUG
  debug("channel: move %d -> %d",from,to);
#endif

  inserting=TRUE;

  //clistp->setAutoDelete ( !clistp->autoDelete ()  );

  channel *cfrom= clistp->at(from);
  channel *cto= clistp->at(to);

  if ( from > to ) { // up
    clistp->remove( clistp->find(cfrom) );
    clistp->insert( clistp->find(cto), cfrom );
  } else { // down
    clistp->insert( clistp->find(cto)+1, cfrom );
    clistp->remove( clistp->find(cfrom) );
  }

  //clistp->setAutoDelete ( !clistp->autoDelete ()  );

  inserting=FALSE;

  newChannelListSLOT( clistp );
#ifdef chanDEBUG
  debug("channel: from %d",clistp->find(cfrom));
#endif
  clistp->at(to);
  channelbox->setCurrentItem( clistp->at() );
  channelbox->centerCurrentItem();
  emit channelSelectedSIGNAL();
}

/* ------------------------------------------------------------------------------*/

void winTVChannelWidget::activeChannelChangedSLOT()
{
  if (inserting) return;

#ifdef chanDEBUG
  debug("channel: activeChannelChangedSLOT");
#endif

  channelbox->setCurrentItem( clistp->at() );
  channelbox->centerCurrentItem();
  channelbox->repaintItem();
}

/* ------------------------------------------------------------------------------*/

void winTVChannelWidget::toggleVisibilitySLOT( QPoint *to )
{
  if (isVisible()) hide();
  else {
    if ( to ) {
#ifdef chanDEBUG
      debug("channel: setGeometry %d,%d,%d,%d",to->x(), to->y(), width(), height());
#endif
      setGeometry( to->x(), to->y(), width(), height() );
    }
    show();
  }
}

/* ------------------------------------------------------------------------------*/

void winTVChannelWidget::channelSelectedEmitSLOT(int index)
{
  if (inserting) return;

#ifdef chanDEBUG
  debug("channel: channelSelectedEmitSLOT");
#endif

  if ( index != clistp->at() ) {
    clistp->at( index );
    emit channelSelectedSIGNAL();
  }
}

/* ------------------------------------------------------------------------------*/

void winTVChannelWidget::channelEditEmitSLOT()
{
  if (inserting) return;

#ifdef chanDEBUG
  debug("channel: channelEditEmitSLOT");
#endif

  emit editChannelSIGNAL();
}

/* ------------------------------------------------------------------------------*/

void winTVChannelWidget::buttonsSLOT(int button)
{
  switch (button) {
  case BID_SC:
    toggleVisibilitySLOT( NULL );
    break;
  case BID_ADD:
    emit addChannelSIGNAL();
    break;
  case BID_EDIT:
    emit editChannelSIGNAL();
    break;
  case BID_DELETE:
    emit deleteChannelSIGNAL();
    break;
  case BID_MUP:
    //    emit copyChannelSIGNAL();
    moveItemSLOT(clistp->at(),clistp->at()-1);
    break;
  case BID_MDOWN:
    //    emit pasteChannelSIGNAL();
    moveItemSLOT(clistp->at(),clistp->at()+1);
    break;
  default:
    // this should never happen...
    fatal(i18n("main: nonexisting button number"));
  }
}


/* ------------------------------------------------------------------------------*/

void winTVChannelWidget::setPopup(QPopupMenu *cp)
{
  channelbox->setCPopup(cp);
}

/* ------------------------------------------------------------------------------*/
/* ------------------------------------------------------------------------------*/

#define Inherited QWidget

winWidget::winWidget(QWidget* parent, const char* name)
  : Inherited( parent, name, 503808 )
{
#ifdef chanDEBUG
  debug("win: init start");
#endif
  channelList = new winListBox( this );
  channelList->setGeometry( 5, 5, 140, 148 );
  channelList->setMinimumSize( 10, 10 );
  channelList->setMaximumSize( 32767, 32767 );

  channelList->setFrameStyle( 51 );
  channelList->setLineWidth( 2 );
  channelList->setMultiSelection ( false );
  channelList->setAllColumnsShowFocus( true );
  channelList->setTreeStepSize( 0 ); 
  channelList->setRootIsDecorated( false );

  channelList->addColumn("Nr.");
  channelList->setColumnWidth(0,30);
  channelList->addColumn("Name");
  channelList->setColumnWidth(1,90);
  channelList->addColumn("Channel");
  channelList->setColumnWidth(2,60);

  QToolTip::add(channelList,i18n("programs"));

  connect(channelList,SIGNAL(selectionChanged ( QListViewItem * ) ),
	  this,SLOT(highlightedSLOT(QListViewItem *)));
  connect(channelList,SIGNAL( doubleClicked ( QListViewItem * ) ),
	  this,SLOT(channelEditSelectSLOT(QListViewItem *)));
  connect(channelList,SIGNAL( returnPressed ( QListViewItem * ) ),
	  this,SLOT(channelEditSelectSLOT(QListViewItem *)));

  connect(channelList,SIGNAL(moveItemSIGNAL( QListViewItem *, QListViewItem * )),
	  this, SLOT(moveItemSLOT( QListViewItem *, QListViewItem * )));
  

  channelList->setFocus();
}

// =============================================================================== 

winWidget::~winWidget()
{

}

// =============================================================================== 

void winWidget::channelEditSelectSLOT(QListViewItem *item)
{
#ifdef chanDEBUG
  debug("win:  channelEditSelectSLOT");
#endif
  emit channelEditSelectSIGNAL();
}

// =============================================================================== 

void winWidget::highlightedSLOT(QListViewItem * item)
{
  if (inserting) return;

  int id= aChannelList.find( item );
  
#ifdef chanDEBUG
  debug("win:  highlightedSLOT %d, %s",id,item->text(2));
#endif

  emit highlighted(id);
}

// =============================================================================== 

void winWidget::insertItem ( const channel* info, bool ins)  
{
  inserting=ins;

  QString id;
  id.sprintf("%2d",info->id);

  QListViewItem *lvi= new QListViewItem( channelList, (const char*)id, info->name, info->channelId );
  aChannelList.append(lvi);

  inserting=false;

#ifdef chanDEBUG
  //debug("win:  insertItem %s, %s, %s, %s, %d",(const char*)id,info->name,(const char*)freq,info->channelId,inserting);
#endif
}


// =============================================================================== 

void winWidget::removeItem ( int index ) 
{
#ifdef chanDEBUG
  debug("win: removeItem");
#endif
  QListViewItem *lvi= aChannelList.at( index );
  aChannelList.remove();
  delete lvi;
  if ( aChannelList.count() ) {
    channelList->setSelected ( aChannelList.current(), true );
    channelList->setCurrentItem ( aChannelList.current() ); 
    channelList->ensureItemVisible(channelList->currentItem());
    //channelList->setSorting( 0 , true );
  }
}

// =============================================================================== 

void winWidget::changeItem ( const char * text, int index ) 
{
#ifdef chanDEBUG
  debug("win: changeItem %d",index);
#endif
  QListViewItem *lvi= aChannelList.at( index );
  lvi->setText( 2, text );
  channelList->repaintItem( lvi );
}

// =============================================================================== 

void winWidget::setCurrentItem ( int index )  
{
#ifdef chanDEBUG
  debug("win: setCurrentItem %d",index);
#endif
  channelList->setSelected ( aChannelList.at( index ), true );
  channelList->setCurrentItem ( aChannelList.at( index ) );  
}

// =============================================================================== 

void winWidget::centerCurrentItem ()  
{
  channelList->ensureItemVisible( aChannelList.current() );
  channelList->repaintItem( aChannelList.current() ); 
}

// =============================================================================== 

void winWidget::repaintItem ()  
{
  channelList->repaintItem( aChannelList.current() );
}

// =============================================================================== 

void winWidget::clear()  
{
#ifdef chanDEBUG
  debug("win: clear");
#endif

  inserting=true;

  channelList->clear();
  if ( !aChannelList.isEmpty() ) aChannelList.clear();
  channelList->setSorting( 0 , true );

  inserting=false;
}

// ================================"=============================================== 

void winWidget::moveItemSLOT( QListViewItem * fromItem, QListViewItem *toItem)
{
#ifdef chanDEBUG
  debug("win: move item %s to pos %s",fromItem->text(1),toItem->text(0));
#endif
  if ( !aChannelList.isEmpty() ) {
    QString from( fromItem->text(0) );
    QString to( toItem->text(0));

    emit moveItemSIGNAL( from.toInt(), to.toInt() ); 
  }
  
}

// =============================================================================== 
void winWidget::resizeEvent( QResizeEvent * re )
{
  channelList->setGeometry( 5, 5, width()-10, height()-10 );
}

// =============================================================================== 
// =============================================================================== 
// =============================================================================== 


void winListBox::mouseReleaseEvent ( QMouseEvent * me )
{
  if (me->button() == RightButton) {
    cPopup->popup(QCursor::pos()/*mapToGlobal(qme->pos())*/);
    return;
  }
#ifdef DRAGDROP
  if ( !pressedButton ) QListView::mouseReleaseEvent( me );

  switch (me->button()) {
  case LeftButton:
#ifdef chanDEBUG
    debug("channelListBox: mouseReleaseEvent, LeftButton released");
#endif
    if ( itemWasMoved ) {
#ifdef chanDEBUG
      debug("channelListBox: mouseReleaseEvent, item moved, LeftButton released");
#endif
      QListViewItem *targetItem= currentItem();
      if ( targetItem && (toMoveItem != targetItem) ) {
#ifdef chanDEBUG
	debug("channelListBox: mouseReleaseEvent, moving item to %s",(const char *)targetItem->text(1));
#endif
	emit moveItemSIGNAL( toMoveItem, targetItem );
      } else {
	setCurrentItem( currentItem() );
      }
      itemWasMoved = false;
      pressedButton= 0;
    } else {
#ifdef chanDEBUG
      debug("channelListBox: mouseReleaseEvent, item not moved, LeftButton released");
#endif
      itemWasMoved = false;
      pressedButton= 0;
    }
  } // switch
#endif
}

// =============================================================================== 
#ifdef DRAGDROP
void winListBox::mousePressEvent ( QMouseEvent * me )
{
  QListView::mousePressEvent ( me );
  
  if (me->button() == LeftButton) {
#ifdef chanDEBUG
    debug("winListBox: mousePressEvent, LeftButton");
#endif
    pressedButton = LeftButton;
    toMoveItem= itemAt( me->pos() );
  } else 
    pressedButton = 0;
  
}
#endif
// =============================================================================== 

void winListBox::mouseMoveEvent ( QMouseEvent * me )
{
#ifdef DRAGDROP
  //if ( !pressedButton ) QListView::mouseMoveEvent( me );

  if (pressedButton == LeftButton) {

#ifdef chanDEBUG
    //debug("winListBox: mouseMoveEvent, LeftButton");
#endif

    oldY= me->y();

    if ( geometry().contains( me->pos() ) && (itemAt( me->pos() ) != NULL) ) {
      lastVisItem=itemAt( me->pos() );
      setCurrentItem( lastVisItem );
      /*
      ensureItemVisible( lastVisItem );

      if ( itemAt( me->pos() ) != lastVisItem ) {
	QPoint pos( mapToGlobal(me->pos()).x() , itemPos( lastVisItem ));
	QCursor::setPos( mapFromGlobal(pos) );
      }
      */
      itemWasMoved=true;
      if ( !cursorChanged ) {
	// change mouse cursor
	cursorChanged= true;
      }
#ifdef chanDEBUG
      //debug("winListBox: mouseMoveEvent: move");
#endif
    } else {
#ifdef chanDEBUG
      debug("winListBox: mouseMoveEvent: exit");
#endif
      itemWasMoved=false;
      setCurrentItem( toMoveItem );
      //ensureItemVisible ( currentItem() );
      pressedButton=0;
      cursorChanged= false;
    }
  }
#endif
}

// =============================================================================== 
// =============================================================================== 
// =============================================================================== 
// never change the next line!
#include "wintvchannel.moc"

// end of file


