#include <stdio.h>
#include <getopt.h>
#include <stdlib.h>
#include <minmaxabs.h>

#include "version.h"
#include "MacroModule.h"
#include "InputModuleFile.h"
#include "ModuleGenerator.h"
#include "SamplingConfig.h"
#include "Dirs.h"

// Function prototypes
int  parse_options(int argc, char **argv);
void print_version();
void print_usage(const char *commandname);
void print_module_types();
bool check_modulefile(const InputModuleFile&);
void execute_module(const char *commandname, const char *);

// Options the user may set
bool   option_verbose      = false;  // -v --verbose
bool   option_dots         = false;  // -d --dots
bool   option_mono         = false;  // -m --mono
SamplingConfig sampling_config;
bool   option_cut_from     = false;  // -c --cut-from
bool   option_cut_to       = false;  // -C --cut-to
const char  *option_plugins_dir  = global_plugins_dir;
double cut_from            = 0;
double cut_to              = 0;

/**
 * Main function
 */
main(int argc, char **argv)
{
    int index_of_parameter = parse_options(argc, argv);
    if (index_of_parameter >= argc) {
	fprintf(stderr, "Give modules to play as arguments.\n");
	print_usage(argv[0]);
	exit(5);
    }
    MacroModule::scanPlugins(option_plugins_dir);
    for (int i = index_of_parameter; i < argc; i++) execute_module(argv[0], argv[i]);
}


/**
 * Parse command line. Return index of first non-option parameter
 */
int parse_options(int argc, char **argv)
{
    static struct option long_options[] = {
	{ "version",              0, 0, 'V' },
	{ "help",                 0, 0, 'h' },
	{ "verbose",              0, 0, 'v' },
	{ "dots",                 0, 0, 'd' },
	{ "print-module-types",   0, 0, 't' },
	{ "sampling-rate",        1, 0, 's' },
	{ "fragment-size",        1, 0, 'f' },
	{ "mono",                 0, 0, 'm' },
	{ "cut-from",             1, 0, 'c' },
	{ "cut-to",               1, 0, 'C' },
	{ "plugins-dir",          1, 0, 'p' },
	{ 0, 0, 0, 0 }
    };

    opterr = 0;
    char option;
    while (-1 != (option = getopt_long(argc, argv, "Vhvdtc:C:s:f:mp:", long_options, 0))) switch (option)
    {
    case 'V':
	print_version();
	exit(0);
	
    case 'h':
	print_usage(argv[0]);
	exit(0);

    case 'v':
	option_verbose = true;
	break;

    case 'd':
	option_dots = true;
	break;

    case 't':
	print_module_types();
	exit(0);

    case 's':
	sampling_config.samplingrate = atol(optarg);
	if (sampling_config.samplingrate <= 0) {
	    fprintf(stderr, "%s: sampling rate must be > 0!\n", argv[0]);
	    exit(5);
	}
	break;

    case 'f':
	sampling_config.fragmentsize = atol(optarg);
	if (sampling_config.fragmentsize <= 0) {
	    fprintf(stderr, "%s: fragment size must be > 0!\n", argv[0]);
	    exit(5);
	}
	break;

    case 'c':
	option_cut_from = true;
	cut_from = atof(optarg);
	break;

    case 'C':
	option_cut_to = true;
	cut_to = atof(optarg);
	break;

    case 'm':
	option_mono = true;
	break;

    case 'p':
	option_plugins_dir = optarg;
	break;

    case '?':
	print_usage(argv[0]);
	exit(5);
    }

    if (option_verbose && option_mono) printf("Forcing mono output.\n");

    return optind;
}


/**
 * Load and execute a module.
 */
void execute_module(const char *commandname, const char *filename)
{
    InputModuleFile imf(filename);
    if (!check_modulefile(imf)) return;

    MacroModule module(MacroModule::ET_FILEREF, "", &imf);
    if (!check_modulefile(imf)) return;

    if (option_verbose) {
	printf("\n");
	printf("     Modulefile:  %s\n", filename);
	printf("     MMM Version: %d.%d.%d\n", 
	       imf.majorVersion(), imf.minorVersion(), imf.subMinorVersion());
	printf("     Name:        %s\n", module.getName().c_str());
	printf("     Description: %s\n", module.getDescription().c_str());
	printf("     Submodules:  %d\n", module.numberOfModules());
	printf("     Wires:       %d\n", module.numberOfWires());
    }

    if (module.isExecutable()) {
	if (option_verbose) {
	    printf("\nUsing sampling rate %ld, ", sampling_config.samplingrate);
	    printf("%d samples per block\n", (1 << sampling_config.fragmentsize) / 2);
	    if (option_cut_from || option_cut_to) {
		printf("Cutting range: %.3f - ", option_cut_from ? cut_from : 0);
		if (option_cut_to) printf("%.3f\n", cut_to);
		else printf("oo\n");
	    }
	}
	module.prepareForExecution(&sampling_config);
	string module_errors;
	if (module.reportModuleErrors(module_errors))
	    fprintf(stderr, "%s\n", module_errors.c_str());
	else 
	{
	    long start_time = option_cut_from ? (long)(cut_from * sampling_config.samplingrate) : 0;
	    long end_time   = option_cut_to   ? (long)(cut_to * sampling_config.samplingrate) : 0;
	    
	    while (true) {
		long number_of_samples = (1 << sampling_config.fragmentsize) / 2;
		if (option_cut_to) number_of_samples = min(number_of_samples, end_time - start_time);
		if (option_dots) { printf("."); fflush(stdout); }
		if (!module.executeBlock(start_time, number_of_samples)) break;
		start_time += (1 << sampling_config.fragmentsize) / 2;
	    }
	}
	module.finishExecution();
	if (option_dots) printf("\n");
    }
    else fprintf(stderr, "%s: Module file %s doesn't contain an executable module.\n",
		 commandname, filename);
}

bool check_modulefile(const InputModuleFile& imf)
{
    if (!imf.isGood()) {
	fprintf(stderr, "%s:%ld: %s.\n",
		imf.getFilename().c_str(), 
		imf.errorLineNumber(),
		imf.errorText().c_str());
	return false;
    }
    else return true;
}


void print_version()
{
    printf("Mathi's Music Magic Module Player %d.%d.%d\n", 
	   MAJOR_VERSION, MINOR_VERSION, REVISION);
}

/**
 * Print syntax of the command
 */
void print_usage(const char *commandname)
{
    fprintf(stderr, 
	    "Usage: %s -V\n"
	    "       %s -h\n"
	    "       %s -t\n"
	    "       %s [-v] [-d] [-s rate] [-b size] [-c time] [-C time] Module1 [Module2 ...]\n"
	    "\n"
	    "       -V,  --version              Print version and exit\n"
	    "       -h,  --help                 Print this help and exit\n"
	    "       -v,  --verbose              Give some verbose information\n"
	    "       -d,  --dots                 Print a dot for each block played\n"
	    "       -t,  --print-module-types   Print list of all known modules types\n"
	    "       -s,  --sampling-rate        Set sampling rate. Default = 44100\n"
	    "       -f,  --fragment-size        Size of sampling fragment in pow of 2. Default=10\n"
	    "       -c,  --cut-from             Give time in seconds to start from\n"
	    "       -C,  --cut-to               Give time in seconds to stop at\n"
	    , commandname, commandname, commandname, commandname);
}


void print_module_types()
{
    printf("%s", ModuleGenerator::allNames().c_str());
}
