// -*- c++ -*-
// **************************************************************
// $Source: /home/proj/mmm/cvsroot/mmm/modules/MVolumeControl.cc,v $
// $Revision: 1.1 $
// $Date: 1999/05/13 08:02:17 $
// $State: Exp $
// **************************************************************

#define MODULE_NAME "volume-control"

#include "ModuleMacros.h"
#include "minmaxabs.h"

BEGIN_MODULE_DEFINITION(VolumeControl);
    Slot  *sslot_input;
    Slot  *sslot_volcon;
END_MODULE_DEFINITION(VolumeControl);

class SSVolumeControlOutput : public Signal
{
public:
    SSVolumeControlOutput(MVolumeControl *m) 
	: Signal(SOUND_TYPE, "output", "Input1 with volume controlled by input2", m) {};
    PreparedSignal *prepareSignal(Metainfo *, const Parameterset *);
};

class PSSVolumeControlOutput : public PreparedSoundSignal
{
    friend class SSVolumeControlOutput;
    PreparedSoundSignal *pssig_input;
    PreparedSoundSignal *pssig_volcon;
    long cut_from, cut_to;
public:
    PSSVolumeControlOutput(PreparedSoundSignal *,PreparedSoundSignal *, long, long);
    ~PSSVolumeControlOutput() { delete pssig_input; delete pssig_volcon; };
private:
    void computeSamples(Number *, long start_time, long number_of_samples);
};


// ---------------------------------------------------------------------------
//                             Implementation
// ---------------------------------------------------------------------------

#include <values.h>

#include "SharedSoundStore.h"

inline PSSVolumeControlOutput::PSSVolumeControlOutput(
    PreparedSoundSignal *pssig_input, PreparedSoundSignal *pssig_volcon, long cut_from, long cut_to)
    :	pssig_input(pssig_input),
	pssig_volcon(pssig_volcon),
	cut_from(cut_from),
	cut_to(cut_to)
{
}


MVolumeControl::MVolumeControl(string)
{
    addConnector(sslot_input  = new Slot(SOUND_TYPE, "input", "Sound signal to control", this, 6));
    addConnector(sslot_volcon = new Slot(SOUND_TYPE, "volcon", "Volume signal", this, 7));
    addConnector(new SSVolumeControlOutput(this));
}


PreparedSignal *SSVolumeControlOutput::prepareSignal(Metainfo *mi, const Parameterset *parset)
{
    Metainfo mi_input  = *mi;
    Metainfo mi_volcon = *mi;
    
    PreparedSoundSignal	*pssig_input 
	= getPreparedSoundSignal(this, ((MVolumeControl *)getModule())->sslot_input, &mi_input, parset);
    PreparedSoundSignal *pssig_volcon 
	= getPreparedSoundSignal(this, ((MVolumeControl *)getModule())->sslot_volcon, &mi_volcon, parset);
	
    Metainfo asked_for = *mi; // Remember what MI are asked for
    *mi = mi_input; // I take all MI from the input signal, instead of cutting range and duration.

    // I save the cutting ranges of the two inputs for later optimizations

    long cut_from_input  = mi_input.containsCutFrom()  ? mi_input.getCutFrom() : MINLONG; 
    long cut_to_input    = mi_input.containsCutTo()    ? mi_input.getCutTo()   : MAXLONG;

    long cut_from_volcon = mi_volcon.containsCutFrom() ? mi_volcon.getCutFrom() : MINLONG;
    long cut_to_volcon   = mi_volcon.containsCutTo()   ? mi_volcon.getCutTo()   : MAXLONG;
    
    // Cutting range computation: If one input signal is 0, so is the output signal.

    if (asked_for.containsCutFrom()) {
	if (mi_input.containsCutFrom() && mi_volcon.containsCutFrom())
	    mi->setCutFrom(max(cut_from_input, cut_from_volcon));
	else if (mi_input.containsCutFrom())  mi->setCutFrom(cut_from_input);
	else if (mi_volcon.containsCutFrom()) mi->setCutFrom(cut_from_volcon);
	else mi->clearCutFrom();
    }

    if (asked_for.containsCutTo()) {
	if (mi_input.containsCutTo() && mi_volcon.containsCutTo())
	    mi->setCutTo(min(cut_to_input, cut_to_volcon));
	else if (mi_input.containsCutTo())  mi->setCutTo(cut_to_input);
	else if (mi_volcon.containsCutTo()) mi->setCutTo(cut_to_volcon);
	else mi->clearCutTo();
    }

    // For the duration the volcon Signal has preference
    
    if (asked_for.containsDuration()) {
	if (mi_volcon.containsDuration()) mi->setDuration(mi_volcon.getDuration());
	else if (mi_input.containsDuration()) mi->setDuration(mi_input.getDuration());
	else mi->clearDuration();
    }
    
    return new PSSVolumeControlOutput(pssig_input, pssig_volcon, 
				      max(cut_from_input, cut_from_volcon),
				      min(cut_to_input, cut_to_volcon));
}


void PSSVolumeControlOutput::computeSamples(Number *output,long start_time, long nsamples)
{
    SoundPortion sp_input  = pssig_input ->getSoundPortion(start_time, nsamples);
    SoundPortion sp_volcon = pssig_volcon->getSoundPortion(start_time, nsamples); 
    const Number *input   = sp_input .getSamples();
    const Number *volcon  = sp_volcon.getSamples();
    while (nsamples--) *output++ = *input++ * *volcon++;
}
