// -*- c++ -*-
// **************************************************************
// $Source: /home/proj/mmm/cvsroot/mmm/modules/MRealtimeAnalyser.cc,v $
// $Revision: 1.1 $
// $Date: 1999/05/13 08:02:16 $
// $State: Exp $
// **************************************************************

#define MODULE_NAME "realtime-analyser"

#include "ModuleMacros.h"
#include "PNSigStoredNumber.h"

BEGIN_MODULE_DEFINITION(RealtimeAnalyser);
    Slot *sslot_input;
    Number rtfactor;
END_MODULE_DEFINITION(RealtimeAnalyser);

class SSigRealtimeAnalyserOutput : public Signal
{
    Number *rtfactor;
public:
    SSigRealtimeAnalyserOutput(MRealtimeAnalyser *m, Number *rtfactor) 
	: Signal(SOUND_TYPE, "output", "Same as input", m), rtfactor(rtfactor) {};
    PreparedSignal *prepareSignal(Metainfo *, const Parameterset *);
};

class NSigRealtimeAnalyserRealtime : public Signal
{
    Number *rtfactor;
public:
    NSigRealtimeAnalyserRealtime(Module *m, Number *rtfactor)
	: Signal(NUMBER_TYPE, "realtime-factor", 
		 "Realtime factor for processing of input", m), 
	  rtfactor(rtfactor) {};
    PreparedSignal *prepareSignal(Metainfo *, const Parameterset *)
	{ return new PNSigStoredNumber(rtfactor); };
};

class PSSigRealtimeAnalyserOutput : public PreparedSoundSignal
{
    PreparedSoundSignal *pssig_input;
    Number *rtfactor;
    Seconds sampling_interval;
public:
    PSSigRealtimeAnalyserOutput(PreparedSoundSignal *pssig_input, Number *rt, Seconds si)
	: pssig_input(pssig_input), rtfactor(rt), sampling_interval(si) {};
    ~PSSigRealtimeAnalyserOutput() { delete pssig_input; };
    SoundPortion getSoundPortion(long start_time, long number_of_samples);
};


// ---------------------------------------------------------------------------
//                             Implementation
// ---------------------------------------------------------------------------

#include <sys/timeb.h>

MRealtimeAnalyser::MRealtimeAnalyser(string)
    : rtfactor(0)
{
    addConnector(sslot_input = new Slot(SOUND_TYPE, "input", "Input sound signal", this, 5));
    addConnector(new NSigRealtimeAnalyserRealtime(this, &rtfactor));
    addConnector(new SSigRealtimeAnalyserOutput(this, &rtfactor));
}


PreparedSignal *SSigRealtimeAnalyserOutput::prepareSignal(Metainfo *mi,
							  const Parameterset *parset)
{
    // I need to know the sampling rate for measuring the realtime factor.
    Seconds sampling_interval = getSamplingInterval(getModule(), mi, parset);

    PSSigRealtimeAnalyserOutput *pps = new PSSigRealtimeAnalyserOutput(
	getPreparedSoundSignal(this,  
			       ((MRealtimeAnalyser *)getModule())->sslot_input, 
			       mi, parset),
	rtfactor, sampling_interval);
    return pps;
}

SoundPortion PSSigRealtimeAnalyserOutput::getSoundPortion(long start_time, 
							  long number_of_samples)
{
    struct timeb before, after;
    ftime(&before);
    SoundPortion sp = pssig_input->getSoundPortion(start_time, number_of_samples);
    ftime(&after);

    unsigned long seconds_elapsed = after.time - before.time;
    long          millis_elapsed = (long)after.millitm - (long)before.millitm;
    double time_elapsed = (double)seconds_elapsed + (double)millis_elapsed / 1000.0;
    double realtime     = (double)sampling_interval * number_of_samples;

    *rtfactor = time_elapsed / realtime;

    return sp;
}
