/*
  fix_fft visualises pcm data
  Copyright (C) 1998  Martin Vogt;Philip VanBaren

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */



/*      fix_fft.c - Fixed-point Fast Fourier Transform  */
/*
        fix_fft()       perform FFT or inverse FFT
        window()        applies a Hanning window to the (time) input
        fix_loud()      calculates the loudness of the signal, for
                        each freq point. Result is an integer array,
                        units are dB (values will be negative).
        iscale()        scale an integer value by (numer/denom).
        fix_mpy()       perform fixed-point multiplication.
        Sinewave[1024]  sinewave normalized to 32767 (= 1.0).
        Loudampl[100]   Amplitudes for lopudnesses from 0 to -99 dB.
        Low_pass        Low-pass filter, cutoff at sample_freq / 4.


        All data are fixed-point short integers, in which
        -32768 to +32768 represent -1.0 to +1.0. Integer arithmetic
        is used for speed, instead of the more natural floating-point.

        For the forward FFT (time -> freq), fixed scaling is
        performed to prevent arithmetic overflow, and to map a 0dB
        sine/cosine wave (i.e. amplitude = 32767) to two -6dB freq
        coefficients; the one in the lower half is reported as 0dB
        by fix_loud(). The return value is always 0.

        For the inverse FFT (freq -> time), fixed scaling cannot be
        done, as two 0dB coefficients would sum to a peak amplitude of
        64K, overflowing the 32k range of the fixed-point integers.
        Thus, the fix_fft() routine performs variable scaling, and
        returns a value which is the number of bits LEFT by which
        the output must be shifted to get the actual amplitude
        (i.e. if fix_fft() returns 3, each value of fr[] and fi[]
        must be multiplied by 8 (2**3) for proper scaling.
        Clearly, this cannot be done within the fixed-point short
        integers. In practice, if the result is to be used as a
        filter, the scale_shift can usually be ignored, as the
        result will be approximately correctly normalized as is.


        TURBO C, any memory model; uses inline assembly for speed
        and for carefully-scaled arithmetic.

        Written by:  Tom Roberts  11/8/89
        Made portable:  Malcolm Slaney 12/15/94 malcolm@interval.com
	C++ class: Martin Vogt 3/25/99

                Timing on a Macintosh PowerBook 180.... (using Symantec C6.0)
                        fix_fft (1024 points)             8 ticks
                        fft (1024 points - Using SANE)  112 Ticks
                        fft (1024 points - Using FPU)    11

*/





#ifndef __FIX_FFT_H
#define __FIX_FFT_H


#define FIX_MPY(DEST,A,B)       DEST = ((long)(A) * (long)(B))>>15


#define N_WAVE          1024    /* dimension of Sinewave[] */
#define LOG2_N_WAVE     10      /* log2(N_WAVE) */
#define N_LOUD          100     /* dimension of Loudampl[] */

typedef short fixed;

class FixFFT {


 public:
  FixFFT();
  ~FixFFT();
  

  /**
        fix_fft() - perform fast Fourier transform.

        if n>0 FFT is done, if n<0 inverse FFT is done
        fr[n],fi[n] are real,imaginary arrays, INPUT AND RESULT.
        size of data = 2**m
        set inverse to 0=dft, 1=idft
  */
  int fix_fft(fixed fr[], fixed fi[], int m, int inverse);

  /**      
	   window() - apply a Hanning window       
  */
  void window(fixed fr[], int n);

  
  /**
     fix_loud() - compute loudness of freq-vis components.
     n should be ntot/2, where ntot was passed to fix_fft();
     6 dB is added to account for the omitted alias components.
     scale_shift should be the result of fix_fft(), if the time-series
     was obtained from an inverse FFT, 0 otherwise.
     loud[] is the loudness, in dB wrt 32767; will be +10 to -N_LOUD.
  */
 void fix_loud(fixed loud[], fixed fr[], fixed fi[], int n, int scale_shift);

 

 private:
   /**
      db_from_ampl() - find loudness (in dB) from
      the complex amplitude.
   */
   int db_from_ampl(fixed re, fixed im);

   
   /**
     fix_mpy() - fixed-point multiplication
   */
   fixed fix_mpy(fixed a, fixed b);
   /**
      iscale() - scale an integer value by (numer/denom)
   */
   int iscale(int value, int numer, int denom);
   /**
      fix_dot() - dot product of two fixed arrays
   */
   fixed fix_dot(fixed *hpa, fixed *pb, int n);
   

   /**
      initilises tables
   */
   void doInit();

   


};
#endif
