/*
  buffer capabilites for a stream
  Copyright (C) 1998  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */

#include <producer/yaf/yafStream.h>
#include "../../../../config.h"

// Counter is used for debugging reasons (to distinguish
// between the threads. And to create an unique (hopefully) fifo name

static int counter=0;




/*Threads and usleep does not work, select is very portable*/
static void doSleep(int msec) {
  struct timeval time;
  time.tv_sec=0;
  time.tv_usec=msec;

  select(0,NULL,NULL,NULL,&time);
}


static void *readloopThread(void *arg){
  ((YafStream*)arg)->readloop();
  return NULL;
}   



YafStream::YafStream(YafGenerator* generator) {
  int rNum;

  allRead=0;
  this->generator=generator;
  pthread_mutex_init(&mut,NULL);
  pthread_cond_init(&cond,NULL);

  randomName=new Buffer (60);
  streamCounter=counter;
  randomName->append("/tmp/.kmpg.");

  rNum=time(NULL);
  randomName->append(rNum);
  randomName->append(counter);
  randomName->append(getuid());
  randomName->append(getpid());
  counter++;
  

  fifoCommand=new Buffer(50);
  fifoCommand->clear();
  fifoCommand->append("outputfile ");
  fifoCommand->append(randomName->getData());
  fifoCommand->append("\n");
  
  size=1024*1024*1;
  isShutdown=true;

  eventQueue=new EventQueue("YafStream");
  eventQueue->addListener(this);
  eventQueue->setNotifyMode(_NOTIFY_ALL);

  ringBufferNT=new RingBufferNT(size,16384);
  bufferUnderrun=__BUFFER_UNDERRUN_NOTIFIED;
  buffUnderRunSize=size/50;
  buffOverRunSize=size/8;
  lSockCreate=false;
  eof=false;
  start();
  while(lSockCreate==false) {
    doSleep(200);
  }
}


YafStream::~YafStream() {
  stop();

  delete ringBufferNT;
  delete fifoCommand;
  delete eventQueue;
  delete randomName;
}




void YafStream::start() {
  if (isShutdown == true) {
    isShutdown=false;
    pthread_create(&tr,NULL,readloopThread,this);
    return;
  }
}



void YafStream::stop() {
  void* ret;

  if (isShutdown == false) {
    lockReaderThread();
    isShutdown=true;
    lWriteToRing=false;
    unlockReaderThread();
    ringBufferNT->exitWaitForSpace();
    debugOutput( cout << "vor join YafStream"<<endl );
    pthread_join(tr,&ret);
    debugOutput( cout << "vor join YafStream -e"<<endl );
   
    return;
  }
}



/**
   These Functions are entered by the "reader" thread [START]

   Note: The Streambuffer is _not_ filled if we have a buffer
   we simply set the StartPtr to the memory of the buffer.
   If we have  not buffer we set the ptr to the read stream.

*/



int YafStream::fillBuffer(AudioBuffer* buffer) {
  int back=true;
  int pSize=buffer->getPreferredFillGrade();
  MemChunk* gmem;
  float percent;
  gmem=ringBufferNT->requestMemChunk(pSize);
  buffer->setMemChunk(gmem);

  // Note : in order to make the threads as parallel as possible
  // we have a single race condition which must be handled
  // seperatly. This race condition comes up, because
  // the reader thread does not wait on the waitForData
  // Mutex (as expected by the ringBufferNT, and which is
  // true in nearyl all cases) We must make sure that if
  // we (think) we have no data, we make another (this time
  // mutexed) try. to be _really_ sure. I think this
  // does not overcomplicate the situation. And the gain
  // that in all other cases the both threads run unmutexed
  // this is ok.

  if (gmem->getLen() == 0) {     // bufferunderrun or a race condition?
    lockReaderThread();

    gmem=ringBufferNT->requestMemChunk(pSize);
    buffer->setMemChunk(gmem);

    if (gmem->getLen() == 0) {
      back=false;
      debugOutput( cout << "underrun" << " r:"<<pSize << " p: "<<buffer->getReadBytes() << " a:"<<ringBufferNT->getFillgrade() << " c:"<<ringBufferNT->getReadBytes() << " w:"<<ringBufferNT->getWriteBytes() << " al:"<<allRead << " wr:"<<ringBufferNT->getWriteBytes()<<endl );
      if (bufferUnderrun == __BUFFER_UNDERRUN_FALSE_NOTIFIED) {
	bufferUnderrun=__BUFFER_UNDERRUN_NOTIFIED;
	unlockReaderThread();
	eventQueue->sendEvent((char)false);
      } else {
	unlockReaderThread();
      }
    } else {
      debugOutput( cout << "backend thread cheated us !!!!!"<<gmem->getLen()<<endl );
      unlockReaderThread();
    }
  }
  buffer->setReadBytes(ringBufferNT->getReadBytes()+gmem->getLen());
  percent=(float)ringBufferNT->getFillgrade()/(float)ringBufferNT->getSize();
  buffer->setMainBufferFillgrade(percent);
  return back;
}
  


/**
   These Functions are entered by the "reader" thread [END]
*/


void YafStream::processEvent(char msgId) {
  generator->bufferFilledNotify((int)msgId);
}


/**
   The readloop is the thread which reads the fifo and fills the
   buffer
*/

void YafStream::readloop() {
  int didRead;
  int fillgrade;
  int lShow=false;
  MemChunk* gmem=new MemChunk(0);
  int preferredSize=8192;
  bindSocket=new BindSocket(randomName->getData());
  bindSocket->bind();
  lSockCreate=true;
  while (isShutdown==false) {
    if (bindSocket->accept() > 0) {
      break;
    }
    doSleep(200);
  }
  pthread_mutex_lock(&mut);
  while(isShutdown==false) {
    if (bufferUnderrun==__BUFFER_UNDERRUN_NOTIFIED) {
      fillgrade=ringBufferNT->getFillgrade();
      if ((fillgrade>buffOverRunSize) || (eof)){
	if (eof == false) {
	  debugOutput( cout << "+++++++++++++ buffer well filled +++++++++++"<<endl );
	} else {
	  debugOutput( cout << "+++++++++++++ notify eof +++++++++++"<<endl );
	}
  
	pthread_mutex_unlock(&mut);
	eventQueue->sendEvent(true);
	pthread_mutex_lock(&mut);
	bufferUnderrun=__BUFFER_UNDERRUN_FALSE_NOTIFIED;
	continue;  
      }
    }

    pthread_mutex_unlock(&mut);
    ringBufferNT->getWriteArea(gmem,preferredSize);
   
    if (gmem->getLen() < preferredSize){
      ringBufferNT->waitForSpace(preferredSize*2);
      ringBufferNT->getWriteArea(gmem,preferredSize);
    }
    if (gmem->getLen() == 0) {
      pthread_mutex_lock(&mut);
      continue;
    }
    if (gmem->getLen() < 0) {
      debugOutput( cout << "gmem->getLen() < 0 this means trouble"<<endl );
    }
    
    // because we block on the fifo and have at least
    // one byte to buffer, we are sure, that didRead<=0 means
    // some nasty bug
    didRead=bindSocket->read(gmem->getPtr(),gmem->getLen());
    if (didRead <= 0) {
      if (lShow==false) {
	debugOutput( cout << "bindSocket error. sleeping for recovery"<<endl );
	lShow=true;
      }
      doSleep(200);
      pthread_mutex_lock(&mut);
      continue;
    }
    allRead+=didRead;
    pthread_mutex_lock(&mut);
    if (lWriteToRing) {
      ringBufferNT->forwardWritePtr(didRead);
     } else {
       //debugOutput( cout << "Skip read bytes :"<<didRead<<endl );
     }
  }
  pthread_mutex_unlock(&mut);
  delete gmem;
  delete bindSocket;
}


/**
   These Functions are entered by the "reader" thread [END]
*/




  
/**

   
*/
char* YafStream::getFifoCommand() {
  return fifoCommand->getData();
}


void YafStream::lockReaderThread() {
  pthread_mutex_lock(&mut);
}


void YafStream::unlockReaderThread() {
  pthread_cond_signal(&cond);
  pthread_mutex_unlock(&mut);
}




void YafStream::close() {
  setWriteRing(false);
  eof=false;
}


void YafStream::open() {
  setWriteRing(true);
}
 

void YafStream::setEOF(int leof) {
  this->eof=leof;
  eventQueue->sendEvent(true);
}


int YafStream::getEOF() {
  return eof;
}


void YafStream::setWriteRing(int lWrite) {
  lockReaderThread();
  lWriteToRing=lWrite;
  unlockReaderThread();
}
  

void YafStream::clear() {
  pthread_mutex_lock(&mut);
  ringBufferNT->emptyBuffer();
  bufferUnderrun=__BUFFER_UNDERRUN_NOTIFIED;
  pthread_mutex_unlock(&mut);
}



