/*
  Copyright (C) 1999 Rainer Maximini

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package
*/

#include "mainwidget.h"

#include <qkeycode.h>
#include <qlabel.h>
#include <kapp.h>
#include <kaccel.h>
#include <qfile.h>
#include <kfiledialog.h>
#include <kmsgbox.h>
#include <iostream.h>
#include <fstream.h>
#include <stdlib.h>
#include <qsplitter.h>
#include <kurl.h>
#include <qfileinfo.h>
#include "ioaccess.h"


MainWidget::MainWidget(QWidget *parent,const char *name) 
  :QWidget(parent,name){

  dataBase = 0;
  splitter = new QSplitter(QSplitter::Horizontal,this, "splitter");
  
  datalist = new SongTree(i18n("DataBase"),  splitter,"datalist");
  playlist = new SongList(i18n("Play List"), splitter,"playlist");

  setTabOrder(datalist, playlist );
  setTabOrder(playlist, datalist );

  connect(datalist,SIGNAL(selectionChanged()),playlist,SLOT(clearSelection()));
  connect(datalist,SIGNAL(doubleClick()),this,SLOT(doubleClickDB()));
  connect(datalist,SIGNAL(showMP3Info(Song)),this,SLOT(showMP3Info(Song)));
  connect(datalist,SIGNAL(playSong(Song)),this,SLOT(findAndPlaySong(Song)));

  connect(playlist,SIGNAL(selectionChanged()),datalist,SLOT(clearSelection()));
  connect(playlist,SIGNAL(doubleClick()),this,SLOT(doubleClickPL()));
  connect(playlist,SIGNAL(showMP3Info(Song)),this,SLOT(showMP3Info(Song)));
  connect(playlist,SIGNAL(playSong(Song)),this,SLOT(findAndPlaySong(Song)));
  

  dropZoneDB = new KDNDDropZone( datalist, DndURL );
  dropZonePL = new KDNDDropZone( playlist, DndURL );
  connect(dropZoneDB,SIGNAL(dropAction(KDNDDropZone*)), SLOT(dropDB(KDNDDropZone *)));
  connect(dropZonePL,SIGNAL(dropAction(KDNDDropZone*)), SLOT(dropPL(KDNDDropZone *)));
  
  mp3Info = new SongInfo();
}

void MainWidget::setDataBase(DataBase *db){
  if (dataBase != 0){
    disconnect(dataBase,0,this,0);
    disconnect(mp3Info,0,dataBase,0);
    disconnect(dataBase,0,mp3Info,0);
  }
  dataBase = db;
  dataBase->attach(playlist);
  connect(dataBase,SIGNAL(dBChanged()), this , SLOT(refreshDataList()));
  connect(dataBase,SIGNAL(genreChanged()), this, SLOT(refreshDataList()));
  connect(mp3Info,SIGNAL(songChanged(Song*)), dataBase, SLOT(songChanged(Song*)));
  connect(dataBase,SIGNAL(referenceChanged(Song*, Song*)), 
	  mp3Info, SLOT(referenceChanged(Song*, Song*)));
  connect(dataBase,SIGNAL(referenceChanged(Song*, Song*)), 
	  playlist, SLOT(referenceChanged(Song*, Song*)));
  connect(dataBase,SIGNAL(songRemovedWithRef(Song*)), 
	  mp3Info, SLOT(referenceRemoved(Song*)));
  connect(dataBase,SIGNAL(dBCleared()), mp3Info, SLOT(songRemoved()));
}


void MainWidget::keyPressEvent(QKeyEvent* e){
  switch (e->key()) {
  case Key_Tab: toggleFocus( ); break;
  }
}

void MainWidget::refreshDataList(){
  QSongList songs;
  KApplication::setOverrideCursor( waitCursor );
  songs = dataBase->getSongList();
  datalist->clear();
  datalist->addSongs(songs);
  KApplication::restoreOverrideCursor();
}

void MainWidget::findAndPlaySong(Song tmpSong){
  Song *song = 0;
  song = dataBase->findInPlayList(tmpSong);
  if(song == 0){
    song = (dataBase->find(tmpSong)).first();
  }
  if(song == 0) return;
  emit playSong(song);
}

void MainWidget::showMP3Info(Song tmpSong){
  QSongList  tmpSongs;
  Song *song;
  
  /* find the song in the database and display it */
  song = dataBase->findInPlayList(tmpSong);
  if(song == 0){
    song = (dataBase->find(tmpSong)).first();
  }
  mp3Info->showMP3Song(song);
}  

void MainWidget::showMP3Info(){
  QSongList  tmpSongs;
  Song  tmpSong;

  /* Frist check database view */  
  tmpSong = datalist->getSelection();
  if (tmpSong.isNull()) { 
    cout << "tmpSong ist Null" << endl;
    /* Then check playlist view */
    tmpSong = playlist->getSelection();
    if (tmpSong.isNull())  return;
  }
  showMP3Info(tmpSong);
}  


void MainWidget::doubleClickPL(){
  KConfig *config;
  Song tmp;
  Song *song;
  config = (KApplication::getKApplication())->getConfig();
  config->setGroup(KCONFIG_GUI);
  if(config->readNumEntry(KCONFIG_GUI_DC_PLAYLIST,PLAYSONG) == PLAYSONG) {
    tmp = playlist->getSelection();
    if(! (tmp.isNull()) ){
      song = dataBase->findInPlayList(tmp);
      if(song == 0)
 	song = dataBase->find(tmp).first();
      emit playSong(song);
    }    
  }
  else{
    showMP3Info();
  }
}

void MainWidget::doubleClickDB(){
  Song tmp;
  KConfig *config;
  config = (KApplication::getKApplication())->getConfig();
  config->setGroup(KCONFIG_GUI);
  if(config->readNumEntry(KCONFIG_GUI_DC_DATABASE,PLAYSONG) == PLAYSONG) {
    tmp = datalist->getSelection();
    if(! (tmp.isNull()) )
      emit playSong(dataBase->find(tmp).first());
  }
  else{
    showMP3Info();
  }
}

void MainWidget::playSelectedSong(){
  Song tmp;
  Song *song;
  tmp = datalist->getSelection();
  if(tmp.isNull() )
    tmp = playlist->getSelection();
  if(!(tmp.isNull())){
    song = dataBase->findInPlayList(tmp);
    if(song == 0)
      song = dataBase->find(tmp).first();
    emit playSong(song);
  }
}


void MainWidget::copySongIntoPlayList(){
  playlist->addSongs(dataBase->find(datalist->getSelection()));
}

void MainWidget::copyAllSongsIntoPlayList(){
  playlist->addSongs(dataBase->getSongList());
}

void MainWidget::removeSong(){  
  Song tmp;
  tmp = datalist->getSelection();
  if (!tmp.isNull()){
    dataBase->remove(tmp);
  }
  else{
    Song *ref;
    ref = playlist->removeSong();
    if (ref!=0) dataBase->removeInPlayList(ref);
  }
}

SongList* MainWidget::getPlayList(){
  return playlist;
}

Song* MainWidget::getSelectedSongRef(){
  Song tmp;
  Song *song;
  tmp = playlist->getSelection();
  if (!tmp.isNull()){
    song = dataBase->findInPlayList(tmp);
    if(song == 0)
      song = dataBase->find(tmp).first();
  }
  else {
    tmp = datalist->getSelection();
    if(tmp.isNull())  return 0;
    song = dataBase->find(tmp).first();

  }
  
  return song;
}
  

void MainWidget::removePlayedSongs(){  
  playlist->removePlayedSongs();
  dataBase->checkPlayList();
}
void MainWidget::clearPlayList(){  
  playlist->removeAllSongs();
  dataBase->clearPlayList();
}

Song MainWidget::getFirstSongToPlay(){
  return playlist->getFirstSongToPlay();
}
Song MainWidget::getRandomSongToPlay(){
  return playlist->getRandomSongToPlay();
}

void MainWidget::setDisplayOrder(int mode){
  datalist->setDisplayOrder(mode);
  refreshDataList();
}


void MainWidget::dropPL( KDNDDropZone *zone){
  QSongList tmp;
  KApplication::setOverrideCursor( waitCursor );
  tmp = drop(zone);
  dataBase->insertPlayList(tmp);
  playlist->addSongs(tmp);
  KApplication::restoreOverrideCursor();
}

void MainWidget::dropDB( KDNDDropZone *zone){
  KApplication::setOverrideCursor( waitCursor );
  dataBase->insert(drop(zone));
  KApplication::restoreOverrideCursor();
}


QSongList MainWidget::drop( KDNDDropZone *zone ){
  unsigned int i;
  QSongList songs, tmpsongs;
  Song     *song;
  QString   tmp;
  QFile     fileTest;
  IOAccess  ioaccess;
  QFileInfo      file;

  for(i=0; i<(zone->getURLList()).count(); i++) {    // process dropped objects
    if (i==0) tmp = (zone->getURLList()).first();
    else tmp = (zone->getURLList()).next();
    KURL url(tmp);
    if(!strcmp(url.protocol(),"file")) {              // is this a local object ?
      file.setFile(url.path());
      if (file.exists()){
	if ((file.fileName()).findRev ( ".mp3", -1, false )!= -1){
	  song= new Song();
	  *song = ioaccess.importMP3File( file.absFilePath() );
	  if(!song->isNull())
	    songs.append(song);
	  else {
	    QString error;
	    error.sprintf(i18n("Error at read file %s"), file.absFilePath().data());
	    //	    status->message( error ), 2000 );
	  }
	}
	else {
	  tmpsongs.clear();
	  if (file.isDir()){
	    tmpsongs = ioaccess.importDirectory(file.absFilePath());
	  }
	  else if (file.fileName().findRev ( ".m3u", -1, false )!= -1){	    
	    tmpsongs = ioaccess.importM3uFileList( file.absFilePath()  );
	  }
	  else {   
	    tmpsongs = ioaccess.importMP3FileList( file.absFilePath() );
	  }
	  for(song=tmpsongs.first(); song!=0; song=tmpsongs.next())
	    songs.append(song);	  
	}
      }
      else debugOutput( cout << "File " << file.absFilePath() << " not found! I skip it." << endl );
    }
    else {
      KMsgBox::message(this,"Drop Error",
			    "Only files and Directories are accepted !!!");
    }
  }
  return songs;
}



void MainWidget::toggleFocus( ){  
  if(playlist->hasFocus()){
    datalist->setFocus();
  }
  else {
    playlist->setFocus();
  }   
}

void MainWidget::resizeEvent( QResizeEvent *tmp ){  
  QWidget::resizeEvent(tmp);
  splitter->setGeometry(5,1,width()-5,height()-1);
}


