// KreateCD - CD recording software for the K desktop environment
//
// 2000 by Alexander Feigl <Alexander.Feigl@gmx.de>
//
// This file is subject to the terms and conditions of the GNU General
// Public License.  See the file COPYING in the main directory of the
// KreateCD distribution for more details.

#include "AudioFileOp.h"
#include "AudioFile.h"
#include "ProcessInterface.h"

#include <stdlib.h>
#include <stdio.h>
#include <strings.h>
#include <unistd.h>
#include <fcntl.h>

#include <sys/stat.h>
#include <sys/ioctl.h>

#include <qtimer.h>
#include <qobject.h>
#include <klocale.h>

#if defined(__FreeBSD__)
# include <machine/soundcard.h>
#else
# include <linux/soundcard.h>
#endif

extern KLocale *locale;

AudioFileOp::AudioFileOp(AudioFile *afil)
 {
  AudioFD=-1;
  ImageFD=-1;
  myAudio=afil;
 }

AudioFileOp::AudioFileOp(AudioFile *afil,int (*fx)(AudioFileOp *afi,const char *fn),const char *fnx):ProcessInterface(fx,this,fnx)
 {
  AudioFD=-1;
  ImageFD=-1;
  myAudio=afil;
 }


AudioFileOp::~AudioFileOp(void)
 {
  if (AudioFD!=-1) close(AudioFD);
 }


int AudioFileOp::readChar(long int *mchar,int words)
 {
  int i;
  if (read(AudioFD,ReadBuffer,1*words)!=1*words) return(0);
  signed temp;
  for (i=0;i<words;++i)
   {
    temp=ReadBuffer[i];
    if (temp>127) temp=-(256-temp);
    *(mchar+i)=temp;
   }
  return(1);
 }

int AudioFileOp::readCharUS(long int *mchar,int words)
 {
  int i;
  if (read(AudioFD,ReadBuffer,1*words)!=1*words) return(0);
  signed temp;
  for (i=0;i<words;++i)
   {
    temp=ReadBuffer[i]-128;
    *(mchar+i)=temp;
   }
  return(1);
 }

int AudioFileOp::readLEShort(long int *mshort,int words)
 {
  int i;
  signed long temp;
  if (read(AudioFD,ReadBuffer,2*words)!=2*words) return(0);
  for (i=0;i<words;++i)
   {
#ifdef WORDS_BIGENDIAN
    temp=ReadBuffer[2*i]+(ReadBuffer[2*i+1]<<8);
#else
    temp=*((short int *) (ReadBuffer+2*i));
#endif
    if (temp>32767) temp=-(65536-temp);
    *(mshort+i)=temp;
   }
  return(1);
 }


int AudioFileOp::readBEShort(long int *mshort,int words)
 {
  int i;
  signed long temp;
  if (read(AudioFD,ReadBuffer,2*words)!=2*words) return(0);
  for (i=0;i<words;++i)
   {
#ifndef WORDS_BIGENDIAN
    temp=ReadBuffer[2*i+1]+(ReadBuffer[2*i]<<8);
#else
    temp=*((short int *) (ReadBuffer+2*i));
#endif
    if (temp>32767) temp=-(65536-temp);
    *(mshort+i)=temp;
   }
  return(1);
 }

int AudioFileOp::readShort(long int *mshort,int words)
 {
  if (myAudio->SampleEndian==AudioFile::Endian_Big) return(readBEShort(mshort,words));
  return(readLEShort(mshort,words));
 }


int AudioFileOp::readLELong(long int *mlong,int words)
 {
#ifdef WORDS_BIGENDIAN
  int i;
  signed long long temp;
  if (read(AudioFD,ReadBuffer,4*words)!=4*words) return(0);
  for (i=0;i<words;++i)
   {
    temp=ReadBuffer[4*i]+(ReadBuffer[4*i+1]<<8)+(ReadBuffer[2*i+2]<<16)+(ReadBuffer[3]<<24);
    if (temp>2147483648LL) temp=temp-4294967296LL;
    *(mlong+i)=temp;
   }
  return(1);
#else
  if (read(AudioFD,mlong,4*words)!=4*words) return(0);
  return(1);
#endif
 }

int AudioFileOp::readBELong(long int *mlong,int words)
 {
#ifndef WORDS_BIGENDIAN
  int i;
  signed long long temp;
  if (read(AudioFD,ReadBuffer,4*words)!=4*words) return(0);
  for (i=0;i<words;++i)
   {
    temp=ReadBuffer[4*i+3]+(ReadBuffer[4*i+2]<<8)+(ReadBuffer[4*i+1]<<16)+(ReadBuffer[4*i]<<24);
    if (temp>2147483648LL) temp=temp-4294967296LL;
    *(mlong+i)=temp;
   }
  return(1);
#else
  if (read(AudioFD,mlong,4*words)!=4*words) return(0);
  return(1);
#endif
 }

int AudioFileOp::readLong(long int *mlong,int words)
 {
  if (myAudio->SampleEndian==AudioFile::Endian_Big) return(readBELong(mlong,words));
  return(readLELong(mlong,words));
 }

int AudioFileOp::fillSampleBuffer(void)
 {
  long int restsamples,readburst;
  restsamples=myAudio->SampleChannels*(myAudio->DataSamples-AudioPos);
  readburst=(restsamples<=(Read_Cachesize-(Read_Cachesize%myAudio->SampleChannels)))?restsamples:(Read_Cachesize-(Read_Cachesize%myAudio->SampleChannels));
  if (readburst==0) return(0);
  switch (myAudio->SampleSize)
   {
    case AudioFile::Samples_8Bit:
     {
      int i;
      if (!readCharUS(AudioBuffer,readburst)) return(0);
      AudioPointer=0;
      AudioPos+=readburst/myAudio->SampleChannels;
      AudioCached=readburst;
      for (i=0;i<readburst;++i)
       {
        AudioBuffer[i]=AudioBuffer[i]<<8;
       }
      return(1);
     }
    case AudioFile::Samples_16Bit:
     {
      if (!readShort(AudioBuffer,readburst)) return(0);
      AudioPointer=0;
      AudioPos+=readburst/myAudio->SampleChannels;
      AudioCached=readburst;
      return(1);
     }
    case AudioFile::Samples_32Bit:
     {
      if (!readLong(AudioBuffer,readburst)) return(0);
      AudioPointer=0;
      AudioPos+=readburst/myAudio->SampleChannels;
      AudioCached=readburst;
      return(1);
     }
   }
  return(0);
 }

int AudioFileOp::flushCDDABuffer(void)
 {
  if (ImageContent==0) return(1);
  if (write(ImageFD,ImageBuffer,2*ImageContent)!=2*ImageContent) return(0);
  ImageContent=0;
  return(1);
 }

int AudioFileOp::openWrite(const char *filename)
 {
  closeWrite();
  DisableSwap=0;
  ImageBuffer=(unsigned char *)malloc(Write_Cachesize*2);
  if (ImageBuffer==0)
   {
    return(0);
   }

  ImageFD=open(filename,O_WRONLY|O_CREAT|O_TRUNC,0644);
  if (ImageFD==-1)
   {
    free(ImageBuffer);
    return(0);
   }
  ImageContent=0;
  return(1);
 }

void AudioFileOp::closeWrite(void)
 {
  if (ImageFD!=-1)
   {
    close(ImageFD);
    ImageFD=-1;
   }
 }

int AudioFileOp::openRead(const char *mfilename)
 {
  if (AudioFD!=-1) close(AudioFD);
  AudioFD=open(mfilename,O_RDONLY);
  if (AudioFD==-1) return(0);
  AudioCached=0;
  AudioPos=0;
  AudioPointer=0;
  ReadBuffer=(unsigned char *) malloc(Read_Cachesize*4);
  if (ReadBuffer==0)
   {
    close(AudioFD);
    AudioFD=-1;
   }
  AudioBuffer=(long int *) malloc(Read_Cachesize*sizeof(long int));
  if (AudioBuffer==0)
   {
    close(AudioFD);
    AudioFD=-1;
    free(ReadBuffer);

   }
  return(1);
 }


void AudioFileOp::closeRead(void)
 {
  if (AudioFD!=-1)
   {
    close(AudioFD);
    free(ReadBuffer);
    free(AudioBuffer);
   }
  AudioFD=-1;
 }

int AudioFileOp::skipRead(int bytes)
 {
  if (lseek(AudioFD,bytes,SEEK_CUR)==-1) return(0);
  return(1);
 }


int AudioFileOp::positionSample(long int sample)
 {
  unsigned long position;
  if  ( (sample>=myAudio->DataSamples) || (sample<0) ) return(0);
  position=myAudio->DataPos+sample*myAudio->SampleBytes;

  if (lseek(AudioFD,position,SEEK_SET)==-1) return(0);
  AudioPointer=0;
  AudioCached=0;
  AudioPos=sample;
  return(1);
 }


