/*
 *   kwrl - a little VRML 2.0 editor
 *   Copyright (C) 1998,99  Mark R. Stevens
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* local headers */
#include <SFToken.h>
#include <Text.h>


/*************************************************************************/
void Text::eventIn(SFString &eventType, SFNode *eventData)
{

#ifdef DEBUG_EVENTS
  cout << "\nEntered Text::eventIn (" << DEF << ")\n";
#endif 

    /* Determine what action to take based on incoming event */
  if ((eventType == "string") || (eventType == "set_string")) {
    MFString *n = (MFString *) eventData;
    astring = *n;

  } else if ((eventType == "fontStyle") || (eventType == "set_fontStyle")) {
    FontStyle *n = (FontStyle *) eventData;
    fontStyle = *n;

  } else if ((eventType == "maxExtent") || (eventType == "set_maxExtent")) {
    SFFloat *n = (SFFloat *) eventData;
    maxExtent = *n;

  } else if ((eventType == "length") || (eventType == "set_length")) {
    MFFloat *n = (MFFloat *) eventData;
    length = *n;

  }

  /* re-generate the call list */
  SFVec3f min, max;
  prepare(min, max);

}
/*************************************************************************/

/*************************************************************************/
void Text::prepare(SFVec3f &, SFVec3f &)
{

#ifdef DEBUG_PREPARE
  cout << "\nEntered Text::prepare (" << DEF << ")\n";
#endif 

  /* free the call list if one already exists */
  if (CallNum != 0) glDeleteLists(CallNum, 1);

  /* create the call list */
  CallNum = glGenLists(1);
  glNewList(CallNum, GL_COMPILE_AND_EXECUTE);

  /* if a string is present, draw */
  if (astring.num() != 0) {

    /* first we are going to see how many lines we have, and what the
    ** widest is. */
    int NumLines = astring.num();
    unsigned int MaxCharsPerLine = 0;
    for (int i = 0; i < NumLines; i++) {
      if (MaxCharsPerLine < strlen(astring[i])) {
	MaxCharsPerLine = strlen(astring[i]);
      }
    }

    /* Set the translation to move the string to the center */
    glPushMatrix();
    glTranslatef(-float(MaxCharsPerLine) * fontStyle() / 2.0, 
		 -float(NumLines       ) * fontStyle() / 2.0, 
		 0.0);

    /* for each string */
    for (int i = 0; i < astring.num(); i++) {

      /* for each character in that string */
      for (unsigned int j = 0; j < strlen(astring[i]); j++) {

	/* set up the texture map */
	fontStyle.render(astring[i][j]);
	
	/* we are going to have to render several quads for each character */
	glBegin(GL_QUADS);

	/* Set the normal */
	glNormal3f  (0.0, 0.0,  -1.0);
	
	/* set up the vertices */	
	int   ii = astring.num() - i - 1;
	float yll = ii       * fontStyle();// * 0.35278;
	float yur = (ii + 1) * fontStyle();// * 0.35278;
	float xll = j        * fontStyle();// * 0.35278;
	float xur = (j + 1)  * fontStyle();// * 0.35278;

	glTexCoord2f(0.01, 0.01);
	glVertex3f(xll, yll, 0.0);

	glTexCoord2f(0.99, 0.01);
    	glVertex3f(xur, yll, 0.0);

	glTexCoord2f(0.99, 0.99);
 	glVertex3f(xur, yur, 0.0);

	glTexCoord2f(0.01, 0.99);
 	glVertex3f(xll, yur, 0.0);

	/* we are done rendering */
	glEnd();	

      }
    }

    /* restore the matrix */
    glPopMatrix();

  }

  /* finish up the list */
  glEndList();

}
/*************************************************************************/

/*************************************************************************/
void Text::render(SFRenderInfo &)
{

#ifdef DEBUG_RENDER
  cout << "\nEntered Text::render (" << DEF << ")\n";
#endif 

  /* call the render function */
  glCallList(CallNum);
  
#ifdef DEBUG_RENDER
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif
 
}
/*************************************************************************/

/*************************************************************************/
void Text::parse(char *header, istream &InFile)
{

  /* we are valid */
  isValid() = true;

  /* a token in the file */
  SFToken Token;

  /* the required labels */
  bool TokenFound  = false;
  bool BeginBracketFound   = false;

  /* commence to search for the matching bracket */
  while (! InFile.eof()) {
    
    /* Remember where the token started from */
    Token.GetToken(InFile);
    
#ifdef DEBUG_PARSE    
    cout << "\tText: (" << Token() << ")\n";
#endif
    
    /* based on what token was found, continue parsing */
    if (Token == "DEF") {
      Token.GetToken(InFile);
      DEF = Token();

    } else if (Token == "USE") {
      Token.GetToken(InFile);
      USE = Token();
      unsatisfiedUSE() = true;
      return;

    } else if (Token == "maxExtent") {
      maxExtent.parse(header, InFile);

    } else if (Token == "string") {
      astring.parse(header, InFile);

    } else if (Token == "length") {
      length.parse(header, InFile);

    } else if (Token == "fontStyle") {
      fontStyle.parse(header, InFile);

    } else if (Token == "Text") {
      TokenFound = true;

    } else if (Token == "{") {
      BeginBracketFound = true;

    } else if (Token == "}") {
      break;
      
    } else {
      parseWarning(Token());

    }
  }

  /* if we did not find the material token we are in trouble */
  if (TokenFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
    cerr << "\tDid not find expected identifier token.\n";
    exit(0);
  }
  if (BeginBracketFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
    cerr << "\tDid not find expected \"{\" token.\n";
    exit(0);
  }
}
/*************************************************************************/
