/*
 *   kwrl - a little VRML 2.0 viewer
 *   Copyright (C) 1998,99  Mark R. Stevens
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* local headers */
#include <SFToken.h>
#include <FontStyle.h>

/* include the rasterized fonts. */
#include <BitmapFonts.h>

/*************************************************************************/
void FontStyle::render(char s) 
{

#ifdef DEBUG_RENDER
  cout << "\nEntered FontStyle::render (" << DEF << ")\n";
#endif 

  /* the soon to be created texture map */
  float Texture[16 * 16 * 4];

  /* zero out the texture */
  for (int i = 0; i < (16 * 16 * 4); i++) Texture[i] = 0.0;

  /* get the current color */
  GLfloat mat_diffuse[]   = { 0.8, 0.8, 0.8, 1.0 };
  glGetMaterialfv(GL_FRONT, GL_DIFFUSE, mat_diffuse);

  /* fill in the texture values */
  for (int i = 0, r = 0; i < 16; i++) {

    /* map the font */
    if (family == "SERIF") {
      if      (style == "PLAIN")       r = SerifPlainFont     [int(s - 32)][i];
      else if (style == "BOLD")        r = SerifBoldFont      [int(s - 32)][i];
      else if (style == "ITALIC")      r = SerifItalicFont    [int(s - 32)][i];
      else if (style == "BOLD ITALIC") r = SerifBoldItalicFont[int(s - 32)][i];
    } else if (family == "SANS") {
      if      (style == "PLAIN")       r = SansPlainFont     [int(s - 32)][i];
      else if (style == "BOLD")        r = SansBoldFont      [int(s - 32)][i];
      else if (style == "ITALIC")      r = SansItalicFont    [int(s - 32)][i];
      else if (style == "BOLD ITALIC") r = SansBoldItalicFont[int(s - 32)][i];
    } else if (family == "TYPEWRITER") {
      if      (style == "PLAIN")       r = TypewriterPlainFont     [int(s - 32)][i];
      else if (style == "BOLD")        r = TypewriterBoldFont      [int(s - 32)][i];
      else if (style == "ITALIC")      r = TypewriterItalicFont    [int(s - 32)][i];
      else if (style == "BOLD ITALIC") r = TypewriterBoldItalicFont[int(s - 32)][i];
    }

    /* convert into a texture map */
    for (int j = 15; j >= 0; j--, r >>= 1) {      
      int t = r & 1;
      int idx = 4 * (j + 16 * (16 - i - 1));
      if (t) {
	Texture[idx + 0] = mat_diffuse[0]; 
	Texture[idx + 1] = mat_diffuse[1]; 
	Texture[idx + 2] = mat_diffuse[2]; 
	Texture[idx + 3] = 1.0; 
      }
    }
  }

  /* set up for texture mapping */
  glPixelStorei  (GL_UNPACK_ALIGNMENT, sizeof(float));
  glTexImage2D   (GL_TEXTURE_2D,
		  0,
		  4,
		  16,
		  16,
		  0,
		  GL_RGBA,
		  GL_FLOAT,
		  (GLvoid *) Texture);
  glTexParameteri(GL_TEXTURE_2D,  GL_TEXTURE_MIN_FILTER, GL_NEAREST);
  glTexParameteri(GL_TEXTURE_2D,  GL_TEXTURE_MAG_FILTER, GL_NEAREST);
  glTexParameteri(GL_TEXTURE_2D,  GL_TEXTURE_WRAP_S,     GL_CLAMP);
  glTexParameteri(GL_TEXTURE_2D,  GL_TEXTURE_WRAP_T,     GL_CLAMP);
  glTexEnvi      (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE,   GL_MODULATE);
  glHint         (GL_PERSPECTIVE_CORRECTION_HINT, GL_FASTEST);
  glEnable       (GL_TEXTURE_2D);

#ifdef DEBUG_RENDER
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif

}
/*************************************************************************/

/*************************************************************************/
void FontStyle::parse(char *header, istream &InFile)
{

  /* we are valid */
  isValid() = true;

  /* a token in the file */
  SFToken Token;

  /* the required labels */
  bool TokenFound  = false;
  bool BeginBracketFound   = false;

  /* commence to search for the matching bracket */
  while (! InFile.eof()) {
    
    /* Remember where the token started from */
    Token.GetToken(InFile);
    
#ifdef DEBUG_PARSE    
    cout << "\tFontStyle: (" << Token() << ")\n";
#endif
    
    /* based on what token was found, continue parsing */
    if (Token == "DEF") {
      Token.GetToken(InFile);
      DEF = Token();

    } else if (Token == "USE") {
      Token.GetToken(InFile);
      USE = Token();
      unsatisfiedUSE() = true;
      return;

    } else if (Token == "size") {
      size.parse(header, InFile);

    } else if (Token == "family") {
      family.parse(header, InFile);

    } else if (Token == "style") {
      style.parse(header, InFile);

    } else if (Token == "horizontal") {
      horizontal.parse(header, InFile);

    } else if (Token == "leftToRight") {
      leftToRight.parse(header, InFile);

    } else if (Token == "topToBottom") {
      topToBottom.parse(header, InFile);

    } else if (Token == "language") {
      language.parse(header, InFile);

    } else if (Token == "spacing") {
      spacing.parse(header, InFile);

    } else if (Token == "justify") {
      justify.parse(header, InFile);

    } else if (Token == "FontStyle") {
      TokenFound = true;

    } else if (Token == "NULL") {
      return;
      
    } else if (Token == "{") {
      BeginBracketFound = true;

    } else if (Token == "}") {
      break;
      
    } else {
      parseWarning(Token());

    }
  }

  /* if we did not find the material token we are in trouble */
  if (TokenFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::operator >>)\n";
    cerr << "\tDid not find expected identifier token.\n";
    exit(0);
  }
  if (BeginBracketFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::operator >>)\n";
    cerr << "\tDid not find expected \"{\" token.\n";
    exit(0);
  }
}
/*************************************************************************/
