/*
 *   kwrl - a little VRML 2.0 editor
 *   Copyright (C) 1998,99  Mark R. Stevens
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* local headers */
#include <SFToken.h>
#include <Appearance.h>

/*************************************************************************/
void Appearance::prepare(SFVec3f &min, SFVec3f &max)
{

#ifdef DEBUG_PREPARE
  cout << "\nEntered Appearance::prepare (" << DEF << ")\n";
#endif 

  /* prepare the material */
  if (material.isValid()) material.prepare(min, max);

  /* prepare the texture */
  if (texture.isValid()) texture.prepare(min, max);
  
  /* prepare the texture transform */
  if (textureTransform.isValid()) textureTransform.prepare(min, max);
  
}
/*************************************************************************/

/*************************************************************************/
void Appearance::render(SFRenderInfo &SFRI)
{

#ifdef DEBUG_RENDER
  cout << "\nEntered Appearance::render (" << DEF << ")\n";
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif 

  /* render the material */
  if (material.isValid()) material.render(SFRI);
  else                    glDisable(GL_LIGHTING);
  
  /* render the texture transform */
  if (textureTransform.isValid()) {
    textureTransform.render(SFRI);
  } else {
    glMatrixMode(GL_TEXTURE);  
    glLoadIdentity();
    glMatrixMode(GL_MODELVIEW);  
  }

  /* render the the texture */
  if (texture.isValid()) texture.render(SFRI);

#ifdef DEBUG_RENDER
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
  cout << "\tDone Appearance\n";
#endif

}
/*************************************************************************/

/*************************************************************************/
void Appearance::eventIn(SFString &eventType, SFNode *eventData)
{

#ifdef DEBUG_EVENTS
  cout << "\nEntered Appearance::eventIn (" << DEF << ")\n";
#endif 
  
  /* Determine what action to take based on incoming event */
  if        ((eventType == "material")    || (eventType == "set_material")) {
    Material *n = (Material *) eventData;
    material    = *n;
    
  } else if ((eventType == "texture") || (eventType == "set_texture")) {
    Texture *n = (Texture *) eventData;
    texture    = *n;

  } else if ((eventType=="textureTransform")||(eventType=="set_textureTransform")) {
    TextureTransform *n = (TextureTransform *) eventData;
    textureTransform    = *n;

  }
}
/*************************************************************************/

/*************************************************************************/
void Appearance::parse(char *header, istream &InFile)
{

  /* the node is valid */
  isValid() = true;

  /* a token in the file */
  SFToken Token;

  /* the required labels */
  char TokenFound  = false;
  char BeginBracketFound   = false;

  /* commence to search for the matching bracket */
  while (! InFile.eof()) {
    
    /* Remember where the token started from */
    Token.GetToken(InFile);

#ifdef DEBUG_PARSE
    cout << "\tAppearance: (" << Token() << ")\n";
#endif

    /* based on what token was found, continue parsing */
    if (Token == "DEF") {
      Token.GetToken(InFile);
      DEF = Token();
      
    } else if (Token == "USE") {
      Token.GetToken(InFile);
      USE = Token();
      unsatisfiedUSE() = true;
      return;

    } else if (Token == "material") {
      material.parse(header, InFile);

    } else if (Token == "texture") {
      texture.parse(header, InFile);

    } else if (Token == "textureTransform") {
      textureTransform.parse(header, InFile);
      
    } else if (Token == "Appearance") {
      TokenFound = true;

    } else if (Token == "{") {
      BeginBracketFound = true;

    } else if (Token == "}") {
      break;
      
    } else {
      parseWarning(Token());

    }
  }

  /* if we did not find the material token we are in trouble */
  if (TokenFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse()\n";
    cerr << "\tDid not find expected identifier token.\n";
    exit(0);
  }
  if (BeginBracketFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse()\n";
    cerr << "\tDid not find expected \"{\" token.\n";
    exit(0);
  }
}
/*************************************************************************/
