/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** mainwin.cpp
*/


#include <version.h>

#include <kapp.h>
#include <klocale.h>
#include <kfiledialog.h>
#include <kmsgbox.h>
#include <kstatusbar.h>

#include <qkeycode.h>

#include <misc.h>

#include "viewwin.h"

#include "../dialogs/dialogs.h"
#include "../dialogs/map2mesh.h"
#include "../dialogs/preferencedialog.h"
#include "../dialogs/aboutdialog.h"

#include "mainwin.h"
#include "mainwin.moc"

#include "../texture/textureview.h"

#include "icons/toolbar/fov.xpm"
#include "icons/toolbar/moveview.xpm"
#include "icons/toolbar/rotateview.xpm"

#include "icons/toolbar/scale.xpm"
#include "icons/toolbar/translate.xpm"
#include "icons/toolbar/rotate.xpm"

#include "icons/toolbar/3dview.xpm"
#include "icons/toolbar/xyview.xpm"
#include "icons/toolbar/xzview.xpm"
#include "icons/toolbar/zyview.xpm"

#include "icons/toolbar/axe-x.xpm"
#include "icons/toolbar/axe-y.xpm"
#include "icons/toolbar/axe-z.xpm"

#include "icons/toolbar/render.xpm"
#include "icons/toolbar/renderlow.xpm"

#include "icons/toolbar/up.xpm"
#include "icons/toolbar/down.xpm"
//#include "icons/toolbar/top.xpm"
//#include "icons/toolbar/bottom.xpm"

#include "icons/menu/fileopen2.xpm"
#include "icons/menu/filefloppy.xpm"
#include "icons/menu/filenew.xpm"
#include "icons/menu/rtree.xpm"

#include "icons/tree/obj.xpm"
#include "icons/tree/csg.xpm"
#include "icons/tree/blob.xpm"
#include "icons/tree/box.xpm"
#include "icons/tree/sphere.xpm"
#include "icons/tree/cylinder.xpm"
#include "icons/tree/cone.xpm"
#include "icons/tree/torus.xpm"
#include "icons/tree/mesh.xpm"
#include "icons/tree/triangle.xpm"
#include "icons/tree/text.xpm"
#include "icons/tree/script.xpm"
#include "icons/tree/heightfield.xpm"
#include "icons/tree/plane.xpm"
#include "icons/tree/ref.xpm"
#include "icons/tree/refptr.xpm"
#include "icons/tree/lightpoint.xpm"
#include "icons/tree/spotlight.xpm"
#include "icons/tree/arealight.xpm"
#include "icons/tree/camera2.xpm"
#include "icons/tree/ear.xpm"



MainWin::MainWin(char *name) :
	KTMainWindow(name)
{
	Preferences		pref;
        int			i,loaded;
	extern KApplication	*k4deapp;


	stbar = 0;

	
	createMenu();	// !! Menus *vor* den Widgets generieren !!
	createWidgets();
	
	createStatusBar(i18n("Welcome to K4DE"));
	
	changed = 0;
	projectname = 0;

	newClicked();

	loaded = FALSE;
	for(i = 0;i < k4deapp->argc() && !loaded;i++)
	{
		QString		qstr;
		
		qstr = k4deapp->argv()[i];
		if(qstr.find(".3dds",0,FALSE) >= 0)
		{
			char		name[1024];
			
#ifdef DEBUG			
			printf("Try to open file %s\n",(const char*)qstr);
#endif
			if(qstr != "")
			{
				strcpy(name,qstr.data());

				lastread = qstr;
				if(lastread.findRev('/'))
					lastread.remove(lastread.findRev('/') + 1,lastread.length() - lastread.findRev('/'));

				if(worldtree->load(name) == 0)
				{
					qstr = projectname = strdup(name);
                         	
					changed = 0;
	
					pref.setRecentFile(qstr);
					createRecentMenu();

					qstr = QString(i18n("Loaded")) + " " + qstr;		
					createStatusBar(qstr);
					
					loaded = TRUE;
				}
				createWindowTitle();
			}
		}
	}
	
	lastread = pref.getPath(pref.getDataPath());
	lastwrite = pref.getPath(pref.getDataPath());
		
	setMinimumSize(500,200);
	resize(pref.getMainWindowWidth(),pref.getMainWindowHeight());
}

MainWin::~MainWin()
{
	Preferences		pref;

	if(projectname)
		free(projectname);
		
	pref.setMainWindowSize(width(),height());
	pref.save();

	//delete viewwidget;
	//delete worldtree;
	
#ifdef DEBUG
	printf("destructed MainWin\n");
#endif
}

void	MainWin::resizedView(int w,int h)
{
	QString		str,sx,sy;
	
	sx.setNum(w);
	sy.setNum(h);
	
	str = QString(i18n("Viewarea")) + " " + sx + " x " + sy;
	stbar->changeItem(str,0);
}

void	MainWin::displayMessage(QString s)
{
	extern KApplication	*k4deapp;
	
	stbar->changeItem(s,1);
	k4deapp->processEvents(10);
}

void	MainWin::createStatusBar(QString message)
{
	QString		str,sx,sy;
	
	if(!stbar)
	{
		stbar = new KStatusBar(this);
		setStatusBar(stbar);

		stbar->setInsertOrder(KStatusBar::RightToLeft);
	
		str = QString(i18n("Viewarea")) + " 8888 x 8888";
		stbar->insertItem(str,0);
		stbar->insertItem(message,1);
	}

	sx.setNum(viewwidget->width());
	sy.setNum(viewwidget->height());
	
	str = QString(i18n("Viewarea")) + " " + sx + " x " + sy;
	stbar->changeItem(str,0);
	stbar->changeItem(message,1);
}

void	MainWin::createWindowTitle()
{
	QString		title;

	title = PROGNAME " V" VERSION "  [";
		
	if(projectname)
  		title += projectname;
	else  		
  		title += i18n("Untitled");

	title += "]";
			
	setCaption(title);
	
	emit changedProjectName(projectname);
}

void	MainWin::createWidgets()
{
	vpanner = new KNewPanner(this,0,KNewPanner::Horizontal,KNewPanner::Percent,40);
	hpanner = new KNewPanner(vpanner,0,KNewPanner::Vertical,KNewPanner::Absolute,300);
	vpanner2 = new KNewPanner(vpanner,0,KNewPanner::Horizontal,KNewPanner::Percent,99);
        vpanner2->setEnabled(0);

	worldtree = new WorldTree(hpanner);
	viewwidget = new ViewWidget(hpanner,"View",0,false);

	tview = new TextureView(vpanner2,i18n("Textures") );

	tbtxtview=new TextureToolBar(this,tview);
	tlview=tview->getTextureListView();

	connect(tlview,SIGNAL(message(QString)),this,SLOT(displayMessage(QString) ) );

	animwidget = new AnimControlWidget(vpanner2);
	animwidget->hide();

	hpanner->activate(worldtree,viewwidget);
	vpanner2->activate(tview,animwidget);
	vpanner->activate(hpanner,vpanner2);

	setView(vpanner);

//	connect(worldtree,SIGNAL(emptyWorld()),this,SLOT(emptyMainViewList()));
	
//	connect(k4deapp,SIGNAL(shutDown()),this,SLOT(quitClicked()));
	
// forward
	connect(worldtree,SIGNAL(changeWorld(world*)),this,SIGNAL(changedWorld(world*)));
	connect(animwidget,SIGNAL(changeWorld(world*)),this,SIGNAL(changedWorld(world*)));
	
// connect
	connect(this,SIGNAL(changedWorld(world*)),viewwidget,SLOT(setWorld(world*)));
	connect(this,SIGNAL(changedWorld(world*)),animwidget,SLOT(setWorld(world*)));
	connect(this,SIGNAL(changedWorld(world*)),worldtree,SLOT(setWorld(world*)));
	connect(this,SIGNAL(changedWorld(world*)),tview,SLOT(setWorld(world*)));

// replaced
/*
	connect(worldtree,SIGNAL(changeWorld(world*)),viewwidget,SLOT(setWorld(world*)));
	connect(worldtree,SIGNAL(changeWorld(world*)),animwidget,SLOT(setWorld(world*)));
	connect(worldtree,SIGNAL(changeWorld(world*)),tview,SLOT(setWorld(world*)));
	connect(animwidget,SIGNAL(changeWorld(world*)),viewwidget,SLOT(setWorld(world*)));
	connect(animwidget,SIGNAL(changeWorld(world*)),worldtree,SLOT(setWorld(world*)));
	connect(animwidget,SIGNAL(changeWorld(world*)),tview,SLOT(setWorld(world*)));
*/

// forward	
	connect(worldtree,SIGNAL(redraw()),this,SIGNAL(redraw()));
	connect(viewwidget,SIGNAL(redraw()),this,SIGNAL(redraw()));
	
// connect
	connect(this,SIGNAL(redraw()),viewwidget,SLOT(draw()));
	
// replaced
/*
	connect(worldtree,SIGNAL(redraw()),viewwidget,SLOT(draw()));
*/

// forward
	connect(worldtree,SIGNAL(changedModel()),this,SIGNAL(changedModel()));
	connect(viewwidget,SIGNAL(changedModel()),this,SIGNAL(changedModel()));
	
// connect
	connect(this,SIGNAL(changedModel()),this,SLOT(setChanged()));
	
// replaced
/*
	connect(worldtree,SIGNAL(changedModel()),this,SLOT(setChanged()));
	connect(viewwidget,SIGNAL(changedModel()),this,SLOT(setChanged()));
*/


// forward
	connect(worldtree,SIGNAL(changedSelected(base*)),this,SIGNAL(changedSelected(base*)));
	
// connect
	connect(this,SIGNAL(changedSelected(base*)),viewwidget,SLOT(setSelected(base*)));
	
// replaced
/*
	connect(worldtree,SIGNAL(setSelected(base*)),viewwidget,SLOT(setSelected(base*)));
*/
	
// forward
	connect(worldtree,SIGNAL(changedViewName()),this,SIGNAL(changedViewName()));
	
// connect
	// none yet
	
	connect(animwidget,SIGNAL(exportPOVanim(const char*)),this,SLOT(exportPOVanim(const char*)));
	connect(animwidget,SIGNAL(drawanim()),viewwidget,SLOT(drawanim()));
	connect(viewwidget,SIGNAL(exportPOV(const char*,int)),this,SLOT(exportPOV(const char*,int)));
	connect(viewwidget,SIGNAL(resizedView(int,int)),this,SLOT(resizedView(int,int)));
	connect(viewwidget,SIGNAL(report(QString)),this,SLOT(displayMessage(QString)));
	connect(worldtree,SIGNAL(report(QString)),this,SLOT(displayMessage(QString)));
	connect(tview,SIGNAL(changedTexture(texture*)),worldtree,SLOT(changeTexture(texture*)));
	connect(viewwidget,SIGNAL(finishedRendering()),this,SLOT(autoSavePicture()));
	
	mainbar = new KToolBar(this);

	mainbar->insertButton(QPixmap((const char**)filenew_xpm),1,SIGNAL(clicked()),this,SLOT(newClicked()),true,i18n("New"));
	mainbar->insertButton(QPixmap((const char**)fileopen_xpm),2,SIGNAL(clicked()),this,SLOT(openClicked()),true,i18n("Open"));
	mainbar->insertButton(QPixmap((const char**)filefloppy_xpm),3,SIGNAL(clicked()),this,SLOT(saveClicked()),true,i18n("Save"));
	mainbar->insertSeparator();
	mainbar->insertSeparator();
      	mainbar->insertButton(QPixmap((const char**)obj_xpm),4,Mobjects,true,i18n("Objects"));
	mainbar->insertSeparator();
     	mainbar->insertButton(QPixmap((const char**)up_xpm),5,SIGNAL(clicked()),worldtree,SLOT(upper()),true,i18n("Object lower"));
     	mainbar->insertButton(QPixmap((const char**)down_xpm),6,SIGNAL(clicked()),worldtree,SLOT(lower()),true,i18n("Object upper"));
	mainbar->insertSeparator();
	mainbar->insertSeparator();



        tbview = new KToolBar(this);
	tbview->insertButton(QPixmap((const char**)fov_xpm),1,true,i18n("Change camera focus"));
	tbview->insertButton(QPixmap((const char**)moveview_xpm),2,true,i18n("Change camera position"));
	tbview->insertButton(QPixmap((const char**)rotateview_xpm),3,true,i18n("Change camera angle"));
	tbview->setToggle(1);
	tbview->setToggle(2);
	tbview->setToggle(3);
	tbview->insertButton(QPixmap((const char**)scale_xpm),4,true,i18n("Change object size"));
	tbview->insertButton(QPixmap((const char**)translate_xpm),5,true,i18n("Change object position"));
	tbview->insertButton(QPixmap((const char**)rotate_xpm),6,true,i18n("Change object angle"));
	tbview->setToggle(4);
	tbview->setToggle(5);
	tbview->setToggle(6);
	tbview->insertSeparator();
	tbview->insertButton(QPixmap((const char**)_3dview_xpm),7,true,i18n("Change camera to 3d-projection"));
	tbview->insertButton(QPixmap((const char**)xyview_xpm),8,true,i18n("Change camera to xy-projection"));
	tbview->insertButton(QPixmap((const char**)xzview_xpm),9,true,i18n("Change camera to xz-projection"));
	tbview->insertButton(QPixmap((const char**)zyview_xpm),10,true,i18n("Change camera to zy-projection"));
	tbview->setToggle(7);
	tbview->setToggle(8);
	tbview->setToggle(9);
	tbview->setToggle(10);
	tbview->insertSeparator();
	tbview->insertButton(QPixmap((const char**)axe_x_xpm),11,true,i18n("Change at x-axe"));
	tbview->insertButton(QPixmap((const char**)axe_y_xpm),12,true,i18n("Change at y-axe"));
	tbview->insertButton(QPixmap((const char**)axe_z_xpm),13,true,i18n("Change at z-axe"));
	tbview->setToggle(11);
	tbview->setToggle(12);
	tbview->setToggle(13);
	tbview->insertSeparator();
	tbview->insertButton(QPixmap((const char**)render_xpm),14,true,i18n("Render the view"));
	tbview->insertButton(QPixmap((const char**)renderlow_xpm),15,true,i18n("Render the view (low quality)"));
	tbview->insertButton(QPixmap((const char**)filefloppy_xpm),16,SIGNAL(clicked()),this,SLOT(savePicture()),true,i18n("Save the view as image"));
	tbview->setButton(1,true);
	tbview->setButton(7,true);
	tbview->setButton(11,true);
	tbview->setButton(12,true);
	tbview->setButton(13,true);



	connect(tlview,SIGNAL(newTextureSelection(textureBase*)),tbtxtview,SLOT(newTextureSel(textureBase*)) );
	connect(tbtxtview,SIGNAL(update()),tlview,SLOT(updateAll()) );

	tb[1]=addToolBar(mainbar);
	tb[2]=addToolBar(tbview);
	tb[3]=addToolBar(tbtxtview);

	connect(tbview,SIGNAL(clicked(int)),this,SLOT(setViewMode(int)));
	connect(tbtxtview,SIGNAL(clicked(int)),tview,SLOT(MenuItemSel(int)));
	connect(this,SIGNAL(viewMode(int)),viewwidget,SLOT(setViewMode(int)));
	connect(this,SIGNAL(projectionMode(int)),viewwidget,SLOT(setProjectionMode(int)));
	connect(this,SIGNAL(axeMode(int)),viewwidget,SLOT(setAxeMode(int)));
	connect(this,SIGNAL(render(int)),viewwidget,SLOT(startRendering(int)));
}

void	MainWin::createRecentMenu()
{
	Preferences	prefs;
	int		i,t;

	i = prefs.numRecentFiles();

	Mopenrecent->clear();
     	
	for(t = 0;t < i;t++)
	{
		QString		qstr;

		qstr = prefs.recentFile(t);
		Mopenrecent->insertItem(qstr,t + 1000);
	}
}

void	MainWin::createMenu()
{
	Preferences	prefs;

	Mopenrecent = new MPopupMenu;
	CHECK_PTR(Mopenrecent);
	
	connect(Mopenrecent,SIGNAL(activated(int)),this,SLOT(openRecentClicked(int)));
	
	createRecentMenu();
	
	Mfile = new MPopupMenu;
	CHECK_PTR(Mfile);

	mid[0] = Mfile->insertItem(QPixmap((const char**)filenew_xpm),i18n("New"),this,SLOT(newClicked()),CTRL+Key_N);
	Mfile->insertSeparator();
	mid[1] = Mfile->insertItem(QPixmap((const char**)fileopen_xpm),i18n("Open"),this,SLOT(openClicked()),CTRL+Key_O);

	mid[2] = Mfile->insertItem(i18n("Open Recent"),Mopenrecent);

	Mfile->insertSeparator();
	mid[3] = Mfile->insertItem(QPixmap((const char**)filefloppy_xpm),i18n("Save"),this,SLOT(saveClicked()),CTRL+Key_S);
	mid[4] = Mfile->insertItem(i18n("Save As"),this,SLOT(saveAsClicked()),0);
	Mfile->insertSeparator();
	mid[5] = Mfile->insertItem(i18n("Dump"),this,SLOT(dumpClicked()),0);
	Mfile->insertSeparator();
	mid[7] = Mfile->insertItem(i18n("Options"),this,SLOT(optionsClicked()),0);
	Mfile->insertSeparator();
	mid[6] = Mfile->insertItem(i18n("Exit"),this,SLOT(quitClicked()),CTRL+Key_Q);


/*
	Medit = new MPopupMenu;
	CHECK_PTR(Medit);

	mid[50] = Medit->insertItem(i18n("Copy"),this,SLOT(copyClicked()),0);
	mid[51] = Medit->insertItem(i18n("Cut"),this,SLOT(cutClicked()),0);
	mid[52] = Medit->insertItem(i18n("Paste"),this,SLOT(pasteClicked()),0);
	Medit->insertSeparator();
	mid[53] = Medit->insertItem(i18n("Add AnimScript"),this,SLOT(addAnimClicked()),0);
	mid[54] = Medit->insertItem(i18n("Delete AnimScript"),this,SLOT(delAnimClicked()),0);
	mid[55] = Medit->insertItem(i18n("Edit AnimScript"),this,SLOT(editAnimClicked()),0);
*/

	Mobjects = new MPopupMenu;
	CHECK_PTR(Mobjects);

	mid[10] = Mobjects->insertItem(QPixmap((const char**)obj_xpm),i18n("Add Object"),this,SLOT(addObjectClicked()),0);
	mid[11] = Mobjects->insertItem(QPixmap((const char**)csg_xpm),i18n("Add CSG"),this,SLOT(addCSGClicked()),0);
	mid[12] = Mobjects->insertItem(QPixmap((const char**)blob_xpm),i18n("Add Blob"),this,SLOT(addBlobClicked()),0);
	Mobjects->insertSeparator();
	mid[13] = Mobjects->insertItem(QPixmap((const char**)ref_xpm),i18n("Add ReferenceObject"),this,SLOT(addRefClicked()),0);
	Mobjects->insertSeparator();
	mid[14] = Mobjects->insertItem(QPixmap((const char**)sphere_xpm),i18n("Add Sphere"),this,SLOT(addSphereClicked()),0);
	mid[15] = Mobjects->insertItem(QPixmap((const char**)cylinder_xpm),i18n("Add Cylinder"),this,SLOT(addCylinderClicked()),0);
	Mobjects->insertSeparator();
	mid[16] = Mobjects->insertItem(QPixmap((const char**)box_xpm),i18n("Add Box"),this,SLOT(addBoxClicked()),0);
	mid[17] = Mobjects->insertItem(QPixmap((const char**)cone_xpm),i18n("Add Cone"),this,SLOT(addConeClicked()),0);
	mid[18] = Mobjects->insertItem(QPixmap((const char**)torus_xpm),i18n("Add Torus"),this,SLOT(addTorusClicked()),0);
	mid[22] = Mobjects->insertItem(QPixmap((const char**)heightfield_xpm),i18n("Add Heightfield"),this,SLOT(addHeightfieldClicked()),0);
	mid[21] = Mobjects->insertItem(QPixmap((const char**)plane_xpm),i18n("Add Plane"),this,SLOT(addPlaneClicked()),0);
	mid[27] = Mobjects->insertItem(QPixmap((const char**)text_xpm),i18n("Add Text"),this,SLOT(addTextClicked()),0);
	mid[27] = Mobjects->insertItem(QPixmap((const char**)script_xpm),i18n("Add Script"),this,SLOT(addScriptClicked()),0);
	Mobjects->insertSeparator();
#ifdef DEBUG
	mid[19] = Mobjects->insertItem(QPixmap((const char**)mesh_xpm),i18n("Add Mesh"),this,SLOT(addMeshClicked()),0);
#endif
	mid[23] = Mobjects->insertItem(QPixmap((const char**)triangle_xpm),i18n("Add Triangle"),this,SLOT(addTriangleClicked()),0);
	Mobjects->insertSeparator();
	mid[20] = Mobjects->insertItem(QPixmap((const char**)refptr_xpm),i18n("Add Reference"),this,SLOT(addRefPtrClicked()),0);
	Mobjects->insertSeparator();
	mid[24] = Mobjects->insertItem(QPixmap((const char**)spotlight_xpm),i18n("Add Spotlight"),this,SLOT(addSpotLightClicked()),0);
#ifdef DEBUG
	mid[25] = Mobjects->insertItem(QPixmap((const char**)arealight_xpm),i18n("Add Arealight"),this,SLOT(addAreaLightClicked()),0);
#endif
	mid[26] = Mobjects->insertItem(QPixmap((const char**)lightpoint_xpm),i18n("Add Pointlight"),this,SLOT(addLightPointClicked()),0);
	Mobjects->insertSeparator();
#ifdef DEBUG
	mid[28] = Mobjects->insertItem(QPixmap((const char**)camera_xpm),i18n("Add Camera"),this,SLOT(addCameraClicked()),0);
#endif
	mid[29] = Mobjects->insertItem(QPixmap((const char**)ear_xpm),i18n("Add Ear"),this,SLOT(addEarClicked()),0);

	Mtools = new MPopupMenu;
	CHECK_PTR(Mtools);

	mid[35] = Mtools->insertItem(QPixmap((const char**)mesh_xpm),i18n("Map2Mesh"),this,SLOT(createMap2MeshClicked()),0);
	mid[38] = Mtools->insertItem(QPixmap((const char**)rtree_xpm),i18n("Recursiv Tree"),this,SLOT(createRecursivTreeClicked()),0);
	mid[39] = Mtools->insertItem(QPixmap((const char**)rtree_xpm),i18n("Forest"),this,SLOT(createForestClicked()),0);
	mid[36] = Mtools->insertItem(QPixmap((const char**)mesh_xpm),i18n("DF3-Tool"),this,SLOT(createDF3Clicked()),0);
	mid[37] = Mtools->insertItem(QPixmap((const char**)mesh_xpm),i18n("Read DF3"),this,SLOT(readDF3Clicked()),0);
	
	Mpov = new MPopupMenu;
	CHECK_PTR(Mpov);

	mid[30] = Mpov->insertItem(QPixmap((const char**)filefloppy_xpm),i18n("Export"),this,SLOT(exportPOVClicked()),0);
	Mpov->insertSeparator();
	mid[31] = Mpov->insertItem(QPixmap((const char**)filefloppy_xpm),i18n("Export Texturelibrary"),this,SLOT(exportTLibClicked()),0);

	Mwindows = new MPopupMenu;
	CHECK_PTR(Mwindows);

	mid[40] = Mwindows->insertItem(i18n("Animcontrol"),this,SLOT(toggleWACClicked()),0);
	mid[41] = Mwindows->insertItem(i18n("Texturecontrol"),this,SLOT(toggleTCCClicked()),0);
        Mwindows->insertSeparator();
	mid[60] = Mwindows->insertItem(i18n("Main-Toolbar"),this,SLOT(toggleMainToolBar()),0);
	mid[61] = Mwindows->insertItem(i18n("View-Toolbar"),this,SLOT(toggleViewToolBar()),0);
	mid[62] = Mwindows->insertItem(i18n("Texture-Toolbar"),this,SLOT(toggleTextureToolBar()),0);
        Mwindows->insertSeparator();
	mid[42] = Mwindows->insertItem(i18n("New Viewwindow"),this,SLOT(newViewClicked()),0);
        Mwindows->insertSeparator();
	mid[43] = Mwindows->insertItem(i18n("View Object"),this,SLOT(viewObjectClicked()),0);
	mid[44] = Mwindows->insertItem(i18n("View Object in Viewwindow"),this,SLOT(viewObjectWindowClicked()),0);

	

        Mwindows->setItemChecked(mid[40],0);
        Mwindows->setItemChecked(mid[41],1);
        Mwindows->setItemChecked(mid[60],1);
        Mwindows->setItemChecked(mid[61],1);
        Mwindows->setItemChecked(mid[62],1);

	Mhelp = new MPopupMenu;
	CHECK_PTR(Mhelp);

	extern KApplication	*k4deapp;

	Mhelp = k4deapp->getHelpMenu(true, "");
	Mhelp->insertItem(i18n("About " PROGNAME),this,SLOT(aboutClicked()),0,-1,2);
	Mhelp->removeItemAt(3);

	mainmenu = new KMenuBar(this,"mainmenu");
	CHECK_PTR(mainmenu);
	mainmenu->insertItem(i18n("&File"),Mfile);
//	mainmenu->insertItem(i18n("&Edit"),Medit);
	mainmenu->insertItem(i18n("&Objects"),Mobjects);
	mainmenu->insertItem(i18n("Tools"),Mtools);
	mainmenu->insertItem(i18n("&PovRay"),Mpov);
//	mainmenu->insertItem(i18n("&Options"),Moptions);
	mainmenu->insertItem(i18n("Windows"),Mwindows);
	mainmenu->insertSeparator();
	mainmenu->insertItem(i18n("&Help"),Mhelp);


	setMenu(mainmenu);
}

void	MainWin::aboutClicked()
{
	AboutDialog	*ad;
	
	ad = new AboutDialog(this,i18n("About " PROGNAME));
	
	ad->show();
	
	delete ad;
}

void	MainWin::newClicked()
{
	int	i;
	
	if(changed)
	{
		QString		str;
		int		result;

		str = PROGNAME " V" VERSION " \n\n";
		str += i18n("The data has changed and is currently not saved!");
		KMsgBox	mb;
		result = mb.yesNoCancel(0,i18n("New world?"),str,8,i18n("Abort"),i18n("Save now"),i18n("New"));
		if(result == 1) return;
		if(result == 2) saveClicked();
	}
	
	// save our main viewwidget
	worldtree->getWorld()->removeView(viewwidget);
	
	// close the other viewwidgets
	for(i = 0;i < worldtree->getWorld()->getNumViews();i++)
	{
		view	*v;
		
		v = worldtree->getWorld()->getView(i);
		if(v)
		{
			worldtree->getWorld()->removeView(v);
		}
	}
	
	// make a new empty world
	worldtree->empty();
	
	viewwidget->setName("Main View");
	
	changed = 0;
	
	if(projectname)
		free(projectname);
	projectname = 0;
	
	createWindowTitle();
	createStatusBar(i18n("New Projekt"));
	displayMessage(i18n("New Projekt"));

	worldtree->updateViews();
}

void	MainWin::quitClicked()
{
	if(changed)
	{
		QString		str;
		int		result;

		str = PROGNAME " V" VERSION " \n\n";
		str += i18n("The data has changed and is currently not saved!");
		
		KMsgBox	mb;
		
		result = mb.yesNoCancel(0,QString(i18n("Quit")) + " " PROGNAME " ?",str,8,i18n("Abort"),i18n("Save now"),i18n("Quit"));
		if(result == 1) return;
		if(result == 2) saveClicked();
	}
	
	{	// hack, lock at ~MainWin()
		Preferences		pref;
	
		pref.setMainWindowSize(width(),height());
	}
	
	deleteAll();
	
//	close();
}

void	MainWin::openClicked()
{
	char		name[1024];
	KFileDialog	*fd;
	QString		qstr;
 	Preferences	prefs;
	
	if(changed)
	{
		QString		str;
		int		result;

		str = PROGNAME " V" VERSION " \n\n";
		str += i18n("The data has changed and is currently not saved!");
		KMsgBox	mb;
		result = mb.yesNoCancel(0,i18n("Load new file?"),str,8,i18n("Abort"),i18n("Save now"),i18n("Load"));
		if(result == 1) return;
		if(result == 2) saveClicked();
	}

	fd = new KFileDialog(0);

	qstr = fd->getOpenFileName(lastread,"*.3dds",this,0);

	if(qstr == "") return;

	strcpy(name,qstr.data());
	delete fd;

	lastread = qstr;
	if(lastread.findRev('/'))
		lastread.remove(lastread.findRev('/') + 1,lastread.length() - lastread.findRev('/'));

	if(worldtree->load(name) == 0)
	{
		int	i;
		world	*w;
		view	*v;
		
		w = worldtree->getWorld();
		
		if(w)
		{
			printf("Nnumber of views: %i\n",w->getNumViews());
			for(i = 1;i < w->getNumViews();i++)
			{
				ViewWin		*nvw;
	
				v = w->getView(i);
				
				nvw = new ViewWin(this);
	
				nvw->show();
	
	                        nvw->setViewSize(v->getWidth(),v->getHeight());
			}
			emit changedWorld(worldtree->getWorld());
		}
		
		qstr = projectname = strdup(name);

		changed = 0;
	
		prefs.setRecentFile(qstr);
		createRecentMenu();

		qstr = QString(i18n("Loaded")) + " " + qstr;		
		createStatusBar(qstr);
	}
	createWindowTitle();
}

void	MainWin::openRecentClicked(int id)
{
	QString		qstr;
	char		name[1024];
	Preferences	prefs;

	qstr = prefs.recentFile(id - 1000);

	if(changed)
	{
		QString		str;
		int		result;

		str = PROGNAME " V" VERSION " \n\n";
		str += i18n("The data has changed and is currently not saved!");
		KMsgBox	mb;
		result = mb.yesNoCancel(0,i18n("Load new file?"),str,8,i18n("Abort"),i18n("Save now"),i18n("Load"));
		if(result == 1) return;
		if(result == 2) saveClicked();
	}

	if(qstr == "") return;

	strcpy(name,qstr.data());

	lastread = qstr;
	
	if(lastread.findRev('/'))
		lastread.remove(lastread.findRev('/') + 1,lastread.length() - lastread.findRev('/'));

	if(worldtree->load(name) == 0)
	{
		int	i;
		world	*w;
		view	*v;
		
		w = worldtree->getWorld();
		
		if(w)
		{
			for(i = 1;i < w->getNumViews();i++)
			{
				ViewWin		*nvw;
	
				v = w->getView(i);
				
				nvw = new ViewWin(this);
	
				emit changedWorld(viewwidget->getWorld());
	
				nvw->show();
				
	                        nvw->setViewSize(v->getWidth(),v->getHeight());
			}
		}
		
		qstr = projectname = strdup(name);

		changed = 0;
	
		prefs.setRecentFile(qstr);
		createRecentMenu();
		
		qstr = QString(i18n("Loaded")) + " " + qstr;		
		createStatusBar(qstr);
	}
	createWindowTitle();
}

void	MainWin::saveClicked()
{
	if(worldtree->save(projectname) == 0)
	{
		QString		qstr;
			
		changed = 0;
		
		qstr = QString(i18n("Saved")) + " ";
		qstr += projectname;		
		createStatusBar(qstr);
	}
	else saveAsClicked();
	createWindowTitle();
}

void	MainWin::saveAsClicked()
{
	char		name[1024];
	KFileDialog	*fd;
	QString		qstr;
	
	fd = new KFileDialog(lastwrite);

	qstr = fd->getSaveFileName(lastwrite,"*.3dds",this,0);

	if(qstr == "") return;

	strcpy(name,qstr.data());
	delete fd;

	lastwrite = qstr;

	if(worldtree->save(name) == 0)
	{
		QString		qstr;
			
		projectname = strdup(name);
		changed = 0;
		
		qstr = QString(i18n("Saved")) + " ";
		qstr += projectname;		
		createStatusBar(qstr);
	}

	createWindowTitle();
}

void	MainWin::dumpClicked()
{
	worldtree->dump();
}

void	MainWin::optionsClicked()
{
	PreferencesDialog	*pd;

	pd = new PreferencesDialog(this);

	connect(pd,SIGNAL(update()),viewwidget,SLOT(reconfigure()));

	pd->show();
//       pd->resize(450,400);

	delete pd;

	// why do the connect do nothing?
	viewwidget->reconfigure();
}

void	MainWin::createMap2MeshClicked()
{
	worldtree->map2Mesh();
}

void	MainWin::createRecursivTreeClicked()
{
	worldtree->recursivTree();
}

void	MainWin::createForestClicked()
{
	worldtree->generateForest();
}

void 	MainWin::createDF3Clicked()
{
	df3Wizard	*df3;

	df3 = new df3Wizard(this,"DF3 Wizard");
	if(df3)
	{
		df3->exec();
		
		delete df3;
	}
}

void	MainWin::readDF3Clicked()
{
	worldtree->readDF3();
}

void	MainWin::exportPOVClicked()
{
	char		name[1024];
	KFileDialog	*fd;
	QString		qstr;

	fd = new KFileDialog(lastpov);

	qstr = fd->getSaveFileName(lastpov,"*.pov",this,0);

	if(qstr == "") return;

	strcpy(name,qstr.data());
	delete fd;

	lastpov = qstr;

	if(worldtree->exportPOV(name,0,0) == 0)
	{
	}
}

void	MainWin::exportPOV(const char *name)
{
	if(worldtree->exportPOV(name,0,0) == 0)
	{
	}
}

void	MainWin::exportPOV(const char *name,int viewnum)
{
	if(worldtree->exportPOV(name,viewnum,0) == 0)
	{
	}
}

void	MainWin::exportPOVanim(const char *name)
{
	if(worldtree->exportPOV(name,0,1) == 0)
	{
	}
}

void	MainWin::addObjectClicked()
{
	worldtree->addObject();
}

void	MainWin::addCSGClicked()
{
	worldtree->addCSG();
}

void	MainWin::addBlobClicked()
{
	worldtree->addBlob();
}

void	MainWin::addRefClicked()
{
	worldtree->addRef();
}

void	MainWin::addSphereClicked()
{
	worldtree->addSphere();
}

void	MainWin::addCylinderClicked()
{
	worldtree->addCylinder();
}

void	MainWin::addBoxClicked()
{
	worldtree->addBox();
}

void	MainWin::addMeshClicked()
{
	worldtree->addMesh();
}

void	MainWin::addRefPtrClicked()
{
	worldtree->addRefPtr();
}

void	MainWin::addConeClicked()
{
	worldtree->addCone();
}

void	MainWin::addPlaneClicked()
{
	worldtree->addPlane();
}

void	MainWin::addTextClicked()
{
	worldtree->addText();
}

void	MainWin::addScriptClicked()
{
	worldtree->addScript();
}

void	MainWin::addTorusClicked()
{
	worldtree->addTorus();
}

void	MainWin::addTriangleClicked()
{
	worldtree->addTriangle();
}

void	MainWin::addHeightfieldClicked()
{
	worldtree->addHeightfield();
}

void	MainWin::addLightPointClicked()
{
	worldtree->addLightPoint();
}

void	MainWin::addSpotLightClicked()
{
	worldtree->addSpotLight();
}

void	MainWin::addAreaLightClicked()
{
	worldtree->addAreaLight();
}

void	MainWin::addCameraClicked()
{
	worldtree->addCamera();
}

void	MainWin::addEarClicked()
{
	worldtree->addEar();
}

void 	MainWin::exportTLibClicked()
{
	if (tlview) tlview->exportTLib();
}

void	MainWin::toggleWACClicked()
{
        if(!Mwindows->isItemChecked(mid[40]))
        {
        	animwidget->show();
        	tview->setMinimumSize(400,vpanner2->height() - animwidget->height() - 4);
        	Mwindows->setItemChecked(mid[40],!0);
		vpanner2->setAbsSeperatorPos(vpanner2->height() - animwidget->height());
        }
        else
        {
        	animwidget->hide();
        	tview->setMinimumSize(400,vpanner2->height() - 4);
        	Mwindows->setItemChecked(mid[40],0);
		vpanner2->setAbsSeperatorPos(vpanner2->height() - 10);
        }
}

void	MainWin::toggleTCCClicked()
{
        if(!Mwindows->isItemChecked(mid[41]))
        {
        	tview->show();
		tbtxtview->show();
        	//tview->setMinimumSize(400,vpanner2->height() - animwidget->height() - 4);
        	Mwindows->setItemChecked(mid[41],!0);
		//vpanner2->setAbsSeperatorPos(vpanner2->height() - animwidget->height());
        }
        else
        {
        	tview->hide();
		tbtxtview->hide();
        	//tview->setMinimumSize(400,vpanner2->height() - 4);
        	Mwindows->setItemChecked(mid[41],0);
		//vpanner2->setAbsSeperatorPos(vpanner2->height() - 10);
        }
}


void MainWin::toggleMainToolBar()
{
	if(!Mwindows->isItemChecked(mid[60]))
	{
		enableToolBar(KToolBar::Show,tb[1]);
        	Mwindows->setItemChecked(mid[60],!0);
	}
	else
	{
		enableToolBar(KToolBar::Hide,tb[1]);
        	Mwindows->setItemChecked(mid[60],0);
	}
}

void MainWin::toggleViewToolBar()
{
	if(!Mwindows->isItemChecked(mid[61]))
	{
		enableToolBar(KToolBar::Show,tb[2]);
        	Mwindows->setItemChecked(mid[61],!0);
	}
	else
	{
		enableToolBar(KToolBar::Hide,tb[2]);
        	Mwindows->setItemChecked(mid[61],0);
	}
}

void MainWin::toggleTextureToolBar()
{
	if(!Mwindows->isItemChecked(mid[62]))
	{
		enableToolBar(KToolBar::Show,tb[3]);
        	Mwindows->setItemChecked(mid[62],!0);
	}
	else
	{
		enableToolBar(KToolBar::Hide,tb[3]);
        	Mwindows->setItemChecked(mid[62],0);
	}
}

void	MainWin::viewObjectClicked()
{
	base		*b;
	
	b = worldtree->getSelectedObject();
	
	viewwidget->setViewObject(b);
}

void	MainWin::viewObjectWindowClicked()
{
	ViewWin		*nvw;
	base		*b;
	
	nvw = new ViewWin(this);
	
	emit changedWorld(viewwidget->getWorld());
	
	b = worldtree->getSelectedObject();
	
	nvw->setViewObject(b);
		
	nvw->show();
}

void	MainWin::newViewClicked()
{
	ViewWin		*nvw;
	
	nvw = new ViewWin(this);
	
	emit changedWorld(viewwidget->getWorld());
	
	nvw->show();
	
//	mainviewlist.append(nvw);
	
	connect(nvw,SIGNAL(deleteMe(ViewWin*)),this,SLOT(deleteViewWin(ViewWin*)));
	
	worldtree->updateViews();
}


void	MainWin::paintEvent(QPaintEvent *pe)
{
	KTMainWindow::paintEvent(pe);
//	KTopLevelWidget::paintEvent(pe);
}

void	MainWin::show()
{
	KTMainWindow::show();
//	KTopLevelWidget::show();
	
	toggleWACClicked();
	toggleWACClicked();
}

void	MainWin::savePicture()
{
	QString		fname,fmt;
	int		i;
	QStrList	strlist;
	QString		str;
	FILE		*fp;
	Preferences	pref;
	
	fmt = "";

	strlist = QImageIO::outputFormats();
		
	for(i = 0;i < (int)strlist.count();i++)
	{
		str = strlist.at(i);
		str = str.lower();
		printf("%i. %s\n",i,(const char*)str);
		if(str == "ppm")
			fmt = strlist.at(i);
		if(fmt == "" && str == "bmp")
			fmt = strlist.at(i);
	}
	
	
	i = 0;
	fp = 0;
	do
	{
		if(fp)
			fclose(fp);
		fname.sprintf("%s/viewimage%0.4i.%s",(const char*)pref.getPath(pref.getImagePath()),i,(const char*)fmt.lower());
		i++;
	}
	while((fp = fopen(fname,"r")));
	
	viewwidget->saveViewImage(fname,fmt);		
}

void	MainWin::autoSavePicture()
{
	Preferences	prefs;
	
	if(prefs.getAutoSavePicture())
		savePicture();
}

void	MainWin::setViewMode(int buttonn)
{
	int	t;

	if(buttonn < 7)
	{
		for(t = 1;t < 7;t++)
			if(t != buttonn)
				tbview->setButton(t,false);

		tbview->setButton(buttonn,true);

		emit viewMode(buttonn);
	}
	else if(buttonn < 11)
	{
		for(t = 7;t < 11;t++)
			if(t != buttonn)
				tbview->setButton(t,false);

		tbview->setButton(buttonn,true);

		emit projectionMode(buttonn - 6);
	}
	else if(buttonn < 14)
	{
		int		mode;

		mode = 0;
		if(tbview->isButtonOn(11)) mode |= 1;
		if(tbview->isButtonOn(12)) mode |= 2;
		if(tbview->isButtonOn(13)) mode |= 4;

		emit axeMode(mode);
	}
	else if(buttonn < 15)
	{
#ifdef DEBUG
		printf("Render clicked\n");
#endif
		emit render(9);
	}
	else if(buttonn < 16)
	{
#ifdef DEBUG
		printf("Render low clicked\n");
#endif
		emit render(1);
	}
}

void	MainWin::setChanged()
{
	changed = TRUE;
}

void	MainWin::createViewWidget()
{
	viewwidget = new ViewWidget(hpanner,"View",0);
	
	hpanner->activate(worldtree,viewwidget);
	
// connect
	connect(this,SIGNAL(changedWorld(world*)),viewwidget,SLOT(setWorld(world*)));

// forward	
	connect(viewwidget,SIGNAL(redraw()),this,SIGNAL(redraw()));
	
// connect
	connect(this,SIGNAL(redraw()),viewwidget,SLOT(draw()));
	
// forward
	connect(viewwidget,SIGNAL(changedModel()),this,SIGNAL(changedModel()));
	
// connect
	connect(this,SIGNAL(changedSelected(base*)),viewwidget,SLOT(setSelected(base*)));
	
	connect(viewwidget,SIGNAL(exportPOV(const char*,int)),this,SLOT(exportPOV(const char*,int)));
	connect(viewwidget,SIGNAL(resizedView(int,int)),this,SLOT(resizedView(int,int)));
	connect(viewwidget,SIGNAL(report(QString)),this,SLOT(displayMessage(QString)));
	connect(viewwidget,SIGNAL(finishedRendering()),this,SLOT(autoSavePicture()));
	
	connect(this,SIGNAL(viewMode(int)),viewwidget,SLOT(setViewMode(int)));
	connect(this,SIGNAL(projectionMode(int)),viewwidget,SLOT(setProjectionMode(int)));
	connect(this,SIGNAL(axeMode(int)),viewwidget,SLOT(setAxeMode(int)));
	connect(this,SIGNAL(render(int)),viewwidget,SLOT(startRendering(int)));
}

void	MainWin::deleteViewWin(ViewWin *vw)
{
	printf("remove ViewWin\n");
	worldtree->updateViews();
	
/*
	printf("rest #%i\n",mainviewlist.count());
	mainviewlist.setAutoDelete(false);
	if(mainviewlist.remove(vw))
		printf("successfully removed: rest #%i\n",mainviewlist.count());
	mainviewlist.setAutoDelete(true);
*/
}

/*
void	MainWin::emptyMainViewList()
{
	while(mainviewlist.count())
	{
		mainviewlist.remove(0U);
	}
}

*/



