/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** dialogs.cpp
*/

#include "dialogs.h"
#include "dialogs.moc"

#include <kapp.h>
#include <qvalidator.h>

NameDialog::NameDialog(QWidget *parent=0,const char *name=0) :
	QDialog(parent,name,TRUE)
{
	setCaption(name);

	line = new QLineEdit(this);

	ok = new QPushButton( this,i18n("OK"));
	ok->setText(i18n("OK"));
	connect( ok, SIGNAL(clicked()), SLOT(accept()) );

	cancel = new QPushButton( this,i18n("Cancel") );
	cancel->setText(i18n("Cancel"));
	connect( cancel, SIGNAL(clicked()), SLOT(reject()) );

	setMaximumSize(150,65);
	QDialog::resize(150,65);
	setMinimumSize(150,65);

	line->setGeometry(5,5,140,25);
	cancel->setGeometry(5,35,67,25);
	ok->setGeometry(77,35,67,25);	
}

NameDialog::~NameDialog()
{
	delete	ok;
	delete	cancel;
	delete	line;
}

QString	NameDialog::getName()
{
	return line->text();
}

void	NameDialog::setName(QString s)
{
	line->setText(s);
}

// ************************************************************************

EarDialog::EarDialog(QWidget *parent=0,const char *name=0) :
	QDialog(parent,name,TRUE)
{
	QIntValidator	*iv;
	
	setCaption(name);

	line = new QLineEdit(this);
	line->setValidator((iv = new QIntValidator(line)));
 	iv->setRange(0,255);

	ok = new QPushButton( this,i18n("OK"));
	ok->setText(i18n("OK"));
	connect( ok, SIGNAL(clicked()), SLOT(accept()) );

	cancel = new QPushButton( this,i18n("Cancel") );
	cancel->setText(i18n("Cancel"));
	connect( cancel, SIGNAL(clicked()), SLOT(reject()) );

	setMaximumSize(150,65);
	QDialog::resize(150,65);
	setMinimumSize(150,65);

	line->setGeometry(5,5,140,25);
	cancel->setGeometry(5,35,67,25);
	ok->setGeometry(77,35,67,25);	
}

EarDialog::~EarDialog()
{
}

unsigned char	EarDialog::getChannel()
{
	unsigned char	c;
	
	c = (unsigned char)atoi(line->text());
	
	return c;
}

void	EarDialog::setChannel(unsigned char c)
{
	QString		qstr;
	
	qstr.setNum((unsigned short)c);
	line->setText(qstr);
}

// ************************************************************************


InfoDialog::InfoDialog(const char *name=0) :
	QDialog(NULL,"Info",true)
{
	QSize qs;

	setCaption(i18n("Info"));

	label = new QLabel(name,this);

	ok = new QPushButton(i18n("OK"), this);

	connect( ok, SIGNAL(clicked()), SLOT(accept()) );

	setMaximumSize(150,65);
	QDialog::resize(150,65);
	setMinimumSize(150,65);

	qs=label->sizeHint();	

	label->setGeometry(qs.width()/4,qs.height()/4,qs.width(),qs.height() );
	ok->setGeometry(width()/2-20,height()-30,40,25);	
}

InfoDialog::~InfoDialog()
{
	delete	ok;
	delete label;
}

// ******************************************************************************************

CameraDialog::CameraDialog(QWidget *parent=0,const char *name=0) :
	QDialog(parent,name,true)
{
	setCaption(name);

	viewselect = new QComboBox(this);
	viewselect->setInsertionPolicy(QComboBox::NoInsertion);

	ok = new QPushButton( this,i18n("OK"));
	ok->setText(i18n("OK"));
	connect( ok, SIGNAL(clicked()), SLOT(accept()) );

	cancel = new QPushButton( this,i18n("Cancel") );
	cancel->setText(i18n("Cancel"));
	connect( cancel, SIGNAL(clicked()), SLOT(reject()) );

	setMaximumSize(150,65);
	QDialog::resize(150,65);
	setMinimumSize(150,65);

	viewselect->setGeometry(5,5,140,25);
	cancel->setGeometry(5,35,67,25);
	ok->setGeometry(77,35,67,25);	
}

CameraDialog::~CameraDialog()
{
}

QString	CameraDialog::getView()
{
	QString		str;
	
	str = viewselect->currentText();
	
	return str;
}

void	CameraDialog::setViewList(QStrList s)
{
	viewselect->insertStrList(&s);
}

// ******************************************************************************************

BlobDialog::BlobDialog(QWidget *parent=0,const char *name=0) :
	QDialog(parent,name,TRUE)
{
	int	w,w2;

	setCaption(name);

	label = new QLabel(this);
	label->setText(i18n("Thresholdvalue:"));

	line = new QLineEdit(this);

	ok = new QPushButton( this,i18n("OK"));
	ok->setText(i18n("OK"));
	connect( ok, SIGNAL(clicked()), SLOT(accept()) );

	cancel = new QPushButton( this,i18n("Cancel") );
	cancel->setText(i18n("Cancel"));
	connect( cancel, SIGNAL(clicked()), SLOT(reject()) );

	w2 = label->sizeHint().width();
	w = w2 * 3 / 2 + 15;

	setMaximumSize(w,65);
	QDialog::resize(w,65);
	setMinimumSize(w,65);

	label->setGeometry(5,5,w2,25);
	line->setGeometry(10 + w2,5,w2 / 2,25);

	w2 = w2 * 2 / 3;
	cancel->setGeometry((w - 5) / 2 - w2,35,w2,25);
	ok->setGeometry((w - 5) / 2 + 5,35,w2,25);	
}

BlobDialog::~BlobDialog()
{
}

double	BlobDialog::thresholdValue()
{
	QString	str;

	str = line->text();

	return str.toDouble();
}

void	BlobDialog::setThresholdValue(double t)
{
	QString	str;

	str.setNum(t);
	line->setText(str);
}


// ******************************************************************************************


TextDialog::TextDialog(QWidget *parent,const char *name) :
	QDialog(parent,name,TRUE)
{
	QString		path;
	Preferences	prefs;

	path = prefs.getPath(prefs.getFontPath());

	setCaption(name);

	lcontext = new QLabel(this);
	lcontext->setText(i18n("Context:"));

	wcontext = new QLineEdit(this);

/*
	lfontname = new QLabel(this);
	lfontname->setText(i18n("Fontname:"));

	wfontname = new QLineEdit(this);
*/

	wfontname = new GetFileWidget(this,i18n("Fontname:"),path);

	lthickness = new QLabel(this);
	lthickness->setText(i18n("Thickness:"));

	wthickness = new QLineEdit(this);

	loffset = new QLabel(this);
	loffset->setText(i18n("Offset:"));

	wxoffset = new QLineEdit(this);
	wyoffset = new QLineEdit(this);

	
	ok = new QPushButton( this,i18n("OK"));
	ok->setText(i18n("OK"));
	connect( ok, SIGNAL(clicked()), SLOT(accept()) );

	cancel = new QPushButton( this,i18n("Cancel") );
	cancel->setText(i18n("Cancel"));
	connect( cancel, SIGNAL(clicked()), SLOT(reject()) );

	setMaximumSize(400,170);
	QDialog::resize(400,170);
	setMinimumSize(400,170);

	lcontext->setGeometry(5,5,140,25);
	wcontext->setGeometry(145,5,150,25);

/*
	lfontname->setGeometry(5,30,140,25);
	wfontname->setGeometry(145,30,150,25);
*/
	wfontname->setGeometry(5,30,390,25);

	lthickness->setGeometry(5,60,140,25);
	wthickness->setGeometry(145,60,150,25);

	loffset->setGeometry(5,90,140,25);
	wxoffset->setGeometry(145,90,74,25);
	wyoffset->setGeometry(221,90,74,25);

	cancel->setGeometry(67,130,80,25);
	ok->setGeometry(153,130,80,25);	
}

TextDialog::~TextDialog()
{
}

void	TextDialog::save(QString &context,QString &fontname,double &thickness,Vector2 &offset)
{
	float		d;
	const char	*str;

	context = wcontext->text();

//	fontname = wfontname->text();
	fontname = wfontname->getFilename();

	str = wthickness->text();
	sscanf(str,"%g",&d);
	thickness = d;

	str = wxoffset->text();
	sscanf(str,"%g",&d);
	offset[0] = d;
	str = wyoffset->text();
	sscanf(str,"%g",&d);
	offset[1] = d;
}

void	TextDialog::load(QString context,QString fontname,double thickness,Vector2 offset)
{
	char		str[128];

	wcontext->setText(context);

	wfontname->setFilename(fontname);

	sprintf(str,"%g",thickness);
	wthickness->setText(str);

	sprintf(str,"%g",offset[0]);
	wxoffset->setText(str);
	sprintf(str,"%g",offset[1]);
	wyoffset->setText(str);
}

// ******************************************************************************************


HeightFieldDialog::HeightFieldDialog(QWidget *parent,const char *name) :
	QDialog(parent,name,TRUE)
{
	Preferences	prefs;
        QString		str;

	setCaption(name);

	lwaterlevel = new QLabel(this);
	lwaterlevel->setText(i18n("Waterlevel:"));

	wwaterlevel = new QLineEdit(this);

	lfilename = new QLabel(this);
	lfilename->setText(i18n("Filename:"));
	
	str = prefs.getPath(prefs.getHeightfieldPath());
#ifdef DEBUG 	
 	printf("heightfield: %s\n",(const char*)str);
#endif
 	wfilename = new GetFileWidget(this,(const char *)"",str);

	csmooth = new QCheckBox(this);
	csmooth->setText(i18n("Smooth"));
	
	ok = new QPushButton( this,i18n("OK"));
	ok->setText(i18n("OK"));
	connect( ok, SIGNAL(clicked()), SLOT(accept()) );

	cancel = new QPushButton( this,i18n("Cancel") );
	cancel->setText(i18n("Cancel"));
	connect( cancel, SIGNAL(clicked()), SLOT(reject()) );

	setMaximumSize(400,170);
	QDialog::resize(400,170);
	setMinimumSize(400,170);

	lfilename->setGeometry(5,5,140,25);
 	wfilename->setGeometry(140,5,250,25);

	lwaterlevel->setGeometry(5,35,140,25);
	wwaterlevel->setGeometry(145,35,150,25);

	csmooth->setGeometry(5,75,190,25);
	
	cancel->setGeometry(102,130,80,25);
	ok->setGeometry(188,130,80,25);	
}

HeightFieldDialog::~HeightFieldDialog()
{
}

void	HeightFieldDialog::save(double &waterlevel,QString &filename,bool &smooth)
{
	QString		str;

	str = wwaterlevel->text();
	waterlevel = str.toDouble();

	filename = wfilename->getFilename();
	
	smooth = csmooth->isChecked();
}

void	HeightFieldDialog::load(double waterlevel,QString filename,bool smooth)
{
	QString		str;

	str.setNum(waterlevel);
	wwaterlevel->setText(str);

	wfilename->setFilename(filename);
	
	csmooth->setChecked(smooth);
}




// ******************************************************************************************




LightPointDialog::LightPointDialog(QWidget *parent,const char *name) :
	QDialog(parent,name,TRUE)
{
	setCaption(name);

	lfadepower = new QLabel(this);
	lfadepower->setText(i18n("Fadepower:"));

	wfadepower = new QLineEdit(this);

	lfadedistance = new QLabel(this);
	lfadedistance->setText(i18n("Fadedistance:"));

	wfadedistance = new QLineEdit(this);

	lcolor = new QLabel(this);
	lcolor->setText(i18n("Color:"));

	wcolor = new KColorButton(this);

	ok = new QPushButton( this,i18n("OK"));
	ok->setText(i18n("OK"));
	connect( ok, SIGNAL(clicked()), SLOT(accept()) );

	cancel = new QPushButton( this,i18n("Cancel") );
	cancel->setText(i18n("Cancel"));
	connect( cancel, SIGNAL(clicked()), SLOT(reject()) );

	setMaximumSize(300,170);
	QDialog::resize(300,170);
	setMinimumSize(300,170);

	lfadedistance->setGeometry(5,5,140,25);
	wfadedistance->setGeometry(145,5,150,25);

	lfadepower->setGeometry(5,35,140,25);
	wfadepower->setGeometry(145,35,150,25);

	lcolor->setGeometry(5,70,140,25);
	wcolor->setGeometry(145,70,150,25);

	cancel->setGeometry(67,130,80,25);
	ok->setGeometry(153,130,80,25);	
}

LightPointDialog::~LightPointDialog()
{
}

void	LightPointDialog::save(double *fadepower,double *fadedistance,Vector3 &color)
{
	QString		str;
	QColor		qc;

	str = wfadepower->text();
	*fadepower = str.toDouble();

	str = wfadedistance->text();
	*fadedistance = str.toDouble();

	qc = wcolor->color();
	color[0] = (double)qc.red() / 255.0;
	color[1] = (double)qc.green() / 255.0;
	color[2] = (double)qc.blue() / 255.0;
}

void	LightPointDialog::load(double fadepower,double fadedistance,Vector3 color)
{
	QString		str;
	QColor		qc;

	str.setNum(fadepower);
	wfadepower->setText(str);

	str.setNum(fadedistance);
	wfadedistance->setText(str);

	qc.setRgb(color[0] * 255,color[1] * 255,color[2] * 255);
	wcolor->setColor(qc);
}






// ******************************************************************************************




SpotLightDialog::SpotLightDialog(QWidget *parent,const char *name) :
	QDialog(parent,name,TRUE)
{
	setCaption(name);

	lfadepower = new QLabel(this);
	lfadepower->setText(i18n("Fadepower:"));

	wfadepower = new QLineEdit(this);

	lfadedistance = new QLabel(this);
	lfadedistance->setText(i18n("Fadedistance:"));

	wfadedistance = new QLineEdit(this);

	lcolor = new QLabel(this);
	lcolor->setText(i18n("Color:"));

	wcolor = new KColorButton(this);

	lminteraction = new QLabel(this);
        lminteraction->setText(i18n("Mediainteraction:"));

	cminteraction = new QCheckBox(this);

	lmattenuation = new QLabel(this);
        lmattenuation->setText(i18n("Mediaattenuation:"));

	cmattenuation = new QCheckBox(this);

	lradius = new QLabel(this);
	lradius->setText(i18n("Radius:"));

	wradius = new QLineEdit(this);

	lfalloff = new QLabel(this);
	lfalloff->setText(i18n("Falloff:"));

	wfalloff = new QLineEdit(this);

	ltightness = new QLabel(this);
	ltightness->setText(i18n("Tightness:"));

	wtightness = new QLineEdit(this);

	ok = new QPushButton( this,i18n("OK"));
	ok->setText(i18n("OK"));
	connect( ok, SIGNAL(clicked()), SLOT(accept()) );

	cancel = new QPushButton( this,i18n("Cancel") );
	cancel->setText(i18n("Cancel"));
	connect( cancel, SIGNAL(clicked()), SLOT(reject()) );

	setMaximumSize(300,290);
	QDialog::resize(300,290);
	setMinimumSize(300,290);

	lfadedistance->setGeometry(5,5,140,25);
	wfadedistance->setGeometry(145,5,150,25);

	lfadepower->setGeometry(5,35,140,25);
	wfadepower->setGeometry(145,35,150,25);

	lcolor->setGeometry(5,70,140,25);
	wcolor->setGeometry(145,70,150,25);

	lminteraction->setGeometry(5,105,140,25);
	cminteraction->setGeometry(145,105,140,25);
	lmattenuation->setGeometry(5,135,140,25);
	cmattenuation->setGeometry(145,135,140,25);

	lradius->setGeometry(5,165,140,25);
	wradius->setGeometry(145,165,150,25);

	lfalloff->setGeometry(5,195,140,25);
	wfalloff->setGeometry(145,195,150,25);

	ltightness->setGeometry(5,225,140,25);
	wtightness->setGeometry(145,225,150,25);

	cancel->setGeometry(67,260,80,25);
	ok->setGeometry(153,260,80,25);	
}

SpotLightDialog::~SpotLightDialog()
{
}

void	SpotLightDialog::save(double &radius,double &falloff,double &tightness,double &fadepower,double &fadedistance,Vector3 &color,bool &mi,bool &ma)
{
	QString		str;
	QColor		qc;

	str = wradius->text();
	radius = str.toDouble();

	str = wfalloff->text();
	falloff = str.toDouble();

	str = wfalloff->text();
	falloff = str.toDouble();

	str = wtightness->text();
	tightness = str.toDouble();

	str = wfadedistance->text();
	fadedistance = str.toDouble();

	qc = wcolor->color();
	color[0] = ((double)qc.red()) / 255.0;
	color[1] = ((double)qc.green()) / 255.0;
	color[2] = ((double)qc.blue()) / 255.0;
	
	mi = cminteraction->isChecked();
	ma = cmattenuation->isChecked();
}

void	SpotLightDialog::load(double radius,double falloff,double tightness,double fadepower,double fadedistance,Vector3 color,bool mi,bool ma)
{
	QString		str;
	QColor		qc;

	str.setNum(radius);
	wradius->setText(str);

	str.setNum(falloff);
	wfalloff->setText(str);

	str.setNum(tightness);
	wtightness->setText(str);

	str.setNum(fadepower);
	wfadepower->setText(str);

	str.setNum(fadedistance);
	wfadedistance->setText(str);

	qc.setRgb(color[0] * 255,color[1] * 255,color[2] * 255);
	wcolor->setColor(qc);
	
	cminteraction->setChecked(mi);
	cmattenuation->setChecked(ma);
}




// ******************************************************************************************




AnimDialog::AnimDialog(QWidget *parent,const char *name,anim *a) :
	QDialog(parent,name,TRUE)
{
	Preferences	prefs;
//	int		n;
	const char	**nl = (const char**)a->getScriptList();

	_anim = a;

	setCaption(name);

	fname = new QComboBox(TRUE,this);
	fname->setAutoCompletion(TRUE);
	fname->insertStrList(nl);
	fname->setEditText("");			// provides nonconfusion
	connect(fname,SIGNAL(activated(const char*)),this,SLOT(selectScript(const char*)));
	
	ok = new QPushButton(this,i18n("OK"));
	ok->setText(i18n("OK"));
	connect(ok,SIGNAL(clicked()),SLOT(saveText()));
	connect(ok,SIGNAL(clicked()),SLOT(accept()));

	add = new QPushButton( this,i18n("Add") );
	add->setText(i18n("Add"));
	connect(add,SIGNAL(clicked()),SLOT(addScript()));

	del = new QPushButton( this,i18n("Delete") );
//	del->setEnabled(FALSE);
	del->setText(i18n("Delete"));
	connect(del,SIGNAL(clicked()),SLOT(delScript()));

	mle = new QMultiLineEdit(this);

	setMaximumSize(1024,960);
	QDialog::resize(prefs.getAnimEditorWidth(),prefs.getAnimEditorHeight());
	setMinimumSize(300,250);

	
/*
	for(n = 0;nl[n] != 0;n++)
		free(nl[n]);
	free(nl);
*/
	layout();
}

AnimDialog::~AnimDialog()
{
}

void	AnimDialog::paintEvent(QPaintEvent *)
{
	Preferences	prefs;
	
	layout();
	
	prefs.setAnimEditorSize(width(),height());
	prefs.save();
}

void	AnimDialog::layout()
{
	int	x,y,w,h,w2;
	
	x = y = 5;
	w2 = MAX(add->sizeHint().width(),MAX(del->sizeHint().width(),ok->sizeHint().width()));
	w = width() - w2 * 3 - 25;
	h = 25;
	fname->setGeometry(x,y,w,h);
	x += w + 5;
	add->setGeometry(x,y,w2,h);
	x += w2 + 5;
	del->setGeometry(x,y,w2,h);
	x += w2 + 5;
	ok->setGeometry(x,y,w2,h);
		
	x = 5;
	y += h + 5;
	w = width() - 10;
	h = height() - 15 - h;
	mle->setGeometry(x,y,w,h);
}

void	AnimDialog::addScript()
{
	QString		str;
	int		n;
	char		**nl = _anim->getScriptList();

	saveText();

	str = fname->currentText();

	if(str.isNull() || str.isEmpty())
		return;

	for(n = 0;nl[n] != 0;n++)
	{
		if(str == nl[n])
		{
			for(n = 0;nl[n] != 0;n++)
				free(nl[n]);
			free(nl);

			return;
		}
	}

	_anim->addScript(str.data(),"#!/bin/sh\n");

	fname->insertItem(str);

	for(n = 0;n < fname->count();n++)
		if(fname->text(n) == str)
			fname->setCurrentItem(n);

	for(n = 0;nl[n] != 0;n++)
		free(nl[n]);
	free(nl);
}

void	AnimDialog::delScript()
{
	QString		str;
	int		n,i;
	char		**nl = _anim->getScriptList();

	saveText();

	str = fname->currentText();

	if(str.isNull() || str.isEmpty())
		return;

	for(n = 0;nl[n] != 0;n++)
	{
		if(str == nl[n])
		{
			for(n = 0;nl[n] != 0;n++)
				free(nl[n]);
			free(nl);

			_anim->delScript(str.data());
			
			for(i = 0;i < fname->count();i++)
				if(fname->text(i) == str)
					fname->removeItem(i);
			return;
		}
	}

	for(n = 0;nl[n] != 0;n++)
		free(nl[n]);
	free(nl);
}

void	AnimDialog::selectScript(const char*)
{
	QString		str;
	int		n;
	char		**nl = _anim->getScriptList();

	saveText();

	str = fname->currentText();
	if(str.isNull() || str.isEmpty())
		return;

	for(n = 0;nl[n] != 0;n++)
		if(str == nl[n])
			break;
	if(nl[n] == 0)
	{
		for(n = 0;nl[n] != 0;n++)
			free(nl[n]);
		free(nl);
		return;
	}
	mle->setText(_anim->getText(str.data()));

	for(n = 0;nl[n] != 0;n++)
		free(nl[n]);
	free(nl);
}

void	AnimDialog::saveText()
{
	QString		str;
	int		n;
	char		**nl = _anim->getScriptList();

	str = fname->currentText();

	if(oldstr.isNull() || oldstr.isEmpty())
	{
		oldstr = str;
		return;
	}

	for(n = 0;nl[n] != 0;n++)
		if(oldstr == nl[n])
			break;
	if(nl[n] == 0)
	{
		for(n = 0;nl[n] != 0;n++)
			free(nl[n]);
		free(nl);
		return;
	}

	_anim->setText(oldstr.data(),mle->text().data());

	oldstr = str;

	for(n = 0;nl[n] != 0;n++)
		free(nl[n]);
	free(nl);
}







// ******************************************************************************************




ScriptDialog::ScriptDialog(QWidget *parent,const char *name,script *s) :
	QDialog(parent,name,TRUE)
{
	Preferences	prefs;

	scriptobj = s;

	setCaption(name);

	ok = new QPushButton(this,i18n("OK"));
	ok->setText(i18n("OK"));
	connect(ok,SIGNAL(clicked()),SLOT(saveText()));
	connect(ok,SIGNAL(clicked()),SLOT(accept()));

	cancel = new QPushButton( this,i18n("Cancel") );
	cancel->setText(i18n("Cancel"));
	connect(cancel,SIGNAL(clicked()),SLOT(reject()));

	mle = new QMultiLineEdit(this);
	if(scriptobj)
		mle->setText(scriptobj->getText());

	setMaximumSize(1024,960);
//	QDialog::resize(prefs.getAnimEditorWidth(),prefs.getAnimEditorHeight());
	setMinimumSize(300,250);

	layout();
}

ScriptDialog::~ScriptDialog()
{
}

void	ScriptDialog::paintEvent(QPaintEvent *)
{
	Preferences	prefs;
	
	layout();
	
//	prefs.setAnimEditorSize(width(),height());
//	prefs.save();
}

void	ScriptDialog::layout()
{
	int	x,y,w,h;
		
	x = 5;
	y = 5;
	w = width() - 10;
	h = height() - 15 - 25;
	mle->setGeometry(x,y,w,h);
	
	w = MAX(cancel->sizeHint().width(),ok->sizeHint().width());
	h = 25;
	x = width() - 10 - w * 2;
	y = height() - 5 - h;
	ok->setGeometry(x,y,w,h);
	x += w + 5;
	cancel->setGeometry(x,y,w,h);
}

void	ScriptDialog::saveText()
{
	if(scriptobj)
		scriptobj->setText(mle->text());
}



// ******************************************************************************************


WorldDialog::WorldDialog(QWidget *parent,const char *name) :
	QDialog(parent,name,TRUE)
{
	setCaption(name);

	lex = new QLineEdit(this,"");
	lex->setText("0");
	ley = new QLineEdit(this,"");
	ley->setText("0");
	lez = new QLineEdit(this,"");
	lez->setText("0");

	ambient_light = Vector3(0,0,0);

	allabel = new QLabel(this,"");
	allabel->setText(i18n("Ambient light:"));

	bgc_label = new QLabel(this,"");
	bgc_label->setText(i18n("Backgroundcolor:"));

	backg_color = new KColorButton(this);

	ok = new QPushButton( this,i18n("OK"));
	ok->setText(i18n("OK"));
	connect( ok, SIGNAL(clicked()), SLOT(accept()) );

	cancel = new QPushButton( this,i18n("Cancel") );
	cancel->setText(i18n("Cancel"));
	connect( cancel, SIGNAL(clicked()), SLOT(reject()) );

	setMaximumSize(300,140);
	QDialog::resize(300,140);
	setMinimumSize(300,140);

	allabel->setGeometry(5,5,155,25);
	lex->setGeometry(165,5,40,25);
	ley->setGeometry(210,5,40,25);
	lez->setGeometry(255,5,40,25);

	bgc_label->setGeometry(5,35,155,25);
	backg_color->setGeometry(165,35,40,25);

	cancel->setGeometry(67,110,80,25);
	ok->setGeometry(153,110,80,25);	
}

WorldDialog::~WorldDialog()
{
}

void	WorldDialog::save(Vector3 &a,Vector3 &b)
{
	float		d;
	const char	*str;
	QColor		qc;

	str = lex->text();
	sscanf(str,"%g",&d);
	ambient_light[0] = d;

	str = ley->text();
	sscanf(str,"%g",&d);
	ambient_light[1] = d;

	str = lez->text();
	sscanf(str,"%g",&d);
	ambient_light[2] = d;

	a = ambient_light;

	qc = backg_color->color();
	b[0] = (double)qc.red() / 255.0;
	b[1] = (double)qc.green() / 255.0;
	b[2] = (double)qc.blue() / 255.0;
}

void	WorldDialog::load(Vector3 a,Vector3 b)
{
	char		str[128];
	QColor		qc;

	ambient_light = a;

	sprintf(str,"%g",ambient_light[0]);
	lex->setText(str);

	sprintf(str,"%g",ambient_light[1]);
	ley->setText(str);

	sprintf(str,"%g",ambient_light[2]);
	lez->setText(str);

	qc.setRgb(b[0] * 255,b[1] * 255,b[2] * 255);
	backg_color->setColor(qc);
}





