/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** turbulence.cpp
*/
#include "turbulence.h"
#include <chunk.h>

/* XPM */
const char*turbulencePic[]={
"20 20 2 1",
"# c #000000",
". c None",
"....................",
"........#####.......",
"......##.....##.....",
"....##.........##...",
"...#.............#..",
"...#......#####..#..",
"..#......#.....#..#.",
"..#.....#.......#.#.",
".#.....#.........#.#",
".#....#......###..##",
".#....#.....#...#.#.",
".#....#....#..#..##.",
".#....#....#.#..###.",
"..#...#....#..##.#..",
"..#....#....#...##..",
"...#....#....#####..",
"...#.....#.......#..",
"....#.....#######...",
".....##.............",
"...................."};


Turbulence::Turbulence(textureBase *p):textureBase(TURBULENCEMODIFIER,p)
{

	turbulence=Vector3(0,0,0);
	octaves=6;
	omega=0.5;
	lambda=2.0;
	setName("Turbulencemodifier");
	setInfo("Turbulencemodifier");
	setSubItems(false);
}

Turbulence::~Turbulence()
{
}

const char **Turbulence::getXPM()
{
	return turbulencePic;
}

void Turbulence::updateInfo()
{
char *buffer=(char *)malloc(512);
	sprintf(buffer,"Turb:(%g/%g/%g)",turbulence[0],turbulence[1],turbulence[2]);
	setInfo(buffer);
	free(buffer);
}

void Turbulence::setTurbulence(Vector3 v)
{
	turbulence=v;
	updateInfo();
}


Vector3 Turbulence::getTurbulence()
{
	return turbulence;
}

void Turbulence::setOctaves(double v)
{
	octaves=v;
	updateInfo();
}


double Turbulence::getOctaves()
{
	return octaves;
}

void Turbulence::setOmega(double v)
{
	omega=v;
	updateInfo();
}


double Turbulence::getOmega()
{
	return omega;
}

void Turbulence::setLambda(double v)
{
	lambda=v;
	updateInfo();
}

bool Turbulence::canAdd(textureBase *t)
{
	if (t->getType()==TURBULENCEMODIFIER) return true;
	return false;
}

bool Turbulence::isEqual(textureBase *tb)
{
Turbulence *t=(Turbulence *)tb;
	if (tb->getType()!=TURBULENCEMODIFIER) return false;

	if (
		(turbulence[0]==t->getTurbulence()[0]) &&
		(turbulence[1]==t->getTurbulence()[1]) &&
		(turbulence[2]==t->getTurbulence()[2]) &&
		(textureBase::isEqual(tb)==true) &&
		(octaves==t->getOctaves() ) &&
		(omega==t->getOmega() ) &&
		(lambda==t->getLambda() )
	) return true;
	return false;
}

void Turbulence::add(textureBase *tb)
{
Turbulence *t=(Turbulence *)tb;
	if (tb->getType()!=TURBULENCEMODIFIER) return;

	turbulence[0]=t->getTurbulence()[0];
	turbulence[1]=t->getTurbulence()[1];
	turbulence[2]=t->getTurbulence()[2];
	octaves=t->getOctaves();
	omega=t->getOmega();
	lambda=t->getLambda();
	updateInfo();
}


double Turbulence::getLambda()
{
	return lambda;
}
int	Turbulence::save(media *m)
{
chunk ch;
	if(!m)	return -1;
	#ifdef DEBUG
		printf("saving Turbulence\n");
	#endif

	setMedia(m);

	writeChunk("TURB");

	textureBase::save(m);
	
	ch.setMedia(m);
	ch.writeChunk("DATA");

		writeVector(turbulence);
		m->write(&octaves,sizeof(double) );
		m->write(&omega,sizeof(double) );
		m->write(&lambda,sizeof(double) );
	ch.writeChunkLen();

	writeChunkLen();

	#ifdef DEBUG
		printf("Turbulence saved\n");
	#endif
	return 0;
}

int	Turbulence::load(media *m,int l)
{
	char	chunk[4];
	bool 	read;
	int	len,pos = m->tell();

	#ifdef DEBUG
		printf("Loading Turbulence\n");
	#endif

	if(!m) return -1;
	setMedia(m);	
	

	do {
		m->read(chunk,4);
		m->read(&len,4);
		read=false;

		if(strncmp(chunk,"BASE",4) == 0) {
			textureBase::load(m,len);
			read=true;
		}
		if(strncmp(chunk,"DATA",4) == 0) {
			turbulence = readVector();
			m->read(&octaves,sizeof(double) );
			m->read(&omega,sizeof(double) );
			m->read(&lambda,sizeof(double) );			
			read=true;
		}
	
		if (read==false) { m->seek(len,SEEK_CUR); }

	} while (m->tell()< (pos+l) );
	m->seek(pos+l,SEEK_SET); // Man weiss ja nie...
	#ifdef DEBUG
		printf("Turbulence loaded\n");
	#endif
	updateInfo();
	return 0;
}

int Turbulence::exportPOV(FILE *fp,int tab,int tabsize, int anim,bool dummy)
{
	if(!fp) return -1;
	#ifdef DEBUG
		printf("exporting Turbulence (POV)\n");
	#endif

	if ((turbulence[0]!=0)&&(turbulence[1]!=0)||(turbulence[2]!=0)) {
		printTab(fp,tab);
		fprintf(fp,"turbulence     <%g,%g,%g>\n",turbulence[0],turbulence[1],turbulence[2]);
	}
	if(omega != 0.5)
	{
		printTab(fp,tab);
		fprintf(fp,"omega  %g\n",omega);
	}
	if(octaves != 6)
	{
		printTab(fp,tab);
		fprintf(fp,"octaves  %g\n",octaves);
	}
	if(lambda != 2)
	{
		printTab(fp,tab);
		fprintf(fp,"lambda  %g\n",lambda);
	}

	#ifdef DEBUG
		printf("Turbulence exported (POV) saved\n");
	#endif
	return 0;
}

