/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** view.cpp
*/
#include "view.h"

#include "camera.h"

view::view(world *wptr,const char *n,bool autodel) :
	base(wptr,n)
{
#if (USE_RTTI == 0)
	bv = 1;
#endif
	numtype = NUM_VIEW;
	
	autodelete = autodel;
	
	cameraptr = 0;

	peye = Vector3(0,0,1000);
	vview = Vector3(0,0,-1000);
	vup = Vector3(0,1,0);
	vright = Vector3(1,0,0);

	fov = 0.3;
	aspect_ratio = 1;

	worldptr = 0;
	subobject = 0;
	soname = 0;
	selectedptr = 0;

	drawselected = 0;

	projmode = PERSPECTIVE;

	// dummy values
	w = h = 100;

	calcVectors();
	createView(0);
	createScreen(w,h);
};

view::view(world *wptr,view &v,bool autodel) :
	base(wptr,v.getName())
{
#if (USE_RTTI == 0)
	bv = 1;
#endif
	numtype = NUM_VIEW;
	
	autodelete = autodel;
	
	cameraptr = 0;

	peye = v.peye;
	vview = v.vview;
	vup = v.vup;
	vright = v.vright;

	fov = v.fov;
	aspect_ratio = v.aspect_ratio;

	subobject = v.subobject;
	if(subobject)
		soname = strdup(subobject->getName());
	else
		soname = 0;
	selectedptr = 0;

	projmode = v.projmode;

	w = v.w;
	h = v.h;

	calcVectors();
	createView(0);
	createScreen(w,h);
};

view::view(world *wptr,view *v,bool autodel) :
	base(wptr,v)
{
#if (USE_RTTI == 0)
	bv = 1;
#endif
	numtype = NUM_VIEW;
	
	autodelete = autodel;
	
	cameraptr = 0;

	peye = v->peye;
	vview = v->vview;
	vup = v->vup;
	vright = v->vright;

	fov = v->fov;
	aspect_ratio = v->aspect_ratio;

	subobject = v->subobject;
	if(subobject)
		soname = strdup(subobject->getName());
	else
		soname = 0;
	selectedptr = 0;

	projmode = v->projmode;

	w = v->w;
	h = v->h;

	calcVectors();
	createView(0);
	createScreen(w,h);
};

view::~view()
{
	if(soname)
		delete soname;
		
	// disconnect of our world
	if(worldptr)
	{
		if(worldptr->removeView(this) >= 0)
		{
#ifdef DEBUG
			printf("view successfully removed from world\n");
#endif
		}
		else
		{
#ifdef DEBUG
			printf("view unsuccessfully not removed from world\n");
			printf("help, please!\n");
#endif
		}
	}
	// workaround
	removeFromParent();
}

bool	view::autoDelete()
{
	return autodelete;
}

void	view::setAutoDelete(bool ad)
{
	autodelete = ad;
}

base	*view::copy(base *p)
{
	if(p && p->getType() == NUM_WORLD)
		return new view((world*)p,this);
		
	return 0;
}

void	view::take(view &v)
{
	if(name)
		free(name);

	if(v.name)
		name = strdup(v.name);
	else
		name = strdup("Unnamed View");

	peye = v.peye;
	vview = v.vview;
	vup = v.vup;
	vright = v.vright;

	fov = v.fov;
	aspect_ratio = v.aspect_ratio;

	w = v.w;
	h = v.h;
	
	calcVectors();
	createView(0);
	createScreen(w,h);
};


void	view::take(view *v)
{
	if(v == 0)
		return;

	if(name)
		free(name);

	if(v->name)
		name = strdup(v->name);
	else
		name = strdup("Unnamed View");

	peye = v->peye;
	vview = v->vview;
	vup = v->vup;
	vright = v->vright;

	fov = v->fov;
	aspect_ratio = v->aspect_ratio;

	w = v->w;
	h = v->h;
	
	calcVectors();
	createView(0);
	createScreen(w,h);
};


void	view::setCamera(camera *c)
{
	cameraptr = c;
}

camera	*view::getCamera()
{
	return cameraptr;
}

int	view::setWorld(world *wp)
{
	worldptr = wp;

	if(worldptr)
	{
		return worldptr->addView(this);
	}

	return 0;
}

world	*view::getWorld()
{
	return worldptr;
}

int		view::setEye(Vector3 p)
{
	peye = p;

	calcVectors();

	return 0;
}

int		view::setLookAt(Vector3 p)
{
	vview = p - peye;

	calcVectors();

	return 0;
}

int		view::setUp(Vector3 p)
{
	vup = p;

	calcVectors();

	return 0;
}
int		view::setRight(Vector3 p)
{
	vright = p;

	calcVectors();

	return 0;
}

int		view::setFOV(double d)
{
	fov = d;

	if(fov < 0.0005) fov = 0.0005;
	if(fov > 179.9999999 / 180 * PI) fov = 179.9999999 / 180 * PI;

	calcVectors();

	return 0;
}

int		view::setAspectRatio(double d)
{
	aspect_ratio = d;

	calcVectors();

	return 0;
}

int		view::setScreenSize(int ww,int hh)
{
	w = ww;
	h = hh;

	calcVectors();

	return 0;
}

int		view::translate(Vector3 v)
{
	peye += v;
	
	return 0;
}

int		view::rotate(Vector3 v)
{
	Matrix44	m;
	Vector4		v4;

	m.unify();	
	m.rotateVector(v);
	
	vright = (v4 = vright) * m;
	vup = (v4 = vup) * m;
	vview = (v4 = vview) * m;
	
	return 0;
}

int		view::scale(Vector3 v)
{
	return 0;
}

void		view::calcVectors(int)
{
	calcVectors();
}

void		view::calcVectors()
{
	double		d,a;
	Vector3		va,vb;

	d = tan(fov / 2) * vview.length();
	a = (double)w / (double)h * aspect_ratio;
	
	vright.normalize();
	vright *= d * a;
	vup.normalize();
	vup *= d;

	va = vright * vup;
	va.normalize();
	vb = vview;
	vb.normalize();
	if(va != vb)
	{
//		printf("Warning: Viewaxes are not perpendicular!\n");
	}

	createView(0);
}


int		view::project(Vector3 v,Vector2 &xy,Matrix44 m,int)
{
	return project(v,xy,m);
}

int		view::project(Vector3 v,Vector2 &xy,Matrix44 m)
{
	switch(projmode)
	{
		case PERSPECTIVE:
		{
			Vector4	v4,vh;

			v4 = v;

			v4 *= m;
			v4 *= mview;
			vh = v4;
			v4 *= mscreen;

			if((vh[2]) < EPSILON) return -1;

			xy[0] = v4[0] / vh[2];
			xy[1] = v4[1] / vh[2];

			//xy.print();
		}
		break;
		case PARALLEL_XY:
		{
			Vector4	v4;

			v4 = v;
			v4 *= m;
			v4 /= tan(fov) / 1.4;

			xy[0] = (w / 2) + v4[0];
			xy[1] = (h / 2) - v4[1];

			//xy.print();
		}
		break;
		case PARALLEL_XZ:
		{
			Vector4	v4;

			v4 = v;
			v4 *= m;
			v4 /= tan(fov) / 1.4;

			xy[0] = (w / 2) + v4[0];
			xy[1] = (h / 2) - v4[2];

			//xy.print();
		}
		break;
		case PARALLEL_ZY:
		{
			Vector4	v4;

			v4 = v;
			v4 *= m;
			v4 /= tan(fov) / 1.4;

			xy[0] = (w / 2) + v4[2];
			xy[1] = (h / 2) - v4[1];

			//xy.print();
		}
		break;
	}

	return 0;
}

int		view::reproject(Vector2 xy,Vector3 &v,Matrix44 m,int)
{
	return reproject(xy,v,m);
}

int		view::reproject(Vector2 xy,Vector3 &v,Matrix44 m)
{
	switch(projmode)
	{
		case PERSPECTIVE:
		{
			Vector4	v4;

			v4 = xy * vview.length();

			v4 *= ~mscreen;
			v4 *= ~mview;

			v = v4 - peye;
		}
		break;
		case PARALLEL_XY:
		{
			Vector4	v4;

			v4 = xy;
			v[0] = (w / 2) - v4[0];
			v[1] = (h / 2) + v4[1];
			v4 *= tan(fov) / 1.4;

			v = v4;
		}
		break;
		case PARALLEL_XZ:
		{
			Vector4	v4;

			v4 = v;
			v4 *= m;
			v4 /= tan(fov) / 1.4;

			xy[0] = (w / 2) + v4[0];
			xy[1] = (h / 2) - v4[2];

			//xy.print();
		}
		break;
		case PARALLEL_ZY:
		{
			Vector4	v4;

			v4 = v;
			v4 *= m;
			v4 /= tan(fov) / 1.4;

			xy[0] = (w / 2) + v4[2];
			xy[1] = (h / 2) - v4[1];

			//xy.print();
		}
		break;
	}

	return 0;
}

void	view::createView(int anim)
{
	Matrix44	mh;
	Vector4		vr,vu,vv,p;
//	Matrix44	m;

/*
	if(cameraptr)
	{
		m = cameraptr->getMatrix(anim);
	}
	else
	{
		m.unify();
	}
	
	vv = Vector4(vview) * m;
	vr = Vector4(vright) * m;
	vu = Vector4(vup) * m;
	p = Vector4(peye) * m;
*/

	vv = Vector4(vview);
	vr = Vector4(vright);
	vu = Vector4(vup);
	p = Vector4(peye);
	
	mview(0,0) = vr[0];
	mview(0,1) = vu[0];
	mview(0,2) = vv[0];
	mview(0,3) = p[0];
	mview(1,0) = vr[1];
	mview(1,1) = vu[1];
	mview(1,2) = vv[1];
	mview(1,3) = p[1];
	mview(2,0) = vr[2];
	mview(2,1) = vu[2];
	mview(2,2) = vv[2];
	mview(2,3) = p[2];
	mview(3,0) = 0;
	mview(3,1) = 0;
	mview(3,2) = 0;
	mview(3,3) = 1;
	mh = mview;
	mview = ~mh;
}

void	view::createScreen(int w,int h)
{
	mscreen.unify();
	mscreen(0,0) = ((double)w - 1) / -2;
	mscreen(0,2) = ((double)w - 1) / 2;
	mscreen(1,1) = ((double)h - 1) / -2;
	mscreen(1,2) = ((double)h - 1) / 2;
}

int	view::exportPOV(FILE*,int,int,int)
{
	return 0;
}

int	view::exportPOV(FILE *fp,int tabsize,int anim)
{
	Vector3		v;
	Vector4		vr,vh,vv,p;
	int		tab = tabsize;
//	Matrix44	m;

	if(!fp)
		return -1;
		
/*
	if(cameraptr)
	{
		m = cameraptr->getMatrix(anim);
	}
	else
	{
		m.unify();
	}
	
	vv = Vector4(vview) * m;
	vr = Vector4(vright) * m;
	vh = Vector4(vup) * m;
	p = Vector4(peye) * m;
*/
	vv = Vector4(vview);
	vr = Vector4(vright);
	vh = Vector4(vup);
	p = Vector4(peye);

		
	fprintf(fp,"// Objectname = %s\n",name);
	fprintf(fp,"// Objecttype = view\n\n"); 
	fprintf(fp,"// This camera is build for ratio %i:%i\n\n\n",w,h); 

	fprintf(fp,"camera\n");
	fprintf(fp,"{\n");

	// This is the way povray looks (handle right and up)
	printTab(fp,tab);
	fprintf(fp,"up        <%g,%g,%g>\n",.0,vup.length() * 2.0,.0);

	printTab(fp,tab);
	fprintf(fp,"right     <%g,%g,%g>\n",vright.length() * 2.0,.0,.0);

	printTab(fp,0);
	fprintf(fp,"#ifdef (LEFTEYE)\n");

	printTab(fp,tab);
	fprintf(fp,"location  <%g - %g * f,%g - %g * f,%g - %g * f>\n",p[0],vr[0],p[1],vr[1],p[2],vr[2]);
	
	printTab(fp,0);
	fprintf(fp,"#else\n");
	
	printTab(fp,2);
	fprintf(fp,"#ifdef (RIGHTEYE)\n");

	printTab(fp,tab); 
	fprintf(fp,"location  <%g + %g * f,%g + %g * f,%g + %g * f>\n",p[0],vr[0],p[1],vr[1],p[2],vr[2]);

	printTab(fp,2);
	fprintf(fp,"#else\n");

	printTab(fp,tab); 
	fprintf(fp,"location  <%g,%g,%g>\n",p[0],p[1],p[2]);
	
	printTab(fp,2);
	fprintf(fp,"#end\n");
	printTab(fp,0);
	fprintf(fp,"#end\n");

	printTab(fp,tab); 
	fprintf(fp,"sky       <%g,%g,%g>\n",vh[0],vh[1],vh[2]);

//	printTab(fp,tab); 
//	fprintf(fp,"direction <%g,%g,%g>\n",vview[0],vview[1],vview[2]);

	
	
	v = p + vv;

	printTab(fp,tab); 
	fprintf(fp,"look_at   <%g,%g,%g>\n",v[0],v[1],v[2]);

	fprintf(fp,"}\n\n");

	return 0;
}

void		view::setProjectionMode(int pm)
{
	projmode = pm;
}

int		view::projectionMode()
{
	return projmode;
}

void		view::setSelected(base *bp)
{
	selectedptr = bp;
}

base		*view::getSelected()
{
	return selectedptr;
}

void		view::setDrawSelected(int sel)
{
	drawselected = sel;
}

int		view::save(media *m,int ver)
{
	if(!m) return -1;

	switch(ver)
	{
		case 0:
			setMedia(m);

			writeChunk("VIEW");

			if(name)
			{
				writeName(name);
			}
			else
			{
				writeName("View\0");
			}

			writeVector(peye);
			writeVector(vview);
			writeVector(vright);
			writeVector(vup);
			writeDouble(fov);
			writeInt(w);
			writeInt(h);

/*
			printf("peye   <%g,%g,%g>\n",peye[0],peye[1],peye[2]);
			printf("vview  <%g,%g,%g>\n",vview[0],vview[1],vview[2]);
			printf("vright <%g,%g,%g>\n",vright[0],vright[1],vright[2]);
			printf("vup    <%g,%g,%g>\n",vup[0],vup[1],vup[2]);
*/

			writeChunkLen();
		break;
		case 1:
		case -1:
			setMedia(m);

			writeChunk("VIEW");

			if(name)
			{
				writeName(name);
			}
			else
			{
				writeName("View\0");
			}

			writeVector(peye);
			writeVector(vview);
			writeVector(vright);
			writeVector(vup);
			writeDouble(fov);
			writeInt(w);
			writeInt(h);

			if(subobject)
				writeName(subobject->getName());
			else
				writeName(0);
			
/*
			printf("peye   <%g,%g,%g>\n",peye[0],peye[1],peye[2]);
			printf("vview  <%g,%g,%g>\n",vview[0],vview[1],vview[2]);
			printf("vright <%g,%g,%g>\n",vright[0],vright[1],vright[2]);
			printf("vup    <%g,%g,%g>\n",vup[0],vup[1],vup[2]);
*/

			writeChunkLen();
		break;
		default:
			return -2;
	}

	return 0;
}

int	view::load(media *m,int,int ver)
{
	if(!m) return -1;

	switch(ver)
	{
		case 0:
			name = readName();

			peye = readVector();
			vview = readVector();
			vright = readVector();
			vup = readVector();
			fov = readDouble();
			w = readInt();
			h = readInt();
 		break;
		case 1:
		case -1:
		{
			name = readName();

			peye = readVector();
//			printf("peye   <%g,%g,%g>\n",peye[0],peye[1],peye[2]);
			vview = readVector();
//			printf("vview  <%g,%g,%g>\n",vview[0],vview[1],vview[2]);
			vright = readVector();
//			printf("vright <%g,%g,%g>\n",vright[0],vright[1],vright[2]);
			vup = readVector();
//			printf("vup    <%g,%g,%g>\n",vup[0],vup[1],vup[2]);
			fov = readDouble();
			w = readInt();
			h = readInt();
			
			soname = readName();
		}
 		break;
		default:
			return -2;
	}

	return 0;
}


int	view::drawLine(Vector3,Vector3,Matrix44) {return -1;};
int	view::drawLine(Vector3,Vector3,Matrix44,int) {return -1;};
int	view::drawCross(Vector3,Matrix44) {return -1;};
int	view::drawCross(Vector3,Matrix44,int) {return -1;};
int	view::drawTriangle(Vector3,Vector3,Vector3,Matrix44) {return -1;};
int	view::drawTriangle(Vector3,Vector3,Vector3,Matrix44,int) {return -1;};
int	view::drawTrapezium(Vector3,Vector3,Vector3,Matrix44) {return -1;};
int	view::drawTrapezium(Vector3,Vector3,Vector3,Matrix44,int) {return -1;};
int	view::drawCircle(Vector3,Vector3,Vector3,Matrix44) {return -1;};
int	view::drawCircle(Vector3,Vector3,Vector3,Matrix44,int) {return -1;};

int	view::drawSphere(Vector3,Vector3,Vector3,Vector3,Matrix44) {return -1;};
int	view::drawSphere(Vector3,Vector3,Vector3,Vector3,Matrix44,int) {return -1;};
int	view::drawSemiSphere(Vector3,Vector3,Vector3,Vector3,Matrix44) {return -1;};
int	view::drawSemiSphere(Vector3,Vector3,Vector3,Vector3,Matrix44,int) {return -1;};
int	view::drawBox(Vector3,Vector3,Vector3,Vector3,Matrix44) {return -1;};
int	view::drawBox(Vector3,Vector3,Vector3,Vector3,Matrix44,int) {return -1;};
int	view::drawCone(Vector3,Vector3,Vector3,Vector3,double,Matrix44) {return -1;};
int	view::drawCone(Vector3,Vector3,Vector3,Vector3,double,Matrix44,int) {return -1;};
int	view::drawCylinder(Vector3,Vector3,Vector3,Vector3,Matrix44) {return -1;};
int	view::drawCylinder(Vector3,Vector3,Vector3,Vector3,Matrix44,int) {return -1;};
int	view::drawTorus(Vector3,Vector3,Vector3,Vector3,double,Matrix44) {return -1;};
int	view::drawTorus(Vector3,Vector3,Vector3,Vector3,double,Matrix44,int) {return -1;};

int	view::drawBlobSphere(Vector3,Vector3,Vector3,Vector3,Matrix44) {return -1;};
int	view::drawBlobSphere(Vector3,Vector3,Vector3,Vector3,Matrix44,int) {return -1;};
int	view::drawBlobCylinder(Vector3,Vector3,Vector3,Vector3,Matrix44) {return -1;};
int	view::drawBlobCylinder(Vector3,Vector3,Vector3,Vector3,Matrix44,int) {return -1;};

int	view::drawSymbol(Vector3,const char*,Matrix44) {return -1;};
int	view::drawSymbol(Vector3,const char*,Matrix44,int) {return -1;};

int	view::drawAxis(Matrix44) {return -1;};
int	view::drawAxis(Matrix44,int) {return -1;};
int	view::drawDragvector(Matrix44,Vector3,Vector3) {return -1;};
int	view::drawDragvector(Matrix44,Vector3,Vector3,int) {return -1;};

Vector3	view::getVView()
{
	return vview;
}

Vector3	view::getVRight()
{
	return vright;
}

Vector3	view::getVUp()
{
	return vup;
}

Vector3	view::getPEye()
{
	return peye;
}

int	view::calculate(int)
{
	setMin(Vector3(-0.5,-0.5,-0.5));
	setMax(Vector3(0.5,0.5,0.5));

	return 0;
}

double	view::volume(int)
{
	return 0;
}

void	view::dumpNames(int tab,int)
{
	printTab(stdout,tab);
	printf("view: %s\n",name);
}

int	view::addToParent(base *p)
{
	if(!p) return -2;

	parent = p;

	return p->addChild(this);
}

int	view::removeFromParent()
{
	if(!parent) return -2;

	return parent->removeChild(this);
}

int	view::draw(view*,Matrix44,int)
{
	return 0;
}

base	*view::getSubObject()
{
	base	*bp;
	
	if(worldptr != 0)
	{
		if(subobject)
			return subobject;
		if(soname != 0)
		{
			bp = worldptr->searchName(soname);
			
			if(bp == 0)
			{
				subobject = 0;
				
				return 0;
			}
			
			if(bp->getType() == NUM_WORLD)
			{
				subobject = 0;
				
				return 0;
			}
			
			if(bp->getType() == NUM_VIEW || bp->getType() == NUM_CAMERA)
			{
				subobject = 0;
				
				return 0;
			}
			
				
			return (subobject = bp);
		}
	}
	
	return 0;
}

int	view::setSubObject(base *so)
{
	subobject = so;
	
	if(soname)
		delete soname;
		
	if(subobject && subobject->getName())
		soname = strdup(subobject->getName());
	else
		soname = 0;
	
	return 0;
}

void	view::resize()
{
}

int	view::getWidth()
{
	return w;
}

int	view::getHeight()
{
	return h;
}

