/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** cmesh.cpp
*/


#include "cmesh.h" 
#include "view.h" 

#include "graph.h" 


cmesh::cmesh(base *p,char *n,texture *t) :
	smesh(p,n,t)
{
	numtype = NUM_CMESH;

	gptr = 0;

	if(parent)
		addToParent(parent);
}


cmesh::cmesh(base *p,char *n,Vector3 &v1,Vector3 &v2,Vector3 &v3):
	smesh(p,n)
{
	numtype = NUM_CMESH;

	if(parent)
		addToParent(parent);

	gptr = 0;

	addSegment(addNode(v1),addNode(v2),addNode(v3));
}

cmesh::cmesh(base *p,char *n,texture *t,Vector3 &v1,Vector3 &v2,Vector3 &v3) :
	smesh(p,n,t)
{
	numtype = NUM_CMESH;

	if(parent)
		addToParent(parent);

	gptr = 0;

	addSegment(addNode(v1),addNode(v2),addNode(v3));
}

cmesh::cmesh(base *p,cmesh *mc) :
	smesh(p,mc)
{
	int		i,s0,s1,s2;
	Vector3		v0,v1;

	numtype = NUM_CMESH;

	gptr = 0;

	if(parent)
		addToParent(parent);

	for(i = 0;i < mc->vectorlist.length();i++)
	{
		if(mc->getNode(i,v0,v1) >= 0)
			addNode(v0,v1);
	}
	for(i = 0;i < mc->trianglelist.length();i++)
	{
		if(mc->getSegment(i,s0,s1,s2) >= 0)
			addSegment(s0,s1,s2);
	}
}

cmesh::cmesh(smesh &m) : 	// Typumwandlung
	smesh(m.getParent(),"Unnamed")
{
	numtype = NUM_CMESH;

	gptr = 0;

	printf("Typecast not implemented\n");
}


cmesh::~cmesh()
{
	if(gptr)
		delete gptr;
}

base		*cmesh::copy(base *p)
{
	return new cmesh(p,this);
}



graph		*cmesh::createGraph()
{
	int		i,s1,s2,s3;

	if(gptr)
		return gptr;

	gptr = new graph();
	for(i = 0;i < vectorlist.length();i++)
	{
		new node(gptr,vectorlist[i].position);
	}
	for(i = 0;i < trianglelist.length();i++)
	{
		s1 = trianglelist[i].pointnum[0];
		s2 = trianglelist[i].pointnum[1];
		s3 = trianglelist[i].pointnum[2];
		new edge(gptr,gptr->getNode(s1),gptr->getNode(s2),gptr->numEdges() + 1);
		new edge(gptr,gptr->getNode(s2),gptr->getNode(s3),gptr->numEdges() + 1);
		new edge(gptr,gptr->getNode(s3),gptr->getNode(s1),gptr->numEdges() + 1);
	}

	return gptr;
}



int		cmesh::createFromGraph(graph *g)
{
	int		i,t,c,l,s1,s2,s3;
	Vector3		v;
	list<edge*>	del;
	edge		*e1,*e2,*e3;
	node		*k;
	graph		*gt;
	bool		sf;

	if(!g)
		return 0;

	vectorlist.empty();
	trianglelist.empty();

	gt = gptr;
	gptr = 0;

	// take all the points
	for(i = 0;i < g->numNodes();i++)
	{
		k = g->getNode(i);
		v = k->vector();
		addNode(v);
	}
	
	// fill the del (double edge list) with every edge twice
	// in a close mesh every edge is part of exactly 2 triangles
	for(i = 0;i < g->numEdges();i++)
	{
		del += g->getEdge(i);
	}
	for(i = g->numEdges() - 1;i >= 0;i--)
	{
		del += g->getEdge(i);
	}

	l = del.length();
	printf("del len %i\n",l);
	// create the segments
	for(c = 0;del.length() > 0 && c < l * 2;c++)
	{
		printf("%i. ",c);
		sf = false;
		e1 = del[c % del.length()];
		for(i = 0;!sf && i < e1->node2()->numEdges();i++)
		{
			e2 = e1->node2()->getEdge(i);
			if(e1 == e2 || del.find(e2) < 0 || e1->node2() != e2->node1())
				continue;
			for(t = 0;!sf && t < e2->node2()->numEdges();t++)
			{
				e3 = e2->node2()->getEdge(t);
				if(e1 == e3 || e2 == e3 || del.find(e3) < 0)
					continue;
/*
				printf("seg(%i,%i,%i) <%g,%g,%g><%g,%g,%g><%g,%g,%g>\n",e1->id(),e2->id(),e3->id(),
					e1->node1()->vector()[0],e1->node1()->vector()[1],e1->node1()->vector()[2],
					e2->node1()->vector()[0],e2->node1()->vector()[1],e2->node1()->vector()[2],
					e3->node1()->vector()[0],e3->node1()->vector()[1],e3->node1()->vector()[2]);
*/
				if(e2->node2() != e3->node1() || e3->node2() != e1->node1())
					continue;
				s1 = getNodeIndex(e1->node1());
				s2 = getNodeIndex(e2->node1());
				s3 = getNodeIndex(e3->node1());
				if(!hasSegment(s1,s2,s3))
				{
					if(addSegment(s1,s2,s3))
					{
						del.deleteAt(del.find(e1));
						del.deleteAt(del.find(e2));
						del.deleteAt(del.find(e3));
//						e1 = del[c % del.length()];
						sf = true;
						break;
					}
				}
			}
			for(t = 0;!sf && t < e2->node2()->numEdges();t++)
			{
				e3 = e2->node2()->getEdge(t)->swap();
				if(e1 == e3 || e2 == e3 || del.find(e3) < 0)
					continue;
/*
				printf("seg(%i,%i,%i) <%g,%g,%g><%g,%g,%g><%g,%g,%g>\n",e1->id(),e2->id(),e3->id(),
					e1->node1()->vector()[0],e1->node1()->vector()[1],e1->node1()->vector()[2],
					e2->node1()->vector()[0],e2->node1()->vector()[1],e2->node1()->vector()[2],
					e3->node1()->vector()[0],e3->node1()->vector()[1],e3->node1()->vector()[2]);
*/
				if(e2->node2() != e3->node1() || e3->node2() != e1->node1())
					continue;
				s1 = getNodeIndex(e1->node1());
				s2 = getNodeIndex(e2->node1());
				s3 = getNodeIndex(e3->node1());
				if(!hasSegment(s1,s2,s3))
				{
					if(addSegment(s1,s2,s3))
					{
						del.deleteAt(del.find(e1));
						del.deleteAt(del.find(e2));
						del.deleteAt(del.find(e3));
//						e1 = del[c % del.length()];
						sf = true;
						break;
					}
				}
			}
		}
		for(i = 0;!sf && i < e1->node2()->numEdges();i++)
		{
			e2 = e1->node2()->getEdge(i)->swap();
			if(e1 == e2 || del.find(e2) < 0 || e1->node2() != e2->node1())
				continue;
			for(t = 0;!sf && t < e2->node2()->numEdges();t++)
			{
				e3 = e2->node2()->getEdge(t);
				if(e1 == e3 || e2 == e3 || del.find(e3) < 0)
					continue;
/*
				printf("seg(%i,%i,%i) <%g,%g,%g><%g,%g,%g><%g,%g,%g>\n",e1->id(),e2->id(),e3->id(),
					e1->node1()->vector()[0],e1->node1()->vector()[1],e1->node1()->vector()[2],
					e2->node1()->vector()[0],e2->node1()->vector()[1],e2->node1()->vector()[2],
					e3->node1()->vector()[0],e3->node1()->vector()[1],e3->node1()->vector()[2]);
*/
				if(e2->node2() != e3->node1() || e3->node2() != e1->node1())
					continue;
				s1 = getNodeIndex(e1->node1());
				s2 = getNodeIndex(e2->node1());
				s3 = getNodeIndex(e3->node1());
				if(!hasSegment(s1,s2,s3))
				{
					if(addSegment(s1,s2,s3))
					{
						del.deleteAt(del.find(e1));
						del.deleteAt(del.find(e2));
						del.deleteAt(del.find(e3));
//						e1 = del[c % del.length()];
						sf = true;
						break;
					}
				}
			}
			for(t = 0;!sf && t < e2->node2()->numEdges();t++)
			{
				e3 = e2->node2()->getEdge(t)->swap();
				if(e1 == e3 || e2 == e3 || del.find(e3) < 0)
					continue;
/*
				printf("seg(%i,%i,%i) <%g,%g,%g><%g,%g,%g><%g,%g,%g>\n",e1->id(),e2->id(),e3->id(),
					e1->node1()->vector()[0],e1->node1()->vector()[1],e1->node1()->vector()[2],
					e2->node1()->vector()[0],e2->node1()->vector()[1],e2->node1()->vector()[2],
					e3->node1()->vector()[0],e3->node1()->vector()[1],e3->node1()->vector()[2]);
*/
				if(e2->node2() != e3->node1() || e3->node2() != e1->node1())
					continue;
				s1 = getNodeIndex(e1->node1());
				s2 = getNodeIndex(e2->node1());
				s3 = getNodeIndex(e3->node1());
				if(!hasSegment(s1,s2,s3))
				{
					if(addSegment(s1,s2,s3))
					{
						del.deleteAt(del.find(e1));
						del.deleteAt(del.find(e2));
						del.deleteAt(del.find(e3));
//						e1 = del[c % del.length()];
						sf = true;
						break;
					}
				}
			}
		}
//		sleep(1);
	}

	if(gt)
		delete gt;

	return 1;
}

bool	cmesh::hasSegment(edge *e1,edge *e2,edge *e3)
{
	int		n,s1,s2,s3;

       	s1 = s2 = s3 = -1;
       	for(n = 0;n < vectorlist.length();n++)
       	{
       		if(vectorlist[n].position == e1->node1()->vector())
       			s1 = n;
       		if(vectorlist[n].position == e1->node2()->vector())
       			s2 = n;
       		if(vectorlist[n].position == e2->node2()->vector())
       			s3 = n;
       	}
       	return hasSegment(s1,s2,s3);
}


int	cmesh::addSegment(edge *e1,edge *e2,edge *e3)
{
	int		n,s1,s2,s3;

	if(e1 != e2 && e1 != e3 && e2 != e3
		&& e1->node2() == e2->node1()
		&& e2->node2() == e3->node1()
		&& e3->node2() == e1->node1())
	{
		s1 = s2 = s3 = -1;
		for(n = 0;n < vectorlist.length();n++)
		{
			if(vectorlist[n].position == e1->node1()->vector())
				s1 = n;
			if(vectorlist[n].position == e1->node2()->vector())
				s2 = n;
			if(vectorlist[n].position == e2->node2()->vector())
				s3 = n;
		}
		addSegment(s1,s2,s3);
		printf("add seg (%i,%i,%i)\n",s1,s2,s3);

		return 1;
	}

	return 0;
}

bool	cmesh::hasSegment(node *n1,node *n2,node *n3)
{
	int		n,s1,s2,s3;

       	s1 = s2 = s3 = -1;
       	for(n = 0;n < vectorlist.length();n++)
       	{
       		if(vectorlist[n].position == n1->vector())
       			s1 = n;
       		if(vectorlist[n].position == n2->vector())
       			s2 = n;
       		if(vectorlist[n].position == n3->vector())
       			s3 = n;
       	}
       	return hasSegment(s1,s2,s3);
}


int	cmesh::addSegment(node *n1,node *n2,node *n3)
{
	int		n,s1,s2,s3;

	if(n1 != n2 && n1 != n3 && n2 != n3)
	{
		s1 = s2 = s3 = -1;
		for(n = 0;n < vectorlist.length();n++)
		{
			if(vectorlist[n].position == n1->vector())
				s1 = n;
			if(vectorlist[n].position == n2->vector())
				s2 = n;
			if(vectorlist[n].position == n3->vector())
				s3 = n;
		}
		addSegment(s1,s2,s3);
		printf("add seg (%i,%i,%i)\n",s1,s2,s3);

		return 1;
	}

	return 0;
}

int	cmesh::getNodeIndex(node *n1)
{
	int		n;

	for(n = 0;n < vectorlist.length();n++)
	{
		if(vectorlist[n].position == n1->vector())
			return n;
	}

	return -1;
}


int		cmesh::isClose()
{
	graph		*g;

	g = createGraph();

	return g->isClose();
}


list<edge*>		cmesh::getOpenPolygone(list<edge*> el)
{
	graph		*g;

	g = createGraph();

	return g->getOpenPolygone(el);
}

int		cmesh::closeMesh()
{
	graph		*g;

	g = createGraph();


	printf("graph has %i points and %i edges\n",g->numNodes(),g->numEdges());

/*
	g->removeDoubleNodes();
	g->removeDoubleEdges();
*/
	if(g->isClose())
	{
		printf("is close\n");
		return 1;
	}

	printf("try to close\n");


	if(g->close())
	{
		if(g->isClose())
		{
			printf("is close\n");
		}
		else
		{
			printf("is not close\n");
		}

		printf("graph has %i points and %i edges\n",g->numNodes(),g->numEdges());
		return createFromGraph(g);
	}

	printf("graph has %i points and %i edges\n",g->numNodes(),g->numEdges());

	createFromGraph(g);

	return 0;
}

int		cmesh::close()
{
	return closeMesh();
}

int		cmesh::isConvex()
{
	return 0;
}


int		cmesh::addMesh(cmesh&)
{
	return -1;
}

int		cmesh::addMesh(cmesh*)
{
	return -1;
}


int		cmesh::subtractMesh(cmesh&)
{
	return -1;
}

int		cmesh::subtractMesh(cmesh*)
{
	return -1;
}



int		cmesh::intersectMesh(cmesh&)
{
	return -1;
}


int		cmesh::intersectMesh(cmesh*)
{
	return -1;
}



int		cmesh::addToParent(base *p)
{
	if(!p) return -2;

	parent = p;
	return p->addChild(this);
}

int		cmesh::removeFromParent()
{
	if(!parent) return -2;

	return parent->removeChild(this);
}

void		cmesh::dumpNames(int tab,int)
{
	printTab(stdout,tab);
	printf("cmesh: %s    ( %i nodes %i segments)\n",name,getNumNodes(),getNumSegments());
}

int		cmesh::exportPOVMesh(FILE *fp,int tab,int tabsize,int anim)
{
	int		i;

	printTab(fp,tab); 
	fprintf(fp,"mesh\n"); 
	printTab(fp,tab); 
	fprintf(fp,"{\n"); 
	for(i = 0;i < trianglelist.length();i++)
	{
		printTab(fp,tab + tabsize);
		fprintf(fp,"triangle\n");
		printTab(fp,tab + tabsize);
		fprintf(fp,"{\n");
		printTab(fp,tab + tabsize + tabsize); 
		fprintf(fp,"<%g, %g, %g>,\n",
			vectorlist[trianglelist[i].pointnum[0]].position[0],
			vectorlist[trianglelist[i].pointnum[0]].position[1],
			vectorlist[trianglelist[i].pointnum[0]].position[2]);
		printTab(fp,tab + tabsize + tabsize); 
		fprintf(fp,"<%g, %g, %g>,\n",
			vectorlist[trianglelist[i].pointnum[1]].position[0],
			vectorlist[trianglelist[i].pointnum[1]].position[1],
			vectorlist[trianglelist[i].pointnum[1]].position[2]);
		printTab(fp,tab + tabsize + tabsize); 
		fprintf(fp,"<%g, %g, %g>\n",
			vectorlist[trianglelist[i].pointnum[2]].position[0],
			vectorlist[trianglelist[i].pointnum[2]].position[1],
			vectorlist[trianglelist[i].pointnum[2]].position[2]);

		printTab(fp,tab + tabsize);
		fprintf(fp,"} \n");
	}
	if(texptr) texptr->exportPOV(fp,tab + tabsize,tabsize,anim);

	transform::exportPOV(fp,tab + tabsize,tabsize,anim);

	printTab(fp,tab);
	fprintf(fp,"}\n\n");

	return 0;
}

int		cmesh::exportPOVCSG(FILE *fp,int tab,int tabsize,int anim)
{
	return -1;
}

int		cmesh::exportPOV(FILE *fp,int tab,int tabsize,int anim)
{
	if(isFlag(NO_EXPORT)) return 0;

	if(trianglelist.length() > 0)
	{
		printTab(fp,tab);
		fprintf(fp,"// Objectname = %s\n",name);
		printTab(fp,tab);
		fprintf(fp,"// Objecttype = cmesh\n");

		if(!isClose())
		{
			close();
		}

		if(!isClose() || !isConvex())
		{
			exportPOVMesh(fp,tab,tabsize,anim);
		}
		else
		{
			exportPOVCSG(fp,tab,tabsize,anim);
		}

	}

	return 0;
}

int		cmesh::save(media *m,int ver)
{
	int	i;

	if(!m)
		return -1;

	switch(ver)
	{
		case 0:
		case -1:
			setMedia(m);

			writeChunk("CMSH");
			writeNameChunk(name);

			saveFlags(m);
			anim::save(m);
			transform::save(m,ver);

			writeInt(vectorlist.length());
			for(i = 0;i < vectorlist.length();i++)
			{
				writeVector(vectorlist[i].position);
				writeVector(vectorlist[i].direction);
			}
			writeInt(trianglelist.length());
			for(i = 0;i < trianglelist.length();i++)
			{
				writeInt(trianglelist[i].pointnum[0]);
				writeInt(trianglelist[i].pointnum[1]);
				writeInt(trianglelist[i].pointnum[2]);
			}

			saveTexture(m);

			writeChunkLen();
		break;
		default:
			return -2;
	}

	return 0;
}

int		cmesh::load(media *m,int l,int ver)
{
	int		i,n,s0,s1,s2;
	int		pos;
	Vector3		v,vn;

	switch(ver)
	{
		case 0:
		case -1:
			setMedia(m);

			pos = m->tell();

			loadFlags(m,l);

			anim::load(m,l - (m->tell() - pos));
			transform::load(m,l - (m->tell() - pos),ver);

			n = readInt();
			for(i = 0;i < n;i++)
			{
				v = readVector();
				vn = readVector();

				addNode(v,vn);
			}

			n = readInt();
			for(i = 0;i < n;i++)
			{
				s0 = readInt();
				s1 = readInt();
				s2 = readInt();
				addSegment(s0,s1,s2);
			}

			if(l - (m->tell() - pos) > 0)
			{
				loadTexture(m,l - (m->tell() - pos));
			}
 		break;
		default:
			return -2;
	}

	return 0;
}



int		cmesh::addChild(box *b)
{
	vectorlist.empty();
	trianglelist.empty();

	addNode(-0.5,-0.5,-0.5);
	addNode( 0.5,-0.5,-0.5);
	addNode(-0.5, 0.5,-0.5);
	addNode( 0.5, 0.5,-0.5);
	addNode(-0.5,-0.5, 0.5);
	addNode( 0.5,-0.5, 0.5);
	addNode(-0.5, 0.5, 0.5);
	addNode( 0.5, 0.5, 0.5);

	addSegment(0,1,3);
	addSegment(0,3,2);
	addSegment(0,4,6);
	addSegment(0,6,2);
	addSegment(0,1,5);
	addSegment(0,4,5);
	addSegment(7,6,4);
	addSegment(7,4,5);
	addSegment(7,3,1);
	addSegment(7,1,5);
	addSegment(7,6,2);
	addSegment(7,2,3);

	return 0;
}

int		cmesh::addChild(cone *c)
{
	int		i,s = 32,p4,p1,p2,p3;
	Vector3		v;
	double		a;

	vectorlist.empty();
	trianglelist.empty();


	addNode(0,0,0.5);
	addNode(0,0,-0.5);
	for(i = 0;i < s;i++)
	{
		a = (double)i / (double)s * 2 * PI;
		v = Vector3(0.5 * sin(a),
			0.5 * cos(a),
			0.5);
		addNode(v);
		v = Vector3(c->capRadius() * sin(a),
			c->capRadius() * cos(a),
			-0.5);
		addNode(v);
	}

	for(i = 0;i < s;i++)
	{
		p1 = i * 2 + 2;
		p2 = i * 2 + 1 + 2;
		p3 = (i * 2 + 2) % (s * 2) + 2;
		p4 = (i * 2 + 3) % (s * 2) + 2;
		addSegment(0,p1,p3);
		addSegment(p1,p2,p3);
		addSegment(p1,p3,p4);
		addSegment(1,p2,p4);
	}

	return 0;
}

int		cmesh::addChild(cylinder *c)
{
	int		i,s = 32,p4,p1,p2,p3;
	Vector3		v;
	double		a;

	vectorlist.empty();
	trianglelist.empty();


	addNode(0,0,0.5);
	addNode(0,0,-0.5);
	for(i = 0;i < s;i++)
	{
		a = (double)i / (double)s * 2 * PI;
		v = Vector3(0.5 * sin(a),
			0.5 * cos(a),
			0.5);
		addNode(v);
		v = Vector3(0.5 * sin(a),
			0.5 * cos(a),
			-0.5);
		addNode(v);
	}

	for(i = 0;i < s;i++)
	{
		p1 = i * 2 + 2;
		p2 = i * 2 + 1 + 2;
		p3 = (i * 2 + 2) % (s * 2) + 2;
		p4 = (i * 2 + 3) % (s * 2) + 2;
		addSegment(0,p1,p3);
		addSegment(p1,p2,p3);
		addSegment(p1,p3,p4);
		addSegment(1,p2,p4);
	}

	return 0;
}

int		cmesh::addChild(sphere *c)
{
	int		i,t,s = 32,p4,p1,p2,p3;
	Vector3		v;
	double		a,b;

	vectorlist.empty();
	trianglelist.empty();


	addNode(0,0.5,0);
	addNode(0,-0.5,0);
	for(t = 1;t < (s / 2);t++)
	{
		b = (double)t / (double)s * 2 * PI;
		for(i = 0;i < s;i++)
		{
			a = (double)i / (double)s * 2 * PI;
			v = Vector3(0.5 * sin(a) * sin(b),
				0.5 * cos(b),
				0.5 * cos(a) * sin(b));
			addNode(v);
		}
	}

	for(i = 0;i < s;i++)
	{
		p1 = i + 2;
		p2 = (i + 1) % s + 2;
		addSegment(0,p1,p2);
	}
	for(t = 0;t < (s / 2) - 1;t++)
	{
		for(i = 0;i < s;i++)
		{
			p1 = t * s + i + 2;
			p2 = (t + 1) * s + i + 2;
			p3 = t * s + (i + 1) % s + 2;
			p4 = (t + 1) * s + (i + 1) % s + 2;
			addSegment(p1,p2,p4);
			addSegment(p1,p3,p4);
		}
	}
	for(i = 0;i < s;i++)
	{
		p1 = i + s * (s / 2 - 2) + 2;
		p2 = (i + 1) % s + s * (s / 2 - 2) + 2;
		addSegment(1,p1,p2);
	}

	return 0;
}

int		cmesh::addChild(text *c)
{
	int			i;

	vectorlist.empty();
	trianglelist.empty();

	if((i = c->getMesh(this)) != 0)
		return i;

/*
	printf("Got %i nodes and %i edges\n",vectorlist.length(),trianglelist.length() * 3);

	for(i = 0;i < vectorlist.length();i++)
		printf("%i. <%g,%g,%g>\n",i,vectorlist[i].position[0],vectorlist[i].position[1],vectorlist[i].position[2]);
	for(i = 0;i < trianglelist.length();i++)
		printf("%i. <%i,%i,%i>\n",i,trianglelist[i].pointnum[0],trianglelist[i].pointnum[1],trianglelist[i].pointnum[2]);
*/

	close();

	return 0;
}

int		cmesh::addChild(csgobj *co)
{
	if(gptr)
		delete gptr;
	gptr = 0;

	switch(co->type())
	{
		case NUM_BOX:
			addChild((box*)co);
		break;
		case NUM_CONE:
			addChild((cone*)co);
		break;
		case NUM_CYLINDER:
			addChild((cylinder*)co);
		break;
		case NUM_SPHERE:
			addChild((sphere*)co);
		break;
		case NUM_TEXT:
			addChild((text*)co);
		break;
		default:
			return -1;
	}

	return -1;
}

int		cmesh::addChild(blobobj *co)
{
	return addChild((csgobj*)co);
}

int		cmesh::subdevide()
{
	int	error;
		
      	error = smesh::subdevide();
		
	if(gptr)
	{
		delete gptr;
		gptr = 0;
	}
	
      	if(error != 0)
      		return error;
			
      	error = close();
		
#ifdef DEBUG
      	printf("Mesh is %s after subdevision\n",(error == 0?"close":"not close"));
#endif

      	return error;
}

int		cmesh::subdevideClever(int base,double grade)
{
      	return smesh::subdevideClever(base,grade);
}

int		cmesh::simplify(double grade)
{
	graph		*g;
        list<node*>	nodelist;
        node		*np,*fn;
        int		i;
        Vector3		v1,v2,v3,vn;

	g = createGraph();

//	printf("graph has %i points and %i edges\n",g->numNodes(),g->numEdges());
	printf("simplify\n");

	if(!g->close())
	{
		printf("is not close => no simplyfication\n");
		return -1;
	}
	
	for(i = 0;i < g->numNodes();i++)
	{
		nodelist += g->getNode(i);
	}
	
	fn = 0;
	while(!nodelist.isEmpty())
	{
#ifdef DEBUG
		printf("length nodelist %i\n",nodelist.length());
#endif

		np = nodelist[0];
		if(!np)
			break;
		
		if(np->numEdges() == 2)
		{
			printf("graph not close!!! double edge found.\n");
			// replace these 2 edges by one new edge
		}
		else if(np->numEdges() == 3)
		{
			v1 = np->getEdge(0)->vector();
			v2 = np->getEdge(1)->vector();
			v3 = np->getEdge(2)->vector();
			vn = v1 * v2;
			if(vn.scalarprod(v3) < grade)
			{
				// delete the 3 edges
				// delete the node
				nodelist.deleteAt(nodelist.find(np));
				delete np;
			}
		}
		else
		{
			if(fn == np)
				break;
			if(!fn)
				fn = np;
			nodelist.moveToEnd(0);
		}
	}
	
	createFromGraph(g);
	
      	return 0;
}

int		cmesh::smooth()
{
        int		i,n;
        node		*np;
        Vector3		v,v1,v2,va,vn;
        graph		*g;

#ifdef DEBUG
	printf("smooth: %i points and %i segments\n",vectorlist.length(),trianglelist.length());
#endif

	if(gptr)
	{
		delete gptr;
		gptr = 0;
	}
	
	g = createGraph();

#ifdef DEBUG
	printf("smooth: graph has %i points and %i edges\n",g->numNodes(),g->numEdges());
#endif

/*
	if(!g->close())
	{
		printf("is not close => no smoothification\n");
		return -1;
	}
*/
	
#ifdef DEBUG
	printf("\n");
#endif
        for(i = 0;i < g->numNodes();i++)
        {
        	v = Vector3(0,0,0);
        	np = g->getNode(i);
        	va = np->vector();
#ifdef DEBUG
		printf("%2i.  (%g,%g,%g) + \n",i,va[0],va[1],va[2]);
#endif
        	for(n = 0;n < np->numEdges();n++)
        	{
        		v1 = np->getEdge(n)->node1()->vector();
        		v2 = np->getEdge(n)->node2()->vector();
        		if(v1 == va)
        		{
        			v += v2;
#ifdef DEBUG
				printf("     (%g,%g,%g) + \n",v2[0],v2[1],v2[2]);
#endif
        		}
        		else
        		{
        			v += v1;
#ifdef DEBUG
				printf("     (%g,%g,%g) + \n",v1[0],v1[1],v1[2]);
#endif
        		}
        	}
        	v /= np->numEdges();
        	vn = (v * 0.2) + (va * 0.8);
        	np->setVector(vn);
#ifdef DEBUG
		printf("=    (%g,%g,%g)\n",vn[0],vn[1],vn[2]);
#endif
        }
	
	createFromGraph(g);
	
	return 0;
}


