/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** transform.cpp
*/
#include <string.h>
#include <stdlib.h>

#include "transform.h"
#include "dragvector.h"

#include <misc.h>

typedef	int (transform::*dmethodfn)(param**,list<param*>*);

transform::transform(base *p,const char *n) :
	base(p,n)
{
	numtype = NUM_TRANSFORM;
	vscale = Vector3(1,1,1);
	vrotate = Vector3(0,0,0);
	vtranslate = Vector3(0,0,0);
	vascale = Vector3(1,1,1);
	varotate = Vector3(0,0,0);
	vatranslate = Vector3(0,0,0);

	dmethodfn	d;

	d = &ascale;
	addMethod("scale",(methodfn)d);
	d = &asetScalation;
	addMethod("setscalation",(methodfn)d);
	d = &agetScalation;
	addMethod("getscalation",(methodfn)d);
	d = &agetSize;
	addMethod("getsize",(methodfn)d);

	d = &arotate;
	addMethod("rotate",(methodfn)d);
	d = &asetRotation;
	addMethod("setrotation",(methodfn)d);
	d = &agetRotation;
	addMethod("getrotation",(methodfn)d);
	d = &agetAngle;
	addMethod("getangle",(methodfn)d);

	d = &atranslate;
	addMethod("translate",(methodfn)d);
	d = &asetTranslation;
	addMethod("settranslation",(methodfn)d);
	d = &agetTranslation;
	addMethod("gettranslation",(methodfn)d);
	d = &agetPosition;
	addMethod("getposition",(methodfn)d);

	d = &agetObjectDistance;
	addMethod("getobjectdistance",(methodfn)d);
	addMethod("getObjectDistance",(methodfn)d);

	addDragvector(new dragvector(Vector3(0.5,0,0),&vscale[0],0,0,2));
	addDragvector(new dragvector(Vector3(0,0.5,0),0,&vscale[1],0,2));
	addDragvector(new dragvector(Vector3(0,0,0.5),0,0,&vscale[2],2));
}

transform::transform(base *p,const char *n,Vector3 &vs,Vector3 &vr,Vector3 &vt) :
	base(p,n)
{
	numtype = NUM_TRANSFORM;
	vscale = vs;
	vrotate = vr;
	vtranslate = vt;
	vascale = Vector3(1,1,1);
	varotate = Vector3(0,0,0);
	vatranslate = Vector3(0,0,0);

	dmethodfn	d;

	d = &ascale;
	addMethod("scale",(methodfn)d);
	d = &asetScalation;
	addMethod("setscalation",(methodfn)d);
	d = &agetScalation;
	addMethod("getscalation",(methodfn)d);
	d = &agetSize;
	addMethod("getsize",(methodfn)d);

	d = &arotate;
	addMethod("rotate",(methodfn)d);
	d = &asetRotation;
	addMethod("setrotation",(methodfn)d);
	d = &agetRotation;
	addMethod("getrotation",(methodfn)d);
	d = &agetAngle;
	addMethod("getangle",(methodfn)d);

	d = &atranslate;
	addMethod("translate",(methodfn)d);
	d = &asetTranslation;
	addMethod("settranslation",(methodfn)d);
	d = &agetTranslation;
	addMethod("gettranslation",(methodfn)d);
	d = &agetPosition;
	addMethod("getposition",(methodfn)d);

	d = &agetObjectDistance;
	addMethod("getobjectdistance",(methodfn)d);
	addMethod("getObjectDistance",(methodfn)d);

	addDragvector(new dragvector(Vector3(0.5,0,0),&vscale[0],0,0,2));
	addDragvector(new dragvector(Vector3(0,0.5,0),0,&vscale[1],0,2));
	addDragvector(new dragvector(Vector3(0,0,0.5),0,0,&vscale[2],2));
}

transform::transform(base *p,transform *dc) :
	base(p,dc)
{
	numtype = NUM_TRANSFORM;
	vscale = dc->vscale;
	vrotate = dc->vrotate;
	vtranslate = dc->vtranslate;
	vascale = dc->vascale;
	varotate = dc->varotate;
	vatranslate = dc->vatranslate;

	dmethodfn	d;

	d = &ascale;
	addMethod("scale",(methodfn)d);
	d = &asetScalation;
	addMethod("setscalation",(methodfn)d);
	d = &agetScalation;
	addMethod("getscalation",(methodfn)d);
	d = &agetSize;
	addMethod("getsize",(methodfn)d);

	d = &arotate;
	addMethod("rotate",(methodfn)d);
	d = &asetRotation;
	addMethod("setrotation",(methodfn)d);
	d = &agetRotation;
	addMethod("getrotation",(methodfn)d);
	d = &agetAngle;
	addMethod("getangle",(methodfn)d);

	d = &atranslate;
	addMethod("translate",(methodfn)d);
	d = &asetTranslation;
	addMethod("settranslation",(methodfn)d);
	d = &agetTranslation;
	addMethod("gettranslation",(methodfn)d);
	d = &agetPosition;
	addMethod("getposition",(methodfn)d);

	d = &agetObjectDistance;
	addMethod("getobjectdistance",(methodfn)d);
	addMethod("getObjectDistance",(methodfn)d);

	addDragvector(new dragvector(Vector3(0.5,0,0),&vscale[0],0,0,2));
	addDragvector(new dragvector(Vector3(0,0.5,0),0,&vscale[1],0,2));
	addDragvector(new dragvector(Vector3(0,0,0.5),0,0,&vscale[2],2));
}

transform::~transform()
{
}

void	transform::transformMatrix(Matrix44 &m,int anim)
{
	Matrix44	h,h2;
	Vector3		vs,vr,vt;

	vs = vscale;
	vr = vrotate;
	vt = vtranslate;
	if(anim)
	{
		vs[0] *= vascale[0];
		vs[1] *= vascale[1];
		vs[2] *= vascale[2];
		vr += varotate;
		vt += vatranslate;
	}

	h2.unify();
	h.scaleVector(vs);
	h2 *= h;
	h.rotateVector(vr);
	h2 *= h;
	h.transposeVector(vt);
	h2 *= h;
	m = h2 * m;
}

int	transform::exportPOV(FILE *fp,int tab,int,int anim)
{
	Vector3		vs,vr,vt;

	vs = vscale;
	vr = vrotate;
	vt = vtranslate;

	if(anim)
	{
		vs[0] *= vascale[0];
		vs[1] *= vascale[1];
		vs[2] *= vascale[2];
		vr += varotate;
		vt += vatranslate;
	}

	printTab(fp,tab);
	fprintf(fp,"scale     <%g,%g,%g>\n",vs[0],vs[1],vs[2]);
	printTab(fp,tab);
	fprintf(fp,"rotate    <%g,%g,%g>\n",vr[0] * 180 / PI,vr[1] * 180 / PI,vr[2] * 180 / PI);
	printTab(fp,tab);
	fprintf(fp,"translate <%g,%g,%g>\n",vt[0],vt[1],vt[2]);

	return 0;
}

void	transform::translate(Vector3 v)
{
	vtranslate += v;
}

void	transform::translate(double x,double y,double z)
{
	vtranslate[0] += x;
	vtranslate[1] += y;
	vtranslate[2] += z;
}

int	transform::atranslate(param**,list<param*> *pl)
{
	Vector3		v;

	if(pl->isEmpty() || pl->at(0)->type() != 5) return -1;

	v = ((pmvector*)pl->at(0))->vector();

	//printf("translating (%g,%g,%g)\n",v(0),v(1),v(2));

	vatranslate = v;

	return 0;
}

int	transform::asetTranslation(param**,list<param*> *pl)
{
	Vector3		v;

	if(pl->isEmpty() || pl->at(0)->type() != 5) return -1;

	v = ((pmvector*)pl->at(0))->vector();

	//printf("translating (%g,%g,%g)\n",v(0),v(1),v(2));

	vatranslate = v;

	return 0;
}

int	transform::agetTranslation(param **p,list<param*>*)
{
	if(p == 0) return -1;

	*p = new pmvector(Vector3(0,0,0));

	((pmvector*)*p)->setVector(vatranslate);

	return 0;
}

int	transform::agetPosition(param **p,list<param*>*)
{
	if(p == 0) return -1;

	*p = new pmvector(Vector3(0,0,0));

	((pmvector*)*p)->setVector(vatranslate + vtranslate);

	return 0;
}

void	transform::rotate(Vector3 v)
{
	vrotate += v;
}

void	transform::rotate(double x,double y,double z)
{
	vrotate[0] += x;
	vrotate[1] += y;
	vrotate[2] += z;
}

void	transform::rotateDeg(Vector3 v)
{
	vrotate += v * PI / 180;
}

void	transform::rotateDeg(double x,double y,double z)
{
	vrotate[0] += x * PI / 180;
	vrotate[1] += y * PI / 180;
	vrotate[2] += z * PI / 180;
}

int	transform::arotate(param**,list<param*> *pl)
{
	Vector3		v;

	if(pl->isEmpty() || pl->at(0)->type() != 5) return -1;

	v = ((pmvector*)pl->at(0))->vector();

	//printf("rotating (%g,%g,%g)\n",v(0),v(1),v(2));

	varotate = v * PI / 180;

	return 0;
}

int	transform::asetRotation(param**,list<param*> *pl)
{
	Vector3		v;

	if(pl->isEmpty() || pl->at(0)->type() != 5) return -1;

	v = ((pmvector*)pl->at(0))->vector();

	//printf("rotating (%g,%g,%g)\n",v(0),v(1),v(2));

	varotate = v * PI / 180;

	return 0;
}

int	transform::agetRotation(param **p,list<param*>*)
{
	Vector3		v;

	if(p == 0) return -1;

	v = varotate * 180 / PI;

	((pmvector*)*p)->setVector(v);

	return 0;
}

int	transform::agetAngle(param **p,list<param*>*)
{
	Vector3		v;

	if(p == 0) return -1;

	v = (varotate + vrotate) * 180 / PI;

	((pmvector*)*p)->setVector(v);

	return 0;
}

void	transform::scale(Vector3 v)
{
	vscale[0] *= v[0];
	vscale[1] *= v[1];
	vscale[2] *= v[2];
}

void	transform::scale(double d)
{
	vscale *= d;
}

void	transform::scale(double x,double y,double z)
{
	vscale[0] *= x;
	vscale[1] *= y;
	vscale[2] *= z;
}

int	transform::ascale(param**,list<param*> *pl)
{
	Vector3		v;

	if(pl->isEmpty() || pl->at(0)->type() != 5) return -1;

	v = ((pmvector*)pl->at(0))->vector();

	//printf("scaling (%g,%g,%g)\n",v(0),v(1),v(2));

	vascale = v;

	return 0;
}

int	transform::asetScalation(param**,list<param*> *pl)
{
	Vector3		v;

	if(pl->isEmpty() || pl->at(0)->type() != 5) return -1;

	v = ((pmvector*)pl->at(0))->vector();

	//printf("scaling (%g,%g,%g)\n",v(0),v(1),v(2));

	vascale = v;

	return 0;
}

int	transform::agetScalation(param **p,list<param*>*)
{
	if(p == 0) return -1;

	((pmvector*)*p)->setVector(vascale);

	return 0;
}

int	transform::agetSize(param **p,list<param*>*)
{
	Vector3		v;

	if(p == 0) return -1;

	v[0] = vscale[0] * vascale[0];
	v[1] = vscale[1] * vascale[1];
	v[2] = vscale[2] * vascale[2];

	((pmvector*)*p)->setVector(v);

	return 0;
}

int	transform::agetObjectDistance(param **p,list<param*> *pl)
{
	base		*bp;
	transform		*dp;
        char		*n = 0;
        Vector3		v,vm,vo,vd,vr;
        Vector4		v4;
        Matrix44	m;

	if(pl->isEmpty() ||
		(pl->at(0)->type() != param::PM_OBJECT &&
		pl->at(0)->type() != param::PM_TEXT)) return -1;
	if(p == 0) return -2;

	if(pl->at(0)->type() == param::PM_OBJECT)
		n = ((pmobject*)pl->at(0))->getName();
	else if(pl->at(0)->type() == param::PM_TEXT)
		n = ((pmtext*)pl->at(0))->getText();
	else return -1;
		
	*p = new pmvector(Vector3(0,0,0));

	vm = mapToGlobal(1);
	vo = Vector3(0,0,0);
	
	bp = getRoot();
	if(bp)
	{
		bp = bp->searchName(n);
#if (USE_RTTI == 1)
		if(bp && (dp = dynamic_cast<transform*>(bp)) != 0)
		{
#else
		if(bp && (bp->getType() != NUM_WORLD))
		{
			dp = (transform*)bp;
#endif
			vo = dp->mapToGlobal(1);
			vr = dp->mapToGlobalAngle(1);
		}		
	}
	
	vd = vo - vm;
	m.unify();
	m.rotateVector(vr);
	
	v4 = vd;
	v = v4 * m;
	
	((pmvector*)*p)->setVector(v);

	return 0;
}

int	transform::save(media *m,int ver)
{
	chunk	c(m);

	if(!m) return -1;

	switch(ver)
	{
		case 0:
			c.writeChunk("DIM ");
			c.writeVector(vscale);
			c.writeVector(vrotate);
			c.writeVector(vtranslate);
			c.writeChunkLen();
		break;
		case 1:
		case -1:
			c.writeChunk("TRNF");
			c.writeVector(vscale);
			c.writeVector(vrotate);
			c.writeVector(vtranslate);
			c.writeChunkLen();
		break;
		default:
			return -2;
	}

	return 0;
}



int	transform::load(media *m,int,int ver)
{
	char	chunkstr[4];
	int	len;
	chunk	c(m);

	if(!m) return -1;

	switch(ver)
	{
		case 0:
			m->read(chunkstr,4);
			m->read(&len,4);

			if(strncmp(chunkstr,"DIM ",4) != 0)
			{
				m->seek(-8,SEEK_CUR);
				return -1;
			}

			vscale = c.readVector();	
			vrotate = c.readVector();	
			vtranslate = c.readVector();	
		break;
		case 1:
		case -1:
			m->read(chunkstr,4);
			m->read(&len,4);

			if(strncmp(chunkstr,"TRNF",4) != 0)
			{
				m->seek(-8,SEEK_CUR);
				return -1;
			}

			vscale = c.readVector();	
			vrotate = c.readVector();	
			vtranslate = c.readVector();	
		break;
		default:
			return -2;
	}

	return 0;
}


Vector3		&transform::getVScale()
{
	return vscale;
}

Vector3		&transform::getVRotate()
{
	return vrotate;
}

Vector3		&transform::getVTranslate()
{
	return vtranslate;
}

void		transform::setVScale(Vector3 &v)
{
	vscale = v;
}

void		transform::setVRotate(Vector3 &v)
{
	vrotate = v;
}

void		transform::setVTranslate(Vector3 &v)
{
	vtranslate = v;
}


Matrix44	transform::getMatrix(int anim)
{
	Matrix44	m;

	if(parent)
	{
		m = parent->getMatrix(anim);
	}
	else
	{
		m.unify();
	}

	transformMatrix(m,anim);
	
	return m;
}

Matrix44	transform::getMatrix(transform *p,int anim)
{
	Matrix44	m;

	if(parent && parent != p)
	{
		m = parent->getMatrix(anim);
	}
	else
	{
		m.unify();
	}

	transformMatrix(m,anim);
	
	return m;
}

Vector3		transform::mapToGlobal(int anim)
{
	Vector3		v;
	Matrix44	m;
	
	m = getMatrix(anim);
	v = Vector4(0,0,0,1) * m;
	
	return v;
}
	
Vector3		transform::mapToParent(transform *p,int anim)
{
	Vector3		v;
	Matrix44	m;
	
	m = getMatrix(p,anim);
	v = Vector4(0,0,0,1) * m;
	
	return v;
}
	
Vector3		transform::mapToGlobalAngle(int anim)
{
	Vector3		v;
	transform		*dp;

	v = vrotate;
	if(anim)
		v += varotate;
	
	if(parent)
	{
#if (USE_RTTI == 1)
		if((dp = dynamic_cast<transform*>(parent)) != 0)
		{
#else
		if((parent->getType() != NUM_WORLD))
		{
			dp = (transform*)parent;
#endif
			v += dp->mapToGlobalAngle(anim);
		}
	}

	return v;
}
	
Vector3		transform::mapToParentAngle(transform *p,int anim)
{
	Vector3		v;
	transform		*dp;
	
	v = vrotate;
	if(anim)
		v += varotate;
	
	if(p == this)
		return v;
		
	if(parent)
	{
#if (USE_RTTI == 1)
		if((dp = dynamic_cast<transform*>(parent)) != 0)
		{
#else
		if((parent->getType() != NUM_WORLD))
		{
			dp = (transform*)parent;
#endif
			v += dp->mapToParentAngle(p,anim);
		}
	}
	
	return v;
}
	
	
	