/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** dragvector.cpp
*/
#include "dragvector.h"


dragvector::dragvector(Vector3 v,Vector3 *vp,double m)
{
	vecmod = vp;
	xmod = 0;
	ymod = 0;
	zmod = 0;
	d_multiplier = m;
	d_direction = v;
	flags = DEFAULT;
}

dragvector::dragvector(Vector3 v,double *x,double *y,double *z,double m)
{
	vecmod = 0;
	xmod = x;
	ymod = y;
	zmod = z;
	d_multiplier = m;
	d_direction = v;
	flags = DEFAULT;
}

void		dragvector::setDirection(Vector3 v)
{
	d_direction = v;
	if(v == Vector3(0,0,0))
	{
		printf("Warning: Set direction to (0,0,0)\n");
		printf("         This may cause floatingpoint-errors\n");
	}
}

void		dragvector::setPosition(Vector3 v)
{
	d_position = v;
}

void		dragvector::setGeometry(Vector3 d,Vector3 p)
{
	setDirection(d);
	setPosition(p);
}

Vector3		dragvector::direction()
{
	return d_direction;
}

Vector3		dragvector::position()
{
	return d_position;
}

int		dragvector::isFlag(DRAG_USE f)
{
	return (flags & f);
}

void		dragvector::setFlag(DRAG_USE f,int set)
{
	if(set)
	{
		flags = (DRAG_USE)((int)flags | (int)f);
	}
	else
	{
		flags = (DRAG_USE)((int)flags & ~(int)f);
	}
}

void		dragvector::multiplyX(Vector3 v)
{
	if(xmod)
	{
		*xmod *= v[0] * d_multiplier;
	}
}

void		dragvector::multiplyY(Vector3 v)
{
	if(ymod)
	{
		*ymod *= v[1] * d_multiplier;
	}
}

void		dragvector::multiplyZ(Vector3 v)
{
	if(zmod)
	{
		*zmod *= v[2] * d_multiplier;
	}
}

void		dragvector::multiplyVector(Vector3 v)
{
	if(vecmod)
	{
		(*vecmod)[0] *= v[0] * d_multiplier;
		(*vecmod)[1] *= v[1] * d_multiplier;
		(*vecmod)[2] *= v[2] * d_multiplier;
	}
}

void		dragvector::multiply(Vector3 v)
{
	multiplyX(v);
	multiplyY(v);
	multiplyZ(v);
	multiplyVector(v);

	if(isFlag(MODIFY_LENGTH))
	{
		modifyLength();
	}
}

void		dragvector::addX(Vector3 v)
{
	if(xmod)
	{
		*xmod += v[0] * d_multiplier;
	}
}

void		dragvector::addY(Vector3 v)
{
	if(ymod)
	{
		*ymod += v[1] * d_multiplier;
	}
}

void		dragvector::addZ(Vector3 v)
{
	if(zmod)
	{
		*zmod += v[2] * d_multiplier;
	}
}

void		dragvector::addVector(Vector3 v)
{
	if(vecmod)
	{
		(*vecmod)[0] += v[0] * d_multiplier;
		(*vecmod)[1] += v[1] * d_multiplier;
		(*vecmod)[2] += v[2] * d_multiplier;
	}
}

void		dragvector::add(Vector3 v)
{
	addX(v);
	addY(v);
	addZ(v);
	addVector(v);

	if(isFlag(MODIFY_LENGTH))
	{
		modifyLength();
	}
}

void		dragvector::modifyLength()
{
	if(xmod)
	{
		d_direction[0] = *xmod / d_multiplier;
	}
	if(ymod)
	{
		d_direction[1] = *ymod / d_multiplier;
	}
	if(zmod)
	{
		d_direction[2] = *zmod / d_multiplier;
	}
	if(vecmod)
	{
		d_direction = *vecmod / d_multiplier;
	}
}

void		dragvector::setMultiplier(double m)
{
	d_multiplier = m;
}

double		dragvector::multiplier()
{
	return d_multiplier;
}


