#include "board.h"
#include <kapp.h>
#include "kpuissance4.h"
#include "resource.h"
#include <qmessagebox.h>
#include <kiconloader.h>
#include <kpixmap.h>
#include <qpaintdevice.h>
#include <kimgio.h>
#include <iostream.h>
#include <qwmatrix.h>

BoardWidget::BoardWidget(BoardOptionsStruct *o, QWidget *parent, const char *name)
	: QWidget(parent,name)
{
	kimgioRegister();

	BoardOptions = o;
	installEventFilter(this);
	CurrentPlayer = 0;	// Player1 poays first
	Players[0] = BoardOptions->Player1;
	Players[1] = BoardOptions->Player2;
	Total = -1;
	Busy = false;
	reloadIcons(CASE_SIZE);
	Table = new InternalPuissance(BoardOptions->NumCols,BoardOptions->NumRows,BoardOptions->NumPuis);
	ButtonGroup = new QButtonGroup(this);
	connect(ButtonGroup,SIGNAL(clicked(int)),SLOT(columnPressed(int)));
	ButtonGroup->setCursor(crossCursor);
	connect(&WinTimer,SIGNAL(timeout()),SLOT(slotTimer()));

	initDisplay();
	resize(BoardOptions->NumCols*CASE_SIZE+20,BoardOptions->NumRows*CASE_SIZE+50);
}

BoardWidget::~BoardWidget()
{
	delete Table;
}

void BoardWidget::resizeEvent(QResizeEvent *e)
{
	int	case_size = QMIN((width()-20)/BoardOptions->NumCols,(height()-50)/BoardOptions->NumRows);
	reloadIcons(case_size);
	int	startx = (width()-10-BoardOptions->NumCols*case_size)/2;
	int	starty = (height()-40-BoardOptions->NumRows*case_size)/2;
	ButtonGroup->setGeometry(startx,starty,case_size*BoardOptions->NumCols+10,40);
	for (int i=0;i<BoardOptions->NumCols;i++) Buttons[i]->setGeometry(5+i*case_size,5,case_size,30);
	Cases->setPixmaps(Pix+1,Pix+2);
	Cases->setGeometry(startx+5,starty+ButtonGroup->height(),case_size*BoardOptions->NumCols,case_size*BoardOptions->NumRows);
}

void BoardWidget::load(const char *filename)
{
	WinTimer.stop();
	emit message(i18n("Loading file..."));
	clearDisplay();
	Table->load(filename);
	BoardOptions->NumCols = Table->numCols();
	BoardOptions->NumRows = Table->numRows();
	BoardOptions->NumPuis = Table->numPuis();
	initDisplay();
	for (int i=0;i<BoardOptions->NumCols;i++) Buttons[i]->show();
	Cases->show();
	resizeEvent(0);
	for (int i=0;i<BoardOptions->NumRows;i++)
	   for (int j=0;j<BoardOptions->NumCols;j++)
		Cases->setState(i,j,Table->state(i,j));
	CurrentPlayer = Table->player();
	NextTurnEvent	*e = new NextTurnEvent;
	QApplication::postEvent(this,e);
}

void BoardWidget::save(const char *filename)
{ Table->save(filename);}

void BoardWidget::dropInColumn(int col, int maxInd, int state)
{
	int	z (col);
	while (z != maxInd) {
		Cases->setState(z/BoardOptions->NumCols,col,state);
		QApplication::flushX(); usleep(20000);
		Cases->setState(z/BoardOptions->NumCols,col,Empty);
		z += BoardOptions->NumCols;
	}
	Cases->setState(maxInd/BoardOptions->NumCols,col,state);
	Total++;
}

void BoardWidget::columnPressed(int col)
{
	if (Busy) return;
	if (CurrentPlayer == -1) {
		QMessageBox::critical(this,i18n("Error"),i18n("Game is finished. Restart first."),QMessageBox::Ok | QMessageBox::Default,0);
		return;
	}
	if (Total == -1) {
		QMessageBox::critical(this,i18n("Error"),i18n("Game is not started. Choose \"New game\" first."),QMessageBox::Ok | QMessageBox::Default,0);
		return;
	}
	int	maxInd = Table->humanPlay(CurrentPlayer+1,col);
	if (maxInd != -1) {
		dropInColumn(col,maxInd,CurrentPlayer+1);
		CurrentPlayer = 1-CurrentPlayer;
		NextTurnEvent	*e = new NextTurnEvent;
		QApplication::postEvent(this,e);
	}
	else QMessageBox::critical(this,i18n("Error"),i18n("Column is full !"),QMessageBox::Ok | QMessageBox::Default,0);
}

bool BoardWidget::eventFilter(QObject *o, QEvent *e)
{
	switch (e->type()) {
	   case Event_NextTurn :
		nextTurn();
		return true;
	}
	return false;
}

void BoardWidget::nextTurn()
{
	// Check for a winner
	int	Previous = (1-CurrentPlayer)+1;
	int	q = Table->check(Previous);
	if (q == Previous) {
		Winner = Table->winner();
		WinTimer.start(500);
		QString		Msg;
		Msg.sprintf(i18n("Player %d won"),q);
		emit message(Msg.data());
		QMessageBox::information(this,i18n("Victory"),Msg.data(),QMessageBox::Ok | QMessageBox::Default,0);
		CurrentPlayer = -1;
		return;
	}

	// Check for status quo
	if (Total == BoardOptions->NumCols*BoardOptions->NumRows) {
		emit message(i18n("Nobody won"));
		QMessageBox::information(this,i18n("Result"),i18n("Nobody won"),QMessageBox::Ok | QMessageBox::Default,0);
		CurrentPlayer = -1;
		return;
	}

	// Process next turn
	if (Players[CurrentPlayer] == Computer) {
		emit message(i18n("Computer thinking..., please wait"));
		Busy = true;
		ButtonGroup->setCursor(waitCursor);
		int	col(-1);
		int	maxInd = Table->computerPlay(CurrentPlayer+1,BoardOptions->Level,col);
		dropInColumn(col,maxInd,CurrentPlayer+1);
		Busy = false;
		ButtonGroup->setCursor(crossCursor);
	}
	else {
		QString		msg;
		msg.sprintf(i18n("Player %d plays"),CurrentPlayer+1);
		emit message(msg.data());
		return;
	}
	CurrentPlayer = 1-CurrentPlayer;
	NextTurnEvent	*e = new NextTurnEvent;
	QApplication::postEvent(this,e);
}

void BoardWidget::restart()
{
	WinTimer.stop();
	if (Table != 0) delete Table;
	CurrentPlayer = 0;	// Player1 poays first
	Players[0] = BoardOptions->Player1;
	Players[1] = BoardOptions->Player2;
	Total = 0;
	Table = new InternalPuissance(BoardOptions->NumCols,BoardOptions->NumRows,BoardOptions->NumPuis);

	Cases->restart();

	// Posting the first event to start the game
	NextTurnEvent	*e = new NextTurnEvent;
	QApplication::postEvent(this,e);
}

void BoardWidget::initDisplay()
{
	for (int i=0;i<BoardOptions->NumCols;i++) {
		QString		numStr;
		numStr.sprintf("%d",i);
		Buttons[i] = new QPushButton(numStr.data(),ButtonGroup);
		Buttons[i]->setFont(QFont("Times",18,QFont::Bold));
	}
	Cases = new CaseWidget(BoardOptions->NumCols,BoardOptions->NumRows,this);
	if (BoardOptions->UseBackgroundPixmap) Cases->setBackgroundPixmap(QPixmap(BoardOptions->BackgroundPixmap.data()));
	else Cases->setBackgroundColor(BoardOptions->BackgroundColor);
}

void BoardWidget::clearDisplay()
{
	for (int i=0;i<BoardOptions->NumCols;i++) delete Buttons[i];
	delete Cases;
}

void BoardWidget::reloadIcons(int size)
{
	QString		rootStr(KApplication::kde_datadir() + "/kpuissance4/pics/");
	QWMatrix	m;
	float		factor = float(size)/CASE_SIZE;
	m.scale(factor,factor);
	KPixmap		pix;
	Pix[0] = pix.xForm(m);
	pix.load(rootStr + "red.xpm",0,KPixmap::LowColor);
	Pix[1] = pix.xForm(m);
	pix.load(rootStr + "yellow.xpm",0,KPixmap::LowColor);
	Pix[2] = pix.xForm(m);
}

void BoardWidget::slotTimer()
{
	int	row(Winner[0]/BoardOptions->NumCols), col(Winner[0]%BoardOptions->NumCols);
	int	winPlayer = Table->state(row,col);
	int	newState = (Cases->state(row,col) != Empty ? Empty : winPlayer);
	for (int i=0;i<BoardOptions->NumPuis;i++) Cases->setState(Winner[i]/BoardOptions->NumCols,Winner[i]%BoardOptions->NumCols,newState);
}

void BoardWidget::undo()
{
	int	index = Table->undo();
	if (index < 0) {
		QMessageBox::critical(this,i18n("Error"),i18n("Can't undo last move"),QMessageBox::Ok | QMessageBox::Default,0);
		return;
	}
	else {
		Cases->setState(index/BoardOptions->NumCols,index%BoardOptions->NumCols,Empty);
		CurrentPlayer = 1-CurrentPlayer;
		// Test if there's one computer player. In such a case
		// what we probably intend to do is undo twice !
		if (Players[CurrentPlayer] == Computer) {
			index = Table->undo();
			if (index >= 0) {
				Cases->setState(index/BoardOptions->NumCols,index%BoardOptions->NumCols,Empty);
				CurrentPlayer = 1-CurrentPlayer;
			}
		}

		NextTurnEvent	*e = new NextTurnEvent;
		QApplication::postEvent(this,e);
	}
}

void BoardWidget::hint()
{
	emit message(i18n("Computer thinking..., please wait"));
	Busy = true;
	ButtonGroup->setCursor(waitCursor);
	int	col(-1);
	Table->computerPlay(CurrentPlayer+1,2,col);	// always use high level
	Table->undo();
	QString	msg;
	msg.sprintf(i18n("Computer would play column n %d"),col);
	QMessageBox::information(this,i18n("Hint"),msg.data(),QMessageBox::Ok | QMessageBox::Default,0);
	Busy = false;
	ButtonGroup->setCursor(crossCursor);

	NextTurnEvent	*e = new NextTurnEvent;
	QApplication::postEvent(this,e);
}
