/***************************************************************************
                          kplayground.cpp  -  description
                             -------------------
    begin                : Fri Dec 10 1999
    copyright            : (C) 1999 by Ralf-Christian Juergensen
    email                : ralf-christian.juergensen@stud.fh-flensburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "kplayground.h"

KPlayGround::KPlayGround(QWidget* parent, const char* name, Player p, QString mapFileName)
 : QWidget(parent, name), PlayGround(p, mapFileName)
{
    loadPixmaps();
    
    fieldWidth = noneEmptyPix.width();
    fieldHeight = noneEmptyPix.height();
    pixelWidth = fieldWidth * getWidth();
    pixelHeight = fieldHeight * getHeight();
    
    // define the keyHandler (cursor, return)
    QAccel* keyHandler = new QAccel(this);
    keyHandler->connectItem(keyHandler->insertItem(Key_Up), this, SLOT(cursorUp()));
    keyHandler->connectItem(keyHandler->insertItem(Key_Down), this, SLOT(cursorDown()));
    keyHandler->connectItem(keyHandler->insertItem(Key_Left), this, SLOT(cursorLeft()));
    keyHandler->connectItem(keyHandler->insertItem(Key_Right), this, SLOT(cursorRight()));
    keyHandler->connectItem(keyHandler->insertItem(Key_Return), this, SLOT(setCursor()));

    currentMapFile = mapFileName;
    	
    partialPaint = false;
    
    setFixedSize(pixelWidth, pixelHeight);
}

KPlayGround::~KPlayGround(void)
{
}

void KPlayGround::paintEvent(QPaintEvent *)
{
    QPainter p;
    p.begin(this);
    
    if(partialPaint)
    {
        paintPartial(&p);
               		
        partialPaint = false;
    }
    else
    {
        paintFields(&p);
        paintCursor(&p);
    }
    
    p.end();
}

void KPlayGround::cursorUp(void)
{
    PlayGround::cursorUp();
    
    partialPaint = true;	
    repaint(false);
}

void KPlayGround::cursorDown(void)
{
    PlayGround::cursorDown();
    	
    partialPaint = true;	
    repaint(false);
}

void KPlayGround::cursorLeft(void)
{
    PlayGround::cursorLeft();
    	
    partialPaint = true;	
    repaint(false);
}

void KPlayGround::cursorRight(void)
{
    PlayGround::cursorRight();
    	
    partialPaint = true;	
    repaint(false);
}

void KPlayGround::setCursor(void)
{
    Player winningPlayer;	// the player who won this game
    	
    PlayGround::setCursor();
    	
    winningPlayer = checkWinner();
    	
    if((winningPlayer == ONE) || (winningPlayer == TWO))
    {
        repaint();
        		
        emit winner(winningPlayer);
        		
        return;
    }
    	
    // prepare for next round
    initRound();
    changePlayer();
    	
    partialPaint = true;	
    repaint(false);
    	
    emit nextRound();
}

void KPlayGround::newGame(Player p)
{
    PlayGround::newGame(p, (const char*)currentMapFile);
    	
    repaint();
    	
    emit nextRound();
}

void KPlayGround::newMap(QString mapFileName)
{
    try
    {
        PlayGround::newGame(PlayGround::getPlayer(), mapFileName);
        
        pixelWidth = fieldWidth * getWidth();	
        pixelHeight = fieldHeight * getHeight();
        
        currentMapFile = mapFileName;
        
        setFixedSize(pixelWidth, pixelHeight);
    }
    catch(char* mapError)
    {
        QString error;
        		
        error.sprintf(i18n("Error in File %s: \n%s"), (const char*)mapFileName, mapError);
        
        QMessageBox::warning(0,
                             i18n("Map error"),
                             error);
    }
    
    repaint();
    
    emit nextRound();
}

int KPlayGround::getPixWidth(void)
{
    return pixelWidth;
}

int KPlayGround::getPixHeight(void)
{
    return pixelHeight;
}

void KPlayGround::paintFields(QPainter* p)
{
    QPixmap ground(pixelWidth, pixelHeight);
    	
    QPainter temp;
    	
    temp.begin(&ground);
    	
    // paint fields
    	
    for(int x=0; x < getWidth(); x++)
    {
        for(int y=0; y < getHeight(); y++)
        {
            paintField(&temp, x, y);
        }
    }
    	
    temp.end();
    	
    p->drawPixmap(0, 0, ground);
}

/** paints a single field of a cursor. */
void KPlayGround::paintCursorField(QPainter* p, int x, int y, int bitNo)
{
    if(getCursorBit(bitNo) == true)
        p->drawPixmap(x, y, cursorOnPix);
    else
        p->drawPixmap(x, y, cursorOffPix);
}

void KPlayGround::paintCursor(QPainter* p)
{
    	
    int cursorX = getCursorX() * fieldWidth;			// current x-position of the cursor in pixels
    int cursorY = getCursorY() * fieldHeight;			// current y-position of the cursor in pixels
    KPixmap* pix = 0;
    
    paintCursorField(p, cursorX                 , cursorY                  , 0);
    paintCursorField(p, cursorX +     fieldWidth, cursorY                  , 1);
    paintCursorField(p, cursorX + 2 * fieldWidth, cursorY                  , 2);
    paintCursorField(p, cursorX                 , cursorY +     fieldHeight, 3);
    paintCursorField(p, cursorX + 2 * fieldWidth, cursorY +     fieldHeight, 4);
    paintCursorField(p, cursorX                 , cursorY + 2 * fieldHeight, 5);
    paintCursorField(p, cursorX +     fieldWidth, cursorY + 2 * fieldHeight, 6);
    paintCursorField(p, cursorX + 2 * fieldWidth, cursorY + 2 * fieldHeight, 7);
    
    switch(getCursorType())
    {
    case AND:   pix = &cursorAndPix;
                break;
    case OR:    pix = &cursorOrPix;
                break;
    case XOR:   pix = &cursorXorPix;
                break;
    case RND:   pix = &cursorRndPix;
                break;
    }
    p->drawPixmap(cursorX + fieldWidth, cursorY + fieldHeight, *pix);
}

void KPlayGround::paintPartial(QPainter* p)
{
    int cursorX = getCursorX();
    int cursorY = getCursorY();
    	
    if(cursorX > 0)
    {
        paintField(p, cursorX - 1, cursorY    );
        paintField(p, cursorX - 1, cursorY + 1);
        paintField(p, cursorX - 1, cursorY + 2);
    }
    	
    if(cursorX < (getWidth() - 3))
    {
        paintField(p, cursorX + 3, cursorY    );
        paintField(p, cursorX + 3, cursorY + 1);
        paintField(p, cursorX + 3, cursorY + 2);
    }
    	
    if(cursorY > 0)
    {
        paintField(p, cursorX    , cursorY - 1);
        paintField(p, cursorX + 1, cursorY - 1);
        paintField(p, cursorX + 2, cursorY - 1);
    }
    	
    if(cursorY < (getHeight() - 3 ))
    {
        paintField(p, cursorX    , cursorY + 3);
        paintField(p, cursorX + 1, cursorY + 3);
        paintField(p, cursorX + 2, cursorY + 3);
    }
    	
    paintCursor(p);
}

void KPlayGround::paintField(QPainter* p, int x, int y)
{
    int xPos = x * fieldWidth;
    int yPos = y * fieldHeight;
    KPixmap* pix = 0;
    Field field = getField(x, y);
    
    switch(field.getValue())
    {
    case ON:
        switch(field.getPlayer())
        {
        case ONE:
            pix = &oneOnPix;
            break;
        case TWO:
            pix = &twoOnPix;
            break;
        case NONE:
            pix = &noneOnPix;
            break;
        }
        break;
    case OFF:
        switch(field.getPlayer())
        {
        case ONE:
            pix = &oneOffPix;
            break;
        case TWO:
            pix = &twoOffPix;
            break;
        case NONE:
            pix = &noneOffPix;
            break;
        }
        break;
    case EMPTY:
        switch(field.getPlayer())
        {
        case ONE:
            pix = &oneEmptyPix;
            break;
        case TWO:
            pix = &twoEmptyPix;
            break;
        case NONE:
            pix = &noneEmptyPix;
            break;
        }
        break;
    case PENGUIN:
        switch(field.getPlayer())
        {
        case ONE:
            pix = &onePenguinPix;
            break;
        case TWO:
            pix = &twoPenguinPix;
            break;
        case NONE:
            pix = &nonePenguinPix;
            break;
        }
        break;
    }
    
    p->drawPixmap(xPos, yPos, *pix);		
}

/** load all pixmaps needed */
void KPlayGround::loadPixmaps()
{
    cursorOnPix.load(QString(DIGIT_PIXMAP_DIR) + "cursor_on.xpm");
    cursorOffPix.load(QString(DIGIT_PIXMAP_DIR) + "cursor_off.xpm");
    cursorAndPix.load(QString(DIGIT_PIXMAP_DIR) + "cursor_and.xpm");
    cursorOrPix.load(QString(DIGIT_PIXMAP_DIR) + "cursor_or.xpm");
    cursorXorPix.load(QString(DIGIT_PIXMAP_DIR) + "cursor_xor.xpm");
    cursorRndPix.load(QString(DIGIT_PIXMAP_DIR) + "cursor_rnd.xpm");
    
    oneOnPix.load(QString(DIGIT_PIXMAP_DIR) + "one_on.xpm");
    oneOffPix.load(QString(DIGIT_PIXMAP_DIR) + "one_off.xpm");
    oneEmptyPix.load(QString(DIGIT_PIXMAP_DIR) + "one_empty.xpm");
    onePenguinPix.load(QString(DIGIT_PIXMAP_DIR) + "one_pin.xpm");
    
    twoOnPix.load(QString(DIGIT_PIXMAP_DIR) + "two_on.xpm");
    twoOffPix.load(QString(DIGIT_PIXMAP_DIR) + "two_off.xpm");
    twoEmptyPix.load(QString(DIGIT_PIXMAP_DIR) + "two_empty.xpm");
    twoPenguinPix.load(QString(DIGIT_PIXMAP_DIR) + "two_pin.xpm");
    
    noneOnPix.load(QString(DIGIT_PIXMAP_DIR) + "none_on.xpm");
    noneOffPix.load(QString(DIGIT_PIXMAP_DIR) + "none_off.xpm");
    noneEmptyPix.load(QString(DIGIT_PIXMAP_DIR) + "none_empty.xpm");
    nonePenguinPix.load(QString(DIGIT_PIXMAP_DIR) + "none_pin.xpm");
}