#include "GameViewWidget.h"
#include "GameViewWidget.moc"
#include <cstdio>
#include <qkeycode.h>
#include <qmsgbox.h>
#include <kapp.h>

#include "QtDraw3D.h"

int GameViewWidget::ActionKeys[GameViewWidget::KEY_NOT_USED]={
	Key_Left,Key_Right,
	Key_Up,Key_Down,
	Key_5,Key_Space,
	Key_Q,Key_W,
	Key_A,Key_S,
	Key_Y,Key_X,
	Key_P};

GameViewWidget::GameViewWidget (QWidget *parent, const char *name) : QWidget (parent, name)
{
	needsRecalc=true;

	performGamePlay=0;
	
	setFocusPolicy (QWidget::StrongFocus);
	setFocus();
	setBackgroundColor (black);

	baseTimeLevel=1;
	
	loweringTimer=new QTimer();
	connect (loweringTimer, SIGNAL(timeout()), SLOT(lowerFigure()));

	levelIncreaseTimer=new QTimer();
	connect (levelIncreaseTimer, SIGNAL(timeout()), SLOT(increaseTimeLevel()));
	
	fastLoweringTimer=new QTimer();
	connect (fastLoweringTimer, SIGNAL(timeout()), SLOT(fastLoweringFigure()));
}

GameViewWidget::~GameViewWidget()
{
	delete loweringTimer;
	delete levelIncreaseTimer;
	delete fastLoweringTimer;
}

void GameViewWidget::newGame()
{
	performGamePlay->newGame();

	needsRecalc=true;

	performGamePlay->setTimeLevel (baseTimeLevel);

	if (loweringTimer->isActive())
		loweringTimer->stop();
	loweringTimer->start (performGamePlay->getDelayTime(), TRUE);
	
	if (levelIncreaseTimer->isActive())
		levelIncreaseTimer->stop();
	levelIncreaseTimer->start (60000);

	repaint (TRUE);

	emit updatePoints (performGamePlay->getPoints());
	emit newLevel (performGamePlay->getTimeLevel());
}

void GameViewWidget::setPerformGamePlay (PerformGamePlay *_performGamePlay)
{
	performGamePlay=_performGamePlay;
}

bool GameViewWidget::lowerFigure()
{
	bool lowered=false;
	
	loweringTimer->stop();		// we have moved one level
	
	if (!performGamePlay->lowerFigureOneLevel())
	{
		lowered=true;
		emit updatePoints (performGamePlay->getPoints());
	}
	

	needsRecalc=true;
	repaint (FALSE);

	if (performGamePlay->hasGameEnded())
		gameEnds();
	else
		loweringTimer->start (performGamePlay->getDelayTime(), TRUE); // same time for moving every level

	return lowered;
}

void GameViewWidget::fastLoweringFigure()
{
	if (lowerFigure())		// have reached bottom
		fastLoweringTimer->stop();
}

void GameViewWidget::gameEnds()
{
	loweringTimer->stop();
	
	levelIncreaseTimer->stop();
	loweringTimer->stop();
	fastLoweringTimer->stop();
	
	emit gameEnded();		// calls modal dialog Box, but disables not timer
}

void GameViewWidget::increaseTimeLevel()
{
	performGamePlay->setTimeLevel (performGamePlay->getTimeLevel()+1);
	emit newLevel (performGamePlay->getTimeLevel());
}


void GameViewWidget::keyPressEvent (QKeyEvent *e)
{
	if (performGamePlay->hasGameEnded())
	{
		e->ignore();
		return;
	}

	ActionKeysEnumerate key=KEY_MOVE_LEFT; // check for pressed key (user configurable)
	while (ActionKeys [key] != e->key() && key != KEY_NOT_USED)
		key=(ActionKeysEnumerate) (((int) key)+1);

	switch (key)
	{
		case KEY_MOVE_LEFT:
		     if (!performGamePlay->moveRel (Position (-1, 0, 0)))
			     return;

		     break;

		case KEY_MOVE_RIGHT:
		     if (!performGamePlay->moveRel (Position (1, 0, 0)))
			     return;

		     break;

		case KEY_MOVE_UP:
		     if (!performGamePlay->moveRel (Position (0, 1, 0)))
			     return;
		     
		     break;

		case KEY_MOVE_DOWN:
		     if (!performGamePlay->moveRel (Position (0, -1, 0)))
			     return;
		     
		     break;

		case KEY_ROTATE_NEG_X:
		     if (!performGamePlay->rotateX (-1))
			     return;
		     
		     break;

		case KEY_ROTATE_POS_X:
		     if (!performGamePlay->rotateX (1))
			     return;
		     
		     break;

		case KEY_ROTATE_NEG_Y:
		     if (!performGamePlay->rotateY (-1))
			     return;

		     break;

		case KEY_ROTATE_POS_Y:
		     if (!performGamePlay->rotateY (1))
			     return;
		     
		     break;

		case KEY_ROTATE_NEG_Z:
		     if (!performGamePlay->rotateZ (-1))
			     return;
		     
		     break;

		case KEY_ROTATE_POS_Z:
		     if (!performGamePlay->rotateZ (1))
			     return;
		     
		     break;

		case KEY_MOVE_TO_BOTTOM:
		     lowerFigure();
		     return;

		case KEY_MOVE_FAST_TO_BOTTOM:
		     fastLoweringTimer->start (100);
		     return;

		case KEY_PAUSE:
		     loweringTimer->stop();
		     fastLoweringTimer->stop();
		     QMessageBox::information (this, "BlockOut", kapp->klocale->translate ("Game paused"));
		     loweringTimer->start (performGamePlay->getDelayTime(), TRUE); // same time for moving every level
		     return;
		     
		case KEY_NOT_USED:
		default:
		     e->ignore();
		     return;
	}

	needsRecalc=true;
	repaint (FALSE);
}

void GameViewWidget::recalcDisplay()
{

	Draw3D::getGlobalInstance()->startDrawing();
	Draw3D::getGlobalInstance()->clear();

	if (performGamePlay)
		performGamePlay->draw();

	Draw3D::getGlobalInstance()->endDrawing();

	needsRecalc=false;
}

void GameViewWidget::paintEvent (QPaintEvent *e)
{
	if (needsRecalc)		// first display?
		recalcDisplay();
	
	Draw3D::getGlobalInstance()->updateDisplay (e->rect().left(), e->rect().top(), e->rect().width(), e->rect().height());
}

void GameViewWidget::resizeEvent (QResizeEvent *)
{
	if (performGamePlay)
	{
		needsRecalc=true;	// redraw lines
		((QtDraw3D *) Draw3D::getGlobalInstance())->updateSize();
	}
}


// Local Variables:
// compile-command: "make GameViewWidget.o"
// End:
