/* This may look like C, but it's really -*- C++ -*- */

/*
    kdb - A Simple but fast database editor for the KDE
    Copyright (C) 1998  Thorsten Niedzwetzki

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "dbviewer.h"
#include "dbviewer.moc"

/*
 * List of all currently active main windows. The Program will
 * terminate itself if the user closes the very last visible
 * main window.
 */
QList<TDBViewer> TDBViewer::windowlist;


// --- constructor -----------------------------------------------


/*
 * The constructor sets object attributes to given arguments or
 * default values. Default values fit well for a new database.
 */
TDBViewer::TDBViewer(char* name, TCharSepDB* db, char* fileName)
  : KTopLevelWidget(name)
{
  this->db = db;
  if (fileName)
    this->fileName = strdup(fileName);
  else
    this->fileName = NULL;
  setupWindow();
  if (db == NULL) {
    lb->hide();
  } else {
    updateView();
  }
  setEditCommands();
}


/*
 * Set up menu, toolbar, status bar and main list box.
 */
void
TDBViewer::setupWindow()
{
  windowlist.append(this);

  KConfig* kcfg = KApplication::getKApplication()->getConfig();
  KStdAccel *keys = new KStdAccel(kcfg);
  
  modified = false;

  readConfigurationFile();

  file = new QPopupMenu;     CHECK_PTR(file);
  help = KApplication::getKApplication()->getHelpMenu(true,
    "\n" APPNAME " " VERSION "\n\n" COPYRIGHT_STRING "\n");
  edit = new QPopupMenu;     CHECK_PTR(edit);
  options = new QPopupMenu;  CHECK_PTR(options);
  menu = new KMenuBar(this); CHECK_PTR(menu);

  KWM::setIcon(winId(), kapp->getIcon());
  KWM::setMiniIcon(winId(), kapp->getMiniIcon());

  // --- File menu -----------------------------------------------

  file->insertItem(i18n("&New..."),
		   this, SLOT (fileNew()), keys->openNew());
  file->insertItem(i18n("&Open..."),
		   this, SLOT (fileOpen()), keys->open());
  file->insertItem(i18n("&Save"),
		   this, SLOT (fileSave()), keys->save());
  file->insertItem(i18n("S&ave as..."),
		   this, SLOT (fileSaveAs()) );
  file->insertItem(i18n("&Close"),
		   this, SLOT (fileClose()), keys->close() );
  file->insertSeparator();
  file->insertItem(i18n("&Print..."),
		   this, SLOT (filePrint()), keys->print() );
  file->insertSeparator();
  file->insertItem(i18n("&Quit"),
		   this, SLOT (fileQuit()), keys->quit());

  // --- Edit menu -----------------------------------------------

  edit->insertItem(i18n("&Header..."),
		   this, SLOT (editHeader()) );
  edit->insertSeparator();
  edit->insertItem(i18n("&New..."),
		   this, SLOT (editNew()) );
  edit->insertItem(i18n("&Edit..."),
		   this, SLOT (editEdit()) );
  edit->insertItem(i18n("&Find..."),
		   this, SLOT (editFind()), keys->find());
  edit->insertSeparator();
  edit->insertItem(i18n("&Copy"),
		   this, SLOT (editCopy()), keys->copy());
  edit->insertItem(i18n("&Paste"),
		   this, SLOT (editPaste()), keys->paste());
  edit->insertItem(i18n("C&ut"),
		   this, SLOT (editCut()), keys->cut() );
  edit->insertItem(i18n("&Delete"),
		   this, SLOT (editDelete()) );

  // --- Options menu --------------------------------------------

  options->setCheckable(true);
  options->insertItem(i18n("&File format..."),
		      this, SLOT (optionsFileFormat()) );
  options->insertItem(i18n("File contains &header"),
		      this, SLOT (optionsFileContainsHeader()) );
  options->insertSeparator();
  options->insertItem(i18n("&Confirm actions"),
		      this, SLOT (optionsConfirmActions()) );
  options->insertItem(i18n("&Always save"),
		      this, SLOT (optionsAlwaysSave()) );

  options->setItemChecked(options->idAt(1), fileContainsHeader);
  options->setItemChecked(options->idAt(3), confirmActions);
  options->setItemChecked(options->idAt(4), alwaysSave);

  // --- main menu -----------------------------------------------

  menu->insertItem(i18n("&File"), file);
  menu->insertItem(i18n("&Edit"), edit, MENU_EDIT);
  menu->insertItem(i18n("&Options"), options);
  menu->insertSeparator();
  menu->insertItem(i18n("&Help"), help);
  setMenu(menu);
	
  // --- tool bar ------------------------------------------------

  KIconLoader *loader = kapp->getIconLoader();
  toolbar = new KToolBar(this);
  toolbar->insertButton(loader->loadIcon("filenew.xpm"),
			BTN_FILE_NEW, SIGNAL(clicked(int)), this,
			SLOT(fileNew()), TRUE, 
			i18n("Create a new file"));
  toolbar->insertButton(loader->loadIcon("fileopen.xpm"),
			BTN_FILE_OPEN, SIGNAL(clicked(int)), this,
			SLOT(fileOpen()), TRUE, 
			i18n("Open a file"));
  toolbar->insertButton(loader->loadIcon("filefloppy.xpm"),
			BTN_FILE_SAVE, SIGNAL(clicked(int)), this,
			SLOT(fileSave()), TRUE,
			i18n("Save the file"));
  addToolBar(toolbar);
  toolbar->show();

  // --- status bar ----------------------------------------------

  stat = new KStatusBar(this);
  stat->insertItem(i18n("Empty table"),
		   STAT_LEFT);
  stat->insertItem(i18n("You may open a file now"),
		   STAT_RIGHT);
  setStatusBar(stat);
  stat->show();

  // --- main window ---------------------------------------------

  lb = new KTabListBox(this, "lb", 1);
  QObject::connect(lb, SIGNAL( highlighted(int, int) ),
		   SLOT( itemHighlighted(int, int) ));
  QObject::connect(lb, SIGNAL( selected(int, int) ),
		   SLOT( itemSelected(int, int) ));

  setView(lb);
  
  // --- tidy up -------------------------------------------------

  delete keys;
}


// --- destructor ------------------------------------------------


/*
 * The Destructor removes all main window data.
 */
TDBViewer::~TDBViewer()
{
  if (toolbar) delete toolbar;
  if (menu) delete menu;
  if (stat) delete stat;
  if (fileName) delete[] fileName;
  if (db) delete db;
}


// --- public member functions -----------------------------------


/*
 * This function should be called whenever data have been modified.
 * If the alwaysSave flag is set, data will be saved.
 */
void
TDBViewer::setModified()
{
  modified = true;
  if (alwaysSave) {
    fileSave();
  }
}


// --- configuration file ----------------------------------------


/*
 * Read configuration file and set flags accordingly.
 */
void
TDBViewer::readConfigurationFile()
{
  KConfig* k = KApplication::getKApplication()->getConfig();

  k->setGroup("settings");

  confirmActions = k->readBoolEntry("confirmActions", true);
  alwaysSave = k->readBoolEntry("alwaysSave", false);
  fileContainsHeader = k->readBoolEntry("fileContainsHeader", true);
  separator = k->readNumEntry("separator", DEFAULT_SEPARATOR);
}


/*
 * Save flags to configuration file.
 */
void
TDBViewer::saveConfig()
{
  KConfig* k = KApplication::getKApplication()->getConfig();

  MESSAGE("saving config");

  k->setGroup("settings");

  k->writeEntry("confirmActions", confirmActions);
  k->writeEntry("alwaysSave", alwaysSave);
  k->writeEntry("fileContainsHeader", fileContainsHeader);
  k->writeEntry("separator", separator);

  k->sync();
}


// --- other private member functions ----------------------------


/*
 * Redraws the status bar displaying the currently selected record
 * and the active database file name.
 */
void
TDBViewer::updateStatusBar(unsigned int selectedRecord)
{
  char recStr[128];

  if (db->getRecordCount() > 0)
    sprintf(recStr, "%u/%u%s", selectedRecord + 1,
	    db->getRecordCount(), modified ? "*" : "");
  else
    strcpy(recStr, i18n("Empty table"));
  stat->changeItem(recStr, STAT_LEFT);
  if (fileName)
    stat->changeItem(fileName, STAT_RIGHT);
  else
    stat->changeItem("New table", STAT_RIGHT);
}


/*
 * Updates listbox from database and redraws listbox.
 */
void
TDBViewer::updateView()
{
  if (fileName) {
    ostrstream caption;
    caption << APPNAME " " VERSION " - " << fileName << ends;
    setCaption(caption.str());
  } else
    setCaption(APPNAME " " VERSION);

  // update list box
  lb->setAutoUpdate(false);
  lb->clear();

  // create list box header
  lb->setNumCols(db->getFieldCount());
  for (uint i = 0; i < db->getFieldCount(); i++)
    lb->setColumn(i, strdup(db->getFieldName(i)), 80);
  
  // write list box data
  for (uint rec = 0; rec < db->getRecordCount(); rec++) {
    lb->appendItem(0);
    for (uint fld = 0; fld < db->getFieldCount(); fld++)
      lb->changeItemPart(db->getItemAsText(rec, fld), rec, fld);
  }
  lb->setAutoUpdate(true);
  lb->show();
  lb->repaint();
  //  lb->setCurrentItem(lb->currentItem()); // highlight current item
  
  // prepare status bar and menu bar
  updateStatusBar();
  setEditCommands();
}


/*
 * This enables or disables certain commands like File/Save and
 * Edit/Delete.
 */
void
TDBViewer::setEditCommands()
{
  /* File/Save, File/Save as, File/Print, whole Edit menu */
  file->setItemEnabled(file->idAt(2), db != NULL);
  file->setItemEnabled(file->idAt(3), db != NULL);
  file->setItemEnabled(file->idAt(6), db != NULL);
  menu->setItemEnabled(MENU_EDIT, db != NULL);

  if (db != NULL) {
    bool enable = db->getRecordCount() > 0;
    edit->setItemEnabled(edit->idAt(1), enable);
    edit->setItemEnabled(edit->idAt(3), enable);
    edit->setItemEnabled(edit->idAt(4), enable);
    edit->setItemEnabled(edit->idAt(6), enable);
    edit->setItemEnabled(edit->idAt(8), enable);
    edit->setItemEnabled(edit->idAt(9), enable);
  }
}

// --- File menu -------------------------------------------------


/*
 * Display a NewTable-Dialog and create a new window.
 */
void
TDBViewer::fileNew() {
  TNewTableDialog *newTableDialog =
    new TNewTableDialog(this, "newtable_dlg", true);

  newTableDialog->setCaption("New Table");
  newTableDialog->show();

  if (!newTableDialog->result()) return;  // cancelled

  TCharSepDB *newDatabase = new TCharSepDB(separator);
  newDatabase->setFieldNames(newTableDialog->strings());

  if (lb->isVisible()) {

    // this window is already used, so create another
    TDBViewer *dbviewer =
      new TDBViewer("dbviewer", newDatabase);
    dbviewer->setModified();
    dbviewer->show();
  } else {

    // use of this previously unused window
    db = newDatabase;
    updateView();
  }
  delete newTableDialog;
}


/*
 * Display a FileOpen-Dialog and create a new database window.
 */
void
TDBViewer::fileOpen()
{
  KFileDialog *fileDialog =
    new KFileDialog(0, "*." DEFAULT_EXTENSION, this, "openFile", true);
  fileDialog->setCaption(i18n("Open file"));
  
  if (fileDialog->exec() == 1) {
    setCursor(waitCursor);

    TCharSepDB *newDatabase = new TCharSepDB(separator);
    newDatabase->setContainsHeader(fileContainsHeader);
    if (newDatabase->loadFromFile(fileDialog->selectedFile().data())) {

      if (lb->isVisible()) {
	
	// this window is already used, so create another
	TDBViewer *dbviewer =
	  new TDBViewer("dbviewer", newDatabase,
			fileDialog->selectedFile().data());
	dbviewer->show();

      } else {

	// use of this previously unused window
	db = newDatabase;
	fileName = strdup(fileDialog->selectedFile().data());
	updateView();

      }
    } else {
      QMessageBox::critical(0, i18n("File read error"),
			    i18n("Could not read file"));
    }
    setCursor(arrowCursor);
  }
  delete fileDialog;
}


/*
 * Save file with old name.
 */
void
TDBViewer::fileSave()
{
  if ( confirmActions && (fileContainsHeader == false)) {
    int r = KMsgBox::yesNoCancel
      (this, i18n("Confirmation"),
       i18n("Header won't be saved. Are you sure?"));
    if (r != 1)
      return;
  }

  if (fileName) {
    if (db->writeToFile(fileName)) {
      modified = false;
      updateStatusBar();
    } else {
      const char *msgText = i18n("Error writing file");
      char *msg = new char[strlen(fileName) + strlen(msgText) + 2];
      strcpy(msg, msgText);
      strcat(msg, " ");
      strcat(msg, fileName);
      QMessageBox::warning(0, APPNAME, msg);
      delete[] msg;
    }   
  } else
    fileSaveAs();
}


/*
 * Save file with new name.
 */
void
TDBViewer::fileSaveAs() {
  KFileDialog *fileDialog =
    new KFileDialog(0, "*." DEFAULT_EXTENSION, this, "saveFile", true);
  fileDialog->setCaption(i18n("Save file"));
  if (fileDialog->exec() == 1) {
    setCursor(waitCursor);

    if (db->writeToFile(fileDialog->selectedFile().data())) {
      modified = false;
      if (fileName) delete[] fileName;
      fileName = strdup(fileDialog->selectedFile().data());
    } else {
      const char *msgText = i18n("Error writing file");
      char *msg = new char[strlen(fileDialog->selectedFile().data())
	    + strlen(msgText) + 2];
      strcpy(msg, msgText);
      strcat(msg, " ");
      strcat(msg, fileDialog->selectedFile().data());
      QMessageBox::warning(0, APPNAME, msg);
      delete[] msg;
    }

   setCursor(arrowCursor);
  }
  delete fileDialog;
}


/*
 *
 */
void
TDBViewer::fileClose()
{
  emit close();
}


/*
 *
 */
void
TDBViewer::filePrint()
{
    QMessageBox::critical
      (0, i18n("Not yet implemented"),
       i18n("This feature is not yet implemented."));
}


/*
 * Quit application. This closes all currently open windows.
 */
void
TDBViewer::fileQuit()
{
  emit closeEvent(new QCloseEvent());
}


// --- Edit menu -------------------------------------------------


/*
 * Edit the field names.
 */
void
TDBViewer::editHeader()
{
  QStrList *header = db->getFields();
  QStrList *labels = new QStrList();
  for (unsigned int i = 0; i < db->getFieldCount(); i++) {
    ostrstream format;
    format << i18n("Field #") << i + 1 << ends;
    labels->append(format.str());
  }

  TEditRecDialog *editDlg =
    new TEditRecDialog(labels, header, this, "edit_header_dlg");

  if (editDlg->exec()) {
    db->setFieldNames(editDlg->strings());
    setModified();
    emit lb->repaint();
    updateStatusBar();
  }
}


/*
 * Append a new record.
 */
void
TDBViewer::editNew()
{
  QStrList *newRecord = new QStrList();
  for (unsigned int i = 0; i < db->getFieldCount(); i++)
    newRecord->append("");

  TEditRecDialog *editDlg =
    new TEditRecDialog(db->getFields(),
		       newRecord,
		       this, "append_dlg");
  editDlg->setCaption(i18n("Append record"));
  if (editDlg->exec()) {
    db->appendRecordAsText(editDlg->strings());
    setModified();
    updateView();
  }
}


/*
 *
 */
void
TDBViewer::editEdit()
{
  TEditRecDialog *editDlg =
    new TEditRecDialog(db->getFields(),
		       db->getRecord(lb->currentItem()),
		       this, "edit_dlg");
  editDlg->setCaption(i18n("Edit record"));
  if (editDlg->exec()) {
    db->replaceRecordAsText(lb->currentItem(), editDlg->strings());
    setModified();
    updateView();
  }
}


/*
 *
 */
void
TDBViewer::editFind()
{
  TFindDialog *findDlg =
    new TFindDialog(this, "findDialog", true);
  findDlg->setCaption(i18n("Find"));

  if (findDlg->exec()) {
    QMessageBox::critical
      (0, i18n("Not yet implemented"),
       i18n("This feature is not yet implemented."));
  }
  delete findDlg;
}


/*
 *
 */
void
TDBViewer::editCopy()
{
}


/*
 *
 */
void
TDBViewer::editPaste()
{
  //  setModified();
}


/*
 * Same as editCopy() and editDelete().
 */
void
TDBViewer::editCut()
{
  //  setModified();
}


/*
 *
 */
void
TDBViewer::editDelete()
{
  int r;

  if (confirmActions)
    r = KMsgBox::yesNoCancel(this, i18n("Confirmation"), i18n(
      "Do you really want to remove the selected record?"));
  else
    r = 1; // Yes by default

  if ( (r == 1) && (lb->currentItem() >= 0) ) {
    db->removeRecord(lb->currentItem());
    lb->removeItem(lb->currentItem());
    lb->repaint();
    setEditCommands();
    setModified();
    updateStatusBar();
  }
}


// --- Options menu ----------------------------------------------


/*
 *
 */
void
TDBViewer::optionsFileFormat()
{
  TFileFormatDialog *fmtDlg =
    new TFileFormatDialog(this, "fileFormatDialog", true);
  fmtDlg->setCaption(i18n("File Format"));

  if (fmtDlg->exec()) {
    QMessageBox::critical
      (0, i18n("Not yet implemented"),
       i18n("This feature is not yet implemented."));
  }
  delete fmtDlg;
}


/*
 *
 */
void
TDBViewer::optionsFileContainsHeader()
{
  fileContainsHeader = !fileContainsHeader;
  options->setItemChecked(options->idAt(1), fileContainsHeader);
}


/*
 *
 */
void
TDBViewer::optionsConfirmActions()
{
  confirmActions = !confirmActions;
  options->setItemChecked(options->idAt(3), confirmActions);
}


/*
 *
 */
void
TDBViewer::optionsAlwaysSave()
{
  alwaysSave = !alwaysSave;
  options->setItemChecked(options->idAt(4), alwaysSave);
}


// --- other slots -----------------------------------------------


/*
 *
 */
void
TDBViewer::itemSelected(int, int)
{
  editEdit();
}


/*
 *
 */
void
TDBViewer::itemHighlighted(int index, int)
{
  updateStatusBar(index);
}


// --- events ----------------------------------------------------


/*
 *
 */
void
TDBViewer::paintEvent(QPaintEvent*)
{
}  


/*
 *
 */
void
TDBViewer::resizeEvent(QResizeEvent*)
{
  menu->resize(width(), menu->height());
  toolbar->resize(width(), toolbar->height());
  stat ->setGeometry(stat->x(), height() - stat->height(),
		     width(), stat->height());
  lb->resize(width(), height()
	     - menu->height() - stat->height() - toolbar->height());
}


/*
 *
 */
void
TDBViewer::closeEvent(QCloseEvent* e)
{
  bool cancel = false;

  if (modified) {
    int r = KMsgBox::yesNoCancel(this, i18n("Confirmation"), i18n(
      "There are unsaved changes.\nDo you want to save your changes?"));
    switch (r) {
    case 1: // Yes
      fileSave();
      cancel = (modified == false);
      break;
    case 2: // No
      break;
    case 3: // Cancel
      cancel = true;
      break;
    }
  }

  if (confirmActions && (cancel == false) && (windowlist.count() == 1) ) {
    int r = KMsgBox::yesNoCancel(this, i18n("Confirmation"), i18n(
      "Do you really want to exit?"));
    cancel = (r != 1);
  }

  if (cancel == false) {
    MESSAGE("confirming quit in closeEvent()");

    e->accept(); // confirm quit
    windowlist.remove(this);
    delete this;
    if (windowlist.isEmpty()) {
      saveConfig(); // save configuration
      MESSAGE("terminating in closeEvent()");
      emit kapp->quit();
      MESSAGE("Whopa!");
    }
  }
  else
    e->ignore();
}
