/*
    File: smif.cpp
    Version: 0.3.1
    Date: 31-DEC-1998
    Copyright: Bernhard Kuhn <kuhn@lpr.e-technik.tu-muenchen.de>

    The Serial Monitor InterFace (SMIF) provides an OO-access
    to the procedural serial monitor utility "sermon-0.23" written
    by Peter Fox <fox@roestock.demon.co.uk>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/



// get smif class definitions

#include <qmsgbox.h> 
#include "smif.h"

#include <iostream.h>


// constructor

smif::smif(char* device=0,int interval=100,int tau=500) :
  interval(interval),tau(tau),fd(0),
  rxf_rate(0),txf_rate(0),timerID(0) {

  // set device, if it exists
  if(device) setDevice(device);

};



// destructor

smif::~smif() {
  if(fd>0) close(fd);
};



// change input device

void smif::setDevice(const char* newDevice) {
  device=newDevice;

  // reset

  emit  RXstatusChanged(rx =0);
  emit  TXstatusChanged(tx =0);
  emit RTSstatusChanged(rts=0);
  emit CTSstatusChanged(cts=0);
  emit DTRstatusChanged(dtr=0);
  emit DSRstatusChanged(cts=0);
  emit DCDstatusChanged(dcd=0);
  emit  RIstatusChanged(ri =0);
  emit RXrateChanged(rx_rate=0);
  emit TXrateChanged(tx_rate=0);
  emit RXcountChanged(rx_count=0);
  emit TXcountChanged(tx_count=0);

  if(fd>0) close(fd);
  if(timerID) killTimer(timerID);

  // open device

  if((fd=open(device,O_NONBLOCK | O_RDONLY))<0) {
    QMessageBox::warning(0,"KSermon",
			 "Error while opening device" + device);
    return;
  };

  // get info structure for rx and tx

  if(ioctl(fd,TIOCSERGSTATS,&info)<0) {
    QMessageBox::warning(0,"KSermon","Error while TIOCSERGSTATS");
    return;
  };

  // get status byte for rts,cts,dtr,dsr,dcd & ri

  if(ioctl(fd,TIOCMGET,&status)<0) {
    QMessageBox::warning(0,"KSermon","Error while TIOCMGET");
    return;
  };

  // Remove DTR which was set by opening the line
  // But only if not yet opened by another process and
  // only if Carrier not detected, don't care if this works
  if(info.count==1) {
    if(!(status & TIOCM_CAR)) {
      status &= ~TIOCM_DTR;
      if(ioctl(fd,TIOCMSET,&status)<0) {
	QMessageBox::warning(0,"KSermon","Error while TIOCMSET");
	return;
      };
    };
  };

  // check line states

  emit RTSstatusChanged(rts=((status&TIOCM_RTS)!=0));
  emit CTSstatusChanged(cts=((status&TIOCM_CTS)!=0));
  emit DTRstatusChanged(dtr=((status&TIOCM_DTR)!=0));
  emit DSRstatusChanged(cts=((status&TIOCM_DSR)!=0));
  emit DCDstatusChanged(dcd=((status&TIOCM_CAR)!=0));
  emit  RIstatusChanged(ri =((status&TIOCM_RNG)!=0));
  emit RXcountChanged(rx_count=info.rx);
  emit TXcountChanged(tx_count=info.tx);

  // finaly start the interval timer for automatic data updates

  timerID=startTimer(interval);
  

  // also start the callibration timer
  // (object interval timers are not exact enough to be sure
  // that the given interval was ok)

  timer.start();

};



// change update interval
void smif::setInterval(int newInterval) {
  if(fd>=0) {
    interval=newInterval;
    if(interval<10) interval=10;

    killTimer(timerID);
    timerID=startTimer(interval);
  };
};



// change tau (RC-lowpass timing constant)
void smif::setTau(int newTau) {
  tau=newTau;
  if(tau<10) tau=10;
};



// get line status
bool smif::getRX()  {return rx; };
bool smif::getTX()  {return tx; };
bool smif::getRTS() {return rts;};
bool smif::getCTS() {return cts;};
bool smif::getDTR() {return dtr;};
bool smif::getDSR() {return dsr;};
bool smif::getDCD() {return dcd;};
bool smif::getRI()  {return ri; };



// determine i/o-transfer rates
float smif::getRXrate() {return rx_rate;};
float smif::getTXrate() {return tx_rate;};



// determine i/o-transfer amounts
float smif::getRXcount() {return rx_count;};
float smif::getTXcount() {return tx_count;};



// check input device and update interval
int smif::getInterval() {return interval;};
const char* smif::getDevice() {return device;};
int smif::getTau() {return tau;};


// periodic update
void smif::timerEvent(QTimerEvent*) {

  // determine elapsed time since last timer event
  // (that is necessary, because object timer intervals
  // are never exact! i.e. if you say startTimer(100),
  // then the intervall may differ between 90 and 110 or
  // even worse. so we use an independend timer)
  elapsed=timer.restart();

  // get status
  if(ioctl(fd,TIOCMGET,&status)<0) {
    // Oh, line must have hupped, reopen it
    setDevice(device);
    return;
  };

  // get info structure for rx and tx
  if(ioctl(fd,TIOCSERGSTATS,&info)<0) {
    // Oh, line must have hupped, reopen it
    setDevice(device);
    return;
  };

  // check line states
  if((rts) != (tmp=((status&TIOCM_RTS)!=0))) emit RTSstatusChanged(rts=tmp);
  if((cts) != (tmp=((status&TIOCM_CTS)!=0))) emit CTSstatusChanged(cts=tmp);
  if((dtr) != (tmp=((status&TIOCM_DTR)!=0))) emit DTRstatusChanged(dtr=tmp);
  if((dsr) != (tmp=((status&TIOCM_DSR)!=0))) emit DSRstatusChanged(cts=tmp);
  if((dcd) != (tmp=((status&TIOCM_CAR)!=0))) emit DCDstatusChanged(dcd=tmp);
  if((ri ) != (tmp=((status&TIOCM_RNG)!=0))) emit RIstatusChanged(ri=tmp);

  // check rx state,count & rate
  diff=info.rx-rx_count;
  if(tmp=(diff>0)) emit RXcountChanged(rx_count=info.rx);
  if(rx != tmp) emit RXstatusChanged(rx=tmp);
  
  if(elapsed) {
    rx_rate=(1000*diff)/elapsed;
    
    if(tau<=elapsed)
      rxf_rate=rx_rate;
    else
      rxf_rate+=elapsed*(rx_rate-rxf_rate)/(float)tau;
  };
  
  emit RXrateChanged(rxf_rate);
  
  // check tx state,count & rate
  diff=info.tx-tx_count;
  if(tmp=(diff>0)) emit TXcountChanged(tx_count=info.tx);
  if(tx != tmp) emit TXstatusChanged(tx=tmp);

  if(elapsed) {
    tx_rate=(1000*diff)/elapsed;

    if(tau<=elapsed)
      txf_rate=tx_rate;
    else
      txf_rate+=elapsed*(tx_rate-txf_rate)/(float)tau;
  };

  emit TXrateChanged(txf_rate);

  // finaly inform the rest of the world
  emit updated();

};



