/*
 *  This file is part of the KDE System Control Tool,
 *  Copyright (C)1999 Thorsten Westheider <twesthei@physik.uni-bielefeld.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ****************************************************************************/

#include <qlist.h>

#include <kapp.h>
#include <kiconloader.h>

#include "deventry.h"
#include "devicelist.h"
#include "devmantab.h"
#include "propdialog.h"


extern DeviceList  devicelist;


DeviceManagerTab::DeviceManagerTab(QWidget *parent, const char *name) : QWidget(parent, name)
{
  _listframe = new QFrame(this, "listframe");
  _listframe->setFrameStyle(QFrame::Panel | QFrame::Sunken);
  _listframe->setLineWidth(2);
  
  _listview = new QListView(_listframe, "listview");
  _listview->setAllColumnsShowFocus(true);
  _listview->setRootIsDecorated(true);
  _listview->setItemMargin(2);
  _listview->addColumn(i18n("Device"));
  connect(_listview, SIGNAL(currentChanged(QListViewItem *)), SLOT(slotCurrentChanged(QListViewItem *)));
  
  _pushprop = new QPushButton(i18n("&Properties"), this, "pushprop");
  _pushprop->adjustSize();
  connect(_pushprop, SIGNAL(clicked()), SLOT(slotProperties()));
  
  _pushrefresh = new QPushButton(i18n("&Refresh"), this, "pushrefresh");
  _pushrefresh->adjustSize();
  connect(_pushrefresh, SIGNAL(clicked()), SLOT(slotRefresh()));  
  
  fillIn();
  slotCurrentChanged(_listview->firstChild());
}


/*
 * Protected methods
 ********************/
 
void  DeviceManagerTab::resizeEvent(QResizeEvent *)
{
  uint  margin = 16;
  uint  w      = width();
  uint  h      = height();
  uint  butx   = margin;
  uint  buty;
  uint  buth   = _pushprop->height();
  uint  butdx  = 4;
  uint  butdy  = 8;
  uint  framew = w-2*margin;
  uint  frameh = h-2*margin-buth-butdy;
  
  _listframe->setGeometry(margin, margin, framew, frameh);
  _listview->setGeometry(2, 2, framew-4, frameh-4);
  
  buty  = margin+frameh+butdy;
  
  _pushprop->move(butx, buty);
  
  butx += _pushprop->width()+butdx;
  
  _pushrefresh->move(butx, buty);
}


/*
 * Private methods
 ******************/
 
void  DeviceManagerTab::fillIn()
{
  KIconLoader  *loader = kapp->getIconLoader();
  DeviceEntry  *deventry, *parent, *newentry;
  QPixmap      xpm;
  Device       *device;
  
  _listview->clear();
  
  for (device = devicelist.first(); device; device = devicelist.next())
  {
    for (deventry = (DeviceEntry *) _listview->firstChild(); deventry; deventry = (DeviceEntry *) deventry->nextSibling())
      if (!deventry->device() && (device->deviceBaseClass() == deventry->deviceBaseClass())) break;
 
    parent   = (deventry) ? deventry : new DeviceEntry(_listview, device->deviceBaseClass(),
  								  device->deviceBaseClassName());
    newentry = new DeviceEntry(parent, device);
    xpm      = loader->loadIcon(device->smallPixmapFile());
 
    parent->setPixmap(0,   xpm);
    newentry->setPixmap(0, xpm);
  }
}


/*
 * Private slots
 ****************/
 
void  DeviceManagerTab::slotCurrentChanged(QListViewItem *item)
{
  _pushprop->setEnabled(((DeviceEntry *) item)->device());
} 


void  DeviceManagerTab::slotProperties()
{
  PropertiesDialog  *propdialog;
  DeviceEntry       *deventry = (DeviceEntry *) _listview->currentItem();
  
  if (deventry)
  {
    propdialog = new PropertiesDialog(deventry->device(), this, "propdialog");  
    propdialog->exec();
    
    delete propdialog;
  }
}


void  DeviceManagerTab::slotRefresh()
{
  devicelist.rescan();
  fillIn();
}

 
