/*
  Copyright (c) 2001 Cornelius Schumacher <schumacher@kde.org>
  Copyright (C) 2003-2004 Reinhold Kainhofer <reinhold@kainhofer.com>
  Copyright (c) 2007 Bruno Virlet <bruno@virlet.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

  As a special exception, permission is given to link this program
  with any edition of Qt, and distribute the resulting executable,
  without including the source code for Qt in the source distribution.
*/
#include "timelabels.h"
#include "agenda.h"
#include "prefs.h"
#include "timelabelszone.h"
#include "timescaleconfigdialog.h"

#include <KLocalizedString>

#include <QFrame>
#include <QIcon>
#include <QMenu>
#include <QPainter>
#include <QPointer>

using namespace EventViews;

TimeLabels::TimeLabels(const QTimeZone &zone, int rows, TimeLabelsZone *parent, Qt::WindowFlags f) :
    QFrame(parent, f),
    mTimezone(zone)
{
    mTimeLabelsZone = parent;
    mRows = rows;
    mMiniWidth = 0;

    mCellHeight = mTimeLabelsZone->preferences()->hourSize() * 4;

    setBackgroundRole(QPalette::Background);

    mMousePos = new QFrame(this);
    mMousePos->setLineWidth(1);
    mMousePos->setFrameStyle(QFrame::HLine | QFrame::Plain);
    mMousePos->setFixedSize(width(), 1);
    colorMousePos();
    mAgenda = Q_NULLPTR;

    if (mTimezone.isValid()) {
        setToolTip(i18n("Timezone:") + i18n(mTimezone.id()));
    }

    setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Fixed);

    updateConfig();
}

void TimeLabels::mousePosChanged(const QPoint &pos)
{
    colorMousePos();
    mMousePos->move(0, pos.y());

    // The repaint somehow prevents that the red line leaves a black artifact when
    // moved down. It's not a full solution, though.
    repaint();
}

void TimeLabels::showMousePos()
{
    // touch screen have no mouse position
    mMousePos->show();
}

void TimeLabels::hideMousePos()
{
    mMousePos->hide();
}

void TimeLabels::colorMousePos()
{
    QPalette pal;
    pal.setColor(QPalette::Window,  // for Oxygen
                 mTimeLabelsZone->preferences()->agendaMarcusBainsLineLineColor());
    pal.setColor(QPalette::WindowText,  // for Plastique
                 mTimeLabelsZone->preferences()->agendaMarcusBainsLineLineColor());
    mMousePos->setPalette(pal);
}

void TimeLabels::setCellHeight(double height)
{
    if (mCellHeight != height) {
        mCellHeight = height;
        updateGeometry();
    }
}

QSize TimeLabels::minimumSizeHint() const
{
    QSize sh = QFrame::sizeHint();
    sh.setWidth(mMiniWidth);
    return sh;
}

static bool use12Clock()
{
    const QString str = QLocale().timeFormat();
    // 'A' or 'a' means am/pm is shown (and then 'h' uses 12-hour format)
    // but 'H' forces a 24-hour format anyway, even with am/pm shown.
    return str.contains(QLatin1Char('a'), Qt::CaseInsensitive) && !str.contains(QLatin1Char('H'));
}

/** updates widget's internal state */
void TimeLabels::updateConfig()
{
    setFont(mTimeLabelsZone->preferences()->agendaTimeLabelsFont());

    QString test = QStringLiteral("20");
    if (use12Clock()) {
        test = QStringLiteral("12");
    }
    mMiniWidth = fontMetrics().width(test);
    if (use12Clock()) {
        test = QStringLiteral("pm");
    } else {
        test = QStringLiteral("00");
    }
    QFont sFont = font();
    sFont.setPointSize(sFont.pointSize() / 2);
    QFontMetrics fmS(sFont);
    mMiniWidth += fmS.width(test) + frameWidth() * 2 + 4;

    /** Can happen if all resources are disabled */
    if (!mAgenda) {
        return;
    }

    // update HourSize
    mCellHeight = mTimeLabelsZone->preferences()->hourSize() * 4;
    // If the agenda is zoomed out so that more than 24 would be shown,
    // the agenda only shows 24 hours, so we need to take the cell height
    // from the agenda, which is larger than the configured one!
    if (mCellHeight < 4 * mAgenda->gridSpacingY()) {
        mCellHeight = 4 * mAgenda->gridSpacingY();
    }

    updateGeometry();

    repaint();
}

/**  */
void TimeLabels::setAgenda(Agenda *agenda)
{
    mAgenda = agenda;

    if (mAgenda) {
        connect(mAgenda, &Agenda::mousePosSignal, this, &TimeLabels::mousePosChanged);
        connect(mAgenda, &Agenda::enterAgenda, this, &TimeLabels::showMousePos);
        connect(mAgenda, &Agenda::leaveAgenda, this, &TimeLabels::hideMousePos);
        connect(mAgenda, &Agenda::gridSpacingYChanged, this, &TimeLabels::setCellHeight);
    }
}

/** This is called in response to repaint() */
void TimeLabels::paintEvent(QPaintEvent *)
{
    QPainter p(this);

    const int ch = height();

    // We won't paint parts that aren't visible
    const int cy = -y();// y() returns a negative value.

    const int beginning =
        !mTimezone.isValid() ?
        0 :
        (mTimezone.offsetFromUtc(QDateTime::currentDateTimeUtc()) -
         mTimeLabelsZone->preferences()->timeZone().offsetFromUtc(QDateTime::currentDateTimeUtc())) / (60 * 60);

    // bug:  the parameters cx and cw are the areas that need to be
    //       redrawn, not the area of the widget.  unfortunately, this
    //       code assumes the latter...

    // now, for a workaround...
    const int cx = frameWidth() * 2;
    const int cw = width();
    // end of workaround

    int cell = static_cast<int>(cy / mCellHeight) + beginning;    // the hour we start drawing with
    double y = (cell - beginning) * mCellHeight;
    QFontMetrics fm = fontMetrics();
    QString hour;
    int timeHeight = fm.ascent();
    QFont hourFont = mTimeLabelsZone->preferences()->agendaTimeLabelsFont();
    p.setFont(font());

    //TODO: rewrite this using QTime's time formats. "am/pm" doesn't make sense
    // in some locale's
    QString suffix;
    if (!use12Clock()) {
        suffix = QStringLiteral("00");
    } else {
        suffix = QStringLiteral("am");
        if (cell > 11) {
            suffix = QStringLiteral("pm");
        }
    }

    // We adjust the size of the hour font to keep it reasonable
    if (timeHeight > mCellHeight) {
        timeHeight = static_cast<int>(mCellHeight - 1);
        int pointS = hourFont.pointSize();
        while (pointS > 4) {   // TODO: use smallestReadableFont() when added to kdelibs
            hourFont.setPointSize(pointS);
            fm = QFontMetrics(hourFont);
            if (fm.ascent() < mCellHeight) {
                break;
            }
            --pointS;
        }
        fm = QFontMetrics(hourFont);
        timeHeight = fm.ascent();
    }
    //timeHeight -= (timeHeight/4-2);
    QFont suffixFont = hourFont;
    suffixFont.setPointSize(suffixFont.pointSize() / 2);
    QFontMetrics fmS(suffixFont);
    const int startW = cw - frameWidth() - 2;
    const int tw2 = fmS.width(suffix);
    const int divTimeHeight = (timeHeight - 1) / 2 - 1;
    //testline
    //p->drawLine(0,0,0,contentsHeight());
    while (y < cy + ch + mCellHeight) {
        QColor lineColor, textColor;
        textColor = palette().color(QPalette::WindowText);
        if (cell < 0 || cell >= 24) {
            textColor.setAlphaF(0.5);
        }
        lineColor = textColor;
        lineColor.setAlphaF(lineColor.alphaF() / 5.);
        p.setPen(lineColor);

        // hour, full line
        p.drawLine(cx, int(y), cw + 2, int(y));

        hour.setNum(cell % 24);
        // handle different timezones
        if (cell < 0) {
            hour.setNum(cell + 24);
        }
        // handle 24h and am/pm time formats
        if (use12Clock()) {
            if (cell == 12) {
                suffix = QStringLiteral("pm");
            }
            if (cell == 0) {
                hour.setNum(12);
            }
            if (cell > 12) {
                hour.setNum(cell - 12);
            }
        }

        // draw the time label
        p.setPen(textColor);
        const int timeWidth = fm.width(hour);
        int offset = startW - timeWidth - tw2 - 1;
        p.setFont(hourFont);
        p.drawText(offset, static_cast<int>(y + timeHeight), hour);
        p.setFont(suffixFont);
        offset = startW - tw2;
        p.drawText(offset, static_cast<int>(y + timeHeight - divTimeHeight), suffix);

        // increment indices
        y += mCellHeight;
        cell++;
    }
}

QSize TimeLabels::sizeHint() const
{
    return QSize(mMiniWidth, mRows * mCellHeight);
}

void TimeLabels::contextMenuEvent(QContextMenuEvent *event)
{
    Q_UNUSED(event);

    QMenu popup(this);
    QAction *editTimeZones =
        popup.addAction(QIcon::fromTheme(QStringLiteral("document-properties")), i18n("&Add Timezones..."));
    QAction *removeTimeZone =
        popup.addAction(QIcon::fromTheme(QStringLiteral("edit-delete")),
                        i18n("&Remove Timezone %1", i18n(mTimezone.id())));
    if (!mTimezone.isValid() ||
            !mTimeLabelsZone->preferences()->timeScaleTimezones().count() ||
            mTimezone == mTimeLabelsZone->preferences()->timeZone()) {
        removeTimeZone->setEnabled(false);
    }

    QAction *activatedAction = popup.exec(QCursor::pos());
    if (activatedAction == editTimeZones) {
        QPointer<TimeScaleConfigDialog> dialog =
            new TimeScaleConfigDialog(mTimeLabelsZone->preferences(), this);
        if (dialog->exec() == QDialog::Accepted) {
            mTimeLabelsZone->reset();
        }
        delete dialog;
    } else if (activatedAction == removeTimeZone) {
        QStringList list = mTimeLabelsZone->preferences()->timeScaleTimezones();
        list.removeAll(QString::fromUtf8(mTimezone.id()));
        mTimeLabelsZone->preferences()->setTimeScaleTimezones(list);
        mTimeLabelsZone->preferences()->writeConfig();
        mTimeLabelsZone->reset();
        hide();
        deleteLater();
    }
}

QTimeZone TimeLabels::timeZone() const
{
    return mTimezone;
}

QString TimeLabels::header() const
{
    return i18n(mTimezone.id());
}

QString TimeLabels::headerToolTip() const
{
    QString toolTip;
    toolTip += QLatin1String("<qt>");
    toolTip += i18n("<b>%1</b>", i18n(mTimezone.id()));
    toolTip += QLatin1String("<hr>");
    //TODO: Once string freeze is lifted, add UTC offset here
    if (mTimezone.country() != QLocale::AnyCountry) {
        toolTip += i18n("<i>Country:</i> %1", QLocale::countryToString(mTimezone.country()));
        toolTip += QLatin1String("<br/>");
    }

    auto abbreviations = QStringLiteral("&nbsp;");
    foreach (const auto &transition, mTimezone.transitions(QDateTime::currentDateTime(), QDateTime::currentDateTime().addYears(1))) {
        abbreviations += transition.abbreviation;
        abbreviations += QLatin1String(",&nbsp;");
    }
    abbreviations.chop(7);
    if (!abbreviations.isEmpty()) {
        toolTip += i18n("<i>Abbreviations:</i>");
        toolTip += abbreviations;
        toolTip += QLatin1String("<br/>");
    }
    if (!mTimezone.comment().isEmpty()) {
        toolTip += i18n("<i>Comment:</i> %1", mTimezone.comment());
    }
    toolTip += QLatin1String("</qt>");

    return toolTip;
}

