"use strict";
var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
Object.defineProperty(exports, "__esModule", { value: true });
var scales_1 = require("./scales");
var specs_1 = require("./specs");
var utils_1 = require("../state/utils");
/**
 * Compute the ticks values and identify max width and height of the labels
 * so we can compute the max space occupied by the axis component.
 * @param axisSpec the spec of the axis
 * @param xDomain the x domain associated
 * @param yDomain the y domain array
 * @param totalBarsInCluster the total number of grouped series
 * @param bboxCalculator an instance of the boundingbox calculator
 * @param chartRotation the rotation of the chart
 */
function computeAxisTicksDimensions(axisSpec, xDomain, yDomain, totalBarsInCluster, bboxCalculator, chartRotation, axisConfig, barsPadding, enableHistogramMode) {
    if (axisSpec.hide) {
        return null;
    }
    var scale = getScaleForAxisSpec(axisSpec, xDomain, yDomain, totalBarsInCluster, chartRotation, 0, 1, barsPadding, enableHistogramMode);
    if (!scale) {
        throw new Error("Cannot compute scale for axis spec " + axisSpec.id);
    }
    var tickLabelPadding = getAxisTickLabelPadding(axisConfig.tickLabelStyle.padding, axisSpec.style);
    var dimensions = computeTickDimensions(scale, axisSpec.tickFormat, bboxCalculator, axisConfig, tickLabelPadding, axisSpec.tickLabelRotation, {
        timeZone: xDomain.timeZone,
    });
    return __assign({}, dimensions);
}
exports.computeAxisTicksDimensions = computeAxisTicksDimensions;
function getAxisTickLabelPadding(axisConfigTickLabelPadding, axisSpecStyle) {
    if (axisSpecStyle && axisSpecStyle.tickLabelPadding !== undefined) {
        return axisSpecStyle.tickLabelPadding;
    }
    return axisConfigTickLabelPadding;
}
exports.getAxisTickLabelPadding = getAxisTickLabelPadding;
function isYDomain(position, chartRotation) {
    var isStraightRotation = chartRotation === 0 || chartRotation === 180;
    if (isVerticalAxis(position)) {
        return isStraightRotation;
    }
    return !isStraightRotation;
}
exports.isYDomain = isYDomain;
function getScaleForAxisSpec(axisSpec, xDomain, yDomain, totalBarsInCluster, chartRotation, minRange, maxRange, barsPadding, enableHistogramMode) {
    var axisIsYDomain = isYDomain(axisSpec.position, chartRotation);
    var range = [minRange, maxRange];
    if (axisIsYDomain) {
        var yScales = scales_1.computeYScales({
            yDomains: yDomain,
            range: range,
            ticks: axisSpec.ticks,
            integersOnly: axisSpec.integersOnly,
        });
        if (yScales.has(axisSpec.groupId)) {
            return yScales.get(axisSpec.groupId);
        }
        return null;
    }
    else {
        return scales_1.computeXScale({
            xDomain: xDomain,
            totalBarsInCluster: totalBarsInCluster,
            range: range,
            barsPadding: barsPadding,
            enableHistogramMode: enableHistogramMode,
            ticks: axisSpec.ticks,
            integersOnly: axisSpec.integersOnly,
        });
    }
}
exports.getScaleForAxisSpec = getScaleForAxisSpec;
function computeRotatedLabelDimensions(unrotatedDims, degreesRotation) {
    var width = unrotatedDims.width, height = unrotatedDims.height;
    var radians = (degreesRotation * Math.PI) / 180;
    var rotatedHeight = Math.abs(width * Math.sin(radians)) + Math.abs(height * Math.cos(radians));
    var rotatedWidth = Math.abs(width * Math.cos(radians)) + Math.abs(height * Math.sin(radians));
    return {
        width: rotatedWidth,
        height: rotatedHeight,
    };
}
exports.computeRotatedLabelDimensions = computeRotatedLabelDimensions;
exports.getMaxBboxDimensions = function (bboxCalculator, fontSize, fontFamily, tickLabelRotation, tickLabelPadding) { return function (acc, tickLabel) {
    var bbox = bboxCalculator.compute(tickLabel, tickLabelPadding, fontSize, fontFamily);
    var rotatedBbox = computeRotatedLabelDimensions(bbox, tickLabelRotation);
    var width = Math.ceil(rotatedBbox.width);
    var height = Math.ceil(rotatedBbox.height);
    var labelWidth = Math.ceil(bbox.width);
    var labelHeight = Math.ceil(bbox.height);
    var prevWidth = acc.maxLabelBboxWidth;
    var prevHeight = acc.maxLabelBboxHeight;
    var prevLabelWidth = acc.maxLabelTextWidth;
    var prevLabelHeight = acc.maxLabelTextHeight;
    return {
        maxLabelBboxWidth: prevWidth > width ? prevWidth : width,
        maxLabelBboxHeight: prevHeight > height ? prevHeight : height,
        maxLabelTextWidth: prevLabelWidth > labelWidth ? prevLabelWidth : labelWidth,
        maxLabelTextHeight: prevLabelHeight > labelHeight ? prevLabelHeight : labelHeight,
    };
}; };
function computeTickDimensions(scale, tickFormat, bboxCalculator, axisConfig, tickLabelPadding, tickLabelRotation, tickFormatOptions) {
    if (tickLabelRotation === void 0) { tickLabelRotation = 0; }
    var tickValues = scale.ticks();
    var tickLabels = tickValues.map(function (d) {
        return tickFormat(d, tickFormatOptions);
    });
    var _a = axisConfig.tickLabelStyle, fontFamily = _a.fontFamily, fontSize = _a.fontSize;
    var _b = tickLabels.reduce(exports.getMaxBboxDimensions(bboxCalculator, fontSize, fontFamily, tickLabelRotation, tickLabelPadding), { maxLabelBboxWidth: 0, maxLabelBboxHeight: 0, maxLabelTextWidth: 0, maxLabelTextHeight: 0 }), maxLabelBboxWidth = _b.maxLabelBboxWidth, maxLabelBboxHeight = _b.maxLabelBboxHeight, maxLabelTextWidth = _b.maxLabelTextWidth, maxLabelTextHeight = _b.maxLabelTextHeight;
    return {
        tickValues: tickValues,
        tickLabels: tickLabels,
        maxLabelBboxWidth: maxLabelBboxWidth,
        maxLabelBboxHeight: maxLabelBboxHeight,
        maxLabelTextWidth: maxLabelTextWidth,
        maxLabelTextHeight: maxLabelTextHeight,
    };
}
/**
 * The Konva api sets the top right corner of a shape as the default origin of rotation.
 * In order to apply rotation to tick labels while preserving their relative position to the axis,
 * we compute offsets to apply to the Text element as well as adjust the x/y coordinates to adjust
 * for these offsets.
 */
function centerRotationOrigin(axisTicksDimensions, coordinates) {
    var maxLabelBboxWidth = axisTicksDimensions.maxLabelBboxWidth, maxLabelBboxHeight = axisTicksDimensions.maxLabelBboxHeight, maxLabelTextWidth = axisTicksDimensions.maxLabelTextWidth, maxLabelTextHeight = axisTicksDimensions.maxLabelTextHeight;
    var offsetX = maxLabelTextWidth / 2;
    var offsetY = maxLabelTextHeight / 2;
    var x = coordinates.x + maxLabelBboxWidth / 2;
    var y = coordinates.y + maxLabelBboxHeight / 2;
    return { offsetX: offsetX, offsetY: offsetY, x: x, y: y };
}
exports.centerRotationOrigin = centerRotationOrigin;
/**
 * Gets the computed x/y coordinates & alignment properties for an axis tick label.
 * @param isVerticalAxis if the axis is vertical (in contrast to horizontal)
 * @param tickLabelRotation degree of rotation of the tick label
 * @param tickSize length of tick line
 * @param tickPadding amount of padding between label and tick line
 * @param tickPosition position of tick relative to axis line origin and other ticks along it
 * @param position position of where the axis sits relative to the visualization
 * @param axisTicksDimensions computed axis dimensions and values (from computeTickDimensions)
 */
function getTickLabelProps(tickLabelRotation, tickSize, tickPadding, tickPosition, position, axisPosition, axisTicksDimensions) {
    var maxLabelBboxWidth = axisTicksDimensions.maxLabelBboxWidth, maxLabelBboxHeight = axisTicksDimensions.maxLabelBboxHeight;
    var isRotated = tickLabelRotation !== 0;
    var align = 'center';
    var verticalAlign = 'middle';
    if (isVerticalAxis(position)) {
        var isLeftAxis = position === specs_1.Position.Left;
        if (!isRotated) {
            align = isLeftAxis ? 'right' : 'left';
        }
        return {
            x: isLeftAxis ? axisPosition.width - tickSize - tickPadding - maxLabelBboxWidth : tickSize + tickPadding,
            y: tickPosition - maxLabelBboxHeight / 2,
            align: align,
            verticalAlign: verticalAlign,
        };
    }
    var isAxisTop = position === specs_1.Position.Top;
    if (!isRotated) {
        verticalAlign = isAxisTop ? 'bottom' : 'top';
    }
    return {
        x: tickPosition - maxLabelBboxWidth / 2,
        y: isAxisTop ? axisPosition.height - tickSize - tickPadding - maxLabelBboxHeight : tickSize + tickPadding,
        align: align,
        verticalAlign: verticalAlign,
    };
}
exports.getTickLabelProps = getTickLabelProps;
function getVerticalAxisTickLineProps(position, axisWidth, tickSize, tickPosition) {
    var isLeftAxis = position === specs_1.Position.Left;
    var y = tickPosition;
    var x1 = isLeftAxis ? axisWidth : 0;
    var x2 = isLeftAxis ? axisWidth - tickSize : tickSize;
    return [x1, y, x2, y];
}
exports.getVerticalAxisTickLineProps = getVerticalAxisTickLineProps;
function getHorizontalAxisTickLineProps(position, axisHeight, tickSize, tickPosition) {
    var isTopAxis = position === specs_1.Position.Top;
    var x = tickPosition;
    var y1 = isTopAxis ? axisHeight - tickSize : 0;
    var y2 = isTopAxis ? axisHeight : tickSize;
    return [x, y1, x, y2];
}
exports.getHorizontalAxisTickLineProps = getHorizontalAxisTickLineProps;
function getVerticalAxisGridLineProps(tickPosition, chartWidth) {
    return [0, tickPosition, chartWidth, tickPosition];
}
exports.getVerticalAxisGridLineProps = getVerticalAxisGridLineProps;
function getHorizontalAxisGridLineProps(tickPosition, chartHeight) {
    return [tickPosition, 0, tickPosition, chartHeight];
}
exports.getHorizontalAxisGridLineProps = getHorizontalAxisGridLineProps;
function getMinMaxRange(axisPosition, chartRotation, chartDimensions) {
    var width = chartDimensions.width, height = chartDimensions.height;
    switch (axisPosition) {
        case specs_1.Position.Bottom:
        case specs_1.Position.Top:
            return getBottomTopAxisMinMaxRange(chartRotation, width);
        case specs_1.Position.Left:
        case specs_1.Position.Right:
            return getLeftAxisMinMaxRange(chartRotation, height);
    }
}
exports.getMinMaxRange = getMinMaxRange;
function getBottomTopAxisMinMaxRange(chartRotation, width) {
    switch (chartRotation) {
        case 0:
            // dealing with x domain
            return { minRange: 0, maxRange: width };
        case 90:
            // dealing with y domain
            return { minRange: 0, maxRange: width };
        case -90:
            // dealing with y domain
            return { minRange: width, maxRange: 0 };
        case 180:
            // dealing with x domain
            return { minRange: width, maxRange: 0 };
    }
}
function getLeftAxisMinMaxRange(chartRotation, height) {
    switch (chartRotation) {
        case 0:
            // dealing with y domain
            return { minRange: height, maxRange: 0 };
        case 90:
            // dealing with x domain
            return { minRange: 0, maxRange: height };
        case -90:
            // dealing with x domain
            return { minRange: height, maxRange: 0 };
        case 180:
            // dealing with y domain
            return { minRange: 0, maxRange: height };
    }
}
function getAvailableTicks(axisSpec, scale, totalBarsInCluster, enableHistogramMode, tickFormatOptions) {
    var ticks = scale.ticks();
    var isSingleValueScale = scale.domain[0] - scale.domain[1] === 0;
    var hasAdditionalTicks = enableHistogramMode && scale.bandwidth > 0;
    if (hasAdditionalTicks) {
        var lastComputedTick = ticks[ticks.length - 1];
        if (!isSingleValueScale) {
            var penultimateComputedTick = ticks[ticks.length - 2];
            var computedTickDistance = lastComputedTick - penultimateComputedTick;
            var numTicks = scale.minInterval / computedTickDistance;
            for (var i = 1; i <= numTicks; i++) {
                ticks.push(i * computedTickDistance + lastComputedTick);
            }
        }
    }
    var shift = totalBarsInCluster > 0 ? totalBarsInCluster : 1;
    var band = scale.bandwidth / (1 - scale.barsPadding);
    var halfPadding = (band - scale.bandwidth) / 2;
    var offset = enableHistogramMode ? -halfPadding : (scale.bandwidth * shift) / 2;
    if (isSingleValueScale && hasAdditionalTicks) {
        var firstTickValue = ticks[0];
        var firstTick = {
            value: firstTickValue,
            label: axisSpec.tickFormat(firstTickValue, tickFormatOptions),
            position: scale.scale(firstTickValue) + offset,
        };
        var lastTickValue = firstTickValue + scale.minInterval;
        var lastTick = {
            value: lastTickValue,
            label: axisSpec.tickFormat(lastTickValue, tickFormatOptions),
            position: scale.bandwidth + halfPadding * 2,
        };
        return [firstTick, lastTick];
    }
    return ticks.map(function (tick) {
        return {
            value: tick,
            label: axisSpec.tickFormat(tick, tickFormatOptions),
            position: scale.scale(tick) + offset,
        };
    });
}
exports.getAvailableTicks = getAvailableTicks;
function getVisibleTicks(allTicks, axisSpec, axisDim) {
    // We sort the ticks by position so that we can incrementally compute previousOccupiedSpace
    allTicks.sort(function (a, b) { return a.position - b.position; });
    var showOverlappingTicks = axisSpec.showOverlappingTicks, showOverlappingLabels = axisSpec.showOverlappingLabels;
    var maxLabelBboxHeight = axisDim.maxLabelBboxHeight, maxLabelBboxWidth = axisDim.maxLabelBboxWidth;
    var requiredSpace = isVerticalAxis(axisSpec.position) ? maxLabelBboxHeight / 2 : maxLabelBboxWidth / 2;
    var previousOccupiedSpace = 0;
    var visibleTicks = [];
    for (var i = 0; i < allTicks.length; i++) {
        var position = allTicks[i].position;
        if (i === 0) {
            visibleTicks.push(allTicks[i]);
            previousOccupiedSpace = position + requiredSpace;
        }
        else if (position - requiredSpace >= previousOccupiedSpace) {
            visibleTicks.push(allTicks[i]);
            previousOccupiedSpace = position + requiredSpace;
        }
        else {
            // still add the tick but without a label
            if (showOverlappingTicks || showOverlappingLabels) {
                var overlappingTick = __assign(__assign({}, allTicks[i]), { label: showOverlappingLabels ? allTicks[i].label : '' });
                visibleTicks.push(overlappingTick);
            }
        }
    }
    return visibleTicks;
}
exports.getVisibleTicks = getVisibleTicks;
function getAxisPosition(chartDimensions, chartMargins, axisTitleHeight, axisSpec, axisDim, cumTopSum, cumBottomSum, cumLeftSum, cumRightSum) {
    var position = axisSpec.position, tickSize = axisSpec.tickSize, tickPadding = axisSpec.tickPadding;
    var maxLabelBboxHeight = axisDim.maxLabelBboxHeight, maxLabelBboxWidth = axisDim.maxLabelBboxWidth;
    var top = chartDimensions.top, left = chartDimensions.left, height = chartDimensions.height, width = chartDimensions.width;
    var dimensions = {
        top: top,
        left: left,
        width: width,
        height: height,
    };
    var topIncrement = 0;
    var bottomIncrement = 0;
    var leftIncrement = 0;
    var rightIncrement = 0;
    if (isVerticalAxis(position)) {
        var dimWidth = maxLabelBboxWidth + tickSize + tickPadding + axisTitleHeight;
        if (position === specs_1.Position.Left) {
            leftIncrement = dimWidth + chartMargins.left;
            dimensions.left = cumLeftSum + chartMargins.left;
        }
        else {
            rightIncrement = dimWidth + chartMargins.right;
            dimensions.left = left + width + cumRightSum;
        }
        dimensions.width = dimWidth;
    }
    else {
        var dimHeight = maxLabelBboxHeight + tickSize + tickPadding + axisTitleHeight;
        if (position === specs_1.Position.Top) {
            topIncrement = dimHeight + chartMargins.top;
            dimensions.top = cumTopSum + chartMargins.top;
        }
        else {
            bottomIncrement = dimHeight + chartMargins.bottom;
            dimensions.top = top + height + cumBottomSum;
        }
        dimensions.height = dimHeight;
    }
    return { dimensions: dimensions, topIncrement: topIncrement, bottomIncrement: bottomIncrement, leftIncrement: leftIncrement, rightIncrement: rightIncrement };
}
exports.getAxisPosition = getAxisPosition;
function isVerticalAxis(axisPosition) {
    return axisPosition === specs_1.Position.Left || axisPosition === specs_1.Position.Right;
}
exports.isVerticalAxis = isVerticalAxis;
function isHorizontalAxis(axisPosition) {
    return axisPosition === specs_1.Position.Top || axisPosition === specs_1.Position.Bottom;
}
exports.isHorizontalAxis = isHorizontalAxis;
function isVerticalGrid(axisPosition) {
    return isHorizontalAxis(axisPosition);
}
exports.isVerticalGrid = isVerticalGrid;
function isHorizontalGrid(axisPosition) {
    return isVerticalAxis(axisPosition);
}
exports.isHorizontalGrid = isHorizontalGrid;
function getAxisTicksPositions(computedChartDims, chartTheme, chartRotation, axisSpecs, axisDimensions, xDomain, yDomain, totalGroupsCount, enableHistogramMode, barsPadding) {
    var chartPaddings = chartTheme.chartPaddings, chartMargins = chartTheme.chartMargins;
    var axisPositions = new Map();
    var axisVisibleTicks = new Map();
    var axisTicks = new Map();
    var axisGridLinesPositions = new Map();
    var chartDimensions = computedChartDims.chartDimensions;
    var cumTopSum = 0;
    var cumBottomSum = chartPaddings.bottom;
    var cumLeftSum = computedChartDims.leftMargin;
    var cumRightSum = chartPaddings.right;
    axisDimensions.forEach(function (axisDim, id) {
        var axisSpec = utils_1.getSpecsById(axisSpecs, id);
        // Consider refactoring this so this condition can be tested
        // Given some of the values that get passed around, maybe re-write as a reduce instead of forEach?
        if (!axisSpec) {
            return;
        }
        var minMaxRanges = getMinMaxRange(axisSpec.position, chartRotation, chartDimensions);
        var scale = getScaleForAxisSpec(axisSpec, xDomain, yDomain, totalGroupsCount, chartRotation, minMaxRanges.minRange, minMaxRanges.maxRange, barsPadding, enableHistogramMode);
        if (!scale) {
            throw new Error("Cannot compute scale for axis spec " + axisSpec.id);
        }
        var tickFormatOptions = {
            timeZone: xDomain.timeZone,
        };
        var allTicks = getAvailableTicks(axisSpec, scale, totalGroupsCount, enableHistogramMode, tickFormatOptions);
        var visibleTicks = getVisibleTicks(allTicks, axisSpec, axisDim);
        if (axisSpec.showGridLines) {
            var isVertical_1 = isVerticalAxis(axisSpec.position);
            var gridLines = visibleTicks.map(function (tick) {
                return computeAxisGridLinePositions(isVertical_1, tick.position, chartDimensions);
            });
            axisGridLinesPositions.set(id, gridLines);
        }
        var _a = chartTheme.axes.axisTitleStyle, fontSize = _a.fontSize, padding = _a.padding;
        var axisTitleHeight = axisSpec.title !== undefined ? fontSize + padding : 0;
        var axisPosition = getAxisPosition(chartDimensions, chartMargins, axisTitleHeight, axisSpec, axisDim, cumTopSum, cumBottomSum, cumLeftSum, cumRightSum);
        cumTopSum += axisPosition.topIncrement;
        cumBottomSum += axisPosition.bottomIncrement;
        cumLeftSum += axisPosition.leftIncrement;
        cumRightSum += axisPosition.rightIncrement;
        axisPositions.set(id, axisPosition.dimensions);
        axisVisibleTicks.set(id, visibleTicks);
        axisTicks.set(id, allTicks);
    });
    return {
        axisPositions: axisPositions,
        axisTicks: axisTicks,
        axisVisibleTicks: axisVisibleTicks,
        axisGridLinesPositions: axisGridLinesPositions,
    };
}
exports.getAxisTicksPositions = getAxisTicksPositions;
function computeAxisGridLinePositions(isVerticalAxis, tickPosition, chartDimensions) {
    var positions = isVerticalAxis
        ? getVerticalAxisGridLineProps(tickPosition, chartDimensions.width)
        : getHorizontalAxisGridLineProps(tickPosition, chartDimensions.height);
    return positions;
}
exports.computeAxisGridLinePositions = computeAxisGridLinePositions;
function isLowerBound(domain) {
    return domain.min != null;
}
exports.isLowerBound = isLowerBound;
function isUpperBound(domain) {
    return domain.max != null;
}
exports.isUpperBound = isUpperBound;
function isCompleteBound(domain) {
    return domain.max != null && domain.min != null;
}
exports.isCompleteBound = isCompleteBound;
function isBounded(domain) {
    return domain.max != null || domain.min != null;
}
exports.isBounded = isBounded;
exports.isDuplicateAxis = function (_a, _b, tickMap, specs) {
    var position = _a.position, title = _a.title;
    var tickLabels = _b.tickLabels;
    var firstTickLabel = tickLabels[0];
    var lastTickLabel = tickLabels.slice(-1)[0];
    var hasDuplicate = false;
    tickMap.forEach(function (_a, axisId) {
        var axisTickLabels = _a.tickLabels;
        if (!hasDuplicate &&
            axisTickLabels &&
            tickLabels.length === axisTickLabels.length &&
            firstTickLabel === axisTickLabels[0] &&
            lastTickLabel === axisTickLabels.slice(-1)[0]) {
            var spec = utils_1.getSpecsById(specs, axisId);
            if (spec && spec.position === position && title === spec.title) {
                hasDuplicate = true;
            }
        }
    });
    return hasDuplicate;
};
//# sourceMappingURL=axis_utils.js.map