#include "yacas/numbers.h"
#include "yacas/errors.h"
#include "yacas/standard.h"

#include <cmath>
#include <sstream>

//////////////////////////////////////////////////
///// bits_to_digits and digits_to_bits implementation
//////////////////////////////////////////////////

// lookup table for transforming the number of digits

static const unsigned log2_table_size = 32;
// report the table size
unsigned log2_table_range()
{
    return log2_table_size;
}

// A lookup table of Ln(n)/Ln(2) for n = 1 .. 32.
// Generated by: PrintList(N(Ln(1 .. 32)/Ln(2)), ",") at precision 40
const double log2_table[log2_table_size] = {
    0.,
    1.,
    1.5849625007211561814537389439478165087598,
    2.,
    2.3219280948873623478703194294893901758648,
    2.5849625007211561814537389439478165087598,
    2.807354922057604107441969317231830808641,
    3.,
    3.1699250014423123629074778878956330175196,
    3.3219280948873623478703194294893901758648,
    3.4594316186372972561993630467257929587032,
    3.5849625007211561814537389439478165087598,
    3.7004397181410921603968126542566947336284,
    3.807354922057604107441969317231830808641,
    3.9068905956085185293240583734372066846246,
    4.,
    4.0874628412503394082540660108104043540112,
    4.1699250014423123629074778878956330175196,
    4.2479275134435854937935194229068344226935,
    4.3219280948873623478703194294893901758648,
    4.3923174227787602888957082611796473174008,
    4.4594316186372972561993630467257929587032,
    4.5235619560570128722941482441626688444988,
    4.5849625007211561814537389439478165087598,
    4.6438561897747246957406388589787803517296,
    4.7004397181410921603968126542566947336284,
    4.7548875021634685443612168318434495262794,
    4.807354922057604107441969317231830808641,
    4.8579809951275721207197733246279847624768,
    4.9068905956085185293240583734372066846246,
    4.9541963103868752088061235991755544235489,
    5.};

// table look-up of small integer logarithms, for converting the number of
// digits to binary and back
double log2_table_lookup(unsigned n)
{
    if (n <= log2_table_size && n >= BASE2)
        return log2_table[n - 1];

    std::ostringstream buf;
    buf << "log2_table_lookup: error: invalid argument " << n;
    throw LispErrGeneric(buf.str());
}

// convert the number of digits in given base to the number of bits, and back.
// need to round the number of digits.
// to make sure that there is no hysteresis, we round upwards on digits_to_bits
// but round down on bits_to_digits
unsigned long digits_to_bits(unsigned long digits, unsigned base)
{
    return static_cast<unsigned long>(
        std::ceil(double(digits) * log2_table_lookup(base)));
}

unsigned long bits_to_digits(unsigned long bits, unsigned base)
{
    return static_cast<unsigned long>(
        std::floor(double(bits) / log2_table_lookup(base)));
}

//////////////////////////////////////////////////
///// End of bits_to_digits and digits_to_bits implementation
//////////////////////////////////////////////////
