/*
 * Copyright (C) 2000-2024 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <math.h>

#include "_xitk.h"
#include "slider.h"
#include "backend.h"

#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif

#define _INVALID_POS 0x7fffffff

typedef struct _slider_private_s {
  xitk_widget_t           w;

  void                    (*paint) (struct _slider_private_s *wp, const widget_event_t *event);
  xitk_slider_type_t      sType;
  uint32_t                wrap;

  /* value related */
  int                     deferred_pos;
  float                   angle;
  int                     upper, lower, range0, range1, step, value;

  xitk_hv_t               paddle_state_list[XITK_IMG_STATE_LIST_SIZE];
  xitk_part_image_t       paddle_skin;
  xitk_part_image_t       bg_skin;

  xitk_state_callback_t   motion_callback;
  xitk_state_callback_t   callback;

  int                     radius;

  int                     bar_mode; /** << 1 (bar |####---|), 0 (pointer |---#----|) */

  xitk_slider_hv_t        hv_info;
  int                     hv_w, hv_h;
  int                     hv_x, hv_y, hv_max_x, hv_max_y;

  struct {
    xitk_hv_t             f;
    int                   x, y, w, h;
    xitk_img_state_t      state;
  }                       paddle_drawn_here;
} _slider_private_t;

static void _xitk_slider_set_type (_slider_private_t *wp, xitk_slider_type_t type) {
  wp->wrap = (uint32_t)type & ((uint32_t)XITK_WRAP_VSLIDER - (uint32_t)XITK_VSLIDER);
  wp->sType = (xitk_slider_type_t)((uint32_t)type - wp->wrap);
}

static void _xitk_slider_update_skin (_slider_private_t *wp) {
  XITK_HV_INIT;

  xitk_part_image_states (&wp->paddle_skin, wp->paddle_state_list, 3);
  wp->hv_w = XITK_HV_H (wp->paddle_state_list[0]);
  wp->hv_h = XITK_HV_V (wp->paddle_state_list[0]);
  wp->bar_mode = 0;
  if (wp->sType == XITK_HSLIDER) {
    if (XITK_HV_H (wp->paddle_state_list[0]) == wp->bg_skin.width)
      wp->bar_mode = 1;
  } else if (wp->sType == XITK_VSLIDER) {
    if (XITK_HV_V (wp->paddle_state_list[0]) == wp->bg_skin.height)
      wp->bar_mode = 1;
  }

  XITK_HV_H (wp->w.size) = wp->bg_skin.width;
  XITK_HV_V (wp->w.size) = wp->bg_skin.height;

  wp->hv_max_x = XITK_HV_H (wp->w.size) - wp->hv_w;
  if (wp->hv_max_x < 0)
    wp->hv_max_x = 0;
  if (wp->hv_x > wp->hv_max_x)
    wp->hv_x = wp->hv_max_x;
  wp->hv_max_y = XITK_HV_V (wp->w.size) - wp->hv_h;
  if (wp->hv_max_y < 0)
    wp->hv_max_y = 0;
  if (wp->hv_y > wp->hv_max_y)
    wp->hv_y = wp->hv_max_y;
}

static int _xitk_slider_report_xy (_slider_private_t *wp, int x, int y) {
  XITK_HV_INIT;

  x = wp->hv_max_x > 0
    ? ((x - XITK_HV_H (wp->w.pos) - (wp->hv_w >> 1)) * (wp->hv_info.h.max - wp->hv_info.h.visible) + (wp->hv_max_x >> 1)) / wp->hv_max_x
    : 0;
  if (x > wp->hv_info.h.max - wp->hv_info.h.visible)
    x = wp->hv_info.h.max - wp->hv_info.h.visible;
  if (x < 0)
    x = 0;
  y = wp->hv_max_y > 0
    ? ((y - XITK_HV_V (wp->w.pos) - (wp->hv_h >> 1)) * (wp->hv_info.v.max - wp->hv_info.v.visible) + (wp->hv_max_y >> 1)) / wp->hv_max_y
    : 0;
  if (y > wp->hv_info.v.max - wp->hv_info.v.visible)
    y = wp->hv_info.v.max - wp->hv_info.v.visible;
  if (y < 0)
    y = 0;
  if ((x != wp->hv_info.h.pos) || (y != wp->hv_info.v.pos)) {
    wp->hv_info.h.pos = x;
    wp->hv_info.v.pos = y;
    return 1;
  }
  return 0;
}

/*
 *
 */
static int _xitk_slider_update_value (_slider_private_t *wp, int value) {
  double fval;
  int nv = (value - wp->lower) % wp->range1;
  /* quick and dirty sign safe modulo div. */
  nv += (nv >> (sizeof (nv) * 8 - 1)) & wp->range1;
  nv += wp->lower;

  if ((value != nv) && !wp->wrap)
    nv = (value < wp->lower) ? wp->lower : wp->upper;
  if (nv == wp->value)
    return 0;

  wp->value = nv;
  fval = (double)(wp->value - wp->lower) / (double)wp->range1;
  wp->angle = wp->wrap
            ? M_PI * 3 / 2 - fval * M_PI * 2
	    : M_PI * 5 / 4 - fval * M_PI * 3 / 2;
  return 1;
}

/**       ********
 *     ***   1/2  ***
 *   **              **
 *  *                  *
 * |  1/1  * M_PI   0/1 |
 * |                    |
 *  *                  *
 *   ** 5/4     -1/4 **
 *     ***  -1/2  ***
 *        ********
 */
static int _xitk_slider_update_xy (_slider_private_t *wp, int x, int y) {
  XITK_HV_INIT;
  int value;

  if (wp->sType == XITK_RSLIDER) {

    int xc = wp->bg_skin.width >> 1, yc = wp->bg_skin.height >> 1;
    double angle = atan2 (yc - y, x - xc);

    if (angle < -M_PI / 2.)
      angle += 2 * M_PI;
    if (wp->wrap) {
      if (angle < M_PI * -1 / 2)
        angle = M_PI * -1 / 2;
      if (angle > M_PI * 3 / 2)
        angle = M_PI * 3 / 2;
      value = (M_PI * 3 / 2 - angle) / (M_PI * 2) * wp->range1 + 0.5;
    } else {
      if (angle < M_PI * -1 / 4)
        angle = M_PI * -1 / 4;
      if (angle > M_PI * 5 / 4)
        angle = M_PI * 5 / 4;
      value = (M_PI * 5 / 4 - angle) / (M_PI * 3 / 2) * wp->range1 + 0.5;
    }

  } else if (wp->sType == XITK_HSLIDER) {

    int width = wp->bg_skin.width;

    if (!wp->bar_mode) {
      x -= XITK_HV_H (wp->paddle_state_list[0]) >> 1;
      width -= XITK_HV_H (wp->paddle_state_list[0]);
    }
    value = (wp->range0 * x + (width >> 1)) / width;

  } else { /* wp->sType == XITK_VSLIDER */

    int height = wp->bg_skin.height;

    if (!wp->bar_mode) {
      y -= XITK_HV_V (wp->paddle_state_list[0]) >> 1;
      height -= XITK_HV_V (wp->paddle_state_list[0]);
    }
    value = (wp->range0 * (height - y) + (height >> 1)) / height;

  }

  return _xitk_slider_update_value (wp, wp->lower + value);
}

/*
 * Draw widget
 */
static void _xitk_slider_paint_n (_slider_private_t *wp, const widget_event_t *event) {
  (void)wp;
  (void)event;
}

static void _xitk_slider_paint_p (_slider_private_t *wp, const widget_event_t *event) {
  XITK_HV_INIT;
  xitk_hv_t p1;
  /* paddle top left */
  XITK_HV_H (p1) = wp->paddle_drawn_here.x;
  XITK_HV_V (p1) = wp->paddle_drawn_here.y;
  p1.w += wp->w.pos.w;
  
  if (event) {
    xitk_hv_t e1, e2, p2;
    /* event.top_left */
    XITK_HV_H (e1) = event->x;
    XITK_HV_V (e1) = event->y;
    /* event.bottom_right */
    XITK_HV_H (e2) = event->width;
    XITK_HV_V (e2) = event->height;
    e2.w += e1.w;
    /* paddle bottom right */
    XITK_HV_H (p2) = wp->paddle_drawn_here.w;
    XITK_HV_V (p2) = wp->paddle_drawn_here.h;
    p2.w += p1.w;
    /* get common area */
    e1.w = xitk_hv_max (p1.w, e1.w);
    e2.w = xitk_hv_min (p2.w, e2.w);
    e2.w -= e1.w;
    if (XITK_HV_IS_RECT (e2)) {
      p2.w = wp->paddle_drawn_here.f.w + e1.w - p1.w;
      xitk_part_image_draw (wp->w.wl, &wp->paddle_skin, NULL,
        XITK_HV_H (p2), XITK_HV_V (p2),
        XITK_HV_H (e2), XITK_HV_V (e2),
        XITK_HV_H (e1), XITK_HV_V (e1));
    }
  } else {
    xitk_part_image_draw (wp->w.wl, &wp->paddle_skin, NULL,
      XITK_HV_H (wp->paddle_drawn_here.f), XITK_HV_V (wp->paddle_drawn_here.f),
      wp->paddle_drawn_here.w, wp->paddle_drawn_here.h,
      XITK_HV_H (p1), XITK_HV_V (p1));
  }
}

static void _xitk_slider_paint_d (_slider_private_t *wp) {
  /* no paddle just background when disabled. */
  wp->paddle_drawn_here.f.w = ~0u;
  wp->paddle_drawn_here.x = 0;
  wp->paddle_drawn_here.y = 0;
  wp->paddle_drawn_here.w = 0;
  wp->paddle_drawn_here.h = 0;
}

static void _xitk_slider_paint_hv (_slider_private_t *wp, const widget_event_t *event) {
  XITK_HV_INIT;

  if (wp->w.state & XITK_WIDGET_STATE_VISIBLE) {
    xitk_img_state_t state = xitk_image_find_state (wp->paddle_skin.num_states - 1, wp->w.state) + 1;


    if (!event) {
      if (wp->paddle_drawn_here.w > 0) {
        /* incremental */
        xitk_part_image_draw (wp->w.wl, &wp->bg_skin, NULL,
          wp->paddle_drawn_here.x, wp->paddle_drawn_here.y,
          wp->paddle_drawn_here.w, wp->paddle_drawn_here.h,
          XITK_HV_H (wp->w.pos) + wp->paddle_drawn_here.x, XITK_HV_V (wp->w.pos) + wp->paddle_drawn_here.y);
      } else {
        /* full */
        xitk_part_image_draw (wp->w.wl, &wp->bg_skin, NULL,
          0, 0,
          wp->bg_skin.width, wp->bg_skin.height,
          XITK_HV_H (wp->w.pos), XITK_HV_V (wp->w.pos));
      }
    } else {
      /* partial */
      xitk_part_image_draw (wp->w.wl, &wp->bg_skin, NULL,
        event->x - XITK_HV_H (wp->w.pos), event->y - XITK_HV_V (wp->w.pos),
        event->width, event->height,
        event->x, event->y);
    }
    wp->paddle_drawn_here.state = state;

    if ((wp->w.state & XITK_WIDGET_STATE_ENABLE) || (wp->paddle_skin.num_states >= 5)) {
      int x, y;

      x = wp->hv_info.h.max - wp->hv_info.h.visible;
      if (x > 0) {
        x = wp->hv_info.h.pos * (XITK_HV_H (wp->w.size) - wp->hv_w) / x;
        if (x > wp->hv_max_x)
          x = wp->hv_max_x;
        if (x < 0)
          x = 0;
      } else {
        x = 0;
      }
      y = wp->hv_info.v.max - wp->hv_info.v.visible;
      if (y > 0) {
        y = wp->hv_info.v.pos * (XITK_HV_V (wp->w.size) - wp->hv_h) / y;
        if (y > wp->hv_max_y)
          y = wp->hv_max_y;
        if (y < 0)
          y = 0;
      } else {
        y = 0;
      }
      wp->hv_x = x;
      wp->hv_y = y;
      wp->paddle_drawn_here.f.w = wp->paddle_state_list[state].w;
      wp->paddle_drawn_here.w = wp->hv_w;
      wp->paddle_drawn_here.h = wp->hv_h;
      wp->paddle_drawn_here.x = x;
      wp->paddle_drawn_here.y = y;

      _xitk_slider_paint_p (wp, event);
    }
  } else {
    _xitk_slider_paint_d (wp);
  }
}

static void _xitk_slider_paint_r (_slider_private_t *wp, const widget_event_t *event) {
  XITK_HV_INIT;

  if (wp->w.state & XITK_WIDGET_STATE_VISIBLE) {
    xitk_img_state_t state = xitk_image_find_state (wp->paddle_skin.num_states - 1, wp->w.state) + 1;
    int paddle_width, paddle_height;

    paddle_width = XITK_HV_H (wp->paddle_state_list[0]);
    paddle_height = XITK_HV_V (wp->paddle_state_list[0]);

    if (!event) {
      if (wp->paddle_drawn_here.w > 0) {
        /* incremental */
        xitk_part_image_draw (wp->w.wl, &wp->bg_skin, NULL,
          wp->paddle_drawn_here.x, wp->paddle_drawn_here.y,
          wp->paddle_drawn_here.w, wp->paddle_drawn_here.h,
          XITK_HV_H (wp->w.pos) + wp->paddle_drawn_here.x, XITK_HV_V (wp->w.pos) + wp->paddle_drawn_here.y);
      } else {
        /* full */
        xitk_part_image_draw (wp->w.wl, &wp->bg_skin, NULL, 0, 0, wp->bg_skin.width, wp->bg_skin.height,
            XITK_HV_H (wp->w.pos), XITK_HV_V (wp->w.pos));
      }
    } else {
      /* partial */
      xitk_part_image_draw (wp->w.wl, &wp->bg_skin, NULL,
        event->x - XITK_HV_H (wp->w.pos), event->y - XITK_HV_V (wp->w.pos),
        event->width, event->height,
        event->x, event->y);
    }
    wp->paddle_drawn_here.state = state;

    if ((wp->w.state & XITK_WIDGET_STATE_ENABLE) || (wp->paddle_skin.num_states >= 5)) {
      int xcenter = wp->bg_skin.width >> 1;
      int ycenter = wp->bg_skin.height >> 1;
      double angle = wp->angle;
      if (angle < M_PI / -2)
        angle = angle + M_PI * 2;
      wp->paddle_drawn_here.x = (int) (0.5 + xcenter + wp->radius * cos (angle)) - (paddle_width / 2);
      wp->paddle_drawn_here.y = (int) (0.5 + ycenter - wp->radius * sin (angle)) - (paddle_height / 2);
      wp->paddle_drawn_here.f.w = wp->paddle_state_list[state].w;
      wp->paddle_drawn_here.w = paddle_width;
      wp->paddle_drawn_here.h = paddle_height;
      _xitk_slider_paint_p (wp, event);
    }
  } else {
    _xitk_slider_paint_d (wp);
  }
}

static void _xitk_slider_paint_h (_slider_private_t *wp, const widget_event_t *event) {
  XITK_HV_INIT;

  if (wp->w.state & XITK_WIDGET_STATE_VISIBLE) {
    xitk_img_state_t state = xitk_image_find_state (wp->paddle_skin.num_states - 1, wp->w.state) + 1;
    int paddle_width, paddle_height, newpos;

    paddle_width = XITK_HV_H (wp->paddle_state_list[0]);
    paddle_height = XITK_HV_V (wp->paddle_state_list[0]);

    newpos = wp->bg_skin.width;
    if (!wp->bar_mode)
      newpos -= paddle_width;
    newpos = ((wp->value - wp->lower) * newpos + (wp->range0 >> 1)) / wp->range0;

    if (!event) {
      if (!(wp->w.state & XITK_WIDGET_STATE_ENABLE) && (wp->paddle_skin.num_states < 5))
        return;
      if (wp->paddle_drawn_here.h > 0) {
        /* incremental */
        if (wp->paddle_drawn_here.state == state) {
          if (wp->bar_mode) {
            /* nasty little shortcut ;-) */
            int oldpos = wp->paddle_drawn_here.w;

            if (newpos < oldpos) {
              xitk_part_image_draw (wp->w.wl, &wp->bg_skin, NULL,
                newpos, 0,
                oldpos - newpos, wp->bg_skin.height,
                XITK_HV_H (wp->w.pos) + newpos, XITK_HV_V (wp->w.pos));
            } else if (newpos > oldpos) {
              xitk_part_image_draw (wp->w.wl, &wp->paddle_skin, NULL,
                XITK_HV_H (wp->paddle_state_list[state]) + oldpos, XITK_HV_V (wp->paddle_state_list[state]),
                newpos - oldpos, paddle_height,
                XITK_HV_H (wp->w.pos) + oldpos, XITK_HV_V (wp->w.pos));
            }
            wp->paddle_drawn_here.w = newpos;
            return;
          } else {
            if (newpos == wp->paddle_drawn_here.x)
              return;
          }
        }
        xitk_part_image_draw (wp->w.wl, &wp->bg_skin, NULL,
          wp->paddle_drawn_here.x, wp->paddle_drawn_here.y,
          wp->paddle_drawn_here.w, wp->paddle_drawn_here.h,
          XITK_HV_H (wp->w.pos) + wp->paddle_drawn_here.x, XITK_HV_V (wp->w.pos) + wp->paddle_drawn_here.y);
      } else {
        /* full */
        xitk_part_image_draw (wp->w.wl, &wp->bg_skin, NULL, 0, 0, wp->bg_skin.width, wp->bg_skin.height,
            XITK_HV_H (wp->w.pos), XITK_HV_V (wp->w.pos));
      }
    } else {
      /* partial */
      xitk_part_image_draw (wp->w.wl, &wp->bg_skin, NULL,
        event->x - XITK_HV_H (wp->w.pos), event->y - XITK_HV_V (wp->w.pos),
        event->width, event->height,
        event->x, event->y);
    }
    wp->paddle_drawn_here.state = state;

    if ((wp->w.state & XITK_WIDGET_STATE_ENABLE) || (wp->paddle_skin.num_states >= 5)) {
      if (wp->bar_mode == 1) {
        wp->paddle_drawn_here.w = newpos;
        wp->paddle_drawn_here.x = 0;
      } else {
        wp->paddle_drawn_here.x = newpos;
        wp->paddle_drawn_here.w  = paddle_width;
      }
      wp->paddle_drawn_here.f.w = wp->paddle_state_list[state].w;
      wp->paddle_drawn_here.h = paddle_height;
      wp->paddle_drawn_here.y = 0;
      _xitk_slider_paint_p (wp, event);
    } else {
      wp->paddle_drawn_here.h = 0;
    }
  } else {
    _xitk_slider_paint_d (wp);
  }
}

static void _xitk_slider_paint_v (_slider_private_t *wp, const widget_event_t *event) {
  XITK_HV_INIT;

  if (wp->w.state & XITK_WIDGET_STATE_VISIBLE) {
    xitk_img_state_t state = xitk_image_find_state (wp->paddle_skin.num_states - 1, wp->w.state) + 1;
    int paddle_width, paddle_height, newpos;

    paddle_width = XITK_HV_H (wp->paddle_state_list[0]);
    paddle_height = XITK_HV_V (wp->paddle_state_list[0]);

    newpos = wp->bg_skin.height;
    if (!wp->bar_mode)
      newpos -= paddle_height;
    newpos = newpos - ((wp->value - wp->lower) * newpos + (wp->range0 >> 1)) / wp->range0;

    if (!event) {
      if (!(wp->w.state & XITK_WIDGET_STATE_ENABLE) && (wp->paddle_skin.num_states < 5))
        return;
      if (wp->paddle_drawn_here.w > 0) {
        /* incremental */
        if (wp->paddle_drawn_here.state == state) {
          if (wp->bar_mode) {
            /* nasty little shortcut ;-) */
            int oldpos = wp->paddle_drawn_here.y;

            if (newpos > oldpos) {
              xitk_part_image_draw (wp->w.wl, &wp->bg_skin, NULL,
                0, oldpos,
                wp->bg_skin.width, newpos - oldpos,
                XITK_HV_H (wp->w.pos), XITK_HV_V (wp->w.pos) + oldpos);
            } else if (newpos < oldpos) {
              xitk_part_image_draw (wp->w.wl, &wp->paddle_skin, NULL,
                XITK_HV_H (wp->paddle_state_list[state]), XITK_HV_V (wp->paddle_state_list[state]) + newpos,
                paddle_width, oldpos - newpos,
                XITK_HV_H (wp->w.pos), XITK_HV_V (wp->w.pos) + newpos);
            }
            wp->paddle_drawn_here.y = newpos;
            wp->paddle_drawn_here.h = paddle_height - newpos;
            return;
          } else {
            if (newpos == wp->paddle_drawn_here.y)
              return;
          }
        }
        xitk_part_image_draw (wp->w.wl, &wp->bg_skin, NULL,
          wp->paddle_drawn_here.x, wp->paddle_drawn_here.y,
          wp->paddle_drawn_here.w, wp->paddle_drawn_here.h,
          XITK_HV_H (wp->w.pos) + wp->paddle_drawn_here.x, XITK_HV_V (wp->w.pos) + wp->paddle_drawn_here.y);
      } else {
        /* full */
        xitk_part_image_draw (wp->w.wl, &wp->bg_skin, NULL, 0, 0, wp->bg_skin.width, wp->bg_skin.height,
            XITK_HV_H (wp->w.pos), XITK_HV_V (wp->w.pos));
      }
    } else {
      /* partial */
      xitk_part_image_draw (wp->w.wl, &wp->bg_skin, NULL,
        event->x - XITK_HV_H (wp->w.pos), event->y - XITK_HV_V (wp->w.pos),
        event->width, event->height,
        event->x, event->y);
    }
    wp->paddle_drawn_here.state = state;

    if ((wp->w.state & XITK_WIDGET_STATE_ENABLE) || (wp->paddle_skin.num_states >= 5)) {
      wp->paddle_drawn_here.x = 0;
      wp->paddle_drawn_here.y = newpos;
      wp->paddle_drawn_here.w = paddle_width;
      wp->paddle_drawn_here.h = paddle_height;
      wp->paddle_drawn_here.f.w = wp->paddle_state_list[state].w;
      if (wp->bar_mode) {
        XITK_HV_V (wp->paddle_drawn_here.f) += newpos;
        wp->paddle_drawn_here.h -= newpos;
      }
      _xitk_slider_paint_p (wp, event);
    } else {
      wp->paddle_drawn_here.w = 0;
    }
  } else {
    _xitk_slider_paint_d (wp);
  }
}

static void _xitk_slider_set_paint (_slider_private_t *wp) {
  if (!wp->bg_skin.width || !wp->paddle_skin.image || !wp->bg_skin.image)
    wp->paint = _xitk_slider_paint_n;
  else if (wp->sType == XITK_HVSLIDER)
    wp->paint = _xitk_slider_paint_hv;
  else if (wp->sType == XITK_RSLIDER)
    wp->paint = _xitk_slider_paint_r;
  else if (wp->sType == XITK_HSLIDER)
    wp->paint = _xitk_slider_paint_h;
  else if (wp->sType == XITK_VSLIDER)
    wp->paint = _xitk_slider_paint_v;
  else
    wp->paint = _xitk_slider_paint_n;
}

static void _xitk_slider_set_skin (_slider_private_t *wp, xitk_skin_config_t *skonfig) {
  const xitk_skin_element_info_t *s = xitk_skin_get_info (skonfig, wp->w.skin_element_name);
  XITK_HV_INIT;
  /* always be there for the application. if this skin does not use us, disable user side. */
  wp->paddle_drawn_here.f.w = ~0u;
  if (s && s->pixmap_img.image && s->slider_pixmap_pad_img.image) {
    XITK_HV_H (wp->w.pos) = s->x;
    XITK_HV_V (wp->w.pos) = s->y;
    xitk_widget_state_from_info (&wp->w, s);
    /* keep the "wrap" property. */
    wp->sType       = (xitk_slider_type_t)((uint32_t)s->slider_type & ((uint32_t)XITK_WRAP_VSLIDER - (uint32_t)XITK_VSLIDER - 1));
    wp->radius      = s->slider_radius;
    wp->bg_skin     = s->pixmap_img;
    wp->paddle_skin = s->slider_pixmap_pad_img;
  } else {
    wp->w.pos.w     = 0;
    wp->w.state    &= ~(XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);
    wp->sType       = XITK_VSLIDER;
    wp->radius      = 8;
    wp->bg_skin.x      = 0;
    wp->bg_skin.y      = 0;
    wp->bg_skin.width  = 0;
    wp->bg_skin.height = 0;
    wp->bg_skin.image  = NULL;
    wp->paddle_skin.x      = 0;
    wp->paddle_skin.y      = 0;
    wp->paddle_skin.width  = 0;
    wp->paddle_skin.height = 0;
    wp->paddle_skin.image  = NULL;
  }
  _xitk_slider_set_paint (wp);
}

/*
 *
 */
static void _xitk_slider_new_skin (_slider_private_t *wp, xitk_skin_config_t *skonfig) {
  XITK_HV_INIT;

  if (wp->w.skin_element_name[0]) {
    _xitk_slider_set_skin (wp, skonfig);
    xitk_set_widget_pos (&wp->w, XITK_HV_H (wp->w.pos), XITK_HV_V (wp->w.pos));
    _xitk_slider_update_skin (wp);
  }
}

/*
 * Got click
 */
static int _xitk_slider_click (_slider_private_t *wp, int button, int bUp, int x, int y) {
  XITK_HV_INIT;

  if (button == 1) {
    /* NOTE: sliders are special. while holding them with left mouse, they will receive move
     * and unclick events even if mouse has left the slider_s host window. */
    if ((wp->w.state & XITK_WIDGET_STATE_FOCUS) || bUp) {
      int moved, value;

      if (wp->sType == XITK_HVSLIDER) {
        /* if this thing really moves x and y, use xitk_slider_hv_sync (). */
        moved = _xitk_slider_report_xy (wp, x, y);
        value = wp->hv_max_y > 0 ? wp->upper - wp->hv_info.v.pos : wp->hv_info.h.pos;
      } else {
        moved = _xitk_slider_update_xy (wp, x - XITK_HV_H (wp->w.pos), y - XITK_HV_V (wp->w.pos));
        value = wp->value;
      }

      if (!bUp) {
        wp->deferred_pos = _INVALID_POS;
        if (moved || !(wp->w.state & XITK_WIDGET_STATE_CLICK)) {
          wp->w.state |= XITK_WIDGET_STATE_CLICK;
          wp->paint (wp, NULL);
          wp->w.shown_state = wp->w.state;
        }
        /* Exec motion callback function (if available) */
        if (moved && wp->motion_callback)
          wp->motion_callback (&wp->w, wp->w.userdata, value);
      } else {
        if (wp->deferred_pos != _INVALID_POS) {
          value = wp->deferred_pos;
          if (wp->sType == XITK_HVSLIDER) {
            if (wp->hv_max_y > 0) {
              value = wp->upper - value;
              if (wp->hv_info.v.pos != value) {
                wp->hv_info.v.pos = value;
                moved = 1;
              }
            } else {
              if (wp->hv_info.h.pos != value) {
                wp->hv_info.h.pos = value;
                moved = 1;
              }
            }
            if ((value >= wp->lower) && (value <= wp->upper))
              wp->value = value;
          } else {
            if (_xitk_slider_update_value (wp, value))
              moved = 1;
          }
          wp->deferred_pos = _INVALID_POS;
          value = wp->value;
        }
        if (moved || (wp->w.state & XITK_WIDGET_STATE_CLICK)) {
          wp->w.state &= ~XITK_WIDGET_STATE_CLICK;
          wp->paint (wp, NULL);
          wp->w.shown_state = wp->w.state;
        }
        if (wp->callback)
          wp->callback (&wp->w, wp->w.userdata, value);
      }
    }
    return 1;
  }
  return 0;
}

/*
 * Return current position.
 */
static int _xitk_slider_get_pos (_slider_private_t *wp) {
  return wp->sType == XITK_HVSLIDER
         ? (wp->hv_max_y > 0 ? wp->upper - wp->hv_info.v.pos : wp->hv_info.h.pos)
         : wp->value;
}

/*
 * Set position.
 */
static void _xitk_slider_set_pos (_slider_private_t *wp, int pos) {
  if (!(wp->w.state & XITK_WIDGET_STATE_CLICK)) {
    if (wp->sType == XITK_HVSLIDER) {
      if (wp->hv_max_y > 0) {
        pos = wp->upper - pos;
        if (wp->hv_info.v.pos != pos) {
          wp->hv_info.v.pos = pos;
          wp->paint (wp, NULL);
        }
      } else {
        if (wp->hv_info.h.pos != pos) {
          wp->hv_info.h.pos = pos;
          wp->paint (wp, NULL);
        }
      }
      if ((pos >= wp->lower) && (pos <= wp->upper))
        wp->value = pos;
    } else {
      if (_xitk_slider_update_value (wp, pos))
        wp->paint (wp, NULL);
    }
  } else {
    /* while user is holding the slider, remember application moves,
     * and apply them later when user ungrabs it. */
    wp->deferred_pos = pos;
  }
}

static int _xitk_slider_key (_slider_private_t *wp, const char *string, int modifier) {
  static const struct {int8_t x, y;} tab_ctrl[XITK_KEY_LASTCODE + 1][6] = {
    /*                        norm      shift    ctrl    hv norm  hv shift hv ctrl */
    [XITK_KEY_LEFT]        = {{-1, 0}, {-3, 0}, {-5, 0}, {-1, 0}, {-2, 0}, {-3, 0}},
    [XITK_KEY_RIGHT]       = {{ 1, 0}, { 3, 0}, { 5, 0}, { 1, 0}, { 2, 0}, { 3, 0}},
    [XITK_KEY_HOME]        = {{-5, 5}, { 0, 0}, { 0, 0}, { 0,-3}, {-3, 0}, {-3,-3}},
    [XITK_KEY_END]         = {{ 5,-5}, { 0, 0}, { 0, 0}, { 0, 3}, { 3, 0}, { 3, 3}},
    [XITK_KEY_PREV]        = {{-3, 3}, { 0, 0}, { 0, 0}, { 0,-2}, {-2, 0}, {-2,-2}},
    [XITK_KEY_NEXT]        = {{ 3,-3}, { 0, 0}, { 0, 0}, { 0, 2}, { 2, 0}, { 2, 2}},
    [XITK_KEY_UP]          = {{ 0, 1}, { 0, 3}, { 0, 5}, { 0,-1}, { 0,-2}, { 0,-3}},
    [XITK_KEY_DOWN]        = {{ 0,-1}, { 0,-3}, { 0,-5}, { 0, 1}, { 0, 2}, { 0, 3}},
    [XITK_KEY_DELETE]      = {{15,15}, { 0, 0}, { 0, 0}, { 0,15}, {15, 0}, {15,15}},
    [XITK_MOUSE_WHEEL_UP]  = {{-1, 1}, {-3, 3}, { 0, 0}, { 0,-1}, {-1, 0}, {-1,-1}},
    [XITK_MOUSE_WHEEL_DOWN]= {{ 1,-1}, { 3,-3}, { 0, 0}, { 0, 1}, { 1, 0}, { 1, 1}},
    [XITK_KEY_LASTCODE]    = {{ 1, 1}, { 0, 0}, { 0, 0}, { 0, 1}, { 1, 0}, { 1, 1}}
  };
  uint32_t md;
  int v, dx, dy;

  if (!(wp->w.state & XITK_WIDGET_STATE_ENABLE) || !string)
    return 0;

  md = modifier & ~MODIFIER_NUML;
  if ((md & ~(MODIFIER_SHIFT | MODIFIER_CTRL)) != MODIFIER_NOMOD)
    return 0;
  md = (modifier & MODIFIER_CTRL) ? 2
     : (modifier & MODIFIER_SHIFT) ? 1
     : 0;
  md += (wp->sType == XITK_HVSLIDER) * 3;

  if (string[0] == XITK_CTRL_KEY_PREFIX) {
    uint32_t z = (uint8_t)string[1];
    if (z >= XITK_KEY_LASTCODE)
      return 0;
    dx = tab_ctrl[z][md].x;
    dy = tab_ctrl[z][md].y;
  } else {
    uint32_t z = (uint8_t)string[0] ^ '0';
    if (z >= 10)
      return 0;
    z += 10;
    dx = z * tab_ctrl[XITK_KEY_LASTCODE][md].x;
    dy = z * tab_ctrl[XITK_KEY_LASTCODE][md].y;
  }

  v = 0;
  if (wp->sType == XITK_HSLIDER)
    v = dx;
  else if (wp->sType == XITK_VSLIDER)
    v = dy;
  else if (wp->sType == XITK_RSLIDER)
    v = dy ? dy : dx;
  if (v) {
    if (v >= 10)
      v = wp->lower + (wp->range0 * (v - 10) + 5) / 10;
    else if (v == -5)
      v = wp->lower;
    else if (v == 5)
      v = wp->upper;
    else
      v = wp->value + v * wp->step;
    if (_xitk_slider_update_value (wp, v)) {
      wp->paint (wp, NULL);
      if (wp->callback)
        wp->callback (&wp->w, wp->w.userdata, wp->value);
      else if (wp->motion_callback)
        wp->motion_callback (&wp->w, wp->w.userdata, wp->value);
    }
    return 1;
  }

  if (wp->sType == XITK_HVSLIDER) {
    xitk_slider_hv_t info;
    int m;

    if (wp->hv_max_x <= 0) {
        dy |= dx;
        dx = 0;
    } else if (wp->hv_max_y <= 0) {
        dx |= dy;
        dy = 0;
    }
    if (!(dx | dy))
      return 0;
    info = wp->hv_info;

    m = info.h.max - info.h.visible;
    dx = dx == -3 ? 0
       : dx == -2 ? info.h.pos - info.h.visible
       : dx == -1 ? info.h.pos - info.h.step
       : dx ==  1 ? info.h.pos + info.h.step
       : dx ==  2 ? info.h.pos + info.h.visible
       : dx ==  3 ? m
       : dx >= 10 ? (m * (dx - 10) + 5) / 10
       : info.h.pos;
    /* we cannot get too far negative here. */
    m += 1;
    v = (uint32_t)(dx + m) % (uint32_t)m;
    if ((dx != v) && !wp->wrap)
      v = (dx < 0) ? 0 : info.h.max - info.h.visible;
    info.h.pos = v;

    m = info.v.max - info.v.visible;
    dy = dy == -3 ? 0
       : dy == -2 ? info.v.pos - info.v.visible
       : dy == -1 ? info.v.pos - info.v.step
       : dy ==  1 ? info.v.pos + info.v.step
       : dy ==  2 ? info.v.pos + info.v.visible
       : dy ==  3 ? m
       : dy >= 10 ? (m * (dy - 10) + 5) / 10
       : info.v.pos;
    m += 1;
    v = (uint32_t)(dy + m) % (uint32_t)m;
    if ((dy != v) && !wp->wrap)
      v = (dy < 0) ? 0 : info.v.max - info.v.visible;
    info.v.pos = v;

    dx = wp->hv_info.h.pos;
    dy = wp->hv_info.v.pos;
    xitk_slider_hv_sync (&wp->w, &info, XITK_SLIDER_SYNC_SET_AND_PAINT);
    if ((dx != wp->hv_info.h.pos) || (dy != wp->hv_info.v.pos)) {
      v = wp->hv_max_y > 0 ? wp->hv_info.v.max - wp->hv_info.v.visible - wp->hv_info.v.pos : wp->hv_info.h.pos;
      if (wp->callback)
        wp->callback (&wp->w, wp->w.userdata, v);
      else if (wp->motion_callback)
        wp->motion_callback (&wp->w, wp->w.userdata, v);
    }
    return 1;
  }

  return 0;
}

static int xitk_slider_event (xitk_widget_t *w, const widget_event_t *event) {
  _slider_private_t *wp;

  xitk_container (wp, w, w);
  if (!wp || !event)
    return 0;
  if ((wp->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_SLIDER)
    return 0;

  switch (event->type) {
    case WIDGET_EVENT_PAINT:
      wp->paint (wp, event);
      return 0;
    case WIDGET_EVENT_SELECT:
      if (event->button != XITK_INT_KEEP) {
        int v = event->button;
        switch (v) {
          case XITK_SLIDER_MIN: v = wp->lower; break;
          case XITK_SLIDER_MAX: v = wp->upper; break;
          case XITK_SLIDER_PLUS: v = xitk_min (_xitk_slider_get_pos (wp) + wp->step, wp->upper); break;
          case XITK_SLIDER_MINUS: v = xitk_max (_xitk_slider_get_pos (wp) - wp->step, wp->lower); break;
	  default: ;
        }
        _xitk_slider_set_pos (wp, v);
      }
      return _xitk_slider_get_pos (wp);
    case WIDGET_EVENT_KEY:
      return event->pressed ? _xitk_slider_key (wp, event->string, event->modifier) : 0;
    case WIDGET_EVENT_CLICK:
      return _xitk_slider_click (wp, event->button, !event->pressed, event->x, event->y);
    case WIDGET_EVENT_INSIDE:
      if (wp->w.state & XITK_WIDGET_STATE_VISIBLE) {
        XITK_HV_INIT;
        xitk_part_image_t *skin = (wp->bar_mode == 1) ? &wp->paddle_skin : &wp->bg_skin;
        return xitk_image_inside (skin->image,
          skin->x + event->x - XITK_HV_H (wp->w.pos),
          skin->y + event->y - XITK_HV_V (wp->w.pos)) ? 1 : 2;
      }
      return 2;
    case WIDGET_EVENT_CHANGE_SKIN:
      _xitk_slider_new_skin (wp, event->skonfig);
      return 0;
    case WIDGET_EVENT_DESTROY:
      if (!wp->w.skin_element_name[0]) {
        xitk_image_free_image (&(wp->paddle_skin.image));
        xitk_image_free_image (&(wp->bg_skin.image));
      }
      return 0;
    case WIDGET_EVENT_GET_SKIN:
      if (event->image) {
        *event->image = (event->skin_layer == FOREGROUND_SKIN) ? wp->paddle_skin.image
                      : (event->skin_layer == BACKGROUND_SKIN) ? wp->bg_skin.image
                      : NULL;
        return 1;
      }
      return 0;
    default:
      return 0;
  }
}

static void _xitk_slider_range (_slider_private_t *wp) {
  wp->range0 = wp->upper - wp->lower;
  wp->range1 = wp->range0 + 1;
  wp->range0 += (uint32_t)(wp->range0 - 1) >> 31;
}

/*
 * Set value MIN.
 */
void xitk_slider_set_range (xitk_widget_t *w, int min, int max, int step) {
  _slider_private_t *wp;

  xitk_container (wp, w, w);
  if (wp && ((wp->w.type & WIDGET_TYPE_MASK) == WIDGET_TYPE_SLIDER)) {
    wp->step = step;
    if (((wp->upper ^ max) | (wp->lower ^ min)) == 0)
      return;
    if (min > max) {
      XITK_WARNING ("%s@%d: slider min value > max value !\n", __FILE__, __LINE__);
      return;
    }
    wp->upper = max;
    wp->lower = min;
    _xitk_slider_range (wp);
    _xitk_slider_update_value (wp, wp->value);
  }
}

void xitk_slider_hv_sync (xitk_widget_t *w, xitk_slider_hv_t *info, xitk_slider_sync_t mode) {
  _slider_private_t *wp;
  XITK_HV_INIT;

  xitk_container (wp, w, w);
  if (!wp || !info)
    return;
  if ((wp->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_SLIDER)
    return;
  if (wp->sType != XITK_HVSLIDER)
    return;

  if (mode == XITK_SLIDER_SYNC_GET) {
    *info = wp->hv_info;
  } else if ((mode == XITK_SLIDER_SYNC_SET) || (mode == XITK_SLIDER_SYNC_SET_AND_PAINT)) {
    if (!wp->w.skin_element_name[0] &&
      ((info->h.visible != wp->hv_info.h.visible) || (info->h.max != wp->hv_info.h.max) ||
       (info->v.visible != wp->hv_info.v.visible) || (info->v.max != wp->hv_info.v.max))) {
      int hv_w, hv_h;
      int no_h = (info->h.visible >= info->h.max) || (info->h.max <= 0);
      int no_v = (info->v.visible >= info->v.max) || (info->v.max <= 0);

      if (no_h) {
        hv_w = XITK_HV_H (wp->w.size);
      } else {
        int minw = no_v ? (XITK_HV_V (wp->w.size) * 5) >> 2 : 10;
        hv_w = XITK_HV_H (wp->w.size) * info->h.visible / info->h.max;
        if (hv_w < minw) {
          hv_w = minw;
          if (hv_w > XITK_HV_H (wp->w.size))
            hv_w = XITK_HV_H (wp->w.size);
        }
      }
      if (no_v) {
        hv_h = XITK_HV_V (wp->w.size);
      } else {
        int minh = no_h ? (XITK_HV_H (wp->w.size) * 5) >> 2 : 10;
        hv_h = XITK_HV_V (wp->w.size) * info->v.visible / info->v.max;
        if (hv_h < minh) {
          hv_h = minh;
          if (hv_h > XITK_HV_V (wp->w.size))
            hv_h = XITK_HV_V (wp->w.size);
        }
      }
      if ((hv_w != wp->hv_w) || (hv_h != wp->hv_h)) {
        xitk_image_draw_paddle_three_state (wp->paddle_skin.image, hv_w, hv_h);
        wp->paddle_skin.width = 3 * hv_w;
        wp->paddle_skin.height = hv_h;
        _xitk_slider_update_skin (wp);
      }
    }

    wp->hv_info = *info;
    wp->lower = 0;
    wp->upper = wp->hv_max_y > 0
              ? (wp->hv_info.v.visible >= wp->hv_info.v.max ? 0 : wp->hv_info.v.max - wp->hv_info.v.visible)
              : (wp->hv_info.h.visible >= wp->hv_info.h.max ? 0 : wp->hv_info.h.max - wp->hv_info.h.visible);
    _xitk_slider_range (wp);
    wp->step = wp->hv_max_y > 0 ? wp->hv_info.v.step : wp->hv_info.h.step;
    if (mode == XITK_SLIDER_SYNC_SET_AND_PAINT)
      wp->paint (wp, NULL);
  }
}

/*
 * Create the widget
 */
static xitk_widget_t *_xitk_slider_create (_slider_private_t *wp, const xitk_slider_widget_t *s) {
  XITK_HV_INIT;

  wp->upper        = (s->min > s->max) ? s->min : s->max;
  wp->lower        = s->min;
  _xitk_slider_range (wp);
  wp->step         = s->step;
  wp->angle        = 0.0;
  wp->deferred_pos = _INVALID_POS;
  _xitk_slider_set_pos (wp, s->value);

  _xitk_slider_update_skin (wp);
#if 0
  wp->hv_info.h.step     = 0;
  wp->hv_info.v.step     = 0;
  wp->hv_w               = 0;
  wp->hv_h               = 0;
  wp->paddle_drawn_here.x = 0;
  wp->paddle_drawn_here.y = 0;
  wp->paddle_drawn_here.w = 0;
  wp->paddle_drawn_here.h = 0;
#endif
  wp->hv_info.h.pos      = -1;
  wp->hv_info.h.visible  = -1;
  wp->hv_info.h.max      = -1;
  wp->hv_info.v.pos      = -1;
  wp->hv_info.v.visible  = -1;
  wp->hv_info.v.max      = -1;
  wp->paddle_drawn_here.f.w = ~0u;

  wp->motion_callback    = s->motion_callback;
  wp->callback           = s->callback;

  XITK_HV_H (wp->w.size) = wp->bg_skin.width;
  XITK_HV_V (wp->w.size) = wp->bg_skin.height;
  wp->w.type             = WIDGET_TYPE_SLIDER | WIDGET_FOCUSABLE | WIDGET_TABABLE | WIDGET_CLICKABLE
                         | WIDGET_KEEP_FOCUS | WIDGET_KEYABLE | WIDGET_PARTIAL_PAINTABLE;
  wp->w.event            = xitk_slider_event;

  _xitk_new_widget_apply (&s->nw, &wp->w);

  return &wp->w;
}

/*
 * Create the widget
 */
xitk_widget_t *xitk_slider_create (const xitk_slider_widget_t *s, xitk_skin_config_t *skonfig) {
  _slider_private_t *wp;

  wp = (_slider_private_t *)xitk_widget_new (&s->nw, sizeof (*wp));
  if (!wp)
    return NULL;

  _xitk_slider_set_type (wp, s->type);
  _xitk_slider_set_skin (wp, skonfig);

  return _xitk_slider_create (wp, s);
}

/*
 * Create the widget (not skined).
 */
xitk_widget_t *xitk_noskin_slider_create (const xitk_slider_widget_t *s, int x, int y, int width, int height) {
  _slider_private_t *wp;
  XITK_HV_INIT;

  wp = (_slider_private_t *)xitk_widget_new (&s->nw, sizeof (*wp));
  if (!wp)
    return NULL;

  _xitk_slider_set_type (wp, s->type);
  wp->w.skin_element_name[0] = 0;

#if 0
  wp->paddle_skin.x = 0;
  wp->paddle_skin.y = 0;
#endif
  wp->paddle_skin.num_states = 3;
  if (wp->sType == XITK_VSLIDER) {
    wp->paddle_skin.image = xitk_image_new (wp->w.wl->xitk, NULL, 0, width * 3, height / 5);
    xitk_image_draw_paddle_three_state_vertical (wp->paddle_skin.image);
  } else if (wp->sType == XITK_HSLIDER) {
    wp->paddle_skin.image = xitk_image_new (wp->w.wl->xitk, NULL, 0, (width / 5) * 3, height);
    xitk_image_draw_paddle_three_state_horizontal (wp->paddle_skin.image);
  } else if (wp->sType == XITK_RSLIDER) {
    int w = ((((width + height) >> 1) >> 1) / 10) * 3;
    wp->paddle_skin.image = xitk_image_new (wp->w.wl->xitk, NULL, 0, (w * 3), w);
    xitk_image_draw_paddle_rotate (wp->paddle_skin.image);
  } else if (wp->sType == XITK_HVSLIDER) {
    wp->paddle_skin.image = xitk_image_new (wp->w.wl->xitk, NULL, 0, width * 3, height);
    /* defer init to xitk_slider_hv_sync (). */
  } else {
    XITK_DIE ("Slider type unhandled.\n");
  }
  wp->paddle_skin.width = wp->paddle_skin.image->width;
  wp->paddle_skin.height = wp->paddle_skin.image->height;

  wp->bg_skin.image = xitk_image_new (wp->w.wl->xitk, NULL, 0, width, height);
  if ((wp->sType == XITK_HSLIDER) || (wp->sType == XITK_VSLIDER) || (wp->sType == XITK_HVSLIDER))
    xitk_image_draw_relief (wp->bg_skin.image, width, height, XITK_DRAW_INNER);
  else if (wp->sType == XITK_RSLIDER) {
    xitk_image_draw_rotate_button (wp->bg_skin.image);
  }
#if 0
  wp->bg_skin.x = 0;
  wp->bg_skin.y = 0;
#endif
  wp->bg_skin.width = wp->bg_skin.image->width;
  wp->bg_skin.height = wp->bg_skin.image->height;

  wp->radius = (wp->bg_skin.height >> 1) - (wp->paddle_skin.height);

  _xitk_slider_set_paint (wp);
  wp->w.state &= ~(XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);
  XITK_HV_H (wp->w.pos) = x;
  XITK_HV_V (wp->w.pos) = y;

  return _xitk_slider_create (wp, s);
}
