# -*- coding:binary -*-

require 'spec_helper'
require 'rex/proto/kerberos/pac/krb5_pac'
require 'rex/proto/kerberos/pac/krb5_pac_element_type'

RSpec.shared_examples 'ticket' do
  before do
    Timecop.freeze(Time.parse('Jul 15, 2022 12:33:40.000000000 GMT'))
  end

  after do
    Timecop.return
  end

  let(:secs_in_day) { 60 * 60 * 24 }

  let(:current_time) { Time.now }
  let(:domain_sid) { 'S-1-5-21-1431996162-3430639357-380615927' }
  let(:domain) { 'DW.LOCAL' }
  let(:spn) { ['mysql', 'dw.local'] }
  let(:username) { 'fake_mysql' }


  let(:flags) do
    [
      Rex::Proto::Kerberos::Model::TicketFlags::FORWARDABLE,
      Rex::Proto::Kerberos::Model::TicketFlags::PROXIABLE,
      Rex::Proto::Kerberos::Model::TicketFlags::RENEWABLE,
      Rex::Proto::Kerberos::Model::TicketFlags::PRE_AUTHENT
    ]
  end


  describe '#create_ticket' do


    subject(:kerberos_ccache) do
      mod = ::Msf::Exploit.new
      mod.extend ::Msf::Exploit::Remote::Kerberos::Client
      mod.extend ::Msf::Exploit::Remote::Kerberos::Ticket
      mod.send(:initialize)
      mod.forge_ticket(
        enc_key: ticket_key,
        enc_type: enc_type,
        start_time: current_time,
        end_time: current_time + secs_in_day * 3650,
        domain_sid: domain_sid,
        domain: domain,
        sname: spn,
        username: username,
        flags: Rex::Proto::Kerberos::Model::TicketFlags.from_flags(flags),
        ticket_checksum: create_ticket_checksum,
        session_key: session_key
      )
    end

    subject(:kerberos_ticket) do
      Rex::Proto::Kerberos::Model::Ticket.decode(kerberos_ccache.credentials.first.ticket.to_s)
    end

    it { is_expected.to be_a(Rex::Proto::Kerberos::Model::Ticket) }

    describe 'decoding ticket' do
      subject(:ticket) { kerberos_ticket }

      it { expect(subject.realm.to_s).to eq(domain) }
      it { expect(subject.tkt_vno).to eq(Rex::Proto::Kerberos::Model::VERSION) }
      it { expect(subject.sname.name_string.length).to eq(spn.length) }
      it { expect(subject.sname.name_string).to include(*spn) }
      it { expect(subject.enc_part).to be_a(Rex::Proto::Kerberos::Model::EncryptedData) }
    end

    describe 'decrypting ticket' do
      subject(:ticket_enc_part) { kerberos_ticket.enc_part }

      it { expect { subject.decrypt_asn1(ticket_key, Rex::Proto::Kerberos::Crypto::KeyUsage::KDC_REP_TICKET) }.not_to raise_error }
      it { expect(subject.decrypt_asn1(ticket_key, Rex::Proto::Kerberos::Crypto::KeyUsage::KDC_REP_TICKET)).to be_a(String) }

      describe 'decoding ticket enc part' do
        subject(:decrypted_ticket_part) { ticket_enc_part.decrypt_asn1(ticket_key, Rex::Proto::Kerberos::Crypto::KeyUsage::KDC_REP_TICKET) }
        it { expect { Rex::Proto::Kerberos::Model::TicketEncPart.decode(subject) }.not_to raise_error }
        it { expect(Rex::Proto::Kerberos::Model::TicketEncPart.decode(subject)).to be_a(Rex::Proto::Kerberos::Model::TicketEncPart) }

        describe 'ticket enc part' do
          let(:expected_flags) do
            [
              Rex::Proto::Kerberos::Model::TicketFlags::FORWARDABLE,
              Rex::Proto::Kerberos::Model::TicketFlags::PROXIABLE,
              Rex::Proto::Kerberos::Model::TicketFlags::RENEWABLE,
              Rex::Proto::Kerberos::Model::TicketFlags::PRE_AUTHENT
            ]
          end

          subject { Rex::Proto::Kerberos::Model::TicketEncPart.decode(decrypted_ticket_part) }

          describe 'flags' do
            it { expect(subject.flags).to include(*expected_flags) }
          end

          describe 'key' do
            it { expect(subject.key.type).to eq(enc_type) }
            it { expect(subject.key.value).to be_a(String) }
            it { expect(subject.key.value).not_to eq(ticket_key) }
            it { expect(subject.key.value.length).to eq(key_length) }
          end

          describe 'crealm' do
            it { expect(subject.crealm).to eq(domain) }
          end

          describe 'cname' do
            it { expect(subject.cname.name_type).to eq(Rex::Proto::Kerberos::Model::NameType::NT_PRINCIPAL) }
            it { expect(subject.cname.name_string).to include(username) }
          end

          describe 'transited' do
            it { expect(subject.transited).to be_a(Rex::Proto::Kerberos::Model::TransitedEncoding) }
          end

          describe 'authtime' do
            it { expect(subject.authtime).to be_a(Time) }
            it { expect(subject.authtime).to eq(current_time) }
          end

          describe 'starttime' do
            it { expect(subject.starttime).to be_a(Time) }
            it { expect(subject.starttime).to eq(current_time) }
          end

          describe 'endtime' do
            it { expect(subject.endtime).to be_a(Time) }
            it { expect(subject.endtime).to be > current_time }
          end

          describe 'renew_till' do
            it { expect(subject.renew_till).to be_a(Time) }
            it { expect(subject.renew_till).to be > current_time }
          end

          describe 'authorization_data' do
            it { expect(subject.authorization_data.elements.length).to eq(1) }
            it { expect(subject.authorization_data.elements[0][:type]).to eq(Rex::Proto::Kerberos::Model::AuthorizationDataType::AD_IF_RELEVANT) }
            it { expect { Rex::Proto::Kerberos::Model::AuthorizationData.decode(subject.authorization_data.elements[0][:data]) }.not_to raise_error }
            it { expect(Rex::Proto::Kerberos::Model::AuthorizationData.decode(subject.authorization_data.elements[0][:data])).to be_a(Rex::Proto::Kerberos::Model::AuthorizationData) }
          end
        end
      end
    end

    describe 'pac_auth_data contents' do
      let(:ticket) { kerberos_ticket }
      let(:ticket_enc_part) { kerberos_ticket.enc_part }
      let(:decrypted_ticket_part) { ticket_enc_part.decrypt_asn1(ticket_key, Rex::Proto::Kerberos::Crypto::KeyUsage::KDC_REP_TICKET) }
      let(:decoded_ticket_part) { Rex::Proto::Kerberos::Model::TicketEncPart.decode(decrypted_ticket_part) }

      subject(:pac_auth_data) { Rex::Proto::Kerberos::Model::AuthorizationData.decode(decoded_ticket_part.authorization_data.elements[0][:data]) }

      it { expect(subject.elements.length).to eq(1) }
      it { expect(subject.elements[0][:type]).to eq(Rex::Proto::Kerberos::Pac::AD_WIN2K_PAC) }
      it { expect(subject.elements[0][:data]).to eq(pac_type) }
    end
  end
end

RSpec.describe Msf::Exploit::Remote::Kerberos::Ticket do
  context 'when using an nthash' do
    let(:nt_hash) { '88E4D9FABAECF3DEC18DD80905521B29' }
    let(:ticket_key) { [nt_hash].pack('H*') }
    let(:key_length) { ticket_key.length }
    let(:enc_type) { Rex::Proto::Kerberos::Crypto::Encryption::RC4_HMAC }
    let(:create_ticket_checksum) { false }
    let(:session_key) { nil }
    let(:pac_type) do
      "\x06\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\xb8\x01\x00\x00" \
      "\x68\x00\x00\x00\x00\x00\x00\x00\x0a\x00\x00\x00\x1e\x00\x00\x00" \
      "\x20\x02\x00\x00\x00\x00\x00\x00\x12\x00\x00\x00\x1c\x00\x00\x00" \
      "\x40\x02\x00\x00\x00\x00\x00\x00\x11\x00\x00\x00\x08\x00\x00\x00" \
      "\x60\x02\x00\x00\x00\x00\x00\x00\x06\x00\x00\x00\x14\x00\x00\x00" \
      "\x68\x02\x00\x00\x00\x00\x00\x00\x07\x00\x00\x00\x14\x00\x00\x00" \
      "\x80\x02\x00\x00\x00\x00\x00\x00\x01\x10\x08\x00\xcc\xcc\xcc\xcc" \
      "\xa8\x01\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\xea\xc1\x1c" \
      "\x47\x98\xd8\x01\xff\xff\xff\xff\xff\xff\xff\x7f\xff\xff\xff\xff" \
      "\xff\xff\xff\x7f\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\xff\xff\xff\xff\xff\xff\xff\x7f\x14\x00\x14\x00" \
      "\x02\x00\x00\x00\x00\x00\x00\x00\x03\x00\x00\x00\x00\x00\x00\x00" \
      "\x04\x00\x00\x00\x00\x00\x00\x00\x05\x00\x00\x00\x00\x00\x00\x00" \
      "\x06\x00\x00\x00\x00\x00\x00\x00\x07\x00\x00\x00\x00\x00\x00\x00" \
      "\xf4\x01\x00\x00\x01\x02\x00\x00\x05\x00\x00\x00\x08\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x09\x00\x00\x00\x10\x00\x10\x00" \
      "\x0a\x00\x00\x00\x0b\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x10\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x0a\x00\x00\x00\x00\x00\x00\x00\x0a\x00\x00\x00" \
      "\x66\x00\x61\x00\x6b\x00\x65\x00\x5f\x00\x6d\x00\x79\x00\x73\x00" \
      "\x71\x00\x6c\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x05\x00\x00\x00\x01\x02\x00\x00\x07\x00\x00\x00\x00\x02\x00\x00" \
      "\x07\x00\x00\x00\x08\x02\x00\x00\x07\x00\x00\x00\x06\x02\x00\x00" \
      "\x07\x00\x00\x00\x07\x02\x00\x00\x07\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x08\x00\x00\x00\x00\x00\x00\x00" \
      "\x08\x00\x00\x00\x44\x00\x57\x00\x2e\x00\x4c\x00\x4f\x00\x43\x00" \
      "\x41\x00\x4c\x00\x04\x00\x00\x00\x01\x04\x00\x00\x00\x00\x00\x05" \
      "\x15\x00\x00\x00\x02\x87\x5a\x55\xfd\x66\x7b\xcc\xf7\xbc\xaf\x16" \
      "\x00\xea\xc1\x1c\x47\x98\xd8\x01\x14\x00\x66\x00\x61\x00\x6b\x00" \
      "\x65\x00\x5f\x00\x6d\x00\x79\x00\x73\x00\x71\x00\x6c\x00\x00\x00" \
      "\x01\x05\x00\x00\x00\x00\x00\x05\x15\x00\x00\x00\x02\x87\x5a\x55" \
      "\xfd\x66\x7b\xcc\xf7\xbc\xaf\x16\xf4\x01\x00\x00\x00\x00\x00\x00" \
      "\x02\x00\x00\x00\x01\x00\x00\x00\x76\xff\xff\xff\x7d\xc5\xca\xfe" \
      "\x2e\xa0\x7d\xa2\x58\xce\x80\xfb\xf8\x59\x49\xd4\x00\x00\x00\x00" \
      "\x76\xff\xff\xff\x93\xf8\x1c\xd8\x40\x44\xe5\x9c\xb6\xa4\x32\x89" \
      "\x57\x06\xed\x28\x00\x00\x00\x00"
    end
    it_behaves_like 'ticket'
  end

  context 'when using AES128' do
    let(:aes_128) { '1a68526c7518ebec78d1965b0e03841e' }
    let(:ticket_key) { [aes_128].pack('H*') }
    let(:key_length) { ticket_key.length }
    let(:enc_type) { Rex::Proto::Kerberos::Crypto::Encryption::AES128 }
    let(:create_ticket_checksum) { false }
    let(:session_key) { nil }
    let(:pac_type) do
      "\x06\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\xb8\x01\x00\x00" \
      "\x68\x00\x00\x00\x00\x00\x00\x00\x0a\x00\x00\x00\x1e\x00\x00\x00" \
      "\x20\x02\x00\x00\x00\x00\x00\x00\x12\x00\x00\x00\x1c\x00\x00\x00" \
      "\x40\x02\x00\x00\x00\x00\x00\x00\x11\x00\x00\x00\x08\x00\x00\x00" \
      "\x60\x02\x00\x00\x00\x00\x00\x00\x06\x00\x00\x00\x10\x00\x00\x00" \
      "\x68\x02\x00\x00\x00\x00\x00\x00\x07\x00\x00\x00\x10\x00\x00\x00" \
      "\x78\x02\x00\x00\x00\x00\x00\x00\x01\x10\x08\x00\xcc\xcc\xcc\xcc" \
      "\xa8\x01\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\xea\xc1\x1c" \
      "\x47\x98\xd8\x01\xff\xff\xff\xff\xff\xff\xff\x7f\xff\xff\xff\xff" \
      "\xff\xff\xff\x7f\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\xff\xff\xff\xff\xff\xff\xff\x7f\x14\x00\x14\x00" \
      "\x02\x00\x00\x00\x00\x00\x00\x00\x03\x00\x00\x00\x00\x00\x00\x00" \
      "\x04\x00\x00\x00\x00\x00\x00\x00\x05\x00\x00\x00\x00\x00\x00\x00" \
      "\x06\x00\x00\x00\x00\x00\x00\x00\x07\x00\x00\x00\x00\x00\x00\x00" \
      "\xf4\x01\x00\x00\x01\x02\x00\x00\x05\x00\x00\x00\x08\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x09\x00\x00\x00\x10\x00\x10\x00" \
      "\x0a\x00\x00\x00\x0b\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x10\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x0a\x00\x00\x00\x00\x00\x00\x00\x0a\x00\x00\x00" \
      "\x66\x00\x61\x00\x6b\x00\x65\x00\x5f\x00\x6d\x00\x79\x00\x73\x00" \
      "\x71\x00\x6c\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x05\x00\x00\x00\x01\x02\x00\x00\x07\x00\x00\x00\x00\x02\x00\x00" \
      "\x07\x00\x00\x00\x08\x02\x00\x00\x07\x00\x00\x00\x06\x02\x00\x00" \
      "\x07\x00\x00\x00\x07\x02\x00\x00\x07\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x08\x00\x00\x00\x00\x00\x00\x00" \
      "\x08\x00\x00\x00\x44\x00\x57\x00\x2e\x00\x4c\x00\x4f\x00\x43\x00" \
      "\x41\x00\x4c\x00\x04\x00\x00\x00\x01\x04\x00\x00\x00\x00\x00\x05" \
      "\x15\x00\x00\x00\x02\x87\x5a\x55\xfd\x66\x7b\xcc\xf7\xbc\xaf\x16" \
      "\x00\xea\xc1\x1c\x47\x98\xd8\x01\x14\x00\x66\x00\x61\x00\x6b\x00" \
      "\x65\x00\x5f\x00\x6d\x00\x79\x00\x73\x00\x71\x00\x6c\x00\x00\x00" \
      "\x01\x05\x00\x00\x00\x00\x00\x05\x15\x00\x00\x00\x02\x87\x5a\x55" \
      "\xfd\x66\x7b\xcc\xf7\xbc\xaf\x16\xf4\x01\x00\x00\x00\x00\x00\x00" \
      "\x02\x00\x00\x00\x01\x00\x00\x00\x0f\x00\x00\x00\xab\xf1\x31\xb6" \
      "\x87\x35\xf2\xce\x2e\xc7\x84\xc8\x0f\x00\x00\x00\x0e\x69\xae\x48" \
      "\x7c\x3b\x8b\xbe\xbe\x08\x88\x3d"
    end
    it_behaves_like 'ticket'
  end

  context 'when using AES256' do
    let(:aes_256) { 'dec977048690dc9278ecea3dd07ca386295bc7ddc0d346676a498104b5f5113e' }
    let(:ticket_key) { [aes_256].pack('H*') }
    let(:key_length) { ticket_key.length }
    let(:enc_type) { Rex::Proto::Kerberos::Crypto::Encryption::AES256 }
    let(:create_ticket_checksum) { false }
    let(:session_key) { nil }
    let(:pac_type) do
      "\x06\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\xb8\x01\x00\x00" \
      "\x68\x00\x00\x00\x00\x00\x00\x00\x0a\x00\x00\x00\x1e\x00\x00\x00" \
      "\x20\x02\x00\x00\x00\x00\x00\x00\x12\x00\x00\x00\x1c\x00\x00\x00" \
      "\x40\x02\x00\x00\x00\x00\x00\x00\x11\x00\x00\x00\x08\x00\x00\x00" \
      "\x60\x02\x00\x00\x00\x00\x00\x00\x06\x00\x00\x00\x10\x00\x00\x00" \
      "\x68\x02\x00\x00\x00\x00\x00\x00\x07\x00\x00\x00\x10\x00\x00\x00" \
      "\x78\x02\x00\x00\x00\x00\x00\x00\x01\x10\x08\x00\xcc\xcc\xcc\xcc" \
      "\xa8\x01\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\xea\xc1\x1c" \
      "\x47\x98\xd8\x01\xff\xff\xff\xff\xff\xff\xff\x7f\xff\xff\xff\xff" \
      "\xff\xff\xff\x7f\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\xff\xff\xff\xff\xff\xff\xff\x7f\x14\x00\x14\x00" \
      "\x02\x00\x00\x00\x00\x00\x00\x00\x03\x00\x00\x00\x00\x00\x00\x00" \
      "\x04\x00\x00\x00\x00\x00\x00\x00\x05\x00\x00\x00\x00\x00\x00\x00" \
      "\x06\x00\x00\x00\x00\x00\x00\x00\x07\x00\x00\x00\x00\x00\x00\x00" \
      "\xf4\x01\x00\x00\x01\x02\x00\x00\x05\x00\x00\x00\x08\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x09\x00\x00\x00\x10\x00\x10\x00" \
      "\x0a\x00\x00\x00\x0b\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x10\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x0a\x00\x00\x00\x00\x00\x00\x00\x0a\x00\x00\x00" \
      "\x66\x00\x61\x00\x6b\x00\x65\x00\x5f\x00\x6d\x00\x79\x00\x73\x00" \
      "\x71\x00\x6c\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x05\x00\x00\x00\x01\x02\x00\x00\x07\x00\x00\x00\x00\x02\x00\x00" \
      "\x07\x00\x00\x00\x08\x02\x00\x00\x07\x00\x00\x00\x06\x02\x00\x00" \
      "\x07\x00\x00\x00\x07\x02\x00\x00\x07\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x08\x00\x00\x00\x00\x00\x00\x00" \
      "\x08\x00\x00\x00\x44\x00\x57\x00\x2e\x00\x4c\x00\x4f\x00\x43\x00" \
      "\x41\x00\x4c\x00\x04\x00\x00\x00\x01\x04\x00\x00\x00\x00\x00\x05" \
      "\x15\x00\x00\x00\x02\x87\x5a\x55\xfd\x66\x7b\xcc\xf7\xbc\xaf\x16" \
      "\x00\xea\xc1\x1c\x47\x98\xd8\x01\x14\x00\x66\x00\x61\x00\x6b\x00" \
      "\x65\x00\x5f\x00\x6d\x00\x79\x00\x73\x00\x71\x00\x6c\x00\x00\x00" \
      "\x01\x05\x00\x00\x00\x00\x00\x05\x15\x00\x00\x00\x02\x87\x5a\x55" \
      "\xfd\x66\x7b\xcc\xf7\xbc\xaf\x16\xf4\x01\x00\x00\x00\x00\x00\x00" \
      "\x02\x00\x00\x00\x01\x00\x00\x00\x10\x00\x00\x00\x99\x45\x39\x47" \
      "\x17\x85\xf7\x6f\x20\x8b\x78\x52\x10\x00\x00\x00\x8c\xce\x2d\x98" \
      "\xda\x1f\x71\x51\x42\x16\x0d\x97"
    end
    it_behaves_like 'ticket'
  end

  context 'when ticket contains a ticket checksum' do
    let(:nt_hash) { '88E4D9FABAECF3DEC18DD80905521B29' }
    let(:ticket_key) { [nt_hash].pack('H*') }
    let(:key_length) { ticket_key.length }
    let(:enc_type) { Rex::Proto::Kerberos::Crypto::Encryption::RC4_HMAC }
    let(:session_key) { "\x7a\x86\xc8\x90\xf7\xd1\x19\x53\x18\xe5\xb6\xe4\xaa\x50\x36\xc4" }
    let(:create_ticket_checksum) { true }
    let(:ticket_checksum_sig) { "\xb0\x90\x67\x70\xea\xf4\x9b\x43\x39\x30\x96\x70\x14\x40\xb2\x24" }

    let(:pac_type) do
      "\x07\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\xb8\x01\x00\x00" \
      "\x78\x00\x00\x00\x00\x00\x00\x00\x0a\x00\x00\x00\x1e\x00\x00\x00" \
      "\x30\x02\x00\x00\x00\x00\x00\x00\x12\x00\x00\x00\x1c\x00\x00\x00" \
      "\x50\x02\x00\x00\x00\x00\x00\x00\x11\x00\x00\x00\x08\x00\x00\x00" \
      "\x70\x02\x00\x00\x00\x00\x00\x00\x06\x00\x00\x00\x14\x00\x00\x00" \
      "\x78\x02\x00\x00\x00\x00\x00\x00\x07\x00\x00\x00\x14\x00\x00\x00" \
      "\x90\x02\x00\x00\x00\x00\x00\x00\x10\x00\x00\x00\x14\x00\x00\x00" \
      "\xa8\x02\x00\x00\x00\x00\x00\x00\x01\x10\x08\x00\xcc\xcc\xcc\xcc" \
      "\xa8\x01\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\xea\xc1\x1c" \
      "\x47\x98\xd8\x01\xff\xff\xff\xff\xff\xff\xff\x7f\xff\xff\xff\xff" \
      "\xff\xff\xff\x7f\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\xff\xff\xff\xff\xff\xff\xff\x7f\x14\x00\x14\x00" \
      "\x02\x00\x00\x00\x00\x00\x00\x00\x03\x00\x00\x00\x00\x00\x00\x00" \
      "\x04\x00\x00\x00\x00\x00\x00\x00\x05\x00\x00\x00\x00\x00\x00\x00" \
      "\x06\x00\x00\x00\x00\x00\x00\x00\x07\x00\x00\x00\x00\x00\x00\x00" \
      "\xf4\x01\x00\x00\x01\x02\x00\x00\x05\x00\x00\x00\x08\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x09\x00\x00\x00\x10\x00\x10\x00" \
      "\x0a\x00\x00\x00\x0b\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x10\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x0a\x00\x00\x00\x00\x00\x00\x00\x0a\x00\x00\x00" \
      "\x66\x00\x61\x00\x6b\x00\x65\x00\x5f\x00\x6d\x00\x79\x00\x73\x00" \
      "\x71\x00\x6c\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x05\x00\x00\x00\x01\x02\x00\x00\x07\x00\x00\x00\x00\x02\x00\x00" \
      "\x07\x00\x00\x00\x08\x02\x00\x00\x07\x00\x00\x00\x06\x02\x00\x00" \
      "\x07\x00\x00\x00\x07\x02\x00\x00\x07\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x08\x00\x00\x00\x00\x00\x00\x00" \
      "\x08\x00\x00\x00\x44\x00\x57\x00\x2e\x00\x4c\x00\x4f\x00\x43\x00" \
      "\x41\x00\x4c\x00\x04\x00\x00\x00\x01\x04\x00\x00\x00\x00\x00\x05" \
      "\x15\x00\x00\x00\x02\x87\x5a\x55\xfd\x66\x7b\xcc\xf7\xbc\xaf\x16" \
      "\x00\xea\xc1\x1c\x47\x98\xd8\x01\x14\x00\x66\x00\x61\x00\x6b\x00" \
      "\x65\x00\x5f\x00\x6d\x00\x79\x00\x73\x00\x71\x00\x6c\x00\x00\x00" \
      "\x01\x05\x00\x00\x00\x00\x00\x05\x15\x00\x00\x00\x02\x87\x5a\x55" \
      "\xfd\x66\x7b\xcc\xf7\xbc\xaf\x16\xf4\x01\x00\x00\x00\x00\x00\x00" \
      "\x02\x00\x00\x00\x01\x00\x00\x00\x76\xff\xff\xff\x3a\xa0\x8f\xb0" \
      "\x6d\x3e\x90\xc0\xd1\xd3\x2d\xdf\xd3\x42\xa9\x16\x00\x00\x00\x00" \
      "\x76\xff\xff\xff\x80\x77\x5b\x5d\x07\x80\x22\xab\x65\x01\x67\xd1" \
      "\x66\xed\x9d\x80\x00\x00\x00\x00\x76\xff\xff\xff\xb0\x90\x67\x70" \
      "\xea\xf4\x9b\x43\x39\x30\x96\x70\x14\x40\xb2\x24\x00\x00\x00\x00"
    end

    it_behaves_like 'ticket'
  end
end
