/*
 * Copyright (c) 2001, Dietmar Kühl, under the Boost Software License,
 * Version 1.0
 * Copyright (c) 2002, Beman Dawes, under the Boost Software License,
 * Version 1.0
 * Copyright (c) 2007-2010, Erik Lindroos <gliptic@gmail.com>, under the
 * BSD-2-Clause License
 * Copyright (c) 2010, "basro", under the BSD-2-Clause License.
 *
 *
 * Boost Software License - Version 1.0 - August 17th, 2003
 *
 * Permission is hereby granted, free of charge, to any person or organization
 * obtaining a copy of the software and accompanying documentation covered by
 * this license (the "Software") to use, reproduce, display, distribute,
 * execute, and transmit the Software, and to prepare derivative works of the
 * Software, and to permit third-parties to whom the Software is furnished to
 * do so, all subject to the following:
 *
 * The copyright notices in the Software and this entire statement, including
 * the above license grant, this restriction and the following disclaimer,
 * must be included in all copies of the Software, in whole or in part, and
 * all derivative works of the Software, unless such copies or derivative
 * works are solely in the form of machine-executable object code generated by
 * a source language processor.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
 * SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
 * FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 *
 * BSD-2-Clause License:
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * * Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "filesystem.hpp"
#include <gvl/support/platform.hpp>
#include <stdexcept>
#include <cassert>
#include <cctype>
#include <sys/stat.h>
#include <fstream>
#include <iostream>
#include <cstring>
#include <cstdlib>
#if GVL_WINDOWS
#include <io.h> // _mktemp(), _mktemp_s()
#endif

std::string changeLeaf(std::string const& path, std::string const& newLeaf)
{
	std::size_t lastSep = path.find_last_of("\\/");

	if(lastSep == std::string::npos)
		return newLeaf; // We assume there's only a leaf in the path
	return path.substr(0, lastSep + 1) + newLeaf;
}

std::string getRoot(std::string const& path)
{
	std::size_t lastSep = path.find_last_of("\\/");

	if(lastSep == std::string::npos)
		return "";
	return path.substr(0, lastSep);
}

std::string getBasename(std::string const& path)
{
	std::size_t lastSep = path.find_last_of(".");

	if(lastSep == std::string::npos)
		return path;
	return path.substr(0, lastSep);
}

std::string getExtension(std::string const& path)
{
	std::size_t lastSep = path.find_last_of(".");

	if(lastSep == std::string::npos)
		return "";
	return path.substr(lastSep + 1);
}

void toUpperCase(std::string& str)
{
	for(std::size_t i = 0; i < str.size(); ++i)
	{
		str[i] = std::toupper(static_cast<unsigned char>(str[i])); // TODO: Uppercase conversion that works for the DOS charset
	}
}

void toLowerCase(std::string& str)
{
	for(std::size_t i = 0; i < str.size(); ++i)
	{
		str[i] = std::tolower(static_cast<unsigned char>(str[i])); // TODO: Lowercase conversion that works for the DOS charset
	}
}

FILE* tolerantFOpen(std::string const& name, char const* mode)
{
	FILE* f = std::fopen(name.c_str(), mode);
	if(f)
		return f;

	std::string ch(name);
	toUpperCase(ch);
	f = std::fopen(ch.c_str(), mode);
	if(f)
		return f;

	toLowerCase(ch);
	f = std::fopen(ch.c_str(), mode);
	if(f)
		return f;

	// Try with first letter capital
	ch[0] = std::toupper(static_cast<unsigned char>(ch[0]));
	f = std::fopen(ch.c_str(), mode);
	if(f)
		return f;

	return 0;
}

std::string joinPath(std::string const& root, std::string const& leaf)
{
	if(!root.empty()
	&& root[root.size() - 1] != '\\'
	&& root[root.size() - 1] != '/')
	{
		return root + '/' + leaf;
	}
	else
	{
		return root + leaf;
	}
}

bool fileExists(std::string const& path)
{
	FILE* f = fopen(path.c_str(), "rb");
	bool state = (f != 0);
	if(f) fclose(f);
	return state;
}

bool isDir(std::string const& path)
{
	struct stat s;
	if(stat(path.c_str(), &s) == 0) {
		if(s.st_mode & S_IFDIR)
			return true;
	}
	return false;
}

void backupFile(std::string const& path)
{
	char tmpFileName[FILENAME_MAX];
	strcpy(tmpFileName, path.c_str());
	strcat(tmpFileName, "_backup_XXXXXX");

#ifdef __MINGW32__
	// _mktemp_s was only very recently implemented for mingw-w64
	if (_mktemp(tmpFileName) == NULL)
#elif	GVL_WINDOWS
	if (_mktemp_s(tmpFileName, strlen(tmpFileName) + 1) != 0)
#else
	if (mkstemp(tmpFileName) == -1)
#endif
		throw std::runtime_error("Unable to create temp file for backup");

	std::ifstream in;
	in.open(path.c_str(), std::ios::binary);
	std::ofstream out;
	out.open(tmpFileName, std::ios::binary | std::ios::out);

	if (in.is_open() && out.is_open())
		out << in.rdbuf();
	else
		throw std::runtime_error("Unable to open files for backup");
}

void rmFile(std::string const& path)
{
	if (remove(path.c_str()) != 0)
		throw std::runtime_error("Could not remove file '" + path + "'");
}

std::size_t fileLength(FILE* f)
{
	long old = ftell(f);
	fseek(f, 0, SEEK_END);
	long len = ftell(f);
	fseek(f, old, SEEK_SET);
	return len;
}

#if GVL_WINDOWS
#  include "windows.h"

#  if defined(__BORLANDC__) || defined(__MWERKS__)
#     if defined(__BORLANDC__)
        using std::time_t;
#     endif
#     include "utime.h"
#   else
#     include "sys/utime.h"
#   endif
# else
#   include "dirent.h"
#   include "unistd.h"
#   include "fcntl.h"
#   include "utime.h"
#   include <errno.h>
# endif

namespace
{

struct filename_result
{
	filename_result()
	: name(0), alt_name(0)
	{
	}

	filename_result(char const* name, char const* alt_name)
	: name(name), alt_name(alt_name)
	{

	}

	operator void const*()
	{
		return name;
	}

	char const* name;
	char const* alt_name;
};

#if GVL_LINUX

# define BOOST_HANDLE DIR *
# define BOOST_INVALID_HANDLE_VALUE 0
# define BOOST_SYSTEM_DIRECTORY_TYPE struct dirent *

inline filename_result find_first_file( const char * dir,
BOOST_HANDLE & handle, BOOST_SYSTEM_DIRECTORY_TYPE & )
// Returns: 0 if error, otherwise name
{
	const char * dummy_first_name = ".";
	return ( (handle = ::opendir( dir ))
		== BOOST_INVALID_HANDLE_VALUE ) ? filename_result() : filename_result(dummy_first_name, dummy_first_name);
}

inline void find_close( BOOST_HANDLE handle )
{
	assert( handle != BOOST_INVALID_HANDLE_VALUE );
	::closedir( handle );
}

inline filename_result find_next_file(
BOOST_HANDLE handle, BOOST_SYSTEM_DIRECTORY_TYPE & )
// Returns: if EOF 0, otherwise name
// Throws: if system reports error
{

	//  TODO: use readdir_r() if available, so code is multi-thread safe.
	//  Fly-in-ointment: not all platforms support readdir_r().

	struct dirent * dp;
	errno = 0;
	if ( (dp = ::readdir( handle )) == 0 )
	{
		if ( errno != 0 )
		{
			throw std::runtime_error("Error iterating directory");
		}
		else { return filename_result(); } // end reached
	}
	return filename_result(dp->d_name, dp->d_name);
}
#elif GVL_WINDOWS

# define BOOST_HANDLE HANDLE
# define BOOST_INVALID_HANDLE_VALUE INVALID_HANDLE_VALUE
# define BOOST_SYSTEM_DIRECTORY_TYPE WIN32_FIND_DATAA


inline filename_result find_first_file( const char * dir,
BOOST_HANDLE & handle, BOOST_SYSTEM_DIRECTORY_TYPE & data )
// Returns: 0 if error, otherwise name
{
	//    std::cout << "find_first_file " << dir << std::endl;
	std::string dirpath( std::string(dir) + "/*" );
	bool fail = ( (handle = ::FindFirstFileA( dirpath.c_str(), &data )) == BOOST_INVALID_HANDLE_VALUE );

	if(fail)
		return filename_result();

	return filename_result(
		data.cFileName,
		data.cAlternateFileName[0] ? data.cAlternateFileName : data.cFileName);
}

inline void find_close( BOOST_HANDLE handle )
{
	//    std::cout << "find_close" << std::endl;
	assert( handle != BOOST_INVALID_HANDLE_VALUE );
	::FindClose( handle );
}

inline filename_result find_next_file(
BOOST_HANDLE handle, BOOST_SYSTEM_DIRECTORY_TYPE & data )
// Returns: 0 if EOF, otherwise name
// Throws: if system reports error
{
	if ( ::FindNextFileA( handle, &data ) == 0 )
	{
		if ( ::GetLastError() != ERROR_NO_MORE_FILES )
		{
			throw std::runtime_error("Error iterating directory");
		}
		else { return filename_result(); } // end reached
	}

	return filename_result(
		data.cFileName,
		data.cAlternateFileName[0] ? data.cAlternateFileName : data.cFileName);
}
#else

#error "Not supported"
#endif

}

struct dir_itr_imp
{
public:
	dir_itr_imp()
	{

	}

	std::string       entry_path;
	std::string       entry_alt_path;
	BOOST_HANDLE      handle;

	~dir_itr_imp()
	{
		if ( handle != BOOST_INVALID_HANDLE_VALUE )
			find_close( handle );
	}
};

DirectoryIterator::DirectoryIterator(std::string const& dir)
{
	dir_itr_init( m_imp, dir.c_str() );
}

DirectoryIterator::~DirectoryIterator()
{
}


inline bool dot_or_dot_dot( char const * name )
{
	return name[0]=='.'
		&& (name[1]=='\0' || (name[1]=='.' && name[2]=='\0'));
}

//  directory_iterator implementation  ---------------------------------------//

void dir_itr_init( dir_itr_imp_ptr & m_imp,
	char const* dir_path )
{
	m_imp.reset( new dir_itr_imp() );
	BOOST_SYSTEM_DIRECTORY_TYPE scratch;
	filename_result name;  // initialization quiets compiler warnings
	if ( !dir_path[0] )
		m_imp->handle = BOOST_INVALID_HANDLE_VALUE;
	else
		name = find_first_file( dir_path, m_imp->handle, scratch );  // sets handle

	if ( m_imp->handle != BOOST_INVALID_HANDLE_VALUE )
	{
		if ( !dot_or_dot_dot( name.name ) )
		{
			m_imp->entry_path = name.name;
			m_imp->entry_alt_path = name.alt_name;
		}
		else
		{
			//m_imp->entry_path.m_path_append( "dummy", no_check );
			dir_itr_increment( m_imp );
		}
	}
	else
	{
		throw std::runtime_error("Directory iterator ctor");
	}
}

std::string & dir_itr_dereference(
	const dir_itr_imp_ptr & m_imp )
{
	assert( m_imp.get() ); // fails if dereference end iterator
	return m_imp->entry_path;
}

std::string & dir_itr_alt_dereference(
	const dir_itr_imp_ptr & m_imp )
{
	assert( m_imp.get() ); // fails if dereference end iterator
	return m_imp->entry_alt_path;
}

void dir_itr_increment( dir_itr_imp_ptr & m_imp )
{
	assert( m_imp.get() ); // fails on increment end iterator
	assert( m_imp->handle != BOOST_INVALID_HANDLE_VALUE ); // reality check

	BOOST_SYSTEM_DIRECTORY_TYPE scratch;
	filename_result name;

	while ( (name = find_next_file( m_imp->handle, scratch )) )
	{
		// append name, except ignore "." or ".."
		if ( !dot_or_dot_dot( name.name ) )
		{
			m_imp->entry_path = name.name;
			m_imp->entry_alt_path = name.alt_name;
			return;
		}
	}

	m_imp.reset(); // make base() the end iterator
}
