package main

import (
	"testing"

	"github.com/knqyf263/go-cpe/naming"
	"github.com/knqyf263/go-cpe/matching"
)

func TestCompare(t *testing.T) {
    vectors := []struct {
		uri       string
		fs        string
	}{
     
    {
        uri: `cpe:/a:%240.99_kindle_books_project:%240.99_kindle_books:6::~~~android~~`,
        fs:  `cpe:2.3:a:hp:nonstop_netbatch:j06.16.02:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:1024cms:1024_cms:0.7`,
        fs:  `cpe:2.3:a:f5:big-ip:9.4.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:1024cms:1024_cms:1.2.5`,
        fs:  `cpe:2.3:a:adobe:acrobat_dc:15.006.30173:*:*:*:classic:*:*:*`,
    },
     
    {
        uri: `cpe:/a:1024cms:1024_cms:1.3.1`,
        fs:  `cpe:2.3:h:softbank:android_smartphone:003p:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:1024cms:1024_cms:1.4.1`,
        fs:  `cpe:2.3:o:microsoft:windows-nt:4.0:sru:terminal_srv:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:1024cms:1024_cms:1.4.2`,
        fs:  `cpe:2.3:h:huawei:ar_18-3x:r0118:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:1024cms:1024_cms:1.4.2:beta`,
        fs:  `cpe:2.3:a:ati:catalyst_control_center_hydravision_full:2009.1124.2131.38610:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:1024cms:1024_cms:2.1.1`,
        fs:  `cpe:2.3:a:ftprush:ftprush:1.1.0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:11in1:11in1:1.2.1:stable_12-31-2011`,
        fs:  `cpe:2.3:a:wireshark:wireshark:1.6.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:129zou:zombie_diary:1.2.2::~~~android~~`,
        fs:  `cpe:2.3:a:emc:watch4net:6.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:12net:login_rebuilder:1.0.0::~~~wordpress~~`,
        fs:  `cpe:2.3:a:ca:itechnology_igateway:4.0.050321:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:12net:login_rebuilder:1.0.1::~~~wordpress~~`,
        fs:  `cpe:2.3:a:google:chrome:4.1.249.1018:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:12net:login_rebuilder:1.0.2::~~~wordpress~~`,
        fs:  `cpe:2.3:a:atlassian:bamboo:5.14.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:12net:login_rebuilder:1.0.3::~~~wordpress~~`,
        fs:  `cpe:2.3:a:redhat:openshift:2.0.5:*:enterprise:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:12net:login_rebuilder:1.1.0::~~~wordpress~~`,
        fs:  `cpe:2.3:a:giflib_project:giflib:5.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:12net:login_rebuilder:1.1.1::~~~wordpress~~`,
        fs:  `cpe:2.3:a:apple:quicktime_broadcaster:4.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:12net:login_rebuilder:1.1.2::~~~wordpress~~`,
        fs:  `cpe:2.3:a:libevent_project:libevent:2.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:12net:login_rebuilder:1.1.3::~~~wordpress~~`,
        fs:  `cpe:2.3:a:perl:perl:2.10.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:12net:login_rebuilder:1.2.0::~~~wordpress~~`,
        fs:  `cpe:2.3:a:ibm:tivoli_storage_manager_for_virtual_environments:6.3.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:12net:login_rebuilder:1.2.1::~~~wordpress~~`,
        fs:  `cpe:2.3:a:emc:networker:7.5.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:12net:login_rebuilder:1.2.2::~~~wordpress~~`,
        fs:  `cpe:2.3:a:pivotal_software:cloud_foundry_uaa:3.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:12net:login_rebuilder:1.2.3::~~~wordpress~~`,
        fs:  `cpe:2.3:a:oracle:supply_chain_products_suite_sql-server:7.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:133:flight_manager:4::~~~android~~`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(15.1\)s:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:163:netease_movie:4.7.2::~~~android~~`,
        fs:  `cpe:2.3:a:moinejf:abcm2ps:3.0.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:1800contacts:1800contacts_app:2.7.0::~~~android~~`,
        fs:  `cpe:2.3:a:atmail:atmail:6.3.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:1kxun:qianxun_yingshi:1.2.3`,
        fs:  `cpe:2.3:a:google:chrome:32.0.1673.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:1kxun:qianxun_yingshi:1.3.4`,
        fs:  `cpe:2.3:a:baseurl:yum:3.2.22:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:2g_live_tv_project:2g_live_tv:0.9::~~~android~~`,
        fs:  `cpe:2.3:a:otrs:otrs:2.0.0:beta1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:2glux:com_sexypolling:0.9.1:-:~-~-~joomla%21~~`,
        fs:  `cpe:2.3:a:teamviewer:teamviewer:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:2glux:com_sexypolling:0.9.2:-:~-~-~joomla%21~~`,
        fs:  `cpe:2.3:a:gnu:fileutils:4.0.36:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:2glux:com_sexypolling:0.9.4:-:~-~-~joomla%21~~`,
        fs:  `cpe:2.3:h:hp:h3c_switch:s5100-26c-pwr-ei_ls-5100-26c-pwr-ei-ovs:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:2glux:com_sexypolling:0.9.5:-:~-~-~joomla%21~~`,
        fs:  `cpe:2.3:o:cisco:pix_firewall_software:6.3\(1\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:2glux:com_sexypolling:0.9.6:-:~-~-~joomla%21~~`,
        fs:  `cpe:2.3:h:cisco:nexus_5596up_switch:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:2glux:com_sexypolling:0.9.7:-:~-~-~joomla%21~~`,
        fs:  `cpe:2.3:o:vmware:photon_os:1.0:2016-08-14:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:2glux:com_sexypolling:1.0.1:-:~-~-~joomla%21~~`,
        fs:  `cpe:2.3:a:isaac_sukin:shorten:6.x-1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:2glux:com_sexypolling:1.0.2:-:~-~-~joomla%21~~`,
        fs:  `cpe:2.3:a:redhat:libvirt:0.6.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:2glux:com_sexypolling:1.0.3:-:~-~-~joomla%21~~`,
        fs:  `cpe:2.3:a:google:chrome:18.0.1025.57:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:2glux:com_sexypolling:1.0.4:-:~-~-~joomla%21~~`,
        fs:  `cpe:2.3:a:pureftpd:pure-ftpd:0.97:pre4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:2glux:com_sexypolling:1.0.5:-:~-~-~joomla%21~~`,
        fs:  `cpe:2.3:a:elasticsearch:logstash:1.0.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:2glux:com_sexypolling:1.0.6:-:~-~-~joomla%21~~`,
        fs:  `cpe:2.3:a:rom_walton:boinc:7.0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:2glux:com_sexypolling:1.0.7:-:~-~-~joomla%21~~`,
        fs:  `cpe:2.3:a:jenkins:config_file_provider:2.6.2:*:*:*:*:jenkins:*:*`,
    },
     
    {
        uri: `cpe:/a:2glux:com_sexypolling:1.0.8:-:~-~-~joomla%21~~`,
        fs:  `cpe:2.3:a:vandyke:absoluteftp:1.9.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:2glux:com_sexypolling:1.0.9:-:~-~-~joomla%21~~`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.4.8-10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:2rv:dr._sheikh_adnan_ibrahim:1::~~~android~~`,
        fs:  `cpe:2.3:a:roxio:creator_content:2011:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:30a:30a:5.26.2::~~~android~~`,
        fs:  `cpe:2.3:a:mozilla:network_security_services:3.21.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:141701:-`,
        fs:  `cpe:2.3:a:simplemachines:simple_machines_forum:1.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c13612:-`,
        fs:  `cpe:2.3:a:ibm:websphere_mq:6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c13613:-`,
        fs:  `cpe:2.3:a:buffalotech:wer-amg54_firmware:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c13616:-`,
        fs:  `cpe:2.3:a:qemu:qemu:1.0:rc4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c13700:-`,
        fs:  `cpe:2.3:a:mahara:mahara:16.04.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c13750:-`,
        fs:  `cpe:2.3:a:vanillaforums:vanilla_forums:2.0.17.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c13754:-`,
        fs:  `cpe:2.3:a:typo3:typo3:6.2.0:beta3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c13758:-`,
        fs:  `cpe:2.3:a:apple:quicktime:6.4.0:-:windows:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3com:3c15100d:-`,
        fs:  `cpe:2.3:a:skyarts:neofiler:2.4.2:-:-:*:lite:android:*:*`,
    },
     
    {
        uri: `cpe:/a:3com:3c16115-us:-`,
        fs:  `cpe:2.3:o:iij:seil_x2_firmware:3.70:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3com:3c16115-us:2.01`,
        fs:  `cpe:2.3:a:gnu:chess:5.02:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3com:3c16116-us:-`,
        fs:  `cpe:2.3:a:xmlsoft:libxslt:1.1.19:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3com:3c16116-us:2.0`,
        fs:  `cpe:2.3:o:cisco:catos:5.2\(2\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c16450-us:-`,
        fs:  `cpe:2.3:o:ibm:power_5_system_firmware:sf240_320_201:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c16450-us:2.15`,
        fs:  `cpe:2.3:a:hp:loadrunner:12.50:p3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c16450-us:2.16`,
        fs:  `cpe:2.3:a:ibm:maximo_asset_management:7.5.0.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c16479:-`,
        fs:  `cpe:2.3:a:cloudbees:jenkins:1.319:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c16485:-`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.39.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c16486:-`,
        fs:  `cpe:2.3:a:live555:streaming_media:2012.08.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c16486:1.0.2`,
        fs:  `cpe:2.3:a:sqltools:sqltools:1.3.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c16811-us:-`,
        fs:  `cpe:2.3:a:python:python:3.4.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c16950-us:-`,
        fs:  `cpe:2.3:a:cisco:telepresence_supervisor_mse_8050_software:2.1\(1.18\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c16950-us:2.67`,
        fs:  `cpe:2.3:a:quest:intrust:10.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c16950-us:2.68`,
        fs:  `cpe:2.3:a:artonx.org:activescriptruby:1.6.3.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c16986a-us:-`,
        fs:  `cpe:2.3:a:piwik:piwik:0.1.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c16987a-us:-`,
        fs:  `cpe:2.3:a:d-bus_project:d-bus:1.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c16988a-us:-`,
        fs:  `cpe:2.3:a:tapatalk:tapatalk:1.1.0:*:*:*:*:vbulletin:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c17100-us:-`,
        fs:  `cpe:2.3:o:yahama:rt250i:8.02.43:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c17203-us:-`,
        fs:  `cpe:2.3:a:apache:axis:1.0:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c17204-us:-`,
        fs:  `cpe:2.3:a:zarafa:zarafa:7.0.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c17205-us:-`,
        fs:  `cpe:2.3:a:ponsoftware:explzh:2.68:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c17210-us:-`,
        fs:  `cpe:2.3:a:cart66:cart66_lite_plugin:1.5.1:-:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c17300:-`,
        fs:  `cpe:2.3:a:moinmo:moinmoin:1.8.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c17302:-`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.0.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c17304:-`,
        fs:  `cpe:2.3:a:horde:imp:4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c17400:-`,
        fs:  `cpe:2.3:a:gnome:eye_of_gnome:3.18.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c17401:-`,
        fs:  `cpe:2.3:a:apple:webkit:312.8.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c17701-us:-`,
        fs:  `cpe:2.3:a:strongswan:strongswan:2.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c17702-us:-`,
        fs:  `cpe:2.3:a:microsoft:internet_explorer:6:sp1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c17706-us:-`,
        fs:  `cpe:2.3:a:digium:asterisk:1.4.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c17707-us:-`,
        fs:  `cpe:2.3:o:iij:seil_b1_firmware:3.77:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c17708-us:-`,
        fs:  `cpe:2.3:o:iij:seil_x1_firmware:2.92:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c17709-us:-`,
        fs:  `cpe:2.3:a:nokia:qt_creator:1.3.0:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c250100a:-`,
        fs:  `cpe:2.3:a:ibm:hardware_management_console:7.3.2.0:sp1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c39024-us:-`,
        fs:  `cpe:2.3:h:ge:multilink_ml1200:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c6012-901:-`,
        fs:  `cpe:2.3:a:ibm:rational_developer_for_system_z:7.6.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c6012-902:-`,
        fs:  `cpe:2.3:a:microsoft:outlook:2010:sp1:*:*:*:x86:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c840-us:-`,
        fs:  `cpe:2.3:a:ibm:inotes:9.0.1.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c8432:-`,
        fs:  `cpe:2.3:a:john_nunemaker:httparty:0.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c8462:-`,
        fs:  `cpe:2.3:a:oracle:sun_products_suite_java_system_web_server:6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3c8532:-`,
        fs:  `cpe:2.3:a:apache:httpclient:4.3:alpha1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3cb9e16:-`,
        fs:  `cpe:2.3:a:sun:ray_server_software:3.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3com:3cdaemon:-`,
        fs:  `cpe:2.3:a:digia:qt:4.6.5:rc:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3cnj220:-`,
        fs:  `cpe:2.3:a:intermesh:group-office:3.5.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3cnj220:2.0.22`,
        fs:  `cpe:2.3:a:inspire_ircd:inspircd:1.1.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3cp4144:-`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.10.87:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3cp4144:1.1.7`,
        fs:  `cpe:2.3:a:ca:directory:r12:sp2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3cp4144:1.1.9`,
        fs:  `cpe:2.3:a:adobe:acrobat_reader:11.0.06:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3cp4144:1.1.9.4`,
        fs:  `cpe:2.3:a:ibm:tivoli_storage_manager:7.1.3.100:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3cr29223:-`,
        fs:  `cpe:2.3:a:sonicwall:aventail_webifiers:10.3.42:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3cr860-95:-`,
        fs:  `cpe:2.3:a:opnet:license_server:16.0.0.10106:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3cradsl72:-`,
        fs:  `cpe:2.3:a:isc:bind:9.4.3:b3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3crtpx505-73:-`,
        fs:  `cpe:2.3:h:rockwellautomation:micrologix_1100:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3crwe454g72:-`,
        fs:  `cpe:2.3:a:gridgain:gridgain:8.0.1:ea7:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3crwe454g72:1.0.2`,
        fs:  `cpe:2.3:a:opera:opera_browser:10.50:beta2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3crwe454g72:1.0.2.11`,
        fs:  `cpe:2.3:o:cisco:ios_xe:3.8s\(.0\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3crwe454g72:1.0.3.5`,
        fs:  `cpe:2.3:a:clamav:clamav:0.95.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3crwe454g72:1.02.00`,
        fs:  `cpe:2.3:a:nodejs:node.js:0.10.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3crwe454g72:1.02.11`,
        fs:  `cpe:2.3:a:cloudbees:jenkins:1.361:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3crwe454g72:1.03.05`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(33\)sxi2a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3crwe454g72:1.03.07`,
        fs:  `cpe:2.3:a:hp:version_control_repository_manager:2.1.1.710:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3crwe454g72:1.03.07a`,
        fs:  `cpe:2.3:a:opnet:license_server:15.0.0.8631:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3crwe554g72t:-`,
        fs:  `cpe:2.3:o:juniper:junos:12.2:r6:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3crwe747a:-`,
        fs:  `cpe:2.3:a:apache:tomcat:8.0.36:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3crwe754g72-a:-`,
        fs:  `cpe:2.3:a:cisco:prime_network_registrar:7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3crwe754g72-a:1.13`,
        fs:  `cpe:2.3:o:iij:seil_b1_firmware:3.27:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3crwe754g72-a:1.23`,
        fs:  `cpe:2.3:a:drupal:drupal:8.0.0:alpha15:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3crwe754g72-a:1.24`,
        fs:  `cpe:2.3:o:cisco:ons_15454_optical_transport_platform:2.3\(5\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3crwe754g72-a:1.27`,
        fs:  `cpe:2.3:a:hummingbird:docs_fusion_server:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:3crx506-96:-`,
        fs:  `cpe:2.3:o:juniper:junos:16.2:r2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3com:3cserver:-`,
        fs:  `cpe:2.3:a:webyog:sqlyog:8.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3com:3cserver:1.1`,
        fs:  `cpe:2.3:h:dell:3110cn:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3com:3ctftpsvc:-`,
        fs:  `cpe:2.3:a:ocportal:ocportal:7.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3com:3ctftpsvc:2.0.1`,
        fs:  `cpe:2.3:a:apache:jackrabbit:2.6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:hiperarc:4.2.29`,
        fs:  `cpe:2.3:a:digium:asterisk:1.4.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:nbx_1000:-`,
        fs:  `cpe:2.3:a:dan_kogai:encode_module:2.08:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:netbuilder_2_bridge_router:-`,
        fs:  `cpe:2.3:h:sophos:unified_threat_management:320:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:superstack_3_switch_4924:-`,
        fs:  `cpe:2.3:a:facebook_like_box_project:facebook_like_box:2.8.2:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:superstack_ii_2000:-`,
        fs:  `cpe:2.3:o:hp:tru64:5.1b1:pk3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:switch_4005:-`,
        fs:  `cpe:2.3:a:docker:docker:0.4.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:switch_4007r:-`,
        fs:  `cpe:2.3:a:compuware:servervantage_agent:10.2.734:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3com:tftp_server:1.05`,
        fs:  `cpe:2.3:a:docker:docker:1.6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:tippingpoint_200:-`,
        fs:  `cpe:2.3:a:ibm:db2_connect:10.1.0.4:*:*:*:application_server:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:tippingpoint_200e:-`,
        fs:  `cpe:2.3:a:swi-prolog:swi-prolog:5.10.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:tippingpoint_2400e:-`,
        fs:  `cpe:2.3:a:hp:insight_management_wbem_providers_for_windows_server_2003\/2008:2.7.0.0:*:*:*:*:*:x64:*`,
    },
     
    {
        uri: `cpe:/h:3com:tippingpoint_50:-`,
        fs:  `cpe:2.3:a:alsunna_project:alsunna:0.1:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:tippingpoint_5000e:-`,
        fs:  `cpe:2.3:a:ibm:iss_realsecure_network:7.0:xpu_20.6:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:tippingpoint_600e:-`,
        fs:  `cpe:2.3:h:hp:j9641a:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3com:tippingpoint_ips:-`,
        fs:  `cpe:2.3:a:apache:http_server:1.3.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:tippingpoint_ips:-`,
        fs:  `cpe:2.3:a:bea:weblogic_server:6.1:sp7:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:tippingpoint_ips_1200e:-`,
        fs:  `cpe:2.3:a:elasticsearch:kibana:6.0.0:beta2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3com:tippingpoint_ips_tos:2.1`,
        fs:  `cpe:2.3:a:octobercms:october_cms:1.0.382:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:3com:tippingpoint_ips_tos:2.1`,
        fs:  `cpe:2.3:a:oracle:jdk:1.7.0:update4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3com:tippingpoint_ips_tos:2.1.3.6323`,
        fs:  `cpe:2.3:a:libtiff:libtiff:3.4:beta34:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:3com:tippingpoint_ips_tos:2.1.3.6323`,
        fs:  `cpe:2.3:a:tedfelix:acpid2:2.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3com:tippingpoint_ips_tos:2.1.4.6324`,
        fs:  `cpe:2.3:a:vmware:vsphere_data_protection:5.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:3com:tippingpoint_ips_tos:2.1.4.6324`,
        fs:  `cpe:2.3:a:smarty:smarty:2.6.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3com:tippingpoint_ips_tos:2.2`,
        fs:  `cpe:2.3:a:gnu:automake:1.11.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:3com:tippingpoint_ips_tos:2.2`,
        fs:  `cpe:2.3:a:apache:hadoop:1.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3com:tippingpoint_ips_tos:2.2.0.6504`,
        fs:  `cpe:2.3:a:postfix:postfix:2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:3com:tippingpoint_ips_tos:2.2.0.6504`,
        fs:  `cpe:2.3:a:sir:gnuboard:4.34.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3com:tippingpoint_ips_tos:2.2.1`,
        fs:  `cpe:2.3:a:google:chrome:4.0.223.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:3com:tippingpoint_ips_tos:2.2.1`,
        fs:  `cpe:2.3:a:novaces:concerto_web:5.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3com:tippingpoint_ips_tos:2.2.1.6506`,
        fs:  `cpe:2.3:a:daniel_friesel:feh:1.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:3com:tippingpoint_ips_tos:2.2.1.6506`,
        fs:  `cpe:2.3:a:ibm:tivoli_storage_manager:6.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3com:tippingpoint_ips_tos:2.2.2`,
        fs:  `cpe:2.3:a:david_paleino:wicd:1.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:3com:tippingpoint_ips_tos:2.2.2`,
        fs:  `cpe:2.3:a:marc_lehmann:rxvt-unicode:9.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3com:tippingpoint_ips_tos:2.2.3`,
        fs:  `cpe:2.3:a:photogallerycreator:flash-album-gallery:1.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:3com:tippingpoint_ips_tos:2.2.3`,
        fs:  `cpe:2.3:a:kolja_schleich:leaguemanager:2.2:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:3com:tippingpoint_ips_tos:2.2.3.6514`,
        fs:  `cpe:2.3:a:automattic:jetpack:2.2.3:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/o:3com:tippingpoint_ips_tos:2.2.3.6514`,
        fs:  `cpe:2.3:a:bitrix:mpbuilder:1.0.11:*:*:*:*:bitrix:*:*`,
    },
     
    {
        uri: `cpe:/a:3com:tippingpoint_ips_tos:2.2.4`,
        fs:  `cpe:2.3:a:apache:synapse:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:3com:tippingpoint_ips_tos:2.2.4`,
        fs:  `cpe:2.3:h:swann:srnvw-470lcd:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3com:tippingpoint_ips_tos:2.5`,
        fs:  `cpe:2.3:a:adobe:photoshop:8.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:3com:tippingpoint_ips_tos:2.5`,
        fs:  `cpe:2.3:o:cisco:ios:12.3\(8\)yh:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3com:tippingpoint_ips_tos:2.5.1`,
        fs:  `cpe:2.3:a:ibm:informix_extended_parallel_server:8.40_uc2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:3com:tippingpoint_ips_tos:2.5.1`,
        fs:  `cpe:2.3:a:moodle:moodle:2.8.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3com:tippingpoint_sms_client:3.1.1.7943`,
        fs:  `cpe:2.3:a:webkitgtk:webkitgtk\&#43;:2.6.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:tippingpoint_sms_server:-`,
        fs:  `cpe:2.3:a:samba:samba:2.2.3:a:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:tippingpoint_sms_server:2.2.1.4477`,
        fs:  `cpe:2.3:a:wave:preboot_manager:3.0.0.154:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:total_control_netserver_card:-`,
        fs:  `cpe:2.3:a:oracle:jdk:1.7.0:update_141:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:total_control_netserver_card:3.7.24`,
        fs:  `cpe:2.3:a:barracudadrive:barracudadrive:4.5.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:webbngss3nbxnts:-`,
        fs:  `cpe:2.3:a:rocomotion:p_up_board:1.13:*:g:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:webbngss3nbxnts:4.0.17`,
        fs:  `cpe:2.3:a:ipswitch:imail:8.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:webbngss3nbxnts:4.1.4`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.5:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:webbngss3nbxnts:4.1.21`,
        fs:  `cpe:2.3:o:linux:linux_kernel:4.12.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:3com:webbngss3nbxnts:4.2.7`,
        fs:  `cpe:2.3:a:fortinet:fortiweb:5.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3dprotect:3d_eqsecure:4.2:-:professional`,
        fs:  `cpe:2.3:a:juniper:networks_mobility_system_software:7.5.1.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3ds:3dvia_composer:6.8.1.1652`,
        fs:  `cpe:2.3:h:3com:3c17302:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3ivx:mpeg-4:4.5`,
        fs:  `cpe:2.3:a:ibm:tivoli_storage_manager_for_space_management:6.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3ivx:mpeg-4:4.5.1`,
        fs:  `cpe:2.3:a:google:chrome:23.0.1271.84:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3ivx:mpeg-4:5.0`,
        fs:  `cpe:2.3:o:hp:procurve_switch_software:h.08.79:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3ivx:mpeg-4:5.0.1`,
        fs:  `cpe:2.3:a:inkscape:inkscape:0.47:pre0:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3ivx:mpeg-4:5.0.2`,
        fs:  `cpe:2.3:a:microsoft:ie:4.0.1:sp2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3par:inform_command_line_interface:2.3.1.330`,
        fs:  `cpe:2.3:a:netscape:navigator:4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3par:inform_management_console:4.1.1.5`,
        fs:  `cpe:2.3:a:otrs:otrs:1.1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3s-software:codesys_gateway-server:2.3.5.1`,
        fs:  `cpe:2.3:a:infor:eclient:7.3.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3s-software:codesys_gateway-server:2.3.5.2`,
        fs:  `cpe:2.3:a:freetype:freetype:2.3.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3s-software:codesys_gateway-server:2.3.5.3`,
        fs:  `cpe:2.3:h:axis:2120_network_camera:2.40:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3s-software:codesys_gateway-server:2.3.6.0`,
        fs:  `cpe:2.3:o:cisco:ios:12.0\(14\)w5\(20\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3s-software:codesys_gateway-server:2.3.7.0`,
        fs:  `cpe:2.3:h:hp:laserjet_9000_mfp:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3s-software:codesys_gateway-server:2.3.8.0`,
        fs:  `cpe:2.3:a:cisco:asr_5000_software:17.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3s-software:codesys_gateway-server:2.3.8.1`,
        fs:  `cpe:2.3:a:google:chrome:5.0.375.50:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3s-software:codesys_gateway-server:2.3.8.2`,
        fs:  `cpe:2.3:a:moinejf:abcm2ps:2.11.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3s-software:codesys_gateway-server:2.3.9`,
        fs:  `cpe:2.3:a:oracle:mysql:5.1.63:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3s-software:codesys_gateway-server:2.3.9.1`,
        fs:  `cpe:2.3:a:cisco:business_edition_3000_software:8.6.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3s-software:codesys_gateway-server:2.3.9.2`,
        fs:  `cpe:2.3:o:f5:tmos:9.4.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3s-software:codesys_gateway-server:2.3.9.3`,
        fs:  `cpe:2.3:a:solarwinds:integrated_virtual_infrastructure_monitor:1.1.674.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3s-software:codesys_gateway-server:2.3.9.4`,
        fs:  `cpe:2.3:a:gitlab:gitlab:2.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3s-software:codesys_gateway-server:2.3.9.5`,
        fs:  `cpe:2.3:a:moinejf:abcm2ps:1.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3s-software:codesys_gateway-server:2.3.9.18`,
        fs:  `cpe:2.3:a:apache:http_server:1.3.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3s-software:codesys_gateway-server:2.3.9.19`,
        fs:  `cpe:2.3:a:apache:traffic_server:2.1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3s-software:codesys_gateway-server:2.3.9.27`,
        fs:  `cpe:2.3:a:apple:pages:5.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3s-software:codesys_gateway-server:2.3.9.38`,
        fs:  `cpe:2.3:a:drupal:drupal:8.0.0:alpha3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3s-software:codesys_runtime_system:-`,
        fs:  `cpe:2.3:a:hitachi:cosminexus_application_server:06-51-\/b:-:standard:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3s-software:codesys_runtime_system:2.3.9.8`,
        fs:  `cpe:2.3:a:lesterchan:wp-postviews:1.00:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:3s-software:codesys_runtime_system:2.3.9.35`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.2.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3s-software:codesys_runtime_system:2.3.9.36`,
        fs:  `cpe:2.3:a:oracle:mysql:5.5.44:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3s-software:codesys_runtime_system:2.3.9.37`,
        fs:  `cpe:2.3:h:softbank:sharp_3g_handset:945shg:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3s-software:codesys_runtime_system:2.3.9.47`,
        fs:  `cpe:2.3:a:rpm:rpm:2.2.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:3sgroup:t2css:2.0`,
        fs:  `cpe:2.3:a:privatemsg_project:privatemsg:7.x-1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:4t-niagara:tray_minimizer_free:4.40`,
        fs:  `cpe:2.3:a:google:chrome:4.0.250.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:4t-niagara:tray_minimizer_pro:4.40`,
        fs:  `cpe:2.3:a:keepass:keepass:1.04:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:4you-studio:com_jphone:1.0:alpha3`,
        fs:  `cpe:2.3:o:cisco:vpn_3060_concentrator:4.7.2.a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:4you-studio:com_jphone:1.0:alpha4`,
        fs:  `cpe:2.3:a:microsoft:visual_studio:97:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:63reasons:supercron:-`,
        fs:  `cpe:2.3:a:irfanview:irfanview:4.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:6677g:girls_games_-_shoes_maker:1.0.1::~~~android~~`,
        fs:  `cpe:2.3:a:webmin:usermin:1.040:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:3.13`,
        fs:  `cpe:2.3:a:nagios:remote_plug_in_executor:2.5.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.20`,
        fs:  `cpe:2.3:a:brian_akers:mod_layout:3.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.23`,
        fs:  `cpe:2.3:a:lunascape:ilunascape_android:1.0.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.24:beta`,
        fs:  `cpe:2.3:a:mozilla:bugzilla:4.2.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.25:beta`,
        fs:  `cpe:2.3:a:nuance:openspeech_recognizer:3.0.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.26:beta`,
        fs:  `cpe:2.3:a:rsyslog:rsyslog:7.3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.27:beta`,
        fs:  `cpe:2.3:a:graphite_project:graphite:0.9.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.28:beta`,
        fs:  `cpe:2.3:o:yamaha:rt102i:2.02.46:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.29:beta`,
        fs:  `cpe:2.3:a:google:chrome:23.0.1271.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.30:beta`,
        fs:  `cpe:2.3:a:apache:tomcat:6.0.48:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.31`,
        fs:  `cpe:2.3:a:google:chrome:31.0.1650.38:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.32`,
        fs:  `cpe:2.3:a:google:chrome:37.0.2062.49:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.33:beta`,
        fs:  `cpe:2.3:o:schneider-electric:telvent_sage_3030_firmware:c3413-500-001d3_p4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.34:beta`,
        fs:  `cpe:2.3:a:rimarts:becky\!_internet_mail:2.30.02:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.35:beta`,
        fs:  `cpe:2.3:a:atmail:atmail:7.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.36:beta`,
        fs:  `cpe:2.3:a:mcafee:antivirus_engine:4.3.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.37:beta`,
        fs:  `cpe:2.3:a:ibm:client_access:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.38:beta`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.0.39:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.39:beta`,
        fs:  `cpe:2.3:a:samba:samba:3.4.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.40:beta`,
        fs:  `cpe:2.3:a:moinmo:moinmoin:0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.41:beta`,
        fs:  `cpe:2.3:a:plume-cms:plume_cms:1.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.42`,
        fs:  `cpe:2.3:a:symantec:netbackup_opscenter:7.6.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.43:beta`,
        fs:  `cpe:2.3:a:ca:etrust_directory:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.44:beta`,
        fs:  `cpe:2.3:o:cisco:ios:15.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.45:beta`,
        fs:  `cpe:2.3:a:gnome:gnome_display_manager:2.21.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.46:beta`,
        fs:  `cpe:2.3:a:hp:application_lifecycle_management:11.51:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.47:beta`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.28:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.48:beta`,
        fs:  `cpe:2.3:a:sophos:web_appliance:3.7.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.49:beta`,
        fs:  `cpe:2.3:o:cisco:ios_xr:3.9\(.1\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.50:beta`,
        fs:  `cpe:2.3:a:ibm:emptoris_strategic_supply_management:10.0.2.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.51:beta`,
        fs:  `cpe:2.3:a:mozilla:bugzilla:3.6.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.52:beta`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.5:p144:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.53:beta`,
        fs:  `cpe:2.3:h:d-link:dap-2230:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.54:beta`,
        fs:  `cpe:2.3:a:plume-cms:plume_cms:1.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.55:beta`,
        fs:  `cpe:2.3:o:wdc:my_net_firmware:1.06.28:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.56:beta`,
        fs:  `cpe:2.3:a:sun:jdk:1.3.1_12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.57`,
        fs:  `cpe:2.3:a:adobe:acrobat_reader:7.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.58:beta`,
        fs:  `cpe:2.3:a:telaxus:epesi:1.8.2.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.59:beta`,
        fs:  `cpe:2.3:a:moinejf:abcm2ps:4.8.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.60:beta`,
        fs:  `cpe:2.3:h:juniper:idp8200:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.61:beta`,
        fs:  `cpe:2.3:a:percona:percona_server:5.6.15-63.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.62`,
        fs:  `cpe:2.3:a:oracle:peoplesoft_enterprise_customer_relationship_management:9.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.63`,
        fs:  `cpe:2.3:a:adobe:widget_browser:2.0_build_230:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.64`,
        fs:  `cpe:2.3:a:ponsoftware:explzh:3.79:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.65`,
        fs:  `cpe:2.3:a:trihedral:vtscada:10.2.08:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:4.65::~~~~x64~`,
        fs:  `cpe:2.3:a:oracle:database_server:9.2.0.8dv:r2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:9.04:beta`,
        fs:  `cpe:2.3:a:oracle:agile_product_lifecycle_management:9.3.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:9.06:beta`,
        fs:  `cpe:2.3:a:recurly:recurly_client_ruby:2.3.0:beta1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:9.07:beta`,
        fs:  `cpe:2.3:o:cisco:ios:12.2yk:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:9.10:beta`,
        fs:  `cpe:2.3:a:ca:messaging:1.07.210.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:9.11:beta`,
        fs:  `cpe:2.3:a:texmacs:texmacs:1.0.6.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:9.12:beta`,
        fs:  `cpe:2.3:a:ibm:emptoris_strategic_supply_management:10.1.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:9.13:beta`,
        fs:  `cpe:2.3:a:ceph:ceph-deploy:1.5.22:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:9.20`,
        fs:  `cpe:2.3:a:digium:asterisk:1.8.17.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:9.20::~~~~x64~`,
        fs:  `cpe:2.3:a:cisco:wide_area_application_services_mobile:3.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:9.22:beta`,
        fs:  `cpe:2.3:a:quagga:quagga:0.99.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7-zip:15.14`,
        fs:  `cpe:2.3:h:asus:rt-n14u:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7zip:9.20`,
        fs:  `cpe:2.3:a:cisco:content_distribution_manager_4630:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:7zip:15.05:beta`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.4.5-7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:0.80`,
        fs:  `cpe:2.3:h:intel:active_management_technology:6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:0.81`,
        fs:  `cpe:2.3:a:gnome:glib:2.15.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:0.90`,
        fs:  `cpe:2.3:a:google:chrome:11.0.696.31:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:0.91`,
        fs:  `cpe:2.3:a:nginx:nginx:1.5.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:4.10`,
        fs:  `cpe:2.3:a:mozilla:bugzilla:4.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:4.12`,
        fs:  `cpe:2.3:o:iij:seil_turbo_firmware:2.06:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:4.13`,
        fs:  `cpe:2.3:a:apache:groovy:1.8.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:4.14`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(31\)sb21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:4.14.01`,
        fs:  `cpe:2.3:a:ibm:tivoli_netview_os_390:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:4.16`,
        fs:  `cpe:2.3:a:yahoo:yui:3.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:4.18`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.31:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:4.20`,
        fs:  `cpe:2.3:a:eviware:soapui:3.6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:4.27`,
        fs:  `cpe:2.3:o:cisco:ios:12.3yj:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:4.29`,
        fs:  `cpe:2.3:a:checkpoint:endpoint_security:r73:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:4.30`,
        fs:  `cpe:2.3:a:ibm:security_guardium:10.01:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:4.33`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.5.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:4.37`,
        fs:  `cpe:2.3:a:x2go:x2go_server:4.0.0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:4.39`,
        fs:  `cpe:2.3:o:ibm:advanced_management_module_firmware:3.64:bpet64g:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:4.42`,
        fs:  `cpe:2.3:a:netsweeper:netsweeper:2.6.29.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:4.43`,
        fs:  `cpe:2.3:a:google:chrome:10.0.645.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:4.44`,
        fs:  `cpe:2.3:a:ains:foiaxpress_ocr_service:6.4.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:4.45`,
        fs:  `cpe:2.3:a:net-snmp:net-snmp:5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:4.47`,
        fs:  `cpe:2.3:a:landesk:landesk_management_suite:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:4.48`,
        fs:  `cpe:2.3:a:exactcode:exactimage:0.0.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:4.49`,
        fs:  `cpe:2.3:a:netweblogic:login_with_ajax:3.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:4.51`,
        fs:  `cpe:2.3:a:cisco:telepresence_manager:1.1.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:4.53`,
        fs:  `cpe:2.3:a:keepass:keepass:2.04:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:4.55`,
        fs:  `cpe:2.3:a:mysql:mysql:5.1.37:sp1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:4.57`,
        fs:  `cpe:2.3:o:greenbone:greenbone_os:3.1.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:4.58`,
        fs:  `cpe:2.3:a:octobercms:october_cms:1.0.138:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:4.61`,
        fs:  `cpe:2.3:a:python:python:2.7.2:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:4.65`,
        fs:  `cpe:2.3:h:cisco:ons_15454_sdh_multiservice_provisioning_platform:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:9.04`,
        fs:  `cpe:2.3:a:web2py:web2py:1.89.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:9.20.1`,
        fs:  `cpe:2.3:a:gnu:tar:1.13.25:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7-zip:p7zip:16.02`,
        fs:  `cpe:2.3:a:ajaydsouza:contextual_related_posts:1.7.2:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:7_habits_personal_development_project:7_habits_personal_development:1::~~~android~~`,
        fs:  `cpe:2.3:o:cisco:vpn_3000_concentrator_series_software:4.1.7.a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7725:7725.com_three_kingdoms:2.4::~~~android~~`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(12\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7andi-fs.co:denny%27s:1.0.1::~~~android~~`,
        fs:  `cpe:2.3:a:citrix:access_gateway:4.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7andi-fs.co:denny%27s:1.0.2::~~~android~~`,
        fs:  `cpe:2.3:a:apache:subversion:1.8.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7andi-fs.co:denny%27s:2.0.0::~~~android~~`,
        fs:  `cpe:2.3:a:ninjaforge:com_ninjaxplorer:1.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7andi-fs.co:denny%27s:2.0.1::~~~android~~`,
        fs:  `cpe:2.3:a:ibm:marketing_platform:10.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7mediaws:edutrac:1.0.0`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.351:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7mediaws:edutrac:1.0.1`,
        fs:  `cpe:2.3:a:ibm:websphere_commerce_suite:4.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7mediaws:edutrac:1.0.2`,
        fs:  `cpe:2.3:a:docker:docker:0.6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7mediaws:edutrac:1.0.3`,
        fs:  `cpe:2.3:a:microsoft:visual_j\#:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7mediaws:edutrac:1.0.4`,
        fs:  `cpe:2.3:a:ibm:rational_directory_server:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7mediaws:edutrac:1.0.5`,
        fs:  `cpe:2.3:a:yahoo:messenger:10.0.0.1241:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7mediaws:edutrac:1.0.6`,
        fs:  `cpe:2.3:a:google:chrome:24.0.1312.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7mediaws:edutrac:1.0.7`,
        fs:  `cpe:2.3:a:john_koleszar:libvpx:0.9.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7mediaws:edutrac:1.0.8`,
        fs:  `cpe:2.3:a:bestpractical:rt:3.8.0:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7mediaws:edutrac:1.0.9`,
        fs:  `cpe:2.3:a:pineapp:mail-secure:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7mediaws:edutrac:1.1.1`,
        fs:  `cpe:2.3:o:linux:linux_kernel:4.1.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7sage:7sage_lsat_prep_-_proctor:2.1.1::~~~android~~`,
        fs:  `cpe:2.3:o:sgi:irix:4.0.5b:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7t:aquis:1.5`,
        fs:  `cpe:2.3:o:redhat:enterprise_linux:3:update5:es:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7t:termis:2.0`,
        fs:  `cpe:2.3:a:ushahidi:ushahidi_platform:2.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:7t:termis:2.10`,
        fs:  `cpe:2.3:a:reportico:php_report_designer:1.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:9.dotpp.net:megalith:12`,
        fs:  `cpe:2.3:h:cisco:catalyst_4224:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:9.dotpp.net:megalith:27`,
        fs:  `cpe:2.3:a:dell:openmanage_server_administrator:6.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:9.dotpp.net:megalith:50`,
        fs:  `cpe:2.3:a:gnu:gnump3d:2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:9.dotpp.net:megalith:51`,
        fs:  `cpe:2.3:a:gnome:eye_of_gnome:3.20.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:9.dotpp.net:megalith:52`,
        fs:  `cpe:2.3:a:gnu:glibc:2.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:90meter:cac_smartcard_manager-90:1.1.1.6`,
        fs:  `cpe:2.3:a:skyphe:file-gallery:1.6.5.2:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:90meter:cac_smartcard_manager-90:1.1.8`,
        fs:  `cpe:2.3:a:gnome:gnome-shell:3.5.90:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:90meter:cac_smartcard_manager-90:1.2.04`,
        fs:  `cpe:2.3:a:hp:webproxy:2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:90meter:cac_smartcard_manager-90:1.2.14`,
        fs:  `cpe:2.3:a:clamav:clamav:0.90.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:90meter:card_issuance_workstation:1.00.130`,
        fs:  `cpe:2.3:a:ftprush:ftprush:1.0.0.616:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:99only:buy_99_cents_only_products:0.1::~~~android~~`,
        fs:  `cpe:2.3:a:cakephp:cakephp:2.6.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:9bis:kitty:0.66.6.3`,
        fs:  `cpe:2.3:o:yahama:rt200i:4.00.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:9gag:9gag_-_funny_pics_and_videos:2.4.10::~~~android~~`,
        fs:  `cpe:2.3:a:ows:scald:7.x-1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:9jacompass:epl_hat_trick:1::~~~android~~`,
        fs:  `cpe:2.3:a:realnetworks:realplayer_sp:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:a-pdf:restrictions_remover:1.6`,
        fs:  `cpe:2.3:a:ibm:tivoli_storage_manager:5.1.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:a10networks:advanced_core_operating_system:2.7.0`,
        fs:  `cpe:2.3:a:vmware:vcenter_chargeback_manager:1.6.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:a10networks:advanced_core_operating_system:2.7.1`,
        fs:  `cpe:2.3:a:cisco:wireless_lan_controller_software:8.0.72.140:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:a9tech:a9cad:2.2.0`,
        fs:  `cpe:2.3:a:mortbay:jetty:3.0.a94:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aas9:zerocms:1.0`,
        fs:  `cpe:2.3:a:tigris:tortoisesvn:1.1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aas9:zerocms:1.3.2`,
        fs:  `cpe:2.3:a:avonesoft:rm_converter:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aas9:zerocms:1.3.3`,
        fs:  `cpe:2.3:a:ibm:lotus_domino:8.5.3.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ab_google_map_travel_project:ab_google_map_travel:3.4::~~~wordpress~~`,
        fs:  `cpe:2.3:a:ntp:ntp:4.3.86:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abb:datamanager:1.0.0`,
        fs:  `cpe:2.3:a:redhat:rhel_virtualization:5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abb:datamanager:6.3.6`,
        fs:  `cpe:2.3:a:openssl:openssl:1.0.0:beta5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abb:panel_builder_800:5.1`,
        fs:  `cpe:2.3:a:stanislas_rolland:sr_feuser_register:1.4.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abb:robotstudio:5.60`,
        fs:  `cpe:2.3:a:bestpractical:rt:3.4.3:pre1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abb:robotstudio:5.61`,
        fs:  `cpe:2.3:a:cisco:adaptive_security_appliance_software:8.2\(5.35\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abb:robotstudio:5.61.01`,
        fs:  `cpe:2.3:a:xnview:xnview:1.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abb:test_signal_viewer:1.5`,
        fs:  `cpe:2.3:a:google:chrome:28.0.1500.39:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abbyy:finereader:4.1`,
        fs:  `cpe:2.3:a:phpjabbers:event_booking_calendar:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abbyy:finereader:5.0:-:office`,
        fs:  `cpe:2.3:a:ovirt:ovirt:3.5.0:beta2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abbyy:finereader:5.0:-:pro`,
        fs:  `cpe:2.3:o:linux:linux_kernel:4.4.24:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abbyy:finereader:6.0`,
        fs:  `cpe:2.3:a:oracle:application_object_library:12.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abbyy:finereader:6.0:-:corporate`,
        fs:  `cpe:2.3:a:connections_project:connections:0.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abbyy:finereader:6.0:-:pro`,
        fs:  `cpe:2.3:a:ibm:spss_modeler:15.0.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abbyy:finereader:7.0:-:corporate`,
        fs:  `cpe:2.3:a:avast:free_antivirus:11.1.2261:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abbyy:finereader:7.0:-:pro`,
        fs:  `cpe:2.3:h:netgear:wgr614:v1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abbyy:finereader:8.0:-:corporate`,
        fs:  `cpe:2.3:a:zend:zend_framework:1.6.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abbyy:finereader:8.0:-:pro`,
        fs:  `cpe:2.3:a:apache:poi:3.1:beta1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abbyy:finereader:9.0:-:corporate`,
        fs:  `cpe:2.3:a:ibm:tivoli_directory_server:6.1.0.51:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abbyy:finereader:9.0:-:pro`,
        fs:  `cpe:2.3:a:adobe:flash_player:11.2.202.238:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abbyy:finereader:10.0:-:corporate`,
        fs:  `cpe:2.3:a:penguinchefshop_project:penguinchefshop:1.0.1:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:abbyy:finereader:10.0:-:pro`,
        fs:  `cpe:2.3:a:rim:blackberry_desktop_software:4.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abbyy:finereader:10.0:-:site_license`,
        fs:  `cpe:2.3:a:claws-mail:claws-mail:2.6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abbyy:finereader_sprint:5.0.0.22227`,
        fs:  `cpe:2.3:o:yahama:rt140p:6.03.33:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abbyy:finereader_sprint:6.0`,
        fs:  `cpe:2.3:a:strongswan:strongswan:4.3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abbyy:finereader_sprint:6.0.1395.4512`,
        fs:  `cpe:2.3:a:postfix:postfix:2.3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abbyy:finereader_sprint:6.0.1735.41615`,
        fs:  `cpe:2.3:o:cisco:email_security_appliance_firmware:9.6.0-051:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abbyy:finereader_sprint:6.0.1990.41618`,
        fs:  `cpe:2.3:a:altiris:recovery_solution:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abbyy:finereader_sprint:9.0`,
        fs:  `cpe:2.3:a:google:chrome:12.0.742.122:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abbyy:finereader_sprint_plus:6.0`,
        fs:  `cpe:2.3:a:mixpanel_project:mixpanel:6.x-1.0:beta1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abeel:simple_php_agenda:0.1`,
        fs:  `cpe:2.3:a:apache:traffic_server:3.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abeel:simple_php_agenda:0.1.1`,
        fs:  `cpe:2.3:a:autodesk:dwf_viewer:7.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abeel:simple_php_agenda:0.1.2`,
        fs:  `cpe:2.3:h:hp:procurve_switch:jc101a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abeel:simple_php_agenda:0.2.0`,
        fs:  `cpe:2.3:a:digium:certified_asterisk:11.6:cert14:*:*:lts:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abeel:simple_php_agenda:0.2.1`,
        fs:  `cpe:2.3:a:glpi-project:glpi:0.84.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abeel:simple_php_agenda:0.2.2`,
        fs:  `cpe:2.3:a:gnome:gnome_display_manager:2.21.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abeel:simple_php_agenda:0.2.3`,
        fs:  `cpe:2.3:a:ibm:maximo_asset_management:7.1.1.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abeel:simple_php_agenda:0.2.4`,
        fs:  `cpe:2.3:a:microsoft:word:97:sr2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abeel:simple_php_agenda:0.2.5`,
        fs:  `cpe:2.3:a:google:chrome:12.0.742.70:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abeel:simple_php_agenda:0.2.6`,
        fs:  `cpe:2.3:a:rocomotion:pplog:3.29:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abeel:simple_php_agenda:0.2.7`,
        fs:  `cpe:2.3:a:gopivotal:grails:2.1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abeel:simple_php_agenda:0.3.0`,
        fs:  `cpe:2.3:a:jenkins:config_file_provider:2.14.2:beta:*:*:*:jenkins:*:*`,
    },
     
    {
        uri: `cpe:/a:abeel:simple_php_agenda:0.3.1`,
        fs:  `cpe:2.3:a:cisco:unified_contact_center_enterprise:7.5\(7\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abeel:simple_php_agenda:0.3.2`,
        fs:  `cpe:2.3:a:google:chrome:12.0.742.22:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abeel:simple_php_agenda:0.3.3`,
        fs:  `cpe:2.3:a:radio_de_la_cato_project:radio_de_la_cato:2.0:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:abeel:simple_php_agenda:1.0.0`,
        fs:  `cpe:2.3:a:c-3.co.jp:webcalenderc3:0.31:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abeel:simple_php_agenda:1.0.1`,
        fs:  `cpe:2.3:a:ibm:tivoli_application_dependency_discovery_manager:7.1.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abeel:simple_php_agenda:2.0.0`,
        fs:  `cpe:2.3:a:git_for_windows_project:git_for_windows:1.8.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abeel:simple_php_agenda:2.1.0`,
        fs:  `cpe:2.3:a:opscode:chef:0.8.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abeel:simple_php_agenda:2.2.0`,
        fs:  `cpe:2.3:a:mcafee:epolicy_orchestrator:4.6.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abeel:simple_php_agenda:2.2.1`,
        fs:  `cpe:2.3:a:lsoft:listserv:14.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abeel:simple_php_agenda:2.2.2`,
        fs:  `cpe:2.3:a:umn:mapserver:5.0.0:beta4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abeel:simple_php_agenda:2.2.3`,
        fs:  `cpe:2.3:a:moodle:moodle:3.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abeel:simple_php_agenda:2.2.4`,
        fs:  `cpe:2.3:a:tryton:tryton:3.2.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abeel:simple_php_agenda:2.2.5`,
        fs:  `cpe:2.3:a:sun:java_system_messaging_server:6.2:-:sparc:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abeel:simple_php_agenda:2.2.6`,
        fs:  `cpe:2.3:a:sun:jre:1.5.0.130:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abeel:simple_php_agenda:2.2.7`,
        fs:  `cpe:2.3:a:cart66:cart66_lite_plugin:1.5.0:-:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:abeel:simple_php_agenda:2.2.8`,
        fs:  `cpe:2.3:a:thulasidas:easy-adsense-lite:2.31:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abeel:simple_php_agenda:2.2.9`,
        fs:  `cpe:2.3:a:ibm:sterling_connect_direct_user_interface:1.4.0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abine:donottrackme_-_mobile_privacy:1.1.8::~~~android~~`,
        fs:  `cpe:2.3:a:microsoft:frontpage:98:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abrahamtours:abraham_tours:1.1.2::~~~android~~`,
        fs:  `cpe:2.3:a:google:chrome:9.0.563.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:absolute:computrace:8.0.885`,
        fs:  `cpe:2.3:a:agilebits:knox:2.0.1:-:*:*:*:mac_os_x:*:*`,
    },
     
    {
        uri: `cpe:/a:absolutefuturity:af_imagegrabber:1.0.3`,
        fs:  `cpe:2.3:a:webyog:sqlyog:6.0:-:enterprise:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:absolutengine:absolut_engine:1.73`,
        fs:  `cpe:2.3:h:hp:9500_vpn_firewall_module:jd245a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abtei-neuburg:stift_neuburg:1.1::~~~android~~`,
        fs:  `cpe:2.3:a:oracle:solaris_cluster:4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abu_ali_anasheeds_project:abu_ali_anasheeds:1.1::~~~android~~`,
        fs:  `cpe:2.3:a:ibm:db2_connect:10.5.0.4:*:*:*:unlimited:*:*:*`,
    },
     
    {
        uri: `cpe:/a:abushhab:alwasel:1.5`,
        fs:  `cpe:2.3:a:merethis:centreon:2.1.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accadvocacy:acc_advocacy_action:2::~~~android~~`,
        fs:  `cpe:2.3:a:vmware:vsphere_client:5.0.0.16964:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accelatech:bizsearch:3.2:-:~~~linux_kernel~~`,
        fs:  `cpe:2.3:a:socketmail:socketmail:2.2.9:-:professional:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accelatech:bizsearch:3.2:-:~~~sunos~~`,
        fs:  `cpe:2.3:a:mozilla:firefox:0.9.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accellion:kiteworks_appliance:kw2016.03.00`,
        fs:  `cpe:2.3:a:microsoft:biztalk_server:2002:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.0`,
        fs:  `cpe:2.3:h:airvana:hubbub_c1-600-rt:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.0.1`,
        fs:  `cpe:2.3:a:polarssl:polarssl:1.3.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.0.2`,
        fs:  `cpe:2.3:o:iij:seil_x1_firmware:2.74:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.0.3`,
        fs:  `cpe:2.3:a:simplemachines:simple_machines_forum:1.0.22:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.0.4`,
        fs:  `cpe:2.3:a:cisco:application_and_content_networking_system_software:5.5.29:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.0.5`,
        fs:  `cpe:2.3:a:ait-pro:bulletproof-security:.48.8:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.0.6`,
        fs:  `cpe:2.3:a:cerberusftp:ftp_server:3.1.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.0.7`,
        fs:  `cpe:2.3:a:zabbix:zabbix:1.6.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.0.8`,
        fs:  `cpe:2.3:a:apple:safari:9.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.0.9`,
        fs:  `cpe:2.3:a:squid-cache:squid:3.0.stable16:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.1.0`,
        fs:  `cpe:2.3:a:arthurdejong:nss-pam-ldapd:0.6.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.1.1`,
        fs:  `cpe:2.3:a:google:chrome:10.0.647.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.1.2`,
        fs:  `cpe:2.3:a:redhat:resteasy:2.3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.1.3`,
        fs:  `cpe:2.3:a:s9y:serendipity:2.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.1.4`,
        fs:  `cpe:2.3:a:cisco:webex_recording_format_player:27.32.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.1.5`,
        fs:  `cpe:2.3:a:videowhisper:videowhisper_live_streaming_integration:4.29.19:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.1.6`,
        fs:  `cpe:2.3:o:cisco:ios:12.0\(14\)s7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.1.7`,
        fs:  `cpe:2.3:a:dolibarr:dolibarr:3.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.1.8`,
        fs:  `cpe:2.3:a:fortinet:fortimail:5.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.1.9`,
        fs:  `cpe:2.3:a:microsoft:lync_basic:2013:-:x86:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.2.0`,
        fs:  `cpe:2.3:a:google:chrome:18.0.1025.47:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.2.0:b`,
        fs:  `cpe:2.3:h:hp:j9850a:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.2.1`,
        fs:  `cpe:2.3:a:stillbreathing:bannerman:0.2.2:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.3.0`,
        fs:  `cpe:2.3:a:adobe:spelling_dictionaries_support_for_adobe_reader:9.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.3.0:b`,
        fs:  `cpe:2.3:a:google:chrome:4.0.302.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.3.1`,
        fs:  `cpe:2.3:a:debian:cifs-utils:6.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.3.2`,
        fs:  `cpe:2.3:a:weathernews:weathernews_touch:2.1.0:-:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.3.3`,
        fs:  `cpe:2.3:o:cisco:unified_wireless_ip_phone_7920_firmware:1.0\(8\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.3.4`,
        fs:  `cpe:2.3:a:gnu:glibc:2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.3.5`,
        fs:  `cpe:2.3:a:apache:httpasyncclient:4.0:alpha2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.4.0`,
        fs:  `cpe:2.3:o:iij:seil_x1_firmware:3.60:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:1.4.1`,
        fs:  `cpe:2.3:a:ultraedit:ultraedit-32:10.20d\&#43;:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:10.1.3`,
        fs:  `cpe:2.3:a:nodejs:node.js:0.10.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:10.4.4`,
        fs:  `cpe:2.3:a:vmware:vma:4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:10.4.5`,
        fs:  `cpe:2.3:a:gitlab:gitlab:8.6.4:*:*:*:enterprise:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:10.4.6`,
        fs:  `cpe:2.3:a:apache:http_server:1.3.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accomplishtechnology:phpmydirectory:11.04.00`,
        fs:  `cpe:2.3:a:skype:skype:1.0.0.100:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accunetix:web_vulnerability_scanner:10`,
        fs:  `cpe:2.3:o:yamaha:rt57i:8.00.71:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:accusoft:imagxpress:7.0.74.0`,
        fs:  `cpe:2.3:a:david_king:vino:2.17.92:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acdsee:photo_editor_2008:5.0:build291`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.5.61:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acdsystems:acdsee:5.0.0::~~standard~~~`,
        fs:  `cpe:2.3:a:google:chrome:25.0.1364.95:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:acer:ac700_chromebook:-`,
        fs:  `cpe:2.3:a:dart:powertcp_zip_compression:1.8.5.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aceviral:angry_gran_toss:1.1.1::~~~android~~`,
        fs:  `cpe:2.3:o:redhat:linux:4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acidcat:acidcat_cms:3.5.1`,
        fs:  `cpe:2.3:a:logitech:lws_gallery:13.20.1166.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acidcat:acidcat_cms:3.5.2`,
        fs:  `cpe:2.3:a:pkware:pkzip:4.00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acidcat:acidcat_cms:3.5.6`,
        fs:  `cpe:2.3:a:xhanch:my_twitter:2.7.4:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:acl:acl:9.1.0.213`,
        fs:  `cpe:2.3:a:yahoo:messenger:5.6.0.1358:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acm.org:er_assistant:2.0`,
        fs:  `cpe:2.3:a:nginx:nginx:1.5.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acmailer:acmailer:3.8`,
        fs:  `cpe:2.3:a:apache:groovy:1.7.0:beta_2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acmailer:acmailer:3.8.17`,
        fs:  `cpe:2.3:o:cisco:catos:5.5\(1\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acmailer:acmailer:3.9`,
        fs:  `cpe:2.3:a:happyworm:jplayer:2.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acmailer:acmailer:3.9.10:beta`,
        fs:  `cpe:2.3:a:oracle:enterprise_manager_plugin_for_database_control:12.1.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:micro_httpd:-`,
        fs:  `cpe:2.3:a:fedoraproject:389_directory_server:1.3.3.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:mini_httpd:1.21`,
        fs:  `cpe:2.3:a:samba:samba:4.0.23:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:1.95`,
        fs:  `cpe:2.3:a:webkitgtk:webkitgtk\&#43;:2.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.00`,
        fs:  `cpe:2.3:a:polycom:rmx_system_software:7.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.01`,
        fs:  `cpe:2.3:a:google:chrome:20.0.1132.27:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.02`,
        fs:  `cpe:2.3:a:webyog:sqlyog:6.55:-:enterprise:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.03`,
        fs:  `cpe:2.3:a:gnu:privacy_guard:8.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.04`,
        fs:  `cpe:2.3:a:adobe:xmp_panels_cs3:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.05`,
        fs:  `cpe:2.3:a:cisco:wireless_lan_controller_software:8.1.122.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.06`,
        fs:  `cpe:2.3:h:cisco:sf302-08pp:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.07`,
        fs:  `cpe:2.3:a:google:chrome:25.0.1364.110:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.08`,
        fs:  `cpe:2.3:o:huawei:honor_4c_firmware:chm-tl00c01_b535:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.09`,
        fs:  `cpe:2.3:a:ati:catalyst_control_center_help:2009.211.2224.40219:*:*:hu:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.10`,
        fs:  `cpe:2.3:a:digium:asterisk:1.8.12.0:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.11`,
        fs:  `cpe:2.3:a:zabbix:zabbix:2.0.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.12`,
        fs:  `cpe:2.3:a:microsoft:sql_srv:7.0:sp4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.13`,
        fs:  `cpe:2.3:a:google:chrome:14.0.811.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.14`,
        fs:  `cpe:2.3:a:dotnetnuke:dotnetnuke:6.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.15`,
        fs:  `cpe:2.3:o:cisco:ios:12.2mb:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.16`,
        fs:  `cpe:2.3:a:emc:networker:8.0.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.17`,
        fs:  `cpe:2.3:a:hp:vertica:7.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.18`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.7:p242:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.19`,
        fs:  `cpe:2.3:a:roundup-tracker:roundup:1.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.20`,
        fs:  `cpe:2.3:a:lesterchan:wp-postviews:1.66:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.20:b`,
        fs:  `cpe:2.3:o:oracle:opus_10g_ethernet_switch_family:1.3.1.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.20:c`,
        fs:  `cpe:2.3:a:umi-cms:umi.cms:2.8.5.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.21`,
        fs:  `cpe:2.3:a:mozilla:mozilla_suite:1.7.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.21:b`,
        fs:  `cpe:2.3:o:microsoft:windows_nt:3.5.1:sp1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.22`,
        fs:  `cpe:2.3:a:hp:wbem:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.23`,
        fs:  `cpe:2.3:a:ovirt:ovirt:3.4.0:beta2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.24`,
        fs:  `cpe:2.3:a:sawmill:sawmill:6.2.8a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.25`,
        fs:  `cpe:2.3:a:horde:groupware:5.0.4:*:*:*:webmail:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acme:thttpd:2.25:b`,
        fs:  `cpe:2.3:a:ibm:db2:11.1:*:*:*:workgroup:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acobot_live_chat_%26_contact_form_project:acobot_live_chat_%26_contact_form:2.0::~~~wordpress~~`,
        fs:  `cpe:2.3:a:git-scm:git:1.5.4.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acquia:commons:6.x-2.4`,
        fs:  `cpe:2.3:o:siemens:simatic_s7-1500_cpu_firmware:1.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acquia:commons:6.x-2.5`,
        fs:  `cpe:2.3:a:ding:ding_ezetop._top-up_any_phone:1.3.4:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:acquia:commons:6.x-2.6`,
        fs:  `cpe:2.3:a:oracle:virtualization_secure_global_desktop:5.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acquia:commons:6.x-2.7`,
        fs:  `cpe:2.3:a:webkitgtk:webkitgtk\&#43;:2.14.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acquia:commons:6.x-2.8`,
        fs:  `cpe:2.3:a:cisco:unified_callmanager:3.3\(5\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acquia:commons:6.x-2.9`,
        fs:  `cpe:2.3:a:x:libxrandr:1.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acquia:commons:6.x-2.10`,
        fs:  `cpe:2.3:a:apple:itunes:4.7.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acquia:commons:6.x-2.x:dev`,
        fs:  `cpe:2.3:a:oracle:marketing:12.2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acquia:commons:7.x-3.0`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.7:p321:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acquia:commons:7.x-3.1`,
        fs:  `cpe:2.3:a:sun:sdk:1.3.1_17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acquia:commons:7.x-3.2`,
        fs:  `cpe:2.3:a:ulli_horlacher:fex:20110813:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acquia:commons:7.x-3.x:dev`,
        fs:  `cpe:2.3:a:prestashop:prestashop:0.9.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acquia:mollom:6.x-2.7::~~~drupal~~`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.5.4-8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acquia:mollom:6.x-2.8::~~~drupal~~`,
        fs:  `cpe:2.3:a:percona:xtradb_cluster:5.6.15-25.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acquia:mollom:6.x-2.9::~~~drupal~~`,
        fs:  `cpe:2.3:o:cisco:nx-os:6.1\(2\)i3\(4\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acquia:mollom:6.x-2.10::~~~drupal~~`,
        fs:  `cpe:2.3:a:phpmyfaq:phpmyfaq:1.2.5a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acquia:mollom:6.x-2.11::~~~drupal~~`,
        fs:  `cpe:2.3:o:cisco:nx-os:6.0\(2\)u1\(2\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acquia:mollom:6.x-2.12::~~~drupal~~`,
        fs:  `cpe:2.3:h:hp:jg405a:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acquia:mollom:6.x-2.13::~~~drupal~~`,
        fs:  `cpe:2.3:o:redhat:certificate_server:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acquia:mollom:6.x-2.14::~~~drupal~~`,
        fs:  `cpe:2.3:a:google:chrome:5.0.375.23:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acra:acra_library:-:-:~~~android~~`,
        fs:  `cpe:2.3:a:hp:serviceguard:a.11.16.00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:backup_%26_recovery_agent:10.0.11639`,
        fs:  `cpe:2.3:a:emc:networker:7.6.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:backup_%26_recovery_agent:10.0.12703`,
        fs:  `cpe:2.3:h:juniper:secure_access_4500:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:backup_%26_recovery_agent:10.0.13545`,
        fs:  `cpe:2.3:h:softbank:android_smartphone:101f:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:backup_%26_recovery_agent:10.0.13762`,
        fs:  `cpe:2.3:a:adobe:shockwave_player:10.1.4.020:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:backup_%26_recovery_agent:11.0.17217`,
        fs:  `cpe:2.3:h:hp:jc748a:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:backup_%26_recovery_agent:11.0.17318`,
        fs:  `cpe:2.3:o:cisco:ios:12.0xs:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:backup_%26_recovery_agent_core:11.0.17217`,
        fs:  `cpe:2.3:a:oracle:endeca_server:7.3.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:backup_%26_recovery_agent_core:11.0.17318`,
        fs:  `cpe:2.3:a:itd-inc:bingo\!cms:1.0:a:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:backup_%26_recovery_bootable_media_builder:10.0.11639`,
        fs:  `cpe:2.3:a:cloudbees:jenkins:1.334:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:backup_%26_recovery_bootable_media_builder:10.0.13762`,
        fs:  `cpe:2.3:a:d.r.commander:libjpeg-turbo:1.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:backup_%26_recovery_bootable_media_builder:11.0.17217`,
        fs:  `cpe:2.3:a:phpmyadmin:phpmyadmin:3.3.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:backup_%26_recovery_management_console:10.0.11639`,
        fs:  `cpe:2.3:a:isc:dhcpd:3.0_pl1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:backup_%26_recovery_management_console:10.0.13545`,
        fs:  `cpe:2.3:a:adobe:digital_editions:4.5.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:backup_%26_recovery_management_console:10.0.13762`,
        fs:  `cpe:2.3:a:oracle:database_server:11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:backup_%26_recovery_management_console:11.0.17318`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.7:p313:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:backup_%26_recovery_management_server:11.0.17318`,
        fs:  `cpe:2.3:a:sixapart:movable_type:5.1:*:open_source:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:backup_%26_recovery_tray_monitor:10.0.11639`,
        fs:  `cpe:2.3:a:symantec:pcanywhere:11.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:backup_%26_recovery_tray_monitor:11.0.17318`,
        fs:  `cpe:2.3:a:wpeasycart:wp_easycart:1.2.8:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:backup_%26_recovery_universal_restore:10.0.13545`,
        fs:  `cpe:2.3:a:gnome:glib:2.9.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:backup_%26_recovery_universal_restore:10.0.13762`,
        fs:  `cpe:2.3:a:bestpractical:request_tracker:3.8.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:backup_%26_recovery_universal_restore:11.0.17217`,
        fs:  `cpe:2.3:a:zingiri:zingiri_web_shop:2.3.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:backup_%26_recovery_universal_restore:11.0.17318`,
        fs:  `cpe:2.3:a:tincan:phplist:1.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:backup_%26_recovery_upgrade_tool:10.0.11639`,
        fs:  `cpe:2.3:o:cisco:vpn_3005_concentrator:4.7.2.f:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:components_for_remote_installation:11.0.17318`,
        fs:  `cpe:2.3:o:cray:unicos:9.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:disk_director_server:10.0.2064`,
        fs:  `cpe:2.3:a:dotnetnuke:dotnetnuke:2.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:license_server:3.0.3470`,
        fs:  `cpe:2.3:a:bouncycastle:legion-of-the-bouncy-castle-c\#-crytography-api:1.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:license_server:10.0.13762`,
        fs:  `cpe:2.3:a:moinmo:moinmoin:1.5.0:beta2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:license_server:11.0.17318`,
        fs:  `cpe:2.3:a:google:chrome:6.0.472.35:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:snapapi:3.1.329`,
        fs:  `cpe:2.3:a:simon_mcvittie:telepathy_gabble:0.16.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:snapdeploy_management_agent:3.0.3470`,
        fs:  `cpe:2.3:a:bib2html_project:bib2html:0.6:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:true_image_agent:9.1.3854`,
        fs:  `cpe:2.3:h:cisco:nexus_9508:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:true_image_agent:9.7.8206`,
        fs:  `cpe:2.3:a:redhat:kdebase:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:true_image_echo_enterprise_server:9.5.8018`,
        fs:  `cpe:2.3:h:hp:laserjet_pro_100_color_mfp_m175:ce866a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:true_image_echo_enterprise_server:9.7.8206`,
        fs:  `cpe:2.3:a:ca:messaging:1.07.220.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:true_image_echo_enterprise_server:9.7.8398`,
        fs:  `cpe:2.3:o:juniper:junos:4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:true_image_echo_server:9.7.8398`,
        fs:  `cpe:2.3:a:horde:gollem:1.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:true_image_echo_workstation:9.7.8206`,
        fs:  `cpe:2.3:a:x.org:libx11:1.3.99.902:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:true_image_management_console:9.7.8206`,
        fs:  `cpe:2.3:a:microsoft:greetings:2000:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:true_image_server:9.1.3534`,
        fs:  `cpe:2.3:a:microsoft:biztalk_server:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acronis:universal_restore_for_true_image_echo_server:9.7.8206`,
        fs:  `cpe:2.3:a:dan_kogai:encode_module:1.01:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acrosoftware:cutepdf:3.4.1.3::~~professional~~~`,
        fs:  `cpe:2.3:a:opnet:sp_guru_transport_planner_documentation:15.0.0.8168:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acrosoftware:cutepdf:3.6.2.0::~~professional~~~`,
        fs:  `cpe:2.3:a:sensiolabs:symfony:2.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acrosoftware:cutepdf:3.6.4.0::~~professional~~~`,
        fs:  `cpe:2.3:a:jeff_sterup:plugin-organizer:2.5.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acrosoftware:cutepdf_writer:2.7`,
        fs:  `cpe:2.3:a:x.org:libx11:1.4.99.901:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acrosoftware:cutepdf_writer:2.8`,
        fs:  `cpe:2.3:a:oracle:database_server:9.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:acs:putty_connection_manager:0.7.1.136beta`,
        fs:  `cpe:2.3:a:custom_search_project:custom_search:6.x-1.9:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:act:compass:-`,
        fs:  `cpe:2.3:a:piwigo:piwigo:2.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:act:compass:3.2.1`,
        fs:  `cpe:2.3:a:boost:boost:1.42.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activcard:activcard_gold_for_cac:-`,
        fs:  `cpe:2.3:a:gnugk:gnu_gatekeeper:2.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activcard:activcard_gold_for_cac:2.24`,
        fs:  `cpe:2.3:a:rsyslog:rsyslog:6.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activcard:activcard_gold_for_cac:3.0`,
        fs:  `cpe:2.3:a:microsoft:access:2003:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:active_24_project:active_24:1.0.1::~~~android~~`,
        fs:  `cpe:2.3:a:gnu:wget:1.7.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activedev:active_cms:1.2`,
        fs:  `cpe:2.3:a:ati:catalyst_install_manager:3.0.816.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activehelper:activehelper_livehelp_live_chat:2.6.0::~~~wordpress~~`,
        fs:  `cpe:2.3:a:ca:network_discovery_gateway:2.5.1.117:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activehelper:activehelper_livehelp_live_chat:2.6.1::~~~wordpress~~`,
        fs:  `cpe:2.3:a:ray_stode:accountsservice:0.6.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activehelper:activehelper_livehelp_live_chat:2.6.2::~~~wordpress~~`,
        fs:  `cpe:2.3:a:cisco:videoscape_distribution_suite_for_internet_streaming:3.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activehelper:activehelper_livehelp_live_chat:2.6.5::~~~wordpress~~`,
        fs:  `cpe:2.3:a:samba:samba:2.2:a:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activehelper:activehelper_livehelp_live_chat:2.6.7::~~~wordpress~~`,
        fs:  `cpe:2.3:a:danielb:cool_aid:6.x-1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activehelper:activehelper_livehelp_live_chat:2.7.0::~~~wordpress~~`,
        fs:  `cpe:2.3:a:intermesh:group-office:3.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activehelper:activehelper_livehelp_live_chat:2.7.3::~~~wordpress~~`,
        fs:  `cpe:2.3:a:macromedia:coldfusion:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activehelper:activehelper_livehelp_live_chat:2.7.4::~~~wordpress~~`,
        fs:  `cpe:2.3:a:redhat:sysreport:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activehelper:activehelper_livehelp_live_chat:2.7.5::~~~wordpress~~`,
        fs:  `cpe:2.3:a:rocomotion:pplog_2:2.23:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activehelper:activehelper_livehelp_live_chat:2.9.0::~~~wordpress~~`,
        fs:  `cpe:2.3:o:juniper:junos:15.1x53:d34:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activehelper:activehelper_livehelp_live_chat:2.9.1::~~~wordpress~~`,
        fs:  `cpe:2.3:a:oracle:secure_backup:10.1.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activehelper:activehelper_livehelp_live_chat:2.9.2::~~~wordpress~~`,
        fs:  `cpe:2.3:a:kolja_schleich:leaguemanager:3.0.3:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:activehelper:activehelper_livehelp_live_chat:2.9.5::~~~wordpress~~`,
        fs:  `cpe:2.3:a:getid3:getid3:1.9.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activehelper:activehelper_livehelp_live_chat:3.0.0::~~~wordpress~~`,
        fs:  `cpe:2.3:a:opera:opera_browser:9.63:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activehelper:activehelper_livehelp_live_chat:3.1.0::~~~wordpress~~`,
        fs:  `cpe:2.3:a:redhat:jboss_enterprise_portal_platform:5.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activehelper:activehelper_livehelp_live_chat:3.1.5::~~~wordpress~~`,
        fs:  `cpe:2.3:a:primetek:primefaces:3.5.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activehelper:activehelper_livehelp_live_chat:3.2.0::~~~wordpress~~`,
        fs:  `cpe:2.3:a:phpmyadmin:phpmyadmin:4.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activehelper:activehelper_livehelp_live_chat:3.4.0::~~~wordpress~~`,
        fs:  `cpe:2.3:a:lsi:power_console_plus_package:5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activehelper:activehelper_livehelp_live_chat:3.4.5::~~~wordpress~~`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.27.22:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activehelper:activehelper_livehelp_live_chat:3.5.0::~~~wordpress~~`,
        fs:  `cpe:2.3:a:horde:groupware:5.1.0:*:*:*:webmail:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:-`,
        fs:  `cpe:2.3:a:icu-project:international_components_for_unicode:3.4.1:*:*:*:*:java:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.6.1`,
        fs:  `cpe:2.3:a:reviewboard:review_board:1.6.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.6.1.629`,
        fs:  `cpe:2.3:a:ibm:tivoli_provisioning_manager_express:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.6.1.630`,
        fs:  `cpe:2.3:a:samba:samba:2.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.6.1.635`,
        fs:  `cpe:2.3:a:bigtreecms:bigtree_cms:4.2.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.6.2`,
        fs:  `cpe:2.3:a:maximus:schoolmax:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.6.3`,
        fs:  `cpe:2.3:a:google:authenticator:0.91:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.6.633`,
        fs:  `cpe:2.3:a:wireshark:wireshark:1.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.6.635`,
        fs:  `cpe:2.3:o:microsoft:windows_nt:4.0:sp6a:workstation:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.6.638`,
        fs:  `cpe:2.3:a:apache:mod_perl:1.99:17:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.7.1`,
        fs:  `cpe:2.3:a:ibm:sterling_selling_and_fulfillment_foundation:9.1.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.7.2`,
        fs:  `cpe:2.3:a:cisco:unified_computing_system:2.0_4a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.7.3`,
        fs:  `cpe:2.3:h:dell:latitude_e6410_atg:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.8`,
        fs:  `cpe:2.3:a:openvas:openvas_scanner:3.2:beta1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.8.1`,
        fs:  `cpe:2.3:a:impresscms:impresscms:1.1:rc3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.8.3`,
        fs:  `cpe:2.3:a:mantisbt:mantisbt:2.0.0:beta1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.8.7`,
        fs:  `cpe:2.3:a:oscommerce:oscommerce:2.2ms1j-r7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.8.8.815`,
        fs:  `cpe:2.3:a:swi-prolog:swi-prolog:6.3.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.8.8.817`,
        fs:  `cpe:2.3:a:dameware:mini_remote_control:7.5.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.8.8.820`,
        fs:  `cpe:2.3:a:xymon:xymon:4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.8.807`,
        fs:  `cpe:2.3:a:splunk:splunk:2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.8.809`,
        fs:  `cpe:2.3:a:ca:etrust_security_command_center:8:sp1:cr2:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.8.810`,
        fs:  `cpe:2.3:a:openldap:openldap:2.4.27:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.8.819`,
        fs:  `cpe:2.3:a:cloudfoundry:cf-release:247:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.8.820`,
        fs:  `cpe:2.3:a:oracle:peoplesoft_enterprise:8.22.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.8.824`,
        fs:  `cpe:2.3:a:cisco:secure_access_control_server:3.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.8.827`,
        fs:  `cpe:2.3:a:phpnuke:php-nuke:7.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.8.828`,
        fs:  `cpe:2.3:a:controlmicrosystems:clearscada_2007:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.10.1002`,
        fs:  `cpe:2.3:a:novell:web_server:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.10.1004`,
        fs:  `cpe:2.3:a:armin_burgmeier:net6:1.3.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.10.1005`,
        fs:  `cpe:2.3:a:cisco:firepower_management_center:5.3.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.10.1006`,
        fs:  `cpe:2.3:a:rubyonrails:ruby_on_rails:3.2.0:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.10.1007`,
        fs:  `cpe:2.3:a:wireshark:wireshark:2.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.10.1008::~~~~x64~`,
        fs:  `cpe:2.3:a:boatmob:boat_browser_mini:2.6.1:-:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.12.1201::~~~~x64~`,
        fs:  `cpe:2.3:a:rocomotion:p_up_board:1.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.12.1202`,
        fs:  `cpe:2.3:a:x:libxext:1.0.99.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.12.1203`,
        fs:  `cpe:2.3:a:digium:asterisk:10.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.12.1203::~~~~x64~`,
        fs:  `cpe:2.3:h:iss:proventia:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.12.1204`,
        fs:  `cpe:2.3:a:bestpractical:rt:3.1.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.14.1402`,
        fs:  `cpe:2.3:a:microsoft:visio:2000:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl:5.14.1402::~~~~x64~`,
        fs:  `cpe:2.3:a:sprockets_project:sprockets:2.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activeperl_pdk:9.0.1`,
        fs:  `cpe:2.3:o:linux:linux_kernel:4.3.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activepython:-`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.10.59:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activepython:2.1`,
        fs:  `cpe:2.3:a:primetek:primefaces:6.1.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activepython:2.5.2.2`,
        fs:  `cpe:2.3:a:x:libxext:1.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activepython:2.5.4.4`,
        fs:  `cpe:2.3:a:google:chrome:13.0.782.44:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:activepython:2.6.5.12`,
        fs:  `cpe:2.3:a:dan_kogai:encode_module:1.34:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:komodo_edit:5.2.4`,
        fs:  `cpe:2.3:a:x.org:libx11:1.4.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:komodo_edit:6.0.3`,
        fs:  `cpe:2.3:a:jenkins-ci:monitoring_plugin:1.48.0:*:*:*:*:cloudbees_jenkins:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:komodo_edit:6.1.0`,
        fs:  `cpe:2.3:a:google:chrome:6.0.408.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activestate:komodo_edit:6.1.1`,
        fs:  `cpe:2.3:a:citrix:offline_plug-in:6.0.0.1304:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:activetemplates:template:1.5`,
        fs:  `cpe:2.3:a:nginx:nginx:1.3.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activcard_gold_for_cac:2.200.66.0`,
        fs:  `cpe:2.3:a:ibm:netezza_performance_portal:2.0.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activcard_gold_for_cac_feature_pack:3.01.1076.1`,
        fs:  `cpe:2.3:o:vmware:esxi:6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activcard_gold_for_cac_feature_pack:3.01.2087.2`,
        fs:  `cpe:2.3:a:bioinformatics:htmlawed:1.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activcard_usb_reader:2.0.3`,
        fs:  `cpe:2.3:a:boehm-demers-weiser:garbage_collector:6.1:alpha4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:-`,
        fs:  `cpe:2.3:a:ssh:tectia_server:5.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.0`,
        fs:  `cpe:2.3:a:spamtitan:spamtitan:6.00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.1.3.99`,
        fs:  `cpe:2.3:o:cisco:catos:7.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.1.3.100`,
        fs:  `cpe:2.3:a:teampass:teampass:2.1.27.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.1.3.100::~~~~x64~`,
        fs:  `cpe:2.3:a:mysql:maxdb:7.5.00.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.1.3.108`,
        fs:  `cpe:2.3:h:hp:3com_router:3013:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.1.3.113`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.30:rc3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.1.3.114`,
        fs:  `cpe:2.3:a:mediawiki:mediawiki:1.21.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.1.3.123`,
        fs:  `cpe:2.3:a:roderick_braun:ya_googlesearch:0.3.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.1.3.130`,
        fs:  `cpe:2.3:o:honeywell:excel_web_xl_1000c1000_600_i\/o_uukl:2.04.00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.1.3.130::~~~~x64~`,
        fs:  `cpe:2.3:a:ibm:access_ibm_message_center:2.103:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.1.3.132`,
        fs:  `cpe:2.3:a:s9y:serendipity:1.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.01.00034`,
        fs:  `cpe:2.3:o:sun:sunos:5.10:-:*:*:*:*:amd64:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.01.00046`,
        fs:  `cpe:2.3:o:cisco:nx-os:7.2\(0\)n1\(1\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.01.00051`,
        fs:  `cpe:2.3:a:cambridge_enterprise:jbig-kit:1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.1.100`,
        fs:  `cpe:2.3:a:cisco:optical_networking_systems_software:7.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.1.120`,
        fs:  `cpe:2.3:o:yahama:rtx1100:8.03.24:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.1.124`,
        fs:  `cpe:2.3:a:labtam-inc:proftp:2.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.1.125`,
        fs:  `cpe:2.3:a:libpng:libpng:1.5.27:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.1.133`,
        fs:  `cpe:2.3:a:web2ldap:web2ldap:1.1.47:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.1.137`,
        fs:  `cpe:2.3:a:webwiz:web_wiz_forums:9.0:beta2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.1.139`,
        fs:  `cpe:2.3:a:ibm:lotus_connections:1.0.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.1.148`,
        fs:  `cpe:2.3:a:modwsgi:mod_wsgi:1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.1.155`,
        fs:  `cpe:2.3:a:commerceguys:commerce:7.x-1.0:alpha1:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.1.157`,
        fs:  `cpe:2.3:a:microsoft:microsoft.aspnetcore.mvc.cors:1.0.3:*:*:*:*:asp.net:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2`,
        fs:  `cpe:2.3:a:iqnect:dubstep_hero:1.9:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2::~~~~x64~`,
        fs:  `cpe:2.3:a:moodle:moodle:1.9.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.54`,
        fs:  `cpe:2.3:o:sun:solaris:5.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.54::~~~~x64~`,
        fs:  `cpe:2.3:a:mozilla:firefox:20.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.69`,
        fs:  `cpe:2.3:a:hp:insight_diagnostics:7.7.0_rev_b:offline:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.69::~~~~x64~`,
        fs:  `cpe:2.3:o:cisco:pix_firewall:3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.71`,
        fs:  `cpe:2.3:a:intermesh:group-office:3.2.48:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.71::~~~~x64~`,
        fs:  `cpe:2.3:a:debian:dpkg:1.18.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.72`,
        fs:  `cpe:2.3:a:codiad:codiad:1.8.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.72::~~~~x64~`,
        fs:  `cpe:2.3:a:avast:premier:11.1.2241:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.73`,
        fs:  `cpe:2.3:o:huawei:s5300_firmware:v200r007c00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.75`,
        fs:  `cpe:2.3:a:formsplayer:formsplayer:1.5.4:1032:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.75::~~~~x64~`,
        fs:  `cpe:2.3:a:juralsulek:realurlmanagement:0.3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.78`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(33\)sb3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.80`,
        fs:  `cpe:2.3:h:hp:3com_switch:4210_pwr:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.80::~~~~x64~`,
        fs:  `cpe:2.3:h:axis:700_network_document_server:1.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.81`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.4.3:*:*:*:*:*:x86:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.81::~~~~x64~`,
        fs:  `cpe:2.3:a:ibm:rational_appscan:8.0.0.3:*:enterprise:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.85`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.7:p419:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.85::~~~~x64~`,
        fs:  `cpe:2.3:a:wireshark:wireshark:1.6.6:*:*:*:*:*:itanium:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.86`,
        fs:  `cpe:2.3:a:alienwp:hatch:7.x-1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.86::~~~~x64~`,
        fs:  `cpe:2.3:a:ponsoftware:explzh:4.19:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.95`,
        fs:  `cpe:2.3:a:ibm:rational_policy_tester:8.5.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.95::~~~~x64~`,
        fs:  `cpe:2.3:a:polycom:cma_system_software:5.2.1j:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.101::~~~~x64~`,
        fs:  `cpe:2.3:a:python:python:2.7.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.107`,
        fs:  `cpe:2.3:o:yahama:rt200i:4.00.02:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.107::~~~~x64~`,
        fs:  `cpe:2.3:a:mcafee:email_gateway:7.6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.111`,
        fs:  `cpe:2.3:a:drupalisme:tinybox:7.x-2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.113`,
        fs:  `cpe:2.3:a:moinejf:abcm2ps:5.9.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.113::~~~~x64~`,
        fs:  `cpe:2.3:a:microsoft:project_server:2010:sp2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.115::~~~~x64~`,
        fs:  `cpe:2.3:o:apple:mac_os_x:10.8.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.116`,
        fs:  `cpe:2.3:a:ibm:inotes:8.5.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.116::~~~~x64~`,
        fs:  `cpe:2.3:a:squid-cache:squid:3.2.0.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.118::~~~~x64~`,
        fs:  `cpe:2.3:a:videowhisper:videowhisper_live_streaming_integration:4.27:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.119`,
        fs:  `cpe:2.3:a:cisco:ips_sensor_software:5.1\(7\)e1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.119::~~~~x64~`,
        fs:  `cpe:2.3:a:roderich_schupp:par-packer_module:0.954:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.120::~~~~x64~`,
        fs:  `cpe:2.3:a:mandrakesoft:mandrake_multi_network_firewall:8.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.122`,
        fs:  `cpe:2.3:a:claws-mail:claws-mail:3.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.122::~~~~x64~`,
        fs:  `cpe:2.3:a:mortbay:jetty:2.4.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.124::~~~~x64~`,
        fs:  `cpe:2.3:o:yahama:rtx1000:7.01.49:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.126::~~~~x64~`,
        fs:  `cpe:2.3:a:solarwinds:ip_address_manager_web_interface:3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.128`,
        fs:  `cpe:2.3:a:google:chrome:23.0.1271.51:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.128::~~~~x64~`,
        fs:  `cpe:2.3:a:ibm:spss_modeler:15.0.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.133`,
        fs:  `cpe:2.3:a:htc:evo_4g_software:3.70.651.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.2.0.133::~~~~x64~`,
        fs:  `cpe:2.3:a:cisco:wireless_control_system_software:4.1.91.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.11.00039::~~~~x64~`,
        fs:  `cpe:2.3:a:php:php:5.4.19:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.11.00050::~~~~x64~`,
        fs:  `cpe:2.3:a:rubyonrails:ruby_on_rails:1.2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:6.11.155::~~~~x64~`,
        fs:  `cpe:2.3:a:strongswan:strongswan:2.4.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac:7.0::~~~~x64~`,
        fs:  `cpe:2.3:a:microsoft:exchange_srv:2000:sp2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac_afr:6.1.100`,
        fs:  `cpe:2.3:a:avaya:enterprise_manager:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac_afr:6.1.125`,
        fs:  `cpe:2.3:a:mantisbt:mantisbt:1.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:activclient_cac_afr:6.1.155`,
        fs:  `cpe:2.3:a:cloudbees:jenkins:1.324:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:actividentity_device_installer:2.00.00012`,
        fs:  `cpe:2.3:a:paperton:allt_om_brollop:1.53:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:actividentity_device_installer:2.2`,
        fs:  `cpe:2.3:a:cloudera:cloudera_manager:3.7.4:*:free:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:actividentity_device_installer:2.2::~~~~x64~`,
        fs:  `cpe:2.3:a:git_for_windows_project:git_for_windows:1.8.3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:actividentity_device_installer:2.3::~~~~x64~`,
        fs:  `cpe:2.3:a:mathjs:math.js:3.16.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:actividentity_device_installer:2.10.00006`,
        fs:  `cpe:2.3:a:conky:conky:1.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:corestreet_desktop_validation_client:2.1.3.3`,
        fs:  `cpe:2.3:a:bea:weblogic_server:5.1:sp3:express:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:corestreet_desktop_validation_client:2.1.6.20`,
        fs:  `cpe:2.3:a:gnu:cfengine:2.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:corestreet_desktop_validation_client:3.1.4.1`,
        fs:  `cpe:2.3:a:moinejf:abcm2ps:6.3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:corestreet_desktop_validation_client:3.1.4.1::~~~~x64~`,
        fs:  `cpe:2.3:a:cisco:unified_computing_system:1.4\(7b\)1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:corestreet_server_validation_extension_dod_settings:1.5.0.0`,
        fs:  `cpe:2.3:a:ibm:websphere_application_server:6.0.2.31:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:corestreet_server_validation_extension_for_microsoft_domain_controller:3.1.4.1::~~~~x64~`,
        fs:  `cpe:2.3:a:cisco:telepresence_e20_software:tenc4.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:corestreet_server_validation_extension_for_microsoft_internet_information_server:3.1.4.1`,
        fs:  `cpe:2.3:a:cisco:unified_service_monitor:2.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:corestreet_validation_authority:5.1.5`,
        fs:  `cpe:2.3:a:ibm:soliddb:6.5.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actividentity:desktop_validation_client:3.1.4.1`,
        fs:  `cpe:2.3:a:udesign_project:udesign:2.4.19:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:activision:call_of_duty_elite:2.0.1:-:~~~iphone_os~~`,
        fs:  `cpe:2.3:a:live555:streaming_media:2013.03.31:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actorskey:actors_key:1.6.24.477::~~~android~~`,
        fs:  `cpe:2.3:a:wp-plugins:wp-print:2.05:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:actualtests:cisco_640-802_demo_in_exam_engine_format:1.0.0`,
        fs:  `cpe:2.3:a:apache:cloudstack:2.1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actualtests:comptia_220-701_in_exam_engine_format:1.0.0`,
        fs:  `cpe:2.3:a:redhat:certificate_server:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actualtests:comptia_220-702_in_exam_engine_format:1.0.0`,
        fs:  `cpe:2.3:a:gnome:gnome_display_manager:3.5.90:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actualtests:comptia_n10-004_in_exam_engine_format:1.0.0`,
        fs:  `cpe:2.3:a:expressjs:express:4.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actualtests:comptia_sy0-201_in_exam_engine_format:1.0.0`,
        fs:  `cpe:2.3:a:ibm:db2:9.1.0.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actualtests:exin_ex0-1_in_exam_engine_format:1.0.0`,
        fs:  `cpe:2.3:a:dotpdn:paint.net:3.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actualtests:questions_and_answers_for_eccouncil_312-50v7:4.2.13135`,
        fs:  `cpe:2.3:a:cisco:jabber:9.1\(.1\):-:-:*:-:windows:*:*`,
    },
     
    {
        uri: `cpe:/a:actuate:iserver:7.2.6`,
        fs:  `cpe:2.3:a:thalesgroup:datacryptor_element_manager:4.4.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actuate:iserver:7.2.19`,
        fs:  `cpe:2.3:a:caucho:resin:4.0.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actuate:iserver:8.0.5`,
        fs:  `cpe:2.3:a:feedweb:feedweb:2.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:actuate:iserver:9.2.0`,
        fs:  `cpe:2.3:o:juniper:netscreen_screenos:3.0.3r4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ad-manager_project:ad-manager:1.1.2::~~~wordpress~~`,
        fs:  `cpe:2.3:a:happyworm:jplayer:2.3.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ad-minister_project:ad-minister:0.4.1`,
        fs:  `cpe:2.3:a:apache:storm:0.10.0:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ad-minister_project:ad-minister:0.4.2`,
        fs:  `cpe:2.3:a:symantec:endpoint_protection:11.0:ru6a:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ad-minister_project:ad-minister:0.4.3`,
        fs:  `cpe:2.3:a:yahoo:yui:3.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ad-minister_project:ad-minister:0.4.4`,
        fs:  `cpe:2.3:a:isc:bind:9.7.0:p1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ad-minister_project:ad-minister:0.5`,
        fs:  `cpe:2.3:a:connections_project:connections:0.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ad-minister_project:ad-minister:0.5.2`,
        fs:  `cpe:2.3:a:oracle:application_server_portal:9.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ad-minister_project:ad-minister:0.6`,
        fs:  `cpe:2.3:a:debian:dpkg:1.6.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adam_zaninovich:sounder:0.0.1`,
        fs:  `cpe:2.3:a:juniper:network_and_security_manager_software:2012.2:r2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adam_zaninovich:sounder:0.0.2`,
        fs:  `cpe:2.3:o:buffalotech:whr-300hp2_firmware:1.70:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adam_zaninovich:sounder:1.0.0`,
        fs:  `cpe:2.3:a:gitlab:gitlab-shell:1.9.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adam_zaninovich:sounder:1.0.1`,
        fs:  `cpe:2.3:a:ibm:websphere_application_server:7.0.0.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.0.0`,
        fs:  `cpe:2.3:a:artifex:afpl_ghostscript:6.50:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.1.2`,
        fs:  `cpe:2.3:a:bit51:better-wp-security:-:alpha10:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.1.3`,
        fs:  `cpe:2.3:a:debian:dpkg:1.16.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.1.6`,
        fs:  `cpe:2.3:a:joomla:joomla\!:2.5.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.1.7`,
        fs:  `cpe:2.3:a:cisco:adaptive_security_appliance_software:8.0.4.31:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.1.8`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.4.34.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.1.9`,
        fs:  `cpe:2.3:a:redhat:libvirt:1.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.1.10`,
        fs:  `cpe:2.3:a:eucalyptus:eucalyptus:2.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.1.11`,
        fs:  `cpe:2.3:a:qualys:qualyssuite:8.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.2.0`,
        fs:  `cpe:2.3:a:emc:networker:8.2.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.2.1`,
        fs:  `cpe:2.3:a:7-zip:7-zip:3.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.3.0`,
        fs:  `cpe:2.3:a:impresscms:impresscms:1.1:beta1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.3.1`,
        fs:  `cpe:2.3:a:zikula:zikula_application_framework:1.3.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.3.2`,
        fs:  `cpe:2.3:o:cisco:ios:15.4\(1\)t1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.3.3`,
        fs:  `cpe:2.3:a:ircd-hybrid:ircd-hybrid:8.0.0:beta1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.3.4`,
        fs:  `cpe:2.3:a:andy_armstrong:cgi.pm:2.67:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.3.5`,
        fs:  `cpe:2.3:a:tm_software:tempo6.3.0:jira42:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.3.6`,
        fs:  `cpe:2.3:a:cisco:adaptive_security_appliance_software:8.4\(3\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.3.7`,
        fs:  `cpe:2.3:a:apache:subversion:0.24.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.3.8`,
        fs:  `cpe:2.3:a:botcha_spam_prevention_project:botcha:7.x-4.x:dev:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.3.9`,
        fs:  `cpe:2.3:a:zingiri:zingiri_web_shop:2.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.3.10`,
        fs:  `cpe:2.3:a:libguestfs:libguestfs:1.16.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.3.11`,
        fs:  `cpe:2.3:h:d-link:dcs-5029l:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.3.12`,
        fs:  `cpe:2.3:a:sun:one_directory_server:5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.3.13`,
        fs:  `cpe:2.3:a:crumbs_project:crumbs:7.x-2.1:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.4.2`,
        fs:  `cpe:2.3:a:tecnick:tcexam:11.0.004:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.4.3`,
        fs:  `cpe:2.3:a:logitech:vid:1.1.1015:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.4.4`,
        fs:  `cpe:2.3:a:ponsoftware:explzh:3.71:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.4.5`,
        fs:  `cpe:2.3:a:mahara:mahara:16.10.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.4.6`,
        fs:  `cpe:2.3:a:katello:katello_installer:0.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.4.7`,
        fs:  `cpe:2.3:a:user_relationships_project:user_relationships:6.x-1.0:rc3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.4.8`,
        fs:  `cpe:2.3:a:mawashimono:nikki:4.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.4.9`,
        fs:  `cpe:2.3:a:emc:documentum_xcp:2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.4.10`,
        fs:  `cpe:2.3:a:zabbix:zabbix:2.2.1:-:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.4.11`,
        fs:  `cpe:2.3:a:libpng:libpng:1.0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.4.12`,
        fs:  `cpe:2.3:a:devincentiis:gazie:4.0.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.4.13`,
        fs:  `cpe:2.3:a:nomachine:nx_web_companion:3.5.0-2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.4.14`,
        fs:  `cpe:2.3:a:gnu:fingerd:1.37:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.4.15`,
        fs:  `cpe:2.3:a:dovecot:dovecot:2.0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.4.16`,
        fs:  `cpe:2.3:a:bea:aqualogic_enterprise_security:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.4.17`,
        fs:  `cpe:2.3:a:live555:streaming_media:2014.01.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.5.0`,
        fs:  `cpe:2.3:a:igor_sysoev:nginx:0.1.19:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.5.1`,
        fs:  `cpe:2.3:h:cisco:cache_engine_570:3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.5.2`,
        fs:  `cpe:2.3:a:ibus_project:ibus:1.5.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.5.3`,
        fs:  `cpe:2.3:a:download_manager_project:download_manager:1.2:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.5.4`,
        fs:  `cpe:2.3:a:magzter:penumbra_emag:3:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.5.5`,
        fs:  `cpe:2.3:h:ibm:netezza:6.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.5.6`,
        fs:  `cpe:2.3:a:vmware:workstation:5.0.0_build_13124:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.5.7`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.7:p55:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.5.8`,
        fs:  `cpe:2.3:a:dell:quest_one_password_manager:5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.5.9`,
        fs:  `cpe:2.3:a:apple:safari:4.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.5.10`,
        fs:  `cpe:2.3:a:keycloak:keycloak-nodejs-auth-utils:2.5.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.5.11`,
        fs:  `cpe:2.3:a:tejimaya:openpne:3.4.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.5.12`,
        fs:  `cpe:2.3:a:gisle_aas:libwww-perl:5.53_90:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:2.5.13`,
        fs:  `cpe:2.3:a:memcache_project:memcache:6.x-1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:3.0.0`,
        fs:  `cpe:2.3:a:botcha_spam_prevention_project:botcha:7.x-3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:3.0.1`,
        fs:  `cpe:2.3:a:sensiolabs:symfony:2.2.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:3.0.2`,
        fs:  `cpe:2.3:a:oracle:mysql_connector\/j:3.1.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:3.0.3`,
        fs:  `cpe:2.3:a:barracudadrive:barracudadrive:4.6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:3.0.4`,
        fs:  `cpe:2.3:a:google:v8:2.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:3.0.5`,
        fs:  `cpe:2.3:a:exim:exim:4.44:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:3.0.6`,
        fs:  `cpe:2.3:a:google:chrome:7.0.536.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:4.0.0`,
        fs:  `cpe:2.3:a:3s-software:codesys_runtime_system:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:4.0.2`,
        fs:  `cpe:2.3:a:sambar:sambar_server:5.0:beta6:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:4.1.0`,
        fs:  `cpe:2.3:a:hp:alm-platform_loader:11.0.5154.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:4.1.1`,
        fs:  `cpe:2.3:a:danny_morris:lazy_seo:1.2.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:4.1.2`,
        fs:  `cpe:2.3:a:kent-web:joyful_note:2.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:4.1.3`,
        fs:  `cpe:2.3:a:plume-cms:plume_cms:1.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:4.1.5.1`,
        fs:  `cpe:2.3:a:tigris:tortoisesvn:0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:4.1.6`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.424.1:*:*:*:lts:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:4.1.7`,
        fs:  `cpe:2.3:a:nginx:nginx:1.9.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:4.2.2`,
        fs:  `cpe:2.3:a:ibm:db2:10.1.0.5:*:*:*:enterprise:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:4.2.3`,
        fs:  `cpe:2.3:a:infotecs:vipnet_personal_firewall:3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:4.2.3.1`,
        fs:  `cpe:2.3:a:lxml:lxml:3.3.0:-:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:4.2.4`,
        fs:  `cpe:2.3:a:rackspace:rackspace:2.1.5:-:*:*:*:iphone_os:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:4.2.4.1`,
        fs:  `cpe:2.3:a:autodesk:vred:2014:sr1_sp8:*:*:professional:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:4.2.5`,
        fs:  `cpe:2.3:a:simplemachines:simple_machines_forum:1.0:beta4.1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:4.2.6`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.7:p324:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:4.2.6.1`,
        fs:  `cpe:2.3:a:bea:weblogic_server:9.2:mp1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:4.2.7`,
        fs:  `cpe:2.3:a:google:chrome:34.0.1847.113:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:4.2.8`,
        fs:  `cpe:2.3:a:sun:jre:6.0.220:update22:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:4.2.9`,
        fs:  `cpe:2.3:a:google:chrome:31.0.1650.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivecomputing:torque_resource_manager:5.0.0-1_43d8f09a`,
        fs:  `cpe:2.3:a:ibm:websphere_extreme_scale_client:8.6.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivelearningsystems:crammaster:1.0`,
        fs:  `cpe:2.3:a:openswan:openswan:2.4.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adaptivelearningsystems:sun_cx-310-200_solaris_10_administrator_%28scsa%29_part_1:1.0`,
        fs:  `cpe:2.3:a:cherokee-project:cherokee:1.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adastra:trace_mode_data_center:-`,
        fs:  `cpe:2.3:a:torproject:tor:0.0.2:pre18:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adastra:trace_mode_data_center:6.0`,
        fs:  `cpe:2.3:a:google:chrome:13.0.753.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adastra:trace_mode_data_center:6.04`,
        fs:  `cpe:2.3:a:mahara:mahara:1.1.0:alpha3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adastra:trace_mode_data_center:6.05`,
        fs:  `cpe:2.3:a:puppetlabs:puppet:3.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adastra:trace_mode_data_center:6.06`,
        fs:  `cpe:2.3:a:google:chrome:17.0.923.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adastra:trace_mode_data_center:6.07`,
        fs:  `cpe:2.3:a:buffalotech:whr-g54s_firmware:1.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adastra:trace_mode_data_center:6.07.7`,
        fs:  `cpe:2.3:a:synology:download_station:3.5-2706:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adastra:trace_mode_data_center:6.08`,
        fs:  `cpe:2.3:a:cakephp:cakephp:2.3.0:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:adb:p.dga4001n:-`,
        fs:  `cpe:2.3:a:nancy_wichmann:realname:6.x-1.1:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:adb:p.dga4001n_firmware:pdg_tef_sp_4.06l.6`,
        fs:  `cpe:2.3:a:ge:intelligent_platforms_proficy_hmi\/scada_ifix:5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.0.0:::de`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.27.50:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.0.0:::en`,
        fs:  `cpe:2.3:a:jruby:jruby:1.1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.0.1:::de`,
        fs:  `cpe:2.3:a:gitlab:gitlab:8.6.6:*:*:*:enterprise:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.0.1:::en`,
        fs:  `cpe:2.3:a:ibm:informix_dynamic_server:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.0.2:::de`,
        fs:  `cpe:2.3:a:netiq:endpoint:5.1.8149:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.0.2:::en`,
        fs:  `cpe:2.3:h:huawei:e200x3:v200r003c00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.1.0`,
        fs:  `cpe:2.3:a:git-scm:git:1.7.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.1.1`,
        fs:  `cpe:2.3:a:debian:apt:0.7.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.1.2`,
        fs:  `cpe:2.3:a:oracle:industry_applications:4.5:sp3c:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.2.0`,
        fs:  `cpe:2.3:a:google:chrome:11.0.686.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.2.1`,
        fs:  `cpe:2.3:a:bayashi:dopvstar\*:0063:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.2.2`,
        fs:  `cpe:2.3:a:igor_sysoev:nginx:0.3.45:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.2.3`,
        fs:  `cpe:2.3:a:postgresql:postgresql:9.1.22:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.2.4`,
        fs:  `cpe:2.3:a:gnome:gnome-keyring:3.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.3.0`,
        fs:  `cpe:2.3:a:gpeasy:gpeasy_cms:1.5:rc4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.3.1`,
        fs:  `cpe:2.3:a:videolan:vlc_media_player:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.3.2`,
        fs:  `cpe:2.3:o:opensuse_project:suse_linux_enterprise_server:11.0:sp3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.3.3`,
        fs:  `cpe:2.3:a:gnome:glib:1.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.4.0`,
        fs:  `cpe:2.3:a:samba:samba:4.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.4.1`,
        fs:  `cpe:2.3:a:google:chrome:12.0.742.100:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.4.2`,
        fs:  `cpe:2.3:a:f5:big-ip_policy_enforcement_manager:11.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.4.3`,
        fs:  `cpe:2.3:a:topdesk:topdesk:6.06.002:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.4.4`,
        fs:  `cpe:2.3:a:ultimus:ultimus_business_process_management_suite:7.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.4.5`,
        fs:  `cpe:2.3:a:gitlab:gitlab:1.0.0:*:*:*:enterprise:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.4.6`,
        fs:  `cpe:2.3:a:pmwiki:pmwiki:2.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.4.7`,
        fs:  `cpe:2.3:a:powerarchiver:powerarchiver:14.02:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.4.8`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.2:rc5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.4.9`,
        fs:  `cpe:2.3:a:teradata:.net_data_provider:13.1.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.4.10`,
        fs:  `cpe:2.3:a:atlassian:bamboo:4.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.4.11`,
        fs:  `cpe:2.3:a:contao:contao_cms:2.5.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.5.0`,
        fs:  `cpe:2.3:a:hardened-php:suhosin:0.9.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.5.1`,
        fs:  `cpe:2.3:a:cisco:pix_firewall_manager:4.3\(2\)g:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.5.2`,
        fs:  `cpe:2.3:a:oracle:mysql_connector\/j:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.5.3`,
        fs:  `cpe:2.3:a:xymon:xymon:4.3.22:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.5.4`,
        fs:  `cpe:2.3:a:webgateinc:webeyeaudio:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.5.5`,
        fs:  `cpe:2.3:a:simon_brown:pebble:2.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:1.5.6`,
        fs:  `cpe:2.3:a:apache:couchdb:1.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:2.0.0`,
        fs:  `cpe:2.3:a:hp:smart_array_sas\/sata_event_notification_service:6.22.0.64:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:2.0.1`,
        fs:  `cpe:2.3:a:nick_copeland:bristol:0.30.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:2.0.2`,
        fs:  `cpe:2.3:a:gomlab:gom_player:2.1.21.4846:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:2.0.3`,
        fs:  `cpe:2.3:a:lansweeper:lansweeper:5.0.0.52:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:2.0.4`,
        fs:  `cpe:2.3:o:hp:procurve_switch_software:r.11.35:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:2.0.5`,
        fs:  `cpe:2.3:a:mediawiki:mediawiki:1.12.0:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:2.1.0`,
        fs:  `cpe:2.3:a:genetechsolutions:pie-register:1.2.91:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:2.1.1`,
        fs:  `cpe:2.3:a:dotcms:dotcms:2.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:2.1.2`,
        fs:  `cpe:2.3:a:ibm:infosphere_replication_server:9.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:2.1.3`,
        fs:  `cpe:2.3:a:google:chrome:6.0.404.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:2.2.0`,
        fs:  `cpe:2.3:a:gitlab:gitlab:8.8.8:*:*:*:enterprise:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:2.2.1`,
        fs:  `cpe:2.3:a:autodesk:autocad_electrical:2006:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:2.2.2`,
        fs:  `cpe:2.3:o:cisco:ios_xe:3.17s:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:2.2.3`,
        fs:  `cpe:2.3:a:apache:struts:2.3.28.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:2.2.4`,
        fs:  `cpe:2.3:a:ocsinventory-ng:ocsinventory_ng:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:2.2.5`,
        fs:  `cpe:2.3:a:iij:seil\/turbo_firmware:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:2.2.6`,
        fs:  `cpe:2.3:a:libpng:libpng:1.5.20:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:2.2.7`,
        fs:  `cpe:2.3:a:ovirt:ovirt:3.5.0:rc1.1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:2.3.0`,
        fs:  `cpe:2.3:o:cisco:unified_computing_system_infrastructure_and_unified_computing_system_software:1.4\(4g\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:2.3.1`,
        fs:  `cpe:2.3:a:3sgroup:t2css:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:2.3.2`,
        fs:  `cpe:2.3:a:digia:qt:4.6.0:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:2.3.3`,
        fs:  `cpe:2.3:a:ca:siteminder:12.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:2.3.4`,
        fs:  `cpe:2.3:h:apple:ipod_touch:1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:2.3.5`,
        fs:  `cpe:2.3:a:dhcpcd_project:dhcpcd:5.5.1:test1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:2.4.0`,
        fs:  `cpe:2.3:a:gitlab:gitlab:9.1.0:*:*:*:enterprise:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:2.4.1`,
        fs:  `cpe:2.3:a:cipherdyne:fwsnort:1.6.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:2.4.2`,
        fs:  `cpe:2.3:a:mozilla:seamonkey:2.0.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:2.4.3`,
        fs:  `cpe:2.3:a:adbnewssender_project:adbnewssender:2.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:3.0.0`,
        fs:  `cpe:2.3:a:apache:zookeeper:3.4.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:3.0.1`,
        fs:  `cpe:2.3:a:phpmyadmin:phpmyadmin:3.4.9.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:3.0.2`,
        fs:  `cpe:2.3:a:fortinet:fortiweb:5.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:3.0.3`,
        fs:  `cpe:2.3:a:adobe:flash_player:23.0.0.207:*:*:*:*:edge:*:*`,
    },
     
    {
        uri: `cpe:/a:adbnewssender_project:adbnewssender:3.0.4`,
        fs:  `cpe:2.3:a:webkitgtk:webkitgtk\&#43;:2.6.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adcillc:simplemeta:6.x-1.0`,
        fs:  `cpe:2.3:a:paolo_bacchilega:file_roller:3.9.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adcillc:simplemeta:6.x-1.1`,
        fs:  `cpe:2.3:a:zarafa:zarafa:6.20.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adcillc:simplemeta:6.x-1.2`,
        fs:  `cpe:2.3:a:adobe:flash_player:11.7.700.202:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adcillc:simplemeta:6.x-1.3`,
        fs:  `cpe:2.3:a:gnu:gnutls:1.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adcillc:simplemeta:6.x-2.x:dev`,
        fs:  `cpe:2.3:a:boehm-demers-weiser:garbage_collector:5.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adcisolutions:node_view_permissions:7.x-1.0`,
        fs:  `cpe:2.3:a:jasig:phpcas:1.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adcisolutions:node_view_permissions:7.x-1.1`,
        fs:  `cpe:2.3:a:alienvault:open_source_security_information_management:3.1.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adcisolutions:node_view_permissions:7.x-1.2`,
        fs:  `cpe:2.3:a:easy_social_project:easy_social:7.x-2.0:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:add-in-express:duplicate_remover_for_microsoft_excel:2.5.0`,
        fs:  `cpe:2.3:o:ibm:vios:2.2.2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:addisgag:addis_gag_funny_amharic_pic:0.1::~~~android~~`,
        fs:  `cpe:2.3:a:ultraseek:ultraseek:5.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:addsoft:storebot:-`,
        fs:  `cpe:2.3:a:vmware:workstation:5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:addsoft:storebot:2002`,
        fs:  `cpe:2.3:a:landing_pages_project:landing_pages_plugin:1.0.4.4:-:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:addsoft:storebot:2005`,
        fs:  `cpe:2.3:a:horde:imp:4.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adersoft:vbsedit:3.4`,
        fs:  `cpe:2.3:a:horde:imp:4.3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adgjm:angel_browser:1.47b`,
        fs:  `cpe:2.3:a:mortbay:jetty:3.1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adgjm:angel_browser:1.62b`,
        fs:  `cpe:2.3:a:trianglemicroworks:scada_data_gateway:3.00.0612:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adgjm:angel_browser:1.68b`,
        fs:  `cpe:2.3:a:gnu:emacs:23.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adgjm:angel_browser:1.76b`,
        fs:  `cpe:2.3:a:mortbay:jetty:4.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adgjm:galapagos_browser:-:-:~~~android~~`,
        fs:  `cpe:2.3:a:cisco:wide_area_application_services_mobile:3.5.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adidas:adidas_eyewear:1.2::~~~android~~`,
        fs:  `cpe:2.3:a:webkitgtk:webkitgtk\&#43;:2.14.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adiscon:loganalyzer:3.2.0`,
        fs:  `cpe:2.3:a:att:communication_manager:5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adiscon:loganalyzer:3.2.1`,
        fs:  `cpe:2.3:a:webyog:sqlyog:6.06:-:community:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adiscon:loganalyzer:3.2.2`,
        fs:  `cpe:2.3:a:google:chrome:6.0.472.37:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adiscon:loganalyzer:3.2.3`,
        fs:  `cpe:2.3:a:isc:bind:9.5.0a3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adiscon:loganalyzer:3.3.0`,
        fs:  `cpe:2.3:a:cybozu:remote_service_manager:2.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adiscon:loganalyzer:3.4.0`,
        fs:  `cpe:2.3:a:ibm:cognos_command_center:10.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adiscon:loganalyzer:3.4.1`,
        fs:  `cpe:2.3:a:adobe:acrobat_dc:15.008.20082:*:*:*:continuous:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adiscon:loganalyzer:3.4.2`,
        fs:  `cpe:2.3:a:marc_deslauriers:software-properties:0.82.7.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adiscon:loganalyzer:3.4.3`,
        fs:  `cpe:2.3:a:infradead:openconnect:4.04:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adiscon:loganalyzer:3.4.4`,
        fs:  `cpe:2.3:a:syntevo:smartsvn:2.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adiscon:loganalyzer:3.5.0`,
        fs:  `cpe:2.3:o:juniper:netscreen_screenos:2.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adiscon:loganalyzer:3.5.1`,
        fs:  `cpe:2.3:a:oxid-esales:eshop:5.0.0:-:enterprise:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adiscon:loganalyzer:3.5.2`,
        fs:  `cpe:2.3:a:oracle:fusion_applications:11.1.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adiscon:loganalyzer:3.5.3`,
        fs:  `cpe:2.3:a:bib2html_project:bib2html:0.8:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adiscon:loganalyzer:3.5.4`,
        fs:  `cpe:2.3:a:microsoft:exchange_instant_messenger:4.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adiscon:loganalyzer:3.5.5`,
        fs:  `cpe:2.3:a:vmware:hyperic_hq:4.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adjam:rekonq:0.0.1`,
        fs:  `cpe:2.3:a:haxx:curl:7.15.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adjam:rekonq:0.0.2`,
        fs:  `cpe:2.3:a:zend:zendservice_nirvanix:2.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adjam:rekonq:0.0.3`,
        fs:  `cpe:2.3:a:idera:pointcheck:1.1.0.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adjam:rekonq:0.0.4`,
        fs:  `cpe:2.3:a:winzip:winzip:11.1.7466:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adjam:rekonq:0.1:alpha`,
        fs:  `cpe:2.3:a:debian:dpkg:1.17.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adjam:rekonq:0.1.0`,
        fs:  `cpe:2.3:h:intel:d845wn_motherboard:p08-0031:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adjam:rekonq:0.1.95`,
        fs:  `cpe:2.3:a:ipswitch:ipswitch_collaboration_suite:8.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adjam:rekonq:0.1.98`,
        fs:  `cpe:2.3:a:web2py:web2py:1.86.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adjam:rekonq:0.2.0`,
        fs:  `cpe:2.3:a:drupal:drupal:8.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adjam:rekonq:0.2.90`,
        fs:  `cpe:2.3:a:hp:product_detection:10.7.9.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adjam:rekonq:0.3.0`,
        fs:  `cpe:2.3:a:mozilla:firefox:46.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adjam:rekonq:0.3.90`,
        fs:  `cpe:2.3:a:monkey-project:monkey_http_daemon:0.6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adjam:rekonq:0.4.0`,
        fs:  `cpe:2.3:a:web2py:web2py:1.94.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adjam:rekonq:0.4.90`,
        fs:  `cpe:2.3:a:businessobjects:crystal_reports_runtime_engine_for_.net_framework_4:13.0.1.220:*:*:*:*:*:x64:*`,
    },
     
    {
        uri: `cpe:/a:adjam:rekonq:0.4.95`,
        fs:  `cpe:2.3:a:componentone:flexgrid:7.1_light:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adjam:rekonq:0.5.0`,
        fs:  `cpe:2.3:a:sun:openoffice.org:3.1.9420:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adjam:rekonq:0.5.80`,
        fs:  `cpe:2.3:a:opennms:opennms:1.10.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adjam:rekonq:0.6.0`,
        fs:  `cpe:2.3:a:simplemachines:simple_machines_forum:1.0.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adjam:rekonq:0.6.1`,
        fs:  `cpe:2.3:a:socketmail:socketmail:2.2.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adjam:rekonq:0.6.80`,
        fs:  `cpe:2.3:a:moodle:moodle:2.3.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adjam:rekonq:0.6.85`,
        fs:  `cpe:2.3:a:primetek:primefaces:6.0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adjam:rekonq:0.6.95`,
        fs:  `cpe:2.3:a:digium:asterisk:1.2.22:netsec:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adjam:rekonq:0.7.0`,
        fs:  `cpe:2.3:a:androidcommunity:hector_leal:13.08.14:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:adjam:rekonq:0.7.1`,
        fs:  `cpe:2.3:a:hp:eprint_and_share:1.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adjam:rekonq:0.7.80`,
        fs:  `cpe:2.3:o:yamaha:rtx1500:8.03.70:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adjam:rekonq:0.7.90`,
        fs:  `cpe:2.3:a:danielb:finder:6.x-1.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adjam:rekonq:0.7.92`,
        fs:  `cpe:2.3:a:lars_hjemli:cgit:0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adjam:rekonq:0.8.0`,
        fs:  `cpe:2.3:a:hp:storage_data_protector:6.20:-:*:*:*:hp-ux:*:*`,
    },
     
    {
        uri: `cpe:/a:admin-font-editor_project:admin-font-editor:1.8::~~~wordpress~~`,
        fs:  `cpe:2.3:a:cisco:unified_computing_system:1.3\(1q\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:administration_views_project:administration_views:7.x-1.0::~~~drupal~~`,
        fs:  `cpe:2.3:a:ca:messaging:1.11.27.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:administration_views_project:administration_views:7.x-1.0:rc1:~~~drupal~~`,
        fs:  `cpe:2.3:a:glub:secure_ftp:2.5:beta_pr1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:administration_views_project:administration_views:7.x-1.1::~~~drupal~~`,
        fs:  `cpe:2.3:a:jenkins-ci:monitoring_plugin:1.50.0:*:*:*:*:cloudbees_jenkins:*:*`,
    },
     
    {
        uri: `cpe:/a:administration_views_project:administration_views:7.x-1.2::~~~drupal~~`,
        fs:  `cpe:2.3:a:promotionalshop:promotional_items:0.1:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:administration_views_project:administration_views:7.x-1.x:dev:~~~drupal~~`,
        fs:  `cpe:2.3:a:ibm:financial_transaction_manager:3.0.2.0:*:*:*:*:ach_services:*:*`,
    },
     
    {
        uri: `cpe:/a:adminsystems_cms_project:adminsystems_cms:4.0.0`,
        fs:  `cpe:2.3:a:ikiwiki:ikiwiki:1.25:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:-`,
        fs:  `cpe:2.3:a:adobe:download_manager:1.6.2.97:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:_reader9.5.3`,
        fs:  `cpe:2.3:o:apple:iphone_os:1.1.0:-:ipodtouch:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:1.0::~~suite~~~`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.31.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:2.0:-:pro`,
        fs:  `cpe:2.3:a:openstack:image_registry_and_delivery_service_\(glance\):juno-3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:3.0`,
        fs:  `cpe:2.3:a:hp:cluster_object_manager:b.02.01.02:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:3.0:-:pro`,
        fs:  `cpe:2.3:a:ecommercesoft:xse_shopping_cart:1.1.1501.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:3.1`,
        fs:  `cpe:2.3:a:tibco:spotfire_statistics_services:5.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:4.0`,
        fs:  `cpe:2.3:a:tweet-blender:tweet-blender:2.2.3:-:-:*:-:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:4.0:-:pro`,
        fs:  `cpe:2.3:a:simplemachines:simple_machines_forum:2.0:beta3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:4.0.5`,
        fs:  `cpe:2.3:a:public_knowledge_project:open_conference_systems:1.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:4.0.5a`,
        fs:  `cpe:2.3:a:meta_tags_quick_project:meta_tags_quick:7.x-2.6:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:4.0.5c`,
        fs:  `cpe:2.3:a:hp:alm_synchronizer:1.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:5.0`,
        fs:  `cpe:2.3:a:boehm-demers-weiser:garbage_collector:4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:5.0:-:pro`,
        fs:  `cpe:2.3:a:phpmyfaq:phpmyfaq:1.5:beta3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:5.0.5`,
        fs:  `cpe:2.3:a:seagullproject:seagull:0.3.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:5.0.6`,
        fs:  `cpe:2.3:a:dotcms:dotcms:1.9.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:5.0.10`,
        fs:  `cpe:2.3:o:juniper:junos:12.1x44:d50:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:5.1`,
        fs:  `cpe:2.3:a:novell:groupwise:7.03:hp3\&#43;ftf:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:5.1:-:pro`,
        fs:  `cpe:2.3:a:matt_johnston:dropbear_ssh_server:0.53:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:6.0`,
        fs:  `cpe:2.3:a:tejimaya:openpne:2.10.4.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:6.0:-:~~standard~~~`,
        fs:  `cpe:2.3:a:ajenti:ajenti:1.2.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:6.0:-:pro`,
        fs:  `cpe:2.3:h:hp:procurve_switch:je071a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:6.0.1`,
        fs:  `cpe:2.3:a:nefarious2_project:nefarious2:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:6.0.2`,
        fs:  `cpe:2.3:a:mozilla:mozilla:1.7.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:6.0.3`,
        fs:  `cpe:2.3:h:d-link:dsl-2730b:c1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:6.0.4`,
        fs:  `cpe:2.3:a:ibm:packaging_utility:1.7.4.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:6.0.5`,
        fs:  `cpe:2.3:a:ibm:tivoli_directory_server:6.1.0.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:6.0.6`,
        fs:  `cpe:2.3:h:ibm:system_x3750_m4:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:7`,
        fs:  `cpe:2.3:o:rockwellautomation:compactlogix:18.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:7.0`,
        fs:  `cpe:2.3:h:cisco:telepresence_profile_55:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:7.0:-:pro`,
        fs:  `cpe:2.3:a:apache:tomcat:8.0.0:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:7.0.1`,
        fs:  `cpe:2.3:a:google:chrome:24.0.1312.34:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:7.0.2`,
        fs:  `cpe:2.3:a:analog:soundmax:5.10.01.5491:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:7.0.3`,
        fs:  `cpe:2.3:h:dell:latitude_e5410:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:7.0.4`,
        fs:  `cpe:2.3:a:philippe_jounin:tftpd32:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:7.0.5`,
        fs:  `cpe:2.3:a:cisco:identity_services_engine_software:1.2\(0.899\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:7.0.6`,
        fs:  `cpe:2.3:o:huawei:s3300_firmware:v100r006c05:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:7.0.7`,
        fs:  `cpe:2.3:a:teamst:testlink:1.7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:7.0.8`,
        fs:  `cpe:2.3:a:gpsd_project:gpsd:3.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:7.0.9`,
        fs:  `cpe:2.3:a:red-gate:sql_prompt:3.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:7.1.0`,
        fs:  `cpe:2.3:a:wireshark:wireshark:1.6.0:*:*:*:*:*:itanium:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:7.1.1`,
        fs:  `cpe:2.3:a:sun:jre:1.5.0:update5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:7.1.2`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.6.7-6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:7.1.3`,
        fs:  `cpe:2.3:a:octopus:octopus_deploy:3.4.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:7.1.4`,
        fs:  `cpe:2.3:a:deliciousdays:cforms:10.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:7.1.4:-`,
        fs:  `cpe:2.3:a:postgresql:postgresql:8.4.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:8`,
        fs:  `cpe:2.3:a:opscode:chef:0.8.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:8.0`,
        fs:  `cpe:2.3:a:dovecot:dovecot:1.2.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:8.0:-:pro`,
        fs:  `cpe:2.3:a:debian:debmake:3.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:8.0.0`,
        fs:  `cpe:2.3:a:ttnetmuzik:ttnet_muzik:3.2:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:8.1`,
        fs:  `cpe:2.3:a:pgp:desktop_for_windows:10.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:8.1.1`,
        fs:  `cpe:2.3:a:funnymonkey:feature_set:7.x-1.0-beta1:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:8.1.2`,
        fs:  `cpe:2.3:a:altiris:console:6.5.1144:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:8.1.2:-:pro`,
        fs:  `cpe:2.3:o:ibm:security_access_manager_for_web_7.0_firmware:7.0.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:8.1.3`,
        fs:  `cpe:2.3:a:e-catchup:basercms:1.6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:8.1.3:-:pro`,
        fs:  `cpe:2.3:a:apple:itunes:12.6.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:8.1.4`,
        fs:  `cpe:2.3:a:apple:quicktime:6.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:8.1.5`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.7.9-6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:8.1.5:-`,
        fs:  `cpe:2.3:a:isc:bind:9.7.0:p2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:8.1.6`,
        fs:  `cpe:2.3:a:contao:contao_cms:2.4.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:8.1.7`,
        fs:  `cpe:2.3:o:d-link:dcs-2132l_firmware:1.08.01:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:8.2`,
        fs:  `cpe:2.3:a:john_albin:zen:6.x-1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:8.2.1`,
        fs:  `cpe:2.3:a:cisco:secure_desktop:3.6.6104:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:8.2.1:-`,
        fs:  `cpe:2.3:a:md-systems:simplenews:7.x-1.0:-:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:8.2.2`,
        fs:  `cpe:2.3:a:samba:samba:3.2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:8.2.3`,
        fs:  `cpe:2.3:a:ibm:infosphere_information_server_metadata_workbench:8.0.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:8.2.4`,
        fs:  `cpe:2.3:a:actividentity:actividentity_device_installer:2.10.00006:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:8.2.4:-`,
        fs:  `cpe:2.3:a:libraw:libraw:0.13.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:8.2.5`,
        fs:  `cpe:2.3:a:rocomotion:p_up_board_i:1.06:*:g:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:8.2.6`,
        fs:  `cpe:2.3:o:apple:mac_os:8.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:8.3`,
        fs:  `cpe:2.3:a:atlassian:bamboo:5.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:8.3::~~professional~~~`,
        fs:  `cpe:2.3:a:avg:anti-virus_plus_firewall:9.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:8.3:-`,
        fs:  `cpe:2.3:a:exactcode:exactimage:0.7.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:8.3.1::~~professional~~~`,
        fs:  `cpe:2.3:a:ibm:forms_viewer:4.0.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:8.3.1:-`,
        fs:  `cpe:2.3:a:krzysztof_kowalczyk:sumatrapdf:2.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9`,
        fs:  `cpe:2.3:a:adodb_project:adodb:5.00:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.0`,
        fs:  `cpe:2.3:o:cisco:vpn_3060_concentrator:4.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.0:-:pro`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.4:p5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.1`,
        fs:  `cpe:2.3:a:gnu:tar:1.13.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.1:-:pro`,
        fs:  `cpe:2.3:a:vbulletin:vbulletin:5.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.1.1`,
        fs:  `cpe:2.3:a:bfgminer:bfgminer:4.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.1.1:-`,
        fs:  `cpe:2.3:a:cloudbees:jenkins:1.447.7.1:-:enterprise:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.1.2`,
        fs:  `cpe:2.3:a:softwaretoolbox:top_server:5.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.1.3`,
        fs:  `cpe:2.3:a:xnview:xnview:1.98.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.1.3:-`,
        fs:  `cpe:2.3:a:apache:struts:1.2.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.2`,
        fs:  `cpe:2.3:o:codeaurora:android-msm:3.12.23:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.2:-`,
        fs:  `cpe:2.3:a:google:chrome:24.0.1312.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.3`,
        fs:  `cpe:2.3:h:hp:procurve_switch:je102a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.3:-:pro`,
        fs:  `cpe:2.3:a:microsoft:windows_mobile:5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.3.1`,
        fs:  `cpe:2.3:o:juniper:junos:9.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.3.1:-`,
        fs:  `cpe:2.3:a:connections_project:connections:0.2.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.3.2`,
        fs:  `cpe:2.3:a:barracudadrive:barracudadrive:6.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.3.2:-`,
        fs:  `cpe:2.3:a:openbsd:openssh:2.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.3.3`,
        fs:  `cpe:2.3:a:papercut:papercut_ng:12.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.3.4`,
        fs:  `cpe:2.3:a:strongswan:strongswan:5.3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.3.4:-`,
        fs:  `cpe:2.3:a:actividentity:activclient_cac:6.2.0.73:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.4`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.7:p123:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.4.1`,
        fs:  `cpe:2.3:a:moodle:moodle:2.9.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.4.1:-`,
        fs:  `cpe:2.3:o:huawei:s6700_firmware:v200r002c00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.4.2`,
        fs:  `cpe:2.3:a:ultraedit:ultracompare:6.30:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.4.2:-`,
        fs:  `cpe:2.3:a:oracle:virtualization:4.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.4.3`,
        fs:  `cpe:2.3:a:mit:kerberos:5-1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.4.3:-`,
        fs:  `cpe:2.3:o:apple:airport_base_station_firmware:7.6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.4.4`,
        fs:  `cpe:2.3:h:alliedtelesyn:at-s24:2.4.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.4.4:-`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(25\)s6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.4.5`,
        fs:  `cpe:2.3:a:pivotal_software:cloud_foundry_elastic_runtime:1.6.25:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.4.5:-`,
        fs:  `cpe:2.3:a:wp-plugins:wp-print:2.30:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.4.6`,
        fs:  `cpe:2.3:a:david_shadoff:mednafen:0.8.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.4.6:-`,
        fs:  `cpe:2.3:o:juniper:junos:13.2x52:d10:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.4.7`,
        fs:  `cpe:2.3:a:cms_tree_page_view_project:cms_tree_page_view:0.4.1:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.5`,
        fs:  `cpe:2.3:a:drupal:drupal:5.0:dev:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.5.1`,
        fs:  `cpe:2.3:o:iij:seil_x86_fuji_firmware:2.32:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.5.2`,
        fs:  `cpe:2.3:o:redhat:linux:6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.5.3`,
        fs:  `cpe:2.3:a:cisco:security_manager:4.4:-:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.5.4`,
        fs:  `cpe:2.3:a:atlassian:bamboo:5.14.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:9.5.5`,
        fs:  `cpe:2.3:a:hp:openview_mib_api_loader_and_mibs:1.30.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:10.0`,
        fs:  `cpe:2.3:a:novell:iprint:5.82:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:10.0:-:pro`,
        fs:  `cpe:2.3:o:iij:seil\/x2_firmware:4.32:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:10.0.1`,
        fs:  `cpe:2.3:a:atlassian:crucible:2.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:10.0.1:-:pro`,
        fs:  `cpe:2.3:a:bea:weblogic_server:5.1:sp6:win32:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:10.0.2`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.6:p3_rc3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:10.0.3`,
        fs:  `cpe:2.3:a:radware:linkproof:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:10.1`,
        fs:  `cpe:2.3:h:foundrynet:fastiron_400_switch:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:10.1.0`,
        fs:  `cpe:2.3:a:oracle:service_bus:12.2.1.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:10.1.1`,
        fs:  `cpe:2.3:a:google:chrome:18.0.1009.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:10.1.2`,
        fs:  `cpe:2.3:a:karen_stevenson:date:6.x-2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:10.1.3`,
        fs:  `cpe:2.3:a:ponsoftware:explzh:4.93:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:10.1.4`,
        fs:  `cpe:2.3:a:rpm:rpm:2.5.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:10.1.5`,
        fs:  `cpe:2.3:a:chatelao:php_address_book:3.2.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:10.1.6`,
        fs:  `cpe:2.3:a:piwik:piwik:0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:10.1.7`,
        fs:  `cpe:2.3:h:oracle:netra_sparc_t3-1:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:10.1.8`,
        fs:  `cpe:2.3:a:mozilla:bugzilla:2.23.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:10.1.9`,
        fs:  `cpe:2.3:a:partytrack_library_project:partytrack_library:-:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:10.1.10`,
        fs:  `cpe:2.3:a:contao:contao_cms:2.1.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:10.1.11`,
        fs:  `cpe:2.3:a:netsweeper:netsweeper:3.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:10.1.12`,
        fs:  `cpe:2.3:a:kolja_schleich:leaguemanager:1.0:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:10.1.13`,
        fs:  `cpe:2.3:a:anthony_mills:s3_video:0.98:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:10.1.14`,
        fs:  `cpe:2.3:a:gnu:lsh:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:10.1.15`,
        fs:  `cpe:2.3:o:huawei:s9700_firmware:v200r001c00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:10.1.16`,
        fs:  `cpe:2.3:a:google:chrome:11.0.697.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:11.0`,
        fs:  `cpe:2.3:a:pivotal_software:rabbitmq:1.5.10:*:*:*:*:pivotal_cloud_foundry:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:11.0.0`,
        fs:  `cpe:2.3:a:buffalotech:bbr-4hg_firmware:1.04:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:11.0.1`,
        fs:  `cpe:2.3:a:peter_kovacs:timesheet_next_gen:1.5.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:11.0.2`,
        fs:  `cpe:2.3:a:bamboosolutions:enterprise_license:1.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:11.0.3`,
        fs:  `cpe:2.3:a:bamboosolutions:bamboo.worldclockandweather.spv3:1.0.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:11.0.4`,
        fs:  `cpe:2.3:a:x:x.org_x11:6.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:11.0.5`,
        fs:  `cpe:2.3:a:webyog:sqlyog:6.05:-:community:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:11.0.5:-:~~~windows~~`,
        fs:  `cpe:2.3:a:netsweeper:netsweeper:3.0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:11.0.6`,
        fs:  `cpe:2.3:o:yahama:rt100i:2.02.39:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:11.0.7`,
        fs:  `cpe:2.3:o:cisco:ios:12.4jda:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:11.0.8`,
        fs:  `cpe:2.3:a:ati:catalyst_control_center_localization:108.2146.2565.38893:*:*:ja:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:11.0.9`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.200:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:11.0.10`,
        fs:  `cpe:2.3:a:greenbone:greenbone_security_assistant:5.0:beta11:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:11.0.11`,
        fs:  `cpe:2.3:a:wireshark:wireshark:2.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:11.0.12`,
        fs:  `cpe:2.3:a:sitracker:support_incident_tracker:3.07:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:11.0.13`,
        fs:  `cpe:2.3:a:ibm:lotus_domino:8.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:11.0.14`,
        fs:  `cpe:2.3:a:infotecs:vipnet_coordinator:3.2.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:11.0.15`,
        fs:  `cpe:2.3:a:palletsprojects:werkzeug:0.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:11.0.15::~~desktop~~~`,
        fs:  `cpe:2.3:a:dotclear:dotclear:2.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:11.0.16`,
        fs:  `cpe:2.3:a:invisionpower:ip.nexus:1.5.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:11.0.17`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(13e\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:11.0.18`,
        fs:  `cpe:2.3:a:otrs:otrs_help_desk:3.3.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:11.0.19`,
        fs:  `cpe:2.3:a:openafs:openafs:1.4.14.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:11.0.20`,
        fs:  `cpe:2.3:o:apple:iphone_os:2.0.1:-:iphone:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:11.0.21`,
        fs:  `cpe:2.3:a:nalin_dahyabhai:vte:0.25.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:11.0.22`,
        fs:  `cpe:2.3:a:ibm:websphere_message_broker:6.1.0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:11.0.23`,
        fs:  `cpe:2.3:a:f5:big-ip:9.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:17.011.30059`,
        fs:  `cpe:2.3:a:spreecommerce:spree:1.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:17.011.30065`,
        fs:  `cpe:2.3:a:symantec:vrtssfmh:2.1.210:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:17.011.30066`,
        fs:  `cpe:2.3:a:f5:big-ip_link_controller:12.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:17.011.30068`,
        fs:  `cpe:2.3:a:tom_braider:count_per_day:2.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat:17.011.30070`,
        fs:  `cpe:2.3:a:altiris:task_server_language_pack:6.0.1404:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_3d:-`,
        fs:  `cpe:2.3:a:symantec:veritas_storage_foundation_for_windows_ccf_client:6.1.10000:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_business_tools:-`,
        fs:  `cpe:2.3:a:octopus:octopus_deploy:3.3.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_business_tools:4.0`,
        fs:  `cpe:2.3:a:skyphe:file-gallery:1.7.6:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_business_tools:4.05`,
        fs:  `cpe:2.3:a:cisco:mobility_services_engine:8.0\(110.0\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.006.30060::~~classic~~~`,
        fs:  `cpe:2.3:a:search.cpan:libwww-perl:5.813:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.006.30094::~~classic~~~`,
        fs:  `cpe:2.3:h:apple:imac:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.006.30096::~~classic~~~`,
        fs:  `cpe:2.3:a:dell:client_configuration_toolkit:1.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.006.30097::~~classic~~~`,
        fs:  `cpe:2.3:a:popuapp:popu_2\:_get_likes_on_instagram:1.7.5:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.006.30119::~~classic~~~`,
        fs:  `cpe:2.3:a:digium:asterisk:1.4.19:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.006.30121::~~classic~~~`,
        fs:  `cpe:2.3:a:drupal:drupal:8.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.006.30172::~~classic~~~`,
        fs:  `cpe:2.3:h:hp:procurve_switch:jd008a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.006.30173::~~classic~~~`,
        fs:  `cpe:2.3:a:gnome:gnome_display_manager:2.24.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.006.30174::~~classic~~~`,
        fs:  `cpe:2.3:o:yahama:rt300i:6.03.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.006.30198::~~classic~~~`,
        fs:  `cpe:2.3:a:webmaster-source:wp125:1.4.9:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.006.30201::~~classic~~~`,
        fs:  `cpe:2.3:a:apache:cxf:2.7.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.006.30243::~~classic~~~`,
        fs:  `cpe:2.3:o:cisco:unified_computing_system_infrastructure_and_unified_computing_system_software:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.006.30244::~~classic~~~`,
        fs:  `cpe:2.3:a:shopware:shopware:5.3.0:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.006.30279::~~classic~~~`,
        fs:  `cpe:2.3:a:tracker-software:pdf-xchange:2.5.0190:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.006.30280::~~classic~~~`,
        fs:  `cpe:2.3:a:bea:weblogic_integration:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.006.30306::~~classic~~~`,
        fs:  `cpe:2.3:a:exactcode:exactimage:0.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.006.30352::~~classic~~~`,
        fs:  `cpe:2.3:a:oracle:weblogic_server:12.1.2.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.006.30354::~~classic~~~`,
        fs:  `cpe:2.3:a:symantec:liveupdate_administrator:2.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.006.30355::~~classic~~~`,
        fs:  `cpe:2.3:a:apple:quicktime:7.1.1:-:windows:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.006.30392::~~classic~~~`,
        fs:  `cpe:2.3:a:shopware:shopware:5.2.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.006.30394::~~classic~~~`,
        fs:  `cpe:2.3:a:ibm:maximo_integration_composer:6.2.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.008.20082::~~continuous~~~`,
        fs:  `cpe:2.3:o:netbsd:netbsd:1.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.009.20069::~~continuous~~~`,
        fs:  `cpe:2.3:a:symantec:vrtsaa:5.0.141:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.009.20071::~~continuous~~~`,
        fs:  `cpe:2.3:a:cisco:telepresence_multipoint_switch_software:1.1.2\(6\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.009.20077::~~continuous~~~`,
        fs:  `cpe:2.3:a:fetchmail:fetchmail:6.3.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.009.20079::~~continuous~~~`,
        fs:  `cpe:2.3:a:netsweeper:netsweeper:2.6.29.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.010.20056::~~continuous~~~`,
        fs:  `cpe:2.3:h:nec:superscript_4400:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.010.20059::~~continuous~~~`,
        fs:  `cpe:2.3:a:igor_sysoev:nginx:0.8.50:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.010.20060::~~continuous~~~`,
        fs:  `cpe:2.3:a:quassel-irc:quassel_irc:0.4.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.016.20039::~~continuous~~~`,
        fs:  `cpe:2.3:a:microsoft:biztalk_server:2000:sp2:standard:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.016.20041::~~continuous~~~`,
        fs:  `cpe:2.3:a:mozilla:seamonkey:2.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.016.20045::~~continuous~~~`,
        fs:  `cpe:2.3:h:3com:3c8532:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.017.20050::~~continuous~~~`,
        fs:  `cpe:2.3:a:rubygems:rubygems:2.4.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.017.20053::~~continuous~~~`,
        fs:  `cpe:2.3:a:libguestfs:libguestfs:1.0.87:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.020.20039::~~continuous~~~`,
        fs:  `cpe:2.3:a:sourcefabric:newscoop:3.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.020.20042::~~continuous~~~`,
        fs:  `cpe:2.3:a:php:php:5.5.0:beta1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.023.20053::~~continuous~~~`,
        fs:  `cpe:2.3:a:google:chrome:6.0.480.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.023.20056::~~continuous~~~`,
        fs:  `cpe:2.3:a:gnu:glibc:2.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:15.023.20070::~~continuous~~~`,
        fs:  `cpe:2.3:a:shopware:shopware:5.2.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:17.009.20044::~~continuous~~~`,
        fs:  `cpe:2.3:a:google:chrome:20.0.1132.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:17.009.20058::~~continuous~~~`,
        fs:  `cpe:2.3:a:ibm:tivoli_storage_manager:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:17.012.20093::~~continuous~~~`,
        fs:  `cpe:2.3:a:tumbleweed:validation_authority:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:17.012.20095::~~continuous~~~`,
        fs:  `cpe:2.3:o:cisco:ios:12.4\(3\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:17.012.20098::~~continuous~~~`,
        fs:  `cpe:2.3:a:hp:cio_components_installer:2.1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:18.009.20044::~~continuous~~~`,
        fs:  `cpe:2.3:a:torproject:tor:0.1.1.14:alpha:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:18.009.20050::~~continuous~~~`,
        fs:  `cpe:2.3:a:veritas:enterprise_administrator:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:2015.006.30060::~~classic~~~`,
        fs:  `cpe:2.3:o:hp:integrated_lights-out_firmware:1.93:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_dc:2015.008.20082::~~continuous~~~`,
        fs:  `cpe:2.3:o:cisco:ios:15.1\(2\)sy6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_elements:-`,
        fs:  `cpe:2.3:a:google:chrome:32.0.1684.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_pdfwriter:3.3`,
        fs:  `cpe:2.3:a:teampass:teampass:2.1.19:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:-`,
        fs:  `cpe:2.3:o:iij:seil_b1_firmware:3.93:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:3.0`,
        fs:  `cpe:2.3:a:commvault:base_client:9.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:3.01`,
        fs:  `cpe:2.3:a:redhat:richfaces:4.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:3.02`,
        fs:  `cpe:2.3:a:apache:tomcat:5.5.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:4.0`,
        fs:  `cpe:2.3:o:yahama:rt250i:8.02.40:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:4.0.5`,
        fs:  `cpe:2.3:a:actividentity:activclient_cac:6.1.3.123:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:4.0.5a`,
        fs:  `cpe:2.3:a:mysql:mysql:3.23.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:4.0.5c`,
        fs:  `cpe:2.3:a:nagios:remote_plug_in_executor:2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:4.5`,
        fs:  `cpe:2.3:a:isc:inn:1.7.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:5.0`,
        fs:  `cpe:2.3:a:mobiloapps:anderson_musaamil:1.4:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:5.0.5`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.7.2-3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:5.0.6`,
        fs:  `cpe:2.3:a:ibm:tririga_application_platform:3.3.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:5.0.7`,
        fs:  `cpe:2.3:a:altiris:pxe_manager:6.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:5.0.9`,
        fs:  `cpe:2.3:a:lenovo:xclarity_administrator:1.3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:5.0.10`,
        fs:  `cpe:2.3:a:sitaram_chamarty:gitolite:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:5.0.11`,
        fs:  `cpe:2.3:o:centos:centos:6.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:5.1`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.447.1:*:*:*:lts:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:6.0`,
        fs:  `cpe:2.3:a:git-scm:git:1.6.2:rc0:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:6.0.1`,
        fs:  `cpe:2.3:a:xmlsoft:libxslt:1.0.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:6.0.2`,
        fs:  `cpe:2.3:a:sun:java_system_web_proxy_server:3.6:sp5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:6.0.3`,
        fs:  `cpe:2.3:a:gnu:sharutils:4.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:6.0.4`,
        fs:  `cpe:2.3:a:adobe:acrobat:9.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:6.0.5`,
        fs:  `cpe:2.3:a:ca:unicenter_data_transport_option:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:7.0`,
        fs:  `cpe:2.3:a:simon_tatham:putty:0.57:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:7.0.1`,
        fs:  `cpe:2.3:a:otrs:otrs_help_desk:3.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:7.0.2`,
        fs:  `cpe:2.3:a:atlassian:jira:3.4.3:*:enterprise:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:7.0.3`,
        fs:  `cpe:2.3:o:blackberry:qnx_neutrino_rtos:6.5.0:sp1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:7.0.4`,
        fs:  `cpe:2.3:a:google:chrome:9.0.597.38:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:7.0.5`,
        fs:  `cpe:2.3:a:vmware:vcenter_converter_client:4.2.0.254483:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:7.0.6`,
        fs:  `cpe:2.3:o:cisco:ios:12.1\(1\)db2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:7.0.7`,
        fs:  `cpe:2.3:a:simon_tatham:putty:0.50:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:7.0.8`,
        fs:  `cpe:2.3:a:mnogosearch:mnogosearch:3.2.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:7.0.9`,
        fs:  `cpe:2.3:h:lexmark:c544:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:7.1.0`,
        fs:  `cpe:2.3:a:cisco:application_and_content_networking_software:4.2.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:8.0`,
        fs:  `cpe:2.3:o:juniper:junos:5.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:8.1`,
        fs:  `cpe:2.3:a:ikiwiki:ikiwiki:3.20110105:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:8.1.1`,
        fs:  `cpe:2.3:a:isc:dhcpd:3.0_b2pl9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:8.1.2`,
        fs:  `cpe:2.3:h:hp:laserjet_m5025_mfp:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:8.1.3`,
        fs:  `cpe:2.3:a:haxx:libcurl:7.15.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:8.1.4`,
        fs:  `cpe:2.3:a:pivotal_software:spring_framework:4.2.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:8.1.5`,
        fs:  `cpe:2.3:a:rocomotion:p_up_board:1.24:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:8.1.6`,
        fs:  `cpe:2.3:a:askbot:askbot:0.7.44:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:8.1.7`,
        fs:  `cpe:2.3:a:commerce:america\&#39;s_economy_for_phone:1.5.2:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:8.2`,
        fs:  `cpe:2.3:a:torproject:tor:0.1.1.4:alpha:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:8.2.1`,
        fs:  `cpe:2.3:h:cisco:3945_integrated_services_router:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:8.2.2`,
        fs:  `cpe:2.3:o:virtualaccess:gw6110a_firmware:10.00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:8.2.3`,
        fs:  `cpe:2.3:a:ca:messaging:1.11.28.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:8.2.4`,
        fs:  `cpe:2.3:a:palletsprojects:werkzeug:0.9.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:8.2.6`,
        fs:  `cpe:2.3:a:kunena:kunena:5.0.2:*:*:*:*:joomla\!:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:8.3`,
        fs:  `cpe:2.3:a:php:php:5.4.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:9.0`,
        fs:  `cpe:2.3:o:sun:sunos:5.11:*:express:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:9.1`,
        fs:  `cpe:2.3:a:cisco:unified_contact_center_enterprise:6.0\(0\):sr8:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:9.1.1`,
        fs:  `cpe:2.3:a:opennms:opennms:1.9.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:9.1.2`,
        fs:  `cpe:2.3:a:openbsd:openssh:2.5.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:9.1.3`,
        fs:  `cpe:2.3:a:gridgain:gridgain:8.0.0:ea2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:9.2`,
        fs:  `cpe:2.3:a:cisco:network_admission_control_manager_and_server_system_software:3.3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:9.3`,
        fs:  `cpe:2.3:a:lyesoft:andexplorer:1.7:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:9.3.1`,
        fs:  `cpe:2.3:a:gnu:mailutils:0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:9.3.2`,
        fs:  `cpe:2.3:h:sun:storedge_3510_fc_array:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:9.3.3`,
        fs:  `cpe:2.3:a:microsoft:ie:7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:9.3.4`,
        fs:  `cpe:2.3:a:munin-monitoring:munin:2.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:9.4`,
        fs:  `cpe:2.3:a:igor_sysoev:nginx:0.3.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:9.4.1`,
        fs:  `cpe:2.3:a:freedesktop:poppler:0.47.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:9.4.2`,
        fs:  `cpe:2.3:a:htc:vivid_software:3.26.502.56:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:9.4.3`,
        fs:  `cpe:2.3:a:cisco:web_security_appliance:9.5_base:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:9.4.4`,
        fs:  `cpe:2.3:a:gnu:gdb:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:9.4.5`,
        fs:  `cpe:2.3:a:sun:jre:1.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:9.4.6`,
        fs:  `cpe:2.3:a:b2evolution:b2evolution:4.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:9.4.7`,
        fs:  `cpe:2.3:a:mywebsiteadvisor:simple_security:1.1.5:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:9.5`,
        fs:  `cpe:2.3:a:symantec:im_manager:8.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:9.5.1`,
        fs:  `cpe:2.3:a:theforeman:foreman:1.15.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:9.5.2`,
        fs:  `cpe:2.3:a:ivanjaros:feed_block:6.x-1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:9.5.3`,
        fs:  `cpe:2.3:a:harfbuzz_project:harfbuzz:1.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:9.5.4`,
        fs:  `cpe:2.3:a:splunk:splunk:3.3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:9.5.5`,
        fs:  `cpe:2.3:a:vmware:player:1.0.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:10.0`,
        fs:  `cpe:2.3:a:ibm:tivoli_directory_server:6.1.0.66:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:10.0.1`,
        fs:  `cpe:2.3:a:deliciousdays:cforms:8.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:10.0.2`,
        fs:  `cpe:2.3:a:cisco:videoscape_delivery_system_origin_server:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:10.0.3`,
        fs:  `cpe:2.3:o:cisco:ios_xr:5.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:10.1`,
        fs:  `cpe:2.3:a:gitlab:gitlab:8.9.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:10.1.0`,
        fs:  `cpe:2.3:a:apple:quicktime:6.0.1:-:mac:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:10.1.1`,
        fs:  `cpe:2.3:a:gnu:gnutls:1.1.23:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:10.1.2`,
        fs:  `cpe:2.3:a:springsignage:xibo:1.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:10.1.3`,
        fs:  `cpe:2.3:a:oracle:hyperion:11.1.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:10.1.4`,
        fs:  `cpe:2.3:a:ulli_horlacher:fex:20120203:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:10.1.5`,
        fs:  `cpe:2.3:a:libpng:libpng:1.0.29:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:10.1.6`,
        fs:  `cpe:2.3:a:oracle:jrockit:r27.7.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:10.1.7`,
        fs:  `cpe:2.3:a:hp:insight_diagnostics:6.4.1_a:unknown:offline:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:10.1.8`,
        fs:  `cpe:2.3:a:webyog:sqlyog:5.32:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:10.1.9`,
        fs:  `cpe:2.3:a:octobercms:october_cms:1.0.157:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:10.1.10`,
        fs:  `cpe:2.3:a:gnome:librsvg:2.13.92:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:10.1.11`,
        fs:  `cpe:2.3:a:ipswitch:ipswitch_collaboration_suite:2006.02:-:standard:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:10.1.12`,
        fs:  `cpe:2.3:a:ibm:lotus_quickr_for_domino:8.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:10.1.13`,
        fs:  `cpe:2.3:a:emc:vplex_geosynchrony:5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:10.1.14`,
        fs:  `cpe:2.3:a:ibm:db2:9.1.0.7:a:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:10.1.15`,
        fs:  `cpe:2.3:a:cisco:adaptive_security_appliance_software:8.4.7.23:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:10.1.16`,
        fs:  `cpe:2.3:a:cisco:unified_computing_system:2.0_5d:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0`,
        fs:  `cpe:2.3:a:ibm:tivoli_storage_manager:7.1.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.0`,
        fs:  `cpe:2.3:a:puppetlabs:facter:2.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.01`,
        fs:  `cpe:2.3:h:juniper:netscreen-5gt:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.1`,
        fs:  `cpe:2.3:a:octopus:octopus_deploy:3.4.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.2`,
        fs:  `cpe:2.3:a:oracle:peoplesoft_enterprise_peopletools:8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.02`,
        fs:  `cpe:2.3:a:roxio:recordnow_data:2.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.03`,
        fs:  `cpe:2.3:a:squid-cache:squid:3.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.3`,
        fs:  `cpe:2.3:a:ibm:worklight:5.0.0.3:*:*:*:consumer:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.04`,
        fs:  `cpe:2.3:h:cisco:aironet_ap350:11.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.4`,
        fs:  `cpe:2.3:a:octopus:octopus_deploy:3.8.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.05`,
        fs:  `cpe:2.3:a:jenkins:git:2.3.1:*:*:*:*:jenkins:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.5`,
        fs:  `cpe:2.3:a:google:chrome:28.0.1500.91:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.5:-:~~~windows~~`,
        fs:  `cpe:2.3:a:ipswitch:whatsup:2005:sp1:professional:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.06`,
        fs:  `cpe:2.3:a:vanillaforums:vanilla_forums:2.0.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.6`,
        fs:  `cpe:2.3:a:moodle:moodle:2.2.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.07`,
        fs:  `cpe:2.3:a:zend:zend_framework:1.11.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.7`,
        fs:  `cpe:2.3:h:siemens:scalance_w788-1rr:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.8`,
        fs:  `cpe:2.3:a:seasarfoundation:s2struts:1.3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.08`,
        fs:  `cpe:2.3:a:adobe:air_sdk:19.0.0.241:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.09`,
        fs:  `cpe:2.3:a:atutor:atutor:1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.9`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.10.91:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.10`,
        fs:  `cpe:2.3:a:roger_padilla_camacho:easy_breadcrumb:7.x-2.4:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.11`,
        fs:  `cpe:2.3:a:dhcpcd_project:dhcpcd:4.0.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.12`,
        fs:  `cpe:2.3:a:connect2id:nimbus_jose\&#43;jwt:4.29:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.13`,
        fs:  `cpe:2.3:h:supermicro:x9sre-3f:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.14`,
        fs:  `cpe:2.3:a:mollify:mollify:1.0.5.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.15`,
        fs:  `cpe:2.3:a:ibm:san_volume_controller_software:7.1.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.16`,
        fs:  `cpe:2.3:a:cisco:secure_access_control_server:3.0:-:windows:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.17`,
        fs:  `cpe:2.3:a:octopus:octopus_deploy:3.5.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.18`,
        fs:  `cpe:2.3:a:google:chrome:21.0.1180.62:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.19`,
        fs:  `cpe:2.3:a:obm:open_business_management:2.3.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.21`,
        fs:  `cpe:2.3:a:google:chrome:23.0.1271.37:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.22`,
        fs:  `cpe:2.3:a:cakefoundation:cakephp:1.2.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:11.0.23`,
        fs:  `cpe:2.3:a:apache:tomcat:7.0.24:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:15.006.30244`,
        fs:  `cpe:2.3:a:isc:bind:9.11.0:b1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:15.020.20042`,
        fs:  `cpe:2.3:a:gnome:librsvg:2.18.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:17.011.30059`,
        fs:  `cpe:2.3:a:intel:prosafe_for_wired_connections:8.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:17.011.30065`,
        fs:  `cpe:2.3:a:cisco:mediasense:10.5\(1\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:17.011.30066`,
        fs:  `cpe:2.3:a:splunk:splunk:4.3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:17.011.30068`,
        fs:  `cpe:2.3:a:gnu:queue:1.20.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader:17.011.30070`,
        fs:  `cpe:2.3:a:ibm:san_volume_controller_software:6.4.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.006.30060::~~classic~~~`,
        fs:  `cpe:2.3:a:roundup-tracker:roundup:1.4.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.006.30094::~~classic~~~`,
        fs:  `cpe:2.3:a:rubygems:rubygems:2.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.006.30096::~~classic~~~`,
        fs:  `cpe:2.3:o:cisco:ons_15454_optical_transport_platform:3.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.006.30097::~~classic~~~`,
        fs:  `cpe:2.3:h:rad:asm-40:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.006.30119::~~classic~~~`,
        fs:  `cpe:2.3:a:debian:adequate:0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.006.30121::~~classic~~~`,
        fs:  `cpe:2.3:a:digium:asterisk:1.6.0.23:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.006.30172::~~classic~~~`,
        fs:  `cpe:2.3:a:ibm:db2_connect:9.7.0.1:*:*:*:unlimited:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.006.30173::~~classic~~~`,
        fs:  `cpe:2.3:a:prestashop:prestashop:1.4.0.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.006.30174::~~classic~~~`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.35.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.006.30198::~~classic~~~`,
        fs:  `cpe:2.3:a:7-zip:7-zip:4.27:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.006.30201::~~classic~~~`,
        fs:  `cpe:2.3:a:ibm:smartcloud_control_desk:7.5.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.006.30243::~~classic~~~`,
        fs:  `cpe:2.3:a:livetecs:timeline:6.2.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.006.30244::~~classic~~~`,
        fs:  `cpe:2.3:o:freebsd:freebsd:2.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.006.30279::~~classic~~~`,
        fs:  `cpe:2.3:a:wingrep:windows_grep:2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.006.30280::~~classic~~~`,
        fs:  `cpe:2.3:a:kolja_schleich:leaguemanager:3.6.6:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.006.30306::~~classic~~~`,
        fs:  `cpe:2.3:a:adobe:flash_player:10.1.82.76:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.006.30352::~~classic~~~`,
        fs:  `cpe:2.3:a:nero:mediahome_ce:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.006.30354::~~classic~~~`,
        fs:  `cpe:2.3:a:oracle:mysql:5.6.34:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.006.30355::~~classic~~~`,
        fs:  `cpe:2.3:h:cisco:ie-4000-8t4g-e:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.006.30392::~~classic~~~`,
        fs:  `cpe:2.3:a:bea:weblogic_server:5.1:sp12:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.006.30394::~~classic~~~`,
        fs:  `cpe:2.3:a:bitcoin:bitcoind:0.5.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.008.20082::~~continuous~~~`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.7:p463:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.009.20069::~~continuous~~~`,
        fs:  `cpe:2.3:a:google:chrome:23.0.1271.33:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.009.20071::~~continuous~~~`,
        fs:  `cpe:2.3:a:kde:kmail:5.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.009.20077::~~continuous~~~`,
        fs:  `cpe:2.3:o:iij:seil_b1_firmware:3.57:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.009.20079::~~continuous~~~`,
        fs:  `cpe:2.3:a:redhat:libreport:2.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.010.20056::~~continuous~~~`,
        fs:  `cpe:2.3:a:mozilla:network_security_services:3.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.010.20059::~~continuous~~~`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.7:p134:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.010.20060::~~continuous~~~`,
        fs:  `cpe:2.3:a:ibm:storwize_v7000_software:6.3.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.016.20039::~~continuous~~~`,
        fs:  `cpe:2.3:a:netscape:navigator:4.61:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.016.20041::~~continuous~~~`,
        fs:  `cpe:2.3:a:apple:quicktime:6.5.2:-:windows:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.016.20045::~~continuous~~~`,
        fs:  `cpe:2.3:a:travis_shirk:eyed3:0.6.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.017.20050::~~continuous~~~`,
        fs:  `cpe:2.3:o:hp:procurve_switch_software:pa.02.08:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.017.20053::~~continuous~~~`,
        fs:  `cpe:2.3:a:o-dyn:collabtive:0.6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.020.20039::~~continuous~~~`,
        fs:  `cpe:2.3:a:winace:unace:1.2b:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.020.20042::~~continuous~~~`,
        fs:  `cpe:2.3:o:huawei:s9700_firmware:v200r003:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.023.20053::~~continuous~~~`,
        fs:  `cpe:2.3:a:itechscripts:itechclassifieds:3.03.057:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.023.20056::~~continuous~~~`,
        fs:  `cpe:2.3:a:gnu:grep:2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:15.023.20070::~~continuous~~~`,
        fs:  `cpe:2.3:a:ssh:tectia_client:6.0.7.28:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:17.009.20044::~~continuous~~~`,
        fs:  `cpe:2.3:a:school_administration_project:school_administration:7.x-1.7:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:17.009.20058::~~continuous~~~`,
        fs:  `cpe:2.3:a:emc:networker:8.1.1.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:17.012.20093::~~continuous~~~`,
        fs:  `cpe:2.3:a:ibm:maximo_industry_solutions:7.5.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:17.012.20095::~~continuous~~~`,
        fs:  `cpe:2.3:a:google_authenticator_login_project:ga_login:7.x-1.0:beta1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:17.012.20098::~~continuous~~~`,
        fs:  `cpe:2.3:a:symantec:client_security:1.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:18.009.20044::~~continuous~~~`,
        fs:  `cpe:2.3:h:juniper:mx240:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:18.009.20050::~~continuous~~~`,
        fs:  `cpe:2.3:a:adobe:acrobat_reader:6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:2015.006.30060::~~classic~~~`,
        fs:  `cpe:2.3:a:perl:perl:2.11.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:acrobat_reader_dc:2015.008.20082::~~continuous~~~`,
        fs:  `cpe:2.3:a:umn:mapserver:4.10.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:add_or_remove_creative_suite_3_design_premium:1.0`,
        fs:  `cpe:2.3:a:activehelper:activehelper_livehelp_live_chat:3.5.0:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:add_or_remove_creative_suite_3_web_premium:1.0`,
        fs:  `cpe:2.3:o:linux:linux_kernel:4.4.51:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:add_or_remove_creative_suite_3_web_standard:1.0`,
        fs:  `cpe:2.3:a:netweblogic:login_with_ajax:3.0b3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:-`,
        fs:  `cpe:2.3:a:moinejf:abcm2ps:2.10.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:1.0`,
        fs:  `cpe:2.3:a:microsoft:expression_web:2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:1.0.1`,
        fs:  `cpe:2.3:o:cisco:ios:12.0\(27\)sv2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:1.0.8.4990`,
        fs:  `cpe:2.3:a:citrix:licensing:6.8.9020:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:1.0.4990`,
        fs:  `cpe:2.3:a:bluecoat:webproxy:5.1:r1d:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:1.1`,
        fs:  `cpe:2.3:a:bitcoin:bitcoind:0.6.0:rc3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:1.1.0.5790`,
        fs:  `cpe:2.3:a:fail2ban:fail2ban:0.8.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:1.5`,
        fs:  `cpe:2.3:a:vmware:server:1.0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:1.5.0.7220`,
        fs:  `cpe:2.3:h:brocade:vyatta_vrouter:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:1.5.1`,
        fs:  `cpe:2.3:h:dlink:dcs-900:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:1.5.1.8210`,
        fs:  `cpe:2.3:a:ibm:tririga_application_platform:3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:1.5.2`,
        fs:  `cpe:2.3:a:nortel:sip_softphone:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:1.5.3`,
        fs:  `cpe:2.3:a:ait-pro:bulletproof-security:.47.4:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:1.5.3.9120`,
        fs:  `cpe:2.3:a:emc:digital_assets_manager:6.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:1.5.3.9130`,
        fs:  `cpe:2.3:a:veritas:netbackup_appliance:2.6.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:2.0.2`,
        fs:  `cpe:2.3:a:apache:groovy:2.1.0:beta_1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:2.0.2.12610`,
        fs:  `cpe:2.3:a:inria:ocaml:3.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:2.0.3`,
        fs:  `cpe:2.3:a:gisle_aas:libwww-perl:5.835:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:2.0.3.13070`,
        fs:  `cpe:2.3:a:ulli_horlacher:fex:20100208:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:2.0.4`,
        fs:  `cpe:2.3:a:welcart:welcart_plugin:0.9.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:2.5.0.16600`,
        fs:  `cpe:2.3:a:ait-pro:bulletproof-security:.46.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:2.5.1.17730`,
        fs:  `cpe:2.3:a:laurent_destailleur:awstats:5.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:2.6`,
        fs:  `cpe:2.3:a:buffalotech:bbr-4mg_firmware:1.30:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:2.6.0.19120`,
        fs:  `cpe:2.3:a:krzysztof_kowalczyk:sumatrapdf:1.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:2.6.0.19140`,
        fs:  `cpe:2.3:a:x:libxext:1.3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:2.7`,
        fs:  `cpe:2.3:a:isc:bind:9.9.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:2.7.0.1948`,
        fs:  `cpe:2.3:a:filemaker:filemaker_pro_advanced:12.0.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:2.7.0.1953`,
        fs:  `cpe:2.3:a:cisco:firewall_services_module_software:3.1\(4\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:2.7.0.19480`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.242:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:2.7.0.19530`,
        fs:  `cpe:2.3:a:bitcoin:wxbitcoin:0.4.1:rc6:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:2.7.1`,
        fs:  `cpe:2.3:o:novell:netware:4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:2.7.1.19610`,
        fs:  `cpe:2.3:a:x.org:xorg-server:1.12.2.901:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:3.0.0.408`,
        fs:  `cpe:2.3:a:whmcs:whmcompletesolution:4.2.0:beta_r3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:3.0.0.4080`,
        fs:  `cpe:2.3:a:libav:libav:11.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:3.1.0.485`,
        fs:  `cpe:2.3:a:oracle:mysql_connector\/j:1.1b:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:3.1.0.488`,
        fs:  `cpe:2.3:a:projeqtor:projeqtor:2.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:3.1.0.4880`,
        fs:  `cpe:2.3:a:abeel:simple_php_agenda:0.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:3.2.0.207`,
        fs:  `cpe:2.3:o:cisco:nx-os:11.1\(2h\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:3.2.0.2070`,
        fs:  `cpe:2.3:o:juniper:junos:15.1x53:d25:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:3.3.0.3670`,
        fs:  `cpe:2.3:a:modx:modx_revolution:2.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:3.4.0.2540`,
        fs:  `cpe:2.3:a:apache:poi:1.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:3.4.0.2710`,
        fs:  `cpe:2.3:a:otrs:otrs:1.2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:3.5.0.600`,
        fs:  `cpe:2.3:a:adobe:acrobat:8.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:3.5.0.880`,
        fs:  `cpe:2.3:a:oracle:jdk:1.5.0:update_36:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:3.5.0.890`,
        fs:  `cpe:2.3:a:hp:storageworks_secure_path_windows:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:3.5.0.1060`,
        fs:  `cpe:2.3:a:attachmate:verastream_process_designer:6.0:sp1_hf1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:3.6.0.597`,
        fs:  `cpe:2.3:h:axis:2420_video_server:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:3.6.0.6090`,
        fs:  `cpe:2.3:o:iij:seil_b1_firmware:3.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:3.7.0.1530`,
        fs:  `cpe:2.3:a:winzip:winzip_command_line_support_add-on:3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:3.7.0.1860`,
        fs:  `cpe:2.3:h:hp:officejet_pro_8500:cn539a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:3.7.0.2090`,
        fs:  `cpe:2.3:a:eric_m_ludlam:cedet:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:3.8.0.870`,
        fs:  `cpe:2.3:h:cisco:adaptive_security_device_manager:5.2.53:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:3.8.0.910`,
        fs:  `cpe:2.3:h:tropos:1410_mesh_router:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:3.9.0.1030`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.33.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:3.9.0.1060`,
        fs:  `cpe:2.3:a:php:pear:1.4.0a5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:3.9.0.1210`,
        fs:  `cpe:2.3:a:jenkins-ci:monitoring_plugin:1.41.0:*:*:*:*:cloudbees_jenkins:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:3.9.0.1380`,
        fs:  `cpe:2.3:a:simon_tatham:putty:0.66:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:4.0.0.1390`,
        fs:  `cpe:2.3:a:oracle:mysql:5.7.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:4.0.0.1628`,
        fs:  `cpe:2.3:a:hitachi:tiered_storage_manager:8.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:13.0.0.83`,
        fs:  `cpe:2.3:a:adobe:indesign:6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:13.0.0.111`,
        fs:  `cpe:2.3:a:cisco:cisco_policy_suite:7.0.4-att:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:14.0.0.110`,
        fs:  `cpe:2.3:a:ibm:iss_realsecure_desktop:7.0:ebm:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air:14.0.0.137`,
        fs:  `cpe:2.3:h:hp:0235a0g7:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air_sdk:3.0.0.4080`,
        fs:  `cpe:2.3:a:roundup-tracker:roundup:0.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air_sdk:3.1.0.488`,
        fs:  `cpe:2.3:a:ibm:tivoli_storage_manager:5.2.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air_sdk:3.2.0.2070`,
        fs:  `cpe:2.3:o:codeaurora:android-msm:3.10.40:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air_sdk:3.3.0.3650`,
        fs:  `cpe:2.3:a:cisco:unified_computing_system:1.4\(4a\)1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air_sdk:3.3.0.3690`,
        fs:  `cpe:2.3:a:apache:apr-util:1.2.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air_sdk:3.4.0.2540`,
        fs:  `cpe:2.3:h:hp:h3c_switch:s5800-32c-pwr:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air_sdk:3.4.0.2710`,
        fs:  `cpe:2.3:a:symantec:sms-smtp-tomcat:5.5.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air_sdk:3.5.0.600`,
        fs:  `cpe:2.3:a:hp:openview_performance_access:10.50.90:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air_sdk:3.5.0.880`,
        fs:  `cpe:2.3:a:sybase:adaptive_server_anywhere:6.0.3.2747:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air_sdk:3.5.0.890`,
        fs:  `cpe:2.3:a:microsoft:microsoft.aspnetcore.mvc.webapicompatshim:1.1.1:*:*:*:*:asp.net:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air_sdk:3.5.0.1060`,
        fs:  `cpe:2.3:a:merethis:centreon:1.4.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air_sdk:3.6.0.599`,
        fs:  `cpe:2.3:a:oracle:jai:1.1.2:01:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air_sdk:3.6.0.6090`,
        fs:  `cpe:2.3:a:mariadb:mariadb:5.5.54:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air_sdk:3.7.0.1530`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.35.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air_sdk:3.7.0.1860`,
        fs:  `cpe:2.3:a:gnu:automake:1.8.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air_sdk:3.7.0.2090`,
        fs:  `cpe:2.3:a:cmsmadesimple:cms_made_simple:1.11.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air_sdk:3.8.0.870`,
        fs:  `cpe:2.3:a:haxx:curl:7.13.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air_sdk:3.8.0.910`,
        fs:  `cpe:2.3:a:icu-project:international_components_for_unicode:3.4.3:*:*:*:*:java:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air_sdk:3.8.0.1430`,
        fs:  `cpe:2.3:a:theforeman:foreman:1.8.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air_sdk:3.9.0.1030`,
        fs:  `cpe:2.3:a:b2evolution:b2evolution:6.9.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air_sdk:3.9.0.1210`,
        fs:  `cpe:2.3:o:asus:dsl-n56u_firmware:3.0.0.4.188:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air_sdk:3.9.0.1380`,
        fs:  `cpe:2.3:a:hp:hp-ux_whitelisting:a.01.02:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air_sdk:4.0.0.1390`,
        fs:  `cpe:2.3:a:git-scm:git:1.0.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air_sdk:4.0.0.1628`,
        fs:  `cpe:2.3:a:xchat:xchat:1.3.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air_sdk:13.0.0.83`,
        fs:  `cpe:2.3:a:igreks:milkystep_light:0.94:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air_sdk:13.0.0.111`,
        fs:  `cpe:2.3:a:git_for_windows_project:git_for_windows:1.8.4.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air_sdk:14.0.0.110`,
        fs:  `cpe:2.3:a:cerberusftp:ftp_server:4.0.0.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_air_sdk:14.0.0.137`,
        fs:  `cpe:2.3:a:freeradius:freeradius:0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_application_manager:2.0.240.0`,
        fs:  `cpe:2.3:a:mathjs:math.js:3.14.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_content_server:-`,
        fs:  `cpe:2.3:a:digium:asterisk:c.1.0:beta7:business:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_content_server:3.0`,
        fs:  `cpe:2.3:a:apache:ignite:1.6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_php_ria_sdk:-`,
        fs:  `cpe:2.3:a:typo3:typo3:4.2.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_presenter:-`,
        fs:  `cpe:2.3:a:oracle:peoplesoft_enterprise:8.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_presenter:7.0`,
        fs:  `cpe:2.3:h:cisco:6400_nrp:12.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_reader:11.1.0::~~~android~~`,
        fs:  `cpe:2.3:o:cisco:adaptive_security_appliance_software:9.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_reader:11.1.3::~~~android~~`,
        fs:  `cpe:2.3:a:ibm:tivoli_monitoring:6.2.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_reader:11.2.0::~~~android~~`,
        fs:  `cpe:2.3:a:deliciousdays:cforms:10.5.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_update_manager_core:1.0.0.0`,
        fs:  `cpe:2.3:a:nordugrid:nordugrid-arc:0.4.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:adobe_update_manager_language_resource:1.0.0.0`,
        fs:  `cpe:2.3:a:bftpd_project:bftpd:1.0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:after_effects:4.0`,
        fs:  `cpe:2.3:a:digium:asterisk:1.8.0:beta4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:after_effects:4.1`,
        fs:  `cpe:2.3:a:nec:socks_5:1.0r5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:after_effects:4.1.1`,
        fs:  `cpe:2.3:a:ffmpeg:ffmpeg:2.4.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:after_effects:5.0`,
        fs:  `cpe:2.3:a:videolan:vlc_media_player:0.2.80:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:after_effects:5.5`,
        fs:  `cpe:2.3:o:fortinet:fortios:5.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:after_effects:5.5.1`,
        fs:  `cpe:2.3:a:cisco:network_admission_control_manager_and_server_system_software:4.1.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:after_effects:6.0`,
        fs:  `cpe:2.3:a:keepass:keepass:2.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:after_effects:6.5.1`,
        fs:  `cpe:2.3:a:reviewboard:review_board:1.7.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:after_effects:7.0`,
        fs:  `cpe:2.3:o:apple:mac_os:8.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:after_effects:7.0.1`,
        fs:  `cpe:2.3:o:google:chrome_os:25.0.1364.23:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:after_effects_cs3:8.0`,
        fs:  `cpe:2.3:a:freetype:freetype2:2.4.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:after_effects_cs3:8.0.2`,
        fs:  `cpe:2.3:o:apple:mac_os_x:10.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:after_effects_cs3_presets:8.0`,
        fs:  `cpe:2.3:a:hitachi:it_operations_director:03-00-08:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:after_effects_cs3_template_projects_%26_footage:8.0`,
        fs:  `cpe:2.3:a:shaarli_project:shaarli:0.6.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:after_effects_cs3_third_party_content:3.0`,
        fs:  `cpe:2.3:a:csp_mysql_user_manager_project:csp_mysql_user_manager:2.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:after_effects_cs4:9.0`,
        fs:  `cpe:2.3:a:lightdm_gtk\&#43;_greeter_project:lightdm_gtk\&#43;_greeter:1.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:after_effects_cs4:9.0.2`,
        fs:  `cpe:2.3:a:7_habits_personal_development_project:7_habits_personal_development:1:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:after_effects_cs4_presets:9.0`,
        fs:  `cpe:2.3:a:foolabs:xpdf:1.01:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:after_effects_cs4_template_projects_%26_footage:9.0`,
        fs:  `cpe:2.3:o:iij:seil_x86_fuji_firmware:2.40:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:after_effects_cs4_third_party_content:9.0`,
        fs:  `cpe:2.3:a:gopivotal:grails-resources:1.2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:air:15.0.0.293`,
        fs:  `cpe:2.3:a:ibm:db2_connect:9.7.0.7:*:*:*:enterprise:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:air:17.0.0.144`,
        fs:  `cpe:2.3:a:sympa:sympa:2.7b.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:air:17.0.0.172`,
        fs:  `cpe:2.3:a:git-scm:git:2.7.0:rc3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:air:18.0.0.180`,
        fs:  `cpe:2.3:a:tmobile:webconnect_manager:2.5.16.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:air:19.0.0.241`,
        fs:  `cpe:2.3:a:monster_menus_module_project:monster_menus:7.x-1.x:dev:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:air:19.0.0.241::~~~android~~`,
        fs:  `cpe:2.3:a:puppetlabs:puppet:0.25.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:air:20.0.0.204`,
        fs:  `cpe:2.3:a:roderich_schupp:par-packer_module:0.942:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:air:20.0.0.233`,
        fs:  `cpe:2.3:a:joomla:joomla\!:3.1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:air:20.0.0.260`,
        fs:  `cpe:2.3:a:libguestfs:libguestfs:1.0.61:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:air_sdk:15.0.0.302`,
        fs:  `cpe:2.3:a:symantec:netbackup_enterprise_server:6.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:air_sdk:17.0.0.144`,
        fs:  `cpe:2.3:o:microsoft:windows_2003_server:-:gold:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:air_sdk:17.0.0.172`,
        fs:  `cpe:2.3:a:apache:subversion:0.24.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:air_sdk:18.0.0.180`,
        fs:  `cpe:2.3:o:yamaha:rt140i:6.02.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:air_sdk:19.0.0.241`,
        fs:  `cpe:2.3:a:node_access_user_reference_project:nodeaccess_userreference_module:6.x-3.0:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:air_sdk:20.0.0.204`,
        fs:  `cpe:2.3:a:puppetlabs:marionette-collective:2.7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:air_sdk:20.0.0.260`,
        fs:  `cpe:2.3:a:unixodbc:unixodbc:2.2.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:air_sdk_%26_compiler:15.0.0.302`,
        fs:  `cpe:2.3:a:plugin-organizer_project:plugin-organizer:0.9:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:air_sdk_%26_compiler:17.0.0.144`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.260:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:air_sdk_%26_compiler:17.0.0.172`,
        fs:  `cpe:2.3:a:google:chrome:33.0.1750.85:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:air_sdk_%26_compiler:18.0.0.180`,
        fs:  `cpe:2.3:o:novell:unixware:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:air_sdk_%26_compiler:19.0.0.241`,
        fs:  `cpe:2.3:a:skype:skype:2.5.0.126:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:air_sdk_%26_compiler:20.0.0.204`,
        fs:  `cpe:2.3:a:search.cpan:libwww-perl:0.04:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:air_sdk_%26_compiler:20.0.0.233`,
        fs:  `cpe:2.3:a:craig_drummond:cantata:0.8.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:air_sdk_%26_compiler:20.0.0.260`,
        fs:  `cpe:2.3:h:3com:nbx_1000:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:air_sdk_%26_compiler:22.0.0.153`,
        fs:  `cpe:2.3:h:valcom:vip-821:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:analytics_appmeasurement_for_flash_library:4.0`,
        fs:  `cpe:2.3:a:openstack:python-keystoneclient:1.5.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:anchor_service:2.1`,
        fs:  `cpe:2.3:a:google:chrome:34.0.1847.103:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:anchor_service_cs3:1.0`,
        fs:  `cpe:2.3:a:category_tokens_project:category_tokens:6.x-1.0:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:anchor_service_cs4:2.0`,
        fs:  `cpe:2.3:a:linuxfoundation:foomatic-filters:4.0.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:anchor_service_cs4:2.0::~~~~x64~`,
        fs:  `cpe:2.3:a:gnu:libtasn1:4.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:animate:15.2.1.95`,
        fs:  `cpe:2.3:h:ibm:system_storage_ds5020_disk_controller:1814-20a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:asset_services:3.0`,
        fs:  `cpe:2.3:a:jenkins:script_security:1.18:*:*:*:*:jenkins:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:asset_services:4.0`,
        fs:  `cpe:2.3:a:zen-cart.jp:zen_cart:1.3.0.1:*:*:ja-jp:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:asset_services_cs3:3.0`,
        fs:  `cpe:2.3:a:buffalotech:bbr-4mg_firmware:1.42:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:asset_services_cs3:3.1.0`,
        fs:  `cpe:2.3:a:cisco:telepresence_system_software:1.9.6\(2\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:asset_services_cs4:4.0`,
        fs:  `cpe:2.3:o:netgear:raidiator:4.2.23:-:-:*:-:-:readynas:*`,
    },
     
    {
        uri: `cpe:/a:adobe:brackets:1.6`,
        fs:  `cpe:2.3:a:gnu:binutils:2.29:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:breeze_licensed_server:-`,
        fs:  `cpe:2.3:a:gnu:coreutils:5.94:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:breeze_licensed_server:5`,
        fs:  `cpe:2.3:a:djangoproject:django:1.8.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:breeze_licensed_server:5.1`,
        fs:  `cpe:2.3:o:cisco:ios_xe:3.11s.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:bridge:-`,
        fs:  `cpe:2.3:a:phusionpassenger:phusion_passenger:5.0.19:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:bridge:1.0.0`,
        fs:  `cpe:2.3:a:citrix:provisioning_services:5.6:sp2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:bridge:1.0.3`,
        fs:  `cpe:2.3:a:igreks:milkystep_professional_oem:1.83:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:bridge:1.0.4`,
        fs:  `cpe:2.3:a:livetecs:timeline:3.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:bridge:2.0`,
        fs:  `cpe:2.3:a:cisco:wireless_control_system_software:4.2.173.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:bridge:2.1`,
        fs:  `cpe:2.3:a:cdsincdesign:simple_dropbox_upload_form:1.5.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:bridge:3.0`,
        fs:  `cpe:2.3:o:d-link:dcs-5000l_firmware:1.02.02:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:bridge_cc:6.1`,
        fs:  `cpe:2.3:a:drupal:drupal:8.0.0:beta3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:bridge_cs3:2.0`,
        fs:  `cpe:2.3:a:webyog:sqlyog:6.53:-:community:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:bridge_cs3:2.1`,
        fs:  `cpe:2.3:a:djangoproject:django:1.3.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:bridge_cs3:2.1.1`,
        fs:  `cpe:2.3:a:google:chrome:10.0.648.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:bridge_cs4:3.0`,
        fs:  `cpe:2.3:a:trendmicro:interscan_web_security_suite:3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:bridge_cs5:4.0.2`,
        fs:  `cpe:2.3:a:wordpress:wordpress:2.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:bridge_cs5:4.0.3`,
        fs:  `cpe:2.3:a:cs-cart:cs-cart:2.0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:bridge_cs5:4.0.4`,
        fs:  `cpe:2.3:a:chamanet:chamacargo:7.0000:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:bridge_cs5:4.0.5`,
        fs:  `cpe:2.3:a:x7chat:x7_chat:2.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:bridge_start_meeting:1.0`,
        fs:  `cpe:2.3:a:google:chrome:32.0.1700.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:bridge_start_meeting:1.0.1`,
        fs:  `cpe:2.3:a:cloudbees:jenkins:1.346:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:bridgetalk_plugin_cs3:1.0`,
        fs:  `cpe:2.3:a:apache:tomcat:7.0.70:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:camera_raw:2.1`,
        fs:  `cpe:2.3:a:ibm:personal_communications:6.0.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:camera_raw:2.2`,
        fs:  `cpe:2.3:o:yamaha:rta50i:3.05.38:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:camera_raw:2.3`,
        fs:  `cpe:2.3:a:rocomotion:pplog_2:2.34:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:camera_raw:2.4`,
        fs:  `cpe:2.3:a:rocomotion:pplog:2.70:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:camera_raw:3.1`,
        fs:  `cpe:2.3:a:icu-project:international_components_for_unicode:52.1:*:*:*:*:c\/c\&#43;\&#43;:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:camera_raw:3.2`,
        fs:  `cpe:2.3:a:opensymphony:xwork:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:camera_raw:3.3`,
        fs:  `cpe:2.3:a:activestate:activeperl:5.14.1402:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:camera_raw:3.4`,
        fs:  `cpe:2.3:a:digium:asterisk:1.8.0:beta5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:camera_raw:3.5`,
        fs:  `cpe:2.3:a:google:android_api:16.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:camera_raw:3.6`,
        fs:  `cpe:2.3:a:octopus:octopus_deploy:3.0.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:camera_raw:3.7`,
        fs:  `cpe:2.3:a:phpmyadmin:phpmyadmin:1.2.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:camera_raw:4.0`,
        fs:  `cpe:2.3:o:netbsd:netbsd:6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:camera_raw:4.1`,
        fs:  `cpe:2.3:a:emery_berger:hoard:3.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:camera_raw:4.2`,
        fs:  `cpe:2.3:a:php:phorum:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:camera_raw:4.3.1`,
        fs:  `cpe:2.3:a:ibm:infosphere_biginsights:1.1.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:camera_raw:4.4.1`,
        fs:  `cpe:2.3:a:git-scm:git:2.12.0:rc0:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:camera_raw:4.5`,
        fs:  `cpe:2.3:a:octobercms:october_cms:1.0.83:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:camera_raw:4.6`,
        fs:  `cpe:2.3:a:ibm:domino:9.0.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:camera_raw:5.0`,
        fs:  `cpe:2.3:a:cisco:firesight_system_software:5.2.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:camera_raw:5.0::~~~~x64~`,
        fs:  `cpe:2.3:a:igniterealtime:smack:3.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:camera_raw:5.1`,
        fs:  `cpe:2.3:a:strongswan:strongswan:2.6.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:camera_raw:5.2`,
        fs:  `cpe:2.3:a:netscape:navigator:4.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:camera_raw:5.3`,
        fs:  `cpe:2.3:o:yamaha:rt105p:6.02.04:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:camera_raw:5.4`,
        fs:  `cpe:2.3:a:ajaydsouza:contextual_related_posts:1.5.1:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:camera_raw:5.5`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(28\)sb2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:captivate:-`,
        fs:  `cpe:2.3:a:gstreamer_project:gstreamer:0.10.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:captivate:2.0`,
        fs:  `cpe:2.3:a:in-mediakg:filterftp:2.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:captivate:3.0`,
        fs:  `cpe:2.3:a:barracudadrive:barracudadrive:6.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:captivate:4.0`,
        fs:  `cpe:2.3:o:yamaha:rt200i:4.00.33:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:captivate:5.0.0.596`,
        fs:  `cpe:2.3:h:hp:elitebook_820_g3:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:captivate_quiz_results_analyzer:1.0`,
        fs:  `cpe:2.3:a:simplejson_project:simplejson:3.1.2:*:*:*:*:python:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:captivate_reviewer:2.0`,
        fs:  `cpe:2.3:a:dotnetblogengine:blogengine.net:1.4.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:cmaps:1.0`,
        fs:  `cpe:2.3:o:cisco:ciscoworks_1105_wireless_lan_solution_engine:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:cmaps_cs4:2.0`,
        fs:  `cpe:2.3:a:ibm:tivoli_storage_manager:6.3.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:cmaps_cs4:2.0::~~~~x64~`,
        fs:  `cpe:2.3:a:tejimaya:openpne:3.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion:-`,
        fs:  `cpe:2.3:a:google:chrome:23.0.1271.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion:6.0`,
        fs:  `cpe:2.3:a:opennms:opennms:1.12.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion:6.1`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.4.6-8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion:7.0`,
        fs:  `cpe:2.3:a:opera:opera_browser:19.00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion:7.0.1`,
        fs:  `cpe:2.3:a:hp:icewall_file_manager:3.0:sp2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion:7.0.2`,
        fs:  `cpe:2.3:h:bluecoat:proxysg_sg210-25:-:-:full_proxy:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion:7.2:unknown:mx`,
        fs:  `cpe:2.3:h:huawei:s3300hi:v100r006:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion:8.0`,
        fs:  `cpe:2.3:a:dameware:mini_remote_control:6.4.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion:8.0.1`,
        fs:  `cpe:2.3:a:octopus:octopus_deploy:3.7.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion:9.0`,
        fs:  `cpe:2.3:a:google:chrome:19.0.1034.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion:9.0.1`,
        fs:  `cpe:2.3:a:ckeditor:fckeditor:6.x-2.2:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion:9.0.2`,
        fs:  `cpe:2.3:a:hp:nonstop_server_software:j06.09.03:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion:10.0`,
        fs:  `cpe:2.3:a:hp:linux_imaging_and_printing_project:3.10.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion:10.0:update_1`,
        fs:  `cpe:2.3:a:hp:systems_insight_manager:5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion:10.0:update_11`,
        fs:  `cpe:2.3:a:apache:archiva:1.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion:10.0:update_14`,
        fs:  `cpe:2.3:a:isc:bind:9.2.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion:10.0:update_15`,
        fs:  `cpe:2.3:a:webyog:sqlyog:5.24:-:enterprise:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion:10.0:update_2`,
        fs:  `cpe:2.3:a:cisco:unified_communications_manager:8.6\(2a\)su2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion:10.0:update_21`,
        fs:  `cpe:2.3:a:ibm:rational_quality_manager:3.0.1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion:10.0:update_3`,
        fs:  `cpe:2.3:a:granita:cloud_browser:2.2.1:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion:10.0:update_4`,
        fs:  `cpe:2.3:a:shibboleth:identity_provider:2.4.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion:10.0:update_8`,
        fs:  `cpe:2.3:o:cisco:catos:5.1\(1a\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion:11.0:update_10`,
        fs:  `cpe:2.3:a:glpi-project:glpi:0.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion:11.0:update_2`,
        fs:  `cpe:2.3:a:apache:apr-util:0.9.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion:11.0:update_4`,
        fs:  `cpe:2.3:o:cisco:ios:12.3\(7\)xi4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion_.net_integration_services:1.0.0.0`,
        fs:  `cpe:2.3:a:tigris:tortoisesvn:0.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion_.net_integration_services:9.0.0.0`,
        fs:  `cpe:2.3:h:huawei:usg5500:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion_8:8.0`,
        fs:  `cpe:2.3:a:atlassian:fisheye:2.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion_8:8.0.1`,
        fs:  `cpe:2.3:a:techsmith:snagit:9.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion_8_.net_integration_services:1.0`,
        fs:  `cpe:2.3:a:nerdico_project:nerdico:1.9:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion_builder:2.0.0.0`,
        fs:  `cpe:2.3:a:nathan_haug:webform:6.x-3.0:beta4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:coldfusion_solr_service:9.0.0.0`,
        fs:  `cpe:2.3:a:rubyonrails:ruby_on_rails:4.2.5:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:color_common_setcmyk:2.0`,
        fs:  `cpe:2.3:a:gnu:libtasn1:0.2.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:color_common_setrgb:2.0`,
        fs:  `cpe:2.3:a:google:chrome:17.0.963.53:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:color_common_settings:1.0`,
        fs:  `cpe:2.3:a:vmturbo:operations_manager:4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:color_common_settings:1.0.1`,
        fs:  `cpe:2.3:h:symantec:clientless_vpn_gateway_4400:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:color_extra_settings:1.0`,
        fs:  `cpe:2.3:a:ibm:rational_clearcase:2.1.1063:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:color_extra_settings_cs4:2.0`,
        fs:  `cpe:2.3:a:rocomotion:pplog_2:3.005:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:color_photoshop_specific:1.0`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.27.27:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:color_photoshop_specific_cs4:2.0`,
        fs:  `cpe:2.3:a:hp:web_registration:140.0.213.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:color_recommended_settings:1.0`,
        fs:  `cpe:2.3:a:google_site_search_project:google_site_search_module:7.x-1.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:color_recommended_settings_cs4:2.0`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.6:p2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:color_video_profiles_cs_cs4:2.0`,
        fs:  `cpe:2.3:o:raritan:px:1.2.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:common_file_installer:1.0.0`,
        fs:  `cpe:2.3:a:videolan:vlc_media_player:0.2.82:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:common_file_installer:1.0.2`,
        fs:  `cpe:2.3:o:cisco:unified_ip_conference_station_7936:3.3\(12\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:community_help:3.0.0`,
        fs:  `cpe:2.3:a:b-e-soft:artweaver:1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:community_help:3.0.0.400`,
        fs:  `cpe:2.3:a:matthew_booth:virt-v2v:0.8.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:community_help:3.4.973`,
        fs:  `cpe:2.3:o:nuuo:nvrmini_2:1.7.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:community_help:3.4.980`,
        fs:  `cpe:2.3:h:cisco:ace_4700_series_application_control_engine_appliance_a4:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:community_help:3.5.23`,
        fs:  `cpe:2.3:a:cisco:unified_communications_manager:7.1\(2b\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:1.0.0.1`,
        fs:  `cpe:2.3:o:cisco:ios:12.0sl:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:6.0`,
        fs:  `cpe:2.3:a:bufferapp:digg_digg:5.3.0:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:6.1`,
        fs:  `cpe:2.3:a:connect2id:nimbus_jose\&#43;jwt:4.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:6.2`,
        fs:  `cpe:2.3:a:microsoft:iis:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:6.3`,
        fs:  `cpe:2.3:o:yahama:rt140p:6.02.07:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:7.0`,
        fs:  `cpe:2.3:a:mozilla:seamonkey:1.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:7.2`,
        fs:  `cpe:2.3:o:sun:solaris:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:7.3`,
        fs:  `cpe:2.3:a:zabbix:zabbix:1.1:beta3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:7.5`,
        fs:  `cpe:2.3:a:gisle_aas:libwww-perl:0.02:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:8.0`,
        fs:  `cpe:2.3:a:danny_morris:lazy_seo:1.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:8.1`,
        fs:  `cpe:2.3:a:glpi-project:glpi:0.51a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:8.2`,
        fs:  `cpe:2.3:a:apache:hbase:0.98.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:8.2.1`,
        fs:  `cpe:2.3:a:ibm:financial_transaction_manager:3.0.0.8:*:*:*:*:ach_services:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:8.2.2`,
        fs:  `cpe:2.3:a:gopivotal:grails:2.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:9.0`,
        fs:  `cpe:2.3:a:ckeditor:fckeditor:6.x-2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:9.0.2`,
        fs:  `cpe:2.3:a:boonex:dolphin:7.1.0:b1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:9.0.3`,
        fs:  `cpe:2.3:h:cisco:wrvs4400n_wireless-n_gigabit_security_router_-_vpn_v2.0:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:9.1`,
        fs:  `cpe:2.3:o:cisco:clean_access:3.3.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:9.1.2`,
        fs:  `cpe:2.3:a:nodejs:node.js:0.10.24:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:9.2`,
        fs:  `cpe:2.3:a:mysql:mysql:5.0.87:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:9.3`,
        fs:  `cpe:2.3:a:perl:perl:2.6.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:9.4.1`,
        fs:  `cpe:2.3:a:thekelleys:dnsmasq:1.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:9.4.2`,
        fs:  `cpe:2.3:a:microsoft:microsoft.aspnetcore.mvc.localization:1.1.0:*:*:*:*:asp.net:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:9.5`,
        fs:  `cpe:2.3:a:jenkins:git:2.0.0:alpha-1:*:*:*:jenkins:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:9.5.2`,
        fs:  `cpe:2.3:a:libpng:libpng:0.81:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:9.5.3`,
        fs:  `cpe:2.3:a:cybozu:office:9.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:9.5.4`,
        fs:  `cpe:2.3:a:tiki:tikiwiki_cms\/groupware:1.9.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:9.5.5`,
        fs:  `cpe:2.3:a:sun:sdk:1.4.2_6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:9.5.6`,
        fs:  `cpe:2.3:a:check_mk_project:check_mk:1.2.8:b6:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:9.5.7`,
        fs:  `cpe:2.3:a:itechscripts:proman_xpress:5.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:9.6`,
        fs:  `cpe:2.3:a:buddypress:buddypress_plugin:1.5.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:9.6.1`,
        fs:  `cpe:2.3:a:easy_social_project:easy_social:7.x-2.1:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:9.6.2`,
        fs:  `cpe:2.3:a:gentoo:linux_webapp-config:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect:9.7`,
        fs:  `cpe:2.3:h:netgear:prosafe_gs752tps:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect_addin_checker:9.3.139.0`,
        fs:  `cpe:2.3:a:microsoft:backoffice_resource_kit:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect_enterprise_server:-`,
        fs:  `cpe:2.3:a:ati:catalyst_control_center_graphics_full_new:2009.113.2222.40119:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect_enterprise_server:6`,
        fs:  `cpe:2.3:o:cisco:wireless_lan_controller_software:4.0.155.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect_outlook_add-in:3.0`,
        fs:  `cpe:2.3:a:augeas:augeas:0.7.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:connect_pro_add-in_for_microsoft_outlook:1.0.0`,
        fs:  `cpe:2.3:a:drupal:drupal:8.0.0:alpha5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:contribute:-`,
        fs:  `cpe:2.3:a:ibm:tivoli_directory_server:6.3.0.25:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:contribute:2`,
        fs:  `cpe:2.3:o:linux:linux_kernel:4.6.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:contribute:3`,
        fs:  `cpe:2.3:a:openproject:openproject:7.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:contribute:4.1`,
        fs:  `cpe:2.3:a:microsoft:msn_chat_control:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:contribute_cs3:4.1`,
        fs:  `cpe:2.3:a:boehm-demers-weiser:garbage_collector:4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:contribute_cs4:5.0`,
        fs:  `cpe:2.3:a:ajenti:ajenti:1.2.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:creative_cloud:3.6.0.248`,
        fs:  `cpe:2.3:a:download_manager_project:download_manager:2.2.1:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:creative_cloud:3.7.0.272`,
        fs:  `cpe:2.3:a:roundup-tracker:roundup:0.7.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:creative_suite:-`,
        fs:  `cpe:2.3:a:gnome:glib:2.28.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:creative_suite:1.0`,
        fs:  `cpe:2.3:a:ibm:urbancode_deploy:6.2.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:creative_suite:1.3`,
        fs:  `cpe:2.3:a:cmscout:cmscout:2.05:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:creative_suite:2.0`,
        fs:  `cpe:2.3:a:iproute2_project:iproute2:3.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:creative_suite:3.0`,
        fs:  `cpe:2.3:o:cisco:vpn_3015_concentrator:4.7.2.a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:creative_suite_3_design_premium:-`,
        fs:  `cpe:2.3:a:simon_tatham:putty:0.53b:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:creative_suite_3_design_premium:1.0`,
        fs:  `cpe:2.3:a:apache:activemq:5.13.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:creative_suite_3_design_premium:1.0.0`,
        fs:  `cpe:2.3:a:splunk:splunk:6.3.0:*:*:*:enterprise:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:creative_suite_3_design_standard:-`,
        fs:  `cpe:2.3:a:openstack:horizon:folsom-3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:creative_suite_3_master_collection:-`,
        fs:  `cpe:2.3:a:rocomotion:pplog_2:2.26:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:creative_suite_3_production_premium:-`,
        fs:  `cpe:2.3:a:andy_armstrong:cgi-simple:1.107:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:creative_suite_3_web_premium:-`,
        fs:  `cpe:2.3:a:gnu:cvs:1.12.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:creative_suite_3_web_premium:1.0`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.518:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:creative_suite_3_web_standard:-`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.7:p113:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:creative_suite_3_web_standard:1.0`,
        fs:  `cpe:2.3:a:cisco:unified_callmanager:3.3\(2\)spc:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:creative_suite_4:1.0.0`,
        fs:  `cpe:2.3:a:skyphe:file-gallery:1.7.4:-:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:creative_suite_4_design_premium:-`,
        fs:  `cpe:2.3:a:dbd-mysql_project:dbd-mysql:4.005:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:creative_suite_4_design_premium:4.0`,
        fs:  `cpe:2.3:o:cisco:sn_5428_storage_router:3.2.2-k9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:creative_suite_4_master_collection:-`,
        fs:  `cpe:2.3:a:bueltge:adminimize:1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:creative_suite_4_production_premium:-`,
        fs:  `cpe:2.3:a:pocoo:jinja2:2.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:creative_suite_4_web_premium:-`,
        fs:  `cpe:2.3:a:hp:unload:4.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:creative_suite_4_web_premium:4.0`,
        fs:  `cpe:2.3:h:huawei:s6300ei:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:creative_suite_4_web_standard:4.0`,
        fs:  `cpe:2.3:a:pocketmags:classic_racer:\@7f0801aa:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:creative_suite_5_design_standard:5.0`,
        fs:  `cpe:2.3:a:horde:gollem:1.0:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:creative_suite_5_master_collection:5.0`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.424.3:*:*:*:lts:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:creative_suite_5_web_premium:5.0`,
        fs:  `cpe:2.3:a:debian:duplicity:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:csi:1.0`,
        fs:  `cpe:2.3:a:actividentity:activclient_cac:7.0:*:*:*:*:*:x64:*`,
    },
     
    {
        uri: `cpe:/a:adobe:csi:1.0::~~~~x64~`,
        fs:  `cpe:2.3:a:preprojects:business_cards_designer:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:csi_cs4:1.0`,
        fs:  `cpe:2.3:a:sonatype:nexus:2.7.0:*:*:*:professional:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:csi_cs4:1.0.1`,
        fs:  `cpe:2.3:a:f5:firepass:7.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:customization_wizard:8.0.0`,
        fs:  `cpe:2.3:a:ati:catalyst_control_center_graphics_previews_common:2010.825.2146.37182:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:customization_wizard:9.0.0`,
        fs:  `cpe:2.3:a:apache:apr-util:1.3.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:customization_wizard_x:10.0.0`,
        fs:  `cpe:2.3:a:hp:insight_management_agents:8.22.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:device_central_cs3:-`,
        fs:  `cpe:2.3:a:apple:safari:3.0.0:-:mac:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:device_central_cs3:1.0`,
        fs:  `cpe:2.3:a:sensiolabs:symfony:2.4.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:device_central_cs4:-`,
        fs:  `cpe:2.3:a:roderich_schupp:par-packer_module:0.71:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:device_central_cs4:2.0`,
        fs:  `cpe:2.3:a:adobe:soundbooth_cs3:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:digital_editions:-`,
        fs:  `cpe:2.3:a:ssh:ssh:1.2.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:digital_editions:2.0.0`,
        fs:  `cpe:2.3:a:barracudadrive:barracudadrive:4.4.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:digital_editions:2.0.1`,
        fs:  `cpe:2.3:h:hp:h3c_switch:s5100-16p-ei_ls-5100-16p-ei-ovs-h3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:digital_editions:4.0.0`,
        fs:  `cpe:2.3:a:microsoft:outlook:2003:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:digital_editions:4.0.1`,
        fs:  `cpe:2.3:a:gnome:gnome-shell:3.7.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:digital_editions:4.0.2`,
        fs:  `cpe:2.3:a:x.org:libxrender:0.9.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:digital_editions:4.0.3`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:7.0.3-8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:digital_editions:4.5.0`,
        fs:  `cpe:2.3:a:connect2id:nimbus_jose\&#43;jwt:3.9.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:digital_editions:4.5.1`,
        fs:  `cpe:2.3:a:usebb:usebb:0.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:digital_editions:4.5.2`,
        fs:  `cpe:2.3:h:siemens:speedstream_wireless_router:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:digital_editions:4.5.3`,
        fs:  `cpe:2.3:a:rubyonrails:ruby_on_rails:0.11.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:digital_editions:4.5.4`,
        fs:  `cpe:2.3:h:siemens:ruggedcom_win5200:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:digital_editions:4.5.5`,
        fs:  `cpe:2.3:a:devsaran:company:7.x-1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:digital_editions:4.5.6`,
        fs:  `cpe:2.3:a:netsweeper:netsweeper:3.1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:digital_editions:4.5.7`,
        fs:  `cpe:2.3:a:zabbix:zabbix:2.0.9:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:director:-`,
        fs:  `cpe:2.3:a:novell:imanager:2.7:sp5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:dispatcher:4.1.4`,
        fs:  `cpe:2.3:a:joomla:joomla\!:3.3.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:dng_converter:9.7`,
        fs:  `cpe:2.3:a:mlmmj:mlmmj:0.7.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:document_server:-`,
        fs:  `cpe:2.3:a:naver:nhn_japan_naver_line:2.5:-:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:document_server:5.0`,
        fs:  `cpe:2.3:a:microsoft:microsoft.aspnetcore.mvc.webapicompatshim:1.1.0:*:*:*:*:asp.net:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:document_server:6.0`,
        fs:  `cpe:2.3:a:wordpress:wordpress:2.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:download_assistant:1.0.2`,
        fs:  `cpe:2.3:o:cisco:ios_xe:3.3sg.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:download_assistant:1.0.5`,
        fs:  `cpe:2.3:a:cmsmadesimple:cms_made_simple:2.0:beta4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:download_manager:-`,
        fs:  `cpe:2.3:h:avaya:cajun_p550:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:download_manager:1.2`,
        fs:  `cpe:2.3:a:naist.jp:chasen:2.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:download_manager:1.6.2.41`,
        fs:  `cpe:2.3:a:freedesktop:dbus-glib:0.82:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:download_manager:1.6.2.44`,
        fs:  `cpe:2.3:a:ibm:rational_test_virtualization_server:8.0.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:download_manager:1.6.2.48`,
        fs:  `cpe:2.3:a:google:chrome:25.0.1364.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:download_manager:1.6.2.49`,
        fs:  `cpe:2.3:a:mailpoet:mailpoet_newsletters:2.1.2:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:download_manager:1.6.2.60`,
        fs:  `cpe:2.3:a:webglimpse:webglimpse:2.12.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:download_manager:1.6.2.63`,
        fs:  `cpe:2.3:a:google:chrome:32.0.1700.103:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:download_manager:1.6.2.87`,
        fs:  `cpe:2.3:o:cisco:ios:12.1\(8b\)e9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:download_manager:1.6.2.90`,
        fs:  `cpe:2.3:a:cybozu:garoon:4.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:download_manager:1.6.2.91`,
        fs:  `cpe:2.3:a:wouter_verhelst:nbd:2.9.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:download_manager:1.6.2.97`,
        fs:  `cpe:2.3:a:apache:camel:1.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:download_manager:1.6.2.99`,
        fs:  `cpe:2.3:a:hp:pki_activex_control:1.1.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:download_manager:1.6.2.100`,
        fs:  `cpe:2.3:a:zabbix:zabbix:2.2.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:download_manager:1.6.2.102`,
        fs:  `cpe:2.3:a:cisco:anyconnect_secure_mobility_client:3.0.1047:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:download_manager:1.6.2.103`,
        fs:  `cpe:2.3:a:avactis:avactis_shopping_cart:1.8.2:-:free:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:download_manager:2.0`,
        fs:  `cpe:2.3:a:mahara:mahara:15.04.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:download_manager:2.1`,
        fs:  `cpe:2.3:a:tecnick:tcexam:11.0.011:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:dreamweaver:-`,
        fs:  `cpe:2.3:a:digium:asterisk:13.2.0:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:dreamweaver:1.0.0`,
        fs:  `cpe:2.3:a:icu-project:international_components_for_unicode:4.0.1:*:*:*:*:java:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:dreamweaver:6.1`,
        fs:  `cpe:2.3:a:roderich_schupp:par-packer_module:0.85:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:dreamweaver:7.0`,
        fs:  `cpe:2.3:o:ibm:power_5_system_firmware:sf240_403_382:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:dreamweaver:8.0`,
        fs:  `cpe:2.3:a:dlo:simple_anti_bot_registration_engine_plugin:0.9.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:dreamweaver:9.0`,
        fs:  `cpe:2.3:a:hi-tech_software:c_compiler_for_the_pic10\/12\/16_mcus:9.81:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:dreamweaver:10.0`,
        fs:  `cpe:2.3:a:codecguide:real_alternative:2.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:dreamweaver:11.0`,
        fs:  `cpe:2.3:a:intervideo:windvd:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:dreamweaver_cs3:-`,
        fs:  `cpe:2.3:a:apache:tomcat:6.0.7:alpha:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:dreamweaver_cs4:-`,
        fs:  `cpe:2.3:a:chat_room_project:chat_room:7.x-2.1:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:dreamweaver_cs5.5:11.5`,
        fs:  `cpe:2.3:a:videolan:vlc_media_player:1.1.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:dreamweaver_mx:6.0`,
        fs:  `cpe:2.3:a:webkitgtk:webkitgtk\&#43;:2.4.2a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:dreamweaver_mx:6.1`,
        fs:  `cpe:2.3:a:hitachi:ucosminexus_application_server:06-70-\/f:-:standard:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:dreamweaver_mx:7.0`,
        fs:  `cpe:2.3:a:mediawiki:mediawiki:1.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:dreamweaver_mx:7.0.1`,
        fs:  `cpe:2.3:a:oracle:database_server:7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:drive_cs4:-`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:drive_cs4:1.0`,
        fs:  `cpe:2.3:a:ulli_horlacher:fex:20110807:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:drive_cs4:1.0::~~~~x64~`,
        fs:  `cpe:2.3:a:modsecurity:modsecurity:2.5.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:dynamiclink_support:1.0`,
        fs:  `cpe:2.3:a:git-scm:git:1.8.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:elicensing:-`,
        fs:  `cpe:2.3:a:transmissionbt:transmission:1.52:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:encore_cs3:-`,
        fs:  `cpe:2.3:a:ibm:tivoli_storage_manager_client:5.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:encore_cs4:-`,
        fs:  `cpe:2.3:a:ixiacom:ixreporter:4.0.20.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:experience_manager:5.6.1`,
        fs:  `cpe:2.3:a:piwik:piwik:0.4.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:experience_manager:6.0.0`,
        fs:  `cpe:2.3:a:haxx:libcurl:7.42.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:experience_manager:6.1.0`,
        fs:  `cpe:2.3:a:hp:hpssupply:2.1.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:extendscript_toolkit:1.0.3`,
        fs:  `cpe:2.3:a:skype:skype:2.0.0.81:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:extendscript_toolkit:2.0`,
        fs:  `cpe:2.3:a:webyog:sqlyog:6.55:-:community:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:extendscript_toolkit:2.0.2`,
        fs:  `cpe:2.3:a:mozilla:firefox:3.5.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:extendscript_toolkit_cs4:3.0.0`,
        fs:  `cpe:2.3:a:gitlab:gitlab:8.6.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:extension_manager:1.5`,
        fs:  `cpe:2.3:a:lucent:security_management_server:9.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:extension_manager:1.6`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.32:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:extension_manager:1.7`,
        fs:  `cpe:2.3:a:mozilla:firefox:6.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:extension_manager:1.8`,
        fs:  `cpe:2.3:a:piwigo:piwigo:2.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:extension_manager_cs3:1.8`,
        fs:  `cpe:2.3:a:fedoraproject:sssd:1.5.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:extension_manager_cs4:2.0`,
        fs:  `cpe:2.3:a:ucdok:tomato:0.0.6:*:*:*:*:node.js:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:extension_manager_cs4:2.1`,
        fs:  `cpe:2.3:a:janrain:rpx:6.x-2.1:beta3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:fireworks:9.0`,
        fs:  `cpe:2.3:a:dan_kogai:encode_module:1.68:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:fireworks_cs3:9.0`,
        fs:  `cpe:2.3:a:symantec:veritas_storage_foundation_for_windows:5.0:rollup_patch_2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:fireworks_cs4:10.0`,
        fs:  `cpe:2.3:a:apport_project:apport:2.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:fireworks_cs4:10.0.3`,
        fs:  `cpe:2.3:a:oracle:mysql:5.5.28:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:fireworks_cs5:11.0`,
        fs:  `cpe:2.3:a:avaya:ip_soft_phone:6.01.85:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:fireworks_mx:6.0`,
        fs:  `cpe:2.3:a:eagletron:trackercam:5.3.0.57:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:fireworks_mx:7.0`,
        fs:  `cpe:2.3:a:botcha_spam_prevention_project:botcha:7.x-2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:fireworks_mx:7.0.2`,
        fs:  `cpe:2.3:a:gpeasy:gpeasy_cms:1.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash:1.0.0`,
        fs:  `cpe:2.3:a:bigtreecms:bigtree_cms:4.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash:8.0.22`,
        fs:  `cpe:2.3:a:mozilla:firefox:1.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_cs3:9.0`,
        fs:  `cpe:2.3:a:xymon:xymon:4.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_cs4:10.0`,
        fs:  `cpe:2.3:a:barebones:textwrangler:3.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_cs4_extension_flash_lite_sti:3.0`,
        fs:  `cpe:2.3:a:galleryproject:gallery:3.0.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_cs4_extension_flash_lite_sti:10.0`,
        fs:  `cpe:2.3:a:silverstripe:silverstripe:2.4.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_cs4_sti:10.0`,
        fs:  `cpe:2.3:a:augeas:augeas:0.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_media_server:3.0`,
        fs:  `cpe:2.3:a:connect2id:nimbus_jose\&#43;jwt:4.26:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_media_server:3.0.1`,
        fs:  `cpe:2.3:a:buffalotech:bbr-4mg_firmware:1.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_media_server:3.0.2`,
        fs:  `cpe:2.3:a:plugin-organizer_project:plugin-organizer:3.0.10:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_media_server:3.0.3`,
        fs:  `cpe:2.3:a:condusiv:diskeeper_2011:15.0.954.64:*:*:*:server:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_media_server:3.0.4`,
        fs:  `cpe:2.3:a:ffmpeg:ffmpeg:2.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_media_server:3.0.5`,
        fs:  `cpe:2.3:a:projeqtor:projeqtor:1.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_media_server:3.0.6`,
        fs:  `cpe:2.3:a:jenkins:git:0.2.0:*:*:*:*:jenkins:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_media_server:3.0.7`,
        fs:  `cpe:2.3:h:philips:xper_flex_cardio:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_media_server:3.5`,
        fs:  `cpe:2.3:a:nullsoft:winamp:5.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_media_server:3.5.1`,
        fs:  `cpe:2.3:a:rom_walton:boinc:6.13.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_media_server:3.5.2`,
        fs:  `cpe:2.3:a:znc:znc-msvc:0.095:-:-:*:-:windows:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_media_server:3.5.3`,
        fs:  `cpe:2.3:a:adbnewssender_project:adbnewssender:1.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_media_server:3.5.4`,
        fs:  `cpe:2.3:a:runtastic:runtastic_mountain_bike:2.0.1:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_media_server:3.5.5`,
        fs:  `cpe:2.3:a:cisco:telepresence_tc_software:7.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_media_server:4.0`,
        fs:  `cpe:2.3:a:git-scm:git:1.5.6:rc0:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_media_server:4.0.1`,
        fs:  `cpe:2.3:a:apache:log4j:2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_mx:6.0`,
        fs:  `cpe:2.3:a:worldwinner:worldwinner_games:1.9.0.23:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_mx:7.0`,
        fs:  `cpe:2.3:o:linux:kernel:2.6.17.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_mx:7.2`,
        fs:  `cpe:2.3:a:haxx:curl:6.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_playe_for_linux:9.0.115.0`,
        fs:  `cpe:2.3:a:sixapart:movable_type:4.15:beta4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:-`,
        fs:  `cpe:2.3:a:apache:groovy:1.9.0:beta_4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:2`,
        fs:  `cpe:2.3:h:buffalotech:wer-am54g54:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:3`,
        fs:  `cpe:2.3:a:google:chrome:33.0.1750.126:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:4`,
        fs:  `cpe:2.3:a:phpmyadmin:phpmyadmin:3.3.8.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:5`,
        fs:  `cpe:2.3:a:apptive:horsepower:2.10.11:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:6`,
        fs:  `cpe:2.3:a:ssh:tectia_server:4.4.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:6.0.21.0`,
        fs:  `cpe:2.3:h:rockwellautomation:compactlogix_1769-l30er-nse:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:6.0.79`,
        fs:  `cpe:2.3:a:wireshark:wireshark:1.2.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:7`,
        fs:  `cpe:2.3:a:systemtap:systemtap:1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:7.0`,
        fs:  `cpe:2.3:a:henri_wahl:nagstamon:0.9.7.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:7.0.1`,
        fs:  `cpe:2.3:a:symantec:netbackup_client:7.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:7.0.14.0`,
        fs:  `cpe:2.3:a:hp:icewall_file_manager:3.0:sp4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:7.0.19.0`,
        fs:  `cpe:2.3:a:google:chrome:11.0.662.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:7.0.24.0`,
        fs:  `cpe:2.3:a:adobe:captivate:3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:7.0.25`,
        fs:  `cpe:2.3:a:abbyy:finereader:6.0:-:corporate:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:7.0.53.0`,
        fs:  `cpe:2.3:a:wpeasycart:wp_easycart:3.0.0:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:7.0.60.0`,
        fs:  `cpe:2.3:a:elgg:elgg:1.8.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:7.0.61.0`,
        fs:  `cpe:2.3:a:ibm:tivoli_federated_identity_manager_business_gateway:6.2.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:7.0.63`,
        fs:  `cpe:2.3:a:ibm:java:7.0.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:7.0.66.0`,
        fs:  `cpe:2.3:a:namazu:namazu:2.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:7.0.67.0`,
        fs:  `cpe:2.3:o:microsoft:windows_xp:sp3:unknown:english:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:7.0.68.0`,
        fs:  `cpe:2.3:a:linuxfoundation:cups-filters:1.0.71:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:7.0.69.0`,
        fs:  `cpe:2.3:a:mysql:mysql:3.23.46:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:7.0.70.0`,
        fs:  `cpe:2.3:a:serve-static_project:serve-static:1.7.1:*:*:*:*:node.js:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:7.0.73.0`,
        fs:  `cpe:2.3:a:google:chrome:32.0.1656.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:7.1`,
        fs:  `cpe:2.3:a:sun:jre:1.4.2:update4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:7.1.1`,
        fs:  `cpe:2.3:a:ebay-kleinanzeigen:ebay_kleinanzeigen_for_germany:5.0.2:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:7.2`,
        fs:  `cpe:2.3:a:orangehrm:orangehrm:2.6.11.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:8`,
        fs:  `cpe:2.3:a:ipswitch:whatsup_gold:8.01:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:8.0`,
        fs:  `cpe:2.3:o:codeaurora:android-msm:3.14.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:8.0.22.0`,
        fs:  `cpe:2.3:a:dm_computer_solutions:ultraedit:14.20.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:8.0.24.0`,
        fs:  `cpe:2.3:a:palemoon:pale_moon:12.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:8.0.33.0`,
        fs:  `cpe:2.3:a:thoughtbot:cocaine:0.3.0:-:*:*:*:ruby:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:8.0.34.0`,
        fs:  `cpe:2.3:a:digium:asterisk:1.8.15.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:8.0.35.0`,
        fs:  `cpe:2.3:a:moinejf:abcm2ps:3.7.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:8.0.39.0`,
        fs:  `cpe:2.3:a:ibm:dashdb_local:1.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:8.0.42.0`,
        fs:  `cpe:2.3:a:barracudadrive:barracudadrive:6.7.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9`,
        fs:  `cpe:2.3:a:haxx:curl:7.18.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.0`,
        fs:  `cpe:2.3:a:tapatalk:tapatalk:4.7.1:*:*:*:*:vbulletin:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.0.8.0`,
        fs:  `cpe:2.3:a:google:chrome:31.0.1650.35:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.0.9.0`,
        fs:  `cpe:2.3:a:webform_project:webform:7.x-3.20:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.0.16`,
        fs:  `cpe:2.3:a:gitlab:gitlab:8.9.1:*:*:*:enterprise:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.0.18d60`,
        fs:  `cpe:2.3:a:tuttophp:happy_chat:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.0.20`,
        fs:  `cpe:2.3:a:grapecity:data_dynamics_reports:0.5.142.0:beta2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.0.20.0`,
        fs:  `cpe:2.3:a:syntevo:smartsvn:6.0:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.0.28`,
        fs:  `cpe:2.3:a:ibm:db2:9.7.0.4:*:*:*:advanced_workgroup:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.0.28.0`,
        fs:  `cpe:2.3:a:libtiff:libtiff:3.5.7:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.0.31`,
        fs:  `cpe:2.3:a:gnupg:libgcrypt:1.6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.0.31.0`,
        fs:  `cpe:2.3:a:telos:amhs_data:11.3.0.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.0.45.0`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.484:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.0.47.0`,
        fs:  `cpe:2.3:h:dahuasecurity:dvr5416:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.0.48.0`,
        fs:  `cpe:2.3:h:hp:procurve_router:jf285a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.0.112.0`,
        fs:  `cpe:2.3:a:xcloner:xcloner:3.1.2:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.0.114.0`,
        fs:  `cpe:2.3:a:mozilla:thunderbird:1.0.5:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.0.115.0`,
        fs:  `cpe:2.3:a:adobe:indesign_cs4_application_feature_set_files:6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.0.124.0`,
        fs:  `cpe:2.3:a:oracle:http_server:9.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.0.125.0`,
        fs:  `cpe:2.3:a:hp:nonstop_seeview_server_gateway:d48.00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.0.151.0`,
        fs:  `cpe:2.3:a:cisco:anyconnect_secure_mobility_client:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.0.152.0`,
        fs:  `cpe:2.3:a:openttd:openttd:0.4.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.0.155.0`,
        fs:  `cpe:2.3:a:debian:reportbug:6.4.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.0.159.0`,
        fs:  `cpe:2.3:a:git_for_windows_project:git_for_windows:1.8.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.0.246.0`,
        fs:  `cpe:2.3:a:wp-plugins:wp-print:2.02:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.0.260.0`,
        fs:  `cpe:2.3:a:microsoft:ie:6.00.2900.2180:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.0.262.0`,
        fs:  `cpe:2.3:h:cisco:identity_services_engine:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.0.277.0`,
        fs:  `cpe:2.3:a:david_king:vino:2.22:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.0.280`,
        fs:  `cpe:2.3:a:adobe:sing_cs4:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.0.283.0`,
        fs:  `cpe:2.3:a:apple:iphoto:6.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:9.125.0`,
        fs:  `cpe:2.3:a:apple:quicktime:7.7.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10`,
        fs:  `cpe:2.3:a:ibm:rational_directory_server:5.1.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.0.0.584`,
        fs:  `cpe:2.3:a:twitter_pull_project:twitter_pull:6.x-1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.0.2.54`,
        fs:  `cpe:2.3:a:apache:jackrabbit:2.2.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.0.12.10`,
        fs:  `cpe:2.3:a:webkitgtk:webkitgtk\&#43;:2.11.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.0.12.36`,
        fs:  `cpe:2.3:a:apache:tomcat:5.5.30:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.0.15.3`,
        fs:  `cpe:2.3:a:gnome:librsvg:2.26.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.0.22.87`,
        fs:  `cpe:2.3:a:blackberry:blackberry_enterprise_service:10.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.0.32.18`,
        fs:  `cpe:2.3:a:tejimaya:openpne:3.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.0.42.34`,
        fs:  `cpe:2.3:a:phpmyfaq:phpmyfaq:2.6.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.0.45.2`,
        fs:  `cpe:2.3:o:hp:procurve_switch_software:r.11.19:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.1`,
        fs:  `cpe:2.3:a:symantec:im_manager:8.4.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.1.52.14`,
        fs:  `cpe:2.3:a:hitachi:cobol2002_net_client_suite:01-02:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.1.52.14.1`,
        fs:  `cpe:2.3:a:libguestfs:libguestfs:1.8.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.1.52.15`,
        fs:  `cpe:2.3:a:apple:quicktime_streaming_server:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.1.53.64`,
        fs:  `cpe:2.3:a:redhat:icedtea6:1.9.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.1.82.76`,
        fs:  `cpe:2.3:a:microsoft:windows_media_rights_manager:4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.1.85.3`,
        fs:  `cpe:2.3:a:libpng:libpng:1.6.3:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.1.92.8`,
        fs:  `cpe:2.3:a:hp:storeonce_4220_backup_system:3.9.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.1.92.10`,
        fs:  `cpe:2.3:h:hp:h3c_router:msr_20-12_t1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.1.95.1`,
        fs:  `cpe:2.3:a:openssl:openssl:0.9.8d:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.1.95.2`,
        fs:  `cpe:2.3:a:php.s3.to:php-i-board:1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.1.102.64`,
        fs:  `cpe:2.3:o:cisco:nx-os:5.2\(1\)n1\(1b\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.1.105.6`,
        fs:  `cpe:2.3:o:cisco:nx-os:5.1\(3\)n2\(1a\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.1.106.16`,
        fs:  `cpe:2.3:a:squid-cache:squid:3.0:-:pre3:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.1.106.17`,
        fs:  `cpe:2.3:a:gnu:libtasn1:0.3.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.2.152`,
        fs:  `cpe:2.3:a:moodle:moodle:2.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.2.152.26`,
        fs:  `cpe:2.3:a:microsoft:windows_digital_rights_management:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.2.152.32`,
        fs:  `cpe:2.3:a:emc:rsa_access_manager:6.1:sp3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.2.152.33`,
        fs:  `cpe:2.3:a:symantec:veritas_dmp_dsm_solution:5.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.2.153.1`,
        fs:  `cpe:2.3:a:sun:jdk:1.4.2:update19:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.2.154.13`,
        fs:  `cpe:2.3:a:actividentity:activclient_cac:6.2.0.95:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.2.154.25`,
        fs:  `cpe:2.3:a:x.org:xorg-server:1.9.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.2.156.12`,
        fs:  `cpe:2.3:h:hp:h3c_switch:s3600-28tp-si_ls-3600-28tp-si-ovs:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.2.157.51`,
        fs:  `cpe:2.3:a:google:chrome:32.0.1671.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.2.159.1`,
        fs:  `cpe:2.3:a:netsweeper:netsweeper:4.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.181.14`,
        fs:  `cpe:2.3:o:compaq:tru64:5.0a_pk3_bl17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.181.16`,
        fs:  `cpe:2.3:a:ibm:lotus_domino:8.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.181.22`,
        fs:  `cpe:2.3:a:samba:samba:4.1.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.181.23`,
        fs:  `cpe:2.3:a:zhang_zhijun_taiwan_visit_2014-06-25_project:zhang_zhijun_taiwan_visit_2014-06-25:1.0:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.181.26`,
        fs:  `cpe:2.3:a:puppetlabs:puppet:2.6.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.181.34`,
        fs:  `cpe:2.3:a:viewvc:viewvc:1.0.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.183.5`,
        fs:  `cpe:2.3:a:barebones:yojimbo:2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.183.7`,
        fs:  `cpe:2.3:a:expressjs:express:4.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.183.10`,
        fs:  `cpe:2.3:a:busybox:busybox:0.41:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.183.11`,
        fs:  `cpe:2.3:a:ibm:rational_developer_for_system_z:8.0.3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.183.15`,
        fs:  `cpe:2.3:a:apache:cloudstack:2.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.183.16`,
        fs:  `cpe:2.3:a:ibm:db2:9.1.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.183.18`,
        fs:  `cpe:2.3:a:wordpress:wordpress:4.4.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.183.19`,
        fs:  `cpe:2.3:o:arubanetworks:arubaos:6.3.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.183.20`,
        fs:  `cpe:2.3:a:google:chrome:32.0.1700.55:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.183.23`,
        fs:  `cpe:2.3:a:vmware:ace:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.183.25`,
        fs:  `cpe:2.3:a:rocomotion:pplog_2:3.004:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.183.29`,
        fs:  `cpe:2.3:o:huawei:s9303_firmware:v200r002:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.183.43`,
        fs:  `cpe:2.3:a:apphp:hotel_site:3.2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.183.48`,
        fs:  `cpe:2.3:a:google:chrome:14.0.805.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.183.50`,
        fs:  `cpe:2.3:a:stephen_adkins:app\:\:context:0.01:*:*:*:*:perl:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.183.51`,
        fs:  `cpe:2.3:a:ecommercesoft:xse_shopping_cart:1.5.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.183.61`,
        fs:  `cpe:2.3:a:ibm:java_sdk:6.0.14.0:*:*:*:technology:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.183.63`,
        fs:  `cpe:2.3:a:gimp:gimp:2.4.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.183.67`,
        fs:  `cpe:2.3:a:drupal:drupal:7.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.183.68`,
        fs:  `cpe:2.3:a:mozilla:bugzilla:3.0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.183.75`,
        fs:  `cpe:2.3:a:tibco:spotfire_automation_services:6.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.183.86`,
        fs:  `cpe:2.3:h:d-link:dcs-5009l:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.185.22`,
        fs:  `cpe:2.3:a:vmware:vrealize_automation:6.0.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.185.24`,
        fs:  `cpe:2.3:a:apache:subversion:0.35.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.186.3`,
        fs:  `cpe:2.3:o:iij:seil_x1_firmware:2.55:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.186.6`,
        fs:  `cpe:2.3:o:d-link:dsr-1000_firmware:1.03b12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:10.3.186.7`,
        fs:  `cpe:2.3:o:cisco:unified_wireless_ip_phone_7920_firmware:1.0\(6\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.0`,
        fs:  `cpe:2.3:a:cisco:firesight_system_software:6.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.0.1.152`,
        fs:  `cpe:2.3:a:openssl:openssl:1.0.0g:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.0.1.152::~~~~x64~`,
        fs:  `cpe:2.3:a:lunascape:lunascape:3.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.0.1.153`,
        fs:  `cpe:2.3:a:geodesicsolutions:geocore_max:7.3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.1`,
        fs:  `cpe:2.3:a:ibm:websphere_application_server:6.0.2.30:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.1.102.55`,
        fs:  `cpe:2.3:a:artonx.org:activescriptruby:1.6.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.1.102.55::~~~~x64~`,
        fs:  `cpe:2.3:a:helios:textpad:5.4.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.1.102.59`,
        fs:  `cpe:2.3:a:pcre:pcre:8.32:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.1.102.62`,
        fs:  `cpe:2.3:a:ibm:infosphere_optim_data_growth_for_oracle_e-business_suite:6.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.1.102.63`,
        fs:  `cpe:2.3:a:novell:libzypp:11.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.1.111.8`,
        fs:  `cpe:2.3:a:hp:openview_network_node_manager:6.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.1.111.44`,
        fs:  `cpe:2.3:a:cisco:telepresence_video_communication_servers_software:x7.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.1.111.50`,
        fs:  `cpe:2.3:o:google:chrome_os:21.0.1180.34:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.1.111.54`,
        fs:  `cpe:2.3:a:redhat:rhel_global_file_system:3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.1.111.64`,
        fs:  `cpe:2.3:h:intel:pro_wireless_3945abg:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.1.111.73`,
        fs:  `cpe:2.3:a:postfix:postfix:2.2.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.1.115.7`,
        fs:  `cpe:2.3:a:gigabyte:dldrv2_activex_control:1.4.206.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.1.115.34`,
        fs:  `cpe:2.3:a:mcafee:portalshield_for_microsoft_sharepoint_server:2.0.216.102:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.1.115.48`,
        fs:  `cpe:2.3:a:openpkg:openpkg:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.1.115.54`,
        fs:  `cpe:2.3:a:perl:perl:1.46:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.1.115.58`,
        fs:  `cpe:2.3:a:mortbay:jetty:6.1.0:pre1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.1.115.59`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.8.7-0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.1.115.63`,
        fs:  `cpe:2.3:h:cisco:content_services_switch_11050:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.1.115.69`,
        fs:  `cpe:2.3:a:moinejf:abcm2ps:3.1.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.1.115.81`,
        fs:  `cpe:2.3:a:oleg_kovalchuk:cctags:7.x-1.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.223`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(18\)sxe1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.228`,
        fs:  `cpe:2.3:a:cisco:telepresence_multipoint_switch_software:1.5.2\(21\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.233`,
        fs:  `cpe:2.3:a:checkpoint:zonealarm:5.0.63.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.235`,
        fs:  `cpe:2.3:a:pedro_castro:gnome-subtitles:0.0.1:-:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.236`,
        fs:  `cpe:2.3:a:hapijs:crumb:2.2.0:*:*:*:*:node.js:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.238`,
        fs:  `cpe:2.3:a:sielcosistemi:winlog_lite:2.07.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.243`,
        fs:  `cpe:2.3:a:cisco:ips_sensor_software:7.2\(2\)e4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.251`,
        fs:  `cpe:2.3:a:haxx:curl:7.47.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.258`,
        fs:  `cpe:2.3:a:mozilla:bugzilla:4.5.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.261`,
        fs:  `cpe:2.3:a:cisco:telepresence_system_software:x7.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.262`,
        fs:  `cpe:2.3:a:heinz_mauelshagen:lvm2:2.02.55:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.270`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.2.3.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.273`,
        fs:  `cpe:2.3:a:ponsoftware:explzh:4.67:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.275`,
        fs:  `cpe:2.3:a:horde:horde_image_api:2.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.280`,
        fs:  `cpe:2.3:o:cisco:ios:12.0\(26\)w5\(28a\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.285`,
        fs:  `cpe:2.3:o:technicolor:tc7200_firmware:std6.01.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.291`,
        fs:  `cpe:2.3:a:libupnp_project:libupnp:1.6.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.297`,
        fs:  `cpe:2.3:a:openvas:openvas_scanner:4.0:beta7:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.310`,
        fs:  `cpe:2.3:h:huawei:cx600:v600r002:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.327`,
        fs:  `cpe:2.3:a:torproject:tor:0.0.9:rc3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.332`,
        fs:  `cpe:2.3:h:linksys:befsru31:1.42.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.335`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:7.0.5-2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.336`,
        fs:  `cpe:2.3:a:mathjs:math.js:0.16.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.341`,
        fs:  `cpe:2.3:a:w1.fi:wpa_supplicant:2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.346`,
        fs:  `cpe:2.3:a:apport_project:apport:2.16.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.350`,
        fs:  `cpe:2.3:a:ibm:rational_clearquest:8.0.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.356`,
        fs:  `cpe:2.3:a:redmine:redmine:1.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.359`,
        fs:  `cpe:2.3:h:lucent:ascend_pipeline_router:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.378`,
        fs:  `cpe:2.3:a:cyberlink:powerdvd:5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.394`,
        fs:  `cpe:2.3:a:abine:donottrackme_-_mobile_privacy:1.1.8:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.411`,
        fs:  `cpe:2.3:a:x.org:xfree86:4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.418`,
        fs:  `cpe:2.3:a:hp:insight_diagnostics:7.4.0.1570:online_windows_server_2003:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.424`,
        fs:  `cpe:2.3:a:rom_walton:boinc:7.3.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.425`,
        fs:  `cpe:2.3:a:keepass:password_safe:1.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.429`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.25:rc6:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.438`,
        fs:  `cpe:2.3:o:huawei:s12700_firmware:v200r008c00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.440`,
        fs:  `cpe:2.3:a:nodejs:node.js:0.12.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.442`,
        fs:  `cpe:2.3:a:horde:groupware:1.2.1:*:webmail:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.451`,
        fs:  `cpe:2.3:a:emery_berger:hoard:3.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.460`,
        fs:  `cpe:2.3:a:term_queue_project:term_queue:6.x-1.0:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.466`,
        fs:  `cpe:2.3:a:microsoft:exchange_server:2010:-:x64:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.468`,
        fs:  `cpe:2.3:a:ati:catalyst_control_center_help:2009.211.2224.40219:*:*:fi:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.475`,
        fs:  `cpe:2.3:a:hp:status:100.0.175.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.491`,
        fs:  `cpe:2.3:h:cisco:pix_firewall_501:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.535`,
        fs:  `cpe:2.3:a:typo3:typo3:4.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.548`,
        fs:  `cpe:2.3:a:bouncycastle:legion-of-the-bouncy-castle-java-crytography-api:1.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.554`,
        fs:  `cpe:2.3:a:dell:sonicwall_analyzer:7.2:sp1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.559`,
        fs:  `cpe:2.3:a:cisco:unified_application_environment_application_server:2.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.569`,
        fs:  `cpe:2.3:a:intel:pro_network_connections:11.2.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.577`,
        fs:  `cpe:2.3:o:huawei:s9300_firmware:v200r006c00spc500:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.626`,
        fs:  `cpe:2.3:a:dimastr:outlook_redemption:1.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.632`,
        fs:  `cpe:2.3:a:oracle:sun_products_suite:10.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.2.202.637::~~~linux~~`,
        fs:  `cpe:2.3:a:magzter:bbc_knowledge_magazine:3.01:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.3.300.257`,
        fs:  `cpe:2.3:a:cisco:application_control_engine_software:a3\(2.2\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.3.300.262`,
        fs:  `cpe:2.3:a:cisco:web_security_virtual_appliance:8.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.3.300.265`,
        fs:  `cpe:2.3:a:connect2id:nimbus_jose\&#43;jwt:4.34:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.3.300.268`,
        fs:  `cpe:2.3:h:hp:0235a19l:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.3.300.270`,
        fs:  `cpe:2.3:a:nodejs:node.js:0.11.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.3.300.271`,
        fs:  `cpe:2.3:a:roundup-tracker:roundup:0.7.0:b2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.3.300.273`,
        fs:  `cpe:2.3:a:bigtreecms:bigtree_cms:4.2.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.4.402.265`,
        fs:  `cpe:2.3:a:argyllcms:argyllcms:1.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.4.402.278`,
        fs:  `cpe:2.3:a:google:chrome:11.0.696.32:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.4.402.287`,
        fs:  `cpe:2.3:a:opnet:license_server:16.0.0.8934:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.5.502.110`,
        fs:  `cpe:2.3:a:primetek:primefaces:5.3.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.5.502.135`,
        fs:  `cpe:2.3:o:yahama:rt250i:8.02.35:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.5.502.136`,
        fs:  `cpe:2.3:a:dlc_solutions:course:6.x-1.0-rc7:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.5.502.146`,
        fs:  `cpe:2.3:a:apache:solr:4.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.5.502.149`,
        fs:  `cpe:2.3:h:toshibatec:e-studio-4520c:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.6.602.167`,
        fs:  `cpe:2.3:a:pidgin:pidgin:2.4.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.6.602.168`,
        fs:  `cpe:2.3:a:hp:update:4.0.7.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.6.602.171`,
        fs:  `cpe:2.3:a:openbsd:openssh:6.7:p1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.6.602.180`,
        fs:  `cpe:2.3:a:microsoft:content_management_server:2002:sp2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.7.700.169`,
        fs:  `cpe:2.3:o:paloaltonetworks:pan-os:5.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.7.700.202`,
        fs:  `cpe:2.3:h:dahuasecurity:dvr2404lf-al:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.7.700.203`,
        fs:  `cpe:2.3:a:freeradius:freeradius:1.1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.7.700.224`,
        fs:  `cpe:2.3:a:microsoft:windows_services_for_unix:3.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.7.700.225`,
        fs:  `cpe:2.3:a:hp:bidi_channel_components_installer:1.1.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.7.700.232`,
        fs:  `cpe:2.3:o:cisco:ons_15454sdh:3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.7.700.242`,
        fs:  `cpe:2.3:a:php:php:5.6.28:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.7.700.252`,
        fs:  `cpe:2.3:o:d-link:dcs-932l_camera_firmware:1.00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.7.700.257`,
        fs:  `cpe:2.3:o:nuuo:nvrsolo:2.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.7.700.260`,
        fs:  `cpe:2.3:a:gnome:gnome-shell:3.8.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.7.700.261`,
        fs:  `cpe:2.3:a:cms_tree_page_view_project:cms_tree_page_view:0.8.10:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.7.700.269`,
        fs:  `cpe:2.3:a:ibm:flex_system_manager:1.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.7.700.272`,
        fs:  `cpe:2.3:a:commentluv:commentluv:2.81.5:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.7.700.275`,
        fs:  `cpe:2.3:a:ovirt:ovirt:3.6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.7.700.279`,
        fs:  `cpe:2.3:a:ibm:tririga_application_platform:10.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.8.800.94`,
        fs:  `cpe:2.3:a:tenable:nessus:6.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.8.800.97`,
        fs:  `cpe:2.3:a:redhat:richfaces:5.0.0:alpha3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.8.800.168`,
        fs:  `cpe:2.3:a:cloudbees:jenkins:1.303:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.9.900.117`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.4:p8:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.9.900.152`,
        fs:  `cpe:2.3:a:redhat:jboss_enterprise_soa_platform:4.3.0:cp05:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:11.9.900.170`,
        fs:  `cpe:2.3:a:microsoft:ie:5.00.3314.2101:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:12.0.0.38`,
        fs:  `cpe:2.3:a:hitachi:cobol_gui_option_server:07-00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:12.0.0.41`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.357:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:12.0.0.43`,
        fs:  `cpe:2.3:o:google:chrome_os:26.0.1410.54:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:12.0.0.44`,
        fs:  `cpe:2.3:a:microsoft:exchange_server:2003:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:12.0.0.70`,
        fs:  `cpe:2.3:a:lockon:ec-cube:2.1.2:a:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:12.0.0.77`,
        fs:  `cpe:2.3:a:invensys:intouch\/wonderware_application_server:10.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:13.0.0.182`,
        fs:  `cpe:2.3:a:hitachi:cobol2002_net_client_suite:01-01:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:13.0.0.201`,
        fs:  `cpe:2.3:a:pinnaclesys:discapi:2.10.57:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:13.0.0.206`,
        fs:  `cpe:2.3:o:juniper:netscreen_screenos:5.0.0r3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:13.0.0.214`,
        fs:  `cpe:2.3:a:adobe:flash_player:11.2.202.559:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:13.0.0.223`,
        fs:  `cpe:2.3:o:huawei:quidway_s9300_firmware:v200r002c00spc100:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:13.0.0.231`,
        fs:  `cpe:2.3:a:puppetlabs:puppet:2.7.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:13.0.0.250`,
        fs:  `cpe:2.3:a:google:chrome:17.0.963.60:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:13.0.0.252`,
        fs:  `cpe:2.3:o:sgi:irix:6.5.9f:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:13.0.0.258`,
        fs:  `cpe:2.3:a:chatelao:php_address_book:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:13.0.0.260`,
        fs:  `cpe:2.3:a:emulex:common_san_management:5.0.66.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:13.0.0.262`,
        fs:  `cpe:2.3:a:fenrir-inc:sleipnir_mobile:2.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:13.0.0.264`,
        fs:  `cpe:2.3:a:postgresql:postgresql:9.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:13.0.0.289`,
        fs:  `cpe:2.3:a:intuit:quickbooks:2010:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:13.0.0.292`,
        fs:  `cpe:2.3:o:huawei:s9300_firmware:v200r003:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:14.0.0.125`,
        fs:  `cpe:2.3:o:cisco:ios:12.3\(14\)yq3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:14.0.0.145`,
        fs:  `cpe:2.3:a:openvpn:openvpn:2.4.0:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:15.0.0.189`,
        fs:  `cpe:2.3:a:hp:randmap:70.0.170.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:15.0.0.239`,
        fs:  `cpe:2.3:a:qemu:qemu:0.11.0-rc1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:16.0.0.234`,
        fs:  `cpe:2.3:a:bas_van_beek:multishop:0.0.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:16.0.0.257`,
        fs:  `cpe:2.3:a:joomla:joomla\!:1.6:beta9:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:16.0.0.287`,
        fs:  `cpe:2.3:a:adobe:search_for_help:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:16.0.0.296`,
        fs:  `cpe:2.3:a:x.org:xorg-server:1.14.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:16.0.0.305`,
        fs:  `cpe:2.3:a:novell:zenworks:6.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:17.0.0.134`,
        fs:  `cpe:2.3:a:prismmicrosys:eventtracker:6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:17.0.0.169`,
        fs:  `cpe:2.3:a:gnu:glibc:2.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:17.0.0.188`,
        fs:  `cpe:2.3:a:envato:complete_gallery_manager_plugin:3.2.6:rev36235:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:18.0.0.161`,
        fs:  `cpe:2.3:a:cmsmadesimple:cms_made_simple:1.11.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:18.0.0.194`,
        fs:  `cpe:2.3:o:compaq:tru64:4.0f_pk7_bl18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:18.0.0.209`,
        fs:  `cpe:2.3:a:google:chrome:25.0.1364.124:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:18.0.0.261`,
        fs:  `cpe:2.3:a:atlassian:jira:3.6.2:*:standard:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:18.0.0.261::~~esr~~~`,
        fs:  `cpe:2.3:a:google:chrome:32.0.1700.66:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:18.0.0.268`,
        fs:  `cpe:2.3:a:zend:zend_framework:1.8.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:18.0.0.324`,
        fs:  `cpe:2.3:a:iss:blackice_defender:2.9cap:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:18.0.0.326`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.5:p164:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:18.0.0.333::~~esr~~~`,
        fs:  `cpe:2.3:a:ibm:java:1.4.2.13.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:18.0.0.343::~~esr~~~`,
        fs:  `cpe:2.3:a:ffmpeg:ffmpeg:0.4.9:pre1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:18.0.0.352::~~esr~~~`,
        fs:  `cpe:2.3:a:fetchmail:fetchmail:6.2.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:18.0.0.360::~~esr~~~`,
        fs:  `cpe:2.3:a:mahara:mahara:1.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:18.0.0.366::~~esr~~~`,
        fs:  `cpe:2.3:a:eucalyptus:eucalyptus:2.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:19.0.0.185`,
        fs:  `cpe:2.3:o:yahama:rt200i:2.02.45:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:19.0.0.207`,
        fs:  `cpe:2.3:a:sendmail:sendmail:8.12.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:19.0.0.226`,
        fs:  `cpe:2.3:o:cisco:ios:15.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:19.0.0.245`,
        fs:  `cpe:2.3:a:f5:big-ip_local_traffic_manager:11.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:19.0.0.245::~~~chrome~~`,
        fs:  `cpe:2.3:a:jeff_sterup:plugin-organizer:0.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:19.0.0.245::~~~edge~~`,
        fs:  `cpe:2.3:a:heimdal_project:heimdal:1.5.0:pre1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:19.0.0.245::~~~internet_explorer~~`,
        fs:  `cpe:2.3:a:kryo:iodine:0.4.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:20.0.0.228`,
        fs:  `cpe:2.3:a:emc:networker:8.2.1.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:20.0.0.235`,
        fs:  `cpe:2.3:a:commentluv:commentluv:2.90.9.9.1:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:20.0.0.286`,
        fs:  `cpe:2.3:a:adobe:flash_player:10.1.92.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:20.0.0.306`,
        fs:  `cpe:2.3:o:juniper:junose_t:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:20.0.0.306::~~~chrome~~`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(28\)sb8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:20.0.0.306::~~~edge~~`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.4.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:20.0.0.306::~~~internet_explorer~~`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.4:*:*:*:*:*:x86:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:21.0.0.97`,
        fs:  `cpe:2.3:a:rubyonrails:ruby_on_rails:2.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:21.0.0.97::~~~chrome~~`,
        fs:  `cpe:2.3:a:jeff_kreitner:hms-testimonials:1.6.2:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:21.0.0.97::~~~edge~~`,
        fs:  `cpe:2.3:a:google:chrome:35.0.1916.46:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:21.0.0.97::~~~internet_explorer~~`,
        fs:  `cpe:2.3:h:brickom:wcb-100ap:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:21.0.0.213`,
        fs:  `cpe:2.3:a:fenrir-inc:sleipnir_mobile:2.8.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:21.0.0.226`,
        fs:  `cpe:2.3:a:oracle:sysfw:7.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:21.0.0.242::~~~chrome~~`,
        fs:  `cpe:2.3:a:enalean:tuleap:7.5.99.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:21.0.0.242::~~~internet_explorer~~`,
        fs:  `cpe:2.3:a:conky:conky:1.4.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:22.0.0.192`,
        fs:  `cpe:2.3:a:cisco:anyconnect_secure_mobility_client:4.2.01035:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:22.0.0.192::~~~chrome~~`,
        fs:  `cpe:2.3:a:openvas:openvas_administrator:1.3:beta1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:22.0.0.192::~~~edge~~`,
        fs:  `cpe:2.3:a:pmwiki:pmwiki:2.2.0:beta6:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:22.0.0.192::~~~internet_explorer~~`,
        fs:  `cpe:2.3:a:techsmith:snagit:10.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:22.0.0.211`,
        fs:  `cpe:2.3:a:cisco:adaptive_security_appliance_software:9.4.3.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:22.0.0.211::~~~chrome~~`,
        fs:  `cpe:2.3:a:sparkpay:capital_one_spark:0.9.81:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:22.0.0.211::~~~edge~~`,
        fs:  `cpe:2.3:a:octobercms:october_cms:1.0.323:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:22.0.0.211::~~~internet_explorer~~`,
        fs:  `cpe:2.3:a:simon_brown:pebble:1.5:beta2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:23.0.0.185`,
        fs:  `cpe:2.3:a:digia:qt:4.8.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:23.0.0.185::~~~chrome~~`,
        fs:  `cpe:2.3:a:gentoo:webmin:1.530:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:23.0.0.185::~~~edge~~`,
        fs:  `cpe:2.3:a:tecnick:tcexam:11.0.009:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:23.0.0.185::~~~internet_explorer~~`,
        fs:  `cpe:2.3:a:hp:insight_diagnostics:7.4.0.1570:online_windows_2000:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:23.0.0.207`,
        fs:  `cpe:2.3:a:bravenewcode:wptouch:1.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:23.0.0.207::~~~chrome~~`,
        fs:  `cpe:2.3:a:fujitsu:serverview_operations_manager:5.00.09:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:23.0.0.207::~~~edge~~`,
        fs:  `cpe:2.3:a:phpmyadmin:phpmyadmin:4.0.10.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:23.0.0.207::~~~internet_explorer~~`,
        fs:  `cpe:2.3:h:cisco:voice_gateway:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:24.0.0.186`,
        fs:  `cpe:2.3:a:southrivertech:titan_ftp_server:1.0.19:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:24.0.0.186::~~~chrome~~`,
        fs:  `cpe:2.3:a:simon_mcvittie:telepathy_gabble:0.17.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:24.0.0.186::~~~edge~~`,
        fs:  `cpe:2.3:o:cisco:ios:15.1\(3\)mra3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:24.0.0.186::~~~internet_explorer~~`,
        fs:  `cpe:2.3:a:bart_feenstra:payment:-:7.x-1.0:beta1:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:24.0.0.194`,
        fs:  `cpe:2.3:a:microsoft:microsoft.aspnetcore.mvc.core:1.0.2:*:*:*:*:asp.net:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:24.0.0.194::~~~chrome~~`,
        fs:  `cpe:2.3:a:igor_sysoev:nginx:0.8.34:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:24.0.0.194::~~~edge~~`,
        fs:  `cpe:2.3:a:sendmail:sendmail:8.7.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:24.0.0.194::~~~internet_explorer~~`,
        fs:  `cpe:2.3:a:t1lib:t1lib:0.1:alpha:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:24.0.0.221`,
        fs:  `cpe:2.3:a:fedoraproject:389_directory_server:1.2.11.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:24.0.0.221::~~~chrome~~`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.10.99:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:24.0.0.221::~~~edge~~`,
        fs:  `cpe:2.3:a:hitachi:electronic_form_workflow-professional_library_set:07-00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:24.0.0.221::~~~internet_explorer~~`,
        fs:  `cpe:2.3:a:digium:asterisk:1.2.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:26.0.0.137`,
        fs:  `cpe:2.3:a:ibm:rational_system_architect:3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:26.0.0.137::~~~chrome~~`,
        fs:  `cpe:2.3:o:yamaha:rt103i:4.00.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:26.0.0.137::~~~edge~~`,
        fs:  `cpe:2.3:a:trihedral:vtscada:11.1.06:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player:26.0.0.137::~~~internet_explorer_11~~`,
        fs:  `cpe:2.3:a:live555:streaming_media:2013.02.05:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_esr:18.0.0.329`,
        fs:  `cpe:2.3:a:gnome:gnome_display_manager:3.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:10.1.106.17`,
        fs:  `cpe:2.3:a:moinejf:abcm2ps:2.9.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:10.2.157.51`,
        fs:  `cpe:2.3:a:ibm:workload_deployer:3.1.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:10.3.186.7`,
        fs:  `cpe:2.3:a:blackbox:inkra_suite:2.1.5-b011:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:11.0.1.153`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.34:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:11.1.102.59`,
        fs:  `cpe:2.3:a:cisco:application_and_content_networking_system_software:5.5.31:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:11.1.111.5`,
        fs:  `cpe:2.3:o:google:chrome_os:20.0.1132.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:11.1.111.6`,
        fs:  `cpe:2.3:a:symantec:norton_antivirus:10.0.2.2021:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:11.1.111.7`,
        fs:  `cpe:2.3:h:cisco:meraki_ms:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:11.1.111.8`,
        fs:  `cpe:2.3:a:cisco:resource_manager:1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:11.1.111.9`,
        fs:  `cpe:2.3:o:iij:seil_b1_firmware:4.62:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:11.1.111.10`,
        fs:  `cpe:2.3:a:connect2id:nimbus_jose\&#43;jwt:2.18.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:11.1.111.16`,
        fs:  `cpe:2.3:a:flurry:flurry-analytics-android:3.3.4:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:11.1.111.19`,
        fs:  `cpe:2.3:a:symantec:norton_antivirus:10.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:11.1.111.24`,
        fs:  `cpe:2.3:a:google:chrome:12.0.716.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:11.1.111.29`,
        fs:  `cpe:2.3:a:videowhisper:videowhisper:7.x-1.1:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:11.1.111.31`,
        fs:  `cpe:2.3:o:univention:univention_corporate_server:2.2:1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:11.1.111.32`,
        fs:  `cpe:2.3:a:google:chrome:14.0.835.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:11.1.111.43`,
        fs:  `cpe:2.3:a:ait-pro:bulletproof-security:.48.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:11.1.112.60`,
        fs:  `cpe:2.3:a:oracle:identity_management_10g:10.1.4.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:11.1.112.61`,
        fs:  `cpe:2.3:a:craig_knudsen:webcalendar:1.2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:11.1.115.6`,
        fs:  `cpe:2.3:a:ssh:tectia_client:4.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:11.1.115.7`,
        fs:  `cpe:2.3:o:freebsd:freebsd:2.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:11.1.115.8`,
        fs:  `cpe:2.3:a:sixapart:movable_type:5.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:11.1.115.11`,
        fs:  `cpe:2.3:a:atlassian:jira:3.1:*:professional:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:11.1.115.12`,
        fs:  `cpe:2.3:a:sun:jdk:1.5.0:update6:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:11.1.115.17`,
        fs:  `cpe:2.3:a:ucweb:ucmobile_blovestorm:2.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:11.1.115.20`,
        fs:  `cpe:2.3:a:cisco:adaptive_security_appliance_software:7.2\(2.48\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:11.1.115.27`,
        fs:  `cpe:2.3:a:splunk:splunk:6.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:11.1.115.34`,
        fs:  `cpe:2.3:a:modssl:mod_ssl:2.0.47:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:11.1.115.36`,
        fs:  `cpe:2.3:a:freetype:freetype:2.4.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:11.1.115.37`,
        fs:  `cpe:2.3:a:sitaram_chamarty:gitolite:3.04:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_android:11.1.115.47`,
        fs:  `cpe:2.3:a:boltwire:boltwire:3.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_linux:9.0.31`,
        fs:  `cpe:2.3:a:isc:bind:9.7.2:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_linux:9.0.48.0`,
        fs:  `cpe:2.3:o:cisco:wireless_lan_controller_software:5.2.157.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_linux:9.0.115.0`,
        fs:  `cpe:2.3:a:twitter_pull_project:twitter_pull:7.x-1.0:rc4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_linux:9.0.124.0`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.14.55:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_linux:9.0.151.0`,
        fs:  `cpe:2.3:a:symantec:enterprise_security_manager:10.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_linux:10.0.12.36`,
        fs:  `cpe:2.3:a:rocomotion:p_board:1.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_linux:11.2.202.559`,
        fs:  `cpe:2.3:a:microsoft:office:2000:sp2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_linux:11.2.202.621`,
        fs:  `cpe:2.3:a:emeric_vernat:javamelody:1.27:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_player_for_linux:11.2.202.644`,
        fs:  `cpe:2.3:a:soraxsoft:sorax_reader:2.0.3129.70:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flash_video_encoder:2.0`,
        fs:  `cpe:2.3:a:cisco:mobility_services_engine:5.2_base:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flex_builder:3.0`,
        fs:  `cpe:2.3:a:mozilla:thunderbird:3.1.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flex_builder_plug-in:3.0.2.214193`,
        fs:  `cpe:2.3:a:hp:insight_diagnostics:7.6.2_rev_a:unknown:offline:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flex_sdk:-`,
        fs:  `cpe:2.3:a:openafs:openafs:1.5.64:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:flex_sdk:1.5`,
        fs:  `cpe:2.3:a:cagintranetworks:getsimple_cms:2.01:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:fonts_all:1.0`,
        fs:  `cpe:2.3:a:sensiolabs:symfony:2.0.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:fonts_all:2.0`,
        fs:  `cpe:2.3:a:redhat:cloudforms:4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:fonts_all:2.0::~~~~x64~`,
        fs:  `cpe:2.3:a:apache:juddi:3.1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:form_client_end-user_components:5.0.0`,
        fs:  `cpe:2.3:a:xerox:freeflow_print_server:8.0:sp2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:formflow_filler:-`,
        fs:  `cpe:2.3:a:context_project:context:7.x-3.1:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:formflow_filler:2.3.5`,
        fs:  `cpe:2.3:a:merethis:centreon:1.4.2.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:formflow_filler:2.22`,
        fs:  `cpe:2.3:a:igor_vlasenko:html-template-pro:0.01:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:formflow_filler:2.23`,
        fs:  `cpe:2.3:a:vips:vips:7.9.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:formflow_filler:2.24`,
        fs:  `cpe:2.3:a:ponsoftware:explzh:2.91:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:framemaker:-`,
        fs:  `cpe:2.3:a:gitlab:gitlab:8.8.0:*:*:*:enterprise:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:framemaker:7.0`,
        fs:  `cpe:2.3:a:oracle:retail_open_commerce_platform_cloud_service:4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:framemaker:7.1`,
        fs:  `cpe:2.3:a:linuxfoundation:cups-filters:1.0.34:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:framemaker:7.2`,
        fs:  `cpe:2.3:a:apple:safari:5.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:framemaker:8.0`,
        fs:  `cpe:2.3:a:cisco:unified_computing_system:2.0_5c:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:framemaker:8.0.1`,
        fs:  `cpe:2.3:a:andy_armstrong:cgi.pm:1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:framemaker:8.0.2`,
        fs:  `cpe:2.3:a:atlassian:bamboo:2.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:framemaker:8.0.3`,
        fs:  `cpe:2.3:a:sonicwall:aventail_access_manager:10.4.32:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:framemaker:9.0`,
        fs:  `cpe:2.3:a:thulasidas:easy-adsense-lite:2.90:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:framemaker:9.0.1`,
        fs:  `cpe:2.3:a:google:chrome:9.0.597.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:framemaker:9.0.2`,
        fs:  `cpe:2.3:a:devexpress:aspxfilemanager_control_for_webforms_and_mvc:12.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:framemaker:9.0.3`,
        fs:  `cpe:2.3:h:cisco:asa_5585-x:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:freehand:11.0`,
        fs:  `cpe:2.3:a:merethis:centreon:2.0:rc4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:freehand:11.0.0.439`,
        fs:  `cpe:2.3:a:torproject:tor:0.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:freehand:11.0.0.458`,
        fs:  `cpe:2.3:a:dan_kogai:encode_module:0.97:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:freehand:11.0.1`,
        fs:  `cpe:2.3:a:apple:quicktime:7.7.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:freehand:11.0.2`,
        fs:  `cpe:2.3:a:cisco:unity_express_software:1.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:freehand_mx:11.0`,
        fs:  `cpe:2.3:a:jgroups:jgroup:3.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:freehand_mx:11.0.1`,
        fs:  `cpe:2.3:a:topdesk:topdesk:7.03.022:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:freehand_mx:11.0.2`,
        fs:  `cpe:2.3:a:smartbear:soapui:4.0:beta2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:golive:-`,
        fs:  `cpe:2.3:a:phorum:phorum:5.0.14:a:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:golive:4.0`,
        fs:  `cpe:2.3:a:ibm:rational_host_on-demand:1.6.0.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:golive:4.0.1`,
        fs:  `cpe:2.3:a:php:php:4.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:golive:5.0`,
        fs:  `cpe:2.3:a:gnu:gnutls:2.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:golive:6.0.1`,
        fs:  `cpe:2.3:a:gnome:glib:2.15.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:golive:9`,
        fs:  `cpe:2.3:a:isc:bind:9.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:golive_cs2:7.0.0`,
        fs:  `cpe:2.3:a:freedesktop:poppler:0.4.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:golive_cs2:7.0.2`,
        fs:  `cpe:2.3:a:gnupg:gnupg:1.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:golive_cs2:8.0`,
        fs:  `cpe:2.3:o:compaq:tru64:5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:golive_cs2:8.0.1`,
        fs:  `cpe:2.3:a:atutor:atutor:0.9.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:graphics_server:-`,
        fs:  `cpe:2.3:a:ibm:tivoli_storage_flashcopy_manager:6.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:graphics_server:2.0`,
        fs:  `cpe:2.3:a:digium:asterisk:1.4.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:graphics_server:2.1`,
        fs:  `cpe:2.3:a:python:pillow:3.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:help_center:1.0.0`,
        fs:  `cpe:2.3:a:obm:open_business_management:2.2.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:help_center:2.1`,
        fs:  `cpe:2.3:a:vmware:esx_server:2.0.1_build_6403:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:help_viewer:1.0`,
        fs:  `cpe:2.3:a:net-snmp:net-snmp:5.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:help_viewer:1.1`,
        fs:  `cpe:2.3:a:dotcms:dotcms:3.6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:help_viewer:1.1.198`,
        fs:  `cpe:2.3:a:altiris:cmdb_and_asset_management_report_language_pack:6.5.1021:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:help_viewer:2.1`,
        fs:  `cpe:2.3:a:cisco:unified_communications_manager:11.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:homesite:5.2`,
        fs:  `cpe:2.3:a:shots:shots:1.0.8:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:homesite:5.5`,
        fs:  `cpe:2.3:a:redhat:sysreport:1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:illustrator:7.0`,
        fs:  `cpe:2.3:a:longtailvideo:jw_player_for_flash_\&amp;_html5_video_plugin:2.0.3:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:illustrator:8.0`,
        fs:  `cpe:2.3:a:ibm:websphere_portal:7.0.0.2:cf021:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:illustrator:9.0`,
        fs:  `cpe:2.3:a:oracle:mysql_connector\/net:6.3.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:illustrator:10.0`,
        fs:  `cpe:2.3:a:ratbox:ircd-ratbox:2.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:illustrator:11.0`,
        fs:  `cpe:2.3:a:vmware:horizon_view:5.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:illustrator:11.0.1`,
        fs:  `cpe:2.3:a:libpng:libpng:0.96:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:illustrator:12.0.0`,
        fs:  `cpe:2.3:a:lyesoft:andexplorer:1.3:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:illustrator:13.0`,
        fs:  `cpe:2.3:a:opera:opera_browser:3.62:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:illustrator:13.0.1`,
        fs:  `cpe:2.3:a:cmscout:cmscout:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:illustrator:13.0.2`,
        fs:  `cpe:2.3:a:adobe:brackets:1.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:illustrator:13.0.3`,
        fs:  `cpe:2.3:a:bea:weblogic_server:6.1:sp5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:illustrator:14.0`,
        fs:  `cpe:2.3:a:mysql:mysql:3.23.31:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:illustrator:15.0`,
        fs:  `cpe:2.3:h:hp:jd675a:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:illustrator:16.0.1`,
        fs:  `cpe:2.3:o:cisco:skinny_client_control_protocol_software:3.3\(14\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:illustrator:16.0.2`,
        fs:  `cpe:2.3:a:veritas:netbackup:7.5.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:illustrator:16.0.3`,
        fs:  `cpe:2.3:a:nordugrid:nordugrid-arc:0.3.37:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:illustrator:16.0.4`,
        fs:  `cpe:2.3:h:t-mobile:g2:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:illustrator:16.0.5`,
        fs:  `cpe:2.3:a:iss:blackice_pc_protection:3.6cbz:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:illustrator:16.2.0`,
        fs:  `cpe:2.3:a:crawlability:vbseo:3.6.0:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:illustrator:16.2.1`,
        fs:  `cpe:2.3:a:adobe:flash_player:11.7.700.232:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:illustrator:16.2.2`,
        fs:  `cpe:2.3:a:hitachi:cobol2002_net_server_suite:01-00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:indesign:3.0`,
        fs:  `cpe:2.3:h:hp:procurve_switch:jd373a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:indesign:3.0.1`,
        fs:  `cpe:2.3:a:sun:seam:1.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:indesign:4.0.0`,
        fs:  `cpe:2.3:a:hp:release_control:9.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:indesign:5.0`,
        fs:  `cpe:2.3:a:piwigo:piwigo:2.7.0:beta2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:indesign:5.0.1`,
        fs:  `cpe:2.3:a:otrs:otrs:1.3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:indesign:5.0.2`,
        fs:  `cpe:2.3:a:andy_armstrong:cgi.pm:2.59:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:indesign:6.0`,
        fs:  `cpe:2.3:a:wordpress:wordpress:3.8.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:indesign_cs3_icon_handler:5.0`,
        fs:  `cpe:2.3:a:davistribe:google_doc_embedder:2.5.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:indesign_cs4_application_feature_set_files:6.0`,
        fs:  `cpe:2.3:a:ibm:hacmp:4.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:indesign_cs4_common_base_files:6.0`,
        fs:  `cpe:2.3:a:apache:derby:10.5.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:indesign_cs4_icon_handler:6.0`,
        fs:  `cpe:2.3:o:seil:x86_fuji_firmware:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:indesign_cs4_icon_handler:6.0::~~~~x64~`,
        fs:  `cpe:2.3:o:yahama:rt80i:2.02.36:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:jabber:5.2.1.30`,
        fs:  `cpe:2.3:h:linksys:wrt54g_v5:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:japanese_fonts_support_for_adobe_reader:9.0.0`,
        fs:  `cpe:2.3:a:google:chrome:14.0.827.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:jrun:-`,
        fs:  `cpe:2.3:a:apache:mod_python:2.7.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:jrun:4.0`,
        fs:  `cpe:2.3:a:pulseaudio:pulseaudio:2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:jrun:4.0:sp1`,
        fs:  `cpe:2.3:a:jeff_ortel:suds:0.2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:jrun:4.0:sp1a`,
        fs:  `cpe:2.3:a:ibm:lotus_mobile_connect:6.1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:jrun:4.0_build_61650`,
        fs:  `cpe:2.3:o:yamaha:rt107e:8.03.08:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:kuler:2.0`,
        fs:  `cpe:2.3:a:atlassian:jira:2.5.3:*:enterprise:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:linguistics_cs3:3.0.0`,
        fs:  `cpe:2.3:o:microsoft:windows_xp:sp3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:linguistics_cs4:4.0.0`,
        fs:  `cpe:2.3:a:ibm:java:7.0.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:linguistics_cs4:4.0.0::~~~~x64~`,
        fs:  `cpe:2.3:h:cisco:2951_integrated_services_router:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:livecycle_data_services:3.0`,
        fs:  `cpe:2.3:a:chillcreations:mod_ccnewsletter:1.0.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:livecycle_data_services:4.5`,
        fs:  `cpe:2.3:a:hp:insight_diagnostics:6.2.1_a:unknown:offline:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:livecycle_data_services:4.6`,
        fs:  `cpe:2.3:a:libdwarf_project:libdwarf:20160613:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:livecycle_data_services:4.7`,
        fs:  `cpe:2.3:a:fedoraproject:389_directory_server:1.3.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:livecycle_designer_es2:9.0.0.20091029.1.612548`,
        fs:  `cpe:2.3:a:moinejf:abcm2ps:1.1.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:livecycle_es:8.2.1`,
        fs:  `cpe:2.3:a:google:chrome:17.0.963.32:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:livecycle_es2:9.0.0.0`,
        fs:  `cpe:2.3:a:jenkins:git:3.0.0:*:*:*:*:jenkins:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:livecycle_form_manager:-`,
        fs:  `cpe:2.3:a:x.org:xorg-server:1.7.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:livecycle_form_manager:7.01`,
        fs:  `cpe:2.3:a:gisle_aas:libwww-perl:5.46:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:livecycle_workflow:-`,
        fs:  `cpe:2.3:o:suse:suse_linux:4.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:media_encoder_cs4:-`,
        fs:  `cpe:2.3:a:ibm:http_server:2.0.47.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:media_encoder_cs4:1.0`,
        fs:  `cpe:2.3:a:atlassian:jira:4.2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:media_encoder_cs4_importer:1.0`,
        fs:  `cpe:2.3:o:ibm:aix:7.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:media_player:-`,
        fs:  `cpe:2.3:a:tag1consulting:support:6.x-1.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:media_player:0.0.0`,
        fs:  `cpe:2.3:a:rocomotion:pplog_2:3.05:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:media_player:1.0`,
        fs:  `cpe:2.3:a:microsoft:data_engine:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:media_player:1.1`,
        fs:  `cpe:2.3:a:fetchmail:fetchmail:4.5.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:media_player:1.5`,
        fs:  `cpe:2.3:a:mediaelementjs:mediaelement.js:2.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:media_player:1.8`,
        fs:  `cpe:2.3:a:fetchmail:fetchmail:4.7.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:motionpicture_color_files:1.0`,
        fs:  `cpe:2.3:h:siemens:scalance_w786-2rr:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:onlocation_cs3:3.0`,
        fs:  `cpe:2.3:a:simplejson_project:simplejson:2.6.2:*:*:*:*:python:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:onlocation_cs4:4.0.1`,
        fs:  `cpe:2.3:h:lexmark:x544:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:onlocation_cs4:4.0.2`,
        fs:  `cpe:2.3:a:isc:bind:9.10.4:b3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:onlocation_cs4:4.0.3`,
        fs:  `cpe:2.3:a:cybozu:garoon:3.5.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:output_module:2.0`,
        fs:  `cpe:2.3:o:yamaha:rta55i:4.06.28:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:output_module:2.1`,
        fs:  `cpe:2.3:a:apache:xalan-java:2.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:pagemaker:-`,
        fs:  `cpe:2.3:a:tigris:tortoisesvn:0.20.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:pagemaker:6.5`,
        fs:  `cpe:2.3:a:visualsoftru:visual_quickbar:4.4.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:pagemaker:7.0`,
        fs:  `cpe:2.3:a:cisco:firewall_services_module_software:3.2\(18\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:pagemaker:7.0.1`,
        fs:  `cpe:2.3:a:varnish-cache:varnish:2.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:pagemaker:7.0.2`,
        fs:  `cpe:2.3:a:gitlab:gitlab:8.17.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:pdf_creation_add-on:9.0.0`,
        fs:  `cpe:2.3:a:postgresql:postgresql:7.4.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:pdf_ifilter:5.0`,
        fs:  `cpe:2.3:a:gnu:gnutls:2.3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:pdf_ifilter:6.0`,
        fs:  `cpe:2.3:a:chatelao:php_address_book:3.3.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:pdf_ifilter:9.0.0::~~~~x64~`,
        fs:  `cpe:2.3:a:fetchmail:fetchmail:6.2.9:rc7:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:pdf_library_sdk:8.0`,
        fs:  `cpe:2.3:o:juniper:junos:15.1:f2-s4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:pdf_library_sdk:9.0`,
        fs:  `cpe:2.3:a:s9y:serendipity:1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:pdf_library_sdk:9.0::~~~~x64~`,
        fs:  `cpe:2.3:o:iij:seil_x2_firmware:3.95:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:pdf_settings:1.0`,
        fs:  `cpe:2.3:a:irfanview:irfanview:2.83:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:pdf_settings_cs4:9.0`,
        fs:  `cpe:2.3:o:cisco:skinny_client_control_protocol_software:5.0\(6\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:pdf_settings_cs5:10.0`,
        fs:  `cpe:2.3:a:netscape:messaging_server:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:pdistiller:8.1`,
        fs:  `cpe:2.3:a:autodesk:autocad_lt:2004:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photodeluxe:-`,
        fs:  `cpe:2.3:a:git-scm:git:1.7.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photodeluxe:3.0`,
        fs:  `cpe:2.3:a:controlsystemworks:csworks:1.0.720.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photodeluxe:3.1`,
        fs:  `cpe:2.3:a:hitachi:cosminexus_client:06-50-\/f:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photodeluxe:4.0`,
        fs:  `cpe:2.3:a:bundler:bundler:1.7.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:-`,
        fs:  `cpe:2.3:a:cgiirc:cgi\:irc:0.3.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:2.5::pro`,
        fs:  `cpe:2.3:a:google:chrome:20.0.1132.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:3.0::pro`,
        fs:  `cpe:2.3:a:cisco:anyconnect_secure_mobility_client:3.0.3050:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:4.0::pro`,
        fs:  `cpe:2.3:a:ffmpeg:ffmpeg:2.2.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:5.0::pro`,
        fs:  `cpe:2.3:a:igor_sysoev:nginx:0.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:6.0`,
        fs:  `cpe:2.3:a:menalto:gallery:1.6:alpha3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:6.0::pro`,
        fs:  `cpe:2.3:a:hp:openview_configuration_management:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:6.0.1`,
        fs:  `cpe:2.3:a:sun:one_directory_server:5.1:sp3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:7.0`,
        fs:  `cpe:2.3:a:ibm:websphere_service_registry_and_repository:7.0.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:7.0.1`,
        fs:  `cpe:2.3:a:cerberusftp:ftp_server:5.0.4.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:8.0`,
        fs:  `cpe:2.3:a:mediawiki:mediawiki:1.19.23:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:9.0`,
        fs:  `cpe:2.3:a:atlassian:jira:3.8:*:enterprise:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:9.0.1`,
        fs:  `cpe:2.3:a:stunnel:stunnel:5.04:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:9.0.2`,
        fs:  `cpe:2.3:a:dropbox:dropbox:0.7.110:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:10.0`,
        fs:  `cpe:2.3:o:iij:seil_plus_firmware:1.90:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:10.0.1`,
        fs:  `cpe:2.3:h:huawei:ne40e\/80e:v600r002:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:11.0`,
        fs:  `cpe:2.3:a:hitachi:cosminexus_developer:06-50:-:standard:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:11.0::~~~~x64~`,
        fs:  `cpe:2.3:a:acronis:license_server:10.0.13762:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:11.0.1`,
        fs:  `cpe:2.3:a:linuxfoundation:cups-filters:1.17.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:11.0.2`,
        fs:  `cpe:2.3:a:libguestfs:libguestfs:1.8.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:11.0.4`,
        fs:  `cpe:2.3:a:owncloud:owncloud:4.0.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:12.0.0`,
        fs:  `cpe:2.3:o:cisco:ios:12.1\(12c\)ev01:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:12.0.1`,
        fs:  `cpe:2.3:a:nullsoft:winamp:5.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:12.0.2`,
        fs:  `cpe:2.3:a:google:chrome:4.0.243.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:12.0.3`,
        fs:  `cpe:2.3:a:microsoft:office_multilingual_user_interface_pack:2003:sp2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:12.0.4`,
        fs:  `cpe:2.3:a:ibm:financial_transaction_manager:3.0.0.14:*:*:*:*:cps_services:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:15.0`,
        fs:  `cpe:2.3:a:gnu:grub2:1.99:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:15.2.2`,
        fs:  `cpe:2.3:a:telaxus:epesi:1.7.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:16.0`,
        fs:  `cpe:2.3:a:david_king:vino:2.25:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:16.1`,
        fs:  `cpe:2.3:a:ibm:websphere_application_server:3.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:17.0`,
        fs:  `cpe:2.3:a:transmissionbt:transmission:2.33:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:17.0.1`,
        fs:  `cpe:2.3:a:trustgo:antivirus_\&amp;_mobile_security:1.3.0:-:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:18.0`,
        fs:  `cpe:2.3:a:ibm:aix_parallel_systems_support_programs:3.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:18.0.1`,
        fs:  `cpe:2.3:o:cisco:ios:15.3\(3\)jpb2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:18.1.0`,
        fs:  `cpe:2.3:a:inspire_ircd:inspircd:1.1.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:18.1.1`,
        fs:  `cpe:2.3:a:redmine:redmine:1.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:19.0.0`,
        fs:  `cpe:2.3:a:blackbox:inkra_center_point:2.1.6-b032:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop:cs`,
        fs:  `cpe:2.3:a:masahiko_watanabe:apsaly:2.60:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop_album:2.0:-:starter`,
        fs:  `cpe:2.3:a:matchalabs:metaslider:2.5:beta2:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop_album:3.0:-:starter`,
        fs:  `cpe:2.3:a:kde:kde_sc:4.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop_album:3.2:-:starter`,
        fs:  `cpe:2.3:a:boldonjames:masterkeyplus:5.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop_cc:16.1.1`,
        fs:  `cpe:2.3:a:batavi:batavi:1.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop_cs3:10.0`,
        fs:  `cpe:2.3:a:sony:vaio_manual_cybersupport:3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop_cs4:11.0`,
        fs:  `cpe:2.3:o:cisco:ons_15454_optical_transport_platform:4.0\(1\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop_cs4_support:11.0`,
        fs:  `cpe:2.3:a:live555:streaming_media:2012.09.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop_cs5:12.0`,
        fs:  `cpe:2.3:a:thulasidas:easy-adsense-lite:6.01:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop_cs5:12.0.1`,
        fs:  `cpe:2.3:a:gnu:gnutls:2.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop_cs5:12.0.2`,
        fs:  `cpe:2.3:a:keycloak:keycloak-nodejs-auth-utils:2.5.0:cr1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop_cs5:12.0.3`,
        fs:  `cpe:2.3:a:formsplayer:formsplayer:1.5.0:1026:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop_cs5:12.0.4`,
        fs:  `cpe:2.3:a:webyog:sqlyog:5.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop_cs5:12.0.5`,
        fs:  `cpe:2.3:h:apple:macbook_pro:ma897xx\/a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop_cs5.1:12.1`,
        fs:  `cpe:2.3:a:rsa:authentication_api:8.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop_cs5.1:12.1.1`,
        fs:  `cpe:2.3:a:gitlab:gitlab:8.9.4:*:*:*:enterprise:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop_cs6:13.0`,
        fs:  `cpe:2.3:a:rpm:rpm:2.2.3.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop_cs6:13.0.1`,
        fs:  `cpe:2.3:o:corega:cg-wlbargnl_firmware:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop_elements:-`,
        fs:  `cpe:2.3:a:steindom:contact_save:6.x-1.x:dev:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop_elements:1.0`,
        fs:  `cpe:2.3:a:josh_fradley:burden:1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop_elements:2.0`,
        fs:  `cpe:2.3:a:debian:dpkg:1.4.0.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop_elements:3.0`,
        fs:  `cpe:2.3:a:postgresql:postgresql:9.4.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop_elements:4.0`,
        fs:  `cpe:2.3:a:nordugrid:nordugrid-arc:0.5.54:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop_elements:5.0`,
        fs:  `cpe:2.3:a:apache:mod_perl:1.999:20:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop_elements:6.0`,
        fs:  `cpe:2.3:a:oracle:jre:1.8.0:update_74:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop_elements:7.0`,
        fs:  `cpe:2.3:a:mortbay:jetty:3.1.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop_elements:8.0`,
        fs:  `cpe:2.3:o:juniper:junos:14.2:r5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop_elements:9.0`,
        fs:  `cpe:2.3:o:hp:hp-ux:10.34:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:photoshop_elements:10.0`,
        fs:  `cpe:2.3:a:tidio-form_project:tidio-form:1.0:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:pixel_bender_toolkit:1.0`,
        fs:  `cpe:2.3:a:ajaxplorer:ajaxplorer:4.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:premiere_elements:3.0.0`,
        fs:  `cpe:2.3:a:sixapart:movable_type:4.29:*:open_source:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:premiere_elements_templates:1.0.0`,
        fs:  `cpe:2.3:a:alquimia:managesite:6.x-1.x:dev:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:premiere_pro_cs3:3.0.1`,
        fs:  `cpe:2.3:a:primetek:primefaces:5.2.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:premiere_pro_cs3:3.1`,
        fs:  `cpe:2.3:o:cisco:ios_xe:3.10s.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:premiere_pro_cs3:3.1.1`,
        fs:  `cpe:2.3:a:fluxbb:fluxbb:1.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:premiere_pro_cs3:3.2`,
        fs:  `cpe:2.3:a:php:php:5.5.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:premiere_pro_cs4:4.0.1`,
        fs:  `cpe:2.3:a:apache:libcloud:0.13.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:premiere_pro_cs4:4.1.0`,
        fs:  `cpe:2.3:a:collectd:collectd:5.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:premiere_pro_cs4:4.2.0`,
        fs:  `cpe:2.3:a:oracle:database_server:7.1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:premiere_pro_cs4:4.2.1`,
        fs:  `cpe:2.3:a:pmwiki:pmwiki:2.2.0:beta5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:presenter:6.2.0`,
        fs:  `cpe:2.3:a:southrivertech:titan_ftp_server:5.03.310:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:presenter:7.0`,
        fs:  `cpe:2.3:a:riken:xoonips:3.44:*:*:*:*:xoops:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:presenter:7.0.6`,
        fs:  `cpe:2.3:a:ipswitch:ws_ftp_server:3.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:reader:7`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.32:rc5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:reader:8`,
        fs:  `cpe:2.3:a:owncloud:owncloud:8.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:reader:9`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(58\)se2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:reader:11.0.15::~~desktop~~~`,
        fs:  `cpe:2.3:a:prestashop:prestashop:1.4.6.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:reader:11.0.16`,
        fs:  `cpe:2.3:a:redhat:jboss_weld:3.0.0:alpha1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:reader_extended_language_support_font_pack:9.0.0`,
        fs:  `cpe:2.3:a:sun:java_auto_updater:2.0.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:robohelp:-`,
        fs:  `cpe:2.3:a:microsoft:biztalk_server:2000:sp2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:robohelp:6`,
        fs:  `cpe:2.3:a:ca:host-based_intrusion_prevention_system:8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:robohelp:7`,
        fs:  `cpe:2.3:a:vmware:workstation:8.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:robohelp:7.0.1`,
        fs:  `cpe:2.3:h:supermicro:x9dre-ln4f:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:robohelp:7.0.2`,
        fs:  `cpe:2.3:a:cloudfoundry:cf-release:168:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:robohelp:7.0.3`,
        fs:  `cpe:2.3:a:dlo:simple_anti_bot_registration_engine_plugin:1.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:robohelp:8`,
        fs:  `cpe:2.3:a:dymo:dymo_label:8.3.0.1242:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:robohelp:8.0.1`,
        fs:  `cpe:2.3:a:symantec:veritas_backup_exec_remote_agent_for_windows_systems:13.0.5204:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:robohelp:8.0.2`,
        fs:  `cpe:2.3:a:southrivertech:titan_ftp_server:2.40.155:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:robohelp:9`,
        fs:  `cpe:2.3:h:hp:procurve_switch:jf844a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:robohelp:9.0.0.228`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.631:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:robohelp:9.0.1`,
        fs:  `cpe:2.3:a:ibm:emptoris_supplier_lifecycle_management:10.0.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:robohelp:9.0.1.232`,
        fs:  `cpe:2.3:a:gitlab:gitlab:8.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:robohelp:9.0.2`,
        fs:  `cpe:2.3:o:xen:xen:4.1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:robohelp:10.0`,
        fs:  `cpe:2.3:a:google:chrome:0.4.154.33:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:robohelp:11.0`,
        fs:  `cpe:2.3:a:horde:imp:4.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:robohelp:2015.0.3`,
        fs:  `cpe:2.3:a:seeddms:seeddms:3.4.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:robohelp:x5`,
        fs:  `cpe:2.3:a:mozilla:seamonkey:2.18:beta2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:robohelp_server:-`,
        fs:  `cpe:2.3:a:project-redcap:redcap:4.14.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:robohelp_server:6`,
        fs:  `cpe:2.3:a:postgresql:postgresql:8.3.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:robohelp_server:7`,
        fs:  `cpe:2.3:a:philippe_jounin:tftpd32:2.70:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:robohelp_server:8`,
        fs:  `cpe:2.3:a:phpmyadmin:phpmyadmin:4.6.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:search_for_help:1.0`,
        fs:  `cpe:2.3:a:apple:remote_desktop:3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:service_manager_extension:1.0`,
        fs:  `cpe:2.3:a:tiger-fish:fancy_slide:6.x-2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:setup:1.0`,
        fs:  `cpe:2.3:a:gnu:tramp:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:setup:2.0`,
        fs:  `cpe:2.3:a:shrew:vpn_client:2.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:sgm_cs4:3.0`,
        fs:  `cpe:2.3:a:ibm:lotus_notes:7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave:8.5.1.102`,
        fs:  `cpe:2.3:a:avocent:dsview:3.1.1.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:-`,
        fs:  `cpe:2.3:a:topdesk:topdesk:7.02.014:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:1.0`,
        fs:  `cpe:2.3:a:c-ares_project:c-ares:1.7.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:2.0`,
        fs:  `cpe:2.3:a:apache:syncope:1.0.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:3.0`,
        fs:  `cpe:2.3:a:cisco:application_and_content_networking_system_software:5.0.3.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:4.0`,
        fs:  `cpe:2.3:a:freedesktop:poppler:0.23.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:5.0`,
        fs:  `cpe:2.3:a:google:chrome:5.0.375.62:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:6.0`,
        fs:  `cpe:2.3:a:google:v8:3.12.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:8.0`,
        fs:  `cpe:2.3:a:oracle:vm_virtualbox:4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:8.0.196`,
        fs:  `cpe:2.3:a:oracle:fusion_middleware:8.0:update2_patch5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:8.0.196a`,
        fs:  `cpe:2.3:a:hitachi:ucosminexus_collaboration:06-30:-:server:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:8.0.204`,
        fs:  `cpe:2.3:a:gnome:glib:2.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:8.0.205`,
        fs:  `cpe:2.3:a:ibm:websphere_commerce:6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:8.5.1`,
        fs:  `cpe:2.3:a:rom_walton:boinc:7.1.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:8.5.1.100`,
        fs:  `cpe:2.3:a:symantec:backupexec_system_recovery:8.0.1.24976:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:8.5.1.103`,
        fs:  `cpe:2.3:a:bitcoin:bitcoind:0.3.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:8.5.1.105`,
        fs:  `cpe:2.3:a:djangoproject:django:1.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:8.5.1.106`,
        fs:  `cpe:2.3:a:gnu:gnutls:3.0.23:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:8.5.321`,
        fs:  `cpe:2.3:a:apache:cloudstack:4.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:8.5.323`,
        fs:  `cpe:2.3:a:glub:secure_ftp:2.5.19:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:8.5.324`,
        fs:  `cpe:2.3:a:opera:opera_browser:3.51:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:8.5.325`,
        fs:  `cpe:2.3:a:coppermine-gallery:coppermine_photo_gallery:1.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:9`,
        fs:  `cpe:2.3:a:huawei:espace_7910:v200r002c00spc100:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:9.0.383`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.651.3:*:*:*:lts:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:9.0.432`,
        fs:  `cpe:2.3:a:inspire_ircd:inspircd:2.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:10.0.0.210`,
        fs:  `cpe:2.3:o:juniper:netscreen_screenos:3.1.0r2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:10.0.1.004`,
        fs:  `cpe:2.3:o:iij:seil_x1_firmware:2.57:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:10.1.0.011`,
        fs:  `cpe:2.3:a:mozilla:bugzilla:4.4.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:10.1.0.11`,
        fs:  `cpe:2.3:a:symantec:reporting_agents:1.0.258.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:10.1.1.016`,
        fs:  `cpe:2.3:a:igor_sysoev:nginx:0.6.26:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:10.1.4.020`,
        fs:  `cpe:2.3:a:jgroups:jgroup:3.0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:10.2.0.021`,
        fs:  `cpe:2.3:a:actividentity:activclient_cac:6.2.0.120:*:*:*:*:*:x64:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:10.2.0.022`,
        fs:  `cpe:2.3:a:mediawiki:mediawiki:1.16.0:beta3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:10.2.0.023`,
        fs:  `cpe:2.3:a:ibm:rational_quality_manager:4.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:11`,
        fs:  `cpe:2.3:a:hp:web_registration:45.4.157.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:11.0.0.456`,
        fs:  `cpe:2.3:o:motorola:android:4.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:11.0.3.471`,
        fs:  `cpe:2.3:a:apache:http_server:1.3.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:11.5.0.595`,
        fs:  `cpe:2.3:a:joomlaboat:com_youtubegallery:3.9.2:*:*:*:*:joomla\!:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:11.5.0.596`,
        fs:  `cpe:2.3:a:sun:java_system_web_server:6.1:sp1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:11.5.1.601`,
        fs:  `cpe:2.3:a:tejimaya:openpne:1.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:11.5.2.602`,
        fs:  `cpe:2.3:a:cisco:telepresence_tc_software:5.1.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:11.5.6.606`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.7:p14:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:11.5.7.609`,
        fs:  `cpe:2.3:h:huawei:ar3200:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:11.5.8.612`,
        fs:  `cpe:2.3:a:gnome:glib:2.3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:11.5.9.615`,
        fs:  `cpe:2.3:a:hp:software_xalana_xslt_processor:1.10.60:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:11.5.9.620`,
        fs:  `cpe:2.3:a:ibm:lotus_foundations_start:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:11.5.10.620`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.8.0-3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:11.6.0.626`,
        fs:  `cpe:2.3:o:cisco:ios_xe_2.5:2.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:11.6.1.629`,
        fs:  `cpe:2.3:a:cloudbees:jenkins:1.408:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:11.6.3.633`,
        fs:  `cpe:2.3:o:cisco:ios_xe_3.2sg:3.2.2sg:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:11.6.4.634`,
        fs:  `cpe:2.3:o:mandrakesoft:mandrake_linux:2007.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:11.6.5.635`,
        fs:  `cpe:2.3:a:wallpaperscript:wallpaperscript:3.5.0082:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:11.6.6.636`,
        fs:  `cpe:2.3:a:oracle:micros_c2:9.89.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:11.6.7.637`,
        fs:  `cpe:2.3:a:openswan:openswan:2.6.02:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:11.6.8.638`,
        fs:  `cpe:2.3:a:mcafee:network_security_manager:7.1.15.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:12.0.0.112`,
        fs:  `cpe:2.3:a:ibm:tealeaf_consumer_experience:9.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:12.0.2.122`,
        fs:  `cpe:2.3:a:mozilla:bugzilla:4.3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:12.0.3.133`,
        fs:  `cpe:2.3:h:nortel:optical_metro_5200:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:12.0.4.144`,
        fs:  `cpe:2.3:a:isc:inn:1.4unoff4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:12.0.6.147`,
        fs:  `cpe:2.3:o:cisco:ios_xe:3.2xo.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:12.0.7.148`,
        fs:  `cpe:2.3:a:cisco:unity_connection:8.6:\(1\):*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:12.0.9.149`,
        fs:  `cpe:2.3:a:lockon:ec-cube:2.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:12.1.0.150`,
        fs:  `cpe:2.3:a:scott_reynen:node_embed:7.x-1.0:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player:12.2.7.197`,
        fs:  `cpe:2.3:a:mollify:mollify:1.6.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:shockwave_player_merge_module:12.0.0.1`,
        fs:  `cpe:2.3:a:bss:continuity_cms:4.2.22640.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:sing_cs3:0.1`,
        fs:  `cpe:2.3:h:emc:rsa_data_protection_manager_appliance:3.2.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:sing_cs4:2.0`,
        fs:  `cpe:2.3:a:google:chrome:22.0.1229.26:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:soundbooth_cs3:1.0`,
        fs:  `cpe:2.3:a:ibm:spss_samplepower:3.0.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:soundbooth_cs4:2.0`,
        fs:  `cpe:2.3:a:openvpn:openvpn:1.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:soundbooth_cs4:2.0.1`,
        fs:  `cpe:2.3:a:umn:mapserver:5.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:soundbooth_cs4_codecs:2.0`,
        fs:  `cpe:2.3:a:custom_search_project:custom_search:7.x-1.12:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:spelling_dictionaries_support_for_adobe_reader:8.0.0`,
        fs:  `cpe:2.3:a:ibm:san_volume_controller_software:6.4.1.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:spelling_dictionaries_support_for_adobe_reader:9.0.0`,
        fs:  `cpe:2.3:a:winzip:winzip:8.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:stock_photos:1.0`,
        fs:  `cpe:2.3:a:oracle:mysql:5.6.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:stock_photos_cs2:-`,
        fs:  `cpe:2.3:a:google:chrome:8.0.560.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:stock_photos_cs2:1.0.9`,
        fs:  `cpe:2.3:a:webmin:webmin:1.680:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:stock_photos_cs3:1.5`,
        fs:  `cpe:2.3:h:cisco:catalyst_2901:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:stock_photos_cs3:1.5.1`,
        fs:  `cpe:2.3:a:hans_nilsson:video_filter:6.x-2.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:studio_mx:7.0`,
        fs:  `cpe:2.3:a:crunchify:facebook_members:4.5.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:suite_shared_configuration_cs4:1.0`,
        fs:  `cpe:2.3:a:ovirt:ovirt:3.6.10.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:suite_specific:2.0.0`,
        fs:  `cpe:2.3:a:redhat:wu_ftpd:2.6.1-18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:svg_viewer:-`,
        fs:  `cpe:2.3:a:apple:preview:3.0.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:svg_viewer:1.0`,
        fs:  `cpe:2.3:a:gnu:cfengine:2.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:svg_viewer:2.0`,
        fs:  `cpe:2.3:a:zenphoto:zenphoto:1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:svg_viewer:3.0`,
        fs:  `cpe:2.3:a:moinejf:abcm2ps:1.2.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:svg_viewer:3.01`,
        fs:  `cpe:2.3:a:magzter:gent_magazine:3:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:svg_viewer:3.02`,
        fs:  `cpe:2.3:a:synology:cloud_station:3.0-3005:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:type_support:1.0`,
        fs:  `cpe:2.3:a:winpcap:winpcap:3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:type_support_cs4:9.0`,
        fs:  `cpe:2.3:a:rsgallery2:com_rsgallery2:1.10.13:alpha:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:type_support_cs4:9.0::~~~~x64~`,
        fs:  `cpe:2.3:a:lunascape:lunascape:4.6.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:update_manager:4.0.3`,
        fs:  `cpe:2.3:o:opensuse:evergreen:11.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:update_manager:4.0.5`,
        fs:  `cpe:2.3:a:netty_project:netty:4.0.30:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:update_manager:5`,
        fs:  `cpe:2.3:a:piwigo:piwigo:2.1.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:update_manager:5.1`,
        fs:  `cpe:2.3:a:cisco:telepresence_advanced_media_gateway:1.0\(.1.13\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:update_manager:6.0`,
        fs:  `cpe:2.3:a:tweet-blender:tweet-blender:3.1.2:-:-:*:-:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:version_cue:1.0`,
        fs:  `cpe:2.3:h:buffalotech:wsr-1166dhp:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:version_cue:1.0.1`,
        fs:  `cpe:2.3:a:ovirt:ovirt:3.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:version_cue_cs2:2.0`,
        fs:  `cpe:2.3:a:condusiv:diskeeper_2008:12.0.781.64:*:*:*:server:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:version_cue_cs2:2.0.1`,
        fs:  `cpe:2.3:a:ibm:data_server_client:9.7.0.441:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:version_cue_cs3_client:3.0.0`,
        fs:  `cpe:2.3:o:yahama:rt103i:4.00.48:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:version_cue_cs3_client:3.1.0`,
        fs:  `cpe:2.3:a:oracle:http_server:1.0.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:version_cue_cs3_server:3.0`,
        fs:  `cpe:2.3:a:cubecart:cubecart:4.3.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:version_cue_cs3_server:3.1.0`,
        fs:  `cpe:2.3:h:f5:big-ip_protocol_security_manager:11.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:version_cue_cs3_server:adobe_version_cue_cs3_server_`,
        fs:  `cpe:2.3:a:manuel_garcia:galleryformatter:6.x-1.0:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:version_cue_cs4_server:4.0`,
        fs:  `cpe:2.3:a:trendmicro:interscan_web_security_suite:2.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:version_cue_cs4_server:4.0.1`,
        fs:  `cpe:2.3:a:mozilla:camino:0.8.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:was_cs3:1.0`,
        fs:  `cpe:2.3:a:apache:tomcat:8.5.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:widget_browser:2.0.230`,
        fs:  `cpe:2.3:a:primetek:primefaces:2.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:widget_browser:2.0_build_230`,
        fs:  `cpe:2.3:a:hp:insight_diagnostics:7.6.0.1984:online_windows_2003_x64:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:winsoft_linguistics_plugin:1.0`,
        fs:  `cpe:2.3:a:f5:big-ip:4.5.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:winsoft_linguistics_plugin:1.1`,
        fs:  `cpe:2.3:a:moinmo:moinmoin:1.8.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:winsoft_linguistics_plugin:1.1::~~~~x64~`,
        fs:  `cpe:2.3:a:adobe:acrobat:9.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:xmp_panels_cs3:1.0`,
        fs:  `cpe:2.3:a:heimdal_project:heimdal:0.2p:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adobe:xmp_panels_cs4:2.0`,
        fs:  `cpe:2.3:o:iij:seil_x2_firmware:3.61:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.00:beta`,
        fs:  `cpe:2.3:h:antlabs:inngate_ig_3101:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.01:beta`,
        fs:  `cpe:2.3:a:hitachi:ucosminexus_operator:07-03-03:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.02`,
        fs:  `cpe:2.3:a:lxml:lxml:3.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.02:a`,
        fs:  `cpe:2.3:a:mozilla:thunderbird:38.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.03`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.12.50:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.04`,
        fs:  `cpe:2.3:a:mantisbt:mantisbt:1.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.04:a`,
        fs:  `cpe:2.3:a:sielcosistemi:winlog_pro:2.06.46:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.05`,
        fs:  `cpe:2.3:a:d-bus_project:d-bus:1.5.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.06`,
        fs:  `cpe:2.3:a:fork-cms:fork_cms:2.6.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.06:a`,
        fs:  `cpe:2.3:a:sendmail:sendmail:2.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.07`,
        fs:  `cpe:2.3:a:oracle:payroll:12.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.08`,
        fs:  `cpe:2.3:a:roderich_schupp:par-packer_module:0.978:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.08:a`,
        fs:  `cpe:2.3:a:google:chrome:26.0.1410.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.09`,
        fs:  `cpe:2.3:o:lacie:lac9000436u_firmware:2.3.0.014:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.09:a`,
        fs:  `cpe:2.3:a:softaculous:webuzo:2.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.10`,
        fs:  `cpe:2.3:a:liftweb:lift:2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.11`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.7:p220:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.12`,
        fs:  `cpe:2.3:a:dotnetnuke:dotnetnuke:6.1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.13`,
        fs:  `cpe:2.3:o:cisco:ios_xe:3.2sg.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.14`,
        fs:  `cpe:2.3:a:ati:catalyst_control_center_graphics_light:2009.211.2225.40219:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.15`,
        fs:  `cpe:2.3:h:emerson:network_power_avocent_mergepoint_unity_2016:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.16`,
        fs:  `cpe:2.3:a:ibm:tivoli_storage_manager:6.4.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.16:a`,
        fs:  `cpe:2.3:a:haudenschilt:family_connections_cms:2.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.17`,
        fs:  `cpe:2.3:a:devexpress:aspxfilemanager_control_for_webforms_and_mvc:11.2.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.18`,
        fs:  `cpe:2.3:h:cisco:rv130_vpn_router:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.18:a`,
        fs:  `cpe:2.3:a:osirix-viewer:osirix:3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.19`,
        fs:  `cpe:2.3:a:plone:plone:5.0:rc3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.20.0`,
        fs:  `cpe:2.3:a:goforandroid:go_email_widget:1.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.20.1`,
        fs:  `cpe:2.3:o:franklinfueling:ts-550_evo_firmware:2.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.20.2`,
        fs:  `cpe:2.3:a:apache:cloudstack:4.6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.20.3`,
        fs:  `cpe:2.3:a:fedoraproject:389_directory_server:1.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.20.4`,
        fs:  `cpe:2.3:a:puppetlabs:puppet_dashboard:1.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.20.5`,
        fs:  `cpe:2.3:a:collabnet:subversion:1.3.0-r17949:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adodb_project:adodb:5.20.6`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(33\)sxh6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adopt_o_pet_project:adopt_o_pet:0.1::~~~android~~`,
        fs:  `cpe:2.3:o:iij:seil\/b1_firmware:4.53:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adp4u:adp_agency_immobiliare:0.1::~~~android~~`,
        fs:  `cpe:2.3:a:alfresco:alfresco:6.x-1.2:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:adrian_chadd:rtg:0.7.4`,
        fs:  `cpe:2.3:a:campustv:campus_link_-_campus_tv_hkusu:2.2:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:adrian_chadd:rtg2:0.9`,
        fs:  `cpe:2.3:a:tigris:tortoisesvn:1.5.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adrian_chadd:rtg2:0.9.1`,
        fs:  `cpe:2.3:h:juniper:nsmexpress:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adrian_chadd:rtg2:0.9.2`,
        fs:  `cpe:2.3:o:hp:procurve_switch_software:r.11.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adrotateplugin:adrotate:3.9.::~free~wordpress~~~`,
        fs:  `cpe:2.3:a:tecnick:tcexam:11.0.012:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adrotateplugin:adrotate:3.9.::~pro~wordpress~~~`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.9.8-7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adrotateplugin:adrotate:3.9.1::~free~wordpress~~~`,
        fs:  `cpe:2.3:a:kde:kde_sc:4.6.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adrotateplugin:adrotate:3.9.1::~pro~wordpress~~~`,
        fs:  `cpe:2.3:a:enanocms:enano_cms:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adrotateplugin:adrotate:3.9.2::~free~wordpress~~~`,
        fs:  `cpe:2.3:o:codeaurora:android-msm:3.4.88:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adrotateplugin:adrotate:3.9.2::~pro~wordpress~~~`,
        fs:  `cpe:2.3:a:jenkins-ci:subversion-plugin:1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adrotateplugin:adrotate:3.9.3::~free~wordpress~~~`,
        fs:  `cpe:2.3:o:cisco:catos:8.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adrotateplugin:adrotate:3.9.3::~pro~wordpress~~~`,
        fs:  `cpe:2.3:a:postgresql:postgresql:9.2.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adrotateplugin:adrotate:3.9.4::~free~wordpress~~~`,
        fs:  `cpe:2.3:a:yahoo:messenger:9.0.0.2018:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adrotateplugin:adrotate:3.9.4::~pro~wordpress~~~`,
        fs:  `cpe:2.3:a:oracle:jdk:1.7.0:update_51:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adrotateplugin:adrotate:3.9.5::~free~wordpress~~~`,
        fs:  `cpe:2.3:a:roundup-tracker:roundup:0.5.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adrotateplugin:adrotate:3.9.5::~pro~wordpress~~~`,
        fs:  `cpe:2.3:h:hp:procurve_switch_2400m:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adrotateplugin:adrotate:3.9.6::~free~wordpress~~~`,
        fs:  `cpe:2.3:a:hitachi:electronic_form_workflow-developer_set:07-50:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adrotateplugin:adrotate:3.9.6::~pro~wordpress~~~`,
        fs:  `cpe:2.3:a:gnu:wget:1.10.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adrotateplugin:adrotate:3.9.7::~pro~wordpress~~~`,
        fs:  `cpe:2.3:a:icu-project:international_components_for_unicode:1.5:*:*:*:*:c\/c\&#43;\&#43;:*:*`,
    },
     
    {
        uri: `cpe:/a:adt-taxis:adt_taxis:6.0::~~~android~~`,
        fs:  `cpe:2.3:a:cisco:firewall_services_module_software:3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:adtran:aos:10.8.0`,
        fs:  `cpe:2.3:a:shavlik:netchk_protect:5.6.0.444:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:adtran:aos:10.8.1`,
        fs:  `cpe:2.3:o:juniper:junos:15.1x49:d35:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:adtran:netvanta_7060:-`,
        fs:  `cpe:2.3:a:owncloud:owncloud:6.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:adtran:netvanta_7100:-`,
        fs:  `cpe:2.3:a:wireshark:wireshark:1.10.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:advanced_forum_signatures_project:advanced_forum_signatures:2.0.4::~~~mybb~~`,
        fs:  `cpe:2.3:a:ntp:ntp:4.3.23:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:advancedsearchbar:advanced_search_bar:3.23`,
        fs:  `cpe:2.3:a:pgp:universal_server:2.10.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:advancedsearchbar:advanced_search_bar:3.25`,
        fs:  `cpe:2.3:a:s9y:serendipity:1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:advanceprotech:advanceware:-`,
        fs:  `cpe:2.3:a:gajennings:this:1.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:advantech:adamview:4.3`,
        fs:  `cpe:2.3:a:google:chrome:5.0.355.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:advantech:advantech_webaccess:5.0`,
        fs:  `cpe:2.3:a:igor_sysoev:nginx:0.3.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:advantech:advantech_webaccess:6.0`,
        fs:  `cpe:2.3:a:yahoo:messenger:9.0.0.922:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:advantech:advantech_webaccess:7.0`,
        fs:  `cpe:2.3:o:greenbone:greenbone_os:3.1.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:advantech:eki-122x_series_firmware:1.49`,
        fs:  `cpe:2.3:a:bitcoin_knots_project:bitcoin_knots:0.11.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:advantech:eki-1321:-`,
        fs:  `cpe:2.3:a:apache:traffic_server:2.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:advantech:eki-1321_series_firmware:1.96`,
        fs:  `cpe:2.3:a:jenkins:config_file_provider:2.15.6:*:*:*:*:jenkins:*:*`,
    },
     
    {
        uri: `cpe:/o:advantech:eki-1321_series_firmware:2015-10-06`,
        fs:  `cpe:2.3:a:c-ares_project:c-ares:1.9.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:advantech:eki-1322:-`,
        fs:  `cpe:2.3:a:virt-who_project:virt-who:0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:advantech:eki-1322_series_firmware:1.96`,
        fs:  `cpe:2.3:a:rocomotion:p_link:1.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:advantech:eki-1322_series_firmware:2015-10-06`,
        fs:  `cpe:2.3:a:ibm:websphere_application_server:8.5.5.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:advantech:eki-1361:-`,
        fs:  `cpe:2.3:a:genetechsolutions:pie-register:1.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:advantech:eki-1361_series_firmware:1.17`,
        fs:  `cpe:2.3:o:cisco:ios:12.3\(4\)xk4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:advantech:eki-1362:-`,
        fs:  `cpe:2.3:a:emc:control_center:6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:advantech:eki-1362_series_firmware:1.17`,
        fs:  `cpe:2.3:a:google:chrome:14.0.816.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:advantech:vesp211-232:-`,
        fs:  `cpe:2.3:a:activehelper:activehelper_livehelp_live_chat:2.6.1:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/o:advantech:vesp211-232_firmware:1.5.1`,
        fs:  `cpe:2.3:a:bftpd_project:bftpd:1.0.19:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:advantech:vesp211-232_firmware:1.7.2`,
        fs:  `cpe:2.3:h:linksys:befsr41_v3:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:advantech:vesp211-eu:-`,
        fs:  `cpe:2.3:a:joomlaboat:com_youtubegallery:3.9.3:*:*:*:*:joomla\!:*:*`,
    },
     
    {
        uri: `cpe:/o:advantech:vesp211-eu_firmware:1.7.2`,
        fs:  `cpe:2.3:h:rockwellautomation:rslogix_emulate_5000:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:advantech:webaccess:7.2`,
        fs:  `cpe:2.3:a:oracle:e-business_suite:12.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:advantech:webaccess:8.0`,
        fs:  `cpe:2.3:a:amos_benari:rbovirt:0.0.7:*:*:*:*:ruby:*:*`,
    },
     
    {
        uri: `cpe:/a:advantech:webaccess:8.1`,
        fs:  `cpe:2.3:a:wireshark:wireshark:2.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:adways:party_track_sdk:1.6.5::~~~iphone_os~~`,
        fs:  `cpe:2.3:a:avaya:libsafe:2.0.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aegirproject:hostmaster:6.x-2.0::~~~drupal~~`,
        fs:  `cpe:2.3:h:bluecoat:proxysg_sg510-10:-:-:full_proxy:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aegirproject:hostmaster:6.x-2.1::~~~drupal~~`,
        fs:  `cpe:2.3:a:winzip:winzip_command_line_support_add-on:2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aegirproject:hostmaster:6.x-2.2::~~~drupal~~`,
        fs:  `cpe:2.3:a:telaxus:epesi:1.8.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aegirproject:hostmaster:6.x-2.3::~~~drupal~~`,
        fs:  `cpe:2.3:a:coppermine-gallery:coppermine_photo_gallery:1.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aegirproject:hostmaster:6.x-3.0:beta1:~~~drupal~~`,
        fs:  `cpe:2.3:a:torproject:tor:0.2.0.12:alpha:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aeroexpress:aeroexpress:2.6.2::~~~android~~`,
        fs:  `cpe:2.3:a:oracle:enterprise_manager_grid_control:10.1.0.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aertherwide:exiftags:0.80`,
        fs:  `cpe:2.3:a:zingiri:zingiri_web_shop:2.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aertherwide:exiftags:0.90`,
        fs:  `cpe:2.3:o:oracle:linux:7:2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aertherwide:exiftags:0.91`,
        fs:  `cpe:2.3:h:cisco:catalyst_2900:lre_xl:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aertherwide:exiftags:0.92`,
        fs:  `cpe:2.3:a:symantec:brightmail_antispam:6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aertherwide:exiftags:0.93`,
        fs:  `cpe:2.3:a:cisco:secure_access_control_server:3.0.3:-:windows:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aertherwide:exiftags:0.94`,
        fs:  `cpe:2.3:o:citrix:netscaler_application_delivery_controller_firmware:10.1.127:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aertherwide:exiftags:0.95`,
        fs:  `cpe:2.3:a:joomla:joomla\!:3.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aertherwide:exiftags:0.96`,
        fs:  `cpe:2.3:a:redhat:lvm2:2.00.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aertherwide:exiftags:0.97`,
        fs:  `cpe:2.3:a:cakefoundation:cakephp:2.2.0:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aertherwide:exiftags:0.98`,
        fs:  `cpe:2.3:a:apache:http_server:2.4.22:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aertherwide:exiftags:0.99`,
        fs:  `cpe:2.3:a:lockon:ec-cube:2.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aertherwide:exiftags:1.00`,
        fs:  `cpe:2.3:a:ca:identity_manager:12.0.7.441:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aertherwide:exiftags:1.01`,
        fs:  `cpe:2.3:h:hp:color_laserjet_2500n:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:affinitycu:affinity_mobile_atm_locator:1.5::~~~android~~`,
        fs:  `cpe:2.3:a:ibm:tivoli_directory_server:6.2.0.37:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:afnetworking_project:afnetworking:2.5.2`,
        fs:  `cpe:2.3:a:git_for_windows_project:git_for_windows:1.8.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:afro-beat_project:afro-beat:0.2::~~~android~~`,
        fs:  `cpe:2.3:a:patrick_przybilla:addtoany:6.x-3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:afsinc:metalcasting_newsstand:3.12.0::~~~android~~`,
        fs:  `cpe:2.3:a:ovirt:ovirt:4.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aftab:tickfa:1.0.1`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.36:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:afterlifewitharchie:afterlife_with_archie:2.4.1::~~~android~~`,
        fs:  `cpe:2.3:a:digium:asterisk:13.7.0:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:afterlogic:mailsuite_pro:6.3`,
        fs:  `cpe:2.3:h:hp:jd429b:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ag-klettern-odenwald:ag_klettern_odenwald:1.2::~~~android~~`,
        fs:  `cpe:2.3:a:cisco:rv042g_dual_gigabit_wan_vpn_firmware:4.2.2.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:age_verification_project:age_verification:0.4`,
        fs:  `cpe:2.3:a:skype:skype:1.0.0.106:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agi:license_manager:3.0.3`,
        fs:  `cpe:2.3:o:bluecoat:cacheos:4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agi:planetary_data_supplement:9.0.0`,
        fs:  `cpe:2.3:a:henri_wahl:nagstamon:0.9.6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agi:stk:9.1.0`,
        fs:  `cpe:2.3:o:cisco:nx-os:5.2\(1\)n1\(4\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agi:stk%2fkml_utility_for_stk:9.1.0`,
        fs:  `cpe:2.3:a:webyog:sqlyog:7.1:-:community:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agi:stk_engine_deployment_resources:9.1.0`,
        fs:  `cpe:2.3:a:symantec:enterprise_security_manager:6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:1password:-:-:~~~android~~`,
        fs:  `cpe:2.3:a:ati:catalyst_control_center:2.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:1password:-:-:~~~iphone_os~~`,
        fs:  `cpe:2.3:a:adobe:acrobat_reader:10.1.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:1password:-:-:~~~windows~~`,
        fs:  `cpe:2.3:a:nodewords_project:nodewords:6.x-1.12:beta8:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:1password:3.0:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:sun:jdk:1.3.1_10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:1password:3.9.9`,
        fs:  `cpe:2.3:a:ibm:rational_clearquest:7.1.2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:-:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:ajaydsouza:contextual_related_posts:1.8.10.1:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:1.5.0:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:microsoft:index_server:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:1.5.1:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:cakephp:cakephp:2.5.0:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:1.5.2:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:opscode:chef:0.8.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:1.5.3:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:samba:samba:2.2.1a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:1.6.0:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:libguestfs:libguestfs:1.21.35:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:1.6.1:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:adobe:studio_mx:7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:1.6.2:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:rocomotion:pplog:3.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:1.6.3:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:mutt:mutt:1.5.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:1.6.4:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:gitlab:gitlab:2.1.0:*:*:*:enterprise:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:2.0.0:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:h:sun:fire_x4150_server:sw_2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:2.0.1:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:cisco:unified_computing_system_central_software:1.2\(1d\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:2.0.2:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:cisco:adaptive_security_appliance_software:8.4\(.6\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:2.0.3:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:o:cisco:vpn_3000_concentrator_series_software:4.7.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:2.0.4:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:siemens:compas:1.5:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:2.0.5:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:atlassian:crucible:2.6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:2.0.6:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:ibm:tivoli_storage_manager_fastback:6.1.5.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:2.0.8:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:sun:jsse:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:2.1.0:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:libpng:libpng:1.4.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:2.1.1:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:hp:status:60.0.155.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:2.1.2:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:f5:big-ip_link_controller:10.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:2.1.3:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:zohocorp:manageengine_password_manager_pro:5.1:*:*:*:managed_service_providers:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:2.1.4:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:cisco:secure_access_control_server_solution_engine:5.5.0.46.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:2.1.5:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:fusiondrupalthemes:fusion:6.x-1.0:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:2.1.6:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:redhat:hawtjni:1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:2.1.7:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:viscacha:viscacha:0.8:rc4_pl1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:2.1.8:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:o:juniper:junos:7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:2.1.9:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:ponsoftware:explzh:2.97:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:2.1.10:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:plugin-organizer_project:plugin-organizer:1.2.2:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:2.1.11:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:ibm:websphere_portal:7.0.0.2:-:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:2.1.12:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:tejimaya:openpne:3.4.11.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:2.1.13:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:ibm:iss_blackice_pc_protection:3.6:cbr:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:2.1.14:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:h:hp:jf807a:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:2.1.15:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:cisco:secure_desktop:3.5.841:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilebits:knox:2.2.0:-:~~~mac_os_x~~`,
        fs:  `cpe:2.3:a:trianglemicroworks:ansi_c_source_code_libraries:3.15.0000:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilefleet:fleetcommander:4.0`,
        fs:  `cpe:2.3:a:phpcheckz:phpcheckz:0.5.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilefleet:fleetcommander:4.08`,
        fs:  `cpe:2.3:a:search.cpan:libwww-perl:5.812:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agilefleet:fleetcommander_kiosk:4.0`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.4:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agnitum:outpost_security_suite:6.7.3.3063.452.0726:-:professional`,
        fs:  `cpe:2.3:a:visualsoftru:visual_quickbar:4.3.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agnitum:outpost_security_suite:7.0.3330.505.1221:beta:professional`,
        fs:  `cpe:2.3:a:featurific_for_wordpress_project:featurific-for-wordpress:1.3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:agrinsoft:argin_all_dvd_ripper:4.0`,
        fs:  `cpe:2.3:a:groups_communities_and_co_project:gcc:7.x-1.1:beta1:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:ahtty:ahtty:1.97.16::~~~android~~`,
        fs:  `cpe:2.3:a:mozilla:thunderbird:32.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aiadp:guess_the_pixel_character_quiz:1.3::~~~android~~`,
        fs:  `cpe:2.3:o:yamaha:rt140p:6.03.34:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aimluck:aipo:-`,
        fs:  `cpe:2.3:a:matt_johnston:dropbear_ssh_server:2013.58:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aimluck:aipo:4.0.1.0`,
        fs:  `cpe:2.3:a:octobercms:october_cms:1.0.260:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aimluck:aipo:4.0.2.0`,
        fs:  `cpe:2.3:a:gnome:glib:2.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aimluck:aipo:4.0.3.0`,
        fs:  `cpe:2.3:a:ibm:rational_test_workbench:8.6.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aimluck:aipo:4.0.4.0`,
        fs:  `cpe:2.3:a:plume-cms:plume_cms:1.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aimluck:aipo:4.0.5.0`,
        fs:  `cpe:2.3:a:cmu:cyrus_imap_server:2.3.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aimluck:aipo:5.0.0.0`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(24\)sv:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aimluck:aipo:5.0.1.0`,
        fs:  `cpe:2.3:a:redhat:libvirt:0.7.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aimluck:aipo:5.0.2.0`,
        fs:  `cpe:2.3:a:sharethis:sharethis:5.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aimluck:aipo:5.1`,
        fs:  `cpe:2.3:o:cisco:ios:12.0\(17\)s4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aimluck:aipo:5.1.0.1`,
        fs:  `cpe:2.3:a:nullsoft:winamp:5.666:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aimluck:aipo:5.1.1`,
        fs:  `cpe:2.3:a:hp:software_certificate_management_client:6.20.50:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aimluck:aipo:5.1.2`,
        fs:  `cpe:2.3:a:netiq:edirectory:8.8.6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aimluck:aipo:6.0.1`,
        fs:  `cpe:2.3:o:cisco:ios:12.0\(17\)sl:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aimluck:aipo-asp:4.0.4.0`,
        fs:  `cpe:2.3:a:ati:catalyst_control_center_help:2009.211.2224.40219:*:*:ja:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aimluck:aipo-asp:5.1.1`,
        fs:  `cpe:2.3:o:linux:linux_kernel:4.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aimluck:aipo-plus:5.1`,
        fs:  `cpe:2.3:a:jenkins-ci:subversion-plugin:1.41:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aimluck:aipo-plus:5.1.0.1`,
        fs:  `cpe:2.3:a:rocomotion:p_forum:1.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aimluck:aipo-plus:5.1.1`,
        fs:  `cpe:2.3:a:w1.fi:hostapd:0.6.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ains:foiaxpress:6.4.0.0`,
        fs:  `cpe:2.3:a:apache:subversion:0.19.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ains:foiaxpress_document_delivery_service:6.4.0.0`,
        fs:  `cpe:2.3:a:oracle:peoplesoft_enterprise:8.8:bundle11:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ains:foiaxpress_ocr_service:6.4.0.0`,
        fs:  `cpe:2.3:a:hp:director:45.4.157.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ains:foiaxpress_sync_service:6.4.0.0`,
        fs:  `cpe:2.3:a:cloudera:cloudera_cdh:cdh3:2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aipluspro:asset_inventory_plus:4.6.0`,
        fs:  `cpe:2.3:a:google:chrome:12.0.742.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:air-watch:air-watch_mobile_device_management_software:6.5`,
        fs:  `cpe:2.3:a:platinum_seo_project:platinum_seo_plugin:1.2.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:air_war_hero_project:air_war_hero:3::~~~android~~`,
        fs:  `cpe:2.3:a:foolabs:xpdf:0.5:a:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aircrack-ng:aircrack-ng:0.1`,
        fs:  `cpe:2.3:o:iij:seil_x1_firmware:3.35:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aircrack-ng:aircrack-ng:0.2`,
        fs:  `cpe:2.3:a:4you-studio:com_jphone:1.0:alpha3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aircrack-ng:aircrack-ng:0.2.1`,
        fs:  `cpe:2.3:a:vmware:player:1.0.5_build_56455:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aircrack-ng:aircrack-ng:0.3`,
        fs:  `cpe:2.3:a:travis_shirk:eyed3:0.6.0:-:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aircrack-ng:aircrack-ng:0.4`,
        fs:  `cpe:2.3:a:mobileeventguide:ids_2013:1.21:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:aircrack-ng:aircrack-ng:0.4.1`,
        fs:  `cpe:2.3:a:cisco:adaptive_security_appliance_software:9.0\(.3\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aircrack-ng:aircrack-ng:0.4.2`,
        fs:  `cpe:2.3:a:realnetworks:realarcade_installer:3.0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aircrack-ng:aircrack-ng:0.4.3`,
        fs:  `cpe:2.3:a:mozilla:firefox:4.0:beta6:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aircrack-ng:aircrack-ng:0.4.4`,
        fs:  `cpe:2.3:a:christos_zoulas:file:5.04:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aircrack-ng:aircrack-ng:0.5`,
        fs:  `cpe:2.3:a:freeipa:freeipa:4.4.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aircrack-ng:aircrack-ng:0.6`,
        fs:  `cpe:2.3:a:atlassian:jira:2.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aircrack-ng:aircrack-ng:0.6.1`,
        fs:  `cpe:2.3:a:matt_johnston:dropbear_ssh_server:0.48:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aircrack-ng:aircrack-ng:0.6.2`,
        fs:  `cpe:2.3:h:cisco:cs-mars:4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aircrack-ng:aircrack-ng:0.7`,
        fs:  `cpe:2.3:a:bananadance:banana_dance:0.9:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aircrack-ng:aircrack-ng:0.8`,
        fs:  `cpe:2.3:a:vmware:vsphere_host_update_utility:4.0.0.12305:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aircrack-ng:aircrack-ng:0.9`,
        fs:  `cpe:2.3:a:caucho:resin:3.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aircrack-ng:aircrack-ng:0.9.1`,
        fs:  `cpe:2.3:a:simon_rycroft:hashcash:6.x-2.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aircrack-ng:aircrack-ng:0.9.2`,
        fs:  `cpe:2.3:a:isc:bind:9.2.3:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aircrack-ng:aircrack-ng:0.9.3`,
        fs:  `cpe:2.3:a:cisco:ons_15216_optical_add_drop_multiplexer_software:2.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aircrack-ng:aircrack-ng:1.0`,
        fs:  `cpe:2.3:a:gameresort:stupid_zombies:1.12:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:aircrack-ng:aircrack-ng:1.0:beta1`,
        fs:  `cpe:2.3:a:netscape:navigator:4.04:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aircrack-ng:aircrack-ng:1.0:beta2`,
        fs:  `cpe:2.3:a:wireshark:wireshark:1.4.7:*:*:*:*:*:sparc:*`,
    },
     
    {
        uri: `cpe:/a:aircrack-ng:aircrack-ng:1.0:rc1`,
        fs:  `cpe:2.3:a:atlassian:jira:3.13.1:*:enterprise:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aircrack-ng:aircrack-ng:1.0:rc2`,
        fs:  `cpe:2.3:a:reasoncms:reasoncms:4.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aircrack-ng:aircrack-ng:1.0:rc3`,
        fs:  `cpe:2.3:a:igor_vlasenko:html-template-pro:0.26:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aircrack-ng:aircrack-ng:1.0:rc4`,
        fs:  `cpe:2.3:h:cisco:asa_5500_csc-ssm:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aircrack-ng:aircrack-ng:1.1`,
        fs:  `cpe:2.3:a:typo3:typo3:4.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aircrack-ng:aircrack-ng:1.2:beta1`,
        fs:  `cpe:2.3:a:boltwire:boltwire:3.4.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:airdroid:airdroid:1.0.1`,
        fs:  `cpe:2.3:a:atlassian:jira:3.2:*:professional:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:airdroid:airdroid:1.0.2`,
        fs:  `cpe:2.3:a:rubygems:rubygems:2.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:airdroid:airdroid:1.0.3`,
        fs:  `cpe:2.3:a:logitech:lws_motion_detection:13.20.1176.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:airdroid:airdroid:1.0.4`,
        fs:  `cpe:2.3:a:oracle:mysql_connector\/net:6.4.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:airdroid:airdroid:1.0.4:beta`,
        fs:  `cpe:2.3:a:ibm:rational_performance_tester:8.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:airdroid:airdroid:1.0.5`,
        fs:  `cpe:2.3:a:virt-who_project:virt-who:0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:airdroid:airdroid:1.0.6:beta`,
        fs:  `cpe:2.3:a:modssl:mod_ssl:2.8.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:airshowbuzz:asb.tv:1.2`,
        fs:  `cpe:2.3:a:intermesh:group-office:3.2.39:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:airties:air_6372:-`,
        fs:  `cpe:2.3:o:apple:mac_os_x_server:10.6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:airties:rt-210:-`,
        fs:  `cpe:2.3:o:sweex:lb000021_router_firmware:3.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:airties:rt-210_firmware:-`,
        fs:  `cpe:2.3:o:yahama:rt140i:6.03.28:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:airvana:hubbub_c1-600-rt:-`,
        fs:  `cpe:2.3:a:recurly:recurly_client_ruby:2.4.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aisquared:zoomtext:9.1`,
        fs:  `cpe:2.3:a:joomla:joomla\!:1.5.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.45.4`,
        fs:  `cpe:2.3:a:microsoft:systems_management_server:1.2:sp3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.45.4::~~~wordpress~~`,
        fs:  `cpe:2.3:o:cisco:catos:5.5\(10\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.45.5`,
        fs:  `cpe:2.3:a:wordpress:wordpress:1.2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.45.5::~~~wordpress~~`,
        fs:  `cpe:2.3:a:hitachi:electronic_form_workflow-developer_client_set:07-20-\/b:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.45.6`,
        fs:  `cpe:2.3:a:ultraedit:ultraedit:8.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.45.6::~~~wordpress~~`,
        fs:  `cpe:2.3:a:ssh:tectia_client_server_connector:5.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.45.7`,
        fs:  `cpe:2.3:a:cisco:unified_customer_voice_portal:8.0\(1\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.45.7::~~~wordpress~~`,
        fs:  `cpe:2.3:a:connect2id:nimbus_jose\&#43;jwt:3.8.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.45.8`,
        fs:  `cpe:2.3:a:ibm:lotus_domino:8.5.1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.45.8::~~~wordpress~~`,
        fs:  `cpe:2.3:a:ibm:sterling_selling_and_fulfillment_foundation:9.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.45.9`,
        fs:  `cpe:2.3:a:rubyonrails:ruby_on_rails:3.1.0:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.45.9::~~~wordpress~~`,
        fs:  `cpe:2.3:a:atlassian:bamboo:2.2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.46`,
        fs:  `cpe:2.3:a:homerelectric:hea_mobile:1.153.0034:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.46::~~~wordpress~~`,
        fs:  `cpe:2.3:a:sympa:sympa:3.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.46.1`,
        fs:  `cpe:2.3:a:cisco:network_admission_control_manager_and_server_system_software:4.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.46.1::~~~wordpress~~`,
        fs:  `cpe:2.3:a:rsgallery2:com_rsgallery2:1.10.10:alpha:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.46.2`,
        fs:  `cpe:2.3:a:redhat:mod_cluster:1.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.46.2::~~~wordpress~~`,
        fs:  `cpe:2.3:a:endeavorsystems:openfisma:2.4.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.46.3`,
        fs:  `cpe:2.3:h:huawei:nip2200:v100r001c00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.46.3::~~~wordpress~~`,
        fs:  `cpe:2.3:a:rack_project:rack:1.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.46.4`,
        fs:  `cpe:2.3:a:moinejf:abcm2ps:4.12.26:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.46.4::~~~wordpress~~`,
        fs:  `cpe:2.3:o:linux:linux_kernel:4.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.46.5`,
        fs:  `cpe:2.3:a:djangoproject:django:1.8.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.46.5::~~~wordpress~~`,
        fs:  `cpe:2.3:a:cisco:unified_communications_manager:9.9\(9\)st1.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.46.6`,
        fs:  `cpe:2.3:o:cisco:pix_firewall_software:6.1\(4\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.46.6::~~~wordpress~~`,
        fs:  `cpe:2.3:a:apple:apple_application_support:1.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.46.7`,
        fs:  `cpe:2.3:a:microsoft:visual_c\&#43;\&#43;:2002:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.46.7::~~~wordpress~~`,
        fs:  `cpe:2.3:a:werdswords:download_shortcode:0.1:-:-:-:-:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.46.8`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.5:p208:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.46.8::~~~wordpress~~`,
        fs:  `cpe:2.3:a:serv-u:serv-u:3.1.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.46.9`,
        fs:  `cpe:2.3:a:ibm:websphere_application_server:8.5.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.46.9::~~~wordpress~~`,
        fs:  `cpe:2.3:a:nagios:remote_plug_in_executor:2.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.47`,
        fs:  `cpe:2.3:a:tryton:tryton:2.6.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.47::~~~wordpress~~`,
        fs:  `cpe:2.3:o:iij:seil_x1_firmware:3.23:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.47.1`,
        fs:  `cpe:2.3:a:git-scm:git:0.10.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.47.1::~~~wordpress~~`,
        fs:  `cpe:2.3:a:skysa:skysa_app_bar_integration_plugin:1.30:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.47.2`,
        fs:  `cpe:2.3:a:postgresql:postgresql:8.3.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.47.2::~~~wordpress~~`,
        fs:  `cpe:2.3:a:mozilla:bugzilla:2.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.47.3`,
        fs:  `cpe:2.3:a:cisco:unified_computing_system:1.4\(3k\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.47.3::~~~wordpress~~`,
        fs:  `cpe:2.3:a:apache:tomcat:4.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.47.4`,
        fs:  `cpe:2.3:a:wptrafficanalyzer:trafficanalyzer:3.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.47.4::~~~wordpress~~`,
        fs:  `cpe:2.3:a:cisco:application_and_content_networking_system_software:4.2.9.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.47.5`,
        fs:  `cpe:2.3:a:bea:weblogic_server:5.1:sp2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.47.5::~~~wordpress~~`,
        fs:  `cpe:2.3:a:microsoft:lync:2010:*:attendant_x86:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.47.6`,
        fs:  `cpe:2.3:a:danielb:finder:6.x-1.0:alpha15:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.47.6::~~~wordpress~~`,
        fs:  `cpe:2.3:a:hp:nonstop_server_software:j06.07.00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.47.7`,
        fs:  `cpe:2.3:a:novell:zenworks_desktops:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.47.7::~~~wordpress~~`,
        fs:  `cpe:2.3:a:m2osw:tableofcontents:6.x-3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.47.8`,
        fs:  `cpe:2.3:h:hp:laserjet_flow_m525c:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.47.8::~~~wordpress~~`,
        fs:  `cpe:2.3:a:ratbox:ircd-ratbox:2.1.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.47.9`,
        fs:  `cpe:2.3:a:owncloud:owncloud:4.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.47.9::~~~wordpress~~`,
        fs:  `cpe:2.3:a:caucho:resin:3.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.48`,
        fs:  `cpe:2.3:a:webglimpse:webglimpse:2.16.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.48::~~~wordpress~~`,
        fs:  `cpe:2.3:o:cisco:ios_xe:3.3sg.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.48.1`,
        fs:  `cpe:2.3:a:webyog:sqlyog:6.52:-:enterprise:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.48.1::~~~wordpress~~`,
        fs:  `cpe:2.3:a:google:chrome:19.0.1084.25:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.48.2`,
        fs:  `cpe:2.3:a:netsweeper:netsweeper:3.1.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.48.2::~~~wordpress~~`,
        fs:  `cpe:2.3:a:inmatrix:zoom_player:8.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.48.3`,
        fs:  `cpe:2.3:a:md-systems:simplenews:6.x-1.0:rc6:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.48.3::~~~wordpress~~`,
        fs:  `cpe:2.3:a:f5:big-ip_advanced_firewall_manager:11.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.48.4`,
        fs:  `cpe:2.3:o:yamaha:rt107e:8.03.80:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.48.4::~~~wordpress~~`,
        fs:  `cpe:2.3:a:mozilla:camino:0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.48.5`,
        fs:  `cpe:2.3:o:cisco:spa8000_8-port_ip_telephony_gateway_firmware:6.1.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.48.5::~~~wordpress~~`,
        fs:  `cpe:2.3:a:audiosharescript:audioshare:2.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.48.6`,
        fs:  `cpe:2.3:a:pivotal_software:cloud_foundry_elastic_runtime:1.7.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.48.6::~~~wordpress~~`,
        fs:  `cpe:2.3:a:nagios:remote_plug_in_executor:2.0b5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.48.7`,
        fs:  `cpe:2.3:o:cisco:catos:2.1\(5\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.48.7::~~~wordpress~~`,
        fs:  `cpe:2.3:a:apache:groovy:2.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.48.8`,
        fs:  `cpe:2.3:a:microsoft:outlook_express:5.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.48.8::~~~wordpress~~`,
        fs:  `cpe:2.3:a:bea:jrockit_mission_control:3.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.48.9`,
        fs:  `cpe:2.3:a:openfabrics:enterprise_distribution:1.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.48.9::~~~wordpress~~`,
        fs:  `cpe:2.3:a:polarssl:polarssl:1.3.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.49`,
        fs:  `cpe:2.3:a:e-isg:crystal_installer:1.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.49::~~~wordpress~~`,
        fs:  `cpe:2.3:a:pivotx:pivotx:2.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.49.1`,
        fs:  `cpe:2.3:a:endonesia:endonesia:8.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.49.1::~~~wordpress~~`,
        fs:  `cpe:2.3:a:erik_hjortsberg:ember:0.5.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.49.2`,
        fs:  `cpe:2.3:a:cisco:webns:8.10.6.02:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.49.2::~~~wordpress~~`,
        fs:  `cpe:2.3:a:heimdal_project:heimdal:0.0u:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.49.3`,
        fs:  `cpe:2.3:a:djogjahotel:liburan_hemat:1:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.49.3::~~~wordpress~~`,
        fs:  `cpe:2.3:a:kolja_schleich:leaguemanager:2.5:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.49.4`,
        fs:  `cpe:2.3:a:samba:samba:4.0.26:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.49.4::~~~wordpress~~`,
        fs:  `cpe:2.3:a:ibm:websphere_extreme_scale_client:8.6.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.49.5`,
        fs:  `cpe:2.3:a:ssh:ssh:1.2.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.49.5::~~~wordpress~~`,
        fs:  `cpe:2.3:a:ocportal:ocportal:4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.49.6`,
        fs:  `cpe:2.3:o:cisco:ios:15.1\(2a\)sg7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.49.6::~~~wordpress~~`,
        fs:  `cpe:2.3:a:lttng:ust:0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.49.7`,
        fs:  `cpe:2.3:a:openvpn:openvpn:2.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.49.7::~~~wordpress~~`,
        fs:  `cpe:2.3:a:ibm:storwize_v5000_software:7.2.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.49.8`,
        fs:  `cpe:2.3:o:juniper:junos:12.3x50:d40:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.49.8::~~~wordpress~~`,
        fs:  `cpe:2.3:a:google:chrome:7.0.529.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.49.9`,
        fs:  `cpe:2.3:a:pulsecms:pulse_cms:1.2.3:-:basic:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.49.9::~~~wordpress~~`,
        fs:  `cpe:2.3:o:cisco:ios_xe:3.8s_base:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.50`,
        fs:  `cpe:2.3:h:linksys:rv082:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof-security:.50::~~~wordpress~~`,
        fs:  `cpe:2.3:a:ssh:tectia_client:4.3.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof_security:.44::~~~wordpress~~`,
        fs:  `cpe:2.3:a:netscape:navigator:6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof_security:.44.1::~~~wordpress~~`,
        fs:  `cpe:2.3:a:magzter:karaf_magazin:3:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof_security:.45::~~~wordpress~~`,
        fs:  `cpe:2.3:a:mailpoet:mailpoet_newsletters:2.5.9.2:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof_security:.45.1::~~~wordpress~~`,
        fs:  `cpe:2.3:a:thulasidas:easy-adsense-lite:2.63:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof_security:.45.2::~~~wordpress~~`,
        fs:  `cpe:2.3:a:pixologic:zbrush:4.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof_security:.45.3::~~~wordpress~~`,
        fs:  `cpe:2.3:a:bftpd_project:bftpd:1.0.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof_security:.50.1::~~~wordpress~~`,
        fs:  `cpe:2.3:a:apache:wicket:6.19.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof_security:.50.2::~~~wordpress~~`,
        fs:  `cpe:2.3:a:axigen:axigen_mail_server:8.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof_security:.50.3::~~~wordpress~~`,
        fs:  `cpe:2.3:a:apache:spamassassin:3.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof_security:.50.4::~~~wordpress~~`,
        fs:  `cpe:2.3:a:atlassian:jira:3.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof_security:.50.5::~~~wordpress~~`,
        fs:  `cpe:2.3:o:arubanetworks:arubaos:6.3.1.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof_security:.50.6::~~~wordpress~~`,
        fs:  `cpe:2.3:a:featurific_for_wordpress_project:featurific-for-wordpress:1.2.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof_security:.50.7::~~~wordpress~~`,
        fs:  `cpe:2.3:a:stuart_watret:sms_adminconsole_additions:1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof_security:.50.8::~~~wordpress~~`,
        fs:  `cpe:2.3:a:pantech:pc_usb_modem_software:3.0.4.823:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof_security:.50.9::~~~wordpress~~`,
        fs:  `cpe:2.3:a:cisco:ips_sensor_software:7.1\(4\)e4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ait-pro:bulletproof_security:.51::~~~wordpress~~`,
        fs:  `cpe:2.3:a:f5:big-ip_policy_enforcement_manager:11.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajax-random-post_project:ajax-random-post:2.00::~~~wordpress~~`,
        fs:  `cpe:2.3:a:nginx:nginx:1.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajax_timeline_project:ajax_timeline:7.x-1.0::~~~drupal~~`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:7.0.6-6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:2.0`,
        fs:  `cpe:2.3:a:mozilla:firefox:1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:2.3.3`,
        fs:  `cpe:2.3:a:wptrafficanalyzer:trafficanalyzer:1.7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:2.3.4`,
        fs:  `cpe:2.3:a:cisco:unified_communications_manager:10.5\(2.10000.5\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:2.5`,
        fs:  `cpe:2.3:a:wpeasycart:wp_easycart:2.1.27:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:2.5.4`,
        fs:  `cpe:2.3:a:jenkins:owasp_dependency-check:1.2.9:*:*:*:*:jenkins:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:2.5.5`,
        fs:  `cpe:2.3:a:xnau:participants_database:1.5.4.1:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:2.6.0`,
        fs:  `cpe:2.3:a:nice:recording_express:6.3.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:2.7.1`,
        fs:  `cpe:2.3:a:mozilla:seamonkey:2.25:-:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:2.7.2`,
        fs:  `cpe:2.3:a:cisco:network_admission_control_manager_and_server_system_software:3.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:2.7.3`,
        fs:  `cpe:2.3:a:jrsoftware:mousetool:3.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:3.0`,
        fs:  `cpe:2.3:a:ibm:rational_performance_tester:8.5.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:3.0.1`,
        fs:  `cpe:2.3:a:google:chrome:9.0.577.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:3.0.2`,
        fs:  `cpe:2.3:a:adobe:photoshop:18.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:3.0.3`,
        fs:  `cpe:2.3:o:cisco:ios:12.3\(2\)jk1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:3.1`,
        fs:  `cpe:2.3:o:siemens:scalance_w700_series_firmware:4.5.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:3.1.1`,
        fs:  `cpe:2.3:a:x.org:x11:6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:3.2`,
        fs:  `cpe:2.3:a:cloudbees:jenkins:1.403:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:3.2.1`,
        fs:  `cpe:2.3:o:cisco:universal_small_cell_firmware:r3.4_base:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:3.2.2`,
        fs:  `cpe:2.3:a:puppetlabs:puppet_dashboard:1.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:3.2.3`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(13\)t:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:3.2.4`,
        fs:  `cpe:2.3:a:x.org:x11:6.8.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:3.2.5`,
        fs:  `cpe:2.3:a:cisco:carrier_routing_system:3.3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:3.3.2`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.30:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:3.3.3`,
        fs:  `cpe:2.3:a:businessobjects:crystal_reports_.net_runtime:10.0.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:3.3.4`,
        fs:  `cpe:2.3:a:atlassian:confluence:3.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:3.3.5`,
        fs:  `cpe:2.3:a:adobe:download_manager:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:4.0`,
        fs:  `cpe:2.3:a:apache:cxf:2.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:4.0.1`,
        fs:  `cpe:2.3:o:yamaha:rtx2000:7.01.41:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:4.0.2`,
        fs:  `cpe:2.3:a:phpmyadmin:phpmyadmin:2.11.6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:4.0.3`,
        fs:  `cpe:2.3:o:cisco:ons_15454sdh:4.0\(1\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:4.0.4`,
        fs:  `cpe:2.3:a:namazu:namazu:2.0.6:rc5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:4.2.0`,
        fs:  `cpe:2.3:a:oracle:database_server:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:4.2.2`,
        fs:  `cpe:2.3:a:xfree86:x11perf:1.5.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:4.2.3`,
        fs:  `cpe:2.3:a:cloudbees:jenkins:1.409.2:*:lts:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:5.0.0`,
        fs:  `cpe:2.3:a:thecartpress:thecartpress:1.1.8.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:5.0.1`,
        fs:  `cpe:2.3:h:tp-link:tl-wrd841n_\(10.0\):-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:5.0.2`,
        fs:  `cpe:2.3:a:tryton:tryton:2.6.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:5.0.3`,
        fs:  `cpe:2.3:a:ssh:tectia_client:5.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaxplorer:ajaxplorer:5.0.4`,
        fs:  `cpe:2.3:o:linux:kernel:2.6.32.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.0::~~~wordpress~~`,
        fs:  `cpe:2.3:h:siemens:simatic_s7-400_cpu_416-3_pn\/dp:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.1::~~~wordpress~~`,
        fs:  `cpe:2.3:a:octobercms:october_cms:1.0.363:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.1.1::~~~wordpress~~`,
        fs:  `cpe:2.3:a:getswish:swish_payments:2.0:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.2::~~~wordpress~~`,
        fs:  `cpe:2.3:h:cisco:spa_301_1_line_ip_phone:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.2.1::~~~wordpress~~`,
        fs:  `cpe:2.3:a:ibm:tririga_application_platform:3.5.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.2.2::~~~wordpress~~`,
        fs:  `cpe:2.3:a:perl:perl:5.10.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.3::~~~wordpress~~`,
        fs:  `cpe:2.3:a:x:libxcb:1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.3.1::~~~wordpress~~`,
        fs:  `cpe:2.3:a:typo3:typo3:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.4::~~~wordpress~~`,
        fs:  `cpe:2.3:a:smart-flv_plugin_project:smart-flv:-:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.4.1::~~~wordpress~~`,
        fs:  `cpe:2.3:a:bundler:bundler:1.1:pre.8:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.4.2::~~~wordpress~~`,
        fs:  `cpe:2.3:a:apple:quicktime:7.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.5::~~~wordpress~~`,
        fs:  `cpe:2.3:a:cisco:adaptive_security_appliance_software:8.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.5.1::~~~wordpress~~`,
        fs:  `cpe:2.3:a:jenkins:config_file_provider:2.15:*:*:*:*:jenkins:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.5.2::~~~wordpress~~`,
        fs:  `cpe:2.3:a:guitartuna:guitar_tuner_free_-_guitartuna:2.4.5:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.6::~~~wordpress~~`,
        fs:  `cpe:2.3:o:hp:procurve_switch_software:h.07.45:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.6.1::~~~wordpress~~`,
        fs:  `cpe:2.3:a:php:php:3.0.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.6.2::~~~wordpress~~`,
        fs:  `cpe:2.3:o:cisco:ios:12.2yr:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.6.3::~~~wordpress~~`,
        fs:  `cpe:2.3:a:gitlab:gitlab:4.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.6.4::~~~wordpress~~`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.5.35:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.6.5::~~~wordpress~~`,
        fs:  `cpe:2.3:a:apple:cups:1.1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.7::~~~wordpress~~`,
        fs:  `cpe:2.3:a:swi-prolog:swi-prolog:5.6.52:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.7.1::~~~wordpress~~`,
        fs:  `cpe:2.3:a:ibm:tivoli_application_dependency_discovery_manager:7.1.2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.7.2::~~~wordpress~~`,
        fs:  `cpe:2.3:a:adbnewssender_project:adbnewssender:2.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.7.3::~~~wordpress~~`,
        fs:  `cpe:2.3:o:cisco:ios:12.0\(16\)s10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.8::~~~wordpress~~`,
        fs:  `cpe:2.3:a:namazu:namazu:2.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.8.1::~~~wordpress~~`,
        fs:  `cpe:2.3:a:wimleers:cdn:7.x-2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.8.2::~~~wordpress~~`,
        fs:  `cpe:2.3:a:polarssl:polarssl:1.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.8.3::~~~wordpress~~`,
        fs:  `cpe:2.3:a:hp:data_protector:7.03:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.8.4::~~~wordpress~~`,
        fs:  `cpe:2.3:a:symantec:vrtssfmh:3.0.357:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.8.5::~~~wordpress~~`,
        fs:  `cpe:2.3:a:deliciousdays:cforms:5.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.8.6::~~~wordpress~~`,
        fs:  `cpe:2.3:a:aircrack-ng:aircrack-ng:0.9.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.8.7::~~~wordpress~~`,
        fs:  `cpe:2.3:a:redhat:jboss_operations_network:3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.8.8::~~~wordpress~~`,
        fs:  `cpe:2.3:a:diversesolutions:dsidxpress_idx_plugin:2.0.11:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.8.9::~~~wordpress~~`,
        fs:  `cpe:2.3:a:synology:cloud_station:2.0-2291:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.8.9.1::~~~wordpress~~`,
        fs:  `cpe:2.3:a:cisco:application_and_content_networking_system_software:5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.8.10::~~~wordpress~~`,
        fs:  `cpe:2.3:a:skype:skype:0.95.0.11:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.8.10.1::~~~wordpress~~`,
        fs:  `cpe:2.3:a:david_shadoff:mednafen:0.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.8.10.2::~~~wordpress~~`,
        fs:  `cpe:2.3:a:vanillaforums:vanilla_forums:2.0.18:b4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.9::~~~wordpress~~`,
        fs:  `cpe:2.3:a:kajona:kajona:4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.9.0.1::~~~wordpress~~`,
        fs:  `cpe:2.3:a:google:chrome:42.0.2311.87:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajaydsouza:contextual_related_posts:1.9.1::~~~wordpress~~`,
        fs:  `cpe:2.3:a:openx:openx:2.6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajenti:ajenti:1.2.0`,
        fs:  `cpe:2.3:a:atheme:atheme:6.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajenti:ajenti:1.2.1`,
        fs:  `cpe:2.3:a:ovirt:ovirt:3.6.7.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajenti:ajenti:1.2.2`,
        fs:  `cpe:2.3:a:usebb:usebb:0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajenti:ajenti:1.2.3`,
        fs:  `cpe:2.3:o:cisco:catos:6.1\(1b\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajenti:ajenti:1.2.4`,
        fs:  `cpe:2.3:a:ntp:ntp:4.3.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajenti:ajenti:1.2.5`,
        fs:  `cpe:2.3:a:apple:quicktime:7.7.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajenti:ajenti:1.2.6`,
        fs:  `cpe:2.3:a:google:chrome:16.0.897.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajenti:ajenti:1.2.7`,
        fs:  `cpe:2.3:a:caucho:resin:4.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajenti:ajenti:1.2.8`,
        fs:  `cpe:2.3:a:sendmail:sendmail:8.7.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajenti:ajenti:1.2.9`,
        fs:  `cpe:2.3:a:symantec:client_security:3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajenti:ajenti:1.2.10`,
        fs:  `cpe:2.3:a:google:chrome:12.0.719.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajenti:ajenti:1.2.11.2`,
        fs:  `cpe:2.3:a:kolja_schleich:leaguemanager:3.0.1:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:ajenti:ajenti:1.2.12`,
        fs:  `cpe:2.3:a:red-gate:sql_data_generator:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajenti:ajenti:1.2.13`,
        fs:  `cpe:2.3:a:zooeffect:zooeffect:1.02:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajenti:ajenti:1.2.14`,
        fs:  `cpe:2.3:a:mortbay:jetty:2.2.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajenti:ajenti:1.2.15`,
        fs:  `cpe:2.3:a:jenkins:config_file_provider:2.3:*:*:*:*:jenkins:*:*`,
    },
     
    {
        uri: `cpe:/a:ajenti:ajenti:1.2.16`,
        fs:  `cpe:2.3:a:cisco:prime_infrastructure:2.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajenti:ajenti:1.2.17`,
        fs:  `cpe:2.3:a:mbtcreations:100_beauty_tips:1.1:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:ajenti:ajenti:1.2.18`,
        fs:  `cpe:2.3:a:haxx:libcurl:7.19.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajenti:ajenti:1.2.19`,
        fs:  `cpe:2.3:a:mozilla:netscape_portable_runtime:4.6.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajenti:ajenti:1.2.20`,
        fs:  `cpe:2.3:o:iodata:rockdisk_firmware:1.05c-2.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajenti:ajenti:1.2.21`,
        fs:  `cpe:2.3:a:webform_project:webform:7.x-3.14:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:ajenti:ajenti:1.2.21.7`,
        fs:  `cpe:2.3:a:safenet-inc:protectdrive:9.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajsquare:aj_auction_pro-oopd:2.0`,
        fs:  `cpe:2.3:a:atlassian:jira:5.2.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajsquare:aj_auction_pro-oopd:3.0`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.14.51:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ajsquare:aj_auction_pro-oopd:3.1`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.7:p36:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:akamai:kona_site_defender_service:-`,
        fs:  `cpe:2.3:a:apple:safari:4.0.0b:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:akamai:netsession:1.9.3.1`,
        fs:  `cpe:2.3:a:cisco:finesse:9.0\(1\)_base:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aker:secure_mail_gateway:2.5.2`,
        fs:  `cpe:2.3:a:ibm:tivoli_directory_server:6.2.0.35:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:akiva:webboard:2.90:beta`,
        fs:  `cpe:2.3:a:emc:avamar_plugin:5.0:-:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:akiva:webboard:8.0`,
        fs:  `cpe:2.3:a:oracle:mysql_connector\/j:5.1.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:akiva:webboard:8.0:sr1`,
        fs:  `cpe:2.3:a:dell:openmanage_server_agent:4.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:akiva:webboard:9.0`,
        fs:  `cpe:2.3:a:gnome:eye_of_gnome:3.20.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:akronchildrens:care4kids:1.03::~~~android~~`,
        fs:  `cpe:2.3:a:imagely:nextgen_gallery:1.8.2:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:aktiv:phantom:2.5.1`,
        fs:  `cpe:2.3:a:ibotta:ibotta_-_better_than_coupons.:2.5.1:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:akunamachata:paypal_express_module:-`,
        fs:  `cpe:2.3:a:apache:xerces-c\&#43;\&#43;:2.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:al-ahsa_news_project:al-ahsa_news:2::~~~android~~`,
        fs:  `cpe:2.3:a:cisco:telepresence_tc_software:7.1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:al_3azmi:ce4arab_market:0.12.13093.40460::~~~android~~`,
        fs:  `cpe:2.3:o:google:chrome_os:25.0.1364.125:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:al_jazeera_project:al_jazeera:6.0::~~~android~~`,
        fs:  `cpe:2.3:o:cisco:ios:12.3\(9a\)bc2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alaaliwat:www.alaaliwat.com:4.9::~~~android~~`,
        fs:  `cpe:2.3:o:google:chrome:21.0.1180.57:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aladdin:ghostview:2.1`,
        fs:  `cpe:2.3:a:ibm:emptoris_sourcing_portfolio:10.0.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aladdin:hasp_srm_run-time:3.10.1.6921`,
        fs:  `cpe:2.3:a:boonex:dolphin:7.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alan_kennington:mod_bwshare:0.1.9`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.338:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alan_kennington:mod_bwshare:0.2.0`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.9:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alawar:motor_town%3a_machine_soul_free:1.1::~~~android~~`,
        fs:  `cpe:2.3:o:codeaurora:android-msm:3.14.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:albasit_artes_y_danza_project:albasit_artes_y_danza:1.2::~~~android~~`,
        fs:  `cpe:2.3:a:nomachine:nx_web_companion:3.0.0-2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alberto_pittoni:alguest:1.0`,
        fs:  `cpe:2.3:a:percona:percona_server:5.6.23-72.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alberto_pittoni:alguest:1.1:c-patched`,
        fs:  `cpe:2.3:o:sun:solaris:5.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alberto_trujillo_gonzalez:protest:6.x-1.0`,
        fs:  `cpe:2.3:a:xmlsoft:libxml2:1.8.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alberto_trujillo_gonzalez:protest:6.x-1.x:dev`,
        fs:  `cpe:2.3:a:rocomotion:pplog_2:3.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alberto_trujillo_gonzalez:protest:7.x-1.0`,
        fs:  `cpe:2.3:a:vbulletin:vbulletin:4.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alberto_trujillo_gonzalez:protest:7.x-1.x:dev`,
        fs:  `cpe:2.3:a:memcache_project:memcache:5.x-1.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alcatel:adsl_modem_1000:-`,
        fs:  `cpe:2.3:a:horde:groupware:1.2.9:*:webmail:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:alcatel:aos:-`,
        fs:  `cpe:2.3:a:apache:apr-util:1.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:alcatel:aos:5.1.1`,
        fs:  `cpe:2.3:a:mediawiki:mediawiki:1.15.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:alcatel:aos:5.1.6.463`,
        fs:  `cpe:2.3:a:chatelao:php_address_book:3.2.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:alcatel:aos:5.4.1.429`,
        fs:  `cpe:2.3:a:obm:open_business_management:2.3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:alcatel:aos:6.1.3.965`,
        fs:  `cpe:2.3:a:joomla:joomla\!:2.5.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:alcatel:aos:6.3.1.966`,
        fs:  `cpe:2.3:a:jruby:jruby:1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alcatel:omnistack:-`,
        fs:  `cpe:2.3:a:vmware:tools:8.8.0.471780:*:*:*:windows:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alcatel:omniswitch:-`,
        fs:  `cpe:2.3:a:ibm:smart_analytics_system_7700:2.0.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alcatel:omniswitch_7800:-`,
        fs:  `cpe:2.3:a:bestpractical:rt:3.4.2:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alcatel:speed_touch_adsl_modem:-`,
        fs:  `cpe:2.3:a:redhat:automatic_bug_reporting_tool:1.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alcatel:speed_touch_home:-`,
        fs:  `cpe:2.3:a:qemu:qemu:0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alcatel:speed_touch_home:khdsaa.108`,
        fs:  `cpe:2.3:o:yahama:rt200i:2.02.31:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alcatel:speed_touch_home:khdsaa.132`,
        fs:  `cpe:2.3:a:adbnewssender_project:adbnewssender:2.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alcatel:speed_touch_home:khdsaa.133`,
        fs:  `cpe:2.3:a:schneider-electric:unity_pro:7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alcatel:speed_touch_home:khdsaa.134`,
        fs:  `cpe:2.3:a:inkscape:inkscape:0.40:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alcatel:speedtouch_7g_router:-`,
        fs:  `cpe:2.3:a:zarafa:zarafa_collaboration_platform:7.0.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alcatel:switched_firewall:-`,
        fs:  `cpe:2.3:a:ovirt:ovirt:3.3.2:beta1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alcatel-lucent:omniaccess_wireless:-`,
        fs:  `cpe:2.3:a:netsweeper:netsweeper:3.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alcatel-lucent:omnipcx:-`,
        fs:  `cpe:2.3:a:sybase:sql_anywhere_studio:9.0.2.3702:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alcatel-lucent:omnipcx:7.1`,
        fs:  `cpe:2.3:a:activestate:activepython:2.6.5.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alcatel-lucent:omnitouch_8400_instant_communications_suite:6.7.2`,
        fs:  `cpe:2.3:a:cisco:adaptive_security_device_manager_idm_launcher:1.5.49:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alcatel-lucent:omnitouch_8400_instant_communications_suite:6.7.3`,
        fs:  `cpe:2.3:a:atlassian:confluence:5.9.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alcatel-lucent:omnitouch_8460_advanced_communication_server:9.0`,
        fs:  `cpe:2.3:a:redhat:jboss_enterprise_application_platform:5.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alcatel-lucent:omnitouch_8460_advanced_communication_server:9.1`,
        fs:  `cpe:2.3:a:splunk:splunk:3.4.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alcatel-lucent:omnitouch_8660_my_teamwork:6.6`,
        fs:  `cpe:2.3:h:apple:iphone:1.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alcatel-lucent:omnitouch_8660_my_teamwork:6.7`,
        fs:  `cpe:2.3:a:cmsmadesimple:cms_made_simple:1.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alcatel-lucent:omnitouch_8670_automated_delivery_message_delivery_system:6.6`,
        fs:  `cpe:2.3:a:moinmo:moinmoin:1.6.0:beta2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alcatel-lucent:omnitouch_8670_automated_delivery_message_delivery_system:6.7`,
        fs:  `cpe:2.3:a:google:chrome:4.1.249.1055:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alcatel-lucent:omnivista:-`,
        fs:  `cpe:2.3:a:intel:pro_network_connections:12.2.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alcatel-lucent:voice_mail_system:-`,
        fs:  `cpe:2.3:a:mediafront:mediafront:7.x-1.0:-:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alejandro_garza:apachesolr_autocomplete:6.x-1.0`,
        fs:  `cpe:2.3:a:moinejf:abcm2ps:6.3.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alejandro_garza:apachesolr_autocomplete:6.x-1.1`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.2.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alejandro_garza:apachesolr_autocomplete:6.x-1.2`,
        fs:  `cpe:2.3:a:kde:kde_sc:2.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alejandro_garza:apachesolr_autocomplete:6.x-1.3`,
        fs:  `cpe:2.3:a:mit:kerberos:5-1.11.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alejandro_garza:apachesolr_autocomplete:6.x-1.x:dev`,
        fs:  `cpe:2.3:a:blinkwebeffects:social-media-widget:2.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alejandro_garza:apachesolr_autocomplete:6.x-1.x-dev`,
        fs:  `cpe:2.3:a:google:chrome:17.0.954.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alejandro_garza:apachesolr_autocomplete:7.x-1.0`,
        fs:  `cpe:2.3:a:feep:libtar:1.2.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alejandro_garza:apachesolr_autocomplete:7.x-1.1`,
        fs:  `cpe:2.3:o:yahama:srt100:10.00.46:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alejandro_garza:apachesolr_autocomplete:7.x-1.2`,
        fs:  `cpe:2.3:a:imember360:imember360:3.5.012:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:alejandro_garza:apachesolr_autocomplete:7.x-1.x:dev`,
        fs:  `cpe:2.3:a:krzysztof_kowalczyk:sumatrapdf:0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alentum:weblog_expert:6.3`,
        fs:  `cpe:2.3:a:rom_walton:boinc:7.0.27:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alephsystem:cms_ariadna:1.1`,
        fs:  `cpe:2.3:h:3com:switch_4005:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_barth:data:6.x-1.0:alpha1`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.8.9-0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_barth:data:6.x-1.0:alpha10`,
        fs:  `cpe:2.3:a:apache:batik:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_barth:data:6.x-1.0:alpha11`,
        fs:  `cpe:2.3:h:cisco:nexus_7000_9-slot:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_barth:data:6.x-1.0:alpha12`,
        fs:  `cpe:2.3:a:ulli_horlacher:fex:20120709:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_barth:data:6.x-1.0:alpha13`,
        fs:  `cpe:2.3:a:andy_armstrong:cgi.pm:3.19:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_barth:data:6.x-1.0:alpha14`,
        fs:  `cpe:2.3:a:fogproject:fog:0.32:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_barth:data:6.x-1.0:alpha2`,
        fs:  `cpe:2.3:a:gnu:glibc:2.3.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_barth:data:6.x-1.0:alpha3`,
        fs:  `cpe:2.3:a:memcache_project:memcache:6.x-1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_barth:data:6.x-1.0:alpha4`,
        fs:  `cpe:2.3:o:cisco:pix_firewall_software:6.2\(3\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_barth:data:6.x-1.0:alpha5`,
        fs:  `cpe:2.3:a:citrix:metaframe_presentation_server_client:9.150.39151:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_barth:data:6.x-1.0:alpha6`,
        fs:  `cpe:2.3:a:opnet:sp_guru_network_planner:16.0.0.10106:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_barth:data:6.x-1.0:alpha7`,
        fs:  `cpe:2.3:a:boatmob:boat_browser_mini:2.8:-:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_barth:data:6.x-1.0:alpha8`,
        fs:  `cpe:2.3:a:keil-software:photokorn_gallery:1.81:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_barth:data:6.x-1.0:alpha9`,
        fs:  `cpe:2.3:a:botan_project:botan:1.8.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_barth:data:6.x-1.x:dev`,
        fs:  `cpe:2.3:a:apple:itunes:6.0.2:-:windows:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_barth:data:7.x-1.0:alpha1`,
        fs:  `cpe:2.3:a:pragyan_cms_project:pragyan_cms:3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_barth:data:7.x-1.0:alpha2`,
        fs:  `cpe:2.3:a:oracle:enterprise_manager_grid_control:10.1.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_barth:data:7.x-1.x:dev`,
        fs:  `cpe:2.3:a:typo3:flow:1.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_feinman:iso_recorder:3.1.0`,
        fs:  `cpe:2.3:a:typo3:typo3:6.2.0:beta1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.0.1`,
        fs:  `cpe:2.3:a:atlassian:confluence:5.8.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.0.2`,
        fs:  `cpe:2.3:a:botan_project:botan:1.9.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.0.3`,
        fs:  `cpe:2.3:o:iodata:hdl-a_firmware:1.07:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.0.4`,
        fs:  `cpe:2.3:a:oracle:mysql:5.1.54:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.0.5`,
        fs:  `cpe:2.3:a:ffmpeg:ffmpeg:0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.0.6`,
        fs:  `cpe:2.3:h:cisco:sf300-08:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.0.7`,
        fs:  `cpe:2.3:a:mozilla:firefox_esr:38.2.1:*:*:*:*:*:x64:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.0.8`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.8:rc3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.0.9`,
        fs:  `cpe:2.3:a:sun:openoffice.org:3.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.0.10`,
        fs:  `cpe:2.3:o:linux:kernel:2.6.34.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.0.11`,
        fs:  `cpe:2.3:a:mozilla:network_security_services:3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.0.12`,
        fs:  `cpe:2.3:a:acquia:mollom:6.x-2.14:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.1.0`,
        fs:  `cpe:2.3:a:php-fusion:team_impact_ti_blog_system_module:_nil_:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.1.1`,
        fs:  `cpe:2.3:a:cisco:videoscape_delivery_system_for_internet_streamer:3.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.1.2`,
        fs:  `cpe:2.3:a:digium:asterisk:1.2.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.1.3`,
        fs:  `cpe:2.3:a:bayashi:dopvstar\*:0090:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.1.4`,
        fs:  `cpe:2.3:a:cloudfoundry:cf-release:228:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.1.5`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.19.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.1.6`,
        fs:  `cpe:2.3:a:oracle:database_server:7.0.64:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.1.7`,
        fs:  `cpe:2.3:a:zarafa:zarafa_collaboration_platform:7.0.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.1.8`,
        fs:  `cpe:2.3:a:rocomotion:p_forum:1.08:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.1.9`,
        fs:  `cpe:2.3:a:ibm:tivoli_directory_server:6.3.0.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.1.10`,
        fs:  `cpe:2.3:a:topdesk:topdesk:7.03.019:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.2.0`,
        fs:  `cpe:2.3:a:alps:touch_pad_driver:6.0.302.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.2.1`,
        fs:  `cpe:2.3:o:suse:suse_linux:6.1:alpha:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.2.2`,
        fs:  `cpe:2.3:a:fetchmail:fetchmail:6.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.2.3`,
        fs:  `cpe:2.3:h:moxa:iologic_e2240:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.2.4`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.30.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.3.1`,
        fs:  `cpe:2.3:a:cisco:cloud_portal:9.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.3.2`,
        fs:  `cpe:2.3:a:caucho:resin:3.1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.3.3`,
        fs:  `cpe:2.3:o:ibm:aix:5.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.3.4`,
        fs:  `cpe:2.3:a:google:chrome:32.0.1678.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.3.5`,
        fs:  `cpe:2.3:a:hotbar:hbtools:4.7.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.3.6`,
        fs:  `cpe:2.3:o:apple:mac_os_x:10.7.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.3.7`,
        fs:  `cpe:2.3:a:bestpractical:rt:3.7.80:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.3.8`,
        fs:  `cpe:2.3:a:sync:oxygen_xml_editor:10.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.3.9`,
        fs:  `cpe:2.3:a:git_for_windows_project:git_for_windows:1.8.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.3.10`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.414:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.3.11`,
        fs:  `cpe:2.3:a:ibm:websphere_extreme_scale:8.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.3.12`,
        fs:  `cpe:2.3:a:digium:asterisk:1.8.11.0:-:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.3.13`,
        fs:  `cpe:2.3:a:magzter:bespoke:3:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.3.14`,
        fs:  `cpe:2.3:a:lesterchan:wp-postviews:1.62:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.3.15`,
        fs:  `cpe:2.3:a:cisco:security_manager:4.4:sp1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.3.16`,
        fs:  `cpe:2.3:a:puppetlabs:facter:1.7.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.4.1`,
        fs:  `cpe:2.3:a:cubecart:cubecart:3.0.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.4.2`,
        fs:  `cpe:2.3:o:cisco:web_security_appliance:8.5.0-497:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.4.3`,
        fs:  `cpe:2.3:a:websense:websense_content_content_gateway:7.7.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.4.4`,
        fs:  `cpe:2.3:h:iodata:hdl-ah:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.4.5`,
        fs:  `cpe:2.3:a:anything-digital:sh404sef:2.0.2.542:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.4.6`,
        fs:  `cpe:2.3:a:pnp4nagios:pnp4nagios:0.6.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.4.7`,
        fs:  `cpe:2.3:a:gitlab:gitlab:8.14.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.4.8`,
        fs:  `cpe:2.3:a:ibm:security_appscan:7.9:-:standard:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.4.9`,
        fs:  `cpe:2.3:a:axigen:axigen_mail_server:8.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.4.10`,
        fs:  `cpe:2.3:a:google:chrome:23.0.1271.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.4.11`,
        fs:  `cpe:2.3:a:globalscape:eft_server:6.2.31.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.4.12`,
        fs:  `cpe:2.3:a:cisco:wireless_control_system_software:4.0.155.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.4.13`,
        fs:  `cpe:2.3:a:sensiolabs:symfony:1.4.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.4.14`,
        fs:  `cpe:2.3:a:zonelabs:zonealarm:2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.4.15`,
        fs:  `cpe:2.3:a:goodiware:goodreader:3.15.0:-:*:*:*:iphone_os:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.4.16`,
        fs:  `cpe:2.3:a:augeas:augeas:0.7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.4.17`,
        fs:  `cpe:2.3:o:sierrawireless:raven_x_ev-do_firmware:4228_4.0.11.003:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.4.18`,
        fs:  `cpe:2.3:h:lexmark:e260:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.5.0`,
        fs:  `cpe:2.3:a:webmin:usermin:1.310:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.5.1`,
        fs:  `cpe:2.3:a:phpcheckz:phpcheckz:0.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.5.3`,
        fs:  `cpe:2.3:a:bitcoin:bitcoin-qt:0.3.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_kellner:powermail:1.6.4`,
        fs:  `cpe:2.3:a:icinga:icinga:1.6.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alex_launi:tangerine:0.3.2.2`,
        fs:  `cpe:2.3:a:alkacon:opencms:7.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alexis_wilke:protected_node:6.x-1.0`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.7:p294:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alexis_wilke:protected_node:6.x-1.2`,
        fs:  `cpe:2.3:a:caucho:resin:2.1.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alexis_wilke:protected_node:6.x-1.3`,
        fs:  `cpe:2.3:a:danielb:finder:7.x-2.0:alpha6:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alexis_wilke:protected_node:6.x-1.4`,
        fs:  `cpe:2.3:a:wireshark:wireshark:1.8.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alexis_wilke:protected_node:6.x-1.5`,
        fs:  `cpe:2.3:a:gnu:gnash:0.8.9:rc4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alexis_wilke:protected_node:6.x-1.x:dev`,
        fs:  `cpe:2.3:a:samba:samba:4.2.0:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alfabank:alfa-bank:5.5.1.1::~~~android~~`,
        fs:  `cpe:2.3:a:splunk:splunk:4.3.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alfresco:alfresco:4.1.6::~~enterprise~~~`,
        fs:  `cpe:2.3:a:sliksvn:slik_subversion:1.6.16.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alfresco:alfresco:4.1.6.13::~~enterprise~~~`,
        fs:  `cpe:2.3:a:glpi-project:glpi:0.78.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alfresco:alfresco:4.2.f::~~community~~~`,
        fs:  `cpe:2.3:a:gnome:librsvg:2.2.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alfresco:alfresco:5.0.a::~~community~~~`,
        fs:  `cpe:2.3:o:cisco:pix_firewall_software:5.1\(4\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alfresco:alfresco:6.x-1.2::~~~drupal~~`,
        fs:  `cpe:2.3:a:altiris:pxe_server:6.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:algisinfo:aicontactsafe:2.0.0`,
        fs:  `cpe:2.3:a:parallels:parallels_plesk_panel:10.2.0:mu_\#5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:algisinfo:aicontactsafe:2.0.1`,
        fs:  `cpe:2.3:o:harman:amx_firmware:1.2.322:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:algisinfo:aicontactsafe:2.0.2`,
        fs:  `cpe:2.3:a:pureftpd:pure-ftpd:0.95:pre4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:algisinfo:aicontactsafe:2.0.3`,
        fs:  `cpe:2.3:a:nginx:nginx:1.1.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:algisinfo:aicontactsafe:2.0.4`,
        fs:  `cpe:2.3:a:ultraedit:ultraedit:16.30.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:algisinfo:aicontactsafe:2.0.5`,
        fs:  `cpe:2.3:a:symantec:norton_antivirus:9.0.4:mr4_build_1000:corporate_edition:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:algisinfo:aicontactsafe:2.0.5.b`,
        fs:  `cpe:2.3:a:drupac:touch:7.x-1.9:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:algisinfo:aicontactsafe:2.0.5.c`,
        fs:  `cpe:2.3:a:redhat:lvm2:2.01.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:algisinfo:aicontactsafe:2.0.6`,
        fs:  `cpe:2.3:a:fetchmail:fetchmail:6.2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:algisinfo:aicontactsafe:2.0.6.b`,
        fs:  `cpe:2.3:a:lockon:ec-cube:2.12.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:algisinfo:aicontactsafe:2.0.6.c`,
        fs:  `cpe:2.3:a:david_king:vino:2.11.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:algisinfo:aicontactsafe:2.0.7`,
        fs:  `cpe:2.3:a:nishishi:fumy_teachers_schedule_board:2.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:algisinfo:aicontactsafe:2.0.8`,
        fs:  `cpe:2.3:a:connectiva:linux:8.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:algisinfo:aicontactsafe:2.0.9`,
        fs:  `cpe:2.3:o:trustix:secure_linux:2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:algisinfo:aicontactsafe:2.0.10`,
        fs:  `cpe:2.3:a:libevent_project:libevent:1.4.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:algisinfo:aicontactsafe:2.0.10.b`,
        fs:  `cpe:2.3:a:softwebsnepal:ananda_real_estate:3.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:algisinfo:aicontactsafe:2.0.11`,
        fs:  `cpe:2.3:a:octobercms:october_cms:1.0.81:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:algisinfo:aicontactsafe:2.0.12`,
        fs:  `cpe:2.3:a:recurly:recurly_client_ruby:2.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:algisinfo:aicontactsafe:2.0.13`,
        fs:  `cpe:2.3:a:apple:xcode:4.3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:algisinfo:aicontactsafe:2.0.19`,
        fs:  `cpe:2.3:a:microsoft:project:2000:sr1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:algisinfo:aicontactsafe:2.0.21`,
        fs:  `cpe:2.3:a:mysql:mysql:3.23.30:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:algosec:fireflow:6.3:b230`,
        fs:  `cpe:2.3:a:contactus:contact_form_7_integrations:1.3:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:algosec:firewall_analyzer:6.1`,
        fs:  `cpe:2.3:a:iss:blackice:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:algosec:firewall_analyzer:6.1:b86`,
        fs:  `cpe:2.3:h:cisco:ip_phone_7905:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:algosec:firewall_analyzer:6.4`,
        fs:  `cpe:2.3:a:danielb:cool_aid:6.x-1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alhazai:leadership_newspapers:1.2::~~~android~~`,
        fs:  `cpe:2.3:a:netbeans:netbeans_ide:7.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ali_visual_project:ali_visual:1::~~~android~~`,
        fs:  `cpe:2.3:a:ibm:openpages_grc_platform:5.5.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aliakay:aptallik_testi:4::~~~android~~`,
        fs:  `cpe:2.3:a:bestpractical:rt:4.0.0:rc7:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alibaba:alibaba:4.1.0.0::~~~android~~`,
        fs:  `cpe:2.3:a:sixapart:movabletype:5.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:1.0.4`,
        fs:  `cpe:2.3:a:adobe:acrobat_dc:15.010.20060:*:*:*:continuous:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:1.0.6`,
        fs:  `cpe:2.3:a:hp:psprinters08:8.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:2.1`,
        fs:  `cpe:2.3:a:heimdal_project:heimdal:0.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:2.1.2`,
        fs:  `cpe:2.3:a:perl:perl:5.10.0:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:2.1.5`,
        fs:  `cpe:2.3:a:debian:axiom:20100701-1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:2.1.5-1`,
        fs:  `cpe:2.3:a:apache:jakarta_isapi_redirector:1.2.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:2.1.5-2`,
        fs:  `cpe:2.3:a:zingiri:zingiri_web_shop:2.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:2.1.5-3`,
        fs:  `cpe:2.3:a:ibm:websphere_application_server:-:-:*:*:*:z\/os:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:3.1`,
        fs:  `cpe:2.3:o:cisco:ios:12.0\(19\)sl4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:3.1.9`,
        fs:  `cpe:2.3:a:autonomy:sharepoint_connector_cfs:1.0.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:3.1.10`,
        fs:  `cpe:2.3:a:microsoft:system.net.security:4.0.0:*:*:*:*:asp.net:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:3.1.12`,
        fs:  `cpe:2.3:a:apple:safari:1.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:4.0`,
        fs:  `cpe:2.3:a:fetchmail:fetchmail:5.9.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:4.0.3`,
        fs:  `cpe:2.3:a:cisco:wireless_lan_solution_engine:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:4.0.4`,
        fs:  `cpe:2.3:o:cisco:ios:12.0\(5\)s:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:4.1`,
        fs:  `cpe:2.3:a:adaptivecomputing:torque_resource_manager:2.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:4.1.2`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.5:p124:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:4.1.3`,
        fs:  `cpe:2.3:a:octopus:octopus_deploy:3.3.25:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:4.2`,
        fs:  `cpe:2.3:h:ibm:s18:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:4.2.2`,
        fs:  `cpe:2.3:a:platinum_seo_project:platinum_seo_plugin:1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:4.2.3`,
        fs:  `cpe:2.3:a:microsoft:visio:2007:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:4.3`,
        fs:  `cpe:2.3:a:nasa_universe_wallpapers_xeus_project:nasa_universe_wallpapers_xeus:1:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:4.3.1`,
        fs:  `cpe:2.3:a:apple:itunes:7.1.1:-:windows:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:4.3.2`,
        fs:  `cpe:2.3:a:debian:dpkg:1.4.1.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:4.3.3`,
        fs:  `cpe:2.3:a:bigtreecms:bigtree_cms:4.1.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:4.4`,
        fs:  `cpe:2.3:o:linux:linux_kernel:4.4.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:4.5`,
        fs:  `cpe:2.3:a:microsoft:exchange_server:2010:sp3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:4.6`,
        fs:  `cpe:2.3:h:hp:laserjet_pro_300_color_mfp_printer_m375:ce903a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:4.6.1`,
        fs:  `cpe:2.3:a:ibm:personal_communications:6.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:4.7.0`,
        fs:  `cpe:2.3:h:kepware:kepserverex_communications_platform:5.3.118.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:open_source_security_information_management:4.8.0`,
        fs:  `cpe:2.3:a:vips:vips:7.20.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienvault:unified_security_management:4.14`,
        fs:  `cpe:2.3:a:cisco:meraki_mr_firmware:2014-09-24:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienwp:hatch:7.x-1.0`,
        fs:  `cpe:2.3:a:tincan:phplist:1.6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienwp:hatch:7.x-1.1`,
        fs:  `cpe:2.3:a:ibm:emptoris_strategic_supply_management:10.0.2.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienwp:hatch:7.x-1.2`,
        fs:  `cpe:2.3:a:citrix:receiver_usb:13.0.0.6685:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienwp:hatch:7.x-1.3`,
        fs:  `cpe:2.3:a:red-gate:sql_data_compare:6.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienwp:hatch:7.x-1.4`,
        fs:  `cpe:2.3:a:primetek:primefaces:4.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alienwp:hatch:7.x-1.x:dev`,
        fs:  `cpe:2.3:o:valvesoftware:steam_link_firmware:435:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alionscience:standard_frequency_action_format_conversion_tool:1.00.0000`,
        fs:  `cpe:2.3:h:rockwellautomation:compactlogix_1769-l30er:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alipay_project:alipay:3.6.0::~~~wordpress~~`,
        fs:  `cpe:2.3:a:ulli_horlacher:fex:20110716:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alkacon:opencms:6.0.0`,
        fs:  `cpe:2.3:a:apple:itunes:11.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alkacon:opencms:6.0.2`,
        fs:  `cpe:2.3:a:ibm:filenet_p8_application_engine:3.5.1:004:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alkacon:opencms:6.0.3`,
        fs:  `cpe:2.3:o:cisco:ios_xe:2.6\(.1\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alkacon:opencms:6.0.4`,
        fs:  `cpe:2.3:a:apache:cassandra:1.2.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alkacon:opencms:6.2`,
        fs:  `cpe:2.3:a:ibm:storwize_v3500_software:7.2.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alkacon:opencms:6.2.1`,
        fs:  `cpe:2.3:a:simple_email_form_project:simple_email_form:1.8.5:*:*:*:*:joomla\!:*:*`,
    },
     
    {
        uri: `cpe:/a:alkacon:opencms:6.2.2`,
        fs:  `cpe:2.3:a:webmin:usermin:1.280:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alkacon:opencms:6.2.3`,
        fs:  `cpe:2.3:a:google:chrome:35.0.1916.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alkacon:opencms:7.0.3`,
        fs:  `cpe:2.3:a:onesolutionapps:aloha_bail_bonds:1.1:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:alkacon:opencms:7.0.4`,
        fs:  `cpe:2.3:a:b2evolution:b2evolution:6.8.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alkacon:opencms:8.5`,
        fs:  `cpe:2.3:a:ibm:sametime:8.0.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alkacon:opencms:8.5.1`,
        fs:  `cpe:2.3:a:theforeman:foreman:1.15.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alkacon:opencms:8.5.2`,
        fs:  `cpe:2.3:a:bestpractical:rt:3.6.0:pre1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alkacon:opencms:9.5.1`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.4.23:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:all_around_cyprus_project:all_around_cyprus:2.11::~~~android~~`,
        fs:  `cpe:2.3:a:nih:libzip:0.9.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:all_navalny_project:all_navalny:1.1::~~~android~~`,
        fs:  `cpe:2.3:o:juniper:junos:15.1x53:d32:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:all_video_gallery_plugin_project:all-video-gallery:1.2::~~~wordpress~~`,
        fs:  `cpe:2.3:a:sendmail:sendmail\&#43;pro:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:all_video_gallery_plugin_project:all_video_gallery_plugin:1.0.0::~~~wordpress~~`,
        fs:  `cpe:2.3:a:ibm:db2:9.7.0.7:*:*:*:advanced_workgroup:*:*:*`,
    },
     
    {
        uri: `cpe:/a:all_video_gallery_plugin_project:all_video_gallery_plugin:1.1.0::~~~wordpress~~`,
        fs:  `cpe:2.3:a:adobe:flash_player:11.2.202.460:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:all_video_gallery_plugin_project:all_video_gallery_plugin:1.2.0::~~~wordpress~~`,
        fs:  `cpe:2.3:a:bas_van_beek:multishop:2.0.31:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:allaire:homesite:1.0`,
        fs:  `cpe:2.3:a:hp:select_identity:4.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:allaire:homesite:2.0`,
        fs:  `cpe:2.3:a:itd-inc:bingo\!cms:1.2:-:core:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:allaire:homesite:3.0`,
        fs:  `cpe:2.3:a:asp-dev:xm_forums:-:rc3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:allaire:homesite:4.0`,
        fs:  `cpe:2.3:a:cmu:cyrus_imap_server:2.3.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:allaire:homesite:4.5`,
        fs:  `cpe:2.3:a:mantisbt:mantisbt:1.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:allamericansportsware:common_controls_pack:1.01.0002`,
        fs:  `cpe:2.3:o:cisco:vpn_3000_concentrator_series_software:2.5.2.c:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:allegrosoft:rompager:4.07`,
        fs:  `cpe:2.3:a:claws-mail:claws-mail:2.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:allegrosoft:rompager:4.34`,
        fs:  `cpe:2.3:a:oracle:mysql:5.6.25:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:allegrosoft:rompager:4.51`,
        fs:  `cpe:2.3:a:libpng:libpng:0.88:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alliedtelesis:at-rg634a:-`,
        fs:  `cpe:2.3:a:germanwings:germanwings:2.1.13:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/o:alliedtelesis:at-rg634a_firmware:3.3%2b`,
        fs:  `cpe:2.3:a:location_module_project:location:6.x-3.0:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alliedtelesis:img616lh:-`,
        fs:  `cpe:2.3:o:microsoft:windows-nt:2003:sp1:itanium:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:alliedtelesis:img616lh_firmware:%2b2.4`,
        fs:  `cpe:2.3:o:microsoft:windows-nt:2003:gold:enterprise:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alliedtelesis:img624a:-`,
        fs:  `cpe:2.3:h:hp:h3c_router:sr8808:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:alliedtelesis:img624a_firmware:3.5`,
        fs:  `cpe:2.3:a:moinejf:abcm2ps:4.7.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alliedtelesis:img646bd:-`,
        fs:  `cpe:2.3:a:foxitsoftware:foxit_reader:5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:alliedtelesis:img646bd_firmware:3.5`,
        fs:  `cpe:2.3:a:google:chrome:8.0.552.23:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alliedtelesyn:at-8024:-`,
        fs:  `cpe:2.3:a:zemanta:related_posts:1.8.1:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/h:alliedtelesyn:at-8024:1.3.1`,
        fs:  `cpe:2.3:a:google:chrome:47.0.2526.73:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alliedtelesyn:at-9000_24_ethernetswitch:-`,
        fs:  `cpe:2.3:a:hp:web_jetadmin:6.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alliedtelesyn:at-9724ts:-`,
        fs:  `cpe:2.3:a:sun:jdk:1.3.0_03:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alliedtelesyn:at-s24:-`,
        fs:  `cpe:2.3:o:iij:seil_x2_firmware:4.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alliedtelesyn:at-s24:2.4.4`,
        fs:  `cpe:2.3:a:nordugrid:nordugrid-arc:0.5.48:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alliedtelesyn:at-s30:-`,
        fs:  `cpe:2.3:a:theforeman:foreman:1.9.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alliedtelesyn:at-s39:-`,
        fs:  `cpe:2.3:a:happyworm:jplayer:2.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alliedtelesyn:at-s39:3.1.1.1`,
        fs:  `cpe:2.3:a:esri:arcsde:9.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alliedtelesyn:at-s39:3.2.0`,
        fs:  `cpe:2.3:a:patrick_przybilla:addtoany:6.x-2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alliedtelesyn:at-s39:3.3.1`,
        fs:  `cpe:2.3:o:apple:mac_os_x_server:10.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alliedtelesyn:at-tftp:-`,
        fs:  `cpe:2.3:h:huawei:s9306:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alliedtelesyn:at-tftp:1.9`,
        fs:  `cpe:2.3:a:yahoo:messenger:5.6.0.1355:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alliedtelesyn:cable_dsl_router_at-ar220e:-`,
        fs:  `cpe:2.3:a:teampass:teampass:2.1.26.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:allnurses:allnurses:3.4.10::~~~android~~`,
        fs:  `cpe:2.3:a:novell:netmail:3.10a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:allplayer:allplayer:5.6.2`,
        fs:  `cpe:2.3:a:orangelampsoftware:killprocess:2.44:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:allplayer:allplayer:5.7.0`,
        fs:  `cpe:2.3:a:oracle:universal_work_queue:12.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:allplayer:allplayer:5.8.1`,
        fs:  `cpe:2.3:a:elfutils_project:elfutils:0.146:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:allqoranvideos:koran_-_alqoranvideos:1::~~~android~~`,
        fs:  `cpe:2.3:h:d-link:dap-2310:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:allroundautomation:pl%2fsql_developer:9.0.2.1621`,
        fs:  `cpe:2.3:a:emc:rsa_archer_egrc:5.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:allroundautomations:pl%2fsql_developer:11.0`,
        fs:  `cpe:2.3:a:elgg:elgg:1.8.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:allroundautomations:pl%2fsql_developer:11.0.1`,
        fs:  `cpe:2.3:a:bas_van_beek:multishop:2.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:allroundautomations:pl%2fsql_developer:11.0.2`,
        fs:  `cpe:2.3:a:location_module_project:location:7.x-5.x:dev:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:allroundautomations:pl%2fsql_developer:11.0.3`,
        fs:  `cpe:2.3:a:hp:san\/iq:8.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:allroundautomations:pl%2fsql_developer:11.0.4`,
        fs:  `cpe:2.3:a:guillaume_gauvrit:pyshop:0.7.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:allroundautomations:pl%2fsql_developer:11.0.5`,
        fs:  `cpe:2.3:a:cambridge_enterprise:jbig-kit:0.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:almanah_project:almanah:0.9.0`,
        fs:  `cpe:2.3:o:apple:mac_os_x:10.12.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:almanah_project:almanah:0.10.0`,
        fs:  `cpe:2.3:a:joomlaboat:com_youtubegallery:3.9.0:*:*:*:*:joomla\!:*:*`,
    },
     
    {
        uri: `cpe:/a:almas:compiere:j253b_a02`,
        fs:  `cpe:2.3:a:octobercms:october_cms:1.0.105:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:almas:compiere:j300_a01`,
        fs:  `cpe:2.3:a:citrix:secure_gateway:3.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:almas:compiere:j300_a02`,
        fs:  `cpe:2.3:a:ratbox:ircd-ratbox:2.0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:almasiapps:healing_bookstore:0.1::~~~android~~`,
        fs:  `cpe:2.3:a:git-scm:git:1.5.4.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aloaha:aloaha_pdf_suite_free:-`,
        fs:  `cpe:2.3:a:apache:opentaps:0.9.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aloaha:aloahapdfviewer:5.0.0.7`,
        fs:  `cpe:2.3:a:ati:catalyst_control_center_localization:2008.717.2343.40629:*:*:fr:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alphabetic_sitemap_project:alphabetic_sitemap:0.0.1::~~~typo3~~`,
        fs:  `cpe:2.3:a:apache_software_foundation:tomcat:4.1.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alphabetic_sitemap_project:alphabetic_sitemap:0.0.2::~~~typo3~~`,
        fs:  `cpe:2.3:o:cisco:catos:6.3\(10\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alphabetic_sitemap_project:alphabetic_sitemap:0.0.3::~~~typo3~~`,
        fs:  `cpe:2.3:a:bea:weblogic_server:4.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alphanetworks:vdsl_asl-55052:-`,
        fs:  `cpe:2.3:a:cisco:telepresence_mcu_software:4.3\(2.32\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:alphanetworks:vdsl_asl-56552:-`,
        fs:  `cpe:2.3:a:google:chrome:31.0.1650.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alphazip:alphazip:1.3.0`,
        fs:  `cpe:2.3:a:umn:mapserver:6.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alps:touch_pad_driver:-`,
        fs:  `cpe:2.3:a:ibm:websphere_application_server:6.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alps:touch_pad_driver:5.4.102.8`,
        fs:  `cpe:2.3:a:ippusbxd_project:ippusbxd:1.21.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alps:touch_pad_driver:5.4.104.4`,
        fs:  `cpe:2.3:a:xymon:xymon:4.3.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alps:touch_pad_driver:5.5.1302.12`,
        fs:  `cpe:2.3:o:aterm:wf800hp_firmware:1.0.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alps:touch_pad_driver:6.0.302.3`,
        fs:  `cpe:2.3:a:merethis:centreon:2.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alps:touch_pad_driver:6.0.302.7`,
        fs:  `cpe:2.3:o:scientificlinux:scientificlinux:6.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alps:touch_pad_driver:6.0.303.4`,
        fs:  `cpe:2.3:a:google:chrome:28.0.1500.31:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alps:touch_pad_driver:6.0.303.5`,
        fs:  `cpe:2.3:a:ibm:business_process_manager:8.5.6.2:*:*:*:advanced:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alps:touch_pad_driver:6.0.304.5`,
        fs:  `cpe:2.3:a:elasticsearch:kibana:6.0.0:alpha2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alps:touch_pad_driver:6.0.304.6`,
        fs:  `cpe:2.3:a:google:chrome:21.0.1180.87:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alps:touch_pad_driver:6.0.305.5`,
        fs:  `cpe:2.3:a:hp:network_node_manager_i:9.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alps:touch_pad_driver:6.0.305.6`,
        fs:  `cpe:2.3:o:nuuo:nvrmini_2:1.7.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alps:touch_pad_driver:7.1.102.7`,
        fs:  `cpe:2.3:a:formsplayer:formsplayer:1.5.0:1047:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alps:touch_pad_driver:7.2.101.219`,
        fs:  `cpe:2.3:a:ibm:maximo_for_government:7.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alps:touch_pad_driver:7.4.102.102`,
        fs:  `cpe:2.3:o:microsoft:windows_2003_server:-:sp1:datacenter:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alps:touch_pad_driver:7.102.101.223`,
        fs:  `cpe:2.3:o:juniper:junos:4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alps:touch_pad_driver:7.102.101.228`,
        fs:  `cpe:2.3:a:ibm:tivoli_storage_manager_for_databases_data_protection_for_microsoft_sql_server:6.3.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alps:touch_pad_driver:7.102.302.203`,
        fs:  `cpe:2.3:a:ibm:security_privileged_identity_manager:2.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alquimia:managesite:6.x-1.0`,
        fs:  `cpe:2.3:a:asial:monaca_debugger:1.4.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alquimia:managesite:6.x-1.x:dev`,
        fs:  `cpe:2.3:a:citrix:receiver_inside:3.0.0.56418:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alrazylabs:lostword:5.9::~~~android~~`,
        fs:  `cpe:2.3:a:cloudbees:jenkins:1.392:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alstom:e-terracontrol:3.5`,
        fs:  `cpe:2.3:a:hp:onboard_administrator:2.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alstom:e-terracontrol:3.6`,
        fs:  `cpe:2.3:a:palemoon:pale_moon:15.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alstom:e-terracontrol:3.7`,
        fs:  `cpe:2.3:a:microsoft:office:2003:*:*:*:student_and_teacher:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alstom:micom_s1_agile:1.0.2`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alstom:micom_s1_agile:1.0.3`,
        fs:  `cpe:2.3:a:cisco:telepresence_system_software:tt4.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alstom:micom_s1_studio:-`,
        fs:  `cpe:2.3:h:avocent:dsr1020:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alsunna_project:alsunna:0.1::~~~android~~`,
        fs:  `cpe:2.3:a:canonical:update-manager:1\:0.142.23.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:3.1.9`,
        fs:  `cpe:2.3:a:mit:kerberos:5-1.6.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:3.1.10`,
        fs:  `cpe:2.3:a:indigorose:setup_factory:7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:3.1.11`,
        fs:  `cpe:2.3:a:graphicsmagick:graphicsmagick:1.3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:3.1.12`,
        fs:  `cpe:2.3:a:ati:catalyst_control_center_graphics_light:2010.302.2233.40412:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:4.0.0`,
        fs:  `cpe:2.3:a:contao:contao_cms:2.2.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:4.0.1`,
        fs:  `cpe:2.3:h:hp:procurve_switch:jc100a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:4.0.2`,
        fs:  `cpe:2.3:a:kbd-project:kbd:1.15.3:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:4.0.3`,
        fs:  `cpe:2.3:a:debian:dpkg:1.6.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:4.0.4`,
        fs:  `cpe:2.3:h:axis:2100_network_camera:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:4.0.5`,
        fs:  `cpe:2.3:a:buffalotech:bbr-4mg_firmware:1.01:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:4.0.6`,
        fs:  `cpe:2.3:a:yokogawa:fast\/tools:r9.03:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:4.0.7`,
        fs:  `cpe:2.3:a:pureftpd:pure-ftpd:0.96:pre1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:4.0.8`,
        fs:  `cpe:2.3:a:google:chrome:14.0.835.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:4.1.0`,
        fs:  `cpe:2.3:a:icinga:icinga:1.9.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:4.1.1`,
        fs:  `cpe:2.3:h:lexmark:cs820dte:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:4.1.2`,
        fs:  `cpe:2.3:a:hitachi:cosminexus_developer:6:-:pro:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:4.1.3`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.4.21:pre7:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:4.1.4`,
        fs:  `cpe:2.3:a:nullsoft:winamp:5.57:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:4.1.5`,
        fs:  `cpe:2.3:a:php:php:5.4.28:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:4.1.6`,
        fs:  `cpe:2.3:a:wordpress:wordpress:2.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:4.1.7`,
        fs:  `cpe:2.3:o:linux:linux_kernel:4.8.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:4.1.8`,
        fs:  `cpe:2.3:a:openstack:swift:1.13.1:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:4.1.8-h3`,
        fs:  `cpe:2.3:a:google:chrome:13.0.751.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:4.1.9`,
        fs:  `cpe:2.3:h:linksys:wet11:1.32:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:4.1.10`,
        fs:  `cpe:2.3:a:digium:asterisk:14.0.0:beta1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:4.1.11`,
        fs:  `cpe:2.3:a:ibm:tealeaf_consumer_experience:8.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:4.1.12`,
        fs:  `cpe:2.3:a:vips:vips:7.11.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:4.1.13`,
        fs:  `cpe:2.3:a:kurt_gusbeth:myquizpoll:0.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:5.0.0`,
        fs:  `cpe:2.3:a:umn:mapserver:5.2.0:beta3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:5.0.0-h1`,
        fs:  `cpe:2.3:a:smartsound:quicktracks_plugin:3.0.2.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:5.0.1-h1`,
        fs:  `cpe:2.3:a:lunascape:lunascape:4.8.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:5.0.2`,
        fs:  `cpe:2.3:a:debian:reprepro:2.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:5.0.3`,
        fs:  `cpe:2.3:a:apple:cups:1.1.6-2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:5.0.4`,
        fs:  `cpe:2.3:a:apple:itunes:4.8.0:-:windows:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:altaware:palo_alto_networks_panos:5.0.5`,
        fs:  `cpe:2.3:a:greenbone:greenbone_security_assistant:4.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alternative_connection_project:alternative_connection:0.1::~~~android~~`,
        fs:  `cpe:2.3:a:fedoraproject:389_directory_server:1.2.8:alpha3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alterpoint:logwatch:1.0.0.0`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.5:p247_rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alterpoint:networkauthority:6.0.0.0`,
        fs:  `cpe:2.3:a:haxx:curl:7.32.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alterpoint:networkauthority_engineering_environment:6.0.1.0`,
        fs:  `cpe:2.3:a:cisco:unified_computing_system:1.4\(5h\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altimatech:netzoom:6.2.2`,
        fs:  `cpe:2.3:a:udesign_project:udesign:2.4.5:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:agent_for_mac:6.2.1302`,
        fs:  `cpe:2.3:a:udesign_project:udesign:2.5.4:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:agent_for_mac_documentation:6.2.1124`,
        fs:  `cpe:2.3:o:cisco:clean_access:3.3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:agent_for_pocket_pc:6.1.1141`,
        fs:  `cpe:2.3:a:isc:bind:9.4.0:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:agent_for_unix_and_linux:6.2.1378`,
        fs:  `cpe:2.3:a:webmin:webmin:1.670:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:agent_for_unix_and_linux_documentation:6.2.1048`,
        fs:  `cpe:2.3:a:microsoft:office:2010:*:x86:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:alert_manager:6.0.308`,
        fs:  `cpe:2.3:a:bestpractical:request_tracker:4.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:application_control_solution_documentation:6.1.1015`,
        fs:  `cpe:2.3:o:google:chrome_os:25.0.1364.121:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:application_management_language_pack:6.1.4082`,
        fs:  `cpe:2.3:a:searchblox:searchblox:7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:application_management_solution:6.1.4088`,
        fs:  `cpe:2.3:a:sensiolabs:symfony:2.3.33:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:application_management_solution_documentation:6.1.4085`,
        fs:  `cpe:2.3:a:ibm:security_appscan_source:9.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:application_metering:6.1.55`,
        fs:  `cpe:2.3:o:cisco:ios:12.3\(8\)t7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:application_metering_agent:6.1.31`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:5.5.5.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:application_metering_documentation:6.1.32`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.559:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:application_metering_language_pack:6.1.32`,
        fs:  `cpe:2.3:a:digium:certified_asterisk:11.3.0:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:appweaver:6.0.308`,
        fs:  `cpe:2.3:a:zope:zope:2.9.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:asset_management_report_pack:6.5.1049`,
        fs:  `cpe:2.3:a:cagintranetworks:getsimple_cms:1.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:asset_management_solution:6.5.1126`,
        fs:  `cpe:2.3:a:adobe:flash_player:10.3.183.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:asset_management_solution_documentation:6.5.1049`,
        fs:  `cpe:2.3:a:postfix:postfix:2.8.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:audit_integration_component:6.2.124`,
        fs:  `cpe:2.3:a:devincentiis:gazie:5.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:audit_integration_component_documentation:6.2.1016`,
        fs:  `cpe:2.3:a:dish:dish_anywhere:3.5.10:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:auditexpress_setup_solution:6.2.124`,
        fs:  `cpe:2.3:a:hp:linux_imaging_and_printing_project:3.11.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:barcode_solution:6.7.8`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.4.1:*:*:*:*:*:x86:*`,
    },
     
    {
        uri: `cpe:/a:altiris:barcode_solution_documentation:6.6.1031`,
        fs:  `cpe:2.3:a:vmware:vcenter_operations:1.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:carbon_copy:-`,
        fs:  `cpe:2.3:a:kiwix:kiwix:0.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:carbon_copy:5.0`,
        fs:  `cpe:2.3:o:cisco:ios:12.0xq:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:carbon_copy:6.0`,
        fs:  `cpe:2.3:h:dahuasecurity:dvr1604hf-l-e:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:carbon_copy_solution:6.2.1144`,
        fs:  `cpe:2.3:a:osticket:osticket:1.6:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:carbon_copy_solution_agent:6.2.1144`,
        fs:  `cpe:2.3:a:microsoft:works_6-9_file_converter:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:carbon_copy_solution_documentation:6.2.1121`,
        fs:  `cpe:2.3:a:sensiolabs:symfony:2.0.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:client_management_suite_quick_start:6.0.1274`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.33.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:client_service:-`,
        fs:  `cpe:2.3:h:hp:color_laserjet_4600:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:client_service:5.6.181`,
        fs:  `cpe:2.3:a:tecnick:tcexam:11.1.026:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:client_service:6.0.88`,
        fs:  `cpe:2.3:a:ibm:db2_universal_database:9.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:client_task_agent:6.0.1524`,
        fs:  `cpe:2.3:a:hp:linux_imaging_and_printing_project:3.11.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:client_task_management:6.0.1524`,
        fs:  `cpe:2.3:a:apache:httpclient:4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:client_task_server:6.0.1524`,
        fs:  `cpe:2.3:a:clamav:clamav:0.60p:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:cmdb:6.5.1126`,
        fs:  `cpe:2.3:a:ibm:maximo_asset_management_essentials:7.5.0.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:cmdb_and_asset_management_language_pack:6.5.1021`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.571:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:cmdb_and_asset_management_report_language_pack:6.5.1021`,
        fs:  `cpe:2.3:a:apache:traffic_server:5.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:cmdb_report_pack:6.5.1049`,
        fs:  `cpe:2.3:a:iconics:bizviz:9.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:cmdb_solution_documentation:6.5.1016`,
        fs:  `cpe:2.3:o:yamaha:rt100i:2.02.31:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:component_for_active_directory_documentation:6.1.1019`,
        fs:  `cpe:2.3:a:gitlab:gitlab:1.2.0:*:*:*:community:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:computer_management:6.2.3529`,
        fs:  `cpe:2.3:a:webyog:sqlyog:3.02:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:computer_management_language_pack:6.2.3533`,
        fs:  `cpe:2.3:a:ruby-lang:ruby:1.9.3:p385:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:connector_for_veritas_configuration_manager:6.0.1039`,
        fs:  `cpe:2.3:a:mortbay:jetty:5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:connector_for_wise:6.1.138`,
        fs:  `cpe:2.3:a:apache:xml_security_for_c\&#43;\&#43;:1.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:connector_solution_data_integration_component:6.5.1149`,
        fs:  `cpe:2.3:a:hp:ignite-ux:c.7.10.474:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:connector_solution_data_integration_component_extensions:6.0.1318`,
        fs:  `cpe:2.3:a:gnu:gnutls:1.2.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:connector_solution_documentation:6.5.1026`,
        fs:  `cpe:2.3:a:ati:catalyst_control_center_help:2009.1104.958.17837:*:*:en:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:connector_solution_event_integration_component:6.0.1823`,
        fs:  `cpe:2.3:a:tapatalk:tapatalk:3.1.4:*:*:*:*:vbulletin:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:connector_solution_quick_start:6.0.1102`,
        fs:  `cpe:2.3:a:gnu:privacy_guard:1.4.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:console:6.5.1144`,
        fs:  `cpe:2.3:h:hp:procurve_router:jf800a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:console_data:6.5.31.0:update_kb34732`,
        fs:  `cpe:2.3:o:iij:seil_x2_firmware:3.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:console_data:6.5.1156`,
        fs:  `cpe:2.3:a:fortinet:fortiweb:5.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:console_data_language_pack:6.5.1146`,
        fs:  `cpe:2.3:o:d-link:dap-2330_firmware:1.06:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:console_documentation:6.5.1024`,
        fs:  `cpe:2.3:a:citadel:hercules_remediation_client_for_windows:4.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:console_language_pack:6.5.1144`,
        fs:  `cpe:2.3:a:podsfoundation:pods:2.4.3:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:context_factory:6.0.1127`,
        fs:  `cpe:2.3:a:webyog:sqlyog:3.7:-:community:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:dell_client_configuration_utility:3.0.1210.0`,
        fs:  `cpe:2.3:o:cisco:ios_xe:3.1s.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:dell_client_configuration_utility:3.0.1213.0`,
        fs:  `cpe:2.3:a:cakephp:cakephp:2.6.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:dell_client_configuration_utility:3.1.1230.0`,
        fs:  `cpe:2.3:a:perl:perl:5.11.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:dell_client_manager_documentation:2.2.1017`,
        fs:  `cpe:2.3:a:lxml:lxml:1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:dell_client_manager_language_pack:2.2.1010`,
        fs:  `cpe:2.3:a:bayashi:dopvstar\*:0037:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:dell_client_manager_solution:2.2.1019`,
        fs:  `cpe:2.3:a:djangoproject:django:1.10.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:deployment_agent:1.0.0`,
        fs:  `cpe:2.3:a:cs-cart:cs-cart:2.2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:deployment_server_extension_for_ibm_director:-`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(25\)seb2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:deployment_server_extension_for_ibm_director:5.0.1`,
        fs:  `cpe:2.3:a:cisofy:lynis:1.5.0:*:*:*:*:aix:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:deployment_server_extension_for_ibm_director:5.5`,
        fs:  `cpe:2.3:o:cisco:unified_ip_phone_6900_series_firmware:9.3\(2\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:deployment_server_extension_for_ibm_director:6.0`,
        fs:  `cpe:2.3:a:mortbay:jetty:3.0.b01:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:deployment_server_extension_for_ibm_director:6.1`,
        fs:  `cpe:2.3:o:arubanetworks:instant_access_point_firmware:4.1.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:deployment_solution:-`,
        fs:  `cpe:2.3:a:mcafee:common_management_agent:3.6.0.569:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:deployment_solution:5.6`,
        fs:  `cpe:2.3:o:cisco:catos:5.4\(2a\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:deployment_solution:5.6:sp1`,
        fs:  `cpe:2.3:h:hp:procurve_switch:jd327a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:deployment_solution:5.6.181`,
        fs:  `cpe:2.3:o:apple:iphone_os:10.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:deployment_solution:6.0`,
        fs:  `cpe:2.3:a:videousermanuals:white-label-cms:1.4.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:deployment_solution:6.8`,
        fs:  `cpe:2.3:a:libtiff:libtiff:4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:deployment_solution:6.8:sp1`,
        fs:  `cpe:2.3:a:lansweeper:lansweeper:5.1.0.50:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:deployment_solution:6.8:sp2`,
        fs:  `cpe:2.3:a:ibm:lotus_notes:8.5.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:deployment_solution:6.9.174`,
        fs:  `cpe:2.3:a:nullsoft:winamp:5.08d:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:deployment_solution:6.9.355`,
        fs:  `cpe:2.3:a:intermesh:group-office:3.2.44:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:deployment_solution_for_network_devices:6.0.1046`,
        fs:  `cpe:2.3:a:jetaudio:jetaudio:8.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:deployment_solution_for_network_devices_documentation:6.0.1013`,
        fs:  `cpe:2.3:a:blackberry:blackberry_link:1.0.1.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:deployment_web_console:6.8`,
        fs:  `cpe:2.3:a:libguestfs:libguestfs:1.0.55:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:deployment_web_console:6.9`,
        fs:  `cpe:2.3:o:cisco:vpn_3030_concentator:4.1.7.l:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:diagnostics:6.1.7325.0`,
        fs:  `cpe:2.3:a:ibm:tivoli_directory_server:6.1.0.62:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:diagnostics:6.1.7631.0`,
        fs:  `cpe:2.3:h:cisco:telepresence_video_communication_server:-:*:starter_pack_express:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:discovery_for_rim:6.1.1091`,
        fs:  `cpe:2.3:a:checkpoint:firewall-1:3.0b:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:endpoint_security_solution:6.0.479`,
        fs:  `cpe:2.3:a:hp:proliant_integrated_lights_out_2:1.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:endpoint_security_solution_documentation:6.0.1007`,
        fs:  `cpe:2.3:a:jease:jease:0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:energy_saver_toolkit:6.0.1214`,
        fs:  `cpe:2.3:a:ibm:security_access_manager_for_web:6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:energy_saver_toolkit_documentation:6.0.1006`,
        fs:  `cpe:2.3:a:x.org:xorg-server:1.15.99.901:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:express_deployment_console:6.8`,
        fs:  `cpe:2.3:a:tryton:trytond:2.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:express_deployment_console:6.9`,
        fs:  `cpe:2.3:a:ibm:financial_transaction_manager:3.0.2.1:*:*:*:*:ach_services:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:express_deployment_datastore:6.8`,
        fs:  `cpe:2.3:a:silverstripe:silverstripe:2.3.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:express_deployment_datastore:6.9`,
        fs:  `cpe:2.3:a:spip:spip:2.0.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:express_deployment_server:6.8`,
        fs:  `cpe:2.3:a:exactcode:exactimage:0.6.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:express_deployment_server:6.9`,
        fs:  `cpe:2.3:a:cisco:adaptive_security_appliance_software:8.4\(.7.15\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:helpdesk_proxy:6.0.308`,
        fs:  `cpe:2.3:o:yahama:rt105e:6.03.28:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:helpdesk_solution:6.0.308`,
        fs:  `cpe:2.3:a:oracle:supply_chain_products_suite:5.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:helpdesk_solution_documentation:6.0.1044`,
        fs:  `cpe:2.3:a:adobe:creative_suite_3_design_premium:1.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:integrated_component_for_microsoft_active_directory:6.1.842.0`,
        fs:  `cpe:2.3:o:advantech:eki-1321_series_firmware:1.96:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:inventory_merge_module_msi:6.1.1075.0`,
        fs:  `cpe:2.3:a:connections_project:connections:0.7.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:inventory_pack_for_servers:6.1.1058.0`,
        fs:  `cpe:2.3:a:google:chrome:35.0.1916.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:inventory_pack_for_servers_documentation:6.1.1019`,
        fs:  `cpe:2.3:a:microfocus:rumba:7.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:inventory_rule_management:6.2.3593`,
        fs:  `cpe:2.3:a:puppetlabs:facter:1.6.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:inventory_rule_management_language_pack:6.2.3595`,
        fs:  `cpe:2.3:a:cisco:hosted_collaboration_solution:10.0\(1\)_base:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:inventory_solution_documentation:6.1.1059`,
        fs:  `cpe:2.3:a:user_relationships_project:user_relationships:6.x-1.0:beta1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:inventory_solution_for_mac:6.2.1054`,
        fs:  `cpe:2.3:o:hp:procurve_switch_software:h.10.30:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:inventory_solution_for_mac_documentation:6.2.1036`,
        fs:  `cpe:2.3:h:apple:imac:mb953xx\/a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:inventory_solution_for_network_devices_documentation:6.0.1015`,
        fs:  `cpe:2.3:o:juniper:junos:14.1x53:d25:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:inventory_solution_for_pocket_pc:6.1.1141`,
        fs:  `cpe:2.3:a:mathjs:math.js:0.12.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:inventory_solution_for_pocket_pc_documentation:6.1.1007`,
        fs:  `cpe:2.3:a:symantec:norton_antivirus:9.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:inventory_solution_for_rim:6.1.1091`,
        fs:  `cpe:2.3:a:altiris:patch_management_agent:6.1.1422:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:inventory_solution_for_rim_documentation:6.1.1010`,
        fs:  `cpe:2.3:a:monster_menus_module_project:monster_menus:7.x-1.23:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:inventory_solution_for_unix_and_linux:6.2.1034`,
        fs:  `cpe:2.3:a:samba:samba:3.2.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:inventory_solution_for_unix_and_linux_documentation:6.2.1032`,
        fs:  `cpe:2.3:a:impresscms:impresscms:1.2.5:final:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:inventory_solution_for_windows:6.1.2004.0`,
        fs:  `cpe:2.3:o:hp:procurve_switch_software:h.08.106:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:it_analytics:6.1.1050`,
        fs:  `cpe:2.3:a:devsaran:professional_theme:7.x-1.x:dev:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:it_analytics_asset_management_pack:6.1.1085`,
        fs:  `cpe:2.3:a:eclipse:eclipse_ide:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:it_analytics_client_management_pack:6.1.1085`,
        fs:  `cpe:2.3:a:cms_tree_page_view_project:cms_tree_page_view:0.8.5:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:it_analytics_service_management_pack:6.1.1085`,
        fs:  `cpe:2.3:a:oracle:mysql:5.5.30:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:it_analytics_solution_documentation:6.1.1010`,
        fs:  `cpe:2.3:a:drupal:drupal:5.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:it_analytics_symantec_endpoint_protection_pack:6.1.1048`,
        fs:  `cpe:2.3:a:labtam-inc:proftp:2.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:knowledge_base_proxy:6.0.308`,
        fs:  `cpe:2.3:a:debian:apt:0.7.17:exp3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:language_pack_for_patch_management_core_linux:6.2.1022`,
        fs:  `cpe:2.3:a:pmwiki:pmwiki:2.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:language_pack_for_patch_management_dell_for_linux:6.2.1015`,
        fs:  `cpe:2.3:h:cisco:web_security_appliance:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:linux_pack_for_patch_management_solution_for_dell_servers:6.2.1128`,
        fs:  `cpe:2.3:a:apache:http_server:1.3.68:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:migration_portal:1.0.4.0`,
        fs:  `cpe:2.3:a:linuxfoundation:cups-filters:1.13.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:monitor_agent_for_unix_and_linux:6.0.3143`,
        fs:  `cpe:2.3:a:hp:instant_share_devices_mfc:70.0.170.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:monitor_agent_for_windows:6.0.1317`,
        fs:  `cpe:2.3:a:busybox:busybox:1.18.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:monitor_agent_for_windows:6.0.1341`,
        fs:  `cpe:2.3:a:mailpoet:mailpoet_newsletters:2.6.5:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:monitor_for_notification_server_metric:6.0.1057`,
        fs:  `cpe:2.3:a:primetek:primefaces:5.3.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:monitor_pack_for_altiris_infrastructure:6.0.1057`,
        fs:  `cpe:2.3:a:restlet:restlet:2.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:monitor_packs_for_servers:6.0.2066`,
        fs:  `cpe:2.3:a:autodesk:autocad_lt:98:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:monitor_solution:6.0.1337`,
        fs:  `cpe:2.3:o:siemens:ccmw3025_firmware:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:monitor_solution:6.0.1341`,
        fs:  `cpe:2.3:a:sun:jdk:1.4.2_8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:monitor_solution_documentation:6.0.1074`,
        fs:  `cpe:2.3:o:tp-link:tl-wrd841n_\(10.0\)_firmware:150104:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:monitor_solution_for_dell_servers:6.1.2046`,
        fs:  `cpe:2.3:a:tincan:phplist:1.1.2b:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:monitor_solution_for_dell_servers_documentation:6.1.1016`,
        fs:  `cpe:2.3:a:postgresql:postgresql:7.4.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:network_discovery:6.0.2120`,
        fs:  `cpe:2.3:a:djangoproject:django:1.2.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:network_discovery_documentation:6.0.1025`,
        fs:  `cpe:2.3:o:tp-link:tl-wr841nd_\(9.0\)_firmware:150104:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:network_discovery_language_pack:6.0.4`,
        fs:  `cpe:2.3:a:roundup-tracker:roundup:0.5.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:network_inventory:6.0.1198`,
        fs:  `cpe:2.3:a:oscommerce:oscommerce:2.2_ms1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:notification_server:6.0.6074`,
        fs:  `cpe:2.3:a:torproject:tor:0.2.0.2:alpha:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:notification_server_configurator:6.0.5010`,
        fs:  `cpe:2.3:a:expressjs:express:4.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:notification_server_configurator:6.1.7325.0`,
        fs:  `cpe:2.3:o:iij:seil_x86_fuji_firmware:2.80:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:notification_server_configurator:6.1.7631.0`,
        fs:  `cpe:2.3:a:mailenable:mailenable:1.704:-:standard:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:notification_server_documentation:6.0.1056`,
        fs:  `cpe:2.3:a:php:php:5.5.26:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:notification_server_email_event_receiver:6.1.1091`,
        fs:  `cpe:2.3:a:mozilla:network_security_services:3.6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:out_of_band_management_language_pack:6.2.1016`,
        fs:  `cpe:2.3:a:mailenable:mailenable:1.97:-:standard:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:out_of_band_management_solution:6.2.1035`,
        fs:  `cpe:2.3:a:jenkins:script_security:1.22:*:*:*:*:jenkins:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:out_of_band_management_solution_documentation:6.2.1012`,
        fs:  `cpe:2.3:a:wordpress:wordpress:2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:out_of_band_setup_and_configuration_language_pack:6.2.1012`,
        fs:  `cpe:2.3:a:pgp:certificate_server:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:out_of_band_setup_and_configuration_solution:6.2.1040`,
        fs:  `cpe:2.3:a:openstack:keystone:juno-3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:package_rule_management:6.1.4085`,
        fs:  `cpe:2.3:a:topdesk:topdesk:7.04.001:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:package_rule_management_language_pack:6.1.4085`,
        fs:  `cpe:2.3:a:mortbay:jetty:5.0:alpha3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:package_server_wmi_provider:6.0.1057`,
        fs:  `cpe:2.3:a:google:chrome:19.0.1077.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:patch_management:6.2.3643`,
        fs:  `cpe:2.3:a:ikiwiki:ikiwiki:3.20100804:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:patch_management_agent:6.1.1422`,
        fs:  `cpe:2.3:a:uangel:realtalk:a.0.9.250:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:patch_management_agent:6.2.2514`,
        fs:  `cpe:2.3:a:isc:bind:9.4:r1:*:*:esv:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:patch_management_core_solution_for_linux:6.2.1161`,
        fs:  `cpe:2.3:a:tm_software:tempo6.3.2:jira42:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:patch_management_dell_agent:6.2.2623`,
        fs:  `cpe:2.3:a:ulli_horlacher:fex:20120605:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:patch_management_for_windows:6.2.3644`,
        fs:  `cpe:2.3:a:google:v8:3.11.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:patch_management_for_windows_dell_servers:6.2.3647`,
        fs:  `cpe:2.3:a:ppfeufer:2-click-social-media-buttons:0.28:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:patch_management_for_windows_dell_servers_language_pack:6.2.3647`,
        fs:  `cpe:2.3:a:typo3:typo3:6.0.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:patch_management_for_windows_language_pack:6.2.3644`,
        fs:  `cpe:2.3:a:mcafee:virusscan:4.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:patch_management_language_pack:6.2.3643`,
        fs:  `cpe:2.3:h:saia_burgess_controls:pcd1.m0xx0:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:patch_management_solution_documentation:6.2.1060`,
        fs:  `cpe:2.3:a:cmsmadesimple:cms_made_simple:0.10.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:patch_management_solution_for_dell_servers_documentation:6.2.2002`,
        fs:  `cpe:2.3:h:dell:latitude_z600:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:power_control_task:6.0.1524`,
        fs:  `cpe:2.3:o:hp:procurve_switch_software:r.11.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:power_management_agent:6.0.1524`,
        fs:  `cpe:2.3:a:xcloner:xcloner:3.5.1:*:*:*:*:joomla\!:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:power_scheme_task:6.0.1015`,
        fs:  `cpe:2.3:a:web2py:web2py:1.78.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:power_scheme_task_language_pack:6.0.1002`,
        fs:  `cpe:2.3:a:springsource:spring_framework:4.0.0:m1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:pxe_manager:6.8`,
        fs:  `cpe:2.3:a:accadvocacy:acc_advocacy_action:2:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:pxe_manager:6.9`,
        fs:  `cpe:2.3:a:quest:knowledge_xpert_for_plsql:9.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:pxe_server:6.9`,
        fs:  `cpe:2.3:a:bestpractical:rt:3.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:real-time_console_infrastructure_language_pack:6.3.1022`,
        fs:  `cpe:2.3:a:ibm:rational_software_architect:8.5.5.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:real-time_system_manager_language_pack:6.3.1016`,
        fs:  `cpe:2.3:a:mcafee:virusscan:7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:realtime_console_infrastructure:6.3.1066`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.16.43:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:realtime_console_infrastructure_documentation:6.3.1006`,
        fs:  `cpe:2.3:a:google:chrome:11.0.663.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:realtime_system_manager:6.3.1066`,
        fs:  `cpe:2.3:a:drei:3kundenzone:2.0:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:realtime_system_manager_solution_documentation:6.3.1009`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(13\)t9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:recovery_solution:-`,
        fs:  `cpe:2.3:a:adobe:acrobat:7.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:report_pack_for_application_metering:6.1.1080`,
        fs:  `cpe:2.3:a:saflink:argus:2.1.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:report_pack_for_audit_integration_component:6.2.1004`,
        fs:  `cpe:2.3:a:bit51:better-wp-security:-:alpha9:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:report_pack_for_carbon_copy_solution:6.2.1024`,
        fs:  `cpe:2.3:a:gitlab:gitlab:8.6.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:report_pack_for_dell_client_manager:2.1.1004`,
        fs:  `cpe:2.3:a:ibm:rational_focal_point:6.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:report_pack_for_helpdesk_solution:6.0.1043`,
        fs:  `cpe:2.3:a:lxml:lxml:1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:report_pack_for_inventory_pack_for_servers:6.1.1039`,
        fs:  `cpe:2.3:a:mariadb:mariadb:5.2.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:report_pack_for_inventory_solution_for_mac:6.2.1007`,
        fs:  `cpe:2.3:a:mozilla:thunderbird:1.5.0.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:report_pack_for_inventory_solution_for_unix_and_linux:6.2.1004`,
        fs:  `cpe:2.3:a:strongswan:strongswan:4.2.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:report_pack_for_inventory_solution_for_windows:6.2.1047`,
        fs:  `cpe:2.3:a:codiad:codiad:1.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:report_pack_for_monitor_solution:6.0.1064`,
        fs:  `cpe:2.3:a:webyog:sqlyog:8.04:-:community:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:report_pack_for_monitor_solution_for_dell_servers:6.1.1063`,
        fs:  `cpe:2.3:o:cisco:adaptive_security_appliance_software:9.1.3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:report_pack_for_monitor_solution_for_servers:6.0.1112`,
        fs:  `cpe:2.3:a:ibm:storwize_v7000_unified_software:1.3.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:report_pack_for_network_discovery_solution:6.0.1026`,
        fs:  `cpe:2.3:a:fetchmail:fetchmail:5.9.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:report_pack_for_out_of_band_management_solution:6.2.1004`,
        fs:  `cpe:2.3:a:tweet-blender:tweet-blender:3.1.5:-:-:*:-:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:report_pack_for_patch_management_solution:6.2.3647`,
        fs:  `cpe:2.3:h:nvidia:geforce_gt_730:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:report_pack_for_patch_management_solution_for_windows:6.2.3644`,
        fs:  `cpe:2.3:h:hp:laserjet_4050:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:report_pack_for_patch_management_solutions_for_dell_servers:6.2.3665`,
        fs:  `cpe:2.3:a:vmware:player:3.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:report_pack_for_real_time_console_infrastructure:6.3.1004`,
        fs:  `cpe:2.3:a:aptana:aflax:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:report_pack_for_real_time_system_manager:6.3.1006`,
        fs:  `cpe:2.3:a:b2evolution:b2evolution:6.1.2:alpha:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:report_pack_for_software_delivery_solution:6.1.1101`,
        fs:  `cpe:2.3:a:ibm:soliddb:7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:report_pack_for_software_delivery_solution_for_windows:6.1.1080`,
        fs:  `cpe:2.3:a:ibm:storwize_v3500_software:7.1.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:report_pack_for_software_virtualization_solution_for_windows:2.1.1026`,
        fs:  `cpe:2.3:a:sap:3d_visual_enterprise_viewer:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:report_pack_for_symantec_ep_integration_component:6.0.1237`,
        fs:  `cpe:2.3:a:libpng:libpng:1.0.5j:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:report_pack_for_task_server:6.0.1007`,
        fs:  `cpe:2.3:o:ibm:power_5_system_firmware:sf240_358_201:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:resource_management:6.0.2927`,
        fs:  `cpe:2.3:a:search.cpan:libwww-perl:5.73:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:script_task:6.0.1524`,
        fs:  `cpe:2.3:a:microsoft:msn_messenger_service:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:script_task_agent:6.0.1524`,
        fs:  `cpe:2.3:a:zabbix:zabbix:2.0.11:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:security_audit_agent_solution:6.2.124`,
        fs:  `cpe:2.3:a:oracle:traffic_director:11.1.1.7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:securityexpressions:3.4.0`,
        fs:  `cpe:2.3:a:otrs:otrs:3.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:server_management_suite:-`,
        fs:  `cpe:2.3:a:checkpoint:web_intelligence:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:service_control_task:6.0.1524`,
        fs:  `cpe:2.3:a:inria:ocaml:2.04:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:service_control_task_agent:6.0.1524`,
        fs:  `cpe:2.3:a:pivotal_software:cloud_foundry_ops_manager:1.7.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:software_delivery_for_mac_documentation:6.1.1020`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(11\)bc3c:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:software_delivery_solution:6.1.1058.0`,
        fs:  `cpe:2.3:a:cmsmadesimple:cms_made_simple:1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:software_delivery_solution_agent:6.1.1016.0`,
        fs:  `cpe:2.3:a:redhat:richfaces:4.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:software_delivery_solution_documentation:6.1.1041`,
        fs:  `cpe:2.3:a:apple:quicktime:7.6.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:software_delivery_solution_for_mac:6.1.2032`,
        fs:  `cpe:2.3:a:apache:myfaces:1.0.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:software_delivery_solution_for_unix_and_linux:6.1.3006`,
        fs:  `cpe:2.3:h:hp:3com_switch:ss4_5500-si:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:software_delivery_solution_for_unix_and_linux_documentation:6.1.1028`,
        fs:  `cpe:2.3:a:watch4net:apg:4.3u3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:software_delivery_task_server_plug-in:6.1.1030`,
        fs:  `cpe:2.3:a:postfix:postfix:2.3.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:software_delivery_task_server_plug-in_language_pack:6.1.1030`,
        fs:  `cpe:2.3:a:ibm:sterling_selling_and_fulfillment_foundation:9.1.0.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:software_management:6.2.4270`,
        fs:  `cpe:2.3:a:oracle:vm_virtualbox:3.1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:software_management_language_pack:6.2.4274`,
        fs:  `cpe:2.3:a:google:chrome:7.0.530.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:software_virtualization_agent:2.1.2095`,
        fs:  `cpe:2.3:a:splunk:splunk:6.1.5:*:*:*:enterprise:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:software_virtualization_solution:2.1.2095`,
        fs:  `cpe:2.3:a:autodesk:revit_structure:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:software_virtualization_solution:2.1.3062`,
        fs:  `cpe:2.3:a:mybb:mybb:1.4.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:software_virtualization_solution_documentation:2.1.1009`,
        fs:  `cpe:2.3:o:juniper:junos:11.4r13:s2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:svs_task:6.3.0`,
        fs:  `cpe:2.3:a:ibm:lotus_notes_sametime:7.5.70413:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:svs_task:6.4.0`,
        fs:  `cpe:2.3:a:ibm:security_access_manager_for_web:7.0.0.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:svs_task_server_plugin_language_pack:6.3.0`,
        fs:  `cpe:2.3:a:netiq:appmanager_console:7.0.10160.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:svs_task_server_plugin_language_pack:6.4.0`,
        fs:  `cpe:2.3:a:oracle:database_server:7.3.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:symantec_endpoint_protection_integration_component:6.0.1171`,
        fs:  `cpe:2.3:a:ibm:security_key_lifecycle_manager:2.6.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:symantec_endpoint_protection_integration_component_documentation:6.0.1008`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.5.52:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:task_management:6.0.1524`,
        fs:  `cpe:2.3:a:dest-unreach:socat:1.2.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:task_management_pack:6.0.1504`,
        fs:  `cpe:2.3:a:live555:streaming_media:2011.10.05:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:task_management_pack_language_pack:6.0.1033`,
        fs:  `cpe:2.3:a:siteminder_agent_for_sharepoint:2010:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:task_management_pack_language_pack:6.0.1504`,
        fs:  `cpe:2.3:a:tecnick:tcexam:11.1.002:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:task_server_documentation:6.0.1013`,
        fs:  `cpe:2.3:a:artifex:gpl_ghostscript:8.54:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:task_server_language_pack:6.0.1404`,
        fs:  `cpe:2.3:a:ruby-i18n:i18n:0.6.8:-:-:*:-:ruby:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:task_server_language_pack:6.0.1509`,
        fs:  `cpe:2.3:a:syntevo:smartsvn:6.0:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:task_synchronization_agent:6.1.1030.0`,
        fs:  `cpe:2.3:a:autodesk:survey:2005:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:task_synchronization_agent:6.1.1034.0`,
        fs:  `cpe:2.3:a:apache:struts:2.5.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:topology_viewer:6.0.1.0`,
        fs:  `cpe:2.3:a:wp_plugin_manager_project:wp_plugin_manager:1.6.4.b:-:-:*:-:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:veritas_configuration_manager_integration_component:6.0.1005`,
        fs:  `cpe:2.3:o:cisco:ios:12.4sw:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altiris:wise_toolkit:6.1.3`,
        fs:  `cpe:2.3:a:pidgin:pidgin:2.10.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altitude:altitude_unified_customer_interaction:7.5`,
        fs:  `cpe:2.3:a:mozilla:bugzilla:3.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altn:mdaemon:9.6.4`,
        fs:  `cpe:2.3:a:acronis:backup_\&amp;_recovery_universal_restore:10.0.13545:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altn:mdaemon:9.6.5`,
        fs:  `cpe:2.3:a:qnap:qsync:4.2.2.0724:*:*:*:*:windows:*:*`,
    },
     
    {
        uri: `cpe:/a:altn:mdaemon:12.5.4:-:free`,
        fs:  `cpe:2.3:a:techsmith:snagit:8.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altnet:altnet_download_manager:4.0.0.2`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.4:rc4:*:*:*:*:x86:*`,
    },
     
    {
        uri: `cpe:/a:altnet:altnet_download_manager:4.0.0.4`,
        fs:  `cpe:2.3:a:sun:java_system_directory_server:6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altnet:altnet_download_manager:4.0.0.6`,
        fs:  `cpe:2.3:a:bandisoft:bandizip:3.07:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altova:altova_installer_for_apache_fop_for_jdk_1.4:0.93`,
        fs:  `cpe:2.3:a:fedoraproject:sssd:1.8.0:beta1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altova:databasespy_2008:2.1`,
        fs:  `cpe:2.3:a:citrix:xendesktop:5.6:fp1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altova:databasespy_2011:2.1::~~enterprise~~~`,
        fs:  `cpe:2.3:a:pivotal_software:rabbitmq:3.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altova:databasespy_2011:2.1::~~professional~~~`,
        fs:  `cpe:2.3:a:phpmyfaq:phpmyfaq:1.4.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altova:databasespy_2011:3.0::~~professional~~~`,
        fs:  `cpe:2.3:a:kaseya:virtual_system_administrator:8.0.0.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altova:databasespy_2012:-::~~enterprise~~~`,
        fs:  `cpe:2.3:a:sun:iplanet_web_server:4.1:sp5:enterprise:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altova:diffdog_2008:2.1`,
        fs:  `cpe:2.3:a:icofx:icofx:1.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altova:diffdog_2011:-:sp1:~~enterprise~~~`,
        fs:  `cpe:2.3:a:libpng:libpng:1.2.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altova:diffdog_2011:2.0::~~professional~~~`,
        fs:  `cpe:2.3:a:mediawiki:mediawiki:1.5:beta2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altova:missionkit_2009:0.1`,
        fs:  `cpe:2.3:o:cisco:clean_access:4.0.4.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altova:missionkit_2011:2.1`,
        fs:  `cpe:2.3:a:oracle:peoplesoft_enterprise:8.97.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altova:missionkit_2012:-`,
        fs:  `cpe:2.3:a:florian_weber:spaces:6.x-3.0:beta4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altova:missionkit_2012:0.1`,
        fs:  `cpe:2.3:a:zarafa:zarafa:6.20.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altova:missionkit_for_enterprise_software_architects:2010.00.00`,
        fs:  `cpe:2.3:a:wordfence_security_project:wordfence_security:5.0.2:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:altova:umodel_2012:-::~~enterprise~~~`,
        fs:  `cpe:2.3:a:krzysztof_kowalczyk:sumatrapdf:1.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altova:xmlspy_2006:30.1::~~professional~~~`,
        fs:  `cpe:2.3:a:nodejs:node.js:4.7.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altova:xmlspy_2007:30.0::~~enterprise~~~`,
        fs:  `cpe:2.3:a:intel:network_connections:14.6.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altova:xmlspy_2008:2.1::~~enterprise~~~`,
        fs:  `cpe:2.3:a:mysql:mysql:5.1.34:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altova:xmlspy_2009:0.1::~~professional~~~`,
        fs:  `cpe:2.3:h:enterasys:6h258-17:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altova:xmlspy_2011:2.0::~~professional~~~`,
        fs:  `cpe:2.3:o:google:chrome:21.0.1180.47:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altova:xmlspy_2011:3.0::~~professional~~~`,
        fs:  `cpe:2.3:a:digi:www_server:compieuw:beta2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altova:xmlspy_2011:3.1::~~professional~~~`,
        fs:  `cpe:2.3:a:disney:maleficent_free_fall:1.2.0:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:altova:xmlspy_2012:-::~~professional~~~`,
        fs:  `cpe:2.3:a:cisco:unity_connection:6.1\(3b\)su1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altova:xmlspy_2012:0.1::~~professional~~~`,
        fs:  `cpe:2.3:a:cisco:telepresence_video_communication_server_software:x6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:altsoft:xml2pdf_workstation_2007:1.2.34`,
        fs:  `cpe:2.3:a:ibm:lotus_inotes:8.0.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alvaro_herrera:pl%2fphp:1.0`,
        fs:  `cpe:2.3:a:google:chrome:9.0.597.107:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alvaro_herrera:pl%2fphp:1.1`,
        fs:  `cpe:2.3:a:tibco:formvine:3.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alvaro_herrera:pl%2fphp:1.2`,
        fs:  `cpe:2.3:a:susumu_terao:terapad:0.03:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alvaro_herrera:pl%2fphp:1.3.1`,
        fs:  `cpe:2.3:a:nlnetlabs:nsd:3.2.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alvaro_herrera:pl%2fphp:1.3.2`,
        fs:  `cpe:2.3:a:redhat:lvm2:2.02.52:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alvaro_herrera:pl%2fphp:1.3.3`,
        fs:  `cpe:2.3:a:otrs:otrs:3.1.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alvaro_herrera:pl%2fphp:1.3.5:beta1`,
        fs:  `cpe:2.3:o:cray:unicos:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:alvaro_herrera:pl%2fphp:1.4`,
        fs:  `cpe:2.3:o:microsoft:windows_nt:4.0:sru:server:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amanda:amanda:-`,
        fs:  `cpe:2.3:a:ffmpeg:ffmpeg:0.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amanda:amanda:2.3.0.4`,
        fs:  `cpe:2.3:a:feedweb:feedweb:1.5.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amazon:ec2_api_tools_java_library:-`,
        fs:  `cpe:2.3:a:keepass:keepass:1.23:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amazon:elastic_load_balancing_api_tools:-`,
        fs:  `cpe:2.3:a:openssl:openssl:0.9.6d:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amazon:elastic_load_balancing_api_tools:1.0`,
        fs:  `cpe:2.3:a:adobe:acrobat_reader_dc:15.009.20071:*:*:*:continuous:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amazon:elastic_load_balancing_api_tools:1.0:1`,
        fs:  `cpe:2.3:a:ibm:qradar_incident_forensics:7.2.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amazon:elastic_load_balancing_api_tools:1.0.3.4`,
        fs:  `cpe:2.3:a:gridgain:gridgain:8.0.3:ea12:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amazon:elastic_load_balancing_api_tools:1.0.9.3`,
        fs:  `cpe:2.3:a:tenable:nessus:6.9.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amazon:elastic_load_balancing_api_tools:1.0.10.0`,
        fs:  `cpe:2.3:a:zulip:zulip_server:1.3.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amazon:elastic_load_balancing_api_tools:1.0.11.1`,
        fs:  `cpe:2.3:a:nttdocomo:overseas_usage:2.0.4:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:amazon:elastic_load_balancing_api_tools:1.0.12.0`,
        fs:  `cpe:2.3:o:ibm:advanced_management_module_firmware:3.64:bpet64b:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amazon:elastic_load_balancing_api_tools:1.0.14.3`,
        fs:  `cpe:2.3:a:views_project:views:7.x-3.0:alpha1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amazon:elastic_load_balancing_api_tools:1.0.15.1`,
        fs:  `cpe:2.3:h:nortel:alteon_2424_application_switch:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amazon:elastic_load_balancing_api_tools:1.0.17.0`,
        fs:  `cpe:2.3:a:icofx:icofx:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amazon:flexible_payments_service:-`,
        fs:  `cpe:2.3:a:andy_armstrong:cgi.pm:2.52:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amazon:kindle:4.4.0::~~~android~~`,
        fs:  `cpe:2.3:a:ibm:tivoli_access_manager_identity_manager_solution:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amazon:kindle:4.4.4::~~~android~~`,
        fs:  `cpe:2.3:a:digium:asterisk:1.6.0:beta8:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amazon:kindle:4.5.0::~~~android~~`,
        fs:  `cpe:2.3:a:polycom:hdx_system_software:3.0.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amazon:kindle:4.6.0::~~~android~~`,
        fs:  `cpe:2.3:a:google:chrome:5.0.342.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amazon:kindle_for_pc:1.3.0.30884`,
        fs:  `cpe:2.3:o:iij:seil_x86_fuji_firmware:2.63:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amazon:merchant_sdk:-`,
        fs:  `cpe:2.3:a:google:chrome:33.0.1750.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amazon_aws_project:amazon_aws:7.x-1.2::~~~drupal~~`,
        fs:  `cpe:2.3:h:apple:ipod_touch:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amberdms:amberdms_billing_system:1.0.0`,
        fs:  `cpe:2.3:a:ibm:db2_recovery_expert:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amberdms:amberdms_billing_system:1.1.0`,
        fs:  `cpe:2.3:a:x:x_display_manager:1.1.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amberdms:amberdms_billing_system:1.2.0`,
        fs:  `cpe:2.3:a:gridgain:gridgain:8.1.3:p2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amberdms:amberdms_billing_system:1.3.0`,
        fs:  `cpe:2.3:a:sixapart:movabletype:5.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amberdms:amberdms_billing_system:1.4.0`,
        fs:  `cpe:2.3:h:cisco:spa_504g:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amberdms:amberdms_billing_system:1.4.1`,
        fs:  `cpe:2.3:a:searchblox:searchblox:8.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amberfog:vk_amberfog:3.5.6::~~~android~~`,
        fs:  `cpe:2.3:a:pulsecms:pulse_cms:1.4.5.1:-:pro:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amcharts:flash:1.0`,
        fs:  `cpe:2.3:a:ca:messaging:1.11.19.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:amd:16h_model_00h_processor:-`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.8.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:amd:16h_model_0fh_processor:-`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.11.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:amd:16h_model_processor_firmware:-`,
        fs:  `cpe:2.3:o:paloaltonetworks:pan-os:5.1.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:amd:amd64:-`,
        fs:  `cpe:2.3:a:avast:email_server_security:8.0.1609:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amd:app_sdk_runtime:2.4.595.10`,
        fs:  `cpe:2.3:a:mysql:eventum:1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amd:app_sdk_runtime:2.4.650.9`,
        fs:  `cpe:2.3:o:asus:rt-g32_firmware:2.0.2.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amd:app_sdk_runtime:2.5.684.213`,
        fs:  `cpe:2.3:o:cisco:ios:12.0\(2\)xc:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amd:app_sdk_runtime:2.5.793.1`,
        fs:  `cpe:2.3:h:xerox:document_centre_440:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amd:catalyst_driver:-`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.7:p37:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amd:power_monitor:1.0.1.47`,
        fs:  `cpe:2.3:a:apple:apple_tv:4.4.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amebra_ameba_project:amebra_ameba:1.0.0::~~~android~~`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.12.39:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amecuae:amgc:6::~~~android~~`,
        fs:  `cpe:2.3:a:novell:zenworks_configuration_management:11:sp1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:americostech:selfshot_front_flash_camera:1.1::~~~android~~`,
        fs:  `cpe:2.3:a:cisco:unified_communications_manager:5.1\(3a\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amiscu:michael_baker_federal_credit_union:1.2.0::~~~android~~`,
        fs:  `cpe:2.3:a:ati:catalyst_control_center_graphics_previews_common:2008.1201.1504.27008:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amiscu:westmoreland_water_fcu:1.2.0::~~~android~~`,
        fs:  `cpe:2.3:a:techsmith:snagit:4.3.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amos_benari:rbovirt:0.0.1::~~~ruby~~`,
        fs:  `cpe:2.3:a:atlassian:crowd:2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amos_benari:rbovirt:0.0.2::~~~ruby~~`,
        fs:  `cpe:2.3:a:caucho:resin:3.1.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amos_benari:rbovirt:0.0.3::~~~ruby~~`,
        fs:  `cpe:2.3:a:mysql:mysql:5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amos_benari:rbovirt:0.0.4::~~~ruby~~`,
        fs:  `cpe:2.3:a:adobe:acrobat:4.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amos_benari:rbovirt:0.0.5::~~~ruby~~`,
        fs:  `cpe:2.3:a:tryton:tryton:3.6.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amos_benari:rbovirt:0.0.6::~~~ruby~~`,
        fs:  `cpe:2.3:a:ponsoftware:explzh:3.62:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amos_benari:rbovirt:0.0.7::~~~ruby~~`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.2:*:*:*:*:*:x86:*`,
    },
     
    {
        uri: `cpe:/a:amos_benari:rbovirt:0.0.8::~~~ruby~~`,
        fs:  `cpe:2.3:a:ibm:http_server:6.0.2.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amos_benari:rbovirt:0.0.9::~~~ruby~~`,
        fs:  `cpe:2.3:a:microsoft:sharepoint_server:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amos_benari:rbovirt:0.0.10::~~~ruby~~`,
        fs:  `cpe:2.3:a:cms_tree_page_view_project:cms_tree_page_view:0.7.2:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:amos_benari:rbovirt:0.0.11::~~~ruby~~`,
        fs:  `cpe:2.3:a:oracle:mysql_connector\/j:3.0.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amos_benari:rbovirt:0.0.12::~~~ruby~~`,
        fs:  `cpe:2.3:a:opera:opera_browser:9.00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amos_benari:rbovirt:0.0.13::~~~ruby~~`,
        fs:  `cpe:2.3:o:hp:procurve_switch_software:r.11.32:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amos_benari:rbovirt:0.0.14::~~~ruby~~`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(18\)sxd7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amos_benari:rbovirt:0.0.15::~~~ruby~~`,
        fs:  `cpe:2.3:a:gitlab:gitlab:8.10.13:*:*:*:enterprise:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amos_benari:rbovirt:0.0.16::~~~ruby~~`,
        fs:  `cpe:2.3:a:ibm:tivoli_monitoring:6.3.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amos_benari:rbovirt:0.0.17::~~~ruby~~`,
        fs:  `cpe:2.3:a:wpeasycart:wp_easycart:2.1.4:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:amos_benari:rbovirt:0.0.18::~~~ruby~~`,
        fs:  `cpe:2.3:o:huawei:s9300e_firmware:v200r001:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amos_benari:rbovirt:0.0.19::~~~ruby~~`,
        fs:  `cpe:2.3:a:altiris:deployment_solution_for_network_devices:6.0.1046:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amos_benari:rbovirt:0.0.20::~~~ruby~~`,
        fs:  `cpe:2.3:a:ronan_dowling:node_hierarchy:6.x-1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amos_benari:rbovirt:0.0.21::~~~ruby~~`,
        fs:  `cpe:2.3:a:fengoffice:feng_office:2.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amos_benari:rbovirt:0.0.22::~~~ruby~~`,
        fs:  `cpe:2.3:a:bigtreecms:bigtree_cms:4.2.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amos_benari:rbovirt:0.0.23::~~~ruby~~`,
        fs:  `cpe:2.3:a:adodb_project:adodb:5.04:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amos_benari:rbovirt:0.0.24::~~~ruby~~`,
        fs:  `cpe:2.3:a:redhen_project:redhen:7.x-1.3:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:amos_benari:rbovirt:0.0.25::~~~ruby~~`,
        fs:  `cpe:2.3:a:freedesktop:poppler:0.23.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amos_benari:rbovirt:0.0.26::~~~ruby~~`,
        fs:  `cpe:2.3:a:microsoft:sql_srv:7.0:alpha:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:ampedwireless:r10000:-`,
        fs:  `cpe:2.3:a:hp:diagnostics_server:9.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:ampedwireless:r10000_firmware:2.5.2.11`,
        fs:  `cpe:2.3:a:git-scm:git:1.5.1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amtelco:misecuremessages:-`,
        fs:  `cpe:2.3:a:kolja_schleich:leaguemanager:3.1.8:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:amtelco:misecuremessages:6.2`,
        fs:  `cpe:2.3:a:webglimpse:webglimpse:2.16.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:amx:resource_management_suite:3.3.85`,
        fs:  `cpe:2.3:a:invisionpower:invision_power_board:3.4.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analects_of_confucius_project:analects_of_confucius:8::~~~android~~`,
        fs:  `cpe:2.3:a:dart:powertcp_ftp_for_activex:2.0.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analog:soundmax:3.0`,
        fs:  `cpe:2.3:a:ibm:worklight:5.0.6.2:*:*:*:consumer:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analog:soundmax:5.10.01.4542`,
        fs:  `cpe:2.3:a:cisco:media_origination_system_suite:2.5\(1\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analog:soundmax:5.10.01.5450`,
        fs:  `cpe:2.3:a:hp:nonstop_seeview_server_gateway:d48.01:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analog:soundmax:5.10.01.5491`,
        fs:  `cpe:2.3:a:standardchartered:breeze_jersey:1:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:analog:soundmax:5.10.01.5851`,
        fs:  `cpe:2.3:a:umn:mapserver:5.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analog:soundmax:5.10.01.6380`,
        fs:  `cpe:2.3:o:cisco:cisco_ons_15454_system_software:9.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analog:soundmax:5.10.01.7255`,
        fs:  `cpe:2.3:a:x.org:xorg-server:1.7.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analog:soundmax:5.10.01.7260`,
        fs:  `cpe:2.3:a:moinejf:abcm2ps:4.9.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analog:soundmax:5.10.01.7265`,
        fs:  `cpe:2.3:h:hp:0235a31n:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analog:soundmax:5.10.01.7270`,
        fs:  `cpe:2.3:a:sap:netweaver_exchange_infrastructure_\(bc-xi\):-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analog:soundmax:5.12.01.3890`,
        fs:  `cpe:2.3:a:rubyonrails:ruby_on_rails:4.2.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analog:soundmax:5.12.01.5246`,
        fs:  `cpe:2.3:a:jenkins:github_branch_source:1.4:beta-1:*:*:*:jenkins:*:*`,
    },
     
    {
        uri: `cpe:/a:analog:soundmax:5.12.01.7000`,
        fs:  `cpe:2.3:a:cisco:adaptive_security_appliance_software:8.0\(5.31\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analog:soundmax:5.12.02.5280`,
        fs:  `cpe:2.3:a:craig_drummond:cantata:0.7.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analog:soundmax:6.10.1.5450`,
        fs:  `cpe:2.3:a:kde:kdelibs:4.12.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analog:soundmax:6.10.2.7260`,
        fs:  `cpe:2.3:a:cisco:wireless_control_system_software:6.0.170.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analogx:atomic_timesync:-`,
        fs:  `cpe:2.3:a:argyllcms:argyllcms:1.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analogx:proxy:-`,
        fs:  `cpe:2.3:a:jquery:jquery:1.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analogx:proxy:4.0`,
        fs:  `cpe:2.3:a:webedition:webedition_cms:6.2.7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analogx:proxy:4.0.1`,
        fs:  `cpe:2.3:a:mysql:mysql:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analogx:proxy:4.0.2`,
        fs:  `cpe:2.3:o:cisco:skinny_client_control_protocol_software:8.2\(2\):sr4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analogx:proxy:4.0.3`,
        fs:  `cpe:2.3:o:cisco:ons_15454_optical_transport_platform:4.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analogx:proxy:4.0.4`,
        fs:  `cpe:2.3:h:dahuasecurity:dvr5108c:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analogx:proxy:4.0.5`,
        fs:  `cpe:2.3:a:gnome:eye_of_gnome:3.17.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analogx:proxy:4.0.6`,
        fs:  `cpe:2.3:a:ibm:iss_blackice_pc_protection:3.6:cpn:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analogx:proxy:4.0.7`,
        fs:  `cpe:2.3:a:avaya:vsu_7500:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analogx:proxy:4.4`,
        fs:  `cpe:2.3:a:alcatel-lucent:omnitouch_8670_automated_delivery_message_delivery_system:6.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analogx:proxy:4.13`,
        fs:  `cpe:2.3:a:apache:qpid:0.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analogx:simpleserver_shout:-`,
        fs:  `cpe:2.3:a:labtam-inc:proftp:2.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analogx:simpleserver_shout:1.0`,
        fs:  `cpe:2.3:a:interworx:web_control_panel:3.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analogx:simpleserver_www:-`,
        fs:  `cpe:2.3:a:atlassian:bamboo:4.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analogx:simpleserver_www:1.0.1`,
        fs:  `cpe:2.3:a:naist.jp:chasen:2.3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analogx:simpleserver_www:1.0.3`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analogx:simpleserver_www:1.0.6`,
        fs:  `cpe:2.3:a:adaptivecomputing:torque_resource_manager:2.4.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analogx:simpleserver_www:1.0.8`,
        fs:  `cpe:2.3:a:ibm:lotus_connections:4.0.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analogx:simpleserver_www:1.01`,
        fs:  `cpe:2.3:a:fedoraproject:sssd:1.11.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analogx:simpleserver_www:1.03`,
        fs:  `cpe:2.3:a:nq:easy_finder_\&amp;_anti-theft:2.0.10.08:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:analogx:simpleserver_www:1.04`,
        fs:  `cpe:2.3:a:otrs:otrs:2.4.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analogx:simpleserver_www:1.05`,
        fs:  `cpe:2.3:o:yamaha:rt105e:6.03.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analogx:simpleserver_www:1.06`,
        fs:  `cpe:2.3:a:bestpractical:rt:3.3.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analogx:simpleserver_www:1.16`,
        fs:  `cpe:2.3:a:david_king:vino:2.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:analytictech:netdraw:2.00.0097`,
        fs:  `cpe:2.3:a:icu-project:international_components_for_unicode:4.2.0.1:*:*:*:*:java:*:*`,
    },
     
    {
        uri: `cpe:/a:ananursespace:american_nurses_association:1.0.0::~~~android~~`,
        fs:  `cpe:2.3:a:bestpractical:rt:2.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anark:client:1.0`,
        fs:  `cpe:2.3:a:zabbix:zabbix:2.0.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anchorcms:anchor_cms:0.9.1`,
        fs:  `cpe:2.3:o:yamaha:rtx1000:7.01.49:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andre_wiethoff:exact_audio_copy:1.0beta1`,
        fs:  `cpe:2.3:a:david_stosik:comment_moderation:6.x-1.x-dev:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andreas_krennmair:tpp:1.3.1`,
        fs:  `cpe:2.3:a:gnu:gnutls:2.10.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_levine:multiblock:6.x-1.1`,
        fs:  `cpe:2.3:a:ibm:financial_transaction_manager:3.0.0.13:*:*:*:*:cps_services:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_levine:multiblock:6.x-1.2`,
        fs:  `cpe:2.3:a:ekahau:real-time_location_system_controller:6.0.5-final:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_levine:multiblock:6.x-1.3`,
        fs:  `cpe:2.3:h:hp:officejet_5610:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_levine:multiblock:6.x-1.x:dev`,
        fs:  `cpe:2.3:a:symantec:im_manager:8.4.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_levine:multiblock:7.x-1.0`,
        fs:  `cpe:2.3:a:bftpd_project:bftpd:4.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_levine:multiblock:7.x-1.0:beta1`,
        fs:  `cpe:2.3:a:google:chrome:30.0.1599.35:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_levine:multiblock:7.x-1.0:beta2`,
        fs:  `cpe:2.3:a:cerberusftp:ftp_server:4.0.7.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_levine:multiblock:7.x-1.x:dev`,
        fs:  `cpe:2.3:a:naist.jp:chasen:2.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_magdy_kamal%27s_network_project:andrew_magdy_kamal%27s_network:0.1::~~~android~~`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.5.43:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:1.3:beta`,
        fs:  `cpe:2.3:a:zeroclipboard_project:zeroclipboard:1.1.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:1.30`,
        fs:  `cpe:2.3:a:tenable:plugin-set:201402092115:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:1.31`,
        fs:  `cpe:2.3:a:sambar:sambar_server:4.3:beta9:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:1.32`,
        fs:  `cpe:2.3:a:ibm:filenet_idm_desktop:4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:1.40`,
        fs:  `cpe:2.3:a:dameware:mini_remote_control:5.0.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:1.41`,
        fs:  `cpe:2.3:a:atlassian:crucible:2.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:1.42`,
        fs:  `cpe:2.3:a:thomas_seidl:search_api:7.x-1.x:dev:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:1.50`,
        fs:  `cpe:2.3:a:southrivertech:titan_ftp_server:5.05.327:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:1.51`,
        fs:  `cpe:2.3:a:postgresql:postgresql:8.3.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:1.60`,
        fs:  `cpe:2.3:a:phpmyfaq:phpmyfaq:2.0.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:1.60a`,
        fs:  `cpe:2.3:a:cisco:jabber:9.0\(.5\):-:-:*:-:windows:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:1.61`,
        fs:  `cpe:2.3:a:webyog:sqlyog:6.07:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:1.62`,
        fs:  `cpe:2.3:a:phusion:passenger:3.0.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:1.62a`,
        fs:  `cpe:2.3:a:invensys:wonderware_application_server:2012:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:1.70`,
        fs:  `cpe:2.3:a:thinkshout:mandrill:7.x-1.x:dev:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:1.71`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.3.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:1.71a`,
        fs:  `cpe:2.3:h:hp:procurve_router:jf234a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:1.80`,
        fs:  `cpe:2.3:o:cisco:wireless_lan_controller_software:4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:1.81`,
        fs:  `cpe:2.3:a:bluecoat:unified_agent:4.6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:2.00`,
        fs:  `cpe:2.3:a:hp:openview_configuration_management:4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:2.01`,
        fs:  `cpe:2.3:a:horde:horde_application_framework:3.0.3:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:2.10`,
        fs:  `cpe:2.3:a:mawashimono:nikki:5.149:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:2.11`,
        fs:  `cpe:2.3:h:cisco:nexus_5010p_switch:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:2.20`,
        fs:  `cpe:2.3:a:webempoweredchurch:wec_discussion:1.6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:2.30`,
        fs:  `cpe:2.3:a:horde:horde_image_api:2.3.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:2.31`,
        fs:  `cpe:2.3:a:citrix:xenserver:6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:2.40`,
        fs:  `cpe:2.3:o:iij:seil\/x1_firmware:4.34:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:2.50`,
        fs:  `cpe:2.3:a:mybb:mybb:1.6.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:2.60`,
        fs:  `cpe:2.3:a:ibm:tivoli_application_dependency_discovery_manager:7.2.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:2.61`,
        fs:  `cpe:2.3:a:nomachine:nx_web_companion:2.0.0-1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:2.70`,
        fs:  `cpe:2.3:a:debian:dpkg:1.4.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:2.71`,
        fs:  `cpe:2.3:a:role_scoper_project:role_scoper:1.3.27:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:3.00`,
        fs:  `cpe:2.3:a:gridgain:gridgain:8.0.3:ea8:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:3.10`,
        fs:  `cpe:2.3:o:iij:seil_x2_firmware:3.40:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:3.20`,
        fs:  `cpe:2.3:a:haudenschilt:family_connections_cms:1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:3.21`,
        fs:  `cpe:2.3:a:topdesk:topdesk:6.09.001:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:3.30`,
        fs:  `cpe:2.3:a:oxid-esales:eshop:4.8.5:*:community:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andrew_simpson:webcollab:3.31`,
        fs:  `cpe:2.3:a:heimdal_project:heimdal:1.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:android_excellence_project:android_excellence:1.4.1::~~~android~~`,
        fs:  `cpe:2.3:a:ipswitch:ws_ftp:5.00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:androidcommunity:hector_leal:13.08.14::~~~android~~`,
        fs:  `cpe:2.3:a:microsoft:active_directory_federation_services:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:androidebookapp:healthy_lunch_diet_recipes:3.6.1::~~~android~~`,
        fs:  `cpe:2.3:a:dovecot:dovecot:2.1.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:androidsu:chainsdd_superuser:3.1.3::~~~android~~`,
        fs:  `cpe:2.3:o:iij:seil_b1_firmware:4.57:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:androkera:las_vegas_lottery_scratch_off:1.2::~~~android~~`,
        fs:  `cpe:2.3:a:cisco:adaptive_security_appliance_software:8.4\(.3.8\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andsocialrew:amkamal_science_portfolio:0.1::~~~android~~`,
        fs:  `cpe:2.3:a:google:chrome:8.0.552.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi-simple:0.078`,
        fs:  `cpe:2.3:a:tryton:tryton:3.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi-simple:0.079`,
        fs:  `cpe:2.3:a:drupal:drupal:7.40:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi-simple:0.080`,
        fs:  `cpe:2.3:a:forgerock:openam:10.1.0:*:*:*:xpress:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi-simple:0.081`,
        fs:  `cpe:2.3:a:ponsoftware:explzh:4.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi-simple:0.082`,
        fs:  `cpe:2.3:a:f5:big-ip_domain_name_system:12.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi-simple:0.83`,
        fs:  `cpe:2.3:a:walrus_digit:walrack:2.0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi-simple:1.0`,
        fs:  `cpe:2.3:a:oracle:human_capital_management_configuration_workbench:12.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi-simple:1.1`,
        fs:  `cpe:2.3:a:freedesktop:libdbus:1.5.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi-simple:1.1.1`,
        fs:  `cpe:2.3:a:apple:itunes:9.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi-simple:1.1.2`,
        fs:  `cpe:2.3:a:ibm:storwize_v7000_unified_software:1.3.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi-simple:1.103`,
        fs:  `cpe:2.3:a:cisco:secure_desktop:3.4.0373:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi-simple:1.104`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.34.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi-simple:1.105`,
        fs:  `cpe:2.3:a:pocketmags:gambling_insider_magazine:\@7f0801aa:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi-simple:1.106`,
        fs:  `cpe:2.3:a:videowhisper:video_presentation:3.25:-:-:*:-:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi-simple:1.107`,
        fs:  `cpe:2.3:a:pivotal_software:spring_framework:4.3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi-simple:1.108`,
        fs:  `cpe:2.3:o:microsoft:ms-dos:6.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi-simple:1.109`,
        fs:  `cpe:2.3:a:octopus:octopus_deploy:3.4.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi-simple:1.110`,
        fs:  `cpe:2.3:a:pivotal_software:cloud_foundry_uaa:3.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi-simple:1.111`,
        fs:  `cpe:2.3:a:mathjs:math.js:0.22.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi-simple:1.112`,
        fs:  `cpe:2.3:a:cakephp:cakephp:2.6.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:1.4`,
        fs:  `cpe:2.3:a:yahoo:messenger:9.0.0.2123:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:1.42`,
        fs:  `cpe:2.3:a:ibm:security_identity_manager:5.1.0.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:1.43`,
        fs:  `cpe:2.3:a:intel:pro_network_connections:10.1.41.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:1.44`,
        fs:  `cpe:2.3:a:hp:openview_performance_core:10.50.40:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:1.45`,
        fs:  `cpe:2.3:o:huawei:secospace_antiddos8000_firmware:v500r001c00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:1.50`,
        fs:  `cpe:2.3:a:kurzweiledu:kurzweil_3000:9.0.0:*:*:*:trial:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:1.51`,
        fs:  `cpe:2.3:a:ubercart:ubercart:7.x-3.0:beta3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:1.52`,
        fs:  `cpe:2.3:a:mediawiki:mediawiki:1.9.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:1.53`,
        fs:  `cpe:2.3:a:cisco:jabber:9.6\(2\):*:*:*:*:windows:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:1.54`,
        fs:  `cpe:2.3:a:shinephp:thank_you_counter_button:1.6.7:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:1.55`,
        fs:  `cpe:2.3:o:cisco:pix_firewall:6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:1.56`,
        fs:  `cpe:2.3:a:artifex:gpl_ghostscript:8.64:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:1.57`,
        fs:  `cpe:2.3:o:cisco:ios_xr:3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.0`,
        fs:  `cpe:2.3:a:sun:jre:1.4.2_18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.01`,
        fs:  `cpe:2.3:a:bestpractical:request_tracker:3.8.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.13`,
        fs:  `cpe:2.3:a:libguestfs:libguestfs:1.9.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.14`,
        fs:  `cpe:2.3:a:git-scm:git:1.6.6:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.15`,
        fs:  `cpe:2.3:a:webmin:usermin:0.90:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.16`,
        fs:  `cpe:2.3:a:syntevo:smartsvn:2.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.17`,
        fs:  `cpe:2.3:o:huawei:5310hi_firmware:v200r005:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.18`,
        fs:  `cpe:2.3:a:git-scm:git:2.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.19`,
        fs:  `cpe:2.3:a:ibm:host_on-demand:11.0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.20`,
        fs:  `cpe:2.3:a:hp:storage_operations_manager_eva_5000:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.21`,
        fs:  `cpe:2.3:a:cisco:telepresence_tc_software:7.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.22`,
        fs:  `cpe:2.3:a:igniterealtime:smack:3.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.23`,
        fs:  `cpe:2.3:a:sophos:web_appliance:3.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.24`,
        fs:  `cpe:2.3:a:hp:nonstop_seeview_server_gateway:d45.01:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.25`,
        fs:  `cpe:2.3:a:dhcpcd_project:dhcpcd:5.5.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.26`,
        fs:  `cpe:2.3:a:silverstripe:silverstripe:2.3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.27`,
        fs:  `cpe:2.3:a:quagga:quagga:0.99.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.28`,
        fs:  `cpe:2.3:a:adaptivecomputing:torque_resource_manager:4.2.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.29`,
        fs:  `cpe:2.3:a:adobe:acrobat:11.0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.30`,
        fs:  `cpe:2.3:a:shopware:shopware:5.0.3:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.31`,
        fs:  `cpe:2.3:a:redhat:subscription_asset_manager:1.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.32`,
        fs:  `cpe:2.3:a:theforeman:kafo:0.3.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.33`,
        fs:  `cpe:2.3:a:apache:batik:1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.34`,
        fs:  `cpe:2.3:a:cisco:adaptive_security_appliance_software:9.1\(1\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.35`,
        fs:  `cpe:2.3:a:rimarts:becky\!_internet_mail:2.64:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.36`,
        fs:  `cpe:2.3:a:redhat:jboss_enterprise_brms_platform:5.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.37`,
        fs:  `cpe:2.3:a:icu-project:international_components_for_unicode:3.0:*:*:*:*:c\/c\&#43;\&#43;:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.38`,
        fs:  `cpe:2.3:a:hp:storeonce_d2d:2.2.00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.39`,
        fs:  `cpe:2.3:a:google:chrome:19.0.1058.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.40`,
        fs:  `cpe:2.3:a:ibm:websphere_mq:7.0.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.41`,
        fs:  `cpe:2.3:a:ibm:storwize_v7000_unified_software:1.4.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.42`,
        fs:  `cpe:2.3:a:cisco:security_agent_management_center:5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.43`,
        fs:  `cpe:2.3:h:supermicro:h8dgt-hlibqf:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.44`,
        fs:  `cpe:2.3:a:prestashop:ebay_module:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.45`,
        fs:  `cpe:2.3:a:redhat:kdelibs_devel:3.1-10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.46`,
        fs:  `cpe:2.3:a:shavlik:netchk_protect:7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.47`,
        fs:  `cpe:2.3:a:gnu:bash:3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.48`,
        fs:  `cpe:2.3:a:docker:docker:0.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.49`,
        fs:  `cpe:2.3:a:hp:insight_management_agent:8.50.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.50`,
        fs:  `cpe:2.3:a:sun:jdk:1.1.6:update7:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.51`,
        fs:  `cpe:2.3:a:mobiloud:mobiloud:1.5.3:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.52`,
        fs:  `cpe:2.3:h:cisco:asr_1004:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.53`,
        fs:  `cpe:2.3:h:f5:big-ip_5110:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.54`,
        fs:  `cpe:2.3:o:apple:watchos:2.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.55`,
        fs:  `cpe:2.3:a:claroline:claroline:1.11.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.56`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.3.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.57`,
        fs:  `cpe:2.3:a:docker:docker-py:0.5.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.58`,
        fs:  `cpe:2.3:a:dlc_solutions:course:6.x-1.x-dev:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.59`,
        fs:  `cpe:2.3:a:splunk:splunk:6.2.10:*:*:*:enterprise:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.60`,
        fs:  `cpe:2.3:h:ovislink:airlive_poe200hd:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.61`,
        fs:  `cpe:2.3:a:activcard:activcard_gold_for_cac:2.24:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.62`,
        fs:  `cpe:2.3:h:d-link:di-624s:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.63`,
        fs:  `cpe:2.3:a:myapp:treves_dance_center:1:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.64`,
        fs:  `cpe:2.3:a:trianglemicroworks:scada_data_gateway:2.54.0571:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.65`,
        fs:  `cpe:2.3:a:apache:struts:2.5.19:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.66`,
        fs:  `cpe:2.3:a:jenkins:git:0.4.0:*:*:*:*:jenkins:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.67`,
        fs:  `cpe:2.3:a:openstack:pycadf:0.1.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.68`,
        fs:  `cpe:2.3:a:weathernews:weathernews_touch:1.23:-:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.69`,
        fs:  `cpe:2.3:h:cisco:ace_4700_series_application_control_engine_appliance:a5\(1.0\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.70`,
        fs:  `cpe:2.3:a:stunnel:stunnel:5.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.71`,
        fs:  `cpe:2.3:o:netgear:prosafe_firmware:5.4.1.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.72`,
        fs:  `cpe:2.3:a:cisco:optical_networking_systems_software:4.1.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.73`,
        fs:  `cpe:2.3:a:checkpoint:vpn-1_firewall_1:4.1.0:sp3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.74`,
        fs:  `cpe:2.3:o:cisco:nx-os:5.0\(5\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.75`,
        fs:  `cpe:2.3:a:tecnick:tcexam:11.2.030:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.76`,
        fs:  `cpe:2.3:a:pictobrowser_project:pictobrowser:0.3.1:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.77`,
        fs:  `cpe:2.3:a:hp:select_identity:4.13.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.78`,
        fs:  `cpe:2.3:a:entity_reference_project:entityreference:7.x-1.0:alpha2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.79`,
        fs:  `cpe:2.3:a:atlassian:jira:3.4.2:*:standard:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.80`,
        fs:  `cpe:2.3:a:apache:cxf:2.4.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.81`,
        fs:  `cpe:2.3:a:axel_jung:js_css_optimizer:1.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.82`,
        fs:  `cpe:2.3:h:cisco:firewall_services_module:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.83`,
        fs:  `cpe:2.3:o:microsoft:windows_vista:-:*:x86-enterprise:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.84`,
        fs:  `cpe:2.3:a:goforandroid:go_email_widget:1.81:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.85`,
        fs:  `cpe:2.3:a:stunnel:stunnel:5.08:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.86`,
        fs:  `cpe:2.3:a:zingiri:zingiri_web_shop:2.4.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.87`,
        fs:  `cpe:2.3:a:symantec:veritas_netbackup_java:1000.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.88`,
        fs:  `cpe:2.3:a:bftpd_project:bftpd:1.0.24:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.89`,
        fs:  `cpe:2.3:a:haudenschilt:family_connections_cms:2.9.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.90`,
        fs:  `cpe:2.3:a:wouter_verhelst:nbd:2.8.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.91`,
        fs:  `cpe:2.3:a:puppetlabs:puppet:2.7.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.92`,
        fs:  `cpe:2.3:a:ati:catalyst_control_center_utility:2011.1025.2152.37348:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.93`,
        fs:  `cpe:2.3:a:ca:etrust_siteminder_documentation:12.0.105.635:sp1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.94`,
        fs:  `cpe:2.3:a:sixapart:movabletype:5.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.95`,
        fs:  `cpe:2.3:a:microsoft:sna_server:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.96`,
        fs:  `cpe:2.3:a:debian:dpkg:1.7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.97`,
        fs:  `cpe:2.3:o:dlink:dsl-2640b_firmware:4.00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.98`,
        fs:  `cpe:2.3:a:ubercart:ubercart:6.x-2.0:rc3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.99`,
        fs:  `cpe:2.3:a:icinga:icinga:1.8.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.751`,
        fs:  `cpe:2.3:a:irfanview:irfanview:4.32:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:2.752`,
        fs:  `cpe:2.3:o:rockwellautomation:guardlogix_5570_controller_firmware:20.00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.00`,
        fs:  `cpe:2.3:a:perl:perl:2.6.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.01`,
        fs:  `cpe:2.3:o:hp:procurve_switch_software:h.07.02:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.02`,
        fs:  `cpe:2.3:a:haxx:libcurl:7.37.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.03`,
        fs:  `cpe:2.3:a:mozilla:netscape_portable_runtime:4.8.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.04`,
        fs:  `cpe:2.3:a:mathjs:math.js:3.7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.05`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.9.6-3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.06`,
        fs:  `cpe:2.3:a:webcreate:web_mart:1.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.07`,
        fs:  `cpe:2.3:a:gnome:glib:2.4.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.08`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.38.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.09`,
        fs:  `cpe:2.3:a:google:chrome:9.0.573.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.10`,
        fs:  `cpe:2.3:a:x.org:xorg-server:1.16.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.11`,
        fs:  `cpe:2.3:a:apache:httpclient:4.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.12`,
        fs:  `cpe:2.3:a:apache:groovy:2.1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.13`,
        fs:  `cpe:2.3:a:b2evolution:b2evolution:4.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.14`,
        fs:  `cpe:2.3:a:phpmyfaq:phpmyfaq:2.6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.15`,
        fs:  `cpe:2.3:a:typo3:typo3:4.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.16`,
        fs:  `cpe:2.3:a:hp:application_lifecycle_management:11.00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.17`,
        fs:  `cpe:2.3:a:gnome:evolution:2.22.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.18`,
        fs:  `cpe:2.3:a:buddypress:buddypress_plugin:1.5.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.19`,
        fs:  `cpe:2.3:a:mozilla:firefox_esr:38.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.20`,
        fs:  `cpe:2.3:a:ibm:websphere_application_server:6.0.2.24:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.21`,
        fs:  `cpe:2.3:a:pureftpd:pure-ftpd:1.0.22:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.22`,
        fs:  `cpe:2.3:a:mantisbt:mantisbt:0.19.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.23`,
        fs:  `cpe:2.3:o:hp:procurve_switch_software:r.11.07:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.24`,
        fs:  `cpe:2.3:a:elfutils_project:elfutils:0.142:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.25`,
        fs:  `cpe:2.3:o:freebsd:freebsd:2.2.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.26`,
        fs:  `cpe:2.3:a:redhat:icedtea-web:1.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.27`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.513:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.28`,
        fs:  `cpe:2.3:a:user_relationships_project:user_relationships:6.x-1.0:beta8:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.29`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.5:p128:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.30`,
        fs:  `cpe:2.3:a:oracle:e-business_suite:12.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.31`,
        fs:  `cpe:2.3:a:openstack:swift:1.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.32`,
        fs:  `cpe:2.3:a:octobercms:october_cms:1.0.348:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.33`,
        fs:  `cpe:2.3:a:hidemaru:editor:8.33:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.34`,
        fs:  `cpe:2.3:a:adobe:air_sdk_\&amp;_compiler:17.0.0.172:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.35`,
        fs:  `cpe:2.3:a:email\:\:address_module_project:email\:\:address:1.80:*:*:*:*:perl:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.36`,
        fs:  `cpe:2.3:a:sun:sdk:1.3.0_05:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.37`,
        fs:  `cpe:2.3:a:atlassian:fisheye:1.6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.38`,
        fs:  `cpe:2.3:a:futomi:mp_form_mail_cgi:2.0.12:*:*:*:ecommerce:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.39`,
        fs:  `cpe:2.3:a:cisco:unified_communications_manager:7.1\(5b\)su4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.40`,
        fs:  `cpe:2.3:a:octopus:octopus_deploy:3.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.41`,
        fs:  `cpe:2.3:a:oracle:mysql_connector\/net:1.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.42`,
        fs:  `cpe:2.3:a:adobe:acrobat:6.0:-:pro:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.43`,
        fs:  `cpe:2.3:a:microsoft:orca:4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.44`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.11.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.45`,
        fs:  `cpe:2.3:h:hp:5500_network_printer:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.46`,
        fs:  `cpe:2.3:a:dadabik:dadabik:1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.47`,
        fs:  `cpe:2.3:a:cisco:adaptive_security_appliance_software:9.2.4.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.48`,
        fs:  `cpe:2.3:a:microsoft:isa_server:2000:fp1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:andy_armstrong:cgi.pm:3.49`,
        fs:  `cpe:2.3:a:nullsoft:winamp:5.32:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anecms:anecms:-`,
        fs:  `cpe:2.3:a:x:libxres:1.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anecms:anecms:2e2c583`,
        fs:  `cpe:2.3:a:hitachi:cosminexus_application_server:06-50-\/b:-:enterprise:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:6.x-1.0`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(5\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:6.x-1.0:alpha1`,
        fs:  `cpe:2.3:h:rockwellautomation:compactlogix_1769-l24er-qbfc1b:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:6.x-1.0:alpha2`,
        fs:  `cpe:2.3:a:atmail:atmail:6.6.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:6.x-1.0:alpha3`,
        fs:  `cpe:2.3:a:novell:zenworks_servers:3.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:6.x-1.0:beta1`,
        fs:  `cpe:2.3:a:octobercms:october_cms:1.0.135:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:6.x-1.0:beta2`,
        fs:  `cpe:2.3:a:hp:insight_control_performance_management:6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:6.x-1.0:beta3`,
        fs:  `cpe:2.3:a:toocharger:myboard:1.0.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:6.x-1.0:beta4`,
        fs:  `cpe:2.3:a:nagios:remote_plug_in_executor:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:6.x-1.0:rc1`,
        fs:  `cpe:2.3:a:puppetlabs:puppet_enterprise_users:1.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:6.x-1.1`,
        fs:  `cpe:2.3:a:digium:asterisk:0.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:6.x-1.2`,
        fs:  `cpe:2.3:a:oracle:financial_services_software:5.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:6.x-1.3`,
        fs:  `cpe:2.3:a:adobe:acrobat_reader:5.0.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:6.x-1.4`,
        fs:  `cpe:2.3:a:dominique_clause:search_autocomplete:6.x-2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:6.x-1.5`,
        fs:  `cpe:2.3:a:debian:trousers:0.3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:6.x-1.6`,
        fs:  `cpe:2.3:a:bas_van_beek:multishop:1.0.24:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:6.x-1.7`,
        fs:  `cpe:2.3:o:redhat:enterprise_linux:5:ga:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:6.x-1.8`,
        fs:  `cpe:2.3:a:topdesk:topdesk:6.06.004:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:6.x-1.9`,
        fs:  `cpe:2.3:a:google:chrome:33.0.1750.67:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:6.x-1.x:dev`,
        fs:  `cpe:2.3:a:cotonti:cotonti_siena:0.9.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:7.x-1.0`,
        fs:  `cpe:2.3:o:cisco:ios:12.1\(4\)dc:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:7.x-1.0:alpha1`,
        fs:  `cpe:2.3:a:faronics:deep_freeze:7.22.020.3453:-:std:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:7.x-1.0:alpha2`,
        fs:  `cpe:2.3:a:ibm:db2_connect:10.1.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:7.x-1.0:alpha3`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.7.4-7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:7.x-1.0:alpha4`,
        fs:  `cpe:2.3:a:cisco:telepresence_server_software:3.1\(1.82\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:7.x-1.0:beta1`,
        fs:  `cpe:2.3:a:emc:powerpath:4.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:7.x-1.0:dev`,
        fs:  `cpe:2.3:a:microsoft:visio:2002:sp2:pro:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:7.x-1.0:rc1`,
        fs:  `cpe:2.3:a:rockmongo:rockmongo:1.0.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:7.x-1.0:rc2`,
        fs:  `cpe:2.3:a:apple:quicktime:7.3.0:-:windows:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:7.x-1.1`,
        fs:  `cpe:2.3:a:apple:safari:3.0.1b:-:windows:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:7.x-1.2`,
        fs:  `cpe:2.3:a:rpm:rpm:2.2.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angrydonuts:ctools:7.x-1.3`,
        fs:  `cpe:2.3:h:intel:active_management_technology:3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:0.5.0.1`,
        fs:  `cpe:2.3:a:eclipse:eclipse_ide:3.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:1.0.0.5`,
        fs:  `cpe:2.3:a:hp:onboard_administrator:3.50:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:2.0.1.2`,
        fs:  `cpe:2.3:a:pocoproject:poco_c\&#43;\&#43;_libraries:1.4.6:p1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:2.1.1.4`,
        fs:  `cpe:2.3:a:cacti:cacti:1.1.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:2.2.0.1`,
        fs:  `cpe:2.3:a:apache:axis:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:2.3.0.1`,
        fs:  `cpe:2.3:a:dell:openmanage_server_administrator:7.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:2.3.0.2`,
        fs:  `cpe:2.3:o:yahama:rt300i:6.03.34:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:2.3.0.3`,
        fs:  `cpe:2.3:a:cms_tree_page_view_project:cms_tree_page_view:0.7.11:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:2.3.0.5`,
        fs:  `cpe:2.3:a:tweet-blender:tweet-blender:3.3.8:-:-:*:-:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:2.3.0.6`,
        fs:  `cpe:2.3:a:microsys:promotic:8.3.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:2.4.0.2`,
        fs:  `cpe:2.3:a:nginx:nginx:1.4.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:2.4.0.3`,
        fs:  `cpe:2.3:a:google:chrome:11.0.696.77:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:2.4.0.4`,
        fs:  `cpe:2.3:a:rom_walton:boinc:7.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:2.5.0`,
        fs:  `cpe:2.3:a:posh_project:posh:3.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:2.6.0`,
        fs:  `cpe:2.3:o:supermicro:intelligent_platform_management_firmware:2.26:-:-:*:-:-:x9_generation_motherboards:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:3.0.0`,
        fs:  `cpe:2.3:a:ibm:urbancode_deploy:6.1.1.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:3.1.0`,
        fs:  `cpe:2.3:h:softbank:samsung_3g_handset:941sc:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:3.2.1`,
        fs:  `cpe:2.3:a:ibm:tivoli_application_dependency_discovery_manager:7.2.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:3.2.2`,
        fs:  `cpe:2.3:a:ibm:qradar_risk_manager:7.2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:3.2.3`,
        fs:  `cpe:2.3:a:redhat:libreport:2.0.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:3.2.4`,
        fs:  `cpe:2.3:a:moinejf:abcm2ps:1.6.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:3.2.5`,
        fs:  `cpe:2.3:o:yahama:rtx1000:7.01.41:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:3.2.6`,
        fs:  `cpe:2.3:a:isaac_sukin:shorten:6.x-1.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:3.2.7`,
        fs:  `cpe:2.3:a:huawei:webui:11.010.06.01.858:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:3.2.8`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.7:p253:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:3.2.9`,
        fs:  `cpe:2.3:a:catalin_florian_radut:zeropoint:7.x-1.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:3.2.10`,
        fs:  `cpe:2.3:a:hp:insight_diagnostics_online_edition_for_windows:8.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:3.3.1`,
        fs:  `cpe:2.3:a:apache:http_server:1.3.37:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:3.3.2`,
        fs:  `cpe:2.3:a:google:chrome:24.0.1312.53:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:3.3.3`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(33\)sxj8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:3.3.4`,
        fs:  `cpe:2.3:a:x.org:xorg-server:1.10.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:3.4.0`,
        fs:  `cpe:2.3:a:robert_ancell:lightdm:1.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:3.5.0`,
        fs:  `cpe:2.3:a:sunil_nanda:blue_wrench_video_widget:1.0.3:-:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:angusj:resource_hacker:3.6.0.92`,
        fs:  `cpe:2.3:o:yahama:rt102i:2.02.33:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:animalcenter:light_for_pets:1::~~~android~~`,
        fs:  `cpe:2.3:a:search.cpan:libwww-perl:5.67:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:animoca:bunny_run:1.1.2::~~~android~~`,
        fs:  `cpe:2.3:h:securecomputing:snapgear:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:animoca:fashion_style:3.4.1::~~~android~~`,
        fs:  `cpe:2.3:a:ca:desktop_management_suite:11.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:animoca:star_girl:3.4.1::~~~android~~`,
        fs:  `cpe:2.3:a:archangelmgt:weblog:0.90.02:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:animoca:star_girl%3a_colors_of_spring:3.4.1::~~~android~~`,
        fs:  `cpe:2.3:a:apache:knox:0.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anixis:password_policy_client:3.5`,
        fs:  `cpe:2.3:a:libevent_project:libevent:2.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anixis:password_policy_client:3.6.0.1`,
        fs:  `cpe:2.3:o:watchguard:fireware:11.7.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anixis:password_policy_client:4.5`,
        fs:  `cpe:2.3:a:redhat:jboss_enterprise_application_platform:7.0.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anixis:password_policy_client:5.1`,
        fs:  `cpe:2.3:a:lansweeper:lansweeper:5.3.0.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anixis:password_policy_enforcer:3.6`,
        fs:  `cpe:2.3:a:apache:subversion:1.5.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anixis:password_policy_enforcer:4.5`,
        fs:  `cpe:2.3:a:sgminer_project:sgminer:4.1.242:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anixis:password_policy_enforcer:5.11`,
        fs:  `cpe:2.3:h:cabletron:6h302-48:4.08.50:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anjuke:anjuke:7.1.7::~~~android~~`,
        fs:  `cpe:2.3:a:connect2id:nimbus_jose\&#43;jwt:1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anl:bcfg2:0.9.6`,
        fs:  `cpe:2.3:a:tejimaya:openpne:2.14.0:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anl:bcfg2:1.0.1`,
        fs:  `cpe:2.3:a:hp:proliant_ilo_management_controller_package:3.3.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anl:bcfg2:1.1.3`,
        fs:  `cpe:2.3:a:apache:apache-ssl:1.37:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anl:bcfg2:1.2.0`,
        fs:  `cpe:2.3:a:connections_project:connections:0.2.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anl:bcfg2:1.2.0:rc2`,
        fs:  `cpe:2.3:a:mathjs:math.js:0.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anl:bcfg2:1.2.3`,
        fs:  `cpe:2.3:a:horde:internet_mail_program:6.1.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anonymous_posting_project:anonymous_posting:7.x-1.2`,
        fs:  `cpe:2.3:a:symantec:data_center_security_server:6.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anonymous_posting_project:anonymous_posting:7.x-1.3`,
        fs:  `cpe:2.3:a:bea:weblogic_workshop:8.1:sp6:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anritsu:traceview:3.1`,
        fs:  `cpe:2.3:a:phponlinechat:phponlinechat:3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anshul_sharma:category-grid-view-gallery:0.1.0`,
        fs:  `cpe:2.3:a:techsmith:camtasia_studio:5.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anshul_sharma:category-grid-view-gallery:0.1.1`,
        fs:  `cpe:2.3:a:irfanview:irfanview:4.35:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anshul_sharma:category-grid-view-gallery:0.2.0`,
        fs:  `cpe:2.3:a:openstack:keystone:juno-1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anshul_sharma:category-grid-view-gallery:2.0.0`,
        fs:  `cpe:2.3:a:google:chrome:5.0.363.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anshul_sharma:category-grid-view-gallery:2.2`,
        fs:  `cpe:2.3:a:polycom:hdx_system_software:3.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anshul_sharma:category-grid-view-gallery:2.2.1`,
        fs:  `cpe:2.3:a:gimp:gimp:2.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anshul_sharma:category-grid-view-gallery:2.2.3`,
        fs:  `cpe:2.3:a:microsoft:office:2008:*:mac:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anshul_sharma:category-grid-view-gallery:2.2.4`,
        fs:  `cpe:2.3:a:ibm:storwize_v5000_software:7.1.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anshul_sharma:category-grid-view-gallery:2.3`,
        fs:  `cpe:2.3:a:cisco:ciscoworks_access_control_list_manager:1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anshul_sharma:category-grid-view-gallery:2.3.1`,
        fs:  `cpe:2.3:a:microstrategy:microstrategy:9.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anshul_sharma:category-grid-view-gallery:2.3.3`,
        fs:  `cpe:2.3:h:linksys:befw11s4_v3:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ansible:tower:2.0.4`,
        fs:  `cpe:2.3:a:horde:groupware:1.0.6:*:webmail:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ansibleworks:ansible:1.1`,
        fs:  `cpe:2.3:a:jenkins:role-based_authorization_strategy:2.5.0:*:*:*:*:jenkins:*:*`,
    },
     
    {
        uri: `cpe:/a:ansibleworks:ansible:1.2`,
        fs:  `cpe:2.3:a:adobe:acrobat_reader_dc:15.016.20041:*:*:*:continuous:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ansibleworks:ansible:1.2.1`,
        fs:  `cpe:2.3:a:nginx:nginx:1.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ansibleworks:ansible:1.2.2`,
        fs:  `cpe:2.3:a:intel:iparty:1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ansibleworks:ansible:1.2.3`,
        fs:  `cpe:2.3:a:google:chrome:24.0.1312.31:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ansibleworks:ansible:1.3.0`,
        fs:  `cpe:2.3:a:emc:vplex_geosynchrony:5.4:sp1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ansibleworks:ansible:1.3.1`,
        fs:  `cpe:2.3:a:marketpress:backwpup_plugin:3.0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ansibleworks:ansible:1.9.1`,
        fs:  `cpe:2.3:a:rubyonrails:ruby_on_rails:0.12.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ansibleworks:ansible:1.9.6`,
        fs:  `cpe:2.3:a:oracle:corporate_time_outlook_connector:3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ansibleworks:ansible:2.0`,
        fs:  `cpe:2.3:a:ca:spectrum:9.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ansibleworks:ansible:2.0.1`,
        fs:  `cpe:2.3:a:yahoo:yahoo\!_browser:1.2.0:-:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:ansys:ls-dyna:-`,
        fs:  `cpe:2.3:a:joedolson:my-calendar:1.9.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ansys:ls-dyna:7.1`,
        fs:  `cpe:2.3:a:litespeedtech:litespeed_web_server:4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ansys:ls-dyna:8.0`,
        fs:  `cpe:2.3:a:mediawiki:mediawiki:1.27.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ansys:ls-dyna:9.0`,
        fs:  `cpe:2.3:a:wesnoth:battle_for_wesnoth:1.9.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ansys:mechanical:-`,
        fs:  `cpe:2.3:a:ssh:tectia_server:5.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ansys:mechanical:7.1`,
        fs:  `cpe:2.3:a:ibm:net.commerce_hosting_server:3.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ansys:mechanical:8.0`,
        fs:  `cpe:2.3:o:cisco:ios_xe:3.9.1as:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:ansys:mechanical:9.0`,
        fs:  `cpe:2.3:a:hitachi:ucosminexus_application_server:07-00-06:-:standard:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anthony_mills:s3_video:0.98`,
        fs:  `cpe:2.3:a:todd_miller:sudo:1.7.4p6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anthony_mills:s3_video:0.981`,
        fs:  `cpe:2.3:a:apache:groovy:2.1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anthony_mills:s3_video:0.982`,
        fs:  `cpe:2.3:h:d-link:dcs-2330l:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anthony_tuininga:cx_oracle-10g:5.0.4`,
        fs:  `cpe:2.3:a:hp:storeonce_4210_fc_backup_system:3.9.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anti-plagiarism_project:anti-plagiarism:3.60::~~~wordpress~~`,
        fs:  `cpe:2.3:a:freebsd:aide:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anti-virus:virusblockada32:3.12.12.4:-:personal`,
        fs:  `cpe:2.3:a:sympa:sympa:3.3.6b.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:antisamy_project:antisamy:1.4.1`,
        fs:  `cpe:2.3:a:sun:java_system_content_delivery_server:5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:antisamy_project:antisamy:1.4.2`,
        fs:  `cpe:2.3:a:fedoraproject:389_directory_server:1.3.2.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:antisamy_project:antisamy:1.4.3`,
        fs:  `cpe:2.3:h:zyxel:p-660hw_t3:v2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:antisamy_project:antisamy:1.4.4`,
        fs:  `cpe:2.3:a:vmware:player:7.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:antisamy_project:antisamy:1.4.5`,
        fs:  `cpe:2.3:a:viscacha:viscacha:0.8:gold:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:antisamy_project:antisamy:1.5`,
        fs:  `cpe:2.3:a:gnome:gnome_display_manager:2.21.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:antisamy_project:antisamy:1.5.1`,
        fs:  `cpe:2.3:a:yourkit:java_profiler:7.0.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:antisamy_project:antisamy:1.5.2`,
        fs:  `cpe:2.3:h:linksys:hpro200:1.42.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:antisamy_project:antisamy:1.5.3`,
        fs:  `cpe:2.3:a:photogallerycreator:flash-album-gallery:1.82:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:antisamy_project:antisamy:1.5.5`,
        fs:  `cpe:2.3:o:linux:linux_kernel:4.9.38:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:antisamy_project:antisamy:1.5.6`,
        fs:  `cpe:2.3:a:jeff_sterup:plugin-organizer:1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:antisamy_project:antisamy:1.5.7`,
        fs:  `cpe:2.3:a:quest:toad_for_oracle:8.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:antiword_project:antiword:0.37`,
        fs:  `cpe:2.3:a:moodle:moodle:2.4.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:antlabs:inngate_ig_3.00_e:-`,
        fs:  `cpe:2.3:a:live555:streaming_media:2011.10.27:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:antlabs:inngate_ig_3.01_e:-`,
        fs:  `cpe:2.3:a:ps_project_management_team:libunity-webapps:0.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:antlabs:inngate_ig_3.02_e:-`,
        fs:  `cpe:2.3:a:symantec:veritas_netbackup:6.5.2a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:antlabs:inngate_ig_3.10_e:-`,
        fs:  `cpe:2.3:a:magzter:where_dallas:3.0.2:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/h:antlabs:inngate_ig_3.10_g:-`,
        fs:  `cpe:2.3:a:splunk:splunk:3.4.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:antlabs:inngate_ig_3100:-`,
        fs:  `cpe:2.3:a:ovirt:ovirt:4.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:antlabs:inngate_ig_3101:-`,
        fs:  `cpe:2.3:h:hp:advanced_server_9000:b.04.05:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:antoine_beaupre:hostmaster:6.x-1.2`,
        fs:  `cpe:2.3:a:haudenschilt:family_connections_cms:2.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:antoine_beaupre:hostmaster:6.x-1.3`,
        fs:  `cpe:2.3:a:3ivx:mpeg-4:5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:antoine_beaupre:hostmaster:6.x-1.4`,
        fs:  `cpe:2.3:a:fusebox:fusebox:5.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:antoine_beaupre:hostmaster:6.x-1.5`,
        fs:  `cpe:2.3:o:suse:linux_enterprise_desktop:11:sp3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:antoine_beaupre:hostmaster:6.x-1.6`,
        fs:  `cpe:2.3:a:digium:asterisk:1.4.23:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:antoine_beaupre:hostmaster:6.x-1.7`,
        fs:  `cpe:2.3:a:f5:big-ip_application_security_manager:11.5.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:antoine_beaupre:hostmaster:6.x-1.8`,
        fs:  `cpe:2.3:a:coppermine-gallery:coppermine_photo_gallery:1.4.22:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:antoine_beaupre:hostmaster:6.x-1.x:dev`,
        fs:  `cpe:2.3:a:cisco:adaptive_security_appliance_software:9.3.3.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:antti_alamki:prh_search:7.x-1.0`,
        fs:  `cpe:2.3:a:cerberusftp:ftp_server:3.0.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:antti_alamki:prh_search:7.x-1.1`,
        fs:  `cpe:2.3:a:datadynamics:activebar:3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:antti_alamki:prh_search:7.x-1.x:dev`,
        fs:  `cpe:2.3:a:andrew_simpson:webcollab:2.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anusthanokarehasya:baglamukhi:0.1::~~~android~~`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.27:rc5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anydesk:anydesk:3.6.0`,
        fs:  `cpe:2.3:a:rocomotion:p_up_board:1.20:*:g:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anyfont_plugin_project:anyfont:2.2.3::~~~wordpress~~`,
        fs:  `cpe:2.3:a:apache:rave:0.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anything-digital:sh404sef:1.5.2.255`,
        fs:  `cpe:2.3:a:torproject:tor:0.2.0.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anything-digital:sh404sef:1.5.3.296`,
        fs:  `cpe:2.3:o:cisco:ios:12.4\(4\)t2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anything-digital:sh404sef:1.5.4.302`,
        fs:  `cpe:2.3:a:drupal:drupal:5.0:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anything-digital:sh404sef:1.5.5.388`,
        fs:  `cpe:2.3:a:yiiframework:yiiframework:1.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anything-digital:sh404sef:1.5.6.398`,
        fs:  `cpe:2.3:a:icu-project:international_components_for_unicode:2.6.1:*:*:*:*:c\/c\&#43;\&#43;:*:*`,
    },
     
    {
        uri: `cpe:/a:anything-digital:sh404sef:1.5.7.407`,
        fs:  `cpe:2.3:a:reasoncms:reasoncms:4.6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anything-digital:sh404sef:1.5.8.432`,
        fs:  `cpe:2.3:a:sun:jdk:1.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anything-digital:sh404sef:1.5.9.434`,
        fs:  `cpe:2.3:a:digium:asterisk:1.8.20.0:patch:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anything-digital:sh404sef:1.5.10.446`,
        fs:  `cpe:2.3:a:hp:smart_array_sas\/sata_event_notification_service:6.12.0.32:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anything-digital:sh404sef:1.5.11.459`,
        fs:  `cpe:2.3:a:iss:blackice_server_protection:3.6cno:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anything-digital:sh404sef:1.5.12.464`,
        fs:  `cpe:2.3:o:d-link:dir505l_shareport_mobile_companion_firmware:1.01:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anything-digital:sh404sef:2.0.0:rc522`,
        fs:  `cpe:2.3:a:trihedral:vtscada:10.2.07:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anything-digital:sh404sef:2.0.1.531`,
        fs:  `cpe:2.3:a:lester_chan:wp-downloadmanager:1.60:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anything-digital:sh404sef:2.0.2.542`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.7:p448:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anything-digital:sh404sef:2.0.3.545`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:7.0.5-6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anything-digital:sh404sef:2.1.0.641`,
        fs:  `cpe:2.3:a:vmware:vcenter_server:4.1:update_1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anything-digital:sh404sef:2.1.1.644`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(29a\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anything-digital:sh404sef:2.1.2.649`,
        fs:  `cpe:2.3:a:redhat:enterprise_mrg:1.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anything-digital:sh404sef:2.1.3.680`,
        fs:  `cpe:2.3:a:rubyonrails:ruby_on_rails:3.2.13:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anything-digital:sh404sef:2.1.4.734`,
        fs:  `cpe:2.3:a:myrephp:myre_vacation_rental:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anything-digital:sh404sef:2.1.5.746`,
        fs:  `cpe:2.3:a:trevor_mckay:cumin:0.1.5137-5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anything-digital:sh404sef:2.1.6.749`,
        fs:  `cpe:2.3:o:cisco:vpn_3030_concentator:4.7.1.f:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anything-digital:sh404sef:2.1.7.761`,
        fs:  `cpe:2.3:a:hp:lights-out_passthrough_service:1.2.26.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anything-digital:sh404sef:2.1.8.777`,
        fs:  `cpe:2.3:a:mozilla:bugzilla:2.21.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anywhere_anytime_yoga_workout_project:anywhere_anytime_yoga_workout:1::~~~android~~`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:anywherepad:anywhere_pad-meet_collaborate:4.0.1031::~~~android~~`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.7:p104:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aokitaka:zip_with_pass:4.5.7:-:~-~-~android~~`,
        fs:  `cpe:2.3:a:symantec:mail_security:5.0.0.204:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aokitaka:zip_with_pass_pro:6.2.1:-:~-~-~android~~`,
        fs:  `cpe:2.3:a:ponsoftware:explzh:3.23a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aokitaka:zip_with_pass_pro:6.2.2:-:~-~-~android~~`,
        fs:  `cpe:2.3:a:wegoi:re-volt_2_\:_multiplayer:1.1.4:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:aokitaka:zip_with_pass_pro:6.3.0:-:~-~-~android~~`,
        fs:  `cpe:2.3:a:sun:java_system_access_manager:7.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aokitaka:zip_with_pass_pro:6.3.4:-:~-~-~android~~`,
        fs:  `cpe:2.3:a:mahara:mahara:1.1.0:beta1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aokitaka:zip_with_pass_pro:6.3.5:-:~-~-~android~~`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.3.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aokitaka:zip_with_pass_pro:6.3.7:-:~-~-~android~~`,
        fs:  `cpe:2.3:a:juniper:netscreen_remote_security_client:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aokitaka:zip_with_pass_pro:6.3.8:-:~-~-~android~~`,
        fs:  `cpe:2.3:a:x.org:xorg-server:1.10.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aokitaka:zip_with_pass_pro:6.3.9:-:~-~-~android~~`,
        fs:  `cpe:2.3:a:ibm:websphere_application_server:8.0.0.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aim:1.0`,
        fs:  `cpe:2.3:o:oracle:linux:6:5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aim:1.0.1.2`,
        fs:  `cpe:2.3:a:lockon:ec-cube:2.5.0:alpha:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aim:1.06`,
        fs:  `cpe:2.3:a:microcart_project:microcart:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aim:1.7`,
        fs:  `cpe:2.3:a:luracast:restler:1.2.1:*:*:*:*:typo3:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aim:2.0`,
        fs:  `cpe:2.3:a:coppermine-gallery:coppermine_photo_gallery:1.4.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aim:2.1`,
        fs:  `cpe:2.3:a:xmlsoft:libxslt:1.0.19:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aim:3.0`,
        fs:  `cpe:2.3:a:google:chrome:8.0.552.213:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aim:3.5`,
        fs:  `cpe:2.3:a:tenable:nessus:6.5.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aim:4.1`,
        fs:  `cpe:2.3:a:cisco:meeting_server:2.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aim:4.2`,
        fs:  `cpe:2.3:a:barebones:bbedit:10.5.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aim:4.3`,
        fs:  `cpe:2.3:h:schneider-electric:bmxp3420302h:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aim:4.4`,
        fs:  `cpe:2.3:a:cubecart:cubecart:4.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aim:4.7`,
        fs:  `cpe:2.3:a:ati:catalyst_control_center_utility:2009.1118.1260.23275:*:*:*:*:*:x64:*`,
    },
     
    {
        uri: `cpe:/a:aol:aim:4.8`,
        fs:  `cpe:2.3:a:splunk:splunk:4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aim:5.0`,
        fs:  `cpe:2.3:a:download_manager_project:download_manager:2.5.5:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aim:5.1`,
        fs:  `cpe:2.3:a:oracle:database_server_lite:5.0.2.9.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aim:5.2`,
        fs:  `cpe:2.3:o:cisco:ios_xe:2.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aim:5.5`,
        fs:  `cpe:2.3:a:paolo_bacchilega:file_roller:3.6.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aim:5.9`,
        fs:  `cpe:2.3:a:cloudfoundry:cf-release:150:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aim:6.0`,
        fs:  `cpe:2.3:a:freeradius:freeradius:0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aim:6.2`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.32.58:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aim:6.5`,
        fs:  `cpe:2.3:a:ati:catalyst_control_center_help:108.2146.2564.38893:*:*:ko:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aim:6.8`,
        fs:  `cpe:2.3:a:macromedia:jrun:4.0_build_61650:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aim_toolbar:-`,
        fs:  `cpe:2.3:a:f5:firepass:6.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aim_toolbar:5.96.6.1`,
        fs:  `cpe:2.3:a:openvas:openvas_manager:1.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aol_toolbar:-`,
        fs:  `cpe:2.3:a:htc:evo_view_4g_software:1.22.651.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aolserver:3.3.1`,
        fs:  `cpe:2.3:a:openvpn:openvpn:2.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aolserver:3.4`,
        fs:  `cpe:2.3:a:apache:tomcat:6.0.50:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aolserver:3.4.2`,
        fs:  `cpe:2.3:a:restlet:restlet:2.1:rc5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aolserver:3.5.5`,
        fs:  `cpe:2.3:a:spamdyke:spamdyke:3.1.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aolserver:3.5.10`,
        fs:  `cpe:2.3:a:libpng:libpng:1.5.23:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aolserver:4.0`,
        fs:  `cpe:2.3:a:wysigot:wysigot:5.50:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aolserver:4.0.3`,
        fs:  `cpe:2.3:a:moinejf:abcm2ps:1.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aolserver:4.0.7`,
        fs:  `cpe:2.3:a:kan-studio:kandidat_cms:1.4.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aolserver:4.0.9`,
        fs:  `cpe:2.3:a:cisco:unified_computing_system:2.0_3b:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aolserver:4.0.10`,
        fs:  `cpe:2.3:a:macromedia:coldfusion_fusebox:4.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:aolserver:4.5.1`,
        fs:  `cpe:2.3:a:artonx.org:activescriptruby:1.6.4.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:browser:1.1`,
        fs:  `cpe:2.3:a:zeus.physik.uni-bonn:mn_fit:4.05\/32:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:browser:1.6:-:embedded`,
        fs:  `cpe:2.3:a:cisco:network_admission_control_manager_and_server_system_software:3.6.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:browser:1.6:-:standalone`,
        fs:  `cpe:2.3:a:ca:arcserve_backup:r16.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:browser:1.7:-:embedded`,
        fs:  `cpe:2.3:a:mahara:mahara:1.4.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:browser:1.7:-:standalone`,
        fs:  `cpe:2.3:h:hp:procurve_router:jf230a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:dailyfinance_-_stocks_%26_news:2.0.2.1::~~~android~~`,
        fs:  `cpe:2.3:a:cisco:secure_desktop:3.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:email_toolbar:-`,
        fs:  `cpe:2.3:a:actividentity:activclient_cac:6.2.0.71:*:*:*:*:*:x64:*`,
    },
     
    {
        uri: `cpe:/a:aol:email_toolbar:5.22.27.1`,
        fs:  `cpe:2.3:a:octopus:octopus_deploy:3.3.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:explorer:-`,
        fs:  `cpe:2.3:a:copadata:zenon_dnp3_process_gateway:7.11:sp0_build_10238:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:explorer:1.5`,
        fs:  `cpe:2.3:a:google:chrome:22.0.1229.49:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:internet_software:4.0`,
        fs:  `cpe:2.3:h:hp:cd645a:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:internet_software:5.0`,
        fs:  `cpe:2.3:a:richard_cook:rgpg:0.2.1:-:*:*:*:ruby:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:internet_software:6.0`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(13\)zj:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:internet_software:7.0`,
        fs:  `cpe:2.3:a:git-scm:git:2.3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:internet_software:8.0`,
        fs:  `cpe:2.3:o:apple:iphone_os:5.0:-:iphone:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:internet_software:9.0`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.7:p457:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aol:toolbar:5.74.1.1`,
        fs:  `cpe:2.3:h:3com:webbngss3nbxnts:4.2.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:-`,
        fs:  `cpe:2.3:a:zonelabs:zonealarm_anti-spyware:6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:4.0`,
        fs:  `cpe:2.3:a:mortbay:jetty:4.1.0:rc4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:4.0:m4`,
        fs:  `cpe:2.3:a:rom_walton:boinc:7.2.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:4.0:rc2`,
        fs:  `cpe:2.3:a:susumu_terao:terapad:0.82:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:4.0.1`,
        fs:  `cpe:2.3:a:mariadb:mariadb:5.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:4.0.2`,
        fs:  `cpe:2.3:a:openstack:keystone:2013.2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:4.1.0`,
        fs:  `cpe:2.3:a:ibm:algo_one:4.7.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:4.1.1`,
        fs:  `cpe:2.3:a:otrs:faq:2.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:4.1.2`,
        fs:  `cpe:2.3:a:kde:kde_sc:4.7.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:5.0.0`,
        fs:  `cpe:2.3:a:rocomotion:pm_forum:1.03:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:5.1.0`,
        fs:  `cpe:2.3:a:mysql:mysql:5.0.0:alpha:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:5.2.0`,
        fs:  `cpe:2.3:a:gitlab:gitlab:9.0.2:*:*:*:enterprise:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:5.3.0`,
        fs:  `cpe:2.3:a:dolibarr:dolibarr:3.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:5.3.1`,
        fs:  `cpe:2.3:a:quest:sql_optimizer_for_oracle:8.0.0.1551:*:*:*:trial:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:5.3.2`,
        fs:  `cpe:2.3:a:apache:poi:3.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:5.4.0`,
        fs:  `cpe:2.3:a:symantec:endpoint_protection_center:12.1.3:-:small_business:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:5.4.1`,
        fs:  `cpe:2.3:a:alienvault:open_source_security_information_management:1.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:5.4.2`,
        fs:  `cpe:2.3:a:mythicsoft_ltd:filelocator_pro:6.0.1225.1:*:*:*:*:*:x64:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:5.4.3`,
        fs:  `cpe:2.3:a:oscommerce:online_merchant:2.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:5.5.0`,
        fs:  `cpe:2.3:o:hp:procurve_switch_software:r.11.38:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:5.5.1`,
        fs:  `cpe:2.3:a:rubyonrails:ruby_on_rails:3.0.7:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:5.6.0`,
        fs:  `cpe:2.3:a:cakefoundation:cakephp:1.3.0:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:5.7.0`,
        fs:  `cpe:2.3:a:atlassian:bamboo:2.1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:5.8.0`,
        fs:  `cpe:2.3:a:apache:struts:2.5.24:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:5.9.0`,
        fs:  `cpe:2.3:a:namazu:namazu:2.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:5.9.1`,
        fs:  `cpe:2.3:a:techsmith:camtasia_studio:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:5.10.0`,
        fs:  `cpe:2.3:o:cisco:ios:15.4\(3\)m2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:5.10.1`,
        fs:  `cpe:2.3:a:automattic:jetpack:2.6.1:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:5.10.2`,
        fs:  `cpe:2.3:a:bylight:20\/20:2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:5.11.0`,
        fs:  `cpe:2.3:a:google:chrome:8.0.552.200:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:5.11.1`,
        fs:  `cpe:2.3:a:social_slider_project:social_slider:7.4.0:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:5.11.2`,
        fs:  `cpe:2.3:a:graphicsmagick:graphicsmagick:1.1.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:5.11.3`,
        fs:  `cpe:2.3:a:webyog:sqlyog:3.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:5.12.0`,
        fs:  `cpe:2.3:a:ibm:tivoli_secureway_policy_director:3.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:5.12.1`,
        fs:  `cpe:2.3:a:roderich_schupp:par-packer_module:0.973:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:5.13.1`,
        fs:  `cpe:2.3:a:jenkins:config_file_provider:2.4:*:*:*:*:jenkins:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq:5.13.3`,
        fs:  `cpe:2.3:a:sensiolabs:symfony:2.3.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq_artemis:1.0.0`,
        fs:  `cpe:2.3:a:pivotal_software:cloud_foundry_elastic_runtime:1.6.34:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq_artemis:1.1.0`,
        fs:  `cpe:2.3:a:ovirt:ovirt:4.1.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq_artemis:1.2.0`,
        fs:  `cpe:2.3:o:novell:suse_manager_proxy:2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:activemq_artemis:1.3.0`,
        fs:  `cpe:2.3:a:devsaran:corporate:7.x-1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ambari:2.1.2`,
        fs:  `cpe:2.3:a:openstack:python_glanceclient:0.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ambari:2.2.0`,
        fs:  `cpe:2.3:a:xnview:xnview:1.80:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ambari:2.2.1`,
        fs:  `cpe:2.3:a:midgard-project:midgard2:10.05.7.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ambari:2.2.2`,
        fs:  `cpe:2.3:a:oracle:retail_order_management_system_cloud_service:5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ambari:2.2.2:rc0`,
        fs:  `cpe:2.3:a:moinejf:abcm2ps:4.12.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ambari:2.2.2:rc1`,
        fs:  `cpe:2.3:a:digium:certified_asterisk:1.8.10.0:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ambari:2.4.0`,
        fs:  `cpe:2.3:a:oracle:javafx:1.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ambari:2.4.0:rc0`,
        fs:  `cpe:2.3:a:gnu:zlib:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ambari:2.4.1`,
        fs:  `cpe:2.3:h:cisco:video_surveillance_sp_isp:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ambari:2.4.1:rc0`,
        fs:  `cpe:2.3:h:google:glass:xe5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ambari:2.4.1:rc1`,
        fs:  `cpe:2.3:a:altiris:application_metering_language_pack:6.1.32:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ambari:2.4.2:rc0`,
        fs:  `cpe:2.3:a:sendcard:sendcard:3.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ambari:2.4.2:rc1`,
        fs:  `cpe:2.3:a:cisco:telepresence_tc_software:4.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ambari:2.5.0:rc0`,
        fs:  `cpe:2.3:a:perl:perl:5.8.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ambari:2.5.0:rc1`,
        fs:  `cpe:2.3:a:pulsecms:pulse_cms:1.3.7:-:pro:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ambari:2.5.0:rc2`,
        fs:  `cpe:2.3:a:yahoo:messenger:6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apache-ssl:1.37`,
        fs:  `cpe:2.3:a:digium:asterisk:11.8.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apache-ssl:1.47`,
        fs:  `cpe:2.3:a:freetype:freetype2:2.6.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apache-ssl:1.48`,
        fs:  `cpe:2.3:a:octobercms:october_cms:1.0.396:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apache-ssl:1.53`,
        fs:  `cpe:2.3:a:google:chrome:17.0.963.26:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apache-ssl:1.55`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(50\)se1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apache-ssl:1.57`,
        fs:  `cpe:2.3:a:jason_flatt:basic_webmail:6.x-1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apache_test:-`,
        fs:  `cpe:2.3:a:openstack:keystone:2013.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apache_test:1.29`,
        fs:  `cpe:2.3:a:ibm:smart_analytics_system_7700:2.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:0.9.1`,
        fs:  `cpe:2.3:a:piwigo:piwigo:1.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:0.9.2`,
        fs:  `cpe:2.3:a:rocomotion:p_up_board:1.05:*:gbo:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:0.9.3`,
        fs:  `cpe:2.3:a:google:android_api:15.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:0.9.4`,
        fs:  `cpe:2.3:a:scott_reynen:node_embed:6.x-1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:0.9.5`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.128:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:0.9.6`,
        fs:  `cpe:2.3:a:lunascape:lunascape:4.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:0.9.7`,
        fs:  `cpe:2.3:a:puppetlabs:hiera:1.3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:0.9.8`,
        fs:  `cpe:2.3:a:projeqtor:projeqtor:2.4.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:0.9.9`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(28\)sb5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:0.9.10`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.26.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:0.9.11`,
        fs:  `cpe:2.3:a:cisco:videoscape_distribution_suite_service_manager:3.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:0.9.12`,
        fs:  `cpe:2.3:a:haproxy:haproxy:1.5:dev2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:0.9.13`,
        fs:  `cpe:2.3:h:3com:3crtpx505-73:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:0.9.14`,
        fs:  `cpe:2.3:a:git-scm:git:0.99.9c:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:0.9.15`,
        fs:  `cpe:2.3:o:cisco:ios:12.1\(14\)e1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:0.9.16`,
        fs:  `cpe:2.3:a:live555:streaming_media:2012.08.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:0.9.17`,
        fs:  `cpe:2.3:a:opera:opera_browser:9.0:beta1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:0.9.18`,
        fs:  `cpe:2.3:a:sun:jre:1.4.2:update1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:1.0`,
        fs:  `cpe:2.3:a:bottlepy:bottle:0.10.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:1.0.1`,
        fs:  `cpe:2.3:o:cisco:nx-os:6.0\(2\)a6\(4\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:1.0.2`,
        fs:  `cpe:2.3:a:fetchmail:fetchmail:6.3.6:rc3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:1.1.0`,
        fs:  `cpe:2.3:a:ibm:storwize_v5000_software:7.1.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:1.1.1`,
        fs:  `cpe:2.3:a:kbs:weblygo:5.04:*:pro:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:1.1.2`,
        fs:  `cpe:2.3:a:mongodb:mongodb:3.0.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:1.2.1`,
        fs:  `cpe:2.3:h:axis:m1034-w_network_camera:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:1.2.2`,
        fs:  `cpe:2.3:a:debian:dpkg:1.4.0.31:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:1.2.6`,
        fs:  `cpe:2.3:a:hp:solution_center:60.0.155.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:1.2.7`,
        fs:  `cpe:2.3:a:openstack:neutron:2012.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:1.2.8`,
        fs:  `cpe:2.3:a:nginx:nginx:1.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:1.2.9`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.22.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:1.2.10`,
        fs:  `cpe:2.3:a:ibm:storwize_unified_v7000_software:1.4.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:1.2.12`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:1.2.13`,
        fs:  `cpe:2.3:o:openvz:vzkernel:2.6.32:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:1.3.0`,
        fs:  `cpe:2.3:a:roundup-tracker:roundup:0.8.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:1.3.1`,
        fs:  `cpe:2.3:a:fedoraproject:389_directory_server:1.2.11.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:1.3.2`,
        fs:  `cpe:2.3:a:good:good_for_enterprise:2.1.4.1518:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:1.3.3`,
        fs:  `cpe:2.3:a:rocomotion:p_board_r:1.16:*:g:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:1.3.4`,
        fs:  `cpe:2.3:a:bestpractical:rt:3.8.6:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:1.3.5`,
        fs:  `cpe:2.3:a:sensiolabs:symfony:2.7.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:1.3.6`,
        fs:  `cpe:2.3:a:cakefoundation:cakephp:2.1.0:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:1.3.7`,
        fs:  `cpe:2.3:a:saflink:netsign_cac:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:1.3.8`,
        fs:  `cpe:2.3:a:ibm:rational_appscan:5.5.0.1:*:enterprise:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:1.3.9`,
        fs:  `cpe:2.3:a:google:v8:3.22.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:1.3.10`,
        fs:  `cpe:2.3:a:hkbn:hkbn_my_account:\@7f070015:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:1.4.3`,
        fs:  `cpe:2.3:a:atlassian:confluence:3.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:apr-util:1.4.4`,
        fs:  `cpe:2.3:a:sap:netweaver_abap:7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:archiva:1.2`,
        fs:  `cpe:2.3:a:cybozu:kunai:3.0.4:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:archiva:1.2.1`,
        fs:  `cpe:2.3:a:chatelao:php_address_book:3.3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:archiva:1.2.2`,
        fs:  `cpe:2.3:a:ssh:tectia_client:5.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:archiva:1.3`,
        fs:  `cpe:2.3:a:dell:openmanage_active_directory_snap-in_utility:6.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:archiva:1.3.1`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.7:p186:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:archiva:1.3.2`,
        fs:  `cpe:2.3:a:libguestfs:libguestfs:1.7.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:archiva:1.3.3`,
        fs:  `cpe:2.3:a:organic_groups_project:organic_groups:7.x-2.4:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:archiva:1.3.4`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.17:rc4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:archiva:1.3.5`,
        fs:  `cpe:2.3:a:gridgain:gridgain:8.0.3:ea14:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:archiva:1.3.6`,
        fs:  `cpe:2.3:a:symantec:veritas_storage_foundation_for_windows_device_driver_installation:5.1.110:sp1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:archiva:1.3.8`,
        fs:  `cpe:2.3:a:isc:bind:9.3.0:b3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:archiva:1.3.9`,
        fs:  `cpe:2.3:a:artis.imag:basilic:1.5.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis:-`,
        fs:  `cpe:2.3:a:owncloud:owncloud:3.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis:-:alpha1`,
        fs:  `cpe:2.3:a:adobe:adobe_air:2.7.0.19530:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis:-:alpha2`,
        fs:  `cpe:2.3:a:phpmyfaq:phpmyfaq:2.7.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis:-:alpha3`,
        fs:  `cpe:2.3:a:apple:cups:1.2.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis:-:beta1`,
        fs:  `cpe:2.3:a:katello:katello_installer:0.0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis:-:beta2`,
        fs:  `cpe:2.3:a:anything-digital:sh404sef:2.1.5.746:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis:-:beta3`,
        fs:  `cpe:2.3:a:widgetfactorylimited:com_jce:2.0.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis:1.0`,
        fs:  `cpe:2.3:a:sap:sql_anywhere:11.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis:1.0:beta`,
        fs:  `cpe:2.3:a:id:id-software:3.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis:1.0:rc1`,
        fs:  `cpe:2.3:a:hp:web_registration:60.0.155.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis:1.0:rc2`,
        fs:  `cpe:2.3:a:nick_copeland:bristol:0.10.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis:1.1`,
        fs:  `cpe:2.3:a:adcillc:simplemeta:6.x-1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis:1.1:beta`,
        fs:  `cpe:2.3:a:x.org:xorg-server:1.14.99.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis:1.1:rc1`,
        fs:  `cpe:2.3:a:haxx:curl:7.13.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis:1.1:rc2`,
        fs:  `cpe:2.3:a:zarafa:zarafa:6.40.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis:1.2`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.9.7-10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis:1.2:alpha`,
        fs:  `cpe:2.3:a:oracle:fusion_middleware:11.1.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis:1.2:beta1`,
        fs:  `cpe:2.3:a:microsoft:outlook:2013:-:-:*:-:-:x64:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis:1.2:beta2`,
        fs:  `cpe:2.3:a:mozilla:firefox:29.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis:1.2:beta3`,
        fs:  `cpe:2.3:a:busybox:busybox:1.12.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis:1.2:rc1`,
        fs:  `cpe:2.3:a:zfcuser_project:zfcuser:1.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis:1.2:rc2`,
        fs:  `cpe:2.3:a:emc:connectrix_manager:12.0.1:-:-:*:converged_network_edition:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis:1.2:rc3`,
        fs:  `cpe:2.3:a:i-doit:i-doit:1.1.2:*:*:*:pro:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis:1.2.1`,
        fs:  `cpe:2.3:a:bea:weblogic_server:5.1:sp9:express:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis:1.3`,
        fs:  `cpe:2.3:a:apache:batik:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis:1.4`,
        fs:  `cpe:2.3:a:phpmyadmin:phpmyadmin:4.0.10.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis2:-`,
        fs:  `cpe:2.3:a:recruit:dokodemo_rikunabi_2013:1.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis2:1.3`,
        fs:  `cpe:2.3:o:microsoft:windows-nt:4.0:sp6:embedded:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis2:1.4`,
        fs:  `cpe:2.3:a:google:chrome:19.0.1084.47:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis2:1.4.1`,
        fs:  `cpe:2.3:a:sun:java_system_application_server:8.1:*:x86:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis2:1.5`,
        fs:  `cpe:2.3:a:nrgglobal:clevereye:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis2:1.5.1`,
        fs:  `cpe:2.3:a:hp:photosmart_essential:1.9.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis2:1.5.2`,
        fs:  `cpe:2.3:o:hp:procurve_switch_software:r.11.06:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis2:1.5.3`,
        fs:  `cpe:2.3:o:iij:seil_x86_fuji_firmware:2.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis2:1.5.4`,
        fs:  `cpe:2.3:a:symantec:sms-smtp-mysql-init:5.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis2:1.5.5`,
        fs:  `cpe:2.3:a:oracle:connector\/j:5.1.40:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis2:1.5.6`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.4.54:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis2:1.6`,
        fs:  `cpe:2.3:a:yahoo:messenger:8.0.0.508:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis2:1.6.1`,
        fs:  `cpe:2.3:a:smarty:smarty:3.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:axis2:1.6.2`,
        fs:  `cpe:2.3:a:gnome:gnome-shell:3.7.91:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:batik:-`,
        fs:  `cpe:2.3:a:apereo:central_authentication_service:3.5.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:batik:1.0`,
        fs:  `cpe:2.3:a:tincan:phplist:2.5.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:batik:1.1`,
        fs:  `cpe:2.3:a:andy_armstrong:cgi.pm:2.62:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:batik:1.1.1`,
        fs:  `cpe:2.3:a:sun:one_web_server:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:batik:1.5`,
        fs:  `cpe:2.3:a:hp:insight_management_wbem_providers:2.8.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:batik:1.7`,
        fs:  `cpe:2.3:a:adobe:flash_player:11.2.202.424:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:1.0.0`,
        fs:  `cpe:2.3:h:ibm:m15:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:1.1.0`,
        fs:  `cpe:2.3:a:splunk:splunk:5.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:1.2.0`,
        fs:  `cpe:2.3:a:emc:unisphere:1.0:-:*:*:*:vmax:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:1.3.0`,
        fs:  `cpe:2.3:h:linksys:befsr41:0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:1.4.0`,
        fs:  `cpe:2.3:a:freedomscientific:synthesizer_eloquence:6.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:1.5.0`,
        fs:  `cpe:2.3:a:serv-u:serv-u:5.1.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:1.6.0`,
        fs:  `cpe:2.3:a:microsoft:microsoft.aspnetcore.mvc.abstractions:1.0.1:*:*:*:*:asp.net:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:1.6.1`,
        fs:  `cpe:2.3:a:s9y:serendipity:1.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:1.6.2`,
        fs:  `cpe:2.3:a:freetype:freetype:2.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:1.6.3`,
        fs:  `cpe:2.3:a:b2evolution:b2evolution:6.9.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:1.6.4`,
        fs:  `cpe:2.3:a:patrick_przybilla:addtoany:6.x-3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.0.0`,
        fs:  `cpe:2.3:a:mysql:eventum:1.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.0.0:m1`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.10.35:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.0.0:m2`,
        fs:  `cpe:2.3:h:axis:2400_video_server:1.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.0.0:m3`,
        fs:  `cpe:2.3:a:ibm:iss_realsecure_network:7.0:xpu_22.2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.1.0`,
        fs:  `cpe:2.3:a:ibm:db2:9.7.0.8:*:*:*:advanced_workgroup:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.2.0`,
        fs:  `cpe:2.3:a:php:pear:1.3.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.3.0`,
        fs:  `cpe:2.3:a:cisco:intelligent_contact_manager:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.4.0`,
        fs:  `cpe:2.3:a:sophos:sophos_anti-virus:4.5.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.5.0`,
        fs:  `cpe:2.3:a:docker:docker:0.7.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.6.0`,
        fs:  `cpe:2.3:a:ubnt:unifi_controller:2.4.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.7.0`,
        fs:  `cpe:2.3:a:sophos:enterprise_console:5.2.1:r2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.7.1`,
        fs:  `cpe:2.3:a:texmacs:texmacs:1.0.4.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.7.2`,
        fs:  `cpe:2.3:a:objective_development:little_snitch:3.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.7.3`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.520:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.7.4`,
        fs:  `cpe:2.3:a:splunk:splunk:5.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.7.5`,
        fs:  `cpe:2.3:a:perl:perl:5.12.0:rc4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.8.0`,
        fs:  `cpe:2.3:a:adiscon:loganalyzer:3.5.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.8.1`,
        fs:  `cpe:2.3:a:apache:myfaces:1.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.8.2`,
        fs:  `cpe:2.3:a:htc:evo_4g_software:4.54.651.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.8.3`,
        fs:  `cpe:2.3:a:google:chrome:14.0.835.122:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.8.4`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(12h\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.8.5`,
        fs:  `cpe:2.3:a:openstack:compute:2014.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.8.6`,
        fs:  `cpe:2.3:a:flowplayer:flowplayer_flash:3.2.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.9.0`,
        fs:  `cpe:2.3:a:dameware:mini_remote_control:5.0.1.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.9.1`,
        fs:  `cpe:2.3:a:andrew_simpson:webcollab:1.61:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.9.2`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.5.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.9.3`,
        fs:  `cpe:2.3:a:mailenable:mailenable:1.93:-:standard:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.9.4`,
        fs:  `cpe:2.3:a:fenrir-inc:sleipnir_mobile:1.7.1:-:black:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.9.5`,
        fs:  `cpe:2.3:a:mozilla:firefox_mobile:10.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.9.6`,
        fs:  `cpe:2.3:a:ecommercesoft:xse_shopping_cart:1.1.1495.34686:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.9.7`,
        fs:  `cpe:2.3:a:ite:smart_guardian:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.9.8`,
        fs:  `cpe:2.3:a:ibm:rational_software_architect_for_websphere_software:8.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.10.0`,
        fs:  `cpe:2.3:o:cisco:ios:15.0\(2a\)se9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.10.1`,
        fs:  `cpe:2.3:a:quest:intrust_monitoring_console:10.2.5.1024:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.10.2`,
        fs:  `cpe:2.3:a:ibm:spectrum_protect_snapshot:4.1:*:*:*:*:vmware:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.10.3`,
        fs:  `cpe:2.3:a:ibm:openpages_grc_platform:5.5.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.10.4`,
        fs:  `cpe:2.3:o:cisco:asa_cx_context-aware_security_software:9.3.2-1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.10.5`,
        fs:  `cpe:2.3:a:goodiware:goodreader:2.8.2:-:*:*:*:iphone_os:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.10.6`,
        fs:  `cpe:2.3:a:bestpractical:rt:4.0.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.10.7`,
        fs:  `cpe:2.3:a:phpmyadmin:phpmyadmin:1.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.11.0`,
        fs:  `cpe:2.3:a:oracle:retail_order_management_system_cloud_service:15.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.11.1`,
        fs:  `cpe:2.3:a:x:libxt:1.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.11.2`,
        fs:  `cpe:2.3:a:moinmo:moinmoin:1.5.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.11.3`,
        fs:  `cpe:2.3:a:google:chrome:4.0.275.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.11.4`,
        fs:  `cpe:2.3:a:hp:openview_network_node_manager:6.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.12.0`,
        fs:  `cpe:2.3:a:ffmpeg:ffmpeg:2.2.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.12.1`,
        fs:  `cpe:2.3:o:wdc:arkeia_virtual_appliance_firmware:10.2.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.12.2`,
        fs:  `cpe:2.3:a:linuxfoundation:cups-filters:1.17.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.12.3`,
        fs:  `cpe:2.3:h:linksys:befcmu10:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.12.4`,
        fs:  `cpe:2.3:a:jenkins:script_security:1.9:*:*:*:*:jenkins:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.12.5`,
        fs:  `cpe:2.3:a:mortbay:jetty:1.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.13.0`,
        fs:  `cpe:2.3:a:kuronecostudio:noble_sticker_\&#34;free\&#34;:1.0.7:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.13.1`,
        fs:  `cpe:2.3:h:hp:laserjet_2200dtn:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.13.2`,
        fs:  `cpe:2.3:a:apple:quicktime:7.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.13.3`,
        fs:  `cpe:2.3:a:rocomotion:p_up_board_random:1.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.13.4`,
        fs:  `cpe:2.3:a:apache:solr:4.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.14.0`,
        fs:  `cpe:2.3:a:linuxfoundation:cups-filters:1.0.19:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.14.1`,
        fs:  `cpe:2.3:a:netgenius:multilink:6.x-2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.14.2`,
        fs:  `cpe:2.3:a:tilman_hausherr:xenu_link_sleuth:1.0u:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.14.3`,
        fs:  `cpe:2.3:a:hp:system_management_homepage:2.1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.14.4`,
        fs:  `cpe:2.3:h:cisco:scientific_atlanta_dpc3008\/epc3008:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.15.0`,
        fs:  `cpe:2.3:a:tejimaya:openpne:2.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.15.1`,
        fs:  `cpe:2.3:o:iij:seil_x86_fuji_firmware:3.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.15.2`,
        fs:  `cpe:2.3:a:omnigroup:omniweb:5.1:563.34:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.15.3`,
        fs:  `cpe:2.3:a:bravenewcode:wptouch:1.9.19.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.15.4`,
        fs:  `cpe:2.3:a:snapapp_project:snapapp:1.5:-:*:*:-:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.16.0`,
        fs:  `cpe:2.3:a:redmine:redmine:1.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.16.1`,
        fs:  `cpe:2.3:a:freetype:freetype:2.3.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.16.2`,
        fs:  `cpe:2.3:a:phimviethoa:chien_binh_bakugan_2_longtieng:2:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.16.3`,
        fs:  `cpe:2.3:o:iij:seil_x2_firmware:3.74:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.16.4`,
        fs:  `cpe:2.3:a:samba:samba:3.6.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.17.0`,
        fs:  `cpe:2.3:a:cmu:cyrus_imap_server:2.1.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.17.1`,
        fs:  `cpe:2.3:h:schneider-electric:tburjr900:05002eh0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.17.2`,
        fs:  `cpe:2.3:a:libguestfs:libguestfs:1.7.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.17.3`,
        fs:  `cpe:2.3:a:openstack:heat:2015.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.17.4`,
        fs:  `cpe:2.3:a:gnome:gnome_display_manager:3.6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.18.0`,
        fs:  `cpe:2.3:a:moinmo:moinmoin:1.7.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.18.1`,
        fs:  `cpe:2.3:a:chaos_tool_suite_project:ctools:6.x-1.0:alpha1:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.19.4`,
        fs:  `cpe:2.3:a:eeye:retina:5.14.1.2510:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:camel:2.20.0`,
        fs:  `cpe:2.3:a:fork-cms:fork_cms:2.6.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:1.2.0`,
        fs:  `cpe:2.3:a:12net:login_rebuilder:1.1.2:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:1.2.1`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(28c\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:1.2.2`,
        fs:  `cpe:2.3:a:hp:arcsight_logger:6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:1.2.3`,
        fs:  `cpe:2.3:o:redhat:enterprise_linux_desktop:4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:1.2.4`,
        fs:  `cpe:2.3:a:buffalotech:whr-g54s_firmware:1.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:1.2.5`,
        fs:  `cpe:2.3:a:jeff_ortel:suds:0.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:1.2.6`,
        fs:  `cpe:2.3:a:rocomotion:p_board_ri:1.09:*:g:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:1.2.7`,
        fs:  `cpe:2.3:h:moxa:iologic_e2242:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:1.2.8`,
        fs:  `cpe:2.3:a:theforeman:kafo:0.3.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:1.2.9`,
        fs:  `cpe:2.3:a:sun:java_system_web_proxy_server:3.6:sp9:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:1.2.10`,
        fs:  `cpe:2.3:a:telligent:evolution:7.6.7.36651:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:1.2.11`,
        fs:  `cpe:2.3:o:google:chrome_os:20.0.1132.19:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:1.2.12`,
        fs:  `cpe:2.3:a:mediawiki:mediawiki:1.7.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:1.2.13`,
        fs:  `cpe:2.3:a:google:chrome:6.0.472.32:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:1.2.14`,
        fs:  `cpe:2.3:a:dest-unreach:socat:1.7.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:1.2.15`,
        fs:  `cpe:2.3:h:hp:procurve_switch:jd310a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:1.2.16`,
        fs:  `cpe:2.3:a:rubyonrails:ruby_on_rails:3.0.4:rc:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:1.2.17`,
        fs:  `cpe:2.3:a:buffalotech:whr-ampg_firmware:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:1.2.18`,
        fs:  `cpe:2.3:a:apache:jackrabbit:2.2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:1.2.19`,
        fs:  `cpe:2.3:a:gnome:at-spi2-atk:2.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:2.0.0`,
        fs:  `cpe:2.3:a:businessobjects:business_intelligence:12.1.0.882:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:2.0.1`,
        fs:  `cpe:2.3:a:libguestfs:libguestfs:1.16.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:2.0.2`,
        fs:  `cpe:2.3:a:apple:itunes:7.4.2:-:windows:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:2.0.3`,
        fs:  `cpe:2.3:a:google:chrome:35.0.1916.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:2.0.4`,
        fs:  `cpe:2.3:a:tejimaya:openpne:3.6.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:2.0.5`,
        fs:  `cpe:2.3:o:greenbone:greenbone_os:1.7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:2.0.6`,
        fs:  `cpe:2.3:a:moxiecode:plupload:1.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:2.0.7`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.38:rc3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:2.0.8`,
        fs:  `cpe:2.3:a:kde:kscreenlocker:5.5.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:2.0.9`,
        fs:  `cpe:2.3:a:david_shadoff:mednafen:0.6.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:2.0.10`,
        fs:  `cpe:2.3:a:bouncycastle:legion-of-the-bouncy-castle-c\#-crytography-api:1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:2.0.11`,
        fs:  `cpe:2.3:o:pheonixcontact-software:proconos_eclr:-:*:*:*:visual_studio:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:2.0.12`,
        fs:  `cpe:2.3:a:altiris:svs_task:6.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:2.0.13`,
        fs:  `cpe:2.3:a:commentluv:commentluv:2.90.9.9.2:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:2.1.0`,
        fs:  `cpe:2.3:a:mailenable:mailenable:4.12:-:enterprise:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:2.1.1`,
        fs:  `cpe:2.3:a:thulasidas:easy-adsense-lite:2.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:2.1.2`,
        fs:  `cpe:2.3:a:oracle:mysql:5.6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cassandra:2.1.3`,
        fs:  `cpe:2.3:a:isc:bind:9.3.5:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:2.0:-:community`,
        fs:  `cpe:2.3:a:adiscon:loganalyzer:3.4.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:2.0.1`,
        fs:  `cpe:2.3:a:hp:operations_manager_i:10.01:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:2.1.0`,
        fs:  `cpe:2.3:a:novell:netmail:3.5.2b:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:2.1.1`,
        fs:  `cpe:2.3:a:devincentiis:gazie:4.0.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:2.1.2`,
        fs:  `cpe:2.3:a:dotnetnuke:dotnetnuke:08.00.04:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:2.1.3`,
        fs:  `cpe:2.3:h:hp:procurve_switch:je168a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:2.1.4`,
        fs:  `cpe:2.3:a:ibm:filenet_p8_application_engine:3.5.1:012:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:2.1.5`,
        fs:  `cpe:2.3:a:hp:array_configuration_utility:8.40.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:2.1.6`,
        fs:  `cpe:2.3:a:apache:xerces-c\&#43;\&#43;:1.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:2.1.7`,
        fs:  `cpe:2.3:a:dlitz:pycrypto:2.1.0:alpha2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:2.1.8`,
        fs:  `cpe:2.3:a:automattic:jetpack:2.8:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:2.1.9`,
        fs:  `cpe:2.3:a:hitachi:ucosminexus_developer:07-00-05:-:pro:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:2.1.10`,
        fs:  `cpe:2.3:a:redhat:jboss_enterprise_application_platform:6.4.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:2.2.0`,
        fs:  `cpe:2.3:a:hp:directories_support_for_proliant_management_processors:3.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:2.2.1`,
        fs:  `cpe:2.3:a:mobiloud:mobiloud:1.6.2:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:2.2.2`,
        fs:  `cpe:2.3:a:ovirt:ovirt:3.6.11.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:2.2.3`,
        fs:  `cpe:2.3:a:ibm:curam_social_program_management:6.0.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:2.2.5`,
        fs:  `cpe:2.3:a:sixapart:movable_type:4.0:beta5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:2.2.6`,
        fs:  `cpe:2.3:a:haxx:libcurl:7.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:2.2.7`,
        fs:  `cpe:2.3:a:octopus:octopus_deploy:3.13.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:2.2.8`,
        fs:  `cpe:2.3:a:apache:hadoop:0.15.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:2.2.9`,
        fs:  `cpe:2.3:a:ibm:rational_automation_framework:3.0.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:2.2.11`,
        fs:  `cpe:2.3:a:google:chrome:11.0.689.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:2.2.12`,
        fs:  `cpe:2.3:a:gnome:librsvg:2.32.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:2.2.13`,
        fs:  `cpe:2.3:h:polycom:mgc-100:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:2.2.14`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.8:rc4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:3.0.0`,
        fs:  `cpe:2.3:a:cisco:video_surveillance_sp_isp_decoder_software:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:3.0.1`,
        fs:  `cpe:2.3:a:magic_hills:wonderplugin_audio_player:2.0:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:3.0.2`,
        fs:  `cpe:2.3:a:ca:single_sign-on:r12.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:4.0.0:incubating`,
        fs:  `cpe:2.3:o:lexmark:printer_firmware:pp.02.048:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:4.0.1`,
        fs:  `cpe:2.3:a:avscripts:av_arcade_pro:5.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:4.0.2`,
        fs:  `cpe:2.3:a:haxx:curl:7.46.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:4.1.0`,
        fs:  `cpe:2.3:a:barebones:bbedit:10.5.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:4.1.1`,
        fs:  `cpe:2.3:a:namazu:namazu:1.2.1.0:beta9:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:4.2.0`,
        fs:  `cpe:2.3:a:ibm:db2_connect:9.8.0.2:*:*:*:application_server:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:4.4.4`,
        fs:  `cpe:2.3:a:wordpress_social_login_project:wordpress_social_login:2.0.3:-:-:*:-:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:4.5.1`,
        fs:  `cpe:2.3:a:hp:solution_center:7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:4.5.2`,
        fs:  `cpe:2.3:a:digium:asterisk:11.8.0:rc3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:4.6.0`,
        fs:  `cpe:2.3:o:cisco:catos:6.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:4.6.1`,
        fs:  `cpe:2.3:o:yahama:rt102i:3.00.31:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:4.6.2`,
        fs:  `cpe:2.3:a:naist.jp:chasen:2.2.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:4.7.0`,
        fs:  `cpe:2.3:a:oracle:mojarra:2.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cloudstack:4.8`,
        fs:  `cpe:2.3:a:yahoo:yui:3.9.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cocoon:-`,
        fs:  `cpe:2.3:a:atlassian:crucible:2.7.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cocoon:2.1`,
        fs:  `cpe:2.3:a:getpixie:pixie:1.01:a:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cocoon:2.1.2`,
        fs:  `cpe:2.3:a:mozilla:bugzilla:3.0.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cocoon:2.2`,
        fs:  `cpe:2.3:a:irislink:readiris:11.0.4965:*:*:*:professional:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons-compress:1.0`,
        fs:  `cpe:2.3:a:emc:networker_module_for_microsoft_applications:2.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons-compress:1.1`,
        fs:  `cpe:2.3:a:cakephp:cakephp:2.7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons-compress:1.2`,
        fs:  `cpe:2.3:a:libpng:libpng:1.5.3:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons-compress:1.3`,
        fs:  `cpe:2.3:a:tecnick:tcexam:11.2.004:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons-compress:1.4`,
        fs:  `cpe:2.3:a:tapatalk:tapatalk:4.4.0:*:*:*:*:vbulletin:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons-compress:1.4.1`,
        fs:  `cpe:2.3:a:mercadolibre:mercadolibre:3.8.7:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons-httpclient:3.0`,
        fs:  `cpe:2.3:a:pivotal_software:rabbitmq:1.7.0:*:*:*:*:pivotal_cloud_foundry:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons-httpclient:4.0`,
        fs:  `cpe:2.3:o:yamaha:rt105e:6.03.34:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons-httpclient:4.0.1`,
        fs:  `cpe:2.3:a:emberjs:ember.js:1.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons-httpclient:4.0.2`,
        fs:  `cpe:2.3:a:dokuwiki:dokuwiki:2011-05-25c:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons-httpclient:4.0.3`,
        fs:  `cpe:2.3:a:apache:httpclient:4.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons-httpclient:4.1.1`,
        fs:  `cpe:2.3:o:iij:seil_x1_firmware:3.57:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons-httpclient:4.1.2`,
        fs:  `cpe:2.3:a:cakephp:cakephp:2.5.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons-httpclient:4.1.3`,
        fs:  `cpe:2.3:o:huawei:s9300_firmware:v100r006c00spc500:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons-httpclient:4.2`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.7:p190:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons-httpclient:4.2.1`,
        fs:  `cpe:2.3:a:sun:jre:1.4.2:update19:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons-httpclient:4.2.2`,
        fs:  `cpe:2.3:a:gecad:axigen_mail_server:4.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons-httpclient:4.2.3`,
        fs:  `cpe:2.3:a:ovirt:ovirt:3.4.3:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons-httpclient:4.2.4`,
        fs:  `cpe:2.3:a:samba:samba:3.0.35:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons-httpclient:4.2.5`,
        fs:  `cpe:2.3:a:cisco:adaptive_security_appliance_software:9.3.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons-httpclient:4.2.6`,
        fs:  `cpe:2.3:a:ibm:lotus_notes:6.5.6.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons_beanutils:1.9.1`,
        fs:  `cpe:2.3:o:cisco:sn_5428_storage_router:3.2.1-k9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons_collections:3.2.1`,
        fs:  `cpe:2.3:a:ibm:rational_requirements_composer:2.0.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons_collections:4.0`,
        fs:  `cpe:2.3:a:joomla:joomla\!:1.6:beta13:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons_email:1.0`,
        fs:  `cpe:2.3:a:ponsoftware:explzh:3.82:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons_email:1.1`,
        fs:  `cpe:2.3:a:nullsoft:winamp:5.09:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons_email:1.2`,
        fs:  `cpe:2.3:a:mailenable:mailenable:1.76:*:professional:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons_email:1.3`,
        fs:  `cpe:2.3:a:perl:perl:0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons_email:1.3.1`,
        fs:  `cpe:2.3:a:cisco:cloud_portal:9.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons_email:1.3.2`,
        fs:  `cpe:2.3:a:gnome:glib:2.27.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons_email:1.3.3`,
        fs:  `cpe:2.3:a:diversesolutions:dsidxpress_idx_plugin:2.0.6:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons_email:1.4`,
        fs:  `cpe:2.3:a:typo3:typo3:7.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons_fileupload:1.0`,
        fs:  `cpe:2.3:o:microsoft:windows_vista:-:sp1:x86-ultimate:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons_fileupload:1.1`,
        fs:  `cpe:2.3:a:igor_sysoev:nginx:1.0.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons_fileupload:1.1.1`,
        fs:  `cpe:2.3:a:gizra:og_vocab:6.x-1.x:dev:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons_fileupload:1.2`,
        fs:  `cpe:2.3:o:cisco:ios:12.0\(27\)s:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons_fileupload:1.2.1`,
        fs:  `cpe:2.3:a:zooeffect:zooeffect:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons_fileupload:1.2.2`,
        fs:  `cpe:2.3:a:mozilla:thunderbird:10.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons_fileupload:1.3`,
        fs:  `cpe:2.3:a:netwin:surgeftp:2.2k1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:commons_fileupload:1.3.1`,
        fs:  `cpe:2.3:a:apache:subversion:1.7.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cordova:3.5.0::~~~android~~`,
        fs:  `cpe:2.3:a:hitachi:ucosminexus_developer:7:-:standard:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cordova:3.9.1::~~~iphone_os~~`,
        fs:  `cpe:2.3:a:google:chrome:25.0.1364.121:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cordova:5.2.2::~~~android~~`,
        fs:  `cpe:2.3:a:mongodb:mongodb:3.0.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cordova_file_transfer:1.2.1::~~~android~~`,
        fs:  `cpe:2.3:a:mediawiki:mediawiki:1.22.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:couchdb:1.0.0`,
        fs:  `cpe:2.3:a:debian:phpbb3:3.0.10-2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:couchdb:1.0.1`,
        fs:  `cpe:2.3:o:cisco:ios:15.0\(2\)se9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:couchdb:1.0.2`,
        fs:  `cpe:2.3:o:microsoft:windows-nt:xp:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:couchdb:1.0.3`,
        fs:  `cpe:2.3:a:hp:toolbox:90.0.146.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:couchdb:1.0.4`,
        fs:  `cpe:2.3:a:cmsmadesimple:cms_made_simple:1.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:couchdb:1.1.0`,
        fs:  `cpe:2.3:a:mawashimono:nikki:5.02:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:couchdb:1.1.1`,
        fs:  `cpe:2.3:a:webglimpse:webglimpse:2.17.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:couchdb:1.1.2`,
        fs:  `cpe:2.3:a:quagga:quagga:0.99.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:couchdb:1.2.0`,
        fs:  `cpe:2.3:a:zabbix:zabbix:1.5.2:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:couchdb:1.2.1`,
        fs:  `cpe:2.3:a:freedesktop:poppler:0.10.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:couchdb:1.5.0`,
        fs:  `cpe:2.3:a:nongnu:oath_toolkit:1.12.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:coyote_http_connector:-`,
        fs:  `cpe:2.3:a:invensys:wonderware_archestra_integrated_development_environment:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:coyote_http_connector:1.0`,
        fs:  `cpe:2.3:a:user_relationships_project:user_relationships:6.x-1.0:beta3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:coyote_http_connector:1.1`,
        fs:  `cpe:2.3:a:ibm:lotus_notes:6.5.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.4.0`,
        fs:  `cpe:2.3:a:phpmyadmin:phpmyadmin:4.4.15.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.4.1`,
        fs:  `cpe:2.3:h:rockwellautomation:rslinx_enterprise:cpr9:sr5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.4.2`,
        fs:  `cpe:2.3:a:efrontlearning:efront:3.6.14.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.4.3`,
        fs:  `cpe:2.3:a:yeast_infection_project:yeast_infection:0.1:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.4.4`,
        fs:  `cpe:2.3:o:huawei:s5700_firmware:v200r005c03:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.4.5`,
        fs:  `cpe:2.3:o:yahama:rt140i:6.01.07:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.4.6`,
        fs:  `cpe:2.3:a:simplemachines:simple_machines_forum:1.0.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.4.7`,
        fs:  `cpe:2.3:a:chaos_tool_suite_project:ctools:6.x-1.0:alpha3:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.5.0`,
        fs:  `cpe:2.3:a:ray_stode:accountsservice:0.6.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.5.1`,
        fs:  `cpe:2.3:a:ikiwiki:ikiwiki:3.1415926:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.5.2`,
        fs:  `cpe:2.3:a:google:chrome:5.0.375.78:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.5.3`,
        fs:  `cpe:2.3:a:adobe:form_client_end-user_components:5.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.5.4`,
        fs:  `cpe:2.3:a:cisco:web_security_appliance:9.0.0-193:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.5.5`,
        fs:  `cpe:2.3:a:jenkins:script_security:1.21:*:*:*:*:jenkins:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.5.6`,
        fs:  `cpe:2.3:a:moodle:moodle:3.2.0:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.5.7`,
        fs:  `cpe:2.3:o:hospira:lifecare_pcainfusion_pump_firmware:412:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.5.8`,
        fs:  `cpe:2.3:a:apache:tomcat:4.1.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.5.9`,
        fs:  `cpe:2.3:a:amos_benari:rbovirt:0.0.4:*:*:*:*:ruby:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.6.0`,
        fs:  `cpe:2.3:a:videolan:vlc_media_player:1.1.10.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.6.1`,
        fs:  `cpe:2.3:a:splunk:splunk:3.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.6.2`,
        fs:  `cpe:2.3:a:iss:realsecure_sentry:3.6:ebs:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.6.3`,
        fs:  `cpe:2.3:a:sun:sdk:1.4.2_26:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.6.4`,
        fs:  `cpe:2.3:a:cloudbees:jenkins:1.409.2:-:lts:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.6.5`,
        fs:  `cpe:2.3:a:ibm:db2_connect:10.1:*:*:*:application_server:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.6.6`,
        fs:  `cpe:2.3:a:moodle:moodle:2.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.6.7`,
        fs:  `cpe:2.3:a:digium:asterisk:10.2.0:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.6.8`,
        fs:  `cpe:2.3:a:gnome:glib:2.29.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.6.9`,
        fs:  `cpe:2.3:a:intermesh:group-office:3.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.6.10`,
        fs:  `cpe:2.3:a:ibm:java:1.4.2.13.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.6.11`,
        fs:  `cpe:2.3:a:boost:boost:1.36.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.6.12`,
        fs:  `cpe:2.3:o:cisco:ios:12.1ea:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.6.13`,
        fs:  `cpe:2.3:a:linksalpha:social_sharing_toolkit_plugin:2.2:-:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.6.14`,
        fs:  `cpe:2.3:a:microsoft:sql_server:2005:sp4:express:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.7.0`,
        fs:  `cpe:2.3:a:hp:icewall_sso_agent_option:10.0:r1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.7.1`,
        fs:  `cpe:2.3:o:cisco:email_security_appliance_firmware:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.7.2`,
        fs:  `cpe:2.3:a:organizer_project:organizer:1.1.0:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.7.3`,
        fs:  `cpe:2.3:a:microsoft:biztalk_srv:2004:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.7.4`,
        fs:  `cpe:2.3:a:christopher_mitchell:smart_breadcrumb:6.x-1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.7.5`,
        fs:  `cpe:2.3:a:ponsoftware:explzh:4.32:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.7.6`,
        fs:  `cpe:2.3:a:oracle:retail_point-of-service:7.1.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.7.7`,
        fs:  `cpe:2.3:h:h3c:s7500e_secblade_vpn_firewall_module:0231a832:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.7.8`,
        fs:  `cpe:2.3:a:cisco:intrusion_prevention_system:7.3\(2\)e4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.7.9`,
        fs:  `cpe:2.3:a:krisonav:krisonav:0.9.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.7.10`,
        fs:  `cpe:2.3:a:igor_sysoev:nginx:1.1.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.7.11`,
        fs:  `cpe:2.3:o:cisco:ios:12.1\(2\)xf:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.7.12`,
        fs:  `cpe:2.3:a:hp:bufferchm:70.0.170.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.7.13`,
        fs:  `cpe:2.3:a:apache:knox:0.8.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:2.7.17`,
        fs:  `cpe:2.3:h:lg:l-04d:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:3.0.0`,
        fs:  `cpe:2.3:a:nlnetlabs:ldns:1.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:3.0.1`,
        fs:  `cpe:2.3:a:ibm:algo_one:5.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:3.0.6`,
        fs:  `cpe:2.3:a:apache:http_server:1.3.35:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:3.1.2`,
        fs:  `cpe:2.3:a:wesnoth:battle_for_wesnoth:1.7.10-1.8:beta3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:3.1.3`,
        fs:  `cpe:2.3:o:cisco:rv130w_wireless-n_multifunction_vpn_router_firmware:1.0.2.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:3.1.4`,
        fs:  `cpe:2.3:a:tejimaya:openpne:2.10.0:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:3.1.5`,
        fs:  `cpe:2.3:a:nttdata:web_analytics_service:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:3.1.6`,
        fs:  `cpe:2.3:a:mawashimono:nikki:5.57:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:3.1.7`,
        fs:  `cpe:2.3:o:cisco:nx-os:6.0\(2\)u6\(3\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:3.1.8`,
        fs:  `cpe:2.3:a:sun:jdk:1.6.0.290:update29:*:*:*:*:x64:*`,
    },
     
    {
        uri: `cpe:/a:apache:cxf:3.1.9`,
        fs:  `cpe:2.3:a:emc:rsa_bsafe:3.2.0:*:*:*:micro_edition_suite:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:derby:-`,
        fs:  `cpe:2.3:a:google:chrome:17.0.923.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:derby:10.1.1.0`,
        fs:  `cpe:2.3:a:igor_vlasenko:html-template-pro:0.51:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:derby:10.1.2.1`,
        fs:  `cpe:2.3:a:plainblack:webgui:7.3.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:derby:10.1.3.1`,
        fs:  `cpe:2.3:a:cloudbees:jenkins:1.313:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:derby:10.2.1.6`,
        fs:  `cpe:2.3:a:f5:big-ip_edge_gateway:11.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:derby:10.2.2.0`,
        fs:  `cpe:2.3:a:jabberd2:jabberd2:2.2.7.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:derby:10.3.3.0`,
        fs:  `cpe:2.3:a:igor_sysoev:nginx:0.8.33:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:derby:10.4.1.3`,
        fs:  `cpe:2.3:a:moinmo:moinmoin:1.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:derby:10.4.2.0`,
        fs:  `cpe:2.3:a:saltstack:salt:0.10.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:derby:10.5.1.1`,
        fs:  `cpe:2.3:a:quassel-irc:quassel:0.12.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:derby:10.5.3.0`,
        fs:  `cpe:2.3:a:interworx:web_control_panel:5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:derby:10.6.1.0`,
        fs:  `cpe:2.3:h:hp:procurve_switch_4000m:c.08.22:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:derby:10.6.2.1`,
        fs:  `cpe:2.3:a:happyworm:jplayer:2.2.24:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:derby:10.7.1.1`,
        fs:  `cpe:2.3:a:parallels:parallels_plesk_panel:10.1.1:mu_\#19:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:derby:10.8.1.2`,
        fs:  `cpe:2.3:a:oracle:application_server_web_cache:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:derby:10.8.2.2`,
        fs:  `cpe:2.3:a:darold:squidclamav:5.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:derby:10.8.3.0`,
        fs:  `cpe:2.3:a:git-scm:git:1.5.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:derby:10.9.1.0`,
        fs:  `cpe:2.3:a:welcart:welcart:1.4.11:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:derby:10.10.1.1`,
        fs:  `cpe:2.3:a:google:chrome:17.0.955.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:derby:10.10.2.0`,
        fs:  `cpe:2.3:a:gnome:librsvg:2.16.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:derby:10.11.1.1`,
        fs:  `cpe:2.3:a:monster_menus_module_project:monster_menus:7.x-1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:derby:10.12.1`,
        fs:  `cpe:2.3:h:radware:certaint_100:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:derby:10.12.1.1`,
        fs:  `cpe:2.3:a:rubyonrails:ruby_on_rails:3.2.4:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:derby:10.13.1.1`,
        fs:  `cpe:2.3:a:goodiware:goodreader:3.8.1:-:*:*:*:iphone_os:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:flex:4.14.0`,
        fs:  `cpe:2.3:a:bigware:bigware_shop:2.1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:geode:1.1.0`,
        fs:  `cpe:2.3:a:redhat:sysstat:4.0.7-3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:geronimo:-`,
        fs:  `cpe:2.3:a:rom_walton:boinc:7.1.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:geronimo:1.0`,
        fs:  `cpe:2.3:a:bueltge:adminimize:1.7.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:geronimo:2.0`,
        fs:  `cpe:2.3:a:mozilla:seamonkey:1.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:geronimo:2.0.1`,
        fs:  `cpe:2.3:a:apache:hadoop:2.6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:geronimo:2.0.2`,
        fs:  `cpe:2.3:o:paloaltonetworks:pan-os:4.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:geronimo:2.1`,
        fs:  `cpe:2.3:a:logitech:khalinstallwrapper:2.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:geronimo:3.0`,
        fs:  `cpe:2.3:a:zingiri:zingiri_web_shop:2.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:geronimo:3.0:beta1`,
        fs:  `cpe:2.3:a:ssh:tectia_server:5.3.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:geronimo:3.0:m1`,
        fs:  `cpe:2.3:a:kde:kde_applications:14.11.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.7.0`,
        fs:  `cpe:2.3:a:torproject:tor:0.2.9.1:alpha:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.7.0:beta_1`,
        fs:  `cpe:2.3:a:entity_reference_project:entityreference:7.x-1.0:beta1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.7.0:beta_2`,
        fs:  `cpe:2.3:a:ibm:java_sdk:5.0.12.3:*:*:*:technology:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.7.0:rc1`,
        fs:  `cpe:2.3:a:zarafa:zarafa:7.0.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.7.0:rc2`,
        fs:  `cpe:2.3:a:david_king:vino:2.32.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.7.1`,
        fs:  `cpe:2.3:a:ultraedit:ultraedit:14.20.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.7.2`,
        fs:  `cpe:2.3:a:mongodb:mongodb:2.2.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.7.3`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.139:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.7.4`,
        fs:  `cpe:2.3:a:rimarts:becky\!_internet_mail:2.58.00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.7.5`,
        fs:  `cpe:2.3:a:grape_project:grape:0.1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.7.6`,
        fs:  `cpe:2.3:o:saia_burgess_controls:pcd7.d4xxwtpf_firmware:1.24.41:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.7.7`,
        fs:  `cpe:2.3:a:novell:groupwise:8.03:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.7.8`,
        fs:  `cpe:2.3:a:gnome:librsvg:2.26.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.7.9`,
        fs:  `cpe:2.3:a:haxx:libcurl:7.21.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.7.10`,
        fs:  `cpe:2.3:a:google:v8:3.23.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.7.11`,
        fs:  `cpe:2.3:a:d-bus_project:d-bus:1.6.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.8.0`,
        fs:  `cpe:2.3:a:mariadb:mariadb:5.5.23:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.8.0:beta_1`,
        fs:  `cpe:2.3:h:hp:color_laserjet_4700:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.8.0:beta_2`,
        fs:  `cpe:2.3:a:perl:perl:2.18.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.8.0:beta_3`,
        fs:  `cpe:2.3:a:qemu:qemu:0.1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.8.0:beta_4`,
        fs:  `cpe:2.3:a:cisco:videoscape_distribution_suite_service_manager:3.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.8.0:rc1`,
        fs:  `cpe:2.3:a:google:android_api:19.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.8.0:rc2`,
        fs:  `cpe:2.3:a:mit:kerberos:5-1.10.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.8.0:rc3`,
        fs:  `cpe:2.3:a:bitbucket:xnbd:0.1.0:pre:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.8.0:rc4`,
        fs:  `cpe:2.3:a:mahara:mahara:1.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.8.1`,
        fs:  `cpe:2.3:a:cisco:prime_infrastructure:1.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.8.2`,
        fs:  `cpe:2.3:a:gnome:librsvg:2.11.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.8.3`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(2\)xk:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.8.4`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.7.0-5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.8.5`,
        fs:  `cpe:2.3:a:pnp4nagios:pnp4nagios:0.6.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.8.6`,
        fs:  `cpe:2.3:a:libguestfs:libguestfs:1.17.25:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.8.7`,
        fs:  `cpe:2.3:a:google:chrome:10.0.648.124:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.8.8`,
        fs:  `cpe:2.3:a:google:chrome:32.0.1700.54:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.8.9`,
        fs:  `cpe:2.3:a:google:chrome:17.0.963.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.9.0`,
        fs:  `cpe:2.3:a:eagletron:dvdriver:1.0.3.26:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.9.0:beta_1`,
        fs:  `cpe:2.3:a:sun:jre:1.4.1:update3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.9.0:beta_3`,
        fs:  `cpe:2.3:a:openstack:swift:1.4.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:1.9.0:beta_4`,
        fs:  `cpe:2.3:o:hp:tru64:5.0a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.0.0`,
        fs:  `cpe:2.3:a:niif:shibb_auth:6.x-4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.0.0:beta_1`,
        fs:  `cpe:2.3:a:getshopped:wp_e-commerce:3.8:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.0.0:beta_2`,
        fs:  `cpe:2.3:a:sfiab:science_fair_in_a_box:2.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.0.0:beta_3`,
        fs:  `cpe:2.3:a:sun:openoffice.org:3.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.0.0:rc1`,
        fs:  `cpe:2.3:a:xchat:xchat:2.8.7:e:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.0.0:rc2`,
        fs:  `cpe:2.3:a:memcache_project:memcache:6.x-1.9:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.0.0:rc3`,
        fs:  `cpe:2.3:a:libevent_project:libevent:2.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.0.0:rc4`,
        fs:  `cpe:2.3:a:cmsmadesimple:cms_made_simple:1.3:beta2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.0.1`,
        fs:  `cpe:2.3:a:openafs:openafs:1.5.55:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.0.2`,
        fs:  `cpe:2.3:a:adobe:acrobat:8.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.0.3`,
        fs:  `cpe:2.3:a:ftprush:ftprush:1.0.0.599:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.0.4`,
        fs:  `cpe:2.3:a:fedoraproject:389_directory_server:1.2.10:alpha8:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.0.5`,
        fs:  `cpe:2.3:a:web2py:web2py:1.83.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.0.6`,
        fs:  `cpe:2.3:a:nordugrid:nordugrid-arc:0.5.31:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.0.7`,
        fs:  `cpe:2.3:o:pheonixcontact-software:proconos_eclr:-:*:*:*:single_chip:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.0.8`,
        fs:  `cpe:2.3:a:readydesk:readydesk:9.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.1.0`,
        fs:  `cpe:2.3:a:oracle:webcenter_sites:7.6.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.1.0:beta_1`,
        fs:  `cpe:2.3:a:yahoo:messenger:9.0.0.1389:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.1.0:rc1`,
        fs:  `cpe:2.3:a:typoheads:formhandler:0.9.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.1.0:rc2`,
        fs:  `cpe:2.3:a:checkpoint:vpn-1_firewall_1:4.1.0:sp2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.1.0:rc3`,
        fs:  `cpe:2.3:a:hitachi:ucosminexus_developer:08-00-02:-:standard:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.1.1`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.0.84:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.1.2`,
        fs:  `cpe:2.3:a:google:chrome:20.0.1132.31:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.1.3`,
        fs:  `cpe:2.3:a:hp:jetadmin:6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.1.4`,
        fs:  `cpe:2.3:a:hp:insight_management_wbem_providers_for_windows_server_2003\/2008:2.4.0.0:*:*:*:*:*:x64:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.1.5`,
        fs:  `cpe:2.3:a:openssl:openssl:0.9.5:beta2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.1.6`,
        fs:  `cpe:2.3:a:ssh:tectia_server:6.0.1.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.1.7`,
        fs:  `cpe:2.3:a:libpng:libpng:1.0.59:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.1.8`,
        fs:  `cpe:2.3:a:ppfeufer:2-click-social-media-buttons:0.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.1.9`,
        fs:  `cpe:2.3:a:oracle:vm_virtualbox:4.1.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.2.0`,
        fs:  `cpe:2.3:a:microsoft:publisher:2002:sp3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.2.0:beta_1`,
        fs:  `cpe:2.3:a:google:web_server:2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.2.0:beta_2`,
        fs:  `cpe:2.3:o:hp:tru64:4.0g_pk4_bl22:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.2.0:rc1`,
        fs:  `cpe:2.3:a:linksalpha:social_sharing_toolkit_plugin:2.0.0:-:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.2.0:rc2`,
        fs:  `cpe:2.3:a:mediawiki:mediawiki:1.3.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.2.0:rc3`,
        fs:  `cpe:2.3:a:shiromuku:bu2_bbs:2.80:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.2.1`,
        fs:  `cpe:2.3:a:curecos:cure_viewer:1.03:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.2.2`,
        fs:  `cpe:2.3:a:cyrus:imap:2.3.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.3.0`,
        fs:  `cpe:2.3:o:fortinet:fortiadc_firmware:3.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.3.0:beta_1`,
        fs:  `cpe:2.3:a:mime_mail_module_project:mimemail:6.x-1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.3.0:beta_2`,
        fs:  `cpe:2.3:a:microsoft:office:2007:sp2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.3.0:rc1`,
        fs:  `cpe:2.3:o:cisco:ios:15.2\(2\)jb4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.3.0:rc2`,
        fs:  `cpe:2.3:a:justsystems:ichitaro:2008:-:government:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.3.0:rc3`,
        fs:  `cpe:2.3:a:ibm:smartcloud_control_desk:7.5.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.3.1`,
        fs:  `cpe:2.3:a:cybozu:kunai:3.1.0:*:*:*:*:iphone_os:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.3.2`,
        fs:  `cpe:2.3:a:ibm:java_sdk:6.0.6.0:*:*:*:technology:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.3.3`,
        fs:  `cpe:2.3:a:cyrus:imap:2.4.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.3.4`,
        fs:  `cpe:2.3:a:w1.f1:hostapd:0.3.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.3.5`,
        fs:  `cpe:2.3:a:invensys:wonderware_archestra_configuration_access_component_activex_control:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.3.6`,
        fs:  `cpe:2.3:o:openbsd:openbsd:4.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.3.7`,
        fs:  `cpe:2.3:a:ppfeufer:2-click-social-media-buttons:0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.3.8`,
        fs:  `cpe:2.3:a:mysql:mysql:5.5.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.3.9`,
        fs:  `cpe:2.3:a:hp:insight_management_wbem_providers:2.2.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.3.10`,
        fs:  `cpe:2.3:a:oracle:application_server:1.0.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.3.11`,
        fs:  `cpe:2.3:a:alex_launi:tangerine:0.3.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.4.0`,
        fs:  `cpe:2.3:a:libtiff:libtiff:3.9.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.4.0:beta_1`,
        fs:  `cpe:2.3:a:apache:http_server:1.3.41:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.4.0:beta_2`,
        fs:  `cpe:2.3:a:cmsmadesimple:cms_made_simple:1.7.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.4.0:beta_3`,
        fs:  `cpe:2.3:a:soprano:fonecta_verify:7.x-1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.4.0:beta_4`,
        fs:  `cpe:2.3:a:ibm:filenet_content_manager:4.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.4.0:rc1`,
        fs:  `cpe:2.3:a:skysa:skysa_app_bar_integration_plugin:1.75:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.4.0:rc2`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(25\)ey3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.4.1`,
        fs:  `cpe:2.3:h:moxa:oncell_gateway_g3251:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.4.2`,
        fs:  `cpe:2.3:a:moinejf:abcm2ps:4.8.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:groovy:2.4.3`,
        fs:  `cpe:2.3:h:cisco:wap4410n:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.14.1`,
        fs:  `cpe:2.3:a:php:php:4.3.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.14.3`,
        fs:  `cpe:2.3:a:mozilla:bugzilla:2.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.14.4`,
        fs:  `cpe:2.3:h:iij:seil\/neu_2fe_plus:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.15.0`,
        fs:  `cpe:2.3:a:primetek:primefaces:6.0.19:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.15.1`,
        fs:  `cpe:2.3:a:atlassian:fisheye:2.1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.15.2`,
        fs:  `cpe:2.3:o:microsoft:windows_server_2012:r2:-:-:*:standard:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.15.3`,
        fs:  `cpe:2.3:a:cart66:cart66_lite_plugin:1.4.0:-:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.16.0`,
        fs:  `cpe:2.3:a:gnu:automake:1.7.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.16.1`,
        fs:  `cpe:2.3:a:phpmanufaktur:kitform:0.38:*:*:*:*:keepintouch:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.16.2`,
        fs:  `cpe:2.3:o:cisco:ios:12.3\(2\)xc2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.16.3`,
        fs:  `cpe:2.3:o:cisco:skinny_client_control_protocol_software:8.1\(2\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.16.4`,
        fs:  `cpe:2.3:a:ibm:curam_social_program_management:6.0.5.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.17.0`,
        fs:  `cpe:2.3:a:kindsoft:kindeditor:4.1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.17.1`,
        fs:  `cpe:2.3:a:hp:system_management_homepage:3.2.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.17.2`,
        fs:  `cpe:2.3:a:ibm:marketing_operations:8.2.0.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.18.0`,
        fs:  `cpe:2.3:a:cart66:cart66_lite_plugin:1.4.9:-:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.18.1`,
        fs:  `cpe:2.3:a:spamdyke:spamdyke:4.0.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.18.2`,
        fs:  `cpe:2.3:a:apache:apr-util:1.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.18.3`,
        fs:  `cpe:2.3:a:isc:dhcp:4.1.0:a1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.19.0`,
        fs:  `cpe:2.3:a:ecommercesoft:xse_shopping_cart:1.5.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.19.1`,
        fs:  `cpe:2.3:a:cisco:secure_desktop:3.5.2008:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.19.2`,
        fs:  `cpe:2.3:a:linuxfoundation:cups-filters:1.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.20.0`,
        fs:  `cpe:2.3:a:moodle:moodle:2.7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.20.1`,
        fs:  `cpe:2.3:a:horde:imp:4.3.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.20.2`,
        fs:  `cpe:2.3:a:cisco:unified_communications_manager:8.6\(2a\)su1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.20.203.0`,
        fs:  `cpe:2.3:a:typo3:typo3:3.8.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.20.204.0`,
        fs:  `cpe:2.3:a:ipswitch:ws_ftp:3.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.20.205.0`,
        fs:  `cpe:2.3:a:vmware:vcenter_orchestrator:5.5.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.21.0`,
        fs:  `cpe:2.3:a:alterpoint:networkauthority_engineering_environment:6.0.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.22.0`,
        fs:  `cpe:2.3:a:mongodb:mongodb:3.0.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.23.0`,
        fs:  `cpe:2.3:a:cisco:firewall_services_module_software:3.2\(12\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.23.1`,
        fs:  `cpe:2.3:a:cisco:unified_contact_center_enterprise:5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.23.3`,
        fs:  `cpe:2.3:o:microsoft:windows-nt:2000:sp3:srv:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.23.4`,
        fs:  `cpe:2.3:a:mortbay:jetty:6.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.23.5`,
        fs:  `cpe:2.3:a:ovirt:ovirt:4.1.7.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.23.6`,
        fs:  `cpe:2.3:a:atlassian:fisheye:2.5.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.23.7`,
        fs:  `cpe:2.3:a:s3ql_project:s3ql:1.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.23.8`,
        fs:  `cpe:2.3:a:maynard_johnson:oprofile:0.7.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.23.9`,
        fs:  `cpe:2.3:a:cakefoundation:cakephp:1.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.23.10`,
        fs:  `cpe:2.3:a:novell:groupwise:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:0.23.11`,
        fs:  `cpe:2.3:a:amberdms:amberdms_billing_system:1.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:1.0.0`,
        fs:  `cpe:2.3:o:extremenetworks:extremeware_xos:11.0.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:1.0.1`,
        fs:  `cpe:2.3:a:microsoft:home_publishing:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:1.0.2`,
        fs:  `cpe:2.3:a:libpng:libpng:1.2.51:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:1.0.3`,
        fs:  `cpe:2.3:a:vips:vips:7.7.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:1.0.4`,
        fs:  `cpe:2.3:a:ghozylab:gallery_-_photo_albums_-_portfolio:1.3.47:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:1.1.0`,
        fs:  `cpe:2.3:a:ntp:ntp:4.3.65:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:1.1.1`,
        fs:  `cpe:2.3:o:huawei:s7703_firmware:v100r006:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:1.1.2`,
        fs:  `cpe:2.3:a:git-scm:git:0.8.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:1.2.0`,
        fs:  `cpe:2.3:a:google:chrome:10.0.648.133:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:2.0.0:alpha`,
        fs:  `cpe:2.3:a:oracle:identity_analytics:11.1.1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:2.0.1:alpha`,
        fs:  `cpe:2.3:o:yamaha:rt100i:1.05.09:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:2.0.2:alpha`,
        fs:  `cpe:2.3:a:southrivertech:titan_ftp_server:1.11.34:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:2.0.3:alpha`,
        fs:  `cpe:2.3:o:iij:seil_x2_firmware:4.46:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:2.0.4:alpha`,
        fs:  `cpe:2.3:a:xnview:xnview:1.32:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:2.0.5:alpha`,
        fs:  `cpe:2.3:a:mcafee:epolicy_orchestrator:4.5.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:2.0.6:alpha`,
        fs:  `cpe:2.3:a:libpng:libpng:1.0.5g:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:2.1.0:beta`,
        fs:  `cpe:2.3:a:powerdns:authoritative:3.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:2.1.1:beta`,
        fs:  `cpe:2.3:a:m2osw:tableofcontents:6.x-3.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:2.2.0`,
        fs:  `cpe:2.3:o:iij:seil_x86_fuji_firmware:2.87:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:2.3.0`,
        fs:  `cpe:2.3:a:google:chrome:8.0.552.218:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:2.4.0`,
        fs:  `cpe:2.3:a:syntevo:smartsvn:5.0:beta_4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:2.4.1`,
        fs:  `cpe:2.3:a:genetechsolutions:pie-register:1.2.8:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:2.5.0`,
        fs:  `cpe:2.3:a:oracle:jdk:1.6.0:update_39:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:2.5.1`,
        fs:  `cpe:2.3:a:hp:centralview_dealer_performance_audit:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:2.6.0`,
        fs:  `cpe:2.3:a:oracle:database_server:11.1.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:2.6.1`,
        fs:  `cpe:2.3:a:soprano:fonecta_verify:7.x-1.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:2.6.2`,
        fs:  `cpe:2.3:a:konqueror:konqueror:3.1:rc6:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:2.6.3`,
        fs:  `cpe:2.3:a:cisco:adaptive_security_appliance_software:9.4.3.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:2.6.4`,
        fs:  `cpe:2.3:a:bestpractical:rt:3.8.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:2.7.0`,
        fs:  `cpe:2.3:a:samba:samba:3.0.28:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:2.7.1`,
        fs:  `cpe:2.3:a:immigrer:forum_ic:3.3.12:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hadoop:2.7.2`,
        fs:  `cpe:2.3:a:xnview:xnview:1.08:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:harmony:6.0:m3`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.4.5-0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.92.0`,
        fs:  `cpe:2.3:a:cloudbees:jenkins:1.379:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.92.1`,
        fs:  `cpe:2.3:a:atlassian:crowd:2.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.92.2`,
        fs:  `cpe:2.3:a:devscripts_devel_team:devscripts:2.13.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.92.3`,
        fs:  `cpe:2.3:a:apache:cxf:2.6.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.94.0`,
        fs:  `cpe:2.3:a:vanilla_perl_project:strawberry_perl:5.12.512:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.94.1`,
        fs:  `cpe:2.3:a:ibm:marketing_platform:8.6.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.94.2`,
        fs:  `cpe:2.3:a:werdswords:download_shortcode:1.1:-:-:-:-:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.94.3`,
        fs:  `cpe:2.3:a:sync:oxygen_xml_editor:6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.94.4`,
        fs:  `cpe:2.3:a:fatfreecrm:fat_free_crm:0.9.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.94.5`,
        fs:  `cpe:2.3:o:suse:suse_linux:7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.94.6`,
        fs:  `cpe:2.3:o:hp:hp-ux:11.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.94.6.1`,
        fs:  `cpe:2.3:a:mautic:mautic:1.2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.94.7`,
        fs:  `cpe:2.3:a:zend:zendservice_slideshare:2.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.94.8`,
        fs:  `cpe:2.3:a:apache:http_server:2.2.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.94.9`,
        fs:  `cpe:2.3:a:rubyonrails:ruby_on_rails:4.0.6:rc3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.98.0`,
        fs:  `cpe:2.3:a:samba:samba:2.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.98.1`,
        fs:  `cpe:2.3:a:pinnaclesys:studio:11.0.13:*:*:*:ultimate:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.98.2`,
        fs:  `cpe:2.3:a:ibm:lotus_inotes:229.191:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.98.3`,
        fs:  `cpe:2.3:a:devexpress:aspxfilemanager_control_for_webforms_and_mvc:12.1.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.98.4`,
        fs:  `cpe:2.3:a:nancy_wichmann:sitedoc:6.x-1.x:dev:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.98.5`,
        fs:  `cpe:2.3:a:piwigo:piwigo:2.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.98.6`,
        fs:  `cpe:2.3:a:sun:jdk:1.6.0.270:update27:*:*:*:*:x64:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.98.6.1`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.584:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.98.7`,
        fs:  `cpe:2.3:a:ulli_horlacher:fex:20111013:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.98.8`,
        fs:  `cpe:2.3:a:prestashop:prestashop:1.4.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.98.9`,
        fs:  `cpe:2.3:a:apache:wss4j:1.6.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.98.10`,
        fs:  `cpe:2.3:h:hp:cm8060_color_mfp:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.98.10.1`,
        fs:  `cpe:2.3:o:iij:seil_b1_firmware:3.76:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.98.11`,
        fs:  `cpe:2.3:a:libguestfs:libguestfs:1.5.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hbase:0.98.12`,
        fs:  `cpe:2.3:h:lexmark:xc8155de:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hive:0.13.0`,
        fs:  `cpe:2.3:a:collectiveaccess:providence:1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hive:0.13.1`,
        fs:  `cpe:2.3:o:cisco:pix_firewall:5.2\(7\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hive:0.14.0`,
        fs:  `cpe:2.3:a:c-ares_project:c-ares:1.7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hive:1.0.0`,
        fs:  `cpe:2.3:a:nick_copeland:bristol:0.40.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hive:1.0.1`,
        fs:  `cpe:2.3:h:hp:procurve_switch_2626:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hive:1.1.0`,
        fs:  `cpe:2.3:a:symantec:veritas_volume_replicator:4.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hive:1.1.1`,
        fs:  `cpe:2.3:a:silexlabs:silex:2.0.0:alpha5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hive:1.2.0`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.14.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:hive:1.2.1`,
        fs:  `cpe:2.3:a:google:chrome:14.0.792.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:-`,
        fs:  `cpe:2.3:a:curvycorners:curvycorners:6.x-1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:0.8.11`,
        fs:  `cpe:2.3:a:tribiq:tribiq_cms:5.0.9a:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:0.8.14`,
        fs:  `cpe:2.3:o:microsoft:windows_nt:4.0:sp6:terminal_server:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.0`,
        fs:  `cpe:2.3:a:wolfssl:wolfssl:3.10.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.0.2`,
        fs:  `cpe:2.3:a:matthew_booth:virt-v2v:0.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.0.3`,
        fs:  `cpe:2.3:o:cisco:ios_xr:4.1\(.2\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.0.5`,
        fs:  `cpe:2.3:a:vmware:workstation:5.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.1`,
        fs:  `cpe:2.3:a:mozilla:seamonkey:1.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.1.1`,
        fs:  `cpe:2.3:a:skyarc:mtcms:5.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.2`,
        fs:  `cpe:2.3:a:apple:itunes:12.6.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.2.4`,
        fs:  `cpe:2.3:a:sawmill:sawmill:8.5.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.2.5`,
        fs:  `cpe:2.3:a:sun:solaris_libfont:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.2.6`,
        fs:  `cpe:2.3:a:netiq:sentinel_agent_manager:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.2.9`,
        fs:  `cpe:2.3:a:vmware:workstation_player:12.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3`,
        fs:  `cpe:2.3:a:avaya:libsafe:2.0.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.0`,
        fs:  `cpe:2.3:a:ikiwiki:ikiwiki:2.62:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.1`,
        fs:  `cpe:2.3:a:wordpress:wordpress:3.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.1.1`,
        fs:  `cpe:2.3:a:ponsoftware:explzh:4.26:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.2`,
        fs:  `cpe:2.3:a:hp:status:82.0.173.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.3`,
        fs:  `cpe:2.3:o:cisco:nx-os:5.0\(3\)u1\(2a\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.4`,
        fs:  `cpe:2.3:a:argyllcms:argyllcms:0.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.5`,
        fs:  `cpe:2.3:a:don_ho:notepad\&#43;\&#43;:4.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.6`,
        fs:  `cpe:2.3:a:apache:solr:6.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.7`,
        fs:  `cpe:2.3:o:cisco:skinny_client_control_protocol_software:3.2\(2\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.8`,
        fs:  `cpe:2.3:h:huawei:mate_7:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.9`,
        fs:  `cpe:2.3:a:haxx:libcurl:7.23.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.10`,
        fs:  `cpe:2.3:a:clamav:clamav:0.97:rc:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.11`,
        fs:  `cpe:2.3:h:hp:eliteone_705_g2_23in_aio_t\/nt:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.12`,
        fs:  `cpe:2.3:a:radius_extension_project:radius:1.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.13`,
        fs:  `cpe:2.3:a:rocomotion:pplog:2.36:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.14`,
        fs:  `cpe:2.3:a:rockwellautomation:1763-l16dwd_series_a:15.000:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.15`,
        fs:  `cpe:2.3:a:mozilla:firefox:1.5.0.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.16`,
        fs:  `cpe:2.3:a:php:php:5.3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.17`,
        fs:  `cpe:2.3:a:oracle:database_server:10.2.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.18`,
        fs:  `cpe:2.3:o:sun:solaris:2.6:hw3:x86:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.19`,
        fs:  `cpe:2.3:a:mattias_hutterer:taxonomy_manager:7.x-1.0:alpha2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.20`,
        fs:  `cpe:2.3:a:piriform:ccleaner:5.02.5101:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.22`,
        fs:  `cpe:2.3:o:codeaurora:android-msm:3.12.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.23`,
        fs:  `cpe:2.3:o:google:chrome_os:20.0.1132.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.24`,
        fs:  `cpe:2.3:o:microsoft:windows:3.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.25`,
        fs:  `cpe:2.3:a:mcafee:e-business_server:8.5.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.26`,
        fs:  `cpe:2.3:a:bayashi:dopvstar\*:0041:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.27`,
        fs:  `cpe:2.3:a:openidm_project:openidm:4.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.28`,
        fs:  `cpe:2.3:a:microsoft:baseline_security_analyzer:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.29`,
        fs:  `cpe:2.3:a:perl:perl:0.91:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.30`,
        fs:  `cpe:2.3:a:ibm:tivoli_opc_tracker_agent:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.31`,
        fs:  `cpe:2.3:a:dlo:simple_anti_bot_registration_engine_plugin:1.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.32`,
        fs:  `cpe:2.3:o:suse:suse_linux_enterprise_desktop:12:sp3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.33`,
        fs:  `cpe:2.3:a:cloudera:cloudera_manager:4.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.34`,
        fs:  `cpe:2.3:a:plone:plone:4.3.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.35`,
        fs:  `cpe:2.3:a:texmacs:texmacs:1.0.6.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.36`,
        fs:  `cpe:2.3:a:google:chrome:18.0.1025.148:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.37`,
        fs:  `cpe:2.3:a:google:chrome:15.0.874.121:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.38`,
        fs:  `cpe:2.3:o:cisco:ios:12.3xg:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.39`,
        fs:  `cpe:2.3:a:php:pear:1.3b1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.41`,
        fs:  `cpe:2.3:a:oracle:mysql_connector\/net:6.9.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.42`,
        fs:  `cpe:2.3:a:openssl:openssl:1.0.1r:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.65`,
        fs:  `cpe:2.3:a:mcafee:change_control:5.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.3.68`,
        fs:  `cpe:2.3:a:ca:messaging:1.11.29.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.4.0`,
        fs:  `cpe:2.3:a:hitachi:cosminexus_developer:06-50-\/f:-:standard:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:1.99`,
        fs:  `cpe:2.3:a:sophos:web_appliance:3.6.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0`,
        fs:  `cpe:2.3:o:google:android:2.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.9`,
        fs:  `cpe:2.3:a:claws-mail:claws-mail:2.9.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.28`,
        fs:  `cpe:2.3:a:ati:software_uninstall_utility:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.28:beta`,
        fs:  `cpe:2.3:a:sightlinesystems:power_agent_for_windows_2003_server:6.3.1700:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.32`,
        fs:  `cpe:2.3:a:huawei:s5700_firmware:v200r003c00spc500:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.32:beta`,
        fs:  `cpe:2.3:a:google:earth:4.3.7204.836:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.34:beta`,
        fs:  `cpe:2.3:a:haxx:curl:7.9.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.35`,
        fs:  `cpe:2.3:a:torproject:tor:0.2.0.23:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.36`,
        fs:  `cpe:2.3:a:djangoproject:django:1.10.0:a1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.37`,
        fs:  `cpe:2.3:o:iij:seil_plus_firmware:2.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.38`,
        fs:  `cpe:2.3:a:mumble:mumble:1.2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.39`,
        fs:  `cpe:2.3:a:cisco:telepresence_system_software:tcu4.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.40`,
        fs:  `cpe:2.3:o:cisco:ios:12.2x:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.41`,
        fs:  `cpe:2.3:a:cisco:unified_communications_manager:10.5_base:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.42`,
        fs:  `cpe:2.3:h:cisco:pix_firewall:6.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.43`,
        fs:  `cpe:2.3:a:linux-sottises:board-tnk:1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.44`,
        fs:  `cpe:2.3:a:google:chrome:6.0.471.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.45`,
        fs:  `cpe:2.3:a:ibm:global_security_kit:7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.46`,
        fs:  `cpe:2.3:a:emulex:hbanyware:4.01a36:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.47`,
        fs:  `cpe:2.3:a:cisco:wide_area_application_services:5.1.1c:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.48`,
        fs:  `cpe:2.3:a:puppetlabs:puppet:3.3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.49`,
        fs:  `cpe:2.3:a:rpm:rpm:2.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.50`,
        fs:  `cpe:2.3:h:hp:0235a31p:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.51`,
        fs:  `cpe:2.3:a:fedoraproject:389_directory_server:1.2.8.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.52`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.9.9-10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.53`,
        fs:  `cpe:2.3:a:bueltge:adminimize:1.6.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.54`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.26.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.55`,
        fs:  `cpe:2.3:a:coppermine-gallery:coppermine_photo_gallery:1.4.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.56`,
        fs:  `cpe:2.3:a:opera:opera_browser:7.01:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.57`,
        fs:  `cpe:2.3:a:cisco:firewall_services_module_software:3.2\(13\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.58`,
        fs:  `cpe:2.3:o:google:chrome:21.0.1180.41:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.59`,
        fs:  `cpe:2.3:a:apache:ofbiz:11.04.02:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.60`,
        fs:  `cpe:2.3:a:cisco:firewall_services_module_software:3.1\(20\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.61`,
        fs:  `cpe:2.3:a:cisco:asr_5000_series_software:19.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.63`,
        fs:  `cpe:2.3:a:jenkins-ci:subversion-plugin:1.32:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.0.64`,
        fs:  `cpe:2.3:a:mislav_marohnic:will_paginate:3.0:-:-:*:-:ruby:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.1`,
        fs:  `cpe:2.3:a:invensys:foxboro_control_software:4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.1.1`,
        fs:  `cpe:2.3:a:southrivertech:titan_ftp_server:5.05.326:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.1.2`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.393:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.1.3`,
        fs:  `cpe:2.3:a:marcel_brinkkemper:lazyest-gallery:1.1.16:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.1.4`,
        fs:  `cpe:2.3:a:visualsoftru:visual_quickbar:4.5.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.1.5`,
        fs:  `cpe:2.3:a:nikola_posa:webfoliocms1.1.3:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.1.6`,
        fs:  `cpe:2.3:a:redhat:hawtjni:1.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.1.7`,
        fs:  `cpe:2.3:a:geoff_davies:contact_forms:6.x-1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.1.8`,
        fs:  `cpe:2.3:o:paloaltonetworks:pan-os:7.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.1.9`,
        fs:  `cpe:2.3:a:sonatype:nexus:2.5.0:*:*:*:open_source:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.2`,
        fs:  `cpe:2.3:a:lighttpd:lighttpd:1.4.28:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.2.0`,
        fs:  `cpe:2.3:a:otrs:otrs_help_desk:3.2.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.2.1`,
        fs:  `cpe:2.3:a:ssh:tectia_server:5.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.2.2`,
        fs:  `cpe:2.3:a:autodesk:building_systems:2005:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.2.3`,
        fs:  `cpe:2.3:a:veritas:netbackup:7.7.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.2.4`,
        fs:  `cpe:2.3:h:siemens:scalance_w786-1pro:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.2.5`,
        fs:  `cpe:2.3:a:cherry-design:wikipad:1.6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.2.6`,
        fs:  `cpe:2.3:a:microsoft:visual_studio:6.0:sp1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.2.8`,
        fs:  `cpe:2.3:a:avg:anti-virus:8.5:-:network:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.2.9`,
        fs:  `cpe:2.3:a:mathew_winstone:mobile_tools:6.x-2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.2.10`,
        fs:  `cpe:2.3:a:mobiloud:mobiloud:1.4.1:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.2.11`,
        fs:  `cpe:2.3:a:ulli_horlacher:fex:20110622:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.2.12`,
        fs:  `cpe:2.3:a:phusion:passenger:3.0.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.2.13`,
        fs:  `cpe:2.3:a:ponsoftware:archive_decoder:1.23:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.2.14`,
        fs:  `cpe:2.3:a:cisco:wireless_lan_controller_software:8.0_base:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.2.15`,
        fs:  `cpe:2.3:o:apple:mac_os_x:10.6.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.2.16`,
        fs:  `cpe:2.3:a:ibm:infosphere_optim_configuration_manager:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.2.17`,
        fs:  `cpe:2.3:a:mysql:mysql:3.23.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.2.18`,
        fs:  `cpe:2.3:a:ibm:sterling_selling_and_fulfillment_foundation:9.2.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.2.19`,
        fs:  `cpe:2.3:a:7mediaws:edutrac:1.0.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.2.20`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:7.0.4-8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.2.21`,
        fs:  `cpe:2.3:a:pdflib:pdflib:3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.2.22`,
        fs:  `cpe:2.3:a:openssl:openssl:1.0.2b:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.2.23`,
        fs:  `cpe:2.3:a:joomla:joomla\!:3.5.0:beta3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.2.24`,
        fs:  `cpe:2.3:a:hitachi:cobol2002_net_server_suite:01-02-\/f:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.2.25`,
        fs:  `cpe:2.3:a:altiris:endpoint_security_solution:6.0.479:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.2.26`,
        fs:  `cpe:2.3:a:shopware:shopware:4.2.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.2.27`,
        fs:  `cpe:2.3:a:nero:photosnap_help:1.53.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.2.34`,
        fs:  `cpe:2.3:a:mozilla:firefox:17.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.3.0`,
        fs:  `cpe:2.3:a:skype:skype:4.0:beta_3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.3.1`,
        fs:  `cpe:2.3:a:google:chrome:4.0.221.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.3.2`,
        fs:  `cpe:2.3:a:southrivertech:titan_ftp_server:3.01.163:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.3.3`,
        fs:  `cpe:2.3:o:cisco:ios:12.2sm:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.3.4`,
        fs:  `cpe:2.3:h:netgear:rt314:3.22:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.3.5`,
        fs:  `cpe:2.3:a:wpeasycart:wp_easycart:2.1.24:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.3.6`,
        fs:  `cpe:2.3:a:google:chrome:4.0.249.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.3.7`,
        fs:  `cpe:2.3:a:spumko_project:hapi_server_framework:2.2.0:*:*:*:*:node.js:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.3.8`,
        fs:  `cpe:2.3:o:iij:seil_b1_firmware:3.97:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.3.9`,
        fs:  `cpe:2.3:a:octopus:octopus_deploy:3.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.3.10`,
        fs:  `cpe:2.3:a:wpeasycart:wp_easycart:2.1.15:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.3.11`,
        fs:  `cpe:2.3:o:juniper:netscreen_screenos:4.0.0r6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.3.12`,
        fs:  `cpe:2.3:a:freeradius:freeradius:1.1.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.3.13`,
        fs:  `cpe:2.3:o:hp:hp-ux:11.00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.3.14`,
        fs:  `cpe:2.3:a:torproject:tor:0.2.3.5:alpha:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.3.15`,
        fs:  `cpe:2.3:a:rocomotion:pplog:2.28:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.3.16`,
        fs:  `cpe:2.3:a:splunk:splunk:3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.4.0`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.18:rc3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.4.1`,
        fs:  `cpe:2.3:a:apache:nutch:0.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.4.2`,
        fs:  `cpe:2.3:a:fitnesse:fitnesse_wiki:20140201:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.4.3`,
        fs:  `cpe:2.3:a:s3ql_project:s3ql:2.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.4.6`,
        fs:  `cpe:2.3:a:bravenewcode:wptouch:1.9.9.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.4.7`,
        fs:  `cpe:2.3:a:simplemachines:simple_machines_forum:2.0:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.4.8`,
        fs:  `cpe:2.3:h:hp:laserjet_2430:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.4.9`,
        fs:  `cpe:2.3:a:mozilla:seamonkey:2.14:beta3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.4.10`,
        fs:  `cpe:2.3:a:ssh:tectia_client:6.0.9.24:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.4.12`,
        fs:  `cpe:2.3:a:tag1consulting:support_timer:6.x-1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.4.14`,
        fs:  `cpe:2.3:a:jordy_meow:media_file_renamer:1.4:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.4.16`,
        fs:  `cpe:2.3:a:vmware:fusion_pro:8.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.4.19`,
        fs:  `cpe:2.3:a:cisco:unified_computing_system:2.1_2a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.4.20`,
        fs:  `cpe:2.3:a:websense:websense_web_security:7.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.4.21`,
        fs:  `cpe:2.3:a:pivotal_software:redis:3.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.4.22`,
        fs:  `cpe:2.3:a:john_nunemaker:httparty:0.2.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.4.23`,
        fs:  `cpe:2.3:a:cloudbees:jenkins:1.301:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:2.4.27`,
        fs:  `cpe:2.3:a:xymon:xymon:4.3.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:http_server:3.1`,
        fs:  `cpe:2.3:h:bluecoat:proxysg_sg210-25:-:-:acceleration:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpasyncclient:4.0`,
        fs:  `cpe:2.3:a:buffalotech:bbr-4mg_firmware:1.00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpasyncclient:4.0:alpha1`,
        fs:  `cpe:2.3:a:wafer:webmatic:3.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpasyncclient:4.0:alpha2`,
        fs:  `cpe:2.3:a:cisco:unity_express_software:7.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpasyncclient:4.0:alpha3`,
        fs:  `cpe:2.3:a:ckeditor:fckeditor:6.x-2.0:alpha1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpasyncclient:4.0:beta1`,
        fs:  `cpe:2.3:h:d-link:dcs-2230l:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpasyncclient:4.0:beta2`,
        fs:  `cpe:2.3:a:gcspublishing:georgia_packing:3.9.16:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpasyncclient:4.0:beta3`,
        fs:  `cpe:2.3:a:freemind:freemind:0.9.0:beta9:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpasyncclient:4.0:beta4`,
        fs:  `cpe:2.3:a:vasthtml:forumpress:1.7.3:-:-:*:-:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpasyncclient:4.0.1`,
        fs:  `cpe:2.3:a:kanboard:kanboard:1.0.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpasyncclient:4.0.2`,
        fs:  `cpe:2.3:a:mcafee:security_center:6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpclient:4.0`,
        fs:  `cpe:2.3:a:cisco:cloud_portal:9.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpclient:4.0:alpha1`,
        fs:  `cpe:2.3:o:iij:seil_x1_firmware:4.00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpclient:4.0:alpha2`,
        fs:  `cpe:2.3:a:sybase:pylon_anywhere:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpclient:4.0:alpha3`,
        fs:  `cpe:2.3:a:email\:\:address_module_project:email\:\:address:1.898:*:*:*:*:perl:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpclient:4.0:alpha4`,
        fs:  `cpe:2.3:a:openssl:openssl:0.9.8e:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpclient:4.0:beta1`,
        fs:  `cpe:2.3:a:cisco:tidal_enterprise_scheduler:3.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpclient:4.0:beta2`,
        fs:  `cpe:2.3:o:mercurycom:mr804_firmware:3.8.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpclient:4.0.1`,
        fs:  `cpe:2.3:a:ati:catalyst_control_center_installproxy:2009.1104.959.17837:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpclient:4.1`,
        fs:  `cpe:2.3:a:debian:trousers:0.3.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpclient:4.1:alpha1`,
        fs:  `cpe:2.3:a:pivotx:pivotx:2.3.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpclient:4.1:alpha2`,
        fs:  `cpe:2.3:a:ricky_morse:excluded_users:6.x-1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpclient:4.1:beta1`,
        fs:  `cpe:2.3:a:lemonldap-ng:lemonldap\:\::0.9.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpclient:4.1.1`,
        fs:  `cpe:2.3:a:hostbillapp:hostbill:4.9.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpclient:4.1.2`,
        fs:  `cpe:2.3:a:ibm:security_siteprotector_system:3.0.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpclient:4.2`,
        fs:  `cpe:2.3:a:oracle:jre:1.5.0:update_39:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpclient:4.2:alpha1`,
        fs:  `cpe:2.3:a:happyworm:jplayer:0.2.5:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpclient:4.2:beta1`,
        fs:  `cpe:2.3:a:ponsoftware:explzh:5.32:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpclient:4.2.1`,
        fs:  `cpe:2.3:a:haxx:libcurl:7.15.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpclient:4.2.2`,
        fs:  `cpe:2.3:a:google:chrome:18.0.1025.140:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpclient:4.2.3`,
        fs:  `cpe:2.3:a:projectsend:projectsend:180:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpclient:4.3`,
        fs:  `cpe:2.3:h:huawei:s7700:v100r006:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpclient:4.3:alpha1`,
        fs:  `cpe:2.3:a:plone:plone:4.2.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpclient:4.3:beta1`,
        fs:  `cpe:2.3:a:polycom:datastore:5.22.109.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpclient:4.3:beta2`,
        fs:  `cpe:2.3:h:lucent:ascend_pipeline_router:6.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpclient:4.3.1`,
        fs:  `cpe:2.3:a:sun:jre:1.6.0.30:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpclient:4.3.2`,
        fs:  `cpe:2.3:h:nortel:ip_phone_1140e:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpclient:4.3.3`,
        fs:  `cpe:2.3:a:adobe:shockwave_player:12.0.2.122:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpclient:4.3.4`,
        fs:  `cpe:2.3:a:ponsoftware:explzh:2.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:httpclient:4.3.5`,
        fs:  `cpe:2.3:a:adobe:photoshop:12.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ignite:1.0.0`,
        fs:  `cpe:2.3:a:bayashi:dopvstar\*:0086:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ignite:1.0.0:rc3`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.370:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ignite:1.1.0`,
        fs:  `cpe:2.3:o:sgi:irix:6.5.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ignite:1.2.0`,
        fs:  `cpe:2.3:o:linux:linux_kernel:4.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ignite:1.3.0`,
        fs:  `cpe:2.3:a:lunarline:scapsync:0.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ignite:1.4.0`,
        fs:  `cpe:2.3:o:yamaha:rt200i:2.02.40:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ignite:1.5.0:b1`,
        fs:  `cpe:2.3:a:jruby:jruby:1.5.0:rc3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ignite:1.5.0:final`,
        fs:  `cpe:2.3:a:ibm:security_appscan:8.0.1.1:-:enterprise:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ignite:1.6.0`,
        fs:  `cpe:2.3:a:bundler:bundler:1.12.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ignite:1.7.0`,
        fs:  `cpe:2.3:a:contao:contao_cms:2.9:beta1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ignite:1.8.0`,
        fs:  `cpe:2.3:o:freebsd:freebsd:2.2.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ignite:1.9.0`,
        fs:  `cpe:2.3:a:heimdal_project:heimdal:1.3.2:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ignite:2.0.0`,
        fs:  `cpe:2.3:a:mahara:mahara:1.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:1.4`,
        fs:  `cpe:2.3:a:bigtreecms:bigtree_cms:4.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:1.5.0`,
        fs:  `cpe:2.3:a:puppetlabs:puppet:0.25.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.0.5`,
        fs:  `cpe:2.3:a:microsoft:excel:97:sr1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.2.0`,
        fs:  `cpe:2.3:o:belkin:n150_f9k1009_firmware:1.00.07:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.2.1`,
        fs:  `cpe:2.3:a:ibm:websphere_service_registry_and_repository:8.0.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.2.2`,
        fs:  `cpe:2.3:a:invisionpower:ip.nexus:1.5.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.2.4`,
        fs:  `cpe:2.3:a:kadu:kadu:0.11.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.2.5`,
        fs:  `cpe:2.3:a:cloudfoundry:cf-release:149:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.2.7`,
        fs:  `cpe:2.3:a:cmu:isilk:0.1.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.2.8`,
        fs:  `cpe:2.3:a:yoshinori_tahara:mycaljp:2.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.2.9`,
        fs:  `cpe:2.3:a:buffalotech:bbr-4mg_firmware:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.2.10`,
        fs:  `cpe:2.3:a:intelliants:esyndicat:2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.2.11`,
        fs:  `cpe:2.3:a:ibm:maximo_asset_management:7.5.0.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.2.12`,
        fs:  `cpe:2.3:a:lansweeper:lansweeper:5.3.0.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.2.13`,
        fs:  `cpe:2.3:a:google:chrome:4.0.249.44:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.4.0`,
        fs:  `cpe:2.3:a:hitachi:cosminexus_application_server:06-51-\/e:-:standard:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.4.1`,
        fs:  `cpe:2.3:a:jenkins:github_branch_source:1.3:*:*:*:*:jenkins:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.4.2`,
        fs:  `cpe:2.3:a:novell:iprint:4.38:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.4.3`,
        fs:  `cpe:2.3:a:rocomotion:p_board_ri:1.03:*:g:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.4.4`,
        fs:  `cpe:2.3:a:rsyslog:rsyslog:7.1.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.4.5`,
        fs:  `cpe:2.3:a:fetchmail:fetchmail:6.2.5.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.6.0`,
        fs:  `cpe:2.3:a:buffalotech:bhr-4rv_firmware:2.46:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.6.1`,
        fs:  `cpe:2.3:o:juniper:netscreen_screenos:3.1.0r3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.6.2`,
        fs:  `cpe:2.3:a:mysql:mysql:5.0.4a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.6.3`,
        fs:  `cpe:2.3:a:apache:hadoop:2.1.0:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.6.4`,
        fs:  `cpe:2.3:a:microsoft:dhcp_client_service:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.6.5`,
        fs:  `cpe:2.3:a:gitlab:gitlab-shell:1.7.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.8.0`,
        fs:  `cpe:2.3:a:zikula:zikula_application_framework:1.3.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.8.1`,
        fs:  `cpe:2.3:a:adobe:connect:7.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.8.2`,
        fs:  `cpe:2.3:a:sync:oxygen_xml_editor:9.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.10.0`,
        fs:  `cpe:2.3:a:google:chrome:8.0.552.315:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.10.1`,
        fs:  `cpe:2.3:a:ubnt:unifi_video:2.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.10.2`,
        fs:  `cpe:2.3:a:openx:openx:2.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.10.3`,
        fs:  `cpe:2.3:a:matchalabs:metaslider:2.5:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.12.0`,
        fs:  `cpe:2.3:a:libpng:libpng:1.5.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.12.1`,
        fs:  `cpe:2.3:a:cisco:desktop_base:6.1.3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.12.2`,
        fs:  `cpe:2.3:a:microsoft:sql_server:2008:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.12.3`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(31\)sb3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.13.0`,
        fs:  `cpe:2.3:a:oracle:database_server:8.0.6.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.13.1`,
        fs:  `cpe:2.3:a:oracle:corporate_time_outlook_connector:3.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jackrabbit:2.13.2`,
        fs:  `cpe:2.3:a:webassist:powerstore:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jakarta_isapi_redirector:1.2.12`,
        fs:  `cpe:2.3:a:hp:insight_diagnostics:7.5.5.1681:unknown:online_windows_2000:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jakarta_isapi_redirector:1.2.14`,
        fs:  `cpe:2.3:a:elasticsearch:kibana:3.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jakarta_isapi_redirector:1.2.15`,
        fs:  `cpe:2.3:a:octopus:octopus_deploy:3.5.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jakarta_slide:-`,
        fs:  `cpe:2.3:a:modx:modx_revolution:2.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:jakarta_slide:2.1`,
        fs:  `cpe:2.3:h:xerox:workcentre_m55:4.84.16.000:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:james:-`,
        fs:  `cpe:2.3:a:oracle:mysql_workbench:6.3.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:james:2.2.0`,
        fs:  `cpe:2.3:a:nvidia:nview:136.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:james_server:2.3.2`,
        fs:  `cpe:2.3:a:openbsd:openssh:6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:james_server:3.0.0`,
        fs:  `cpe:2.3:a:atvise:webmi2ads:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:juddi:0.9:rc1`,
        fs:  `cpe:2.3:a:debian:bsdmainutils:6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:juddi:0.9:rc2`,
        fs:  `cpe:2.3:a:oracle:mysql:5.6.38:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:juddi:0.9:rc3`,
        fs:  `cpe:2.3:o:cisco:ios:12.0xr:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:juddi:0.9:rc4`,
        fs:  `cpe:2.3:a:moinejf:abcm2ps:4.12.28:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:juddi:2.0:rc5`,
        fs:  `cpe:2.3:a:adobe:flash_player:22.0.0.211:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:juddi:3.1.2`,
        fs:  `cpe:2.3:a:shopware:shopware:5.1.0:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:juddi:3.1.3`,
        fs:  `cpe:2.3:o:yamaha:rt103i:4.00.38:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:juddi:3.1.4`,
        fs:  `cpe:2.3:o:d-link:dsr-1000n_firmware:1.08b51:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:juddi:3.1.5`,
        fs:  `cpe:2.3:a:htc:evo_4g_software:1.47.651.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:knox:0.2.0`,
        fs:  `cpe:2.3:a:google:chrome:25.0.1364.87:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:knox:0.3.0`,
        fs:  `cpe:2.3:a:atlassian:jira:3.13.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:knox:0.4.0`,
        fs:  `cpe:2.3:a:google:chrome:27.0.1453.52:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:knox:0.5.0`,
        fs:  `cpe:2.3:h:juniper:junos_space_ja1500_appliance:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:knox:0.6.0`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.0.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:knox:0.7.0`,
        fs:  `cpe:2.3:a:rocomotion:pplog_2:3.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:knox:0.8.0`,
        fs:  `cpe:2.3:a:gnupg:gnupg:1.9.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:knox:0.9.0`,
        fs:  `cpe:2.3:a:ibm:traveler:9.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:knox:0.10.0`,
        fs:  `cpe:2.3:a:ratbox:ircd-ratbox:2.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:knox:0.11.0`,
        fs:  `cpe:2.3:a:pyyaml:libyaml:0.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:libcloud:0.2.0`,
        fs:  `cpe:2.3:a:micropact:icomplaints:8.0.2.1.8.8014:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:libcloud:0.3.0`,
        fs:  `cpe:2.3:a:petition_project:petition:6.x-1.0:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:libcloud:0.3.1`,
        fs:  `cpe:2.3:o:cisco:ios_xe:3.2.0sg:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:libcloud:0.4.0`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.3.4-5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:libcloud:0.4.2`,
        fs:  `cpe:2.3:a:adobe:robohelp:8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:libcloud:0.5.0`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.2.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:libcloud:0.5.2`,
        fs:  `cpe:2.3:a:claroline:claroline:1.11.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:libcloud:0.6.0:beta1`,
        fs:  `cpe:2.3:a:telaxus:epesi:1.8.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:libcloud:0.6.1`,
        fs:  `cpe:2.3:a:altiris:language_pack_for_patch_management_core_linux:6.2.1022:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:libcloud:0.6.2`,
        fs:  `cpe:2.3:a:lxml:lxml:2.2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:libcloud:0.7.0`,
        fs:  `cpe:2.3:h:supermicro:x9drw-itpf\&#43;:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:libcloud:0.7.1`,
        fs:  `cpe:2.3:a:google:chrome:19.0.1084.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:libcloud:0.8.0`,
        fs:  `cpe:2.3:a:gnu:grub2:2.01:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:libcloud:0.9.1`,
        fs:  `cpe:2.3:a:dokuwiki:dokuwiki:release_2005-02-18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:libcloud:0.10.1`,
        fs:  `cpe:2.3:a:apache:solr:5.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:libcloud:0.11.0`,
        fs:  `cpe:2.3:a:joedolson:my-calendar:2.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:libcloud:0.11.1`,
        fs:  `cpe:2.3:a:mozilla:firefox_esr:45.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:libcloud:0.11.2`,
        fs:  `cpe:2.3:a:gpsd_project:gpsd:3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:libcloud:0.11.3`,
        fs:  `cpe:2.3:a:ibm:sterling_selling_and_fulfillment_foundation:9.1.0.42:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:libcloud:0.12.3`,
        fs:  `cpe:2.3:a:redhat:icedtea-web:1.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:libcloud:0.12.4`,
        fs:  `cpe:2.3:a:nullsoft:winamp:5.54:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:libcloud:0.13.0`,
        fs:  `cpe:2.3:a:apache:cordova_file_transfer:1.2.1:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:libcloud:0.13.1`,
        fs:  `cpe:2.3:a:ibm:storwize_v3500_software:7.1.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:libcloud:0.13.2`,
        fs:  `cpe:2.3:a:freedomscientific:talking_installer:8.0.2173.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:log4j:2.0:alpha1`,
        fs:  `cpe:2.3:a:altiris:patch_management_for_windows_dell_servers:6.2.3647:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:log4j:2.0:alpha2`,
        fs:  `cpe:2.3:a:deliciousdays:cforms:6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:log4j:2.0:beta1`,
        fs:  `cpe:2.3:a:hp:diagnostics_server:9.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:log4j:2.0:beta2`,
        fs:  `cpe:2.3:a:impresscms:impresscms:1.2.1:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:log4j:2.0:beta3`,
        fs:  `cpe:2.3:a:cyrus:imap:2.3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:log4j:2.0:beta4`,
        fs:  `cpe:2.3:a:cisco:meeting_server:2.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:log4j:2.0:beta5`,
        fs:  `cpe:2.3:a:mozilla:bugzilla:2.16:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:log4j:2.0:beta6`,
        fs:  `cpe:2.3:o:suse:suse_linux_enterprise_desktop:12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:log4j:2.0:beta7`,
        fs:  `cpe:2.3:a:dell:system_manager:1.7.10000:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:log4j:2.0:beta8`,
        fs:  `cpe:2.3:a:iss:blackice_defender:2.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:log4j:2.0:beta9`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.16.39:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:log4j:2.0:rc1`,
        fs:  `cpe:2.3:a:avaya:call_management_system_server:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:log4j:2.0:rc2`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.8.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:log4j:2.0.1`,
        fs:  `cpe:2.3:a:ibm:marketing_platform:9.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:log4j:2.0.2`,
        fs:  `cpe:2.3:a:ge:intelligent_platforms_proficy_plant_applications:4.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:log4j:2.1`,
        fs:  `cpe:2.3:a:google:chrome:25.0.1364.38:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:log4j:2.2`,
        fs:  `cpe:2.3:a:biscom:faxcom_cover_page_editor:9.6.700:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:log4j:2.3`,
        fs:  `cpe:2.3:a:ruby-i18n:i18n:0.6.6:-:-:*:-:ruby:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:log4j:2.4`,
        fs:  `cpe:2.3:a:jenkins-ci:subversion-plugin:1.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:log4j:2.4.1`,
        fs:  `cpe:2.3:a:oracle:collaboration_suite:10.1.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:log4j:2.5`,
        fs:  `cpe:2.3:h:sophos:unified_threat_management:110:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:log4j:2.6`,
        fs:  `cpe:2.3:a:apache:roller:5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:log4j:2.6.1`,
        fs:  `cpe:2.3:a:redhat:stronghold:2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:log4j:2.6.2`,
        fs:  `cpe:2.3:a:liferay:liferay_portal:2.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:log4j:2.7`,
        fs:  `cpe:2.3:a:heimdal_project:heimdal:0.4c:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:log4j:2.8`,
        fs:  `cpe:2.3:a:pocketmags:model_laboratory:\@7f080193:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:log4j:2.8.1`,
        fs:  `cpe:2.3:a:maynard_johnson:oprofile:0.9.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:log4net:-`,
        fs:  `cpe:2.3:a:igor_sysoev:nginx:0.1.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:log4net:1.2.9_beta`,
        fs:  `cpe:2.3:a:hp:destinations:60.0.155.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:maven:3.0`,
        fs:  `cpe:2.3:o:centos:centos:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:maven:3.0.1`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.32.55:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:maven:3.0.2`,
        fs:  `cpe:2.3:a:isc:bind:9.9.8:p3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:maven:3.0.3`,
        fs:  `cpe:2.3:a:hp:serviceguard:a.11.15.00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:maven:3.0.4`,
        fs:  `cpe:2.3:a:python:python:3.1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:maven:3.0.5`,
        fs:  `cpe:2.3:a:stephen_adkins:app\:\:context:0.9661:*:*:*:*:perl:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:maven_wagon:2.1`,
        fs:  `cpe:2.3:o:cisco:catos:2.1\(6\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_auth_radius:-`,
        fs:  `cpe:2.3:a:hp:xp_p9000_tiered_storage_manager:8.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_auth_radius:1.5.4`,
        fs:  `cpe:2.3:a:isc:bind:9.3.3:rc3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_dav_svn:-`,
        fs:  `cpe:2.3:h:hp:cm8050_color_mfp:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_dontdothat:-`,
        fs:  `cpe:2.3:o:yamaha:rt140e:6.03.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_fcgid:2.3.1`,
        fs:  `cpe:2.3:a:oswald_buddenhagen:isync:0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_fcgid:2.3.2`,
        fs:  `cpe:2.3:h:hp:procurve_switch:jg221a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_fcgid:2.3.3`,
        fs:  `cpe:2.3:a:ikiwiki:ikiwiki:3.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_fcgid:2.3.4`,
        fs:  `cpe:2.3:a:phpcheckz:phpcheckz:0.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_fcgid:2.3.5`,
        fs:  `cpe:2.3:a:sixapart:movable_type:4.2:rc5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_fcgid:2.3.6`,
        fs:  `cpe:2.3:a:vips:vips:7.11.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_fcgid:2.3.7`,
        fs:  `cpe:2.3:a:jce-tech:video_niche_script:4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_fcgid:2.3.8`,
        fs:  `cpe:2.3:a:xmlsoft:libxslt:0.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_fcgid:2.3.9`,
        fs:  `cpe:2.3:a:hp:release_control:9.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_imap:-`,
        fs:  `cpe:2.3:a:lansweeper:lansweeper:5.3.0.31:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:-`,
        fs:  `cpe:2.3:a:nginx:nginx:1.4.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:0.93`,
        fs:  `cpe:2.3:a:ibm:sterling_selling_and_fulfillment_foundation:9.1.0.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:1.21`,
        fs:  `cpe:2.3:a:atlassian:crowd:2.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:1.22`,
        fs:  `cpe:2.3:a:htc:hero_software:2.31.651.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:1.23`,
        fs:  `cpe:2.3:a:tiki:tikiwiki_cms\/groupware:1.9.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:1.24`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.27.24:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:1.24_01`,
        fs:  `cpe:2.3:a:google:chrome:24.0.1296.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:1.25`,
        fs:  `cpe:2.3:o:apple:mac_os_x:10.4.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:1.26`,
        fs:  `cpe:2.3:a:oracle:mysql:5.6.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:1.27`,
        fs:  `cpe:2.3:a:public_knowledge_project:open_conference_systems:1.1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:1.28`,
        fs:  `cpe:2.3:o:slackware:slackware_linux:12.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:1.29`,
        fs:  `cpe:2.3:a:tweet-blender:tweet-blender:3.3.9:-:-:*:-:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:1.30`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.0:test9:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:1.99`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.0.90:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:1.99:08`,
        fs:  `cpe:2.3:a:jenkins:role-based_authorization_strategy:2.3.2:*:*:*:*:jenkins:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:1.99:09`,
        fs:  `cpe:2.3:a:rimarts:becky\!_internet_mail:2.51.06:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:1.99:10`,
        fs:  `cpe:2.3:a:adobe:acrobat_dc:15.006.30244:*:*:*:classic:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:1.99:11`,
        fs:  `cpe:2.3:a:cisco:adaptive_security_device_manager_idm_launcher:1.5.47:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:1.99:12`,
        fs:  `cpe:2.3:a:phplist:phplist:3.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:1.99:13`,
        fs:  `cpe:2.3:h:hp:deskjet_3420_color_inkjet:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:1.99:14`,
        fs:  `cpe:2.3:a:rocomotion:pplog:2.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:1.99:16`,
        fs:  `cpe:2.3:o:google:chrome_os:25.0.1364.161:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:1.99:17`,
        fs:  `cpe:2.3:a:seasarfoundation:s2struts:1.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:1.99:18`,
        fs:  `cpe:2.3:a:reviewboard:review_board:1.6.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:1.999`,
        fs:  `cpe:2.3:a:debian:apt:0.8.15:exp2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:1.999:20`,
        fs:  `cpe:2.3:a:merethis:centreon:2.3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:1.999:21`,
        fs:  `cpe:2.3:a:openstack:swift:1.7.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:1.999:22`,
        fs:  `cpe:2.3:a:google:chrome:9.0.597.72:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:2.0.0`,
        fs:  `cpe:2.3:a:ibm:java:6.1.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:2.0.1`,
        fs:  `cpe:2.3:h:hp:8\/20q_fibre_channel_switch_16_port:ak242b:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:2.0.2`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.409.1:*:*:*:lts:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_perl:2.0.3`,
        fs:  `cpe:2.3:a:digium:asterisk:1.4.19:rc4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:-`,
        fs:  `cpe:2.3:h:nvidia:quadro_m3000m:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:1.9a`,
        fs:  `cpe:2.3:o:google:chrome_os:26.0.1410.31:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:2.0`,
        fs:  `cpe:2.3:o:google:chrome_os:25.0.1364.114:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:2.1`,
        fs:  `cpe:2.3:a:adobe:flash_player:18.0.0.333:*:*:*:esr:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:2.2`,
        fs:  `cpe:2.3:a:citrix:presentation_server_for_windows:4.5.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:2.3`,
        fs:  `cpe:2.3:o:moxa:edr_g903_firmware:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:2.4`,
        fs:  `cpe:2.3:a:octobercms:october_cms:1.0.51:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:2.4.1`,
        fs:  `cpe:2.3:a:playscape:pets_fun_house:1.0.1:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:2.5`,
        fs:  `cpe:2.3:a:ibm:websphere_portal:8.0.0.1:cf12:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:2.6`,
        fs:  `cpe:2.3:a:dokuwiki:dokuwiki:release_2004-07-07:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:2.6.1`,
        fs:  `cpe:2.3:a:apache:tomcat:8.0.25:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:2.6.2`,
        fs:  `cpe:2.3:a:cisco:unified_communications_manager:8.5\(1\)su3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:2.6.3`,
        fs:  `cpe:2.3:a:fourkitchens:block_class:5.x-1.0:rc:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:2.6.4`,
        fs:  `cpe:2.3:a:openswan:openswan:2.6.31:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:2.7`,
        fs:  `cpe:2.3:h:hp:t620:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:2.7.1`,
        fs:  `cpe:2.3:a:hitachi:cobol2002_net_client_suite:01-03-\/f:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:2.7.2`,
        fs:  `cpe:2.3:o:ibm:xiv_storage_system_gen3_firmware:11.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:2.7.3`,
        fs:  `cpe:2.3:a:mozilla:seamonkey:2.10:beta1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:2.7.4`,
        fs:  `cpe:2.3:a:horde:horde_application_framework:2.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:2.7.5`,
        fs:  `cpe:2.3:o:nec:up-ux_v:4.2mp:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:2.7.6`,
        fs:  `cpe:2.3:a:recurly:recurly_client_ruby:2.1.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:2.7.7`,
        fs:  `cpe:2.3:a:atutor:atutor:1.5.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:2.7.8`,
        fs:  `cpe:2.3:a:heaventools:pe_explorer:1.99.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:2.7.9`,
        fs:  `cpe:2.3:a:google:toolbar:1.1.48:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:3.0`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.14.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:3.0.1`,
        fs:  `cpe:2.3:a:eclipse:eclipse_ide:4.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:3.0.2`,
        fs:  `cpe:2.3:a:chatelao:php_address_book:3.3.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:3.0.3`,
        fs:  `cpe:2.3:a:memcache_project:memcache:6.x-1.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:3.1.3`,
        fs:  `cpe:2.3:a:cisco:unified_computing_system_central_software:1.4\(1a\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:3.1.4`,
        fs:  `cpe:2.3:o:cisco:catos:5.5\(11\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:3.2.7`,
        fs:  `cpe:2.3:a:isc:dhcp:4.1-esv:r5_rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:3.2.8`,
        fs:  `cpe:2.3:a:apache:hbase:0.94.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:3.2.10`,
        fs:  `cpe:2.3:a:topdesk:topdesk:6.06.006:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:mod_python:3.3.1`,
        fs:  `cpe:2.3:a:cisco:network_admission_control_manager_and_server_system_software:4.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.0.0`,
        fs:  `cpe:2.3:a:photogallerycreator:flash-album-gallery:0.40:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.0.1`,
        fs:  `cpe:2.3:a:nagios:plugins:1.4.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.0.2`,
        fs:  `cpe:2.3:a:mislav_marohnic:will_paginate:3.0.1:-:-:*:-:ruby:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.0.3`,
        fs:  `cpe:2.3:a:google:chrome:6.0.450.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.0.4`,
        fs:  `cpe:2.3:a:marked_project:marked:0.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.0.5`,
        fs:  `cpe:2.3:a:check_mk_project:check_mk:1.2.5:i4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.0.6`,
        fs:  `cpe:2.3:h:apple:macbook_pro:mc373xx\/a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.0.7`,
        fs:  `cpe:2.3:a:bmc:remedy_action_request_system_administrator:6.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.0.8`,
        fs:  `cpe:2.3:a:mw6tech:aztec_activex_control:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.0.9`,
        fs:  `cpe:2.3:a:netjapan:tsushima_travel_guide:1.9:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.0.10`,
        fs:  `cpe:2.3:a:digium:asterisk:1.2.28.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.0.11`,
        fs:  `cpe:2.3:a:google:chrome:28.0.1500.36:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.0.12`,
        fs:  `cpe:2.3:a:andy_armstrong:cgi.pm:3.30:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.0.13`,
        fs:  `cpe:2.3:h:huawei:5710ei:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.1.0`,
        fs:  `cpe:2.3:a:netiq:security_manager_reporting:6.0.0.194:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.1.1`,
        fs:  `cpe:2.3:a:squid-cache:squid:3.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.1.2`,
        fs:  `cpe:2.3:a:tcpdump:tcpdump:4.7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.1.3`,
        fs:  `cpe:2.3:a:seeds:acmailer:3.8.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.1.4`,
        fs:  `cpe:2.3:o:hp:integrated_lights-out_firmware:1.6a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.1.5`,
        fs:  `cpe:2.3:a:mozilla:firefox_esr:38.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.1.6`,
        fs:  `cpe:2.3:a:symantec:mail_security:5.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.1.7`,
        fs:  `cpe:2.3:a:cisco:firesight_system_software:5.2.0.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.1.8`,
        fs:  `cpe:2.3:a:oracle:mysql:5.5.19:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.2.1`,
        fs:  `cpe:2.3:a:x.org:xorg-server:1.8.1.901:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.2.2`,
        fs:  `cpe:2.3:a:isc:bind:9.2.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.2.3`,
        fs:  `cpe:2.3:a:jenkins:datadog:0.2.0:*:*:*:*:jenkins:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.2.4`,
        fs:  `cpe:2.3:a:ibm:db2:9.7.0.4:*:*:*:workgroup:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.2.5`,
        fs:  `cpe:2.3:a:isc:bind:9.6-esv:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.2.6`,
        fs:  `cpe:2.3:a:nuance:paperport:11.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.2.7`,
        fs:  `cpe:2.3:a:exim:exim:4.40:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.2.8`,
        fs:  `cpe:2.3:a:octobercms:october_cms:1.0.412:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.2.9`,
        fs:  `cpe:2.3:a:zohocorp:manageengine_assetexplorer:5.6:5003:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.2.10`,
        fs:  `cpe:2.3:a:rimarts:becky\!_internet_mail:2.64.02:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.2.11`,
        fs:  `cpe:2.3:o:iij:seil_x86_fuji_firmware:2.29:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.2.12`,
        fs:  `cpe:2.3:a:hp:snmp_agents_for_linux:8.2.5-50.sles10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.2.13`,
        fs:  `cpe:2.3:o:vmware:esx:3.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:1.2.14`,
        fs:  `cpe:2.3:a:trillian:trillian:0.71:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.0.0`,
        fs:  `cpe:2.3:a:qemu:qemu:2.1.0:rc5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.0.1`,
        fs:  `cpe:2.3:a:altiris:securityexpressions:3.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.0.2`,
        fs:  `cpe:2.3:a:ponsoftware:explzh:3.16b:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.0.3`,
        fs:  `cpe:2.3:a:rubygems:rubygems:1.8.19:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.0.4`,
        fs:  `cpe:2.3:a:e107:e107:0.7.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.0.5`,
        fs:  `cpe:2.3:a:igor_sysoev:nginx:1.1.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.0.6`,
        fs:  `cpe:2.3:a:oleg_kovalchuk:cctags:6.x-1.x:dev:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.0.7`,
        fs:  `cpe:2.3:a:joomla:joomla\!:1.6:beta7:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.0.8`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.2.23:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.0.9`,
        fs:  `cpe:2.3:a:novell:imanager:2.7:sp4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.0.10`,
        fs:  `cpe:2.3:a:hp:insight_diagnostics_online_edition_for_windows:7.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.0.11`,
        fs:  `cpe:2.3:a:mailpoet:mailpoet_newsletters:2.5.2:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.0.12`,
        fs:  `cpe:2.3:h:motorola:razr_m:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.0.13`,
        fs:  `cpe:2.3:a:sun:one_application_server:6.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.0.14`,
        fs:  `cpe:2.3:a:hybridauth_social_login_project:hybridauth_social_login:7.x-2.10:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.0.15`,
        fs:  `cpe:2.3:a:ibm:websphere_mq:7.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.0.16`,
        fs:  `cpe:2.3:a:postfix:postfix:2.3.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.0.17`,
        fs:  `cpe:2.3:a:fail2ban:fail2ban:0.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.0.18`,
        fs:  `cpe:2.3:a:unisys:enterprise_output_manager_secure_e-mail:1.0.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.0.19`,
        fs:  `cpe:2.3:h:hp:3com_router:jg005a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.0.20`,
        fs:  `cpe:2.3:a:march-hare:cvs_suite:2.5.03:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.0.21`,
        fs:  `cpe:2.3:a:dolibarr:dolibarr:3.5.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.1.0`,
        fs:  `cpe:2.3:a:bit51:better-wp-security:0.14:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.1.1`,
        fs:  `cpe:2.3:o:compaq:tru64:5.1_pk5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.1.2`,
        fs:  `cpe:2.3:a:ffmpeg:ffmpeg:0.4.9_pre1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.1.3`,
        fs:  `cpe:2.3:a:openwebanalytics:open_web_analytics:1.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.1.4`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.451:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.1.5`,
        fs:  `cpe:2.3:a:download_manager_project:download_manager:2.4.7:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.1.6`,
        fs:  `cpe:2.3:a:ibm:tririga_application_platform:3.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.1.7`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.36.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.1.8`,
        fs:  `cpe:2.3:a:tibco:activematrix_management_agent:1.1.0:*:*:*:*:windows_communication_foundation:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.1.9`,
        fs:  `cpe:2.3:a:vips:vips:7.10.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.1.10`,
        fs:  `cpe:2.3:a:gitlab:gitlab:6.7.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.1.11`,
        fs:  `cpe:2.3:a:nvidia:install_application:2.275.82.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.1.12`,
        fs:  `cpe:2.3:h:lexmark:x546:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.1.13`,
        fs:  `cpe:2.3:a:openwebanalytics:open_web_analytics:1.2.0:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.1.14`,
        fs:  `cpe:2.3:a:google:chrome:5.0.375.66:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces:2.1.15`,
        fs:  `cpe:2.3:a:robo-ftp:robo-ftp:1.50.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces_tomahawk:-`,
        fs:  `cpe:2.3:a:roderick_baier:weberknecht:0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:myfaces_tomahawk:1.1.5`,
        fs:  `cpe:2.3:a:sage-mozdev:sage:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:nutch:0.8.1`,
        fs:  `cpe:2.3:a:cisco:adaptive_security_appliance_software:8.3\(1\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:nutch:0.9`,
        fs:  `cpe:2.3:a:nodejs:node.js:0.11.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ofbiz:11.04`,
        fs:  `cpe:2.3:a:adobe:svg_viewer:3.01:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ofbiz:11.04.01`,
        fs:  `cpe:2.3:a:textpattern:textpattern:4.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ofbiz:11.04.02`,
        fs:  `cpe:2.3:a:nancy_wichmann:taxonomy_list:6.x-1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ofbiz:11.04.03`,
        fs:  `cpe:2.3:a:redmine:redmine:0.8.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ofbiz:11.04.04`,
        fs:  `cpe:2.3:a:cloudbees:jenkins:1.424.2.1:-:enterprise:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ofbiz:11.04.05`,
        fs:  `cpe:2.3:a:ganglia:ganglia-web:2.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ofbiz:11.04.06`,
        fs:  `cpe:2.3:a:mcafee:dlp_agent:9.1.100.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ofbiz:12.04`,
        fs:  `cpe:2.3:a:horizon_quick_content_management_system_project:horizon_quick_content_management_system:3.5.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ofbiz:12.04.01`,
        fs:  `cpe:2.3:a:git-scm:git:1.6.2.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ofbiz:12.04.02`,
        fs:  `cpe:2.3:a:nick_copeland:bristol:0.20.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ofbiz:12.04.03`,
        fs:  `cpe:2.3:a:gnupg:gnupg:0.9.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ofbiz:12.04.04`,
        fs:  `cpe:2.3:a:jrbcs:webform_report:6.x-1.x:dev:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ofbiz:12.04.06`,
        fs:  `cpe:2.3:a:debian:dpkg:1.18.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ofbiz:13.07`,
        fs:  `cpe:2.3:a:scilab:scilab:5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ofbiz:13.07.01`,
        fs:  `cpe:2.3:h:hp:procurve_switch:jg237a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ofbiz:13.07.03`,
        fs:  `cpe:2.3:a:wali:walisms_cn:2.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:open_for_business_project:-`,
        fs:  `cpe:2.3:a:isc:bind:9.6.2b1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:open_for_business_project:09.04`,
        fs:  `cpe:2.3:a:symantec:product_authentication_service_client:4.3.28.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:open_for_business_project:09.04.01`,
        fs:  `cpe:2.3:a:monkey-project:monkey_http_daemon:0.20.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:open_for_business_project:10.04`,
        fs:  `cpe:2.3:a:mortbay:jetty:6.0.0:beta5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:open_for_business_project:10.04.01`,
        fs:  `cpe:2.3:h:cisco:unified_ip_phone_7970g:8.0\(4\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:open_for_business_project:10.04.02`,
        fs:  `cpe:2.3:o:cisco:staros:12.2\(300\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:open_for_business_project:10.04.03`,
        fs:  `cpe:2.3:a:ishii:pukiwikimod:1.6.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:open_for_business_project:10.04.04`,
        fs:  `cpe:2.3:a:skyarc:mtcms:5.24:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:open_for_business_project:10.04.05`,
        fs:  `cpe:2.3:a:libtiff:libtiff:4.0:beta5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:open_for_business_project:11.04.01`,
        fs:  `cpe:2.3:a:cisco:telepresence_server_software:3.0\(2.48\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:open_for_business_project:11.04.02`,
        fs:  `cpe:2.3:h:hp:jc808a:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:open_for_business_project:12.04.01`,
        fs:  `cpe:2.3:a:openvpn:openvpn:1.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:openjpa:1.0.0`,
        fs:  `cpe:2.3:a:freka:yr_verdata:6.x-1.5:rc3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:openjpa:1.0.1`,
        fs:  `cpe:2.3:a:apache:tomcat:5.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:openjpa:1.0.2`,
        fs:  `cpe:2.3:o:cisco:ios:12.1\(4\)xz:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:openjpa:1.0.3`,
        fs:  `cpe:2.3:a:roriis.codeplex.com:rails_for_iis_fast-cgi:1.0.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:openjpa:1.0.4`,
        fs:  `cpe:2.3:a:sixnet:sixview_manager:2.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:openjpa:1.1.0`,
        fs:  `cpe:2.3:a:eucalyptus:eucalyptus:3.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:openjpa:1.2.0`,
        fs:  `cpe:2.3:a:rpm:rpm:4.12.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:openjpa:1.2.1`,
        fs:  `cpe:2.3:a:cisco:application_and_content_networking_system_software:5.0.11.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:openjpa:1.2.2`,
        fs:  `cpe:2.3:a:ibm:rational_focal_point:6.4.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:openjpa:2.0.0`,
        fs:  `cpe:2.3:a:foobla:com_obsuggest:1.5.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:openjpa:2.0.1`,
        fs:  `cpe:2.3:a:phpmyadmin:phpmyadmin:1.2.9.4:b:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:openjpa:2.1.0`,
        fs:  `cpe:2.3:a:mozilla:bonsai:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:openjpa:2.1.1`,
        fs:  `cpe:2.3:a:sybase:adaptive_server_enterprise:11.03.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:openjpa:2.2.0`,
        fs:  `cpe:2.3:a:sun:jre:1.5.0:update24:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:openjpa:2.2.1`,
        fs:  `cpe:2.3:a:jabberd2:jabberd2:2.1.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:openmeetings:3.1.1`,
        fs:  `cpe:2.3:a:oracle:database_server:10.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:opennlp:1.5.0`,
        fs:  `cpe:2.3:a:oracle:siebel_suite:7.8.2.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:opennlp:1.5.1`,
        fs:  `cpe:2.3:a:humhub:humhub:0.10.0:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:opennlp:1.5.2`,
        fs:  `cpe:2.3:a:ibm:security_guardium:9.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:opennlp:1.5.3`,
        fs:  `cpe:2.3:o:iij:seil\/x1_firmware:4.40:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:opennlp:1.6.0`,
        fs:  `cpe:2.3:a:intel:inbusiness_email_station:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:opennlp:1.7.0`,
        fs:  `cpe:2.3:a:cubecart:cubecart:5.2.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:opennlp:1.7.1`,
        fs:  `cpe:2.3:a:cisco:unified_customer_voice_portal:3.6\(10\):es01:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:opennlp:1.7.2`,
        fs:  `cpe:2.3:a:bueltge:adminimize:1.7.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:opennlp:1.8.0`,
        fs:  `cpe:2.3:a:microsoft:word:2010:sp1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:opennlp:1.8.1`,
        fs:  `cpe:2.3:o:linux:linux_kernel:4.4.35:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:openoffice:4.0.0`,
        fs:  `cpe:2.3:a:schedmd:slurm:16.05.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:openoffice:4.0.1`,
        fs:  `cpe:2.3:a:koha:koha:3.20.00:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:openoffice:4.1.0`,
        fs:  `cpe:2.3:a:southrivertech:titan_ftp_server:4.02.248:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:openoffice:4.1.1`,
        fs:  `cpe:2.3:a:bueltge:adminimize:1.7.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:openoffice:4.1.2`,
        fs:  `cpe:2.3:a:google:chrome:5.0.375.96:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:openoffice.org:3.2.0`,
        fs:  `cpe:2.3:a:torproject:tor:0.2.2.28:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:openoffice.org:3.2.1`,
        fs:  `cpe:2.3:a:microsoft:system_center_configuration_manager:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:openoffice.org:3.3.0`,
        fs:  `cpe:2.3:a:ocportal:ocportal:7.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:openoffice.org:3.4:beta`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.7.0-8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:openoffice.org:3.4:beta_1`,
        fs:  `cpe:2.3:a:gnu:gnutls:2.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:openoffice.org:3.4.0`,
        fs:  `cpe:2.3:a:openbsd:openssh:4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:openoffice.org:3.4.1`,
        fs:  `cpe:2.3:a:rocomotion:p_up_board:1.00:*:g:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:openoffice.org:4.0`,
        fs:  `cpe:2.3:h:schneider_electric:struxureware_building_operations_automation_server_as:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:opentaps:-`,
        fs:  `cpe:2.3:a:coppermine-gallery:coppermine_photo_gallery:1.5.1:alpha:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:opentaps:0.9.3`,
        fs:  `cpe:2.3:a:git-scm:git:2.4.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:org.apache.sling.servlets.post:2.1.0`,
        fs:  `cpe:2.3:a:wouter_verhelst:nbd:2.9.23:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:org.apache.sling.servlets.post:2.1.2`,
        fs:  `cpe:2.3:o:cisco:ios:12.1\(5\)xm7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:org.apache.sling.servlets.post:2.2.0`,
        fs:  `cpe:2.3:a:tipsandtricks-hq:wordpress_simple_paypal_shopping_cart:2.3:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:org.apache.sling.servlets.post:2.3.0`,
        fs:  `cpe:2.3:a:avaya:wireless_ap-3:2.5.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:pdfbox:1.8.0`,
        fs:  `cpe:2.3:a:tipsandtricks-hq:wordpress_simple_paypal_shopping_cart:1.4:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:pdfbox:1.8.1`,
        fs:  `cpe:2.3:a:smarty:smarty:2.6.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:pdfbox:1.8.2`,
        fs:  `cpe:2.3:a:fortinet:fortimail:5.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:pdfbox:1.8.3`,
        fs:  `cpe:2.3:a:tejimaya:openpne:3.0.6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:pdfbox:1.8.4`,
        fs:  `cpe:2.3:a:torproject:tor:0.2.1.19:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:pdfbox:1.8.5`,
        fs:  `cpe:2.3:a:dm_computer_solutions:ultraedit:11.00a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:pdfbox:1.8.6`,
        fs:  `cpe:2.3:a:rocomotion:pplog:2.74:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:pdfbox:1.8.7`,
        fs:  `cpe:2.3:a:sixapart:movable_type:5.0:beta1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:pdfbox:1.8.8`,
        fs:  `cpe:2.3:a:nathan_haug:webform:7.x-3.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:pdfbox:1.8.9`,
        fs:  `cpe:2.3:a:nancy_wichmann:announcements:6.x-1.x:dev:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:pdfbox:1.8.10`,
        fs:  `cpe:2.3:a:webform_validation_project:webform_validation:7.x-1.3:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:pdfbox:1.8.11`,
        fs:  `cpe:2.3:h:cisco:secure_pix_firewall:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:pdfbox:2.0`,
        fs:  `cpe:2.3:a:octobercms:october_cms:1.0.339:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:pdfbox:2.0:rc1`,
        fs:  `cpe:2.3:h:sony:playstation_portable:2.50:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:pdfbox:2.0:rc2`,
        fs:  `cpe:2.3:a:skype:skype:0.90.0.5:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:pdfbox:2.0:rc3`,
        fs:  `cpe:2.3:a:connect2id:nimbus_jose\&#43;jwt:4.26.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:0.1`,
        fs:  `cpe:2.3:a:ibm:marketing_operations:9.0.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:0.2`,
        fs:  `cpe:2.3:a:pidgin:pidgin:2.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:0.3`,
        fs:  `cpe:2.3:a:systemtools:hyena:4.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:0.4`,
        fs:  `cpe:2.3:a:ucweb:uc_browser:5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:0.5`,
        fs:  `cpe:2.3:h:hp:procurve_switch:je072a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:0.6`,
        fs:  `cpe:2.3:a:netwitness:netwitness_investigator:8.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:0.7`,
        fs:  `cpe:2.3:o:y-cam:ycw004_firmware:4.30:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:0.10.0`,
        fs:  `cpe:2.3:o:siemens:synco_ozw_web_server_firmware:3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:0.11.0`,
        fs:  `cpe:2.3:o:google:android:2.3.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:0.12.0`,
        fs:  `cpe:2.3:a:mawashimono:nikki:5.0:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:0.13.0`,
        fs:  `cpe:2.3:o:apple:iphone_os:3.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:0.14.0`,
        fs:  `cpe:2.3:a:utorrent:utorrent:1.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:1.0.0`,
        fs:  `cpe:2.3:a:erikwebb:password_policy:7.x-1.0:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:1.0.1`,
        fs:  `cpe:2.3:a:apache:xerces2_java:2.10.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:1.0.2`,
        fs:  `cpe:2.3:a:ckeditor:fckeditor:6.x-2.0:alpha4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:1.1.0`,
        fs:  `cpe:2.3:a:thulasidas:easy-adsense-lite:2.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:1.2.0`,
        fs:  `cpe:2.3:a:libraw:libraw:0.14.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:1.5`,
        fs:  `cpe:2.3:a:autodesk:survey:2006:sp1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:1.5.1`,
        fs:  `cpe:2.3:a:flag_module_project:flag:7.x-3.5:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:1.7:dev`,
        fs:  `cpe:2.3:a:ettercap_project:ettercap:0.6.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:1.8:dev`,
        fs:  `cpe:2.3:a:icu-project:international_components_for_unicode:1.4.1.2:*:*:*:*:c\/c\&#43;\&#43;:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:1.10:dev`,
        fs:  `cpe:2.3:a:oracle:jdk:1.7.0:update_101:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:2.0`,
        fs:  `cpe:2.3:a:ibm:java:6.0.8.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:2.0:pre1`,
        fs:  `cpe:2.3:a:kbd-project:kbd:1.15.1:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:2.0:pre2`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.409.3:*:*:*:lts:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:2.0:pre3`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.12.42:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:2.0:rc1`,
        fs:  `cpe:2.3:a:autodesk:autocad:2007:*:*:en:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:2.0:rc2`,
        fs:  `cpe:2.3:o:cisco:ios:12.0\(3.4\)t:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:2.5`,
        fs:  `cpe:2.3:a:serv-u:serv-u:7.3.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:2.5.1`,
        fs:  `cpe:2.3:a:vanillaforums:vanilla_forums:2.0.17.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.0`,
        fs:  `cpe:2.3:a:apple:iphoto:6.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.0:alpha1`,
        fs:  `cpe:2.3:a:fenrir-inc:sleipnir_mobile:1.7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.0:alpha2`,
        fs:  `cpe:2.3:o:apple:a_ux:2.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.0:alpha3`,
        fs:  `cpe:2.3:a:google:chrome:19.0.1084.37:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.0.1`,
        fs:  `cpe:2.3:a:qualys:qualysguardsuite:7.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.0.2`,
        fs:  `cpe:2.3:a:adobe:flash_player:7.0.61.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.0.2:beta1`,
        fs:  `cpe:2.3:a:git-scm:git:1.6.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.0.2:beta2`,
        fs:  `cpe:2.3:o:microsoft:windows_rt:-:gold:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.1`,
        fs:  `cpe:2.3:a:gnupg:gnupg:1.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.1:beta1`,
        fs:  `cpe:2.3:a:adobe:connect:9.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.1:beta2`,
        fs:  `cpe:2.3:a:gnu:libmicrohttpd:0.9.24:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.2`,
        fs:  `cpe:2.3:a:landing_pages_project:landing_pages_plugin:1.0.5.1:-:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.5`,
        fs:  `cpe:2.3:a:ibm:rational_software_architect_design_manager:4.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.5:beta1`,
        fs:  `cpe:2.3:a:squid-cache:squid:3.2.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.5:beta2`,
        fs:  `cpe:2.3:a:gajim:gajim:0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.5:beta3`,
        fs:  `cpe:2.3:a:cisco:unified_contact_center_enterprise:7.0\(0\):sr2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.5:beta4`,
        fs:  `cpe:2.3:a:microsoft:jscript:5.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.5:beta5`,
        fs:  `cpe:2.3:a:vmware:vcenter_orchestrator:4.0.1.4502:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.5:beta6`,
        fs:  `cpe:2.3:a:mongodb:mongodb:2.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.6`,
        fs:  `cpe:2.3:a:cart66:cart66_lite_plugin:1.0.7:-:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.7`,
        fs:  `cpe:2.3:o:iij:seil\/x1_firmware:3.60:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.7:beta1`,
        fs:  `cpe:2.3:a:octopus:octopus_deploy:3.2.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.7:beta2`,
        fs:  `cpe:2.3:a:ajaxplorer:ajaxplorer:3.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.7:beta3`,
        fs:  `cpe:2.3:a:adobe:camera_raw:3.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.8`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.3.44:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.8:beta1`,
        fs:  `cpe:2.3:o:yamaha:rtx1000:7.01.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.8:beta2`,
        fs:  `cpe:2.3:a:gnu:mailman:2.1.22:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.8:beta3`,
        fs:  `cpe:2.3:a:ftprush:ftprush:1.0.0.621:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.8:beta4`,
        fs:  `cpe:2.3:a:quagga:quagga:0.99.22:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.8:beta5`,
        fs:  `cpe:2.3:h:iij:seil_x2:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.9`,
        fs:  `cpe:2.3:a:tag1consulting:support:6.x-1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.10`,
        fs:  `cpe:2.3:a:hidemaru:editor:8.40:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.10:beta1`,
        fs:  `cpe:2.3:a:welcart:welcart:1.4.14:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.10:beta2`,
        fs:  `cpe:2.3:a:mozilla:thunderbird:24.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.11:beta1`,
        fs:  `cpe:2.3:a:cisco:application_policy_infrastructure_controller:1.0\(1k\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:poi:3.14`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.12.55:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:qpid:0.5`,
        fs:  `cpe:2.3:a:philippe_jounin:tftpd32:3.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:qpid:0.6`,
        fs:  `cpe:2.3:a:google:chrome:8.0.552.328:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:qpid:0.7`,
        fs:  `cpe:2.3:a:vips:vips:7.7.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:qpid:0.8`,
        fs:  `cpe:2.3:a:ibm:business_process_manager:7.5.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:qpid:0.9`,
        fs:  `cpe:2.3:a:ca:desktop_protection_suite:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:qpid:0.10`,
        fs:  `cpe:2.3:a:zingiri:zingiri_web_shop:1.2.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:qpid:0.11`,
        fs:  `cpe:2.3:a:shibboleth:shibboleth-sp:2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:qpid:0.12`,
        fs:  `cpe:2.3:a:mortbay:jetty:4.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:qpid:0.13`,
        fs:  `cpe:2.3:a:xnview:xnview:1.97.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:qpid:0.14`,
        fs:  `cpe:2.3:o:yamaha:rt140p:6.03.08:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:qpid:0.15`,
        fs:  `cpe:2.3:a:oracle:vm_virtualbox:4.3.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:qpid:0.16`,
        fs:  `cpe:2.3:a:bundler:bundler:1.1:rc.3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:qpid:0.17`,
        fs:  `cpe:2.3:a:google:chrome:16.0.912.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:qpid:0.18`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.3:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:qpid:0.19`,
        fs:  `cpe:2.3:a:tapatalk:tapatalk:1.1.1:*:*:*:*:wotlab_burning_board:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:qpid:0.20`,
        fs:  `cpe:2.3:a:adobe:creative_suite:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:qpid:0.22`,
        fs:  `cpe:2.3:a:rpm:rpm:1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:qpid:0.30`,
        fs:  `cpe:2.3:a:sap:netweaver_development_infrastructure:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:qpid_java:6.0.2`,
        fs:  `cpe:2.3:h:juniper:router_m20:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ranger:0.4.0`,
        fs:  `cpe:2.3:a:moinejf:abcm2ps:1.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ranger:0.5.0`,
        fs:  `cpe:2.3:a:zend:zend_framework:2.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ranger:0.5.1`,
        fs:  `cpe:2.3:o:cisco:ios:12.2mx:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ranger:0.5.2`,
        fs:  `cpe:2.3:a:tecnick:tcexam:11.3.007:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ranger:0.5.3`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.5.47:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:ranger:0.6.0`,
        fs:  `cpe:2.3:a:atlassian:fisheye:2.4.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:rave:0.11`,
        fs:  `cpe:2.3:a:ups:worldship_msichecker:9.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:rave:0.12`,
        fs:  `cpe:2.3:a:dm_computer_solutions:ultraedit:11.10a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:rave:0.13`,
        fs:  `cpe:2.3:a:mozilla:bugzilla:4.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:rave:0.14`,
        fs:  `cpe:2.3:o:openbsd:openbsd:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:rave:0.15`,
        fs:  `cpe:2.3:a:gecad:axigen_free_mail_server:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:rave:0.16`,
        fs:  `cpe:2.3:a:apple:apple_remote_desktop:3.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:rave:0.17`,
        fs:  `cpe:2.3:a:php:pear:1.4.0a10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:rave:0.18`,
        fs:  `cpe:2.3:a:libupnp_project:libupnp:1.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:rave:0.19`,
        fs:  `cpe:2.3:a:contao:contao_cms:2.2.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:rave:0.20`,
        fs:  `cpe:2.3:o:cisco:ios:15.1\(4\)m12a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:0.9.5`,
        fs:  `cpe:2.3:a:ca:etrust_siteminder_policy_server:12.0.102.414:sp1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:0.9.6`,
        fs:  `cpe:2.3:a:apache:tomcat:9.0.0:m16:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:0.9.6.3`,
        fs:  `cpe:2.3:a:rockwellautomation:rslogix_5000_design_and_configuration_software:20.03:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:0.9.6.4`,
        fs:  `cpe:2.3:a:libguestfs:libguestfs:1.17.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:0.9.7`,
        fs:  `cpe:2.3:a:torproject:tor:0.2.1.5:alpha:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:0.9.7.1`,
        fs:  `cpe:2.3:o:valvesoftware:steam_link_firmware:564:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:0.9.7.2`,
        fs:  `cpe:2.3:a:mysql:mysql:4.0.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:0.9.8`,
        fs:  `cpe:2.3:o:iij:seil_x1_firmware:3.93:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:0.9.8.1`,
        fs:  `cpe:2.3:o:google:chrome_os:0.10.140.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:0.9.8.2`,
        fs:  `cpe:2.3:a:bestpractical:rt:3.8.1:rc3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:0.9.9`,
        fs:  `cpe:2.3:a:ibm:lotus_domino:6.0.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:1.0`,
        fs:  `cpe:2.3:a:isc:bind:9.10.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:1.0:rc1`,
        fs:  `cpe:2.3:a:rack_project:rack:1.3.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:1.0:rc2`,
        fs:  `cpe:2.3:a:joedolson:my-calendar:1.11.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:1.0.1`,
        fs:  `cpe:2.3:a:apache:traffic_server:6.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:1.1`,
        fs:  `cpe:2.3:a:obm:open_business_management:2.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:1.1.1`,
        fs:  `cpe:2.3:a:google:toolbar:5:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:1.1.2`,
        fs:  `cpe:2.3:a:qemu:qemu:2.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:1.2`,
        fs:  `cpe:2.3:a:ganglia:ganglia-web:2.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:1.3`,
        fs:  `cpe:2.3:a:apple:quicktime:7.7.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:2.0`,
        fs:  `cpe:2.3:a:digium:asterisk:c:-:business:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:2.0.1`,
        fs:  `cpe:2.3:a:ibm:java:6.0.14.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:2.0.2`,
        fs:  `cpe:2.3:a:wptrafficanalyzer:trafficanalyzer:1.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:2.1`,
        fs:  `cpe:2.3:a:xchat:xchat:1.8.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:2.1.1`,
        fs:  `cpe:2.3:a:sourcetreesolutions:mojoportal:2.3.8.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:2.3`,
        fs:  `cpe:2.3:a:uiu4you:universal_imaging_utility:3.0.0.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:3.0`,
        fs:  `cpe:2.3:a:cisco:telepresence_mcu_4501_series_software:4.1\(1.51\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:3.1`,
        fs:  `cpe:2.3:a:boehm-demers-weiser:garbage_collector:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:4.0`,
        fs:  `cpe:2.3:a:digium:asterisk:b.2.5.3:-:business:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:4.0.1`,
        fs:  `cpe:2.3:a:ibm:websphere_mq:7.0.1.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:5.0`,
        fs:  `cpe:2.3:a:ibm:emptoris_contract_management:10.0.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:5.0.1`,
        fs:  `cpe:2.3:a:aircrack-ng:aircrack-ng:0.9.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:roller:5.0.2`,
        fs:  `cpe:2.3:a:oracle:reports:6.0.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:santuario_xml_security_for_java:2.0.0`,
        fs:  `cpe:2.3:a:ibm:websphere_mq:7.1.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:santuario_xml_security_for_java:2.0.1`,
        fs:  `cpe:2.3:a:gopivotal:grails:2.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:santuario_xml_security_for_java:2.0.2`,
        fs:  `cpe:2.3:a:google:chrome:37.0.2062.43:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:sentry:1.5.1`,
        fs:  `cpe:2.3:a:igor_vlasenko:html-template-pro:0.63:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:sentry:1.6.0`,
        fs:  `cpe:2.3:a:rsyslog:rsyslog:7.1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:shindig:2.5.0`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.7.7-8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:shiro:1.2.4`,
        fs:  `cpe:2.3:o:iij:seil_x2_firmware:3.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:shiro:1.3.1`,
        fs:  `cpe:2.3:o:dlink:des-3800_firmware:4.50:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:sling:-`,
        fs:  `cpe:2.3:a:pligg:pligg_cms:1.0.0:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:sling_auth_core_component:1.0.2`,
        fs:  `cpe:2.3:a:cloudfoundry:cf-release:264:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:sling_auth_core_component:1.0.4`,
        fs:  `cpe:2.3:o:cisco:catos:6.4\(3\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:sling_auth_core_component:1.0.6`,
        fs:  `cpe:2.3:a:jason_a_donenfeld:cgit:0.9.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:sling_auth_core_component:1.1.0`,
        fs:  `cpe:2.3:a:cybozu:garoon:4.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:sling_auth_core_component:1.1.2`,
        fs:  `cpe:2.3:a:digium:asterisk:1.4.16.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:3.6.0`,
        fs:  `cpe:2.3:a:mozilla:thunderbird_esr:10.0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:3.6.1`,
        fs:  `cpe:2.3:a:qemu:qemu:0.11.0-rc0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:3.6.2`,
        fs:  `cpe:2.3:a:spi\/si263b:message_classification:2.2.0.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:4.0.0`,
        fs:  `cpe:2.3:a:microsoft:office_word_viewer:2003:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:4.0.0:alpha`,
        fs:  `cpe:2.3:a:macromedia:director:5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:4.0.0:beta`,
        fs:  `cpe:2.3:a:openx:openx:2.6.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:4.1.0`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.4.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:4.2.0`,
        fs:  `cpe:2.3:a:primetek:primefaces:6.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:4.2.1`,
        fs:  `cpe:2.3:a:adobe:fireworks_mx:7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:4.3.0`,
        fs:  `cpe:2.3:h:hp:service_manager:9.30:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:4.3.1`,
        fs:  `cpe:2.3:a:gizra:og_vocab:6.x-1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:4.4.0`,
        fs:  `cpe:2.3:a:mahara:mahara:1.5.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:4.5.0`,
        fs:  `cpe:2.3:a:jordy_meow:media_file_renamer:0.54:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:4.5.1`,
        fs:  `cpe:2.3:a:microsoft:data_access_components:2.7:sp1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:4.6.0`,
        fs:  `cpe:2.3:a:mortbay:jetty:4.1.0:rc5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:4.6.1`,
        fs:  `cpe:2.3:a:git_for_windows_project:git_for_windows:1.8.0:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:4.7.0`,
        fs:  `cpe:2.3:a:apache:tomcat:1.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:4.7.1`,
        fs:  `cpe:2.3:a:qemu:qemu:0.12.0:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:4.7.2`,
        fs:  `cpe:2.3:h:huawei:5310ei:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:4.8.0`,
        fs:  `cpe:2.3:a:bestpractical:rt:1.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:4.8.1`,
        fs:  `cpe:2.3:h:apple:imac:m9250xx\/a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:4.9.0`,
        fs:  `cpe:2.3:a:ssh:secure_shell_for_servers:3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:4.9.1`,
        fs:  `cpe:2.3:a:google:chrome:9.0.565.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:4.10.0`,
        fs:  `cpe:2.3:a:sun:jre:1.4.2_19:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:4.10.1`,
        fs:  `cpe:2.3:a:djangoproject:django:1.7.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:4.10.2`,
        fs:  `cpe:2.3:o:netbsd:netbsd:2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:5.0`,
        fs:  `cpe:2.3:a:libupnp_project:libupnp:1.6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:5.1`,
        fs:  `cpe:2.3:a:bradfordnetworks:network_sentry_appliance_software:5.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:5.2.1`,
        fs:  `cpe:2.3:a:debian:shadow-utils:4.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:5.3`,
        fs:  `cpe:2.3:a:bmc:patrol_agent:3.4.00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:5.3.0`,
        fs:  `cpe:2.3:a:isc:bind:9.6.3:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:5.3.1`,
        fs:  `cpe:2.3:a:cisco:peap_module:1.1.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:5.3.2`,
        fs:  `cpe:2.3:a:google:chrome:14.0.835.204:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:5.4.0`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.32.19:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:5.4.1`,
        fs:  `cpe:2.3:a:redhat:application_server:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:5.5.0`,
        fs:  `cpe:2.3:a:cms_tree_page_view_project:cms_tree_page_view:1.2.12:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:5.5.1`,
        fs:  `cpe:2.3:a:mozilla:netscape_portable_runtime:4.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:5.5.2`,
        fs:  `cpe:2.3:a:microsoft:ie:5.50.4134.0600:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:5.5.3`,
        fs:  `cpe:2.3:o:cisco:wireless_lan_controller_software:7.6.100.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:5.5.4`,
        fs:  `cpe:2.3:a:hp:toolbox:120.0.194.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:6.0.0`,
        fs:  `cpe:2.3:o:yamaha:rt105i:6.02.07:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:6.0.1`,
        fs:  `cpe:2.3:a:tigris:tortoisesvn:1.3.6219:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:6.1.0`,
        fs:  `cpe:2.3:a:apache:hadoop:0.23.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:6.2.0`,
        fs:  `cpe:2.3:a:tibco:spotfire_automation_services:6.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:6.2.1`,
        fs:  `cpe:2.3:a:jenkins:owasp_dependency-check:2.0.1:*:*:*:*:jenkins:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:6.3.0`,
        fs:  `cpe:2.3:a:google:chrome:5.0.307.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:6.4.0`,
        fs:  `cpe:2.3:a:adobe:acrobat_reader:11.0.22:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:6.4.1`,
        fs:  `cpe:2.3:a:ibm:b2b_advanced_communications:1.0.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:6.4.2`,
        fs:  `cpe:2.3:a:cisco:secure_desktop:3.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:6.5.0`,
        fs:  `cpe:2.3:a:octobercms:october_cms:1.0.218:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:solr:6.5.1`,
        fs:  `cpe:2.3:a:postgresql:postgresql:9.3.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:spamassassin:-`,
        fs:  `cpe:2.3:a:torproject:tor:0.2.4.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:spamassassin:3.0.1`,
        fs:  `cpe:2.3:a:ibm:tealeaf_consumer_experience:8.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:spamassassin:3.0.2`,
        fs:  `cpe:2.3:a:apache:struts:1.1:b3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:spamassassin:3.0.3`,
        fs:  `cpe:2.3:a:xnview:xnview:1.99:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:spamassassin:3.0.4`,
        fs:  `cpe:2.3:a:rocomotion:pplog:3.28:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:spamassassin:3.1.0`,
        fs:  `cpe:2.3:a:igor_sysoev:nginx:0.6.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:spamassassin:3.1.1`,
        fs:  `cpe:2.3:a:coppermine-gallery:coppermine_photo_gallery:1.2.1:b:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:spamassassin:3.1.2`,
        fs:  `cpe:2.3:a:cacti:cacti:0.8.2a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:spamassassin:3.1.7`,
        fs:  `cpe:2.3:a:intel:proset_wireless:12.4.3000:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:standard_taglibs:1.2.1`,
        fs:  `cpe:2.3:a:amos_benari:rbovirt:0.0.23:*:*:*:*:ruby:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:storm:0.9.0.1`,
        fs:  `cpe:2.3:a:f5:big-ip_global_traffic_manager:10.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:storm:0.10.0:beta`,
        fs:  `cpe:2.3:o:cisco:ios:12.1\(7\)ec:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:storm:1.0`,
        fs:  `cpe:2.3:a:cisco:unified_communications_manager:4.3\(1\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:storm:1.0.1`,
        fs:  `cpe:2.3:a:oscommerce:online_merchant:2.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:storm:1.0.2`,
        fs:  `cpe:2.3:a:phpmyfaq:phpmyfaq:2.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:storm:1.0.3`,
        fs:  `cpe:2.3:a:google:chrome:11.0.696.23:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:storm:1.1`,
        fs:  `cpe:2.3:a:ssh:tectia_client:5.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:-`,
        fs:  `cpe:2.3:a:dotpdn:paint.net:3.5.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:1.0`,
        fs:  `cpe:2.3:a:jan_engelhardt:libhx:3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:1.0.2`,
        fs:  `cpe:2.3:a:google:chrome:32.0.1670.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:1.1`,
        fs:  `cpe:2.3:a:recurly:recurly_client_ruby:2.11.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:1.1:b1`,
        fs:  `cpe:2.3:a:hp:universal_configuration_management_database:10.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:1.1:b2`,
        fs:  `cpe:2.3:a:c-ares_project:c-ares:1.5.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:1.1:b3`,
        fs:  `cpe:2.3:h:linksys:wrt54g:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:1.1:rc1`,
        fs:  `cpe:2.3:a:dalbum:dalbum:1.03:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:1.1:rc2`,
        fs:  `cpe:2.3:a:otrs:otrs:1.0:rc3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:1.2.2`,
        fs:  `cpe:2.3:a:cisco:prime_data_center_network_manager:6.2\(1\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:1.2.4`,
        fs:  `cpe:2.3:o:juniper:junos:6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:1.2.6`,
        fs:  `cpe:2.3:o:linux:linux_kernel:3.9.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:1.2.7`,
        fs:  `cpe:2.3:a:dameware:mini_remote_control:7.5.6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:1.2.8`,
        fs:  `cpe:2.3:a:percona:xtrabackup:2.1.0:beta1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:1.2.9`,
        fs:  `cpe:2.3:a:mozilla:thunderbird:2.0.0.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:1.3.5`,
        fs:  `cpe:2.3:a:ovirt:ovirt:4.1.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:1.3.8`,
        fs:  `cpe:2.3:a:toshibatec:e-studio-2500c_firmware:t380sy0j354:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:1.3.10`,
        fs:  `cpe:2.3:a:apache:solr:4.0.0:alpha:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.0.0`,
        fs:  `cpe:2.3:a:openafs:openafs:1.5.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.0.1`,
        fs:  `cpe:2.3:a:altiris:console_language_pack:6.5.1144:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.0.2`,
        fs:  `cpe:2.3:a:ponsoftware:explzh:3.53:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.0.3`,
        fs:  `cpe:2.3:a:gnu:emacs:21.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.0.4`,
        fs:  `cpe:2.3:a:ibm:tivoli_directory_server:6.3.0.23:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.0.5`,
        fs:  `cpe:2.3:o:sun:sunos:5.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.0.6`,
        fs:  `cpe:2.3:a:webyog:sqlyog:8.0:-:enterprise:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.0.7`,
        fs:  `cpe:2.3:a:ibm:websphere_service_registry_and_repository:7.5.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.0.8`,
        fs:  `cpe:2.3:a:op5:monitor:6.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.0.9`,
        fs:  `cpe:2.3:a:wireshark:wireshark:1.4.10:*:*:*:*:*:itanium:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.0.10`,
        fs:  `cpe:2.3:o:juniper:junos:12.2:r5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.0.11`,
        fs:  `cpe:2.3:a:hp:web_registration:70.0.170.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.0.11.1`,
        fs:  `cpe:2.3:a:mozilla:bugzilla:2.18.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.0.11.2`,
        fs:  `cpe:2.3:o:cisco:ios:11.2sa:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.0.12`,
        fs:  `cpe:2.3:a:wpeasycart:wp_easycart:1.2.11:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.0.13`,
        fs:  `cpe:2.3:a:bowenehs:cih_quiz_game:1.3:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.0.14`,
        fs:  `cpe:2.3:o:yamaha:rt300i:6.00.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.1.0`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:7.0.6-0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.1.1`,
        fs:  `cpe:2.3:a:apple:safari:5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.1.2`,
        fs:  `cpe:2.3:a:ca:messaging:1.11.26:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.1.3`,
        fs:  `cpe:2.3:a:altiris:deployment_web_console:6.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.1.4`,
        fs:  `cpe:2.3:a:cloudfoundry:capi-release:1.14.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.1.5`,
        fs:  `cpe:2.3:a:plumtree:logging_utilities:6.0.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.1.6`,
        fs:  `cpe:2.3:a:apache:axis2:1.5.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.1.8`,
        fs:  `cpe:2.3:a:emc:eroom:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.1.8.1`,
        fs:  `cpe:2.3:a:cyrus:imap:2.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.2.1`,
        fs:  `cpe:2.3:a:oracle:peoplesoft_enterprise:8.22.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.2.1.1`,
        fs:  `cpe:2.3:a:silverstripe:silverstripe:2.3.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.2.3`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.614:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.2.3.1`,
        fs:  `cpe:2.3:a:ibm:rational_policy_tester:5.6.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.1`,
        fs:  `cpe:2.3:a:linuxfoundation:cups-filters:1.11.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.1.1`,
        fs:  `cpe:2.3:a:ibm:marketing_operations:7.4.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.1.2`,
        fs:  `cpe:2.3:a:libav:libav:0.5.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.3`,
        fs:  `cpe:2.3:a:digium:asterisk:1.2.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.4`,
        fs:  `cpe:2.3:a:bestpractical:rt:4.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.4.1`,
        fs:  `cpe:2.3:a:ffmpeg:ffmpeg:0.8.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.5`,
        fs:  `cpe:2.3:a:citrix:license_server_administration:5.0.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.6`,
        fs:  `cpe:2.3:a:kde:kde_sc:4.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.7`,
        fs:  `cpe:2.3:h:sony:snc_dh240:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.8`,
        fs:  `cpe:2.3:a:gitlab:gitlab-shell:1.7.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.9`,
        fs:  `cpe:2.3:a:serv-u:serv-u:6.4.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.10`,
        fs:  `cpe:2.3:a:apache:struts:2.3.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.11`,
        fs:  `cpe:2.3:a:mozilla:thunderbird:1.5.0.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.12`,
        fs:  `cpe:2.3:a:linuxcontainers:cgmanager:0.32:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.13`,
        fs:  `cpe:2.3:h:juniper:srx240:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.14`,
        fs:  `cpe:2.3:a:mozilla:network_security_services:3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.14.1`,
        fs:  `cpe:2.3:a:git-scm:git:1.7.11.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.14.2`,
        fs:  `cpe:2.3:a:ati:control_panel:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.14.3`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.15`,
        fs:  `cpe:2.3:a:mortbay:jetty:3.0.a9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.15.1`,
        fs:  `cpe:2.3:a:businessobjects:crystal_reports_update_service:1.0.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.15.2`,
        fs:  `cpe:2.3:a:ibm:tivoli_directory_server:6.2.0.22:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.15.3`,
        fs:  `cpe:2.3:a:moinejf:abcm2ps:3.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.16`,
        fs:  `cpe:2.3:o:juniper:junos:10.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.16.1`,
        fs:  `cpe:2.3:a:f5:big-ip_access_policy_manager:11.5.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.16.2`,
        fs:  `cpe:2.3:a:apache:axis2:1.5.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.16.3`,
        fs:  `cpe:2.3:a:altiris:application_management_language_pack:6.1.4082:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.17`,
        fs:  `cpe:2.3:o:cisco:ios:12.1ex:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.19`,
        fs:  `cpe:2.3:a:sun:java_web_start:1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.20`,
        fs:  `cpe:2.3:a:git-scm:git:1.7.7.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.20.1`,
        fs:  `cpe:2.3:a:checkpoint:vpn-1_secureclient:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.20.2`,
        fs:  `cpe:2.3:a:ipswitch:imserver:2.0.5.30:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.21`,
        fs:  `cpe:2.3:a:apache:tomcat:7.0.64:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.22`,
        fs:  `cpe:2.3:a:samba:samba:3.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.23`,
        fs:  `cpe:2.3:a:location_module_project:location:6.x-3.x:dev:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.24.2`,
        fs:  `cpe:2.3:a:icu-project:international_components_for_unicode:52.1:*:*:*:*:java:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.24.3`,
        fs:  `cpe:2.3:a:google:chrome:8.0.552.320:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.25`,
        fs:  `cpe:2.3:h:tp-link:tl-sc3171g:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.26`,
        fs:  `cpe:2.3:a:schedmd:slurm:17.02.0:pre2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.27`,
        fs:  `cpe:2.3:a:gnupg:gnupg:1.2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.28`,
        fs:  `cpe:2.3:a:paymentsplus:payments_plus:3.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.28.1`,
        fs:  `cpe:2.3:a:live555:streaming_media:2013.10.01:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.29`,
        fs:  `cpe:2.3:a:ponsoftware:explzh:2.45:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.30`,
        fs:  `cpe:2.3:o:google:chrome_os:25.0.1364.66:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.31`,
        fs:  `cpe:2.3:a:tejimaya:openpne:2.6.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.32`,
        fs:  `cpe:2.3:a:symantec:liveupdate:3.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.3.33`,
        fs:  `cpe:2.3:a:schneider-electric:interactive_graphical_scada_system:10.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5`,
        fs:  `cpe:2.3:a:isc:bind:9.5.3:b1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5:beta1`,
        fs:  `cpe:2.3:o:apple:watchos:2.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5:beta2`,
        fs:  `cpe:2.3:a:typo3:typo3:4.2.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5:beta3`,
        fs:  `cpe:2.3:a:umn:mapserver:5.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.1`,
        fs:  `cpe:2.3:a:spambot_module_project:spambot:6.x-1.x:dev:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.2`,
        fs:  `cpe:2.3:a:torproject:tor:0.1.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.3`,
        fs:  `cpe:2.3:h:cisco:scientific_atlanta_dpx\/epx2203c:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.4`,
        fs:  `cpe:2.3:a:nng:naviextras_toolbox_prerequesities:1.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.5`,
        fs:  `cpe:2.3:a:redmine:redmine:0.8.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.6`,
        fs:  `cpe:2.3:a:dell:system_e-support_tool:1.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.7`,
        fs:  `cpe:2.3:o:yahama:rt107e:8.03.90:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.8`,
        fs:  `cpe:2.3:a:happyworm:jplayer:2.2.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.9`,
        fs:  `cpe:2.3:a:sybase:easerver:5.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.10`,
        fs:  `cpe:2.3:a:transmissionbt:transmission:1.41:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.11`,
        fs:  `cpe:2.3:a:isc:dhcpd:3.0.1:rc7:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.13`,
        fs:  `cpe:2.3:a:autonomy:keyview_export_sdk:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.14`,
        fs:  `cpe:2.3:a:cloudbees:jenkins:1.409:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.15`,
        fs:  `cpe:2.3:a:cisco:secure_access_control_server:2.3\(1\):-:windows:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.16`,
        fs:  `cpe:2.3:a:gnu:glibc:2.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.17`,
        fs:  `cpe:2.3:a:citrix:cloudportal_services_manager:10.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.18`,
        fs:  `cpe:2.3:a:igreks:milkystep_professional:1.56:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.19`,
        fs:  `cpe:2.3:a:atlassian:bamboo:2.0:beta2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.20`,
        fs:  `cpe:2.3:h:cisco:unified_contact_center_express:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.21`,
        fs:  `cpe:2.3:o:supermicro:intelligent_platform_management_firmware:3.15:-:-:*:-:-:x9_generation_motherboards:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.22`,
        fs:  `cpe:2.3:a:cisco:firewall_services_module_software:4.0\(11\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.23`,
        fs:  `cpe:2.3:o:cisco:ios:12.0\(19\)st6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.24`,
        fs:  `cpe:2.3:a:alienvault:open_source_security_information_management:2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.25`,
        fs:  `cpe:2.3:a:oracle:financial_services_asset_liability_management:8.0.1.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.26`,
        fs:  `cpe:2.3:a:ibm:lotus_notes:5.0.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.27`,
        fs:  `cpe:2.3:a:lightscribe:lightscribe:1.6.43.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.28`,
        fs:  `cpe:2.3:a:nodejs:node.js:0.10.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.29`,
        fs:  `cpe:2.3:a:tribiq:tribiq_cms:5.0.9:b:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.30`,
        fs:  `cpe:2.3:a:xiph:icecast:2.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.31`,
        fs:  `cpe:2.3:a:ait-pro:bulletproof-security:.45.9:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.32`,
        fs:  `cpe:2.3:o:codeaurora:android-msm:3.14:rc8:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts:2.5.33`,
        fs:  `cpe:2.3:a:services_twitter_group:services_twitter:0.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts2-showcase:2.0.0`,
        fs:  `cpe:2.3:a:robert_ancell:lightdm:0.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:struts2-showcase:2.3.13`,
        fs:  `cpe:2.3:a:cmu:cyrus-sasl:2.1.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.6`,
        fs:  `cpe:2.3:a:adbnewssender_project:adbnewssender:1.4.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.7`,
        fs:  `cpe:2.3:a:apache:subversion:0.18.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.8`,
        fs:  `cpe:2.3:a:ibm:tivoli_storage_manager:6.1.5.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.9`,
        fs:  `cpe:2.3:o:cisco:skinny_client_control_protocol_software:4.1\(7\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.10.0`,
        fs:  `cpe:2.3:o:mandrakesoft:mandrake_linux_corporate_server:2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.10.1`,
        fs:  `cpe:2.3:h:linksys:befsru31:1.41:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.10.2`,
        fs:  `cpe:2.3:a:htmlcleaner_project:htmlcleaner:2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.11.1`,
        fs:  `cpe:2.3:a:moinejf:abcm2ps:4.7.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.12.0`,
        fs:  `cpe:2.3:h:sierrawireless:airlink_mp_sprint_wifi:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.13.0`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.5:p135:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.13.1`,
        fs:  `cpe:2.3:a:ibm:db2_universal_database:9.1:ga:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.13.2`,
        fs:  `cpe:2.3:a:mediafront:mediafront:6.x-1.0:beta4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.14.0`,
        fs:  `cpe:2.3:a:unisys:enterprise_output_manager_service:7.1.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.14.1`,
        fs:  `cpe:2.3:a:ibm:db2_universal_database:9.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.14.2`,
        fs:  `cpe:2.3:a:quest:installer_logging:3.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.14.3`,
        fs:  `cpe:2.3:a:divx:web_player:1.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.14.4`,
        fs:  `cpe:2.3:a:octopus:octopus_deploy:3.7.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.14.5`,
        fs:  `cpe:2.3:a:nokia:affix:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.15`,
        fs:  `cpe:2.3:a:oracle:mysql:5.5.34:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.16`,
        fs:  `cpe:2.3:a:hp:openview_nnm_advanced_edition_device_support_for_endp:1.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.16.1`,
        fs:  `cpe:2.3:a:yahoo:messenger:5.6.0.1347:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.17.0`,
        fs:  `cpe:2.3:a:boeing:imis_message_system_rabbit:1.1.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.17.1`,
        fs:  `cpe:2.3:a:debian:dpkg:1.4.0.25:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.18.0`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.15.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.18.1`,
        fs:  `cpe:2.3:a:jenkins:config_file_provider:2.15.5:*:*:*:*:jenkins:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.19.0`,
        fs:  `cpe:2.3:a:haxx:libcurl:7.36.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.19.1`,
        fs:  `cpe:2.3:a:redmine:redmine:1.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.20.0`,
        fs:  `cpe:2.3:a:ibm:maximo_asset_management:7.5.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.20.1`,
        fs:  `cpe:2.3:a:mnogosearch:mnogosearch:3.3.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.21.0`,
        fs:  `cpe:2.3:a:isc:dhcp:4.1.1:b3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.22.0`,
        fs:  `cpe:2.3:a:ifdefined:bugtracker.net:3.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.22.1`,
        fs:  `cpe:2.3:a:adobe:photoshop:11.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.22.2`,
        fs:  `cpe:2.3:o:juniper:junos:12.1x46:d35:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.23.0`,
        fs:  `cpe:2.3:a:bea:weblogic_server:6.1:sp2:win32:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.24.0`,
        fs:  `cpe:2.3:a:mozilla:bugzilla:3.6.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.24.1`,
        fs:  `cpe:2.3:a:kajona:kajona:3.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.24.2`,
        fs:  `cpe:2.3:a:eclipse:eclipse_ide:3.6:rc3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.25.0`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.3.2-1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.26.0`,
        fs:  `cpe:2.3:a:lemonldap-ng:lemonldap\:\::1.0:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.27.0`,
        fs:  `cpe:2.3:a:google:chrome:29.0.1547.28:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.28.0`,
        fs:  `cpe:2.3:a:apple:apple_tv:6.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.28.1`,
        fs:  `cpe:2.3:a:videowhisper:live_streaming_integration_plugin:4.25.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.28.2`,
        fs:  `cpe:2.3:h:hp:procurve_switch:je045a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.29.0`,
        fs:  `cpe:2.3:a:chaos_tool_suite_project:ctools:6.x-1.11:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.30.0`,
        fs:  `cpe:2.3:a:opnet:ace_analyst:15.0.0.8631:*:*:*:standard:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.31.0`,
        fs:  `cpe:2.3:a:xchat:xchat:2.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.32.1`,
        fs:  `cpe:2.3:a:kajona:kajona:3.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.33.0`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.5:p186:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.33.1`,
        fs:  `cpe:2.3:a:ati:catalyst_control_center:2.8.225.2152:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.34.0`,
        fs:  `cpe:2.3:h:hp:0235a31q:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.35.0`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(12b\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.35.1`,
        fs:  `cpe:2.3:a:cisco:unified_contact_center_enterprise:10.0\(1\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.36.0`,
        fs:  `cpe:2.3:a:sqltools:sqltools:1.3.5.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:0.37.0`,
        fs:  `cpe:2.3:a:torproject:tor:0.2.0.4:alpha:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.0.0`,
        fs:  `cpe:2.3:a:wesnoth:battle_for_wesnoth:1.11.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.0.1`,
        fs:  `cpe:2.3:a:alex_kellner:powermail:1.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.0.2`,
        fs:  `cpe:2.3:a:sendmail:sendmail:8.14.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.0.3`,
        fs:  `cpe:2.3:a:mit:kerberos:5-1.3:alpha1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.0.4`,
        fs:  `cpe:2.3:a:redhat:gdk_pixbuf:0.18.0-7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.0.5`,
        fs:  `cpe:2.3:a:google:chrome:32.0.1687.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.0.6`,
        fs:  `cpe:2.3:a:google:chrome:33.0.1750.27:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.0.7`,
        fs:  `cpe:2.3:a:emberjs:ember.js:1.12.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.0.8`,
        fs:  `cpe:2.3:a:oracle:sysfw:8.0.3:b:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.0.9`,
        fs:  `cpe:2.3:a:aircrack-ng:aircrack-ng:0.4.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.1.0`,
        fs:  `cpe:2.3:a:hitachi:cosminexus_application_server:06-50-\/e:-:standard:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.1.1`,
        fs:  `cpe:2.3:a:ibm:rational_engineering_lifecycle_manager:4.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.1.2`,
        fs:  `cpe:2.3:a:diziturky:diziturky_hd_2015:2014:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.1.3`,
        fs:  `cpe:2.3:a:cisco:unity_connection:1.1\(1\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.1.4`,
        fs:  `cpe:2.3:a:octobercms:october_cms:1.0.400:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.2.0`,
        fs:  `cpe:2.3:a:gnome:at-spi2-atk:2.7.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.2.1`,
        fs:  `cpe:2.3:a:hp:nonstop_server_software:h06.18.02:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.2.2`,
        fs:  `cpe:2.3:a:novell:bordermanager:3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.2.3`,
        fs:  `cpe:2.3:a:e107:e107:0.7.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.3.0`,
        fs:  `cpe:2.3:a:drupal:drupal:4.7.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.3.1`,
        fs:  `cpe:2.3:a:openafs:openafs:1.6.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.3.2`,
        fs:  `cpe:2.3:a:ibm:storwize_v7000_unified_software:1.4.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.4.0`,
        fs:  `cpe:2.3:a:keepass:keepass:1.03:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.4.1`,
        fs:  `cpe:2.3:o:raritan:px:1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.4.2`,
        fs:  `cpe:2.3:a:vmware:tools:3.1.2.12548:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.4.3`,
        fs:  `cpe:2.3:a:apple:weblog_server:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.4.4`,
        fs:  `cpe:2.3:a:oleggo_livestream_project:oleggo_livestream:0.2.6:-:-:*:-:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.4.5`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.37.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.4.6`,
        fs:  `cpe:2.3:a:webyog:sqlyog:2.5:-:community:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.5.0`,
        fs:  `cpe:2.3:a:gitlab:gitlab:9.0.9:*:*:*:enterprise:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.5.1`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.32.46:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.5.2`,
        fs:  `cpe:2.3:a:novell:imanager:2.7.3:ftf4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.5.3`,
        fs:  `cpe:2.3:a:plugin-organizer_project:plugin-organizer:5.1:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.5.4`,
        fs:  `cpe:2.3:a:andrew_simpson:webcollab:1.42:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.5.5`,
        fs:  `cpe:2.3:a:jeremy_massel:underconstruction:1.12:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.5.6`,
        fs:  `cpe:2.3:a:openbsd:openssh:4.0p1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.5.7`,
        fs:  `cpe:2.3:a:busybox:busybox:0.42:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.5.8`,
        fs:  `cpe:2.3:a:ibm:tivoli_directory_server:6.1.0.37:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.6.0`,
        fs:  `cpe:2.3:a:sgi:performance_co-pilot:2.1.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.6.1`,
        fs:  `cpe:2.3:a:phpmyadmin:phpmyadmin:3.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.6.2`,
        fs:  `cpe:2.3:a:marksdailyapple:mark\&#39;s_daily_apple_forum:2.9.4.3:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.6.3`,
        fs:  `cpe:2.3:a:ibm:marketing_platform:7.5.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.6.4`,
        fs:  `cpe:2.3:a:hp:hpssupply:100.0.172.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.6.5`,
        fs:  `cpe:2.3:a:google:chrome:33.0.1750.56:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.6.6`,
        fs:  `cpe:2.3:a:google:chrome:4.0.249.28:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.6.7`,
        fs:  `cpe:2.3:o:cisco:vpn_3000_concentrator_series_software:4.7.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.6.8`,
        fs:  `cpe:2.3:a:oleg_kovalchuk:cctags:7.x-1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.6.9`,
        fs:  `cpe:2.3:a:compaq:web-enabled_management:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.6.10`,
        fs:  `cpe:2.3:o:cisco:asa_with_firepower_services:5.3.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.6.11`,
        fs:  `cpe:2.3:a:ati:catalyst_control_center_core_static:2010.211.1740.31663:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.6.12`,
        fs:  `cpe:2.3:a:live555:streaming_media:2011.08.22:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.6.13`,
        fs:  `cpe:2.3:a:ibm:tivoli_storage_manager:6.1.5.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.6.14`,
        fs:  `cpe:2.3:a:hp:status:70.0.170.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.6.15`,
        fs:  `cpe:2.3:a:puppetlabs:facter:1.6.15:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.6.16`,
        fs:  `cpe:2.3:a:cisco:trust_agent:2.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.6.17`,
        fs:  `cpe:2.3:a:osticket:osticket:1.9.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.6.18`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.144:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.6.19`,
        fs:  `cpe:2.3:a:oracle:applications_framework:12.2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.6.20`,
        fs:  `cpe:2.3:a:topdesk:topdesk:6.05.008:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.6.21`,
        fs:  `cpe:2.3:a:thecartpress:thecartpress:1.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.6.23`,
        fs:  `cpe:2.3:a:google:chrome:29.0.1547.32:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.7.0`,
        fs:  `cpe:2.3:a:google:chrome:5.0.342.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.7.1`,
        fs:  `cpe:2.3:a:arellia:arellia_file_inventory_language_pack:6.1.1100:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.7.2`,
        fs:  `cpe:2.3:a:littlecms:little_cms_color_engine:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.7.3`,
        fs:  `cpe:2.3:a:digium:certified_asterisk:1.8.4.0:-:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.7.4`,
        fs:  `cpe:2.3:a:bestpractical:rt:3.6.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.7.5`,
        fs:  `cpe:2.3:a:mortbay:jetty:3.0.b02:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.7.6`,
        fs:  `cpe:2.3:a:atlassian:confluence:5.9.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.7.7`,
        fs:  `cpe:2.3:a:webkitgtk:webkitgtk\&#43;:2.8.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.7.8`,
        fs:  `cpe:2.3:h:3com:3crwe454g72:1.0.2.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.7.9`,
        fs:  `cpe:2.3:a:civicrm:civicrm_private_report:6.x-1.1:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.7.10`,
        fs:  `cpe:2.3:a:dell:openmanage:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.7.11`,
        fs:  `cpe:2.3:a:larry_garfield:autosave:6.x-2.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.7.12`,
        fs:  `cpe:2.3:a:cisco:identity_services_engine_software:1.1.2:p8:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.7.13`,
        fs:  `cpe:2.3:a:rocomotion:p_up_board:1.02:*:gbo:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.7.14`,
        fs:  `cpe:2.3:a:cloudera:cdh:4.1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.7.15`,
        fs:  `cpe:2.3:a:clausmuus:spitfire:1.0436:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.7.16`,
        fs:  `cpe:2.3:o:cisco:catos:6.2\(1\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.7.17`,
        fs:  `cpe:2.3:h:bradfordnetworks:network_sentry_appliance:ns8200rx:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.7.18`,
        fs:  `cpe:2.3:o:cisco:ios:12.4jdc:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.7.19`,
        fs:  `cpe:2.3:a:apache:subversion:0.19.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.7.20`,
        fs:  `cpe:2.3:a:huawei:vcn500:v100r002c00spc200:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.8.0`,
        fs:  `cpe:2.3:o:novell:opensuse:12.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.8.1`,
        fs:  `cpe:2.3:a:citadel:ux:6.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.8.2`,
        fs:  `cpe:2.3:a:mutt:mutt:1.5.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.8.3`,
        fs:  `cpe:2.3:a:ushahidi:ushahidi_platform:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.8.4`,
        fs:  `cpe:2.3:a:adjam:rekonq:0.7.92:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.8.5`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.5:p194:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.8.6`,
        fs:  `cpe:2.3:a:google:chrome:15.0.874.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.8.7`,
        fs:  `cpe:2.3:a:pidgin:libpurple:2.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.8.8`,
        fs:  `cpe:2.3:a:gentoo:poppassd_pam:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.8.9`,
        fs:  `cpe:2.3:a:redhat:lvm2:2.01.01:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.8.10`,
        fs:  `cpe:2.3:a:sympa:sympa:6.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.8.11`,
        fs:  `cpe:2.3:a:symantec:antivirus_scan_engine:5.0.0.24:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.8.12`,
        fs:  `cpe:2.3:o:cisco:ios_xe:3.1sg.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.8.13`,
        fs:  `cpe:2.3:a:openwebanalytics:open_web_analytics:1.1.0:rc4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.8.14`,
        fs:  `cpe:2.3:a:apache:struts:2.5.30:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.8.15`,
        fs:  `cpe:2.3:a:apache:tomcat:7.0.79:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.8.18`,
        fs:  `cpe:2.3:o:yamaha:rt102i:3.00.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.9.0`,
        fs:  `cpe:2.3:a:adobe:adobe_air:1.5.0.7220:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.9.1`,
        fs:  `cpe:2.3:a:google:chrome:7.0.507.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.9.2`,
        fs:  `cpe:2.3:a:ibm:elastic_storage_server:2.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.9.3`,
        fs:  `cpe:2.3:a:oracle:supply_chain_products_suite:9.3.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.9.4`,
        fs:  `cpe:2.3:a:dadabik:dadabik:4.3:beta2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.9.5`,
        fs:  `cpe:2.3:a:deliciousdays:cforms:5.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.9.6`,
        fs:  `cpe:2.3:a:cisco:prime_infrastructure:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.10.0`,
        fs:  `cpe:2.3:a:fenrir-inc:sleipnir_mobile:1.5.1:-:black:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.10.0:alpha1`,
        fs:  `cpe:2.3:a:southrivertech:titan_ftp_server:3.30.186:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.10.0:alpha2`,
        fs:  `cpe:2.3:a:dell:sonicwall_scrutinizer_with_flow_analytics_module:8.6.2:-:pay:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:1.10.0:alpha3`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.611:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:m1`,
        fs:  `cpe:2.3:a:redhat:lvm2:2.02.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:m2`,
        fs:  `cpe:2.3:a:mozilla:netscape_portable_runtime:4.9.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:m3`,
        fs:  `cpe:2.3:a:freedomscientific:jaws:11.0.756.400:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:subversion:m4%2fm5`,
        fs:  `cpe:2.3:a:moinejf:abcm2ps:5.9.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:synapse:1.0`,
        fs:  `cpe:2.3:a:mit:kerberos:5_1.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:synapse:1.1`,
        fs:  `cpe:2.3:a:cisco:firewall_services_module_software:3.2\(3\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:synapse:1.1.1`,
        fs:  `cpe:2.3:a:fork-cms:fork_cms:2.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:synapse:1.1.2`,
        fs:  `cpe:2.3:a:roundup-tracker:roundup:0.7.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:synapse:1.2`,
        fs:  `cpe:2.3:a:mortbay:jetty:7.0.0:pre0:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:synapse:2.0.0`,
        fs:  `cpe:2.3:o:yahama:rta50i:3.03.35:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:synapse:2.1.0`,
        fs:  `cpe:2.3:a:oracle:odac_documentation_for_visual_studio_2010:11.2.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:synapse:3.0.0`,
        fs:  `cpe:2.3:a:sun:jdk:1.4.2_11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:syncope:1.0.0`,
        fs:  `cpe:2.3:a:theforeman:foreman:0.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:syncope:1.0.0:rc1`,
        fs:  `cpe:2.3:a:google:chrome:37.0.2062.99:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:syncope:1.0.0:rc2`,
        fs:  `cpe:2.3:a:apache:tomcat:4.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:syncope:1.0.0:rc3`,
        fs:  `cpe:2.3:a:google:chrome:12.0.742.111:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:syncope:1.0.1`,
        fs:  `cpe:2.3:a:openafs:openafs:1.5.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:syncope:1.0.2`,
        fs:  `cpe:2.3:a:mahara:mahara:1.1.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:syncope:1.0.3`,
        fs:  `cpe:2.3:a:oracle:javafx:1.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:syncope:1.0.4`,
        fs:  `cpe:2.3:a:heimdal_project:heimdal:0.2b:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:syncope:1.0.5`,
        fs:  `cpe:2.3:a:cisco:ips_sensor_software:6.2\(1\)e3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:syncope:1.0.6`,
        fs:  `cpe:2.3:a:pivotal_software:cloud_foundry_uaa:2.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:syncope:1.0.7`,
        fs:  `cpe:2.3:a:google:chrome:4.1:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:syncope:1.0.8`,
        fs:  `cpe:2.3:o:cisco:ios:12.1\(5\)xs:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:syncope:1.0.9`,
        fs:  `cpe:2.3:a:adobe:flash_player:10.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:syncope:1.1.0`,
        fs:  `cpe:2.3:a:xmlsoft:libxml2:2.5.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:syncope:1.1.1`,
        fs:  `cpe:2.3:a:ibm:emptoris_sourcing_portfolio:9.5.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:syncope:1.1.2`,
        fs:  `cpe:2.3:o:f5:tmos:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:syncope:1.1.3`,
        fs:  `cpe:2.3:a:adobe:fireworks_mx:7.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:syncope:1.1.4`,
        fs:  `cpe:2.3:a:libevent_project:libevent:1.4.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:syncope:1.1.5`,
        fs:  `cpe:2.3:a:joomla:joomla\!:3.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:syncope:1.1.6`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.7:p25:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:syncope:1.1.7`,
        fs:  `cpe:2.3:a:apple:apple_tv:3.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:syncope:1.1.8`,
        fs:  `cpe:2.3:a:x.org:xorg-server:1.14.99.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tika:1.9`,
        fs:  `cpe:2.3:o:cisco:ios:12.4\(15\)md:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tika:1.13`,
        fs:  `cpe:2.3:a:bbs_technologies:sharepoint_security_manager:3.50.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:-`,
        fs:  `cpe:2.3:a:cisco:secure_access_control_server:4.2.1.15.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:1.1.3`,
        fs:  `cpe:2.3:a:microsoft:sql_server:2012:-:standard:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:3.0`,
        fs:  `cpe:2.3:a:google:chrome:5.0.347.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:3.1`,
        fs:  `cpe:2.3:a:webkitgtk:webkitgtk\&#43;:2.14.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:3.1.1`,
        fs:  `cpe:2.3:a:elgg:elgg:1.8.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:3.2`,
        fs:  `cpe:2.3:a:ssh:tectia_connector:4.3.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:3.2.1`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:5.2.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:3.2.2`,
        fs:  `cpe:2.3:a:mortbay:jetty:6.1.0:rc3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:3.2.2:beta2`,
        fs:  `cpe:2.3:a:microsoft:ie:4.72.2106.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:3.2.3`,
        fs:  `cpe:2.3:a:faronics:deep_freeze:7.30.020.3852:-:std:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:3.2.4`,
        fs:  `cpe:2.3:a:solarwinds:orion_network_performance_monitor:11.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:3.3`,
        fs:  `cpe:2.3:a:sap:enterprise_portal:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:3.3.1`,
        fs:  `cpe:2.3:a:ibm:db2:9.8.0.4:*:*:*:advanced_enterprise:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:3.3.1a`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.8.6-0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:3.3.2`,
        fs:  `cpe:2.3:a:hp:photosmart_essential:2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:4`,
        fs:  `cpe:2.3:a:apple:iphoto:6.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:4.0.0`,
        fs:  `cpe:2.3:a:google:chrome:19.0.1055.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:4.0.1`,
        fs:  `cpe:2.3:a:gitlab:gitlab:5.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:4.0.2`,
        fs:  `cpe:2.3:a:mariadb:mariadb:5.5.33:a:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:4.0.3`,
        fs:  `cpe:2.3:a:octobercms:october_cms:1.0.155:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:4.0.4`,
        fs:  `cpe:2.3:a:zend:zendservice_audioscrobbler:2.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:4.0.5`,
        fs:  `cpe:2.3:a:live555:streaming_media:2012.10.22:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:4.0.6`,
        fs:  `cpe:2.3:a:debian:apt:0.7.15:exp3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:4.1.0`,
        fs:  `cpe:2.3:a:google:chrome:5.0.327.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:4.1.1`,
        fs:  `cpe:2.3:a:google:chrome:5.0.335.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:4.1.2`,
        fs:  `cpe:2.3:a:libav:libav:0.5.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:4.1.3`,
        fs:  `cpe:2.3:a:wireshark:wireshark:1.8.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:4.1.3:beta`,
        fs:  `cpe:2.3:o:cisco:wireless_lan_controller_software:7.2.103.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:4.1.9:beta`,
        fs:  `cpe:2.3:a:adobe:flash_player:9.0.246.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:4.1.10`,
        fs:  `cpe:2.3:a:cisco:wireless_control_system_software:4.2.110.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:4.1.12`,
        fs:  `cpe:2.3:a:clearone:converge_console:5.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:4.1.15`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.24.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:4.1.24`,
        fs:  `cpe:2.3:h:huawei:ar120:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:4.1.28`,
        fs:  `cpe:2.3:a:gnome:glib:2.22.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:4.1.29`,
        fs:  `cpe:2.3:a:juniper:netscreen-security_manager:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:4.1.31`,
        fs:  `cpe:2.3:a:rocomotion:p_board:1.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:4.1.36`,
        fs:  `cpe:2.3:a:avast:endpoint_protection_plus:8.0.1609:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5`,
        fs:  `cpe:2.3:a:hp:onboard_administrator:3.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.0`,
        fs:  `cpe:2.3:a:igor_sysoev:nginx:0.4.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.1`,
        fs:  `cpe:2.3:a:cloudfoundry:cf-release:141:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.2`,
        fs:  `cpe:2.3:a:oracle:collaboration_suite:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.3`,
        fs:  `cpe:2.3:a:google:chrome:17.0.925.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.4`,
        fs:  `cpe:2.3:h:cisco:scientific_atlanta_dpc\/epc3212:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.5`,
        fs:  `cpe:2.3:a:gnu:mailman:2.0.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.6`,
        fs:  `cpe:2.3:a:ibm:tivoli_application_dependency_discovery_manager:7.2.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.7`,
        fs:  `cpe:2.3:a:prestashop:prestashop:0.9.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.8`,
        fs:  `cpe:2.3:o:cisco:ios:12.2xc:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.9`,
        fs:  `cpe:2.3:a:altiris:report_pack_for_task_server:6.0.1007:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.10`,
        fs:  `cpe:2.3:a:jenkins:git:2.2.1:*:*:*:*:jenkins:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.11`,
        fs:  `cpe:2.3:a:ponsoftware:explzh:2.80:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.12`,
        fs:  `cpe:2.3:o:huawei:s5300_firmware:v200r005c00spc500:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.13`,
        fs:  `cpe:2.3:o:cisco:unified_computing_system_infrastructure_and_unified_computing_system_software:2.0\(1x\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.14`,
        fs:  `cpe:2.3:a:symantec:message_gateway_for_service_providers:10.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.15`,
        fs:  `cpe:2.3:a:cisco:web_security_virtual_appliance:8.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.16`,
        fs:  `cpe:2.3:o:google:chrome_os:26.0.1410.46:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.17`,
        fs:  `cpe:2.3:a:vmware:vcenter_converter_standalone:4.3.0.292238:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.18`,
        fs:  `cpe:2.3:a:typo3:typo3:4.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.19`,
        fs:  `cpe:2.3:a:seeds:acmailer:3.9.5:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.21`,
        fs:  `cpe:2.3:a:mozilla:seamonkey:2.20:beta1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.22`,
        fs:  `cpe:2.3:a:s9y:serendipity:1.5.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.23`,
        fs:  `cpe:2.3:a:izarc:izarc:4.1.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.24`,
        fs:  `cpe:2.3:a:ibm:websphere_extended_deployment_compute_grid:8.0.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.25`,
        fs:  `cpe:2.3:o:iij:seil_b1_firmware:4.38:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.26`,
        fs:  `cpe:2.3:a:ibm:websphere_application_server:8.5.5.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.27`,
        fs:  `cpe:2.3:o:cisco:ios:12.1\(22\)eb:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.28`,
        fs:  `cpe:2.3:a:exactcode:exactimage:0.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.29`,
        fs:  `cpe:2.3:a:iss:realsecure_sentry:3.6ecb:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.0.30`,
        fs:  `cpe:2.3:a:gilles_lamiral:imapsync:1.580:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.0`,
        fs:  `cpe:2.3:a:disqus:disqus_comment_system:2.66:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.1`,
        fs:  `cpe:2.3:a:invensys:wonderware_information_server:5.0:-:portal:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.2`,
        fs:  `cpe:2.3:a:attachmate:verastream_host_integrator:4.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.3`,
        fs:  `cpe:2.3:o:linux:linux_kernel:4.10.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.4`,
        fs:  `cpe:2.3:a:zabbix:zabbix:1.8.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.5`,
        fs:  `cpe:2.3:a:gisle_aas:libwww-perl:5.831:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.6`,
        fs:  `cpe:2.3:a:zemanta:related_posts:2.7.1:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.7`,
        fs:  `cpe:2.3:a:owncloud:owncloud:4.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.8`,
        fs:  `cpe:2.3:a:cisco:identity_services_engine_software:1.1.3:p2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.9`,
        fs:  `cpe:2.3:a:sync:oxygen_xml_editor:7.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.10`,
        fs:  `cpe:2.3:a:dhcpcd_project:dhcpcd:5.2.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.11`,
        fs:  `cpe:2.3:a:cloudfoundry:cf-release:258:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.12`,
        fs:  `cpe:2.3:a:gnu:gnutls:3.1.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.13`,
        fs:  `cpe:2.3:a:apache:cocoon:2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.14`,
        fs:  `cpe:2.3:a:securecomputing:sidewinder_g2_admin_console:3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.15`,
        fs:  `cpe:2.3:a:cisco:meeting_server:2.0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.16`,
        fs:  `cpe:2.3:a:phpmyadmin:phpmyadmin:1.3:alpha:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.17`,
        fs:  `cpe:2.3:a:dhcpcd_project:dhcpcd:6.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.18`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.0.22:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.19`,
        fs:  `cpe:2.3:h:hp:procurve_switch:je095a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.20`,
        fs:  `cpe:2.3:a:oracle:weblogic_server:9.1.0.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.21`,
        fs:  `cpe:2.3:a:axel_jung:js_css_optimizer:1.1.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.22`,
        fs:  `cpe:2.3:a:catalin_florian_radut:zeropoint:6.x-1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.23`,
        fs:  `cpe:2.3:a:gnu:gnutls:2.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.24`,
        fs:  `cpe:2.3:o:nuuo:nvrmini_2:1.6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.25`,
        fs:  `cpe:2.3:a:karen_stevenson:date:6.x-2.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.26`,
        fs:  `cpe:2.3:a:altiris:script_task:6.0.1524:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.27`,
        fs:  `cpe:2.3:a:hp:system_management_homepage:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.28`,
        fs:  `cpe:2.3:a:ssh:tectia_server:4.4.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.29`,
        fs:  `cpe:2.3:a:ember-entertainment:towers_n\&#39;_trolls:1.6.4:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.30`,
        fs:  `cpe:2.3:a:rocomotion:p_link:1.05:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.31`,
        fs:  `cpe:2.3:a:securityinnovation:ntru_hybrid_tss:1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.32`,
        fs:  `cpe:2.3:h:cisco:aironet_ap1200:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.33`,
        fs:  `cpe:2.3:o:cisco:staros:18.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.34`,
        fs:  `cpe:2.3:a:mcafee:network_security_manager:8.1.7.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:5.5.35`,
        fs:  `cpe:2.3:a:lockon:ec-cube:2.12.5en:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6`,
        fs:  `cpe:2.3:o:moxa:miineport_e1_4641_firmware:1.1.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0`,
        fs:  `cpe:2.3:a:quest:toad_for_oracle:8.5.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.0`,
        fs:  `cpe:2.3:a:geeklog:geeklog:1.5.0:*:ja:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.0:alpha`,
        fs:  `cpe:2.3:a:tejimaya:openpne:2.6.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.1`,
        fs:  `cpe:2.3:a:squid-cache:squid:3.0.stable9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.1:alpha`,
        fs:  `cpe:2.3:a:isc:bind:8.2.3_t1a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.2`,
        fs:  `cpe:2.3:a:oracle:fusion_middleware:8.3.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.2:alpha`,
        fs:  `cpe:2.3:o:google:chrome_os:8.0.552.343:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.2:beta`,
        fs:  `cpe:2.3:a:google:chrome:14.0.835.97:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.3`,
        fs:  `cpe:2.3:o:oracle:linux:6.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.4`,
        fs:  `cpe:2.3:a:microsoft:internet_information_server:6.0:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.4:alpha`,
        fs:  `cpe:2.3:a:cisco:customer_response_solutions:6.0.1068.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.5`,
        fs:  `cpe:2.3:a:sophos:web_appliance:3.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.6`,
        fs:  `cpe:2.3:a:octobercms:october_cms:1.0.115:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.6:alpha`,
        fs:  `cpe:2.3:a:cisco:adaptive_security_appliance_software:9.5.3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.7`,
        fs:  `cpe:2.3:o:cisco:catos:6.4\(6\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.7:alpha`,
        fs:  `cpe:2.3:a:nortel:meridian_option_61c:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.7:beta`,
        fs:  `cpe:2.3:a:apache:hive:1.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.8`,
        fs:  `cpe:2.3:a:tribiq:tribiq_cms:5.0.10b:-:community:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.8:alpha`,
        fs:  `cpe:2.3:a:proftpd:proftpd:1.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.9`,
        fs:  `cpe:2.3:a:google:chrome:7.0.517.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.9:beta`,
        fs:  `cpe:2.3:a:ibm:tivoli_identity_manager:5.0.0.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.10`,
        fs:  `cpe:2.3:a:webyog:sqlyog:5.18:-:community:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.11`,
        fs:  `cpe:2.3:a:dell:client_configuration_toolkit:1.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.12`,
        fs:  `cpe:2.3:a:openssl:openssl:1.0.2d:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.13`,
        fs:  `cpe:2.3:a:mozilla:bugzilla:3.4.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.14`,
        fs:  `cpe:2.3:a:photocati_media:photocrati:4.07:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.15`,
        fs:  `cpe:2.3:a:libguestfs:libguestfs:1.16.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.16`,
        fs:  `cpe:2.3:a:oracle:vm_virtualbox:4.0.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.17`,
        fs:  `cpe:2.3:a:git-scm:git:1.7.3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.18`,
        fs:  `cpe:2.3:a:xyzscripts:newsletter_manager:1.2.3:-:-:*:-:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.19`,
        fs:  `cpe:2.3:a:perl:perl:5.8.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.20`,
        fs:  `cpe:2.3:a:apache:apr-util:1.2.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.21`,
        fs:  `cpe:2.3:a:apache:cxf:2.6.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.22`,
        fs:  `cpe:2.3:o:microsoft:windows_vista:-:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.23`,
        fs:  `cpe:2.3:a:hp:version_control_repository_manager:1.0.2345.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.24`,
        fs:  `cpe:2.3:a:ibm:informix_dynamic_database_server:9.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.25`,
        fs:  `cpe:2.3:a:wpeasycart:wp_easycart:2.1.19:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.26`,
        fs:  `cpe:2.3:a:plone:plone:2.5.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.27`,
        fs:  `cpe:2.3:a:digium:asterisk:11.0.0:beta1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.28`,
        fs:  `cpe:2.3:a:bradfordnetworks:network_sentry_appliance_software:5.3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.29`,
        fs:  `cpe:2.3:o:blackberry:qnx_neutrino_rtos:6.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.30`,
        fs:  `cpe:2.3:a:apache:poi:3.0:alpha3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.31`,
        fs:  `cpe:2.3:o:yahama:rtx1000:7.01.04:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.32`,
        fs:  `cpe:2.3:a:postfix:postfix:2.0.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.33`,
        fs:  `cpe:2.3:a:digium:asterisk:1.8.11.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.34`,
        fs:  `cpe:2.3:a:oracle:mysql_connector\/net:6.8.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.35`,
        fs:  `cpe:2.3:a:powerarchiver:powerarchiver:14.02.05:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.36`,
        fs:  `cpe:2.3:a:bundler:bundler:1.7.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.37`,
        fs:  `cpe:2.3:a:jenkins:config_file_provider:2.2.1:*:*:*:*:jenkins:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.38`,
        fs:  `cpe:2.3:a:google:chrome:37.0.2062.29:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.40`,
        fs:  `cpe:2.3:a:grapecity:data_dynamics_reports:1.5.905.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.41`,
        fs:  `cpe:2.3:a:bluecoat:content_analysis_system_software:1.1.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.42`,
        fs:  `cpe:2.3:a:gnu:gnutls:2.8.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.43`,
        fs:  `cpe:2.3:a:netsweeper:netsweeper:5.0.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.44`,
        fs:  `cpe:2.3:a:adobe:color_extra_settings:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.45`,
        fs:  `cpe:2.3:a:hp:linux_imaging_and_printing_project:3.9.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.46`,
        fs:  `cpe:2.3:a:pgp:desktop_security:7.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.47`,
        fs:  `cpe:2.3:a:oracle:hyperion_production_reporting_server:11.1.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.48`,
        fs:  `cpe:2.3:a:proftpd:proftpd:1.2.2:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.49`,
        fs:  `cpe:2.3:a:degraaff:checkbot:1.77:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.50`,
        fs:  `cpe:2.3:a:octobercms:october_cms:1.0.122:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.51`,
        fs:  `cpe:2.3:a:narr8:secret_city_-_motion_comic:2.1.7:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:6.0.52`,
        fs:  `cpe:2.3:a:itd-inc:bingo\!cms:1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0`,
        fs:  `cpe:2.3:a:tejimaya:openpne:2.12.0:beta1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.0`,
        fs:  `cpe:2.3:a:cerberusftp:ftp_server:3.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.0:beta`,
        fs:  `cpe:2.3:a:springsource:spring_framework:3.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.1`,
        fs:  `cpe:2.3:a:ibm:tivoli_storage_manager_fastback:6.1.8.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.2`,
        fs:  `cpe:2.3:a:wave:private_information_manager:7.0.0.26:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.2:beta`,
        fs:  `cpe:2.3:a:altiris:deployment_solution:6.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.3`,
        fs:  `cpe:2.3:a:freedesktop:poppler:0.25.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.4`,
        fs:  `cpe:2.3:h:cisco:guard_ddos_mitigation_appliance:5.1\(5\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.4:beta`,
        fs:  `cpe:2.3:a:digium:asterisk:11.23.0:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.5`,
        fs:  `cpe:2.3:o:cisco:catos:5.5\(1a\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.5:beta`,
        fs:  `cpe:2.3:a:hp:mercury_loadrunner_agent:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.6`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.6.1-3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.7`,
        fs:  `cpe:2.3:a:nagios:remote_plug_in_executor:1.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.8`,
        fs:  `cpe:2.3:a:rom_walton:boinc:7.1.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.9`,
        fs:  `cpe:2.3:a:apple:apple_tv:2.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.10`,
        fs:  `cpe:2.3:a:ibm:net.commerce:3.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.11`,
        fs:  `cpe:2.3:a:fileopen:fileopen_client:3.0.67.914:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.12`,
        fs:  `cpe:2.3:a:google:chrome:16.0.912.41:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.13`,
        fs:  `cpe:2.3:a:mortbay:jetty:4.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.14`,
        fs:  `cpe:2.3:a:igor_sysoev:nginx:0.3.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.15`,
        fs:  `cpe:2.3:o:hp:procurve_switch_software:h.08.99:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.16`,
        fs:  `cpe:2.3:a:bestpractical:rtfm:2.4.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.17`,
        fs:  `cpe:2.3:a:apache:myfaces:2.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.18`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.602:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.19`,
        fs:  `cpe:2.3:a:sophos:sophos_anti-virus:6.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.20`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.21`,
        fs:  `cpe:2.3:a:sielcosistemi:winlog_lite:2.07.01:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.22`,
        fs:  `cpe:2.3:a:eterna:bozohttpd:20050410:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.23`,
        fs:  `cpe:2.3:a:tibco:spotfire_server:5.5.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.24`,
        fs:  `cpe:2.3:a:devsaran:fresh:7.x-1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.25`,
        fs:  `cpe:2.3:a:adobe:air:20.0.0.233:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.26`,
        fs:  `cpe:2.3:a:seeddms:seeddms:4.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.27`,
        fs:  `cpe:2.3:o:microsoft:windows_10:-:gold:*:*:*:*:x86:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.28`,
        fs:  `cpe:2.3:o:apple:iphone_os:3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.29`,
        fs:  `cpe:2.3:a:ibm:websphere_application_server:6.0.2.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.30`,
        fs:  `cpe:2.3:h:iss:proventia_a_series_xpu:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.31`,
        fs:  `cpe:2.3:o:yahama:rt56v:4.07.19:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.32`,
        fs:  `cpe:2.3:a:octobercms:october_cms:1.0.154:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.33`,
        fs:  `cpe:2.3:a:joomla:joomla\!:1.0.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.34`,
        fs:  `cpe:2.3:h:cisco:ubr10012:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.35`,
        fs:  `cpe:2.3:a:accomplishtechnology:phpmydirectory:1.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.36`,
        fs:  `cpe:2.3:h:intel:express_510t:2.63:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.37`,
        fs:  `cpe:2.3:h:xerox:document_centre_555:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.38`,
        fs:  `cpe:2.3:a:sun:one_directory_server:4.16:sp1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.39`,
        fs:  `cpe:2.3:a:rockmongo:rockmongo:1.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.40`,
        fs:  `cpe:2.3:a:webkitgtk:webkitgtk\&#43;:2.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.41`,
        fs:  `cpe:2.3:a:hp:vertica:7.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.42`,
        fs:  `cpe:2.3:a:quest:toad_for_oracle:10.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.43`,
        fs:  `cpe:2.3:a:pidgin:libpurple:2.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.44`,
        fs:  `cpe:2.3:a:hp:openview_cross_platform_component:3.10.40:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.45`,
        fs:  `cpe:2.3:a:hp:rapid_deployment_pack:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.46`,
        fs:  `cpe:2.3:a:mm_forum_project:mm_forum:1.9.1:*:*:*:*:typo3:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.47`,
        fs:  `cpe:2.3:a:google:chrome:13.0.782.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.48`,
        fs:  `cpe:2.3:a:divx:player:6.4.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.49`,
        fs:  `cpe:2.3:a:kerio:control:8.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.50`,
        fs:  `cpe:2.3:a:shaarli_project:shaarli:0.0.43:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.51`,
        fs:  `cpe:2.3:a:devexpress:aspxfilemanager_control_for_webforms_and_mvc:11.2.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.54`,
        fs:  `cpe:2.3:a:oxid-esales:eshop:5.1.0:*:enterprise:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.55`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.187:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.56`,
        fs:  `cpe:2.3:o:iij:seil_x1_firmware:3.03:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.57`,
        fs:  `cpe:2.3:o:microsoft:windows_2000:-:beta3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.58`,
        fs:  `cpe:2.3:a:ibm:rational_team_concert:5.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.59`,
        fs:  `cpe:2.3:o:juniper:netscreen_screenos:4.0.0r10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.60`,
        fs:  `cpe:2.3:a:libguestfs:libguestfs:1.15.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.61`,
        fs:  `cpe:2.3:a:videolan:vlc_media_player:2.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.62`,
        fs:  `cpe:2.3:a:isc:bind:9.6.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.63`,
        fs:  `cpe:2.3:o:cisco:nx-os:11.1\(4g\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.64`,
        fs:  `cpe:2.3:a:intel:network_connections:16.6.126.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.65`,
        fs:  `cpe:2.3:a:solarwinds:collector:2.2.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.66`,
        fs:  `cpe:2.3:a:norman:security_suite:10.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.67`,
        fs:  `cpe:2.3:a:google:chrome:11.0.672.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.68`,
        fs:  `cpe:2.3:a:videolan:vlc_media_player:0.2.73:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.69`,
        fs:  `cpe:2.3:a:ibm:u2_universe:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.70`,
        fs:  `cpe:2.3:o:freebsd:freebsd:5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.71`,
        fs:  `cpe:2.3:a:git-scm:git:0.99.9h:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.72`,
        fs:  `cpe:2.3:a:ibm:sametime:8.5.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.73`,
        fs:  `cpe:2.3:a:unisys:sentinel_alert_service:9.2.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.74`,
        fs:  `cpe:2.3:a:unixodbc:unixodbc:1.8.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.75`,
        fs:  `cpe:2.3:a:symantec:norton_security_scan:1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.76`,
        fs:  `cpe:2.3:a:apache:roller:1.0:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.77`,
        fs:  `cpe:2.3:a:ibm:pureapplication_system:1.1.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.79`,
        fs:  `cpe:2.3:h:lexmark:cs820de:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.80`,
        fs:  `cpe:2.3:a:lighttpd:lighttpd:1.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:7.0.81`,
        fs:  `cpe:2.3:o:cisco:ios:12.3\(11\)yl:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0`,
        fs:  `cpe:2.3:a:glub:secure_ftp:2.5.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.0:rc1`,
        fs:  `cpe:2.3:a:mediawiki:mediawiki:1.4.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.0:rc10`,
        fs:  `cpe:2.3:o:novell:opensuse:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.0:rc2`,
        fs:  `cpe:2.3:a:reviewboard:djblets:0.8.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.0:rc5`,
        fs:  `cpe:2.3:a:digium:asterisk:1.6.2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.1`,
        fs:  `cpe:2.3:a:sixapart:movabletype:6.0.7:*:*:*:advanced:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.2`,
        fs:  `cpe:2.3:a:adjam:rekonq:0.6.80:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.4`,
        fs:  `cpe:2.3:a:theforeman:kafo:0.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.6`,
        fs:  `cpe:2.3:h:dell:poweredge_sc1435:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.7`,
        fs:  `cpe:2.3:a:herpin_time_radio_project:herpin_time_radio:2:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.9`,
        fs:  `cpe:2.3:a:perl:perl:5.12.0:rc0:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.10`,
        fs:  `cpe:2.3:a:adobe:acrobat_reader:11.0.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.11`,
        fs:  `cpe:2.3:a:ibm:storwize_v7000_software:6.1.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.12`,
        fs:  `cpe:2.3:o:huawei:s9312_firmware:v100r006:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.13`,
        fs:  `cpe:2.3:a:hp:insight_diagnostics:7.5.4:offline:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.14`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.142:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.15`,
        fs:  `cpe:2.3:a:adbnewssender_project:adbnewssender:1.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.16`,
        fs:  `cpe:2.3:a:checkpoint:security_gateway:r75.47:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.17`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(15\)zj3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.18`,
        fs:  `cpe:2.3:a:gstreamer_project:gstreamer:0.10.25:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.19`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.26:rc5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.20`,
        fs:  `cpe:2.3:a:oracle:mysql:5.7.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.21`,
        fs:  `cpe:2.3:a:sielcosistemi:winlog_pro:2.06.04:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.22`,
        fs:  `cpe:2.3:o:iij:seil_x2_firmware:2.64:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.23`,
        fs:  `cpe:2.3:a:cisco:evolved_programmable_network_manager:1.2.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.24`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.7.5-1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.25`,
        fs:  `cpe:2.3:a:mariadb:mariadb:10.0.22:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.26`,
        fs:  `cpe:2.3:a:hp:cluster_object_manager:b.03.00.00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.27`,
        fs:  `cpe:2.3:a:mailenable:mailenable:1.26:-:enterprise:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.28`,
        fs:  `cpe:2.3:o:iij:seil_x2_firmware:2.78:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.29`,
        fs:  `cpe:2.3:o:linux:linux_kernel:4.9.39:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.30`,
        fs:  `cpe:2.3:a:dalbum:dalbum:1.22:sp5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.31`,
        fs:  `cpe:2.3:a:atlassian:jira:3.13.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.32`,
        fs:  `cpe:2.3:a:ca:itechnology_igateway:4.0.050526:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.33`,
        fs:  `cpe:2.3:a:f5:big-ip_analytics:11.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.34`,
        fs:  `cpe:2.3:a:bestsoftinc:advance_hotel_booking_system:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.35`,
        fs:  `cpe:2.3:h:iss:proventia_server:1.0.914.1880:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.36`,
        fs:  `cpe:2.3:a:mit:kerberos:5_1.2:beta2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.37`,
        fs:  `cpe:2.3:a:zimbra:collaboration_server:8.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.38`,
        fs:  `cpe:2.3:a:tenable:nessus:5.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.39`,
        fs:  `cpe:2.3:a:cisco:application_and_content_networking_system_software:5.5.11.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.40`,
        fs:  `cpe:2.3:h:rockwellautomation:1769_compactlogix_5370_l2_controller:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.41`,
        fs:  `cpe:2.3:a:ponsoftware:explzh:2.78:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.42`,
        fs:  `cpe:2.3:a:macromedia:extension_manager:1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.0.43`,
        fs:  `cpe:2.3:a:git-scm:git:0.8.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.5.0`,
        fs:  `cpe:2.3:a:gnome:gnome-shell:3.9.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.5.1`,
        fs:  `cpe:2.3:a:mahara:mahara:1.0.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.5.2`,
        fs:  `cpe:2.3:a:f5:big-ip_global_traffic_manager:10.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.5.3`,
        fs:  `cpe:2.3:a:vmware:vcenter_server_appliance:5.0:update_1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.5.4`,
        fs:  `cpe:2.3:o:cisco:ios:11.2\(15a\)p:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.5.5`,
        fs:  `cpe:2.3:a:hummingbird:cyberdocs:3.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.5.6`,
        fs:  `cpe:2.3:a:search.cpan:libwww-perl:5.63:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.5.7`,
        fs:  `cpe:2.3:a:hidemaru:editor:8.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.5.8`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.27.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.5.9`,
        fs:  `cpe:2.3:a:sap:solution_manager:7.20:sp01:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.5.10`,
        fs:  `cpe:2.3:a:opera:opera_browser:6.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.5.11`,
        fs:  `cpe:2.3:a:igreks:milkystep_professional:1.75:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.5.12`,
        fs:  `cpe:2.3:a:quest:benchmark_factory_for_databases:5.8.1.571:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.5.13`,
        fs:  `cpe:2.3:o:cisco:asa_cx_context-aware_security_software:9.3.4-2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:8.5.14`,
        fs:  `cpe:2.3:a:ikiwiki:ikiwiki:1.33.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:9.0.0:m1`,
        fs:  `cpe:2.3:h:d-link:dir-619l:a1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:9.0.0:m10`,
        fs:  `cpe:2.3:a:sendcard:sendcard:3.2.0:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:9.0.0:m11`,
        fs:  `cpe:2.3:a:google:chrome:19.0.1077.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:9.0.0:m12`,
        fs:  `cpe:2.3:a:wdc:smartware:1.3.0.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:9.0.0:m13`,
        fs:  `cpe:2.3:a:avsmedia:dvdmenu_editor:1.2.1.19:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:9.0.0:m14`,
        fs:  `cpe:2.3:h:cisco:unified_ip_phone_7940g:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:9.0.0:m15`,
        fs:  `cpe:2.3:a:google:chrome:35.0.1916.84:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:9.0.0:m16`,
        fs:  `cpe:2.3:a:prosody:prosody:0.9.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:9.0.0:m17`,
        fs:  `cpe:2.3:o:yamaha:srt100:10.00.27:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:9.0.0:m18`,
        fs:  `cpe:2.3:a:yard_radius_project:yard_radius:1.1.2-4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:9.0.0:m19`,
        fs:  `cpe:2.3:a:phorum:phorum:5.1.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:9.0.0:m2`,
        fs:  `cpe:2.3:o:cisco:nx-os:6.0\(3\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:9.0.0:m20`,
        fs:  `cpe:2.3:a:openafs:openafs:1.4.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:9.0.0:m3`,
        fs:  `cpe:2.3:a:libguestfs:libguestfs:1.21.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:9.0.0:m4`,
        fs:  `cpe:2.3:a:dest-unreach:socat:1.3.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:9.0.0:m5`,
        fs:  `cpe:2.3:a:janrain:ruby-openid:2.2.3:-:-:*:-:ruby:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:9.0.0:m6`,
        fs:  `cpe:2.3:a:rik_de_boer:revisioning:7.x-1.x:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:9.0.0:m7`,
        fs:  `cpe:2.3:a:cisco:desktop_administrator:80.2.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:9.0.0:m8`,
        fs:  `cpe:2.3:a:vmware:infrastructure:3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat:9.0.0:m9`,
        fs:  `cpe:2.3:a:ibm:infosphere_information_server_metadata_workbench:8.1.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat_connectors:1.2.40`,
        fs:  `cpe:2.3:a:drupal:drupal:7.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat_jk_web_server_connector:-`,
        fs:  `cpe:2.3:o:oracle:linux:5.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat_jk_web_server_connector:1.2.19`,
        fs:  `cpe:2.3:a:xymon:xymon:4.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat_jk_web_server_connector:1.2.20`,
        fs:  `cpe:2.3:a:hotbar:hbtools:4.6.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomcat_jk_web_server_connector:1.2.22`,
        fs:  `cpe:2.3:a:spreecommerce:spree:0.7.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:tomee:-`,
        fs:  `cpe:2.3:a:fetchmail:fetchmail:5.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:2.0.0`,
        fs:  `cpe:2.3:o:moxa:edr_g903_firmware:2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:2.0.0:alpha`,
        fs:  `cpe:2.3:a:cmsdevelopment:business_card_web_builder:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:2.0.1`,
        fs:  `cpe:2.3:a:ati:catalyst_control_center_localization:2008.717.2343.40629:*:*:ko:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:2.1.0`,
        fs:  `cpe:2.3:a:ibm:websphere_extreme_scale_client:7.1.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:2.1.1`,
        fs:  `cpe:2.3:a:gnu:glibc:2.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:2.1.2`,
        fs:  `cpe:2.3:a:plugin-organizer_project:plugin-organizer:5.7.5:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:2.1.3`,
        fs:  `cpe:2.3:a:no-margin-for-errors:prettyphoto:3.1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:2.1.4`,
        fs:  `cpe:2.3:o:juniper:junos:13.2x51:d20:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:2.1.5`,
        fs:  `cpe:2.3:o:cisco:catos:7.2\(1\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:2.1.6`,
        fs:  `cpe:2.3:a:ooorl_project:ooorl:-:-:-:*:-:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:2.1.7`,
        fs:  `cpe:2.3:a:nullsoft:winamp:2.91:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:2.1.8`,
        fs:  `cpe:2.3:a:cisco:unified_callmanager:3.3\(5\)sr2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:2.1.9`,
        fs:  `cpe:2.3:a:mariadb:mariadb:10.0.26:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:3.0.0`,
        fs:  `cpe:2.3:a:libguestfs:libguestfs:1.17.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:3.0.1`,
        fs:  `cpe:2.3:a:moinejf:abcm2ps:4.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:3.0.2`,
        fs:  `cpe:2.3:h:arris:touchstone_tg862g\/ct:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:3.0.3`,
        fs:  `cpe:2.3:a:bouncycastle:legion-of-the-bouncy-castle-c\#-cryptography-api:1.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:3.0.4`,
        fs:  `cpe:2.3:a:bravenewcode:wptouch:1.9.9.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:3.1.0`,
        fs:  `cpe:2.3:a:bravenewcode:wptouch:1.9.9.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:3.1.1`,
        fs:  `cpe:2.3:a:oracle:mysql:5.5.39:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:3.1.2`,
        fs:  `cpe:2.3:a:ibm:spss_collaboration_and_deployment_services:5.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:3.1.3`,
        fs:  `cpe:2.3:a:gitphp_project:gitphp:0.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:3.1.4`,
        fs:  `cpe:2.3:a:fedoraproject:sssd:1.13.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:3.2.0`,
        fs:  `cpe:2.3:a:google:chrome:14.0.835.96:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:3.3.0`,
        fs:  `cpe:2.3:a:trexart:campaignmonitor:6.x-3.0:beta3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:3.3.1`,
        fs:  `cpe:2.3:a:ibm:websphere_application_server:5.0.2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:3.3.2`,
        fs:  `cpe:2.3:a:smartsound:quicktracks:5.1.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:3.3.3`,
        fs:  `cpe:2.3:o:ibm:security_access_manager_for_web_8.0_firmware:8.0.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:3.3.4`,
        fs:  `cpe:2.3:a:sendcard:sendcard:3.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:3.3.5`,
        fs:  `cpe:2.3:a:wireshark:wireshark:1.6.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:4.0.1`,
        fs:  `cpe:2.3:a:apache:cloudstack:4.5.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:4.1.0`,
        fs:  `cpe:2.3:a:susumu_terao:terapad:0.40:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:4.2.0`,
        fs:  `cpe:2.3:a:totalseminars:a\&#43;_2006_demo:4.10.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:4.2.1`,
        fs:  `cpe:2.3:a:rockwellautomation:1763-l16bwa_series_a:15.000:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:4.2.1.1`,
        fs:  `cpe:2.3:a:ifdefined:bugtracker.net:2.4.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:5.0.0`,
        fs:  `cpe:2.3:a:xmlsoft:libxml2:2.4.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:5.0.1`,
        fs:  `cpe:2.3:a:ibm:websphere_application_server_feature_pack_for_web_services:6.1.0.29:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:5.1.1`,
        fs:  `cpe:2.3:a:mm_forum_project:mm_forum:0.1.4:*:*:*:*:typo3:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:5.3.0`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:5.3.1`,
        fs:  `cpe:2.3:a:sun:jre:1.4.2_28:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:6.0.0`,
        fs:  `cpe:2.3:a:moodle:moodle:2.9.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:6.1.0`,
        fs:  `cpe:2.3:a:drupal:drupal:5.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:6.1.1`,
        fs:  `cpe:2.3:a:apache:wink:1.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:traffic_server:6.2.0`,
        fs:  `cpe:2.3:a:dan_kogai:encode_module:1.33:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.4.0`,
        fs:  `cpe:2.3:a:philippe_jounin:tftpd32:2.74:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.4.1`,
        fs:  `cpe:2.3:a:weathernews:weathernews_touch:1.39:-:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.4.2`,
        fs:  `cpe:2.3:a:oracle:developer_suite:9.0.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.4.3`,
        fs:  `cpe:2.3:a:busybox:busybox:1.0.0:pre10:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.4.4`,
        fs:  `cpe:2.3:a:apache:axis:1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.4.5`,
        fs:  `cpe:2.3:a:robo-ftp:robo-ftp:3.7.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.4.6`,
        fs:  `cpe:2.3:a:gajim:gajim:0.16.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.4.7`,
        fs:  `cpe:2.3:a:debian:dpkg:1.4.1.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.4.8`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.225:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.4.9`,
        fs:  `cpe:2.3:a:google:chrome:32.0.1672.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.4.10`,
        fs:  `cpe:2.3:a:git-scm:git:1.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.4.11`,
        fs:  `cpe:2.3:o:cisco:ios:12.0\(5\)xn1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.4.12`,
        fs:  `cpe:2.3:a:profile2_privacy_project:profile2_privacy:7.x-1.4:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.4.13`,
        fs:  `cpe:2.3:a:ibm:urbancode_deploy:6.0.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.4.14`,
        fs:  `cpe:2.3:a:entity_reference_project:entityreference:7.x-1.0:beta5:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.4.15`,
        fs:  `cpe:2.3:a:macromedia:shockwave_flash_plugin:8.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.4.16`,
        fs:  `cpe:2.3:a:inkscape:inkscape:0.42:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.4.17`,
        fs:  `cpe:2.3:a:apache:httpasyncclient:4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.4.18`,
        fs:  `cpe:2.3:a:digium:asterisk:11.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.4.19`,
        fs:  `cpe:2.3:a:teradata:shared_icu_libraries:0.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.4.20`,
        fs:  `cpe:2.3:a:vmware:workstation:8.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.4.21`,
        fs:  `cpe:2.3:a:oracle:industry_applications:12.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.5.0`,
        fs:  `cpe:2.3:a:hp:ignite-ux:b.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.5.1`,
        fs:  `cpe:2.3:a:ibm:tivoli_storage_manager:6.1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.5.2`,
        fs:  `cpe:2.3:a:fedoraproject:sssd:1.8.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.5.3`,
        fs:  `cpe:2.3:h:huawei:h3c_ar\(oem_in\):r2209:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.5.4`,
        fs:  `cpe:2.3:a:call-cc:chicken:4.5.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.5.5`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.37:rc6:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.5.6`,
        fs:  `cpe:2.3:a:rack_project:rack:1.2.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.5.7`,
        fs:  `cpe:2.3:a:ibm:distributed_marketing:10.0.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.5.8`,
        fs:  `cpe:2.3:a:morequick:greenbrowser:6.1.0216:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:1.5.14`,
        fs:  `cpe:2.3:a:ibm:installation_manager:1.8.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:6.0.0`,
        fs:  `cpe:2.3:a:bloodjournal:blood:2.1:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:6.9.0`,
        fs:  `cpe:2.3:a:microsoft:works:3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:6.10.0`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.7:p478:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:6.11.0`,
        fs:  `cpe:2.3:a:search.cpan:libwww-perl:5b6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:6.12.0`,
        fs:  `cpe:2.3:a:mysql:mysql:4.1.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:6.13.0`,
        fs:  `cpe:2.3:a:migrate_project:migrate:7.x-2.0:rc2:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:6.14.0`,
        fs:  `cpe:2.3:a:role_scoper_project:role_scoper:1.3:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:6.15.0`,
        fs:  `cpe:2.3:a:ibm:inventory_scout:2.2.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:6.16.0`,
        fs:  `cpe:2.3:a:red-gate:sql_doc:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:6.17.0`,
        fs:  `cpe:2.3:a:mavenhut:solitaire_arena:1.0.15:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:6.18.0`,
        fs:  `cpe:2.3:a:libraw:libraw:0.14.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:6.19.0`,
        fs:  `cpe:2.3:o:cisco:ios:12.2fz:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:6.20.0`,
        fs:  `cpe:2.3:a:drupal:drupal:7.51:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:6.21.0`,
        fs:  `cpe:2.3:o:sgi:irix:4.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:6.22.0`,
        fs:  `cpe:2.3:a:python:python:3.2:alpha:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:6.23.0`,
        fs:  `cpe:2.3:a:acquia:commons:6.x-2.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:6.24.0`,
        fs:  `cpe:2.3:a:mozilla:mozilla:1.8:alpha4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wicket:7.1.0`,
        fs:  `cpe:2.3:a:horde:imp:2.2.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wink:0.1`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.7:p24:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wink:1.0`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.6.32.34:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wink:1.1.0`,
        fs:  `cpe:2.3:a:sun:java_studio_enterprise:8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wink:1.1.1`,
        fs:  `cpe:2.3:a:hp:product_assistant:100.0.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wink:1.1.2`,
        fs:  `cpe:2.3:a:opera:opera_mini:7.5:-:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wink:1.2.0`,
        fs:  `cpe:2.3:a:ntp:ntp:4.2.7:p163:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wink:1.2.1`,
        fs:  `cpe:2.3:a:git-scm:git:1.7.3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wink:1.3.0`,
        fs:  `cpe:2.3:a:apache:subversion:1.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wss4j:1.6.10`,
        fs:  `cpe:2.3:a:gitlab:gitlab:8.10.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wss4j:1.6.11`,
        fs:  `cpe:2.3:o:cisco:ios_xr:3.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wss4j:1.6.12`,
        fs:  `cpe:2.3:o:microsoft:windows_2000:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wss4j:1.6.13`,
        fs:  `cpe:2.3:a:gitlab:gitlab:0.9.6:*:*:*:community:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wss4j:1.6.14`,
        fs:  `cpe:2.3:a:dm_computer_solutions:ultraedit:8.20a:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wss4j:1.6.15`,
        fs:  `cpe:2.3:a:bundler:bundler:1.11.0:pre.2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wss4j:1.6.16`,
        fs:  `cpe:2.3:h:hp:h3c_router:rt-msr2040-ac-ovs-h3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wss4j:2.0.0`,
        fs:  `cpe:2.3:o:microsoft:windows_2000:-:gold:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wss4j:2.0.0:rc1`,
        fs:  `cpe:2.3:a:adobe:acrobat:10.0.1:-:pro:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:wss4j:2.0.1`,
        fs:  `cpe:2.3:a:marcel_brinkkemper:lazyest-gallery:1.1.3.3:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xalan-java:1.0.0`,
        fs:  `cpe:2.3:a:dynpg:dynpg_cms:3.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xalan-java:2.0.0`,
        fs:  `cpe:2.3:a:haxx:libcurl:7.16.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xalan-java:2.0.1`,
        fs:  `cpe:2.3:a:ibm:tivoli_service_request_manager:7.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xalan-java:2.1.0`,
        fs:  `cpe:2.3:a:dell:dataengine:5.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xalan-java:2.2.0`,
        fs:  `cpe:2.3:a:hp:storage_data_protector:7.01:-:*:*:*:windows_server_2008:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xalan-java:2.4.0`,
        fs:  `cpe:2.3:a:tipsandtricks-hq:wordpress_simple_paypal_shopping_cart:1.3:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xalan-java:2.4.1`,
        fs:  `cpe:2.3:a:horde:horde_application_framework:3.0:rc3:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xalan-java:2.5.0`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.3.5-7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xalan-java:2.5.1`,
        fs:  `cpe:2.3:o:yahama:rt56v:4.07.37:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xalan-java:2.5.2`,
        fs:  `cpe:2.3:o:huawei:s2350ei_firmware:v200r006c00:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xalan-java:2.6.0`,
        fs:  `cpe:2.3:a:cerberusftp:ftp_server:4.0.8.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xalan-java:2.7.0`,
        fs:  `cpe:2.3:a:adodb_project:adodb:5.20.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xalan-java:2.7.1`,
        fs:  `cpe:2.3:a:ati:catalyst_control_center_graphics_previews_common:2009.211.2225.40219:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xalan-java:2.7.2`,
        fs:  `cpe:2.3:a:mysql:mysql:6.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xerces-c%2b%2b:-`,
        fs:  `cpe:2.3:o:cisco:nx-os:5.0\(3\)n1\(1a\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xerces-c%2b%2b:1.0.0`,
        fs:  `cpe:2.3:a:palemoon:pale_moon:9.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xerces-c%2b%2b:1.0.1`,
        fs:  `cpe:2.3:a:google:chrome:28.0.1500.19:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xerces-c%2b%2b:1.1.0`,
        fs:  `cpe:2.3:a:ibm:tivoli_asset_discovery_for_distributed:7.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xerces-c%2b%2b:1.2.0`,
        fs:  `cpe:2.3:h:cisco:router_3600:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xerces-c%2b%2b:1.3.0`,
        fs:  `cpe:2.3:a:endeavorsystems:openfisma:2.14.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xerces-c%2b%2b:1.4.0`,
        fs:  `cpe:2.3:a:tiki:tikiwiki_cms\/groupware:6.7:-:lts:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xerces-c%2b%2b:1.5.0`,
        fs:  `cpe:2.3:a:orangehrm:orangehrm:2.6.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xerces-c%2b%2b:1.6.0`,
        fs:  `cpe:2.3:a:webkitgtk:webkitgtk\&#43;:2.9.90:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xerces-c%2b%2b:1.7.0`,
        fs:  `cpe:2.3:a:mawashimono:nikki:5.111:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xerces-c%2b%2b:2.0.0`,
        fs:  `cpe:2.3:a:chatelao:php_address_book:3.2.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xerces-c%2b%2b:2.1.0`,
        fs:  `cpe:2.3:a:bit51:better-wp-security:2.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xerces-c%2b%2b:2.2.0`,
        fs:  `cpe:2.3:a:adobe:flash_player_for_android:11.1.111.19:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xerces-c%2b%2b:2.3.0`,
        fs:  `cpe:2.3:a:quest:sql_optimizer_for_oracle:7.5.0.1113:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xerces-c%2b%2b:2.4.0`,
        fs:  `cpe:2.3:a:mysqldumper:mysqldumper:1.21_b6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xerces-c%2b%2b:2.5.0`,
        fs:  `cpe:2.3:a:subsplash:renny_mclean_ministries:2.8.1:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xerces-c%2b%2b:2.6.0`,
        fs:  `cpe:2.3:a:gnu:readline:4.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xerces-c%2b%2b:2.7.0`,
        fs:  `cpe:2.3:h:supermicro:x9dbi-f:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xerces-c%2b%2b:2.8.0`,
        fs:  `cpe:2.3:h:hp:laserjet_4200dnt_network_printer:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xerces-c%2b%2b:3.1.3`,
        fs:  `cpe:2.3:a:laurent_destailleur:awstats:6.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xerces2_java:-`,
        fs:  `cpe:2.3:a:fiverrscript:fiverrscript:7.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xerces2_java:2.9.1`,
        fs:  `cpe:2.3:a:apache:tomcat:7.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xerces2_java:2.10.0`,
        fs:  `cpe:2.3:a:splunk:splunk:6.2.0:*:*:*:enterprise:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xerces2_java:2.11.0`,
        fs:  `cpe:2.3:a:bmc:atrium_integration_engine:7.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_c%2b%2b:0.1.0`,
        fs:  `cpe:2.3:a:apache:http_server:2.4.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_c%2b%2b:0.2.0`,
        fs:  `cpe:2.3:a:citrix:xenapp_administration:4.7.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_c%2b%2b:1.1.0`,
        fs:  `cpe:2.3:o:microsoft:windows_8.1:-:-:-:*:-:-:x64:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_c%2b%2b:1.2.0`,
        fs:  `cpe:2.3:a:oscommerce:oscommerce:2.2_ms2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_c%2b%2b:1.2.1`,
        fs:  `cpe:2.3:a:appstros:appstros_-_free_gift_cards\!:1.1.3:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_c%2b%2b:1.3.0`,
        fs:  `cpe:2.3:a:octobercms:october_cms:1.0.426:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_c%2b%2b:1.3.1`,
        fs:  `cpe:2.3:a:macromedia:shockwave_flash_plugin:6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_c%2b%2b:1.4.0`,
        fs:  `cpe:2.3:a:google:chrome:28.0.1500.33:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_c%2b%2b:1.5.0`,
        fs:  `cpe:2.3:a:sun:jre:6.0.220:update22:*:*:*:*:x64:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_c%2b%2b:1.5.1`,
        fs:  `cpe:2.3:a:atlassian:confluence:2.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_c%2b%2b:1.6.0`,
        fs:  `cpe:2.3:a:quick_tabs_module_project:quicktabs:6.x-2.0:rc4:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_c%2b%2b:1.6.1`,
        fs:  `cpe:2.3:a:thulasidas:easy-adsense-lite:5.12:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_c%2b%2b:1.7.0`,
        fs:  `cpe:2.3:a:oracle:vm_virtualbox:5.0.36:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_c%2b%2b:1.7.1`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(6c\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_java:1.2.0`,
        fs:  `cpe:2.3:a:skype:skype:1.3.0.48:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_java:1.2.1`,
        fs:  `cpe:2.3:a:debian:kernel-patch-vserver:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_java:1.3.0`,
        fs:  `cpe:2.3:a:alex_kellner:powermail:1.3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_java:1.4.0`,
        fs:  `cpe:2.3:o:hp:hp-ux:9.09:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_java:1.4.1`,
        fs:  `cpe:2.3:a:graphisoft:archicad:13.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_java:1.4.2`,
        fs:  `cpe:2.3:a:trustgo:antivirus_\&amp;_mobile_security:1.2.8:-:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_java:1.4.3`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.35:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_java:1.4.4`,
        fs:  `cpe:2.3:a:joakim_nygard:webgrind:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_java:1.4.5`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.4.28:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_java:1.4.6`,
        fs:  `cpe:2.3:a:gpg4win:gpg4win:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_java:1.4.7`,
        fs:  `cpe:2.3:a:hardened-php:suhosin:0.9.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_java:1.4.8`,
        fs:  `cpe:2.3:a:opera:opera_browser:6.0:beta1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_java:1.5.0`,
        fs:  `cpe:2.3:a:zenphoto:zenphoto:1.3.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_java:1.5.1`,
        fs:  `cpe:2.3:a:mcafee:virusscan_security_center:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_java:1.5.2`,
        fs:  `cpe:2.3:a:ssh:tectia_client:6.1.9.95:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_java:1.5.3`,
        fs:  `cpe:2.3:a:sgi:performance_co-pilot:3.6.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_java:1.5.4`,
        fs:  `cpe:2.3:a:vmware:vcloud_automation_center:6.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_java:1.5.5`,
        fs:  `cpe:2.3:a:philippe_jounin:tftpd32:2.84:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:xml_security_for_java:1.5.6`,
        fs:  `cpe:2.3:o:broadcom:widcomm_bluetooth:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:zookeeper:-`,
        fs:  `cpe:2.3:a:emc:rsa_archer_egrc:5.4:sp1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:zookeeper:3.4.0`,
        fs:  `cpe:2.3:a:google:chrome:19.0.1084.33:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:zookeeper:3.4.1`,
        fs:  `cpe:2.3:a:google:chrome:5.0.375.48:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:zookeeper:3.4.2`,
        fs:  `cpe:2.3:a:docker:docker:0.9.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:zookeeper:3.4.3`,
        fs:  `cpe:2.3:a:batavi:batavi:1.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:zookeeper:3.4.4`,
        fs:  `cpe:2.3:a:designsandcode:forget-about-shorcode-buttons:1.1.1:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:zookeeper:3.4.5`,
        fs:  `cpe:2.3:a:cisco:adaptive_security_appliance_software:7.2.2.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:zookeeper:3.4.6`,
        fs:  `cpe:2.3:a:ibm:content_collector:3.0.0.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:zookeeper:3.4.7`,
        fs:  `cpe:2.3:a:openssl:openssl:1.0.1s:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:zookeeper:3.4.8`,
        fs:  `cpe:2.3:a:rom_walton:boinc:7.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:zookeeper:3.4.9`,
        fs:  `cpe:2.3:a:haxx:libcurl:7.15.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:zookeeper:3.5.0`,
        fs:  `cpe:2.3:h:alliedtelesis:img646bd:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:zookeeper:3.5.1`,
        fs:  `cpe:2.3:a:oracle:jre:1.7.0:update_76:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache:zookeeper:3.5.2`,
        fs:  `cpe:2.3:a:oracle:jre:1.7.0:update21:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache_friends:xampp:-`,
        fs:  `cpe:2.3:a:nvidia:gpu_driver:313.00:-:*:*:freebsd:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache_friends:xampp:1.5.2`,
        fs:  `cpe:2.3:a:logitech:lws_launcher:13.30.1379.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache_software_foundation:tomcat:4.1.4`,
        fs:  `cpe:2.3:a:git-scm:git:1.5.0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache_software_foundation:tomcat:4.1.5`,
        fs:  `cpe:2.3:h:zyxel:p-660hw_d1:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache_software_foundation:tomcat:4.1.6`,
        fs:  `cpe:2.3:o:redhat:enterprise_linux:5:*:client_workstation:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache_software_foundation:tomcat:4.1.7`,
        fs:  `cpe:2.3:h:polycom:viewstation_128:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache_software_foundation:tomcat:4.1.8`,
        fs:  `cpe:2.3:a:ibm:db2:9.1.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache_software_foundation:tomcat:4.1.11`,
        fs:  `cpe:2.3:a:ati:catalyst_control_center_utility:2008.225.2153.39091:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache_software_foundation:tomcat:4.1.13`,
        fs:  `cpe:2.3:a:invisionpower:invision_power_board:3.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache_software_foundation:tomcat:4.1.14`,
        fs:  `cpe:2.3:a:google:chrome:15.0.874.46:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache_software_foundation:tomcat:4.1.16`,
        fs:  `cpe:2.3:a:ssh:tectia_server:4.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache_software_foundation:tomcat:4.1.17`,
        fs:  `cpe:2.3:a:ibm:storwize_unified_v7000_software:1.4.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache_software_foundation:tomcat:4.1.18`,
        fs:  `cpe:2.3:a:wppa.opajaap:wp-photo-album-plus:5.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache_software_foundation:tomcat:4.1.19`,
        fs:  `cpe:2.3:a:cisco:adaptive_security_appliance_software:9.4.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache_software_foundation:tomcat:4.1.20`,
        fs:  `cpe:2.3:a:libpng:libpng:1.0.54:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache_software_foundation:tomcat:4.1.21`,
        fs:  `cpe:2.3:a:cisco:carrier_routing_system:4.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache_software_foundation:tomcat:4.1.22`,
        fs:  `cpe:2.3:a:nextcloud:nextcloud_server:9.0.51:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache_software_foundation:tomcat:4.1.23`,
        fs:  `cpe:2.3:a:cisco:application_control_engine_software:a4\(2.1\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache_software_foundation:tomcat:4.1.25`,
        fs:  `cpe:2.3:a:falconsc:wisepoint:3.6.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache_software_foundation:tomcat:4.1.26`,
        fs:  `cpe:2.3:a:devexpress:aspxfilemanager_control_for_webforms_and_mvc:12.2.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache_software_foundation:tomcat:4.1.27`,
        fs:  `cpe:2.3:h:huawei:s12708:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache_software_foundation:tomcat:4.1.30`,
        fs:  `cpe:2.3:a:kde:kde_sc:4.4.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apache_software_foundation:tomcat:4.1.33`,
        fs:  `cpe:2.3:a:busybox:busybox:1.12.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:apc:1500:-`,
        fs:  `cpe:2.3:o:bluecoat:sgos:3.2.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:apc:2200:-`,
        fs:  `cpe:2.3:a:netscape:fasttrack_server:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:apc:3000:-`,
        fs:  `cpe:2.3:a:jordy_meow:media_file_renamer:0.1:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/h:apc:700:-`,
        fs:  `cpe:2.3:a:apache:http_server:1.3.34:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:apc:ap9606:-`,
        fs:  `cpe:2.3:a:git_for_windows_project:git_for_windows:1.8.3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:apc:ap9606:3.0`,
        fs:  `cpe:2.3:a:google:chrome:14.0.835.112:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:apc:ap9606:3.0.1`,
        fs:  `cpe:2.3:a:boltwire:boltwire:3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apc:apcupsd:-`,
        fs:  `cpe:2.3:a:promotic:promotic:8.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apc:powerchute:-`,
        fs:  `cpe:2.3:a:boost:boost:1.54.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apc:powerchute:1.3.4:-:personal`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(5d\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apc:powerchute:1.5.0:-:personal`,
        fs:  `cpe:2.3:a:freelancerkit:freelancerkit:2.34:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apc:powerchute:2.0:-:personal`,
        fs:  `cpe:2.3:a:commentluv:commentluv:2.90.7:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apc:powerchute:2.1.1:-:personal`,
        fs:  `cpe:2.3:a:foobla:com_obsuggest:1.6.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apc:powerchute:2.2:-:personal`,
        fs:  `cpe:2.3:a:adaptivecomputing:torque_resource_manager:4.1.5.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apc:powerchute:5.0.2`,
        fs:  `cpe:2.3:a:acme:thttpd:2.25:b:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apc:powerchute:9.0.1.606::~~business~~~`,
        fs:  `cpe:2.3:a:heinz_mauelshagen:lvm2:2.02.58:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apc:powerchute_network_shutdown:-`,
        fs:  `cpe:2.3:a:gisle_aas:libwww-perl:5.814:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apc:powerchute_network_shutdown:3.0.0`,
        fs:  `cpe:2.3:a:ibm:spectrum_lsf:8.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:apc:smart-ups_2200:-`,
        fs:  `cpe:2.3:a:john_nunemaker:httparty:0.2.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apc:switched_rack_pdu_firmware:-`,
        fs:  `cpe:2.3:a:drupal:drupal:6.34:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apc:switched_rack_pdu_firmware:3.5.5`,
        fs:  `cpe:2.3:a:mediaelementjs:mediaelement.js:2.8.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apereo:central_authentication_service:3.5.2`,
        fs:  `cpe:2.3:o:conectiva:linux:7.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aperturemobilemedia:aperture_mobile_media:1.404::~~~android~~`,
        fs:  `cpe:2.3:a:mit:kerberos:5-1.15.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apexoo:spider:1.1`,
        fs:  `cpe:2.3:o:cisco:ios:12.2\(4\)xl4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apexsql:apexsql_log:2008`,
        fs:  `cpe:2.3:a:spamdyke:spamdyke:4.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apheliontechnologies:hydfm:1.1.9::~~~android~~`,
        fs:  `cpe:2.3:a:hp:universal_cmbd_foundation:10.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aphpkb:aphpkb:0.95.5`,
        fs:  `cpe:2.3:o:canonical:ubuntu_linux:13.04:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aphpkb:aphpkb:0.95.6`,
        fs:  `cpe:2.3:a:cloudera:cdh:5.3.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:aphpkb:aphpkb:0.95.7`,
        fs:  `cpe:2.3:o:sun:sunos:5.11:*:*:*:*:*:sparc:*`,
    },
     
    {
        uri: `cpe:/a:apollogrp:riverpoint_writer:1.0`,
        fs:  `cpe:2.3:a:libtiff:libtiff:3.7.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:app_maker_ks:buy_books:0.1::~~~android~~`,
        fs:  `cpe:2.3:a:ibm:lotus_domino:5.0.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:appa-apps:top_roller_coasters_europe_1:%407f050001::~~~android~~`,
        fs:  `cpe:2.3:a:sun:java_db:10.2.2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:appa-apps:top_roller_coasters_europe_2:%407f050001::~~~android~~`,
        fs:  `cpe:2.3:o:yahama:rtx2000:7.00.19:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apparound:apparound_blend:4.9.0::~~~android~~`,
        fs:  `cpe:2.3:a:dan_kogai:encode_module:1.63:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:appbasedtechnologies:belaire_family_orthodontics:1.304::~~~android~~`,
        fs:  `cpe:2.3:a:ca:unicenter_nsm:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:appbelle:stop_headaches_and_migraines:1.2::~~~android~~`,
        fs:  `cpe:2.3:a:sebastian_heinlein:aptdaemon:0.30:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:appbelle:top_hangover_cures:1.2::~~~android~~`,
        fs:  `cpe:2.3:a:abbyy:finereader:6.0:-:pro:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:appdeploy:free_appdeploy_repackager:1.2.53`,
        fs:  `cpe:2.3:a:openldap:openldap:2.4.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:appeak:poker:2.4.5::~~~android~~`,
        fs:  `cpe:2.3:o:cisco:ios:12.4\(24\)mda6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:appearingbusiness:magic_balloonman_marty_boone:1.4::~~~android~~`,
        fs:  `cpe:2.3:a:hp:insight_management_wbem_providers_for_windows_server_2003\/2008:2.3.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apphp:hotel_site:3.0.9`,
        fs:  `cpe:2.3:a:digium:asterisk:1.4.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apphp:hotel_site:3.1.3`,
        fs:  `cpe:2.3:a:mediawiki:rssreader:0.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apphp:hotel_site:3.2.4`,
        fs:  `cpe:2.3:o:cisco:ios:12.3yl:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apphp:hotel_site:3.3.0`,
        fs:  `cpe:2.3:a:fenrir-inc:sleipnir_mobile:1.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apphp:hotel_site:3.4.4`,
        fs:  `cpe:2.3:a:mathjs:math.js:3.10.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apphp:hotel_site:3.5.1`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.3.4-9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apphp:hotel_site:3.6.1`,
        fs:  `cpe:2.3:a:octobercms:october_cms:1.0.380:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apphp:hotel_site:3.7.5`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.3.8-3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apphp:hotel_site:3.8.4`,
        fs:  `cpe:2.3:h:amd:16h_model_00h_processor:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apphp:hotel_site:3.9.1`,
        fs:  `cpe:2.3:a:ponsoftware:explzh:3.34:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:apple:802.11n:7.3.1`,
        fs:  `cpe:2.3:a:lennart_poettering:systemd:018:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:apple:a_ux:-`,
        fs:  `cpe:2.3:a:ibm:general_parallel_file_system_storage_server:2.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:apple:a_ux:2.0.1`,
        fs:  `cpe:2.3:h:hp:cz257a:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:apple:a_ux:3.1.1`,
        fs:  `cpe:2.3:a:apache:tomcat:8.0.32:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:afp_server:-`,
        fs:  `cpe:2.3:a:mediawiki:mediawiki:1.6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:apple:airport_base_station:-`,
        fs:  `cpe:2.3:h:cisco:7920_wireless_ip_phone:1.0\(8\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:apple:airport_base_station_firmware:7.3.2`,
        fs:  `cpe:2.3:a:tejimaya:openpne:3.0.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:apple:airport_base_station_firmware:7.4.1`,
        fs:  `cpe:2.3:a:mozilla:mozilla:0.9.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:apple:airport_base_station_firmware:7.4.2`,
        fs:  `cpe:2.3:a:lunascape:lunascape:6.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:apple:airport_base_station_firmware:7.5.2`,
        fs:  `cpe:2.3:a:djvulibre_project:djvulibre:3.5.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:apple:airport_base_station_firmware:7.6`,
        fs:  `cpe:2.3:a:avaya:vsu_100:3.2.40:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:apple:airport_base_station_firmware:7.6.1`,
        fs:  `cpe:2.3:a:tylertech:taxweb:3.13.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:apple:airport_base_station_firmware:7.6.2`,
        fs:  `cpe:2.3:h:oracle:sparc_t3-4:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:apple:airport_base_station_firmware:7.6.3`,
        fs:  `cpe:2.3:a:hp:scan:7.0.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:apple:airport_card:-`,
        fs:  `cpe:2.3:a:cisco:jabber:9.7\(3\):*:*:*:*:windows:*:*`,
    },
     
    {
        uri: `cpe:/h:apple:airport_express:-`,
        fs:  `cpe:2.3:a:apache:solr:6.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:apple:airport_express:6.1`,
        fs:  `cpe:2.3:o:yahama:rt100i:2.01.19:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:apple:airport_express_base_station_firmware:6.1`,
        fs:  `cpe:2.3:a:google:chrome:4.0.249.68:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:apple:airport_extreme:-`,
        fs:  `cpe:2.3:a:apple:quicktime:7.1.2:-:windows:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:apple:airport_extreme:5.5`,
        fs:  `cpe:2.3:a:fetchmail:fetchmail:5.2.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:apple:airport_extreme_base_station_firmware:5.5`,
        fs:  `cpe:2.3:a:bftpd_project:bftpd:1.6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apache_mod_digest_apple:-`,
        fs:  `cpe:2.3:a:igniterealtime:smack:4.0.0:snapshot-2014-04-09:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:apple:apple_airport_base_station_firmware:-`,
        fs:  `cpe:2.3:a:linuxfoundation:cups-filters:1.0.75:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_application_support:1.0`,
        fs:  `cpe:2.3:a:samba:samba:3.6.17:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_application_support:1.2.1`,
        fs:  `cpe:2.3:o:cisco:catos:6.2\(3a\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_application_support:1.3.1`,
        fs:  `cpe:2.3:a:serv-u:serv-u:8.0.0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_application_support:1.3.2`,
        fs:  `cpe:2.3:a:sourcetreesolutions:mojoportal:2.3.9.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_application_support:1.4.1`,
        fs:  `cpe:2.3:a:adobe:add_or_remove_creative_suite_3_design_premium:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_application_support:1.5.1`,
        fs:  `cpe:2.3:a:igor_sysoev:nginx:1.3.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_application_support:1.5.2`,
        fs:  `cpe:2.3:a:matthew_booth:virt-v2v:0.4.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_application_support:2.0.1`,
        fs:  `cpe:2.3:h:hp:jf233a:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_application_support:2.1.5`,
        fs:  `cpe:2.3:a:ibm:lotus_inotes:8.5.3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/h:apple:apple_laserwriter:-`,
        fs:  `cpe:2.3:o:yahama:rt100i:2.02.33:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_mobile_device_support:2.1.0.25`,
        fs:  `cpe:2.3:a:google_authenticator_login_project:ga_login:6.x-1.0:alpha1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_mobile_device_support:3.2.0.47`,
        fs:  `cpe:2.3:h:cisco:video_surveillance_2600:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_mobile_device_support:3.3.0.69`,
        fs:  `cpe:2.3:o:juniper:junos:8.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_mobile_device_support:3.4.0.25`,
        fs:  `cpe:2.3:a:ibm:http_server:2.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_mobile_device_support:3.4.1.2`,
        fs:  `cpe:2.3:a:emc:rsa_security_analytics:10.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_mobile_device_support:4.0.0.97`,
        fs:  `cpe:2.3:a:csc:serviceclient:1.2.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_remote_desktop:1.2.4`,
        fs:  `cpe:2.3:a:zen-cart:zen_cart:1.2.6d:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_remote_desktop:2.0.0`,
        fs:  `cpe:2.3:a:haproxy:haproxy:1.5:dev11:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_remote_desktop:2.1.0`,
        fs:  `cpe:2.3:h:dell:3010cn:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_remote_desktop:2.2`,
        fs:  `cpe:2.3:a:divx:web_player:1.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_remote_desktop:3.0`,
        fs:  `cpe:2.3:o:linux:kernel:2.6.23:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_remote_desktop:3.0.0`,
        fs:  `cpe:2.3:a:pcre:pcre:8.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_remote_desktop:3.1`,
        fs:  `cpe:2.3:o:cisco:vpn_3000_concentrator_series_software:4.1.5.b:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_remote_desktop:3.2`,
        fs:  `cpe:2.3:a:radvision:l2w-323:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_remote_desktop:3.2.1`,
        fs:  `cpe:2.3:a:sysax:multi_server:4.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_remote_desktop:3.2.2`,
        fs:  `cpe:2.3:a:zimbra:collaboration_server:8.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_remote_desktop:3.3`,
        fs:  `cpe:2.3:a:atlassian:confluence:3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_remote_desktop:3.3.1`,
        fs:  `cpe:2.3:a:catalin_florian_radut:zeropoint:6.x-1.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_remote_desktop:3.3.2`,
        fs:  `cpe:2.3:a:groups_communities_and_co_project:gcc:7.x-1.0:alpha4:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_remote_desktop:3.4`,
        fs:  `cpe:2.3:a:techsmith:camtasia_studio:4.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_remote_desktop:3.5`,
        fs:  `cpe:2.3:a:apache:activemq:5.12.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_remote_desktop:3.5.1`,
        fs:  `cpe:2.3:a:pulsesecure:pulse_connect_secure:8.0r3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_remote_desktop:3.5.2`,
        fs:  `cpe:2.3:a:mozilla:bugzilla:4.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_remote_desktop:3.5.3`,
        fs:  `cpe:2.3:h:tp-link:tl-wrd841nd_\(9.0\):-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_remote_desktop:3.5.4`,
        fs:  `cpe:2.3:a:speed_software:explorer:2.1.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_remote_desktop:3.6`,
        fs:  `cpe:2.3:a:dlo:simple_anti_bot_registration_engine_plugin:0.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_remote_desktop:3.6.1`,
        fs:  `cpe:2.3:a:mozilla:firefox_esr:45.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_remote_desktop:3.6.2`,
        fs:  `cpe:2.3:a:postfix:postfix:2.0.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_remote_desktop:3.7`,
        fs:  `cpe:2.3:a:roxio:recordnow_audio:3.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:1.0.0`,
        fs:  `cpe:2.3:a:ibm:lotus_inotes:8.5.3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:1.1.0`,
        fs:  `cpe:2.3:a:puppetlabs:stdlib:2.3.3:*:*:*:*:puppet:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:2.0.0`,
        fs:  `cpe:2.3:h:cisco:nexus_3064t:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:2.0.1`,
        fs:  `cpe:2.3:a:cisco:ips_sensor_software:5.1\(1d\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:2.0.2`,
        fs:  `cpe:2.3:a:isc:bind:9.4:r5-b1:*:*:esv:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:2.1.0`,
        fs:  `cpe:2.3:a:owncloud:owncloud:4.5.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:2.2.0`,
        fs:  `cpe:2.3:a:exactcode:exactimage:0.7.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:2.3.0`,
        fs:  `cpe:2.3:o:vmware:esx:3.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:2.3.1`,
        fs:  `cpe:2.3:a:shopware:shopware:5.2.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:2.4.0`,
        fs:  `cpe:2.3:a:tryton:tryton:2.6.15:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:3.0.0`,
        fs:  `cpe:2.3:a:git-scm:git:1.8.1.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:3.0.1`,
        fs:  `cpe:2.3:a:bftpd_project:bftpd:1.0.16:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:3.0.2`,
        fs:  `cpe:2.3:h:cisco:unified_ip_phone_6921:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:4.1.0`,
        fs:  `cpe:2.3:a:hp:insight_diagnostics_online_edition_for_windows:8.6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:4.1.1`,
        fs:  `cpe:2.3:a:ibm:websphere_application_server:6.0.1.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:4.2.0`,
        fs:  `cpe:2.3:a:ibm:spss_statistics:22.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:4.2.1`,
        fs:  `cpe:2.3:h:nokia:n70:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:4.2.2`,
        fs:  `cpe:2.3:a:avscripts:av_arcade_pro:5.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:4.3.0`,
        fs:  `cpe:2.3:a:torproject:tor:0.2.2.27:beta:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:4.4.0`,
        fs:  `cpe:2.3:a:digium:asterisk:0.1.9.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:4.4.2`,
        fs:  `cpe:2.3:a:webprancer:garfield\&#39;s_defense:1.5.4:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:4.4.3`,
        fs:  `cpe:2.3:h:axis:2411_video_server:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:4.4.4`,
        fs:  `cpe:2.3:a:commerceguys:commerce_reorder:7.x-1.x:dev:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:5.0.0`,
        fs:  `cpe:2.3:a:google:chrome:27.0.1453.103:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:5.0.1`,
        fs:  `cpe:2.3:a:gnupg:gnupg:1.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:5.0.2`,
        fs:  `cpe:2.3:a:octopus:octopus_deploy:3.12.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:5.1.0`,
        fs:  `cpe:2.3:a:mozilla:mozilla:0.9.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:5.1.1`,
        fs:  `cpe:2.3:a:ibm:tivoli_netcool\/reporter:2.2.0.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:5.2.0`,
        fs:  `cpe:2.3:a:fedoraproject:sssd:1.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:6.0`,
        fs:  `cpe:2.3:a:openstack:icehouse:2014.1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:6.0.1`,
        fs:  `cpe:2.3:a:imagemagick:imagemagick:6.8.4-1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:6.0.2`,
        fs:  `cpe:2.3:a:cisco:asa_with_firepower_services:6.0.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:6.1`,
        fs:  `cpe:2.3:a:openstack:keystone:2013:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:6.1.1`,
        fs:  `cpe:2.3:o:microsoft:windows_server_2008:-:gold:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:6.1.2`,
        fs:  `cpe:2.3:o:freebsd:freebsd:3.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:6.2`,
        fs:  `cpe:2.3:a:jenkins:jenkins:1.555:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:6.2.1`,
        fs:  `cpe:2.3:a:otrs:otrs:2.4.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:7.0`,
        fs:  `cpe:2.3:o:iij:seil_b1_firmware:2.89:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:7.0.1`,
        fs:  `cpe:2.3:o:apple:apple_tv:9.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:7.0.3`,
        fs:  `cpe:2.3:a:foolabs:xpdf:0.92:b:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:7.1`,
        fs:  `cpe:2.3:a:bea:weblogic_server:8.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:apple:apple_tv:9.0`,
        fs:  `cpe:2.3:a:pizzashack:rssh:2.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:9.0.1`,
        fs:  `cpe:2.3:a:bea:aqualogic_interaction_content_service_for_windows_files:1.0.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:apple:apple_tv:9.1`,
        fs:  `cpe:2.3:o:cisco:ios:12.0\(7a\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:9.1.1`,
        fs:  `cpe:2.3:a:roderich_schupp:par-packer_module:0.92:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:apple:apple_tv:9.2`,
        fs:  `cpe:2.3:a:citrix:command_center:5.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:apple:apple_tv:9.2.1`,
        fs:  `cpe:2.3:a:adobe:acrobat:9.4.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/o:apple:apple_tv:9.2.2`,
        fs:  `cpe:2.3:a:novell:edirectory:8.7.3:sp9:windows:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:10.0`,
        fs:  `cpe:2.3:a:fedoraproject:sssd:1.12.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:10.0.1`,
        fs:  `cpe:2.3:a:oracle:mysql_connector\/j:3.0.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:10.1`,
        fs:  `cpe:2.3:a:stunnel:stunnel:5.09:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:10.1.1`,
        fs:  `cpe:2.3:a:apache:cloudstack:2.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:apple_tv:10.2`,
        fs:  `cpe:2.3:o:cisco:catos:5.2\(5\):*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:applescript:-`,
        fs:  `cpe:2.3:a:cacti:cacti:0.8.6h:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:bomarchivehelper:-`,
        fs:  `cpe:2.3:o:cisco:ios:12.3\(4\)xe4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:bonjour:1.0.0:-:windows`,
        fs:  `cpe:2.3:a:mongodb:mongodb:3.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:bonjour:1.0.1:-:windows`,
        fs:  `cpe:2.3:a:ibm:web_content_manager:7.0.0.1:cf002:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:bonjour:1.0.2:-:windows`,
        fs:  `cpe:2.3:a:citrix:metaframe_secure_access_manager:2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:bonjour:1.0.3:-:windows`,
        fs:  `cpe:2.3:a:ibm:iss_realsecure_network:6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:bonjour:1.0.4:-:windows`,
        fs:  `cpe:2.3:a:google:chrome:20.0.1132.18:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:bonjour:1.0.105`,
        fs:  `cpe:2.3:a:debian:xbuffy:3.3.bl.3-19:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:bonjour:1.0.106`,
        fs:  `cpe:2.3:a:webkitgtk:webkitgtk\&#43;:1.10.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:bonjour:2.0.3.0`,
        fs:  `cpe:2.3:a:redhat:kernel:2.4.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:bonjour:2.0.4.0`,
        fs:  `cpe:2.3:a:gnome:evolution:2.3.6.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:bonjour:2.0.5.0`,
        fs:  `cpe:2.3:a:microsoft:publisher:2013:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:bonjour:3.0.0.2`,
        fs:  `cpe:2.3:a:bit51:better-wp-security:-:alpha1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:bonjour:3.0.0.10`,
        fs:  `cpe:2.3:a:webform_project:webform:7.x-3.18:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:boot_camp:5.0`,
        fs:  `cpe:2.3:a:hp:unified_functional_testing:12.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:boot_camp:5.1`,
        fs:  `cpe:2.3:a:compaq:version_control_agent:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cfnetwork:-`,
        fs:  `cpe:2.3:a:vanillaforums:vanilla_forums:2.0.14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:claris_emailer:-`,
        fs:  `cpe:2.3:a:nordugrid:nordugrid-arc:0.5.41:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:claris_emailer:2.0v2`,
        fs:  `cpe:2.3:a:ibm:security_identity_manager:6.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:core_audio_technologies:-`,
        fs:  `cpe:2.3:h:samsung:galaxy_s2:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1`,
        fs:  `cpe:2.3:a:haxx:libcurl:6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.1`,
        fs:  `cpe:2.3:a:tincan:phplist:2.9.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.2`,
        fs:  `cpe:2.3:a:deon_george:phpldapadmin:1.2.0.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.3`,
        fs:  `cpe:2.3:a:dlitz:pycrypto:2.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.4`,
        fs:  `cpe:2.3:a:fenrir-inc:sleipnir_mobile:2.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.5`,
        fs:  `cpe:2.3:a:thekelleys:dnsmasq:2.24:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.5-1`,
        fs:  `cpe:2.3:a:webyog:sqlyog:4.05:-:community:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.5-2`,
        fs:  `cpe:2.3:o:yahama:rta55i:4.06.54:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.6`,
        fs:  `cpe:2.3:a:novell:groupwise:7.02:hp2r1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.6-1`,
        fs:  `cpe:2.3:a:modwsgi:mod_wsgi:3.5:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.6-2`,
        fs:  `cpe:2.3:a:fetchmail:fetchmail:6.2.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.6-3`,
        fs:  `cpe:2.3:a:analogx:simpleserver_shout:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.7`,
        fs:  `cpe:2.3:a:apple:cups:1.3:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.8`,
        fs:  `cpe:2.3:a:feedweb:feedweb:1.8.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.9`,
        fs:  `cpe:2.3:a:jrbcs:webform_report:6.x-1.10:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.9-1`,
        fs:  `cpe:2.3:a:tinc-vpn:tinc:1.0.20:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.10`,
        fs:  `cpe:2.3:a:mozilla:seamonkey:2.7.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.10-1`,
        fs:  `cpe:2.3:a:google:android_api:3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.11`,
        fs:  `cpe:2.3:a:zabbix:zabbix:2.2.7:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.12`,
        fs:  `cpe:2.3:a:emberjs:ember.js:1.13.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.13`,
        fs:  `cpe:2.3:a:juniper:networks_mobility_system_software:7.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.14`,
        fs:  `cpe:2.3:a:bouncycastle:legion-of-the-bouncy-castle-c\#-crytography-api:1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.15`,
        fs:  `cpe:2.3:a:rack_project:rack:1.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.16`,
        fs:  `cpe:2.3:a:joomla:joomla\!:3.5.0:rc4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.17`,
        fs:  `cpe:2.3:a:avast:endpoint_protection_suite_plus:8.0.1609:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.18`,
        fs:  `cpe:2.3:a:google:chrome:12.0.742.13:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.19`,
        fs:  `cpe:2.3:a:ssh:tectia_client:4.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.19:rc1`,
        fs:  `cpe:2.3:a:dotnetnuke:dotnetnuke:4.4.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.19:rc2`,
        fs:  `cpe:2.3:a:google:chrome:19.0.1084.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.19:rc3`,
        fs:  `cpe:2.3:a:cisco:transport_controller:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.19:rc4`,
        fs:  `cpe:2.3:a:djangoproject:django:1.10.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.19:rc5`,
        fs:  `cpe:2.3:a:bestpractical:rt:3.5.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.20`,
        fs:  `cpe:2.3:a:spreecommerce:spree:0.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.20:rc1`,
        fs:  `cpe:2.3:a:invisionpower:ip.nexus:1.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.20:rc2`,
        fs:  `cpe:2.3:a:atlassian:jira:6.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.20:rc3`,
        fs:  `cpe:2.3:a:php:php:4.1.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.20:rc4`,
        fs:  `cpe:2.3:a:moodle:moodle:2.7.7:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.20:rc5`,
        fs:  `cpe:2.3:h:siemens:scalance_x204irt_pro:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.20:rc6`,
        fs:  `cpe:2.3:a:isaac_sukin:shorten:6.x-1.x:dev:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.21`,
        fs:  `cpe:2.3:a:fenrir-inc:sleipnir_mobile:2.5.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.21:rc1`,
        fs:  `cpe:2.3:a:axel_jung:js_css_optimizer:0.2.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.21:rc2`,
        fs:  `cpe:2.3:a:rocomotion:pplog:2.47:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.22`,
        fs:  `cpe:2.3:a:apache:knox:0.3.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.22:rc1`,
        fs:  `cpe:2.3:a:kernel:linux-pam:1.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.22:rc2`,
        fs:  `cpe:2.3:a:acquia:mollom:6.x-2.8:*:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.23`,
        fs:  `cpe:2.3:a:eucalyptus:eucalyptus:4.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.1.23:rc1`,
        fs:  `cpe:2.3:a:secunia:csi_agent:7.0.0.21:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.2:b1`,
        fs:  `cpe:2.3:a:microsoft:netmeeting:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.2:b2`,
        fs:  `cpe:2.3:o:linux:linux_kernel:2.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.2:rc1`,
        fs:  `cpe:2.3:a:digium:asterisk:c.1.8.0:-:business:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.2:rc2`,
        fs:  `cpe:2.3:a:actividentity:actividentity_device_installer:2.2:*:*:*:*:*:x64:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.2:rc3`,
        fs:  `cpe:2.3:a:pidgin:libpurple:2.5.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.2.0`,
        fs:  `cpe:2.3:a:3com:tippingpoint_ips_tos:2.2.0.6504:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.2.1`,
        fs:  `cpe:2.3:a:hitachi:ucosminexus_developer:07-03-03:-:pro:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.2.2`,
        fs:  `cpe:2.3:a:netiq:appmanager_web_management_server:7.0.41039.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.2.3`,
        fs:  `cpe:2.3:a:moinejf:abcm2ps:5.9.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.2.4`,
        fs:  `cpe:2.3:a:cisco:content_security_management_appliance:9.6.6-068:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.2.5`,
        fs:  `cpe:2.3:a:idrt:marvin_teaches_fingerspelling_ii:1.0.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.2.6`,
        fs:  `cpe:2.3:a:splunk:splunk:6.0.5:*:*:*:enterprise:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.2.7`,
        fs:  `cpe:2.3:a:lsoft:listserv:1.8d:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.2.8`,
        fs:  `cpe:2.3:a:web2py:web2py:1.79.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.2.9`,
        fs:  `cpe:2.3:a:forgerock:openam:11.0.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.2.10`,
        fs:  `cpe:2.3:a:rubyonrails:ruby_on_rails:4.0.10:rc1:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.2.11`,
        fs:  `cpe:2.3:a:rom_walton:boinc:7.0.46:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.2.12`,
        fs:  `cpe:2.3:a:atlassian:jira:7.2.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.3:b1`,
        fs:  `cpe:2.3:a:usr:faxtools:5.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.3:rc1`,
        fs:  `cpe:2.3:a:apache:tomcat:5.0.27:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.3:rc2`,
        fs:  `cpe:2.3:o:yamaha:rt107e:8.03.26:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.3.0`,
        fs:  `cpe:2.3:a:microsoft:office_infopath:2003:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.3.1`,
        fs:  `cpe:2.3:a:isc:bind:9.2.0:rc4:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.3.2`,
        fs:  `cpe:2.3:a:cmcm:cm_backup_-restorecloudphoto:1.1.0.135:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.3.3`,
        fs:  `cpe:2.3:a:reviewboard:review_board:1.7.11:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.3.4`,
        fs:  `cpe:2.3:a:goahead:goahead_webserver:2.1.4:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.3.5`,
        fs:  `cpe:2.3:a:flowplayer:flowplayer_flash:3.1.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.3.6`,
        fs:  `cpe:2.3:a:boson:netsim:8.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.3.7`,
        fs:  `cpe:2.3:a:alienvault:open_source_security_information_management:3.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.3.8`,
        fs:  `cpe:2.3:a:jetaudio:jetaudio:8.0.6:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.3.9`,
        fs:  `cpe:2.3:a:b2evolution:b2evolution:5.2.2:rc2:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.3.10`,
        fs:  `cpe:2.3:a:quiz_module_project:quiz:6.x-4.0:beta6:*:*:*:drupal:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.3.11`,
        fs:  `cpe:2.3:a:ibm:websphere_transformation_extender:8.4.1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.4:b1`,
        fs:  `cpe:2.3:a:photogallerycreator:flash-album-gallery:0.32:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.4:b2`,
        fs:  `cpe:2.3:a:dave_airlie:pam_smb:1.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.4:b3`,
        fs:  `cpe:2.3:o:codeaurora:android-msm:3.10.38:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.4:rc1`,
        fs:  `cpe:2.3:a:fedoraproject:389_directory_server:1.2.8.1:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.4.0`,
        fs:  `cpe:2.3:a:nodejs:node.js:8.6.0:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.4.1`,
        fs:  `cpe:2.3:a:katello:katello:1.5.0-14:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.4.2`,
        fs:  `cpe:2.3:h:tp-link:tl-sc3171:-:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.4.3`,
        fs:  `cpe:2.3:a:bundler:bundler:1.3.3:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.4.4`,
        fs:  `cpe:2.3:o:hp:procurve_switch_software:r.11.50:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.4.5`,
        fs:  `cpe:2.3:a:lyesoft:andexplorer:1.2:*:*:*:*:android:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.4.6`,
        fs:  `cpe:2.3:a:commentluv:commentluv:2.90.8.2:*:*:*:*:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.4.7`,
        fs:  `cpe:2.3:a:libraw:libraw:0.15.2:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.4.8`,
        fs:  `cpe:2.3:a:debian:adequate:0.8:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.5:b1`,
        fs:  `cpe:2.3:a:wu-rating_project:wu-rating:1.0_12319:-:-:*:-:wordpress:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.5:b2`,
        fs:  `cpe:2.3:a:fork-cms:fork_cms:2.6.9:*:*:*:*:*:*:*`,
    },
     
    {
        uri: `cpe:/a:apple:cups:1.5:rc1`,
        fs:  `cpe:2.3:a:google:chrome:0.1.40.1:*:*:*:*:*:*:*`,
   