from ..dojo_test_case import DojoTestCase
from dojo.models import Test
from dojo.tools.github_vulnerability.parser import GithubVulnerabilityParser


class TestGithubVulnerabilityParser(DojoTestCase):
    def test_parse_file_with_no_vuln_has_no_findings(self):
        """sample with zero vulnerability"""
        testfile = open("unittests/scans/github_vulnerability/github-0-vuln.json")
        parser = GithubVulnerabilityParser()
        findings = parser.get_findings(testfile, Test())
        self.assertEqual(0, len(findings))

    def test_parse_file_with_one_vuln_has_one_findings(self):
        """sample with one vulnerability"""
        testfile = open("unittests/scans/github_vulnerability/github-1-vuln.json")
        parser = GithubVulnerabilityParser()
        findings = parser.get_findings(testfile, Test())
        self.assertEqual(1, len(findings))
        for finding in findings:
            finding.clean()

        with self.subTest(i=0):
            finding = findings[0]
            self.assertEqual(finding.title, "Critical severity vulnerability that affects package")
            self.assertEqual(
                finding.description,
                "This is a sample description for sample description from Github API.",
            )
            self.assertEqual(finding.severity, "Critical")
            self.assertEqual(finding.component_name, "package")
            self.assertEqual(finding.unique_id_from_tool, "aabbccddeeff1122334401")

    def test_parse_file_with_multiple_vuln_has_multiple_findings(self):
        """sample with five vulnerability"""
        testfile = open("unittests/scans/github_vulnerability/github-5-vuln.json")
        parser = GithubVulnerabilityParser()
        findings = parser.get_findings(testfile, Test())
        self.assertEqual(5, len(findings))

    def test_parse_file_issue2984(self):
        testfile = open("unittests/scans/github_vulnerability/github_issue2984.json")
        parser = GithubVulnerabilityParser()
        findings = parser.get_findings(testfile, Test())
        self.assertEqual(4, len(findings))
        for finding in findings:
            finding.clean()

        with self.subTest(i=0):
            finding = findings[0]
            self.assertEqual(finding.title, "XXXXXXXXXXXXXXX")
            self.assertEqual(finding.severity, "Medium")
            self.assertEqual(finding.unique_id_from_tool, "xxxxxxxxx")
        with self.subTest(i=1):
            finding = findings[1]
            self.assertEqual(finding.title, "AMSVNASCMASNCADNNJSADC")
            self.assertEqual(finding.severity, "Medium")
            self.assertEqual(finding.unique_id_from_tool, "AFDSFSDAFSDASFDAFSDASFD=")
        with self.subTest(i=3):
            finding = findings[3]
            self.assertEqual(finding.title, "SDKPKÁSMNMKSDANJDOPASJOKNDOSAJ")
            self.assertEqual(finding.severity, "Medium")
            self.assertEqual(finding.unique_id_from_tool, "DASFMMFKLNKDSAKFSDLANJKKFDSNJSAKDFNJKDFS=")

    def test_parse_file_search(self):
        testfile = open("unittests/scans/github_vulnerability/github_search.json")
        parser = GithubVulnerabilityParser()
        findings = parser.get_findings(testfile, Test())
        self.assertEqual(2, len(findings))
        for finding in findings:
            finding.clean()

        with self.subTest(i=0):
            finding = findings[0]
            self.assertEqual(finding.title, "Deserialization of Untrusted Data in Log4j")
            self.assertEqual(finding.severity, "Critical")
            self.assertEqual(len(finding.unsaved_vulnerability_ids), 2)
            self.assertEqual(finding.unsaved_vulnerability_ids[0], "GHSA-2qrg-x229-3v8q")
            self.assertEqual(finding.unsaved_vulnerability_ids[1], "CVE-2019-17571")
            self.assertEqual(finding.component_name, "log4j:log4j")
            self.assertEqual(finding.unique_id_from_tool, "MDI4OlJlcG9zaXRvcnlWdWxuZXJhYmlsaXR5QWxlcnQyMDg2Nzc5NzY=")
        with self.subTest(i=1):
            finding = findings[1]
            self.assertEqual(finding.title, "Deserialization of Untrusted Data in Log4j")
            self.assertEqual(finding.severity, "Critical")
            self.assertEqual(len(finding.unsaved_vulnerability_ids), 2)
            self.assertEqual(finding.unsaved_vulnerability_ids[0], "GHSA-2qrg-x229-3v8q")
            self.assertEqual(finding.unsaved_vulnerability_ids[1], "CVE-2019-17571")
            self.assertEqual(finding.component_name, "log4j:log4j")
            self.assertEqual(finding.unique_id_from_tool, "MDI4OlJlcG9zaXRvcnlWdWxuZXJhYmlsaXR5QWxlcnQ1NTE5NTI2OTM=")

    def test_parse_file_search2(self):
        """Search result with more data/attributes"""
        testfile = open("unittests/scans/github_vulnerability/github_search2.json")
        parser = GithubVulnerabilityParser()
        findings = parser.get_findings(testfile, Test())
        self.assertEqual(2, len(findings))
        for finding in findings:
            finding.clean()

        with self.subTest(i=0):
            finding = findings[0]
            self.assertEqual(finding.title, "Deserialization of Untrusted Data in Log4j")
            self.assertEqual(finding.severity, "Critical")
            self.assertEqual(len(finding.unsaved_vulnerability_ids), 2)
            self.assertEqual(finding.unsaved_vulnerability_ids[0], "GHSA-2qrg-x229-3v8q")
            self.assertEqual(finding.unsaved_vulnerability_ids[1], "CVE-2019-17571")
            self.assertEqual(finding.component_name, "log4j:log4j")
            self.assertEqual(finding.unique_id_from_tool, "MDI4OlJlcG9zaXRvcnlWdWxuZXJhYmlsaXR5QWxlcnQyMDg2Nzc5NzY=")
        with self.subTest(i=1):
            finding = findings[1]
            self.assertEqual(finding.title, "Deserialization of Untrusted Data in Log4j")
            self.assertEqual(finding.severity, "Critical")
            self.assertEqual(len(finding.unsaved_vulnerability_ids), 2)
            self.assertEqual(finding.unsaved_vulnerability_ids[0], "GHSA-2qrg-x229-3v8q")
            self.assertEqual(finding.unsaved_vulnerability_ids[1], "CVE-2019-17571")
            self.assertEqual(finding.component_name, "log4j:log4j")
            self.assertEqual(finding.unique_id_from_tool, "MDI4OlJlcG9zaXRvcnlWdWxuZXJhYmlsaXR5QWxlcnQ1NTE5NTI2OTM=")

    def test_parse_file_search3(self):
        """Search result with more data/attributes"""
        testfile = open("unittests/scans/github_vulnerability/github_search3.json")
        parser = GithubVulnerabilityParser()
        findings = parser.get_findings(testfile, Test())
        self.assertEqual(2, len(findings))
        for finding in findings:
            finding.clean()

        with self.subTest(i=0):
            finding = findings[0]
            self.assertEqual(finding.title, "Deserialization of Untrusted Data in Log4j")
            self.assertEqual(finding.severity, "Critical")
            self.assertEqual(len(finding.unsaved_vulnerability_ids), 2)
            self.assertEqual(finding.unsaved_vulnerability_ids[0], "GHSA-2qrg-x229-3v8q")
            self.assertEqual(finding.unsaved_vulnerability_ids[1], "CVE-2019-17571")
            self.assertEqual(finding.component_name, "log4j:log4j")
            self.assertEqual(finding.cvssv3, "CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H")
            self.assertEqual(finding.file_path, "gogoph-crawler/pom.xml")
            self.assertEqual(finding.unique_id_from_tool, "MDI4OlJlcG9zaXRvcnlWdWxuZXJhYmlsaXR5QWxlcnQyMDg2Nzc5NzY=")
        with self.subTest(i=1):
            finding = findings[1]
            self.assertEqual(finding.title, "Deserialization of Untrusted Data in Log4j")
            self.assertEqual(finding.severity, "Critical")
            self.assertEqual(len(finding.unsaved_vulnerability_ids), 2)
            self.assertEqual(finding.unsaved_vulnerability_ids[0], "GHSA-2qrg-x229-3v8q")
            self.assertEqual(finding.unsaved_vulnerability_ids[1], "CVE-2019-17571")
            self.assertEqual(finding.component_name, "log4j:log4j")
            self.assertEqual(finding.cvssv3, "CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H")
            self.assertEqual(finding.file_path, "gogoph/pom.xml")
            self.assertEqual(finding.unique_id_from_tool, "MDI4OlJlcG9zaXRvcnlWdWxuZXJhYmlsaXR5QWxlcnQ1NTE5NTI2OTM=")
