import json

from dojo.models import Finding


class HarborVulnerabilityParser(object):
    """
    Read JSON data from Harbor compatible format and import it to DefectDojo
    """

    def get_scan_types(self):
        return ["Harbor Vulnerability Scan"]

    def get_label_for_scan_types(self, scan_type):
        return scan_type  # no custom label for now

    def get_description_for_scan_types(self, scan_type):
        return "Import vulnerabilities from Harbor API."

    def get_findings(self, filename, test):

        tree = filename.read()
        try:
            data = json.loads(str(tree, "utf-8"))
        except:
            data = json.loads(tree)

        # When doing dictionary, we can detect duplications
        dupes = dict()

        # To be compatible with update in version
        try:
            vulnerability = data[next(iter(data.keys()))]["vulnerabilities"]
        except (KeyError, StopIteration):
            return list()

        # Early exit if empty
        if vulnerability is None:
            return list()

        for item in vulnerability:

            id = item.get('id')
            package_name = item.get('package')
            package_version = item.get('version')
            description = item.get('description', 'No description found')
            severity = item.get('severity')
            fix_version = item.get('fix_version')
            links = item.get('links')

            title = f'{id} - {package_name} ({package_version})'
            severity = transpose_severity(severity)

            if fix_version:
                mitigation = f'Upgrade {package_name} to version {fix_version}'
            else:
                mitigation = None

            if links:
                references = ''
                for link in links:
                    references += f'{link}\n'
            else:
                references = None

            if id and id.startswith('CVE'):
                vulnerability_id = id
            else:
                vulnerability_id = None

            dupe_key = title

            if dupe_key in dupes:
                find = dupes[dupe_key]
            else:
                dupes[dupe_key] = True

                find = Finding(
                    title=title,
                    test=test,
                    description=description,
                    severity=severity,
                    mitigation=mitigation,
                    references=references,
                    static_finding=True,
                    component_name=package_name,
                    component_version=package_version,
                )
                if vulnerability_id:
                    find.unsaved_vulnerability_ids = [vulnerability_id]
                dupes[dupe_key] = find

        return list(dupes.values())


def transpose_severity(severity):
    if severity in Finding.SEVERITIES:
        return severity
    else:
        return 'Info'
