/*
 This file is part of TALER
 (C) 2016 GNUnet e.V.

 TALER is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 TALER is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
System.register(["./wallet", "./http", "./checkable", "./types", "./chromeBadge", "./logging", "./query"], function (exports_1, context_1) {
    "use strict";
    var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
        return new (P || (P = Promise))(function (resolve, reject) {
            function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
            function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
            function step(result) { result.done ? resolve(result.value) : new P(function (resolve) { resolve(result.value); }).then(fulfilled, rejected); }
            step((generator = generator.apply(thisArg, _arguments)).next());
        });
    };
    var __moduleName = context_1 && context_1.id;
    function makeHandlers(db, wallet) {
        return {
            ["balances"]: function (detail, sender) {
                return wallet.getBalances();
            },
            ["dump-db"]: function (detail, sender) {
                return exportDb(db);
            },
            ["get-tab-cookie"]: function (detail, sender) {
                if (!sender || !sender.tab || !sender.tab.id) {
                    return Promise.resolve();
                }
                let id = sender.tab.id;
                let info = paymentRequestCookies[id];
                delete paymentRequestCookies[id];
                return Promise.resolve(info);
            },
            ["ping"]: function (detail, sender) {
                return Promise.resolve();
            },
            ["reset"]: function (detail, sender) {
                if (db) {
                    let tx = db.transaction(Array.from(db.objectStoreNames), 'readwrite');
                    for (let i = 0; i < db.objectStoreNames.length; i++) {
                        tx.objectStore(db.objectStoreNames[i]).clear();
                    }
                }
                deleteDb();
                chrome.browserAction.setBadgeText({ text: "" });
                console.log("reset done");
                // Response is synchronous
                return Promise.resolve({});
            },
            ["create-reserve"]: function (detail, sender) {
                const d = {
                    exchange: detail.exchange,
                    amount: detail.amount,
                };
                const req = wallet_1.CreateReserveRequest.checked(d);
                return wallet.createReserve(req);
            },
            ["confirm-reserve"]: function (detail, sender) {
                // TODO: make it a checkable
                const d = {
                    reservePub: detail.reservePub
                };
                const req = wallet_1.ConfirmReserveRequest.checked(d);
                return wallet.confirmReserve(req);
            },
            ["confirm-pay"]: function (detail, sender) {
                let offer;
                try {
                    offer = wallet_1.OfferRecord.checked(detail.offer);
                }
                catch (e) {
                    if (e instanceof checkable_1.Checkable.SchemaError) {
                        console.error("schema error:", e.message);
                        return Promise.resolve({
                            error: "invalid contract",
                            hint: e.message,
                            detail: detail
                        });
                    }
                    else {
                        throw e;
                    }
                }
                return wallet.confirmPay(offer);
            },
            ["check-pay"]: function (detail, sender) {
                let offer;
                try {
                    offer = wallet_1.OfferRecord.checked(detail.offer);
                }
                catch (e) {
                    if (e instanceof checkable_1.Checkable.SchemaError) {
                        console.error("schema error:", e.message);
                        return Promise.resolve({
                            error: "invalid contract",
                            hint: e.message,
                            detail: detail
                        });
                    }
                    else {
                        throw e;
                    }
                }
                return wallet.checkPay(offer);
            },
            ["execute-payment"]: function (detail, sender) {
                if (sender.tab && sender.tab.id) {
                    rateLimitCache[sender.tab.id]++;
                    if (rateLimitCache[sender.tab.id] > 10) {
                        console.warn("rate limit for execute payment exceeded");
                        let msg = {
                            error: "rate limit exceeded for execute-payment",
                            rateLimitExceeded: true,
                            hint: "Check for redirect loops",
                        };
                        return Promise.resolve(msg);
                    }
                }
                return wallet.executePayment(detail.H_contract);
            },
            ["exchange-info"]: function (detail) {
                if (!detail.baseUrl) {
                    return Promise.resolve({ error: "bad url" });
                }
                return wallet.updateExchangeFromUrl(detail.baseUrl);
            },
            ["hash-contract"]: function (detail) {
                if (!detail.contract) {
                    return Promise.resolve({ error: "contract missing" });
                }
                return wallet.hashContract(detail.contract).then((hash) => {
                    return { hash };
                });
            },
            ["put-history-entry"]: function (detail) {
                if (!detail.historyEntry) {
                    return Promise.resolve({ error: "historyEntry missing" });
                }
                return wallet.putHistory(detail.historyEntry);
            },
            ["save-offer"]: function (detail) {
                let offer = detail.offer;
                if (!offer) {
                    return Promise.resolve({ error: "offer missing" });
                }
                console.log("handling safe-offer");
                return wallet.saveOffer(offer);
            },
            ["reserve-creation-info"]: function (detail, sender) {
                if (!detail.baseUrl || typeof detail.baseUrl !== "string") {
                    return Promise.resolve({ error: "bad url" });
                }
                let amount = types_1.AmountJson.checked(detail.amount);
                return wallet.getReserveCreationInfo(detail.baseUrl, amount);
            },
            ["check-repurchase"]: function (detail, sender) {
                let contract = types_2.Contract.checked(detail.contract);
                return wallet.checkRepurchase(contract);
            },
            ["get-history"]: function (detail, sender) {
                // TODO: limit history length
                return wallet.getHistory();
            },
            ["get-offer"]: function (detail, sender) {
                return wallet.getOffer(detail.offerId);
            },
            ["get-exchanges"]: function (detail, sender) {
                return wallet.getExchanges();
            },
            ["get-reserves"]: function (detail, sender) {
                if (typeof detail.exchangeBaseUrl !== "string") {
                    return Promise.reject(Error("exchangeBaseUrl missing"));
                }
                return wallet.getReserves(detail.exchangeBaseUrl);
            },
            ["get-coins"]: function (detail, sender) {
                if (typeof detail.exchangeBaseUrl !== "string") {
                    return Promise.reject(Error("exchangBaseUrl missing"));
                }
                return wallet.getCoins(detail.exchangeBaseUrl);
            },
            ["get-precoins"]: function (detail, sender) {
                if (typeof detail.exchangeBaseUrl !== "string") {
                    return Promise.reject(Error("exchangBaseUrl missing"));
                }
                return wallet.getPreCoins(detail.exchangeBaseUrl);
            },
            ["get-denoms"]: function (detail, sender) {
                if (typeof detail.exchangeBaseUrl !== "string") {
                    return Promise.reject(Error("exchangBaseUrl missing"));
                }
                return wallet.getDenoms(detail.exchangeBaseUrl);
            },
            ["refresh-coin"]: function (detail, sender) {
                if (typeof detail.coinPub !== "string") {
                    return Promise.reject(Error("coinPub missing"));
                }
                return wallet.refresh(detail.coinPub);
            },
            ["payment-failed"]: function (detail, sender) {
                // For now we just update exchanges (maybe the exchange did something
                // wrong and the keys were messed up).
                // FIXME: in the future we should look at what actually went wrong.
                console.error("payment reported as failed");
                wallet.updateExchanges();
                return Promise.resolve();
            },
            ["payment-succeeded"]: function (detail, sender) {
                let contractHash = detail.contractHash;
                if (!contractHash) {
                    return Promise.reject(Error("contractHash missing"));
                }
                return wallet.paymentSucceeded(contractHash);
            },
        };
    }
    function dispatch(handlers, req, sender, sendResponse) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!(req.type in handlers)) {
                console.error(`Request type ${JSON.stringify(req)} unknown, req ${req.type}`);
                try {
                    sendResponse({ error: "request unknown" });
                }
                catch (e) {
                }
            }
            try {
                const p = handlers[req.type](req.detail, sender);
                let r = yield p;
                try {
                    sendResponse(r);
                }
                catch (e) {
                }
            }
            catch (e) {
                console.log(`exception during wallet handler for '${req.type}'`);
                console.log("request", req);
                console.error(e);
                try {
                    sendResponse({
                        error: "exception",
                        hint: e.message,
                        stack: e.stack.toString()
                    });
                }
                catch (e) {
                }
            }
        });
    }
    function handleHttpPayment(headerList, url, tabId) {
        const headers = {};
        for (let kv of headerList) {
            if (kv.value) {
                headers[kv.name.toLowerCase()] = kv.value;
            }
        }
        const contractUrl = headers["x-taler-contract-url"];
        if (contractUrl !== undefined) {
            paymentRequestCookies[tabId] = { type: "fetch", contractUrl };
            return;
        }
        const contractHash = headers["x-taler-contract-hash"];
        if (contractHash !== undefined) {
            const payUrl = headers["x-taler-pay-url"];
            if (payUrl === undefined) {
                console.log("malformed 402, X-Taler-Pay-Url missing");
                return;
            }
            // Offer URL is optional
            const offerUrl = headers["x-taler-offer-url"];
            paymentRequestCookies[tabId] = {
                type: "execute",
                offerUrl,
                payUrl,
                contractHash
            };
            return;
        }
        // looks like it's not a taler request, it might be
        // for a different payment system (or the shop is buggy)
        console.log("ignoring non-taler 402 response");
    }
    function handleBankRequest(wallet, headerList, url, tabId) {
        const headers = {};
        for (let kv of headerList) {
            if (kv.value) {
                headers[kv.name.toLowerCase()] = kv.value;
            }
        }
        const reservePub = headers["x-taler-reserve-pub"];
        if (reservePub !== undefined) {
            console.log(`confirming reserve ${reservePub} via 201`);
            wallet.confirmReserve({ reservePub });
            return;
        }
        const amount = headers["x-taler-amount"];
        if (amount) {
            let callbackUrl = headers["x-taler-callback-url"];
            if (!callbackUrl) {
                console.log("202 not understood (X-Taler-Callback-Url missing)");
                return;
            }
            let amountParsed;
            try {
                amountParsed = JSON.parse(amount);
            }
            catch (e) {
                let uri = URI(chrome.extension.getURL("/src/pages/error.html"));
                let p = {
                    message: `Can't parse amount ("${amount}"): ${e.message}`,
                };
                let redirectUrl = uri.query(p).href();
                return { redirectUrl };
            }
            let wtTypes = headers["x-taler-wt-types"];
            if (!wtTypes) {
                console.log("202 not understood (X-Taler-Wt-Types missing)");
                return;
            }
            let params = {
                amount: amount,
                callback_url: URI(callbackUrl)
                    .absoluteTo(url),
                bank_url: url,
                wt_types: wtTypes,
            };
            let uri = URI(chrome.extension.getURL("/src/pages/confirm-create-reserve.html"));
            let redirectUrl = uri.query(params).href();
            return { redirectUrl };
        }
        // no known headers found, not a taler request ...
    }
    function clearRateLimitCache() {
        rateLimitCache = {};
    }
    function wxMain() {
        return __awaiter(this, void 0, void 0, function* () {
            window.onerror = (m, source, lineno, colno, error) => {
                logging.record("error", m + error, undefined, source || "(unknown)", lineno || 0, colno || 0);
            };
            chrome.browserAction.setBadgeText({ text: "" });
            exports_1("badge", badge = new chromeBadge_1.ChromeBadge());
            chrome.tabs.query({}, function (tabs) {
                for (let tab of tabs) {
                    if (!tab.url || !tab.id) {
                        return;
                    }
                    let uri = URI(tab.url);
                    if (uri.protocol() == "http" || uri.protocol() == "https") {
                        console.log("injecting into existing tab", tab.id);
                        chrome.tabs.executeScript(tab.id, { file: "/src/vendor/URI.js" });
                        chrome.tabs.executeScript(tab.id, { file: "/src/taler-wallet-lib.js" });
                        chrome.tabs.executeScript(tab.id, { file: "/src/content_scripts/notify.js" });
                        let code = `
          if (("taler" in window) || document.documentElement.getAttribute("data-taler-nojs")) {
            document.dispatchEvent(new Event("taler-probe-result"));
          }
        `;
                        chrome.tabs.executeScript(tab.id, { code, runAt: "document_idle" });
                    }
                }
            });
            const tabTimers = {};
            chrome.tabs.onRemoved.addListener((tabId, changeInfo) => {
                let tt = tabTimers[tabId] || [];
                for (let t of tt) {
                    chrome.extension.getBackgroundPage().clearTimeout(t);
                }
            });
            chrome.tabs.onUpdated.addListener((tabId, changeInfo) => {
                if (changeInfo.status != 'complete') {
                    return;
                }
                const timers = [];
                const addRun = (dt) => {
                    const id = chrome.extension.getBackgroundPage().setTimeout(run, dt);
                    timers.push(id);
                };
                const run = () => {
                    timers.shift();
                    chrome.tabs.get(tabId, (tab) => {
                        if (chrome.runtime.lastError) {
                            return;
                        }
                        if (!tab.url || !tab.id) {
                            return;
                        }
                        let uri = URI(tab.url);
                        if (!(uri.protocol() == "http" || uri.protocol() == "https")) {
                            return;
                        }
                        let code = `
          if (("taler" in window) || document.documentElement.getAttribute("data-taler-nojs")) {
            document.dispatchEvent(new Event("taler-probe-result"));
          }
        `;
                        chrome.tabs.executeScript(tab.id, { code, runAt: "document_start" });
                    });
                };
                addRun(0);
                addRun(50);
                addRun(300);
                addRun(1000);
                addRun(2000);
                addRun(4000);
                addRun(8000);
                addRun(16000);
                tabTimers[tabId] = timers;
            });
            chrome.extension.getBackgroundPage().setInterval(clearRateLimitCache, 5000);
            let db;
            try {
                db = yield openTalerDb();
            }
            catch (e) {
                console.error("could not open database", e);
                return;
            }
            let http = new http_1.BrowserHttpLib();
            let notifier = new ChromeNotifier();
            console.log("setting wallet");
            exports_1("wallet", wallet = new wallet_1.Wallet(db, http, badge, notifier));
            // Handlers for messages coming directly from the content
            // script on the page
            let handlers = makeHandlers(db, wallet);
            chrome.runtime.onMessage.addListener((req, sender, sendResponse) => {
                dispatch(handlers, req, sender, sendResponse);
                return true;
            });
            // Handlers for catching HTTP requests
            chrome.webRequest.onHeadersReceived.addListener((details) => {
                if (details.statusCode == 402) {
                    console.log(`got 402 from ${details.url}`);
                    return handleHttpPayment(details.responseHeaders || [], details.url, details.tabId);
                }
                else if (details.statusCode == 202) {
                    return handleBankRequest(wallet, details.responseHeaders || [], details.url, details.tabId);
                }
            }, { urls: ["<all_urls>"] }, ["responseHeaders", "blocking"]);
        });
    }
    exports_1("wxMain", wxMain);
    /**
     * Return a promise that resolves
     * to the taler wallet db.
     */
    function openTalerDb() {
        return new Promise((resolve, reject) => {
            const req = indexedDB.open(DB_NAME, DB_VERSION);
            req.onerror = (e) => {
                reject(e);
            };
            req.onsuccess = (e) => {
                resolve(req.result);
            };
            req.onupgradeneeded = (e) => {
                const db = req.result;
                console.log("DB: upgrade needed: oldVersion = " + e.oldVersion);
                switch (e.oldVersion) {
                    case 0:
                        for (let n in wallet_2.Stores) {
                            if (wallet_2.Stores[n] instanceof query_1.Store) {
                                let si = wallet_2.Stores[n];
                                const s = db.createObjectStore(si.name, si.storeParams);
                                for (let indexName in si) {
                                    if (si[indexName] instanceof query_1.Index) {
                                        let ii = si[indexName];
                                        s.createIndex(ii.indexName, ii.keyPath);
                                    }
                                }
                            }
                        }
                        break;
                    default:
                        if (e.oldVersion != DB_VERSION) {
                            window.alert("Incompatible wallet dababase version, please reset" +
                                " db.");
                            chrome.browserAction.setBadgeText({ text: "err" });
                            chrome.browserAction.setBadgeBackgroundColor({ color: "#F00" });
                            throw Error("incompatible DB");
                        }
                        break;
                }
            };
        });
    }
    function exportDb(db) {
        let dump = {
            name: db.name,
            version: db.version,
            stores: {},
        };
        return new Promise((resolve, reject) => {
            let tx = db.transaction(Array.from(db.objectStoreNames));
            tx.addEventListener("complete", () => {
                resolve(dump);
            });
            for (let i = 0; i < db.objectStoreNames.length; i++) {
                let name = db.objectStoreNames[i];
                let storeDump = {};
                dump.stores[name] = storeDump;
                let store = tx.objectStore(name)
                    .openCursor()
                    .addEventListener("success", (e) => {
                    let cursor = e.target.result;
                    if (cursor) {
                        storeDump[cursor.key] = cursor.value;
                        cursor.continue();
                    }
                });
            }
        });
    }
    function deleteDb() {
        indexedDB.deleteDatabase(DB_NAME);
    }
    var wallet_1, http_1, checkable_1, types_1, types_2, chromeBadge_1, logging, DB_NAME, DB_VERSION, wallet_2, query_1, ChromeNotifier, paymentRequestCookies, wallet, badge, log, rateLimitCache;
    return {
        setters: [
            function (wallet_1_1) {
                wallet_1 = wallet_1_1;
                wallet_2 = wallet_1_1;
            },
            function (http_1_1) {
                http_1 = http_1_1;
            },
            function (checkable_1_1) {
                checkable_1 = checkable_1_1;
            },
            function (types_1_1) {
                types_1 = types_1_1;
                types_2 = types_1_1;
            },
            function (chromeBadge_1_1) {
                chromeBadge_1 = chromeBadge_1_1;
            },
            function (logging_1) {
                logging = logging_1;
            },
            function (query_1_1) {
                query_1 = query_1_1;
            }
        ],
        execute: function () {/*
             This file is part of TALER
             (C) 2016 GNUnet e.V.
            
             TALER is free software; you can redistribute it and/or modify it under the
             terms of the GNU General Public License as published by the Free Software
             Foundation; either version 3, or (at your option) any later version.
            
             TALER is distributed in the hope that it will be useful, but WITHOUT ANY
             WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
             A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
            
             You should have received a copy of the GNU General Public License along with
             TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
             */
            "use strict";
            DB_NAME = "taler";
            DB_VERSION = 12;
            ChromeNotifier = class ChromeNotifier {
                constructor() {
                    this.ports = [];
                    chrome.runtime.onConnect.addListener((port) => {
                        console.log("got connect!");
                        this.ports.push(port);
                        port.onDisconnect.addListener(() => {
                            let i = this.ports.indexOf(port);
                            if (i >= 0) {
                                this.ports.splice(i, 1);
                            }
                            else {
                                console.error("port already removed");
                            }
                        });
                    });
                }
                notify() {
                    for (let p of this.ports) {
                        p.postMessage({ notify: true });
                    }
                }
            };
            /**
             * Mapping from tab ID to payment information (if any).
             */
            paymentRequestCookies = {};
            // Useful for debugging ...
            exports_1("wallet", wallet = undefined);
            exports_1("badge", badge = undefined);
            exports_1("log", log = logging.log);
            // Rate limit cache for executePayment operations, to break redirect loops
            rateLimitCache = {};
        }
    };
});
//# sourceMappingURL=wxBackend.js.map