/*
 This file is part of TALER
 (C) 2015 GNUnet e.V.

 TALER is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 TALER is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
System.register(["./types", "./http", "./query", "./checkable", "./helpers", "./cryptoApi"], function (exports_1, context_1) {
    "use strict";
    var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
        var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
        if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
        else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
        return c > 3 && r && Object.defineProperty(target, key, r), r;
    };
    var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
        return new (P || (P = Promise))(function (resolve, reject) {
            function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
            function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
            function step(result) { result.done ? resolve(result.value) : new P(function (resolve) { resolve(result.value); }).then(fulfilled, rejected); }
            step((generator = generator.apply(thisArg, _arguments)).next());
        });
    };
    var __moduleName = context_1 && context_1.id;
    function setTimeout(f, t) {
        return chrome.extension.getBackgroundPage().setTimeout(f, t);
    }
    function isWithdrawableDenom(d) {
        const now_sec = (new Date).getTime() / 1000;
        const stamp_withdraw_sec = helpers_1.getTalerStampSec(d.stampExpireWithdraw);
        const stamp_start_sec = helpers_1.getTalerStampSec(d.stampStart);
        // Withdraw if still possible to withdraw within a minute
        if ((stamp_withdraw_sec + 60 > now_sec) && (now_sec >= stamp_start_sec)) {
            return true;
        }
        return false;
    }
    function selectCoins(cds, paymentAmount, depositFeeLimit) {
        if (cds.length == 0) {
            return undefined;
        }
        // Sort by ascending deposit fee
        cds.sort((o1, o2) => types_1.Amounts.cmp(o1.denom.feeDeposit, o2.denom.feeDeposit));
        let currency = cds[0].denom.value.currency;
        let cdsResult = [];
        let accFee = types_1.Amounts.getZero(currency);
        let accAmount = types_1.Amounts.getZero(currency);
        let isBelowFee = false;
        let coversAmount = false;
        let coversAmountWithFee = false;
        for (let i = 0; i < cds.length; i++) {
            let { coin, denom } = cds[i];
            if (coin.suspended) {
                continue;
            }
            if (coin.dirty) {
                continue;
            }
            if (types_1.Amounts.cmp(denom.feeDeposit, coin.currentAmount) >= 0) {
                continue;
            }
            cdsResult.push(cds[i]);
            accFee = types_1.Amounts.add(denom.feeDeposit, accFee).amount;
            accAmount = types_1.Amounts.add(coin.currentAmount, accAmount).amount;
            coversAmount = types_1.Amounts.cmp(accAmount, paymentAmount) >= 0;
            coversAmountWithFee = types_1.Amounts.cmp(accAmount, types_1.Amounts.add(paymentAmount, denom.feeDeposit).amount) >= 0;
            isBelowFee = types_1.Amounts.cmp(accFee, depositFeeLimit) <= 0;
            if ((coversAmount && isBelowFee) || coversAmountWithFee) {
                return cdsResult;
            }
        }
        return undefined;
    }
    exports_1("selectCoins", selectCoins);
    /**
     * Get a list of denominations (with repetitions possible)
     * whose total value is as close as possible to the available
     * amount, but never larger.
     */
    function getWithdrawDenomList(amountAvailable, denoms) {
        let remaining = types_1.Amounts.copy(amountAvailable);
        const ds = [];
        denoms = denoms.filter(isWithdrawableDenom);
        denoms.sort((d1, d2) => types_1.Amounts.cmp(d2.value, d1.value));
        // This is an arbitrary number of coins
        // we can withdraw in one go.  It's not clear if this limit
        // is useful ...
        for (let i = 0; i < 1000; i++) {
            let found = false;
            for (let d of denoms) {
                let cost = types_1.Amounts.add(d.value, d.feeWithdraw).amount;
                if (types_1.Amounts.cmp(remaining, cost) < 0) {
                    continue;
                }
                found = true;
                remaining = types_1.Amounts.sub(remaining, cost).amount;
                ds.push(d);
                break;
            }
            if (!found) {
                break;
            }
        }
        return ds;
    }
    var types_1, http_1, query_1, checkable_1, helpers_1, cryptoApi_1, KeysJson, CreateReserveRequest, ConfirmReserveRequest, OfferRecord, HistoryLevel, Stores, Wallet;
    return {
        setters: [
            function (types_1_1) {
                types_1 = types_1_1;
            },
            function (http_1_1) {
                http_1 = http_1_1;
            },
            function (query_1_1) {
                query_1 = query_1_1;
            },
            function (checkable_1_1) {
                checkable_1 = checkable_1_1;
            },
            function (helpers_1_1) {
                helpers_1 = helpers_1_1;
            },
            function (cryptoApi_1_1) {
                cryptoApi_1 = cryptoApi_1_1;
            }
        ],
        execute: function () {/*
             This file is part of TALER
             (C) 2015 GNUnet e.V.
            
             TALER is free software; you can redistribute it and/or modify it under the
             terms of the GNU General Public License as published by the Free Software
             Foundation; either version 3, or (at your option) any later version.
            
             TALER is distributed in the hope that it will be useful, but WITHOUT ANY
             WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
             A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
            
             You should have received a copy of the GNU General Public License along with
             TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
             */
            "use strict";
            KeysJson = class KeysJson {
            };
            __decorate([
                checkable_1.Checkable.List(checkable_1.Checkable.Value(types_1.Denomination))
            ], KeysJson.prototype, "denoms", void 0);
            __decorate([
                checkable_1.Checkable.String
            ], KeysJson.prototype, "master_public_key", void 0);
            __decorate([
                checkable_1.Checkable.Any
            ], KeysJson.prototype, "auditors", void 0);
            __decorate([
                checkable_1.Checkable.String
            ], KeysJson.prototype, "list_issue_date", void 0);
            __decorate([
                checkable_1.Checkable.Any
            ], KeysJson.prototype, "signkeys", void 0);
            __decorate([
                checkable_1.Checkable.String
            ], KeysJson.prototype, "eddsa_pub", void 0);
            __decorate([
                checkable_1.Checkable.String
            ], KeysJson.prototype, "eddsa_sig", void 0);
            KeysJson = __decorate([
                checkable_1.Checkable.Class
            ], KeysJson);
            exports_1("KeysJson", KeysJson);
            CreateReserveRequest = class CreateReserveRequest {
            };
            __decorate([
                checkable_1.Checkable.Value(types_1.AmountJson)
            ], CreateReserveRequest.prototype, "amount", void 0);
            __decorate([
                checkable_1.Checkable.String
            ], CreateReserveRequest.prototype, "exchange", void 0);
            CreateReserveRequest = __decorate([
                checkable_1.Checkable.Class
            ], CreateReserveRequest);
            exports_1("CreateReserveRequest", CreateReserveRequest);
            ConfirmReserveRequest = class ConfirmReserveRequest {
            };
            __decorate([
                checkable_1.Checkable.String
            ], ConfirmReserveRequest.prototype, "reservePub", void 0);
            ConfirmReserveRequest = __decorate([
                checkable_1.Checkable.Class
            ], ConfirmReserveRequest);
            exports_1("ConfirmReserveRequest", ConfirmReserveRequest);
            OfferRecord = class OfferRecord {
            };
            __decorate([
                checkable_1.Checkable.Value(types_1.Contract)
            ], OfferRecord.prototype, "contract", void 0);
            __decorate([
                checkable_1.Checkable.String
            ], OfferRecord.prototype, "merchant_sig", void 0);
            __decorate([
                checkable_1.Checkable.String
            ], OfferRecord.prototype, "H_contract", void 0);
            __decorate([
                checkable_1.Checkable.Number
            ], OfferRecord.prototype, "offer_time", void 0);
            __decorate([
                checkable_1.Checkable.Optional(checkable_1.Checkable.Number)
            ], OfferRecord.prototype, "id", void 0);
            OfferRecord = __decorate([
                checkable_1.Checkable.Class
            ], OfferRecord);
            exports_1("OfferRecord", OfferRecord);
            (function (HistoryLevel) {
                HistoryLevel[HistoryLevel["Trace"] = 1] = "Trace";
                HistoryLevel[HistoryLevel["Developer"] = 2] = "Developer";
                HistoryLevel[HistoryLevel["Expert"] = 3] = "Expert";
                HistoryLevel[HistoryLevel["User"] = 4] = "User";
            })(HistoryLevel || (HistoryLevel = {}));
            exports_1("HistoryLevel", HistoryLevel);
            (function (Stores) {
                class ExchangeStore extends query_1.Store {
                    constructor() {
                        super("exchanges", { keyPath: "baseUrl" });
                        this.pubKeyIndex = new query_1.Index(this, "pubKey", "masterPublicKey");
                    }
                }
                class CoinsStore extends query_1.Store {
                    constructor() {
                        super("coins", { keyPath: "coinPub" });
                        this.exchangeBaseUrlIndex = new query_1.Index(this, "exchangeBaseUrl", "exchangeBaseUrl");
                    }
                }
                class HistoryStore extends query_1.Store {
                    constructor() {
                        super("history", {
                            keyPath: "id",
                            autoIncrement: true
                        });
                        this.timestampIndex = new query_1.Index(this, "timestamp", "timestamp");
                    }
                }
                class OffersStore extends query_1.Store {
                    constructor() {
                        super("offers", {
                            keyPath: "id",
                            autoIncrement: true
                        });
                    }
                }
                class TransactionsStore extends query_1.Store {
                    constructor() {
                        super("transactions", { keyPath: "contractHash" });
                        this.repurchaseIndex = new query_1.Index(this, "repurchase", [
                            "contract.merchant_pub",
                            "contract.repurchase_correlation_id"
                        ]);
                    }
                }
                class DenominationsStore extends query_1.Store {
                    constructor() {
                        // case needed because of bug in type annotations
                        super("denominations", { keyPath: ["exchangeBaseUrl", "denomPub"] });
                        this.exchangeBaseUrlIndex = new query_1.Index(this, "exchangeBaseUrl", "exchangeBaseUrl");
                        this.denomPubIndex = new query_1.Index(this, "denomPub", "denomPub");
                    }
                }
                Stores.exchanges = new ExchangeStore();
                Stores.transactions = new TransactionsStore();
                Stores.reserves = new query_1.Store("reserves", { keyPath: "reserve_pub" });
                Stores.coins = new CoinsStore();
                Stores.refresh = new query_1.Store("refresh", { keyPath: "meltCoinPub" });
                Stores.history = new HistoryStore();
                Stores.offers = new OffersStore();
                Stores.precoins = new query_1.Store("precoins", { keyPath: "coinPub" });
                Stores.denominations = new DenominationsStore();
            })(Stores || (Stores = {}));
            exports_1("Stores", Stores);
            Wallet = class Wallet {
                constructor(db, http, badge, notifier) {
                    this.processPreCoinConcurrent = 0;
                    this.processPreCoinThrottle = {};
                    /**
                     * Set of identifiers for running operations.
                     */
                    this.runningOperations = new Set();
                    this.db = db;
                    this.http = http;
                    this.badge = badge;
                    this.notifier = notifier;
                    this.cryptoApi = new cryptoApi_1.CryptoApi();
                    this.resumePendingFromDb();
                }
                q() {
                    return new query_1.QueryRoot(this.db);
                }
                startOperation(operationId) {
                    this.runningOperations.add(operationId);
                    this.badge.startBusy();
                }
                stopOperation(operationId) {
                    this.runningOperations.delete(operationId);
                    if (this.runningOperations.size == 0) {
                        this.badge.stopBusy();
                    }
                }
                updateExchanges() {
                    return __awaiter(this, void 0, void 0, function* () {
                        console.log("updating exchanges");
                        let exchangesUrls = yield this.q()
                            .iter(Stores.exchanges)
                            .map((e) => e.baseUrl)
                            .toArray();
                        for (let url of exchangesUrls) {
                            this.updateExchangeFromUrl(url)
                                .catch((e) => {
                                console.error("updating exchange failed", e);
                            });
                        }
                    });
                }
                /**
                 * Resume various pending operations that are pending
                 * by looking at the database.
                 */
                resumePendingFromDb() {
                    console.log("resuming pending operations from db");
                    this.q()
                        .iter(Stores.reserves)
                        .reduce((reserve) => {
                        console.log("resuming reserve", reserve.reserve_pub);
                        this.processReserve(reserve);
                    });
                    this.q()
                        .iter(Stores.precoins)
                        .reduce((preCoin) => {
                        console.log("resuming precoin");
                        this.processPreCoin(preCoin);
                    });
                    this.q()
                        .iter(Stores.refresh)
                        .reduce((r) => {
                        this.continueRefreshSession(r);
                    });
                    // FIXME: optimize via index
                    this.q()
                        .iter(Stores.coins)
                        .reduce((c) => {
                        if (c.dirty && !c.transactionPending) {
                            this.refresh(c.coinPub);
                        }
                    });
                }
                /**
                 * Get exchanges and associated coins that are still spendable,
                 * but only if the sum the coins' remaining value exceeds the payment amount.
                 */
                getCoinsForPayment(paymentAmount, depositFeeLimit, allowedExchanges) {
                    return __awaiter(this, void 0, void 0, function* () {
                        for (let exchangeHandle of allowedExchanges) {
                            let exchange = yield this.q().get(Stores.exchanges, exchangeHandle.url);
                            if (!exchange) {
                                console.error("db inconsistent");
                                continue;
                            }
                            let coins = yield this.q()
                                .iterIndex(Stores.coins.exchangeBaseUrlIndex, exchangeHandle.url)
                                .toArray();
                            if (!coins || coins.length == 0) {
                                continue;
                            }
                            // Denomination of the first coin, we assume that all other
                            // coins have the same currency
                            let firstDenom = yield this.q().get(Stores.denominations, [
                                exchangeHandle.url,
                                coins[0].denomPub
                            ]);
                            if (!firstDenom) {
                                throw Error("db inconsistent");
                            }
                            let currency = firstDenom.value.currency;
                            let cds = [];
                            for (let i = 0; i < coins.length; i++) {
                                let coin = coins[i];
                                let denom = yield this.q().get(Stores.denominations, [exchangeHandle.url, coin.denomPub]);
                                if (!denom) {
                                    throw Error("db inconsistent");
                                }
                                if (denom.value.currency != currency) {
                                    console.warn(`same pubkey for different currencies at exchange ${exchange.baseUrl}`);
                                    continue;
                                }
                                if (coin.suspended) {
                                    continue;
                                }
                                if (coin.dirty) {
                                    continue;
                                }
                                if (coin.transactionPending) {
                                    continue;
                                }
                                cds.push({ coin, denom });
                            }
                            let res = selectCoins(cds, paymentAmount, depositFeeLimit);
                            if (res) {
                                return {
                                    exchangeUrl: exchangeHandle.url,
                                    cds: res,
                                };
                            }
                        }
                        return undefined;
                    });
                }
                /**
                 * Record all information that is necessary to
                 * pay for a contract in the wallet's database.
                 */
                recordConfirmPay(offer, payCoinInfo, chosenExchange) {
                    return __awaiter(this, void 0, void 0, function* () {
                        let payReq = {
                            amount: offer.contract.amount,
                            coins: payCoinInfo.map((x) => x.sig),
                            H_contract: offer.H_contract,
                            max_fee: offer.contract.max_fee,
                            merchant_sig: offer.merchant_sig,
                            exchange: URI(chosenExchange).href(),
                            refund_deadline: offer.contract.refund_deadline,
                            pay_deadline: offer.contract.pay_deadline,
                            timestamp: offer.contract.timestamp,
                            transaction_id: offer.contract.transaction_id,
                            instance: offer.contract.merchant.instance
                        };
                        let t = {
                            contractHash: offer.H_contract,
                            contract: offer.contract,
                            payReq: payReq,
                            merchantSig: offer.merchant_sig,
                            finished: false,
                        };
                        let historyEntry = {
                            type: "pay",
                            timestamp: (new Date).getTime(),
                            subjectId: `contract-${offer.H_contract}`,
                            detail: {
                                merchantName: offer.contract.merchant.name,
                                amount: offer.contract.amount,
                                contractHash: offer.H_contract,
                                fulfillmentUrl: offer.contract.fulfillment_url,
                            },
                            level: HistoryLevel.User
                        };
                        yield this.q()
                            .put(Stores.transactions, t)
                            .put(Stores.history, historyEntry)
                            .putAll(Stores.coins, payCoinInfo.map((pci) => pci.updatedCoin))
                            .finish();
                        this.notifier.notify();
                    });
                }
                putHistory(historyEntry) {
                    return __awaiter(this, void 0, void 0, function* () {
                        yield this.q().put(Stores.history, historyEntry).finish();
                        this.notifier.notify();
                    });
                }
                saveOffer(offer) {
                    return __awaiter(this, void 0, void 0, function* () {
                        console.log(`saving offer in wallet.ts`);
                        let id = yield this.q().putWithResult(Stores.offers, offer);
                        this.notifier.notify();
                        console.log(`saved offer with id ${id}`);
                        if (typeof id !== "number") {
                            throw Error("db schema wrong");
                        }
                        return id;
                    });
                }
                /**
                 * Add a contract to the wallet and sign coins,
                 * but do not send them yet.
                 */
                confirmPay(offer) {
                    return __awaiter(this, void 0, void 0, function* () {
                        console.log("executing confirmPay");
                        let transaction = yield this.q().get(Stores.transactions, offer.H_contract);
                        if (transaction) {
                            // Already payed ...
                            return {};
                        }
                        let res = yield this.getCoinsForPayment(offer.contract.amount, offer.contract.max_fee, offer.contract.exchanges);
                        console.log("max_fee", offer.contract.max_fee);
                        console.log("coin selection result", res);
                        if (!res) {
                            console.log("not confirming payment, insufficient coins");
                            return {
                                error: "coins-insufficient",
                            };
                        }
                        let { exchangeUrl, cds } = res;
                        let ds = yield this.cryptoApi.signDeposit(offer, cds);
                        yield this.recordConfirmPay(offer, ds, exchangeUrl);
                        return {};
                    });
                }
                /**
                 * Add a contract to the wallet and sign coins,
                 * but do not send them yet.
                 */
                checkPay(offer) {
                    return __awaiter(this, void 0, void 0, function* () {
                        // First check if we already payed for it.
                        let transaction = yield this.q().get(Stores.transactions, offer.H_contract);
                        if (transaction) {
                            return { isPayed: true };
                        }
                        // If not already payed, check if we could pay for it.
                        let res = yield this.getCoinsForPayment(offer.contract.amount, offer.contract.max_fee, offer.contract.exchanges);
                        if (!res) {
                            console.log("not confirming payment, insufficient coins");
                            return {
                                error: "coins-insufficient",
                            };
                        }
                        return { isPayed: false };
                    });
                }
                /**
                 * Retrieve all necessary information for looking up the contract
                 * with the given hash.
                 */
                executePayment(H_contract) {
                    return __awaiter(this, void 0, void 0, function* () {
                        let t = yield this.q().get(Stores.transactions, H_contract);
                        if (!t) {
                            return {
                                success: false,
                                contractFound: false,
                            };
                        }
                        let resp = {
                            success: true,
                            payReq: t.payReq,
                            contract: t.contract,
                        };
                        return resp;
                    });
                }
                /**
                 * First fetch information requred to withdraw from the reserve,
                 * then deplete the reserve, withdrawing coins until it is empty.
                 */
                processReserve(reserveRecord, retryDelayMs = 250) {
                    return __awaiter(this, void 0, void 0, function* () {
                        const opId = "reserve-" + reserveRecord.reserve_pub;
                        this.startOperation(opId);
                        try {
                            let exchange = yield this.updateExchangeFromUrl(reserveRecord.exchange_base_url);
                            let reserve = yield this.updateReserve(reserveRecord.reserve_pub, exchange);
                            let n = yield this.depleteReserve(reserve, exchange);
                            if (n != 0) {
                                let depleted = {
                                    type: "depleted-reserve",
                                    subjectId: `reserve-progress-${reserveRecord.reserve_pub}`,
                                    timestamp: (new Date).getTime(),
                                    detail: {
                                        exchangeBaseUrl: reserveRecord.exchange_base_url,
                                        reservePub: reserveRecord.reserve_pub,
                                        requestedAmount: reserveRecord.requested_amount,
                                        currentAmount: reserveRecord.current_amount,
                                    },
                                    level: HistoryLevel.User
                                };
                                yield this.q().put(Stores.history, depleted).finish();
                            }
                        }
                        catch (e) {
                            // random, exponential backoff truncated at 3 minutes
                            let nextDelay = Math.min(2 * retryDelayMs + retryDelayMs * Math.random(), 3000 * 60);
                            console.warn(`Failed to deplete reserve, trying again in ${retryDelayMs} ms`);
                            setTimeout(() => this.processReserve(reserveRecord, nextDelay), retryDelayMs);
                        }
                        finally {
                            this.stopOperation(opId);
                        }
                    });
                }
                processPreCoin(preCoin, retryDelayMs = 200) {
                    return __awaiter(this, void 0, void 0, function* () {
                        if (this.processPreCoinConcurrent >= 4 || this.processPreCoinThrottle[preCoin.exchangeBaseUrl]) {
                            console.log("delaying processPreCoin");
                            setTimeout(() => this.processPreCoin(preCoin, Math.min(retryDelayMs * 2, 5 * 60 * 1000)), retryDelayMs);
                            return;
                        }
                        console.log("executing processPreCoin");
                        this.processPreCoinConcurrent++;
                        try {
                            const exchange = yield this.q().get(Stores.exchanges, preCoin.exchangeBaseUrl);
                            if (!exchange) {
                                console.error("db inconsistend: exchange for precoin not found");
                                return;
                            }
                            const denom = yield this.q().get(Stores.denominations, [preCoin.exchangeBaseUrl, preCoin.denomPub]);
                            if (!denom) {
                                console.error("db inconsistent: denom for precoin not found");
                                return;
                            }
                            const coin = yield this.withdrawExecute(preCoin);
                            const mutateReserve = (r) => {
                                console.log(`before committing coin: current ${helpers_1.amountToPretty(r.current_amount)}, precoin: ${helpers_1.amountToPretty(r.precoin_amount)})}`);
                                let x = types_1.Amounts.sub(r.precoin_amount, preCoin.coinValue, denom.feeWithdraw);
                                if (x.saturated) {
                                    console.error("database inconsistent");
                                    throw query_1.AbortTransaction;
                                }
                                r.precoin_amount = x.amount;
                                return r;
                            };
                            const historyEntry = {
                                type: "withdraw",
                                timestamp: (new Date).getTime(),
                                level: HistoryLevel.Expert,
                                detail: {
                                    coinPub: coin.coinPub,
                                }
                            };
                            yield this.q()
                                .mutate(Stores.reserves, preCoin.reservePub, mutateReserve)
                                .delete("precoins", coin.coinPub)
                                .add(Stores.coins, coin)
                                .add(Stores.history, historyEntry)
                                .finish();
                            this.notifier.notify();
                        }
                        catch (e) {
                            console.error("Failed to withdraw coin from precoin, retrying in", retryDelayMs, "ms", e);
                            // exponential backoff truncated at one minute
                            let nextRetryDelayMs = Math.min(retryDelayMs * 2, 5 * 60 * 1000);
                            setTimeout(() => this.processPreCoin(preCoin, nextRetryDelayMs), retryDelayMs);
                            this.processPreCoinThrottle[preCoin.exchangeBaseUrl] = (this.processPreCoinThrottle[preCoin.exchangeBaseUrl] || 0) + 1;
                            setTimeout(() => { this.processPreCoinThrottle[preCoin.exchangeBaseUrl]--; }, retryDelayMs);
                        }
                        finally {
                            this.processPreCoinConcurrent--;
                        }
                    });
                }
                /**
                 * Create a reserve, but do not flag it as confirmed yet.
                 */
                createReserve(req) {
                    return __awaiter(this, void 0, void 0, function* () {
                        let keypair = yield this.cryptoApi.createEddsaKeypair();
                        const now = (new Date).getTime();
                        const canonExchange = helpers_1.canonicalizeBaseUrl(req.exchange);
                        const reserveRecord = {
                            reserve_pub: keypair.pub,
                            reserve_priv: keypair.priv,
                            exchange_base_url: canonExchange,
                            created: now,
                            last_query: null,
                            current_amount: null,
                            requested_amount: req.amount,
                            confirmed: false,
                            precoin_amount: types_1.Amounts.getZero(req.amount.currency),
                        };
                        const historyEntry = {
                            type: "create-reserve",
                            level: HistoryLevel.Expert,
                            timestamp: now,
                            subjectId: `reserve-progress-${reserveRecord.reserve_pub}`,
                            detail: {
                                requestedAmount: req.amount,
                                reservePub: reserveRecord.reserve_pub,
                            }
                        };
                        yield this.q()
                            .put(Stores.reserves, reserveRecord)
                            .put(Stores.history, historyEntry)
                            .finish();
                        let r = {
                            exchange: canonExchange,
                            reservePub: keypair.pub,
                        };
                        return r;
                    });
                }
                /**
                 * Mark an existing reserve as confirmed.  The wallet will start trying
                 * to withdraw from that reserve.  This may not immediately succeed,
                 * since the exchange might not know about the reserve yet, even though the
                 * bank confirmed its creation.
                 *
                 * A confirmed reserve should be shown to the user in the UI, while
                 * an unconfirmed reserve should be hidden.
                 */
                confirmReserve(req) {
                    return __awaiter(this, void 0, void 0, function* () {
                        const now = (new Date).getTime();
                        let reserve = yield (this.q().get(Stores.reserves, req.reservePub));
                        if (!reserve) {
                            console.error("Unable to confirm reserve, not found in DB");
                            return;
                        }
                        console.log("reserve confirmed");
                        const historyEntry = {
                            type: "confirm-reserve",
                            timestamp: now,
                            subjectId: `reserve-progress-${reserve.reserve_pub}`,
                            detail: {
                                exchangeBaseUrl: reserve.exchange_base_url,
                                reservePub: req.reservePub,
                                requestedAmount: reserve.requested_amount,
                            },
                            level: HistoryLevel.User,
                        };
                        reserve.confirmed = true;
                        yield this.q()
                            .put(Stores.reserves, reserve)
                            .put(Stores.history, historyEntry)
                            .finish();
                        this.notifier.notify();
                        this.processReserve(reserve);
                    });
                }
                withdrawExecute(pc) {
                    return __awaiter(this, void 0, void 0, function* () {
                        let reserve = yield this.q().get(Stores.reserves, pc.reservePub);
                        if (!reserve) {
                            throw Error("db inconsistent");
                        }
                        let wd = {};
                        wd.denom_pub = pc.denomPub;
                        wd.reserve_pub = pc.reservePub;
                        wd.reserve_sig = pc.withdrawSig;
                        wd.coin_ev = pc.coinEv;
                        let reqUrl = URI("reserve/withdraw").absoluteTo(reserve.exchange_base_url);
                        let resp = yield this.http.postJson(reqUrl, wd);
                        if (resp.status != 200) {
                            throw new http_1.RequestException({
                                hint: "Withdrawal failed",
                                status: resp.status
                            });
                        }
                        let r = JSON.parse(resp.responseText);
                        let denomSig = yield this.cryptoApi.rsaUnblind(r.ev_sig, pc.blindingKey, pc.denomPub);
                        let coin = {
                            coinPub: pc.coinPub,
                            coinPriv: pc.coinPriv,
                            denomPub: pc.denomPub,
                            denomSig: denomSig,
                            currentAmount: pc.coinValue,
                            exchangeBaseUrl: pc.exchangeBaseUrl,
                            dirty: false,
                            transactionPending: false,
                        };
                        return coin;
                    });
                }
                /**
                 * Withdraw coins from a reserve until it is empty.
                 */
                depleteReserve(reserve, exchange) {
                    return __awaiter(this, void 0, void 0, function* () {
                        console.log("depleting reserve");
                        if (!reserve.current_amount) {
                            throw Error("can't withdraw when amount is unknown");
                        }
                        let currentAmount = reserve.current_amount;
                        if (!currentAmount) {
                            throw Error("can't withdraw when amount is unknown");
                        }
                        let denomsForWithdraw = yield this.getVerifiedWithdrawDenomList(exchange.baseUrl, currentAmount);
                        console.log(`withdrawing ${denomsForWithdraw.length} coins`);
                        let ps = denomsForWithdraw.map((denom) => __awaiter(this, void 0, void 0, function* () {
                            function mutateReserve(r) {
                                let currentAmount = r.current_amount;
                                if (!currentAmount) {
                                    throw Error("can't withdraw when amount is unknown");
                                }
                                r.precoin_amount = types_1.Amounts.add(r.precoin_amount, denom.value, denom.feeWithdraw).amount;
                                let result = types_1.Amounts.sub(currentAmount, denom.value, denom.feeWithdraw);
                                if (result.saturated) {
                                    console.error("can't create precoin, saturated");
                                    throw query_1.AbortTransaction;
                                }
                                r.current_amount = result.amount;
                                console.log(`after creating precoin: current ${helpers_1.amountToPretty(r.current_amount)}, precoin: ${helpers_1.amountToPretty(r.precoin_amount)})}`);
                                return r;
                            }
                            let preCoin = yield this.cryptoApi
                                .createPreCoin(denom, reserve);
                            yield this.q()
                                .put(Stores.precoins, preCoin)
                                .mutate(Stores.reserves, reserve.reserve_pub, mutateReserve);
                            yield this.processPreCoin(preCoin);
                        }));
                        yield Promise.all(ps);
                        return ps.length;
                    });
                }
                /**
                 * Update the information about a reserve that is stored in the wallet
                 * by quering the reserve's exchange.
                 */
                updateReserve(reservePub, exchange) {
                    return __awaiter(this, void 0, void 0, function* () {
                        let reserve = yield this.q()
                            .get(Stores.reserves, reservePub);
                        if (!reserve) {
                            throw Error("reserve not in db");
                        }
                        let reqUrl = URI("reserve/status").absoluteTo(exchange.baseUrl);
                        reqUrl.query({ 'reserve_pub': reservePub });
                        let resp = yield this.http.get(reqUrl);
                        if (resp.status != 200) {
                            throw Error();
                        }
                        let reserveInfo = JSON.parse(resp.responseText);
                        if (!reserveInfo) {
                            throw Error();
                        }
                        let oldAmount = reserve.current_amount;
                        let newAmount = reserveInfo.balance;
                        reserve.current_amount = reserveInfo.balance;
                        let historyEntry = {
                            type: "reserve-update",
                            timestamp: (new Date).getTime(),
                            subjectId: `reserve-progress-${reserve.reserve_pub}`,
                            detail: {
                                reservePub,
                                requestedAmount: reserve.requested_amount,
                                oldAmount,
                                newAmount
                            }
                        };
                        yield this.q()
                            .put(Stores.reserves, reserve)
                            .finish();
                        this.notifier.notify();
                        return reserve;
                    });
                }
                /**
                 * Get the wire information for the exchange with the given base URL.
                 */
                getWireInfo(exchangeBaseUrl) {
                    return __awaiter(this, void 0, void 0, function* () {
                        exchangeBaseUrl = helpers_1.canonicalizeBaseUrl(exchangeBaseUrl);
                        let reqUrl = URI("wire").absoluteTo(exchangeBaseUrl);
                        let resp = yield this.http.get(reqUrl);
                        if (resp.status != 200) {
                            throw Error("/wire request failed");
                        }
                        let wiJson = JSON.parse(resp.responseText);
                        if (!wiJson) {
                            throw Error("/wire response malformed");
                        }
                        return wiJson;
                    });
                }
                getPossibleDenoms(exchangeBaseUrl) {
                    return __awaiter(this, void 0, void 0, function* () {
                        return (this.q().iterIndex(Stores.denominations.exchangeBaseUrlIndex, exchangeBaseUrl)
                            .filter((d) => d.status == types_1.DenominationStatus.Unverified || d.status == types_1.DenominationStatus.VerifiedGood)
                            .toArray());
                    });
                }
                /**
                 * Get a list of denominations to withdraw from the given exchange for the
                 * given amount, making sure that all denominations' signatures are verified.
                 *
                 * Writes to the DB in order to record the result from verifying
                 * denominations.
                 */
                getVerifiedWithdrawDenomList(exchangeBaseUrl, amount) {
                    return __awaiter(this, void 0, void 0, function* () {
                        const exchange = yield this.q().get(Stores.exchanges, exchangeBaseUrl);
                        if (!exchange) {
                            throw Error(`exchange ${exchangeBaseUrl} not found`);
                        }
                        const possibleDenoms = yield (this.q().iterIndex(Stores.denominations.exchangeBaseUrlIndex, exchange.baseUrl)
                            .filter((d) => d.status == types_1.DenominationStatus.Unverified || d.status == types_1.DenominationStatus.VerifiedGood)
                            .toArray());
                        let allValid = false;
                        let currentPossibleDenoms = possibleDenoms;
                        let selectedDenoms;
                        do {
                            allValid = true;
                            let nextPossibleDenoms = [];
                            selectedDenoms = getWithdrawDenomList(amount, possibleDenoms);
                            for (let denom of selectedDenoms || []) {
                                if (denom.status == types_1.DenominationStatus.Unverified) {
                                    console.log(`verifying denom ${denom.denomPub.substr(0, 15)}`);
                                    let valid = yield this.cryptoApi.isValidDenom(denom, exchange.masterPublicKey);
                                    if (!valid) {
                                        denom.status = types_1.DenominationStatus.VerifiedBad;
                                        allValid = false;
                                    }
                                    else {
                                        denom.status = types_1.DenominationStatus.VerifiedGood;
                                        nextPossibleDenoms.push(denom);
                                    }
                                    yield this.q().put(Stores.denominations, denom).finish();
                                }
                                else {
                                    nextPossibleDenoms.push(denom);
                                }
                            }
                            currentPossibleDenoms = nextPossibleDenoms;
                        } while (selectedDenoms.length > 0 && !allValid);
                        return selectedDenoms;
                    });
                }
                getReserveCreationInfo(baseUrl, amount) {
                    return __awaiter(this, void 0, void 0, function* () {
                        let exchangeInfo = yield this.updateExchangeFromUrl(baseUrl);
                        let selectedDenoms = yield this.getVerifiedWithdrawDenomList(baseUrl, amount);
                        let acc = types_1.Amounts.getZero(amount.currency);
                        for (let d of selectedDenoms) {
                            acc = types_1.Amounts.add(acc, d.feeWithdraw).amount;
                        }
                        let actualCoinCost = selectedDenoms
                            .map((d) => types_1.Amounts.add(d.value, d.feeWithdraw).amount)
                            .reduce((a, b) => types_1.Amounts.add(a, b).amount);
                        let wireInfo = yield this.getWireInfo(baseUrl);
                        let ret = {
                            exchangeInfo,
                            selectedDenoms,
                            wireInfo,
                            withdrawFee: acc,
                            overhead: types_1.Amounts.sub(amount, actualCoinCost).amount,
                        };
                        return ret;
                    });
                }
                /**
                 * Update or add exchange DB entry by fetching the /keys information.
                 * Optionally link the reserve entry to the new or existing
                 * exchange entry in then DB.
                 */
                updateExchangeFromUrl(baseUrl) {
                    return __awaiter(this, void 0, void 0, function* () {
                        baseUrl = helpers_1.canonicalizeBaseUrl(baseUrl);
                        let reqUrl = URI("keys").absoluteTo(baseUrl);
                        let resp = yield this.http.get(reqUrl);
                        if (resp.status != 200) {
                            throw Error("/keys request failed");
                        }
                        let exchangeKeysJson = KeysJson.checked(JSON.parse(resp.responseText));
                        return this.updateExchangeFromJson(baseUrl, exchangeKeysJson);
                    });
                }
                suspendCoins(exchangeInfo) {
                    return __awaiter(this, void 0, void 0, function* () {
                        let suspendedCoins = yield (this.q()
                            .iterIndex(Stores.coins.exchangeBaseUrlIndex, exchangeInfo.baseUrl)
                            .indexJoinLeft(Stores.denominations.exchangeBaseUrlIndex, (e) => e.exchangeBaseUrl)
                            .reduce((cd, suspendedCoins) => {
                            if ((!cd.right) || (!cd.right.isOffered)) {
                                return Array.prototype.concat(suspendedCoins, [cd.left]);
                            }
                            return Array.prototype.concat(suspendedCoins);
                        }, []));
                        let q = this.q();
                        suspendedCoins.map((c) => {
                            console.log("suspending coin", c);
                            c.suspended = true;
                            q.put(Stores.coins, c);
                        });
                        yield q.finish();
                    });
                }
                updateExchangeFromJson(baseUrl, exchangeKeysJson) {
                    return __awaiter(this, void 0, void 0, function* () {
                        const updateTimeSec = helpers_1.getTalerStampSec(exchangeKeysJson.list_issue_date);
                        if (updateTimeSec === null) {
                            throw Error("invalid update time");
                        }
                        const r = yield this.q().get(Stores.exchanges, baseUrl);
                        let exchangeInfo;
                        if (!r) {
                            exchangeInfo = {
                                baseUrl,
                                lastUpdateTime: updateTimeSec,
                                masterPublicKey: exchangeKeysJson.master_public_key,
                            };
                            console.log("making fresh exchange");
                        }
                        else {
                            if (updateTimeSec < r.lastUpdateTime) {
                                console.log("outdated /keys, not updating");
                                return r;
                            }
                            exchangeInfo = r;
                            exchangeInfo.lastUpdateTime = updateTimeSec;
                            console.log("updating old exchange");
                        }
                        let updatedExchangeInfo = yield this.updateExchangeInfo(exchangeInfo, exchangeKeysJson);
                        yield this.suspendCoins(updatedExchangeInfo);
                        yield this.q()
                            .put(Stores.exchanges, updatedExchangeInfo)
                            .finish();
                        return updatedExchangeInfo;
                    });
                }
                updateExchangeInfo(exchangeInfo, newKeys) {
                    return __awaiter(this, void 0, void 0, function* () {
                        if (exchangeInfo.masterPublicKey != newKeys.master_public_key) {
                            throw Error("public keys do not match");
                        }
                        const existingDenoms = yield (this.q().iterIndex(Stores.denominations.exchangeBaseUrlIndex, exchangeInfo.baseUrl)
                            .reduce((x, acc) => (acc[x.denomPub] = x, acc), {}));
                        const newDenoms = {};
                        const newAndUnseenDenoms = {};
                        for (let d of newKeys.denoms) {
                            let dr = types_1.denominationRecordFromKeys(exchangeInfo.baseUrl, d);
                            if (!(d.denom_pub in existingDenoms)) {
                                newAndUnseenDenoms[dr.denomPub] = dr;
                            }
                            newDenoms[dr.denomPub] = dr;
                        }
                        for (let oldDenomPub in existingDenoms) {
                            if (!(oldDenomPub in newDenoms)) {
                                let d = existingDenoms[oldDenomPub];
                                d.isOffered = false;
                            }
                        }
                        yield this.q()
                            .putAll(Stores.denominations, Object.keys(newAndUnseenDenoms).map((d) => newAndUnseenDenoms[d]))
                            .putAll(Stores.denominations, Object.keys(existingDenoms).map((d) => existingDenoms[d]))
                            .finish();
                        return exchangeInfo;
                    });
                }
                /**
                 * Retrieve a mapping from currency name to the amount
                 * that is currenctly available for spending in the wallet.
                 */
                getBalances() {
                    return __awaiter(this, void 0, void 0, function* () {
                        function ensureEntry(balance, currency) {
                            let entry = balance[currency];
                            let z = types_1.Amounts.getZero(currency);
                            if (!entry) {
                                balance[currency] = entry = {
                                    available: z,
                                    pendingIncoming: z,
                                    pendingPayment: z,
                                };
                            }
                            return entry;
                        }
                        function collectBalances(c, balance) {
                            if (c.suspended) {
                                return balance;
                            }
                            let currency = c.currentAmount.currency;
                            let entry = ensureEntry(balance, currency);
                            entry.available = types_1.Amounts.add(entry.available, c.currentAmount).amount;
                            return balance;
                        }
                        function collectPendingWithdraw(r, balance) {
                            if (!r.confirmed) {
                                return balance;
                            }
                            let entry = ensureEntry(balance, r.requested_amount.currency);
                            let amount = r.current_amount;
                            if (!amount) {
                                amount = r.requested_amount;
                            }
                            amount = types_1.Amounts.add(amount, r.precoin_amount).amount;
                            if (types_1.Amounts.cmp(smallestWithdraw[r.exchange_base_url], amount) < 0) {
                                entry.pendingIncoming = types_1.Amounts.add(entry.pendingIncoming, amount).amount;
                            }
                            return balance;
                        }
                        function collectPendingRefresh(r, balance) {
                            if (!r.finished) {
                                return balance;
                            }
                            let entry = ensureEntry(balance, r.valueWithFee.currency);
                            entry.pendingIncoming = types_1.Amounts.add(entry.pendingIncoming, r.valueOutput).amount;
                            return balance;
                        }
                        function collectPayments(t, balance) {
                            if (t.finished) {
                                return balance;
                            }
                            let entry = ensureEntry(balance, t.contract.amount.currency);
                            entry.pendingPayment = types_1.Amounts.add(entry.pendingPayment, t.contract.amount).amount;
                            return balance;
                        }
                        function collectSmallestWithdraw(e, sw) {
                            let min = sw[e.left.baseUrl];
                            let v = types_1.Amounts.add(e.right.value, e.right.feeWithdraw).amount;
                            if (!min) {
                                min = v;
                            }
                            else if (types_1.Amounts.cmp(v, min) < 0) {
                                min = v;
                            }
                            sw[e.left.baseUrl] = min;
                            return sw;
                        }
                        let balance = {};
                        // Mapping from exchange pub to smallest
                        // possible amount we can withdraw
                        let smallestWithdraw = {};
                        smallestWithdraw = yield (this.q()
                            .iter(Stores.exchanges)
                            .indexJoin(Stores.denominations.exchangeBaseUrlIndex, (x) => x.baseUrl)
                            .reduce(collectSmallestWithdraw, {}));
                        let tx = this.q();
                        tx.iter(Stores.coins)
                            .reduce(collectBalances, balance);
                        tx.iter(Stores.refresh)
                            .reduce(collectPendingRefresh, balance);
                        tx.iter(Stores.reserves)
                            .reduce(collectPendingWithdraw, balance);
                        tx.iter(Stores.transactions)
                            .reduce(collectPayments, balance);
                        yield tx.finish();
                        return balance;
                    });
                }
                createRefreshSession(oldCoinPub) {
                    return __awaiter(this, void 0, void 0, function* () {
                        let coin = yield this.q().get(Stores.coins, oldCoinPub);
                        if (!coin) {
                            throw Error("coin not found");
                        }
                        let exchange = yield this.updateExchangeFromUrl(coin.exchangeBaseUrl);
                        if (!exchange) {
                            throw Error("db inconsistent");
                        }
                        let oldDenom = yield this.q().get(Stores.denominations, [exchange.baseUrl, coin.denomPub]);
                        if (!oldDenom) {
                            throw Error("db inconsistent");
                        }
                        let availableDenoms = yield (this.q()
                            .iterIndex(Stores.denominations.exchangeBaseUrlIndex, exchange.baseUrl)
                            .toArray());
                        let availableAmount = types_1.Amounts.sub(coin.currentAmount, oldDenom.feeRefresh).amount;
                        let newCoinDenoms = getWithdrawDenomList(availableAmount, availableDenoms);
                        console.log("refreshing into", newCoinDenoms);
                        if (newCoinDenoms.length == 0) {
                            console.log("not refreshing, value too small");
                            return undefined;
                        }
                        let refreshSession = yield (this.cryptoApi.createRefreshSession(exchange.baseUrl, 3, coin, newCoinDenoms, oldDenom.feeRefresh));
                        function mutateCoin(c) {
                            let r = types_1.Amounts.sub(c.currentAmount, refreshSession.valueWithFee);
                            if (r.saturated) {
                                // Something else must have written the coin value
                                throw query_1.AbortTransaction;
                            }
                            c.currentAmount = r.amount;
                            return c;
                        }
                        yield this.q()
                            .put(Stores.refresh, refreshSession)
                            .mutate(Stores.coins, coin.coinPub, mutateCoin)
                            .finish();
                        return refreshSession;
                    });
                }
                refresh(oldCoinPub) {
                    return __awaiter(this, void 0, void 0, function* () {
                        let refreshSession;
                        let oldSession = yield this.q().get(Stores.refresh, oldCoinPub);
                        if (oldSession) {
                            refreshSession = oldSession;
                        }
                        else {
                            refreshSession = yield this.createRefreshSession(oldCoinPub);
                        }
                        if (!refreshSession) {
                            // refreshing not necessary
                            return;
                        }
                        this.continueRefreshSession(refreshSession);
                    });
                }
                continueRefreshSession(refreshSession) {
                    return __awaiter(this, void 0, void 0, function* () {
                        if (refreshSession.finished) {
                            return;
                        }
                        if (typeof refreshSession.norevealIndex !== "number") {
                            let coinPub = refreshSession.meltCoinPub;
                            yield this.refreshMelt(refreshSession);
                            let r = yield this.q().get(Stores.refresh, coinPub);
                            if (!r) {
                                throw Error("refresh session does not exist anymore");
                            }
                            refreshSession = r;
                        }
                        yield this.refreshReveal(refreshSession);
                    });
                }
                refreshMelt(refreshSession) {
                    return __awaiter(this, void 0, void 0, function* () {
                        if (refreshSession.norevealIndex != undefined) {
                            console.error("won't melt again");
                            return;
                        }
                        let coin = yield this.q().get(Stores.coins, refreshSession.meltCoinPub);
                        if (!coin) {
                            console.error("can't melt coin, it does not exist");
                            return;
                        }
                        let reqUrl = URI("refresh/melt").absoluteTo(refreshSession.exchangeBaseUrl);
                        let meltCoin = {
                            coin_pub: coin.coinPub,
                            denom_pub: coin.denomPub,
                            denom_sig: coin.denomSig,
                            confirm_sig: refreshSession.confirmSig,
                            value_with_fee: refreshSession.valueWithFee,
                        };
                        let coinEvs = refreshSession.preCoinsForGammas.map((x) => x.map((y) => y.coinEv));
                        let req = {
                            "new_denoms": refreshSession.newDenoms,
                            "melt_coin": meltCoin,
                            "transfer_pubs": refreshSession.transferPubs,
                            "coin_evs": coinEvs,
                        };
                        console.log("melt request:", req);
                        let resp = yield this.http.postJson(reqUrl, req);
                        console.log("melt request:", req);
                        console.log("melt response:", resp.responseText);
                        if (resp.status != 200) {
                            console.error(resp.responseText);
                            throw Error("refresh failed");
                        }
                        let respJson = JSON.parse(resp.responseText);
                        if (!respJson) {
                            throw Error("exchange responded with garbage");
                        }
                        let norevealIndex = respJson.noreveal_index;
                        if (typeof norevealIndex != "number") {
                            throw Error("invalid response");
                        }
                        refreshSession.norevealIndex = norevealIndex;
                        yield this.q().put(Stores.refresh, refreshSession).finish();
                    });
                }
                refreshReveal(refreshSession) {
                    return __awaiter(this, void 0, void 0, function* () {
                        let norevealIndex = refreshSession.norevealIndex;
                        if (norevealIndex == undefined) {
                            throw Error("can't reveal without melting first");
                        }
                        let privs = Array.from(refreshSession.transferPrivs);
                        privs.splice(norevealIndex, 1);
                        let req = {
                            "session_hash": refreshSession.hash,
                            "transfer_privs": privs,
                        };
                        let reqUrl = URI("refresh/reveal")
                            .absoluteTo(refreshSession.exchangeBaseUrl);
                        console.log("reveal request:", req);
                        let resp = yield this.http.postJson(reqUrl, req);
                        console.log("session:", refreshSession);
                        console.log("reveal response:", resp);
                        if (resp.status != 200) {
                            console.log("error:  /refresh/reveal returned status " + resp.status);
                            return;
                        }
                        let respJson = JSON.parse(resp.responseText);
                        if (!respJson.ev_sigs || !Array.isArray(respJson.ev_sigs)) {
                            console.log("/refresh/reveal did not contain ev_sigs");
                        }
                        let exchange = yield this.q().get(Stores.exchanges, refreshSession.exchangeBaseUrl);
                        if (!exchange) {
                            console.error(`exchange ${refreshSession.exchangeBaseUrl} not found`);
                            return;
                        }
                        let coins = [];
                        for (let i = 0; i < respJson.ev_sigs.length; i++) {
                            let denom = yield (this.q()
                                .get(Stores.denominations, [
                                refreshSession.exchangeBaseUrl,
                                refreshSession.newDenoms[i]
                            ]));
                            if (!denom) {
                                console.error("denom not found");
                                continue;
                            }
                            let pc = refreshSession.preCoinsForGammas[refreshSession.norevealIndex][i];
                            let denomSig = yield this.cryptoApi.rsaUnblind(respJson.ev_sigs[i].ev_sig, pc.blindingKey, denom.denomPub);
                            let coin = {
                                coinPub: pc.publicKey,
                                coinPriv: pc.privateKey,
                                denomPub: denom.denomPub,
                                denomSig: denomSig,
                                currentAmount: denom.value,
                                exchangeBaseUrl: refreshSession.exchangeBaseUrl,
                                dirty: false,
                                transactionPending: false,
                            };
                            coins.push(coin);
                        }
                        refreshSession.finished = true;
                        yield this.q()
                            .putAll(Stores.coins, coins)
                            .put(Stores.refresh, refreshSession)
                            .finish();
                    });
                }
                /**
                 * Retrive the full event history for this wallet.
                 */
                getHistory() {
                    return __awaiter(this, void 0, void 0, function* () {
                        function collect(x, acc) {
                            acc.push(x);
                            return acc;
                        }
                        let history = yield (this.q()
                            .iterIndex(Stores.history.timestampIndex)
                            .reduce(collect, []));
                        return { history };
                    });
                }
                getDenoms(exchangeUrl) {
                    return __awaiter(this, void 0, void 0, function* () {
                        let denoms = yield this.q().iterIndex(Stores.denominations.exchangeBaseUrlIndex, exchangeUrl).toArray();
                        return denoms;
                    });
                }
                getOffer(offerId) {
                    return __awaiter(this, void 0, void 0, function* () {
                        let offer = yield this.q().get(Stores.offers, offerId);
                        return offer;
                    });
                }
                getExchanges() {
                    return __awaiter(this, void 0, void 0, function* () {
                        return this.q()
                            .iter(Stores.exchanges)
                            .flatMap((e) => [e])
                            .toArray();
                    });
                }
                getReserves(exchangeBaseUrl) {
                    return __awaiter(this, void 0, void 0, function* () {
                        return this.q()
                            .iter(Stores.reserves)
                            .filter((r) => r.exchange_base_url === exchangeBaseUrl)
                            .toArray();
                    });
                }
                getCoins(exchangeBaseUrl) {
                    return __awaiter(this, void 0, void 0, function* () {
                        return this.q()
                            .iter(Stores.coins)
                            .filter((c) => c.exchangeBaseUrl === exchangeBaseUrl)
                            .toArray();
                    });
                }
                getPreCoins(exchangeBaseUrl) {
                    return __awaiter(this, void 0, void 0, function* () {
                        return this.q()
                            .iter(Stores.precoins)
                            .filter((c) => c.exchangeBaseUrl === exchangeBaseUrl)
                            .toArray();
                    });
                }
                hashContract(contract) {
                    return __awaiter(this, void 0, void 0, function* () {
                        return this.cryptoApi.hashString(helpers_1.canonicalJson(contract));
                    });
                }
                /**
                 * Check if there's an equivalent contract we've already purchased.
                 */
                checkRepurchase(contract) {
                    return __awaiter(this, void 0, void 0, function* () {
                        if (!contract.repurchase_correlation_id) {
                            console.log("no repurchase: no correlation id");
                            return { isRepurchase: false };
                        }
                        let result = yield (this.q()
                            .getIndexed(Stores.transactions.repurchaseIndex, [
                            contract.merchant_pub,
                            contract.repurchase_correlation_id
                        ]));
                        if (result) {
                            console.assert(result.contract.repurchase_correlation_id == contract.repurchase_correlation_id);
                            return {
                                isRepurchase: true,
                                existingContractHash: result.contractHash,
                                existingFulfillmentUrl: result.contract.fulfillment_url,
                            };
                        }
                        else {
                            return { isRepurchase: false };
                        }
                    });
                }
                paymentSucceeded(contractHash) {
                    return __awaiter(this, void 0, void 0, function* () {
                        const doPaymentSucceeded = () => __awaiter(this, void 0, void 0, function* () {
                            let t = yield this.q().get(Stores.transactions, contractHash);
                            if (!t) {
                                console.error("contract not found");
                                return;
                            }
                            t.finished = true;
                            let modifiedCoins = [];
                            for (let pc of t.payReq.coins) {
                                let c = yield this.q().get(Stores.coins, pc.coin_pub);
                                if (!c) {
                                    console.error("coin not found");
                                    return;
                                }
                                c.transactionPending = false;
                                modifiedCoins.push(c);
                            }
                            yield this.q()
                                .putAll(Stores.coins, modifiedCoins)
                                .put(Stores.transactions, t)
                                .finish();
                            for (let c of t.payReq.coins) {
                                this.refresh(c.coin_pub);
                            }
                        });
                        doPaymentSucceeded();
                        return;
                    });
                }
            };
            exports_1("Wallet", Wallet);
        }
    };
});
//# sourceMappingURL=wallet.js.map