/*
  @source https://www.git.taler.net/?p=web-common.git;a=blob_plain;f=taler-wallet-lib.ts;hb=HEAD
  @license magnet:?xt=urn:btih:5de60da917303dbfad4f93fb1b985ced5a89eac2&dn=lgpl-2.1.txt LGPL v21

  @licstart  The following is the entire license notice for the
  JavaScript code in this page.

  Copyright (C) 2015, 2016 INRIA

  The JavaScript code in this page is free software: you can
  redistribute it and/or modify it under the terms of the GNU
  Lesser General Public License (GNU LGPL) as published by the Free Software
  Foundation, either version 2.1 of the License, or (at your option)
  any later version.  The code is distributed WITHOUT ANY WARRANTY;
  without even the implied warranty of MERCHANTABILITY or FITNESS
  FOR A PARTICULAR PURPOSE.  See the GNU LGPL for more details.

  As additional permission under GNU LGPL version 2.1 section 7, you
  may distribute non-source (e.g., minimized or compacted) forms of
  that code without the copy of the GNU LGPL normally required by
  section 4, provided you include this license notice and a URL
  through which recipients can access the Corresponding Source.

  @licend  The above is the entire license notice
  for the JavaScript code in this page.

  @author Marcello Stanisci
  @author Florian Dold
*/
var taler;
/*
  @source https://www.git.taler.net/?p=web-common.git;a=blob_plain;f=taler-wallet-lib.ts;hb=HEAD
  @license magnet:?xt=urn:btih:5de60da917303dbfad4f93fb1b985ced5a89eac2&dn=lgpl-2.1.txt LGPL v21

  @licstart  The following is the entire license notice for the
  JavaScript code in this page.

  Copyright (C) 2015, 2016 INRIA

  The JavaScript code in this page is free software: you can
  redistribute it and/or modify it under the terms of the GNU
  Lesser General Public License (GNU LGPL) as published by the Free Software
  Foundation, either version 2.1 of the License, or (at your option)
  any later version.  The code is distributed WITHOUT ANY WARRANTY;
  without even the implied warranty of MERCHANTABILITY or FITNESS
  FOR A PARTICULAR PURPOSE.  See the GNU LGPL for more details.

  As additional permission under GNU LGPL version 2.1 section 7, you
  may distribute non-source (e.g., minimized or compacted) forms of
  that code without the copy of the GNU LGPL normally required by
  section 4, provided you include this license notice and a URL
  through which recipients can access the Corresponding Source.

  @licend  The above is the entire license notice
  for the JavaScript code in this page.

  @author Marcello Stanisci
  @author Florian Dold
*/
(function (taler) {
    "use strict";
    let logVerbose = false;
    try {
        logVerbose = !!localStorage.getItem("taler-log-verbose");
    }
    catch (e) {
    }
    const presentHandlers = [];
    const absentHandlers = [];
    // Are we running as the content script of an 
    // extension (and not just from a normal page)?
    let runningInExtension = false;
    let callSeqId = 1;
    let installed = false;
    let probeExecuted = false;
    let pageLoaded = false;
    let errorHandler = undefined;
    let sheet;
    function onError(handler) {
        if (errorHandler) {
            console.warn("Overriding error handler");
        }
        errorHandler = handler;
    }
    taler.onError = onError;
    /**
     * Error handler for things that go wrong in the merchant
     * frontend browser code.
     */
    function raise_error(reason, detail) {
        if (errorHandler) {
            errorHandler(reason, detail);
            return;
        }
        alert(`Failure: ${reason}.  No error handler installed.  Open the developer console for more information.`);
        console.error(reason, detail);
        console.warn("No custom error handler set.");
    }
    function callWallet(funcName, args, onResult) {
        const detail = JSON.parse(JSON.stringify(args || {}));
        const callId = callSeqId++;
        detail.callId = callId;
        let onTimeout = () => {
            console.warn("timeout for invocation of " + funcName);
        };
        const timeoutHandle = setTimeout(onTimeout, 1000);
        let handler = (evt) => {
            if (evt.detail.callId !== callId) {
                return;
            }
            if (onResult) {
                onResult(evt.detail);
            }
            clearTimeout(timeoutHandle);
            document.removeEventListener(funcName + "-result", handler);
        };
        document.addEventListener(funcName + "-result", handler);
        const evt = new CustomEvent(funcName, { detail });
        document.dispatchEvent(evt);
    }
    /**
     * Confirm that a reserve was created.
     *
     * Used by tightly integrated bank portals.
     */
    function confirmReserve(reservePub) {
        if (!installed) {
            logVerbose && console.log("delaying confirmReserve");
            taler.onPresent(() => {
                confirmReserve(reservePub);
            });
            return;
        }
        callWallet("taler-confirm-reserve", { reserve_pub: reservePub });
    }
    taler.confirmReserve = confirmReserve;
    function createReserve(callbackUrl, amount, wtTypes) {
        if (!installed) {
            logVerbose && console.log("delaying createReserve");
            taler.onPresent(() => {
                createReserve(callbackUrl, amount, wtTypes);
            });
            return;
        }
        let args = {
            callback_url: callbackUrl,
            amount: amount,
            wt_types: wtTypes,
        };
        callWallet("taler-create-reserve", args);
    }
    taler.createReserve = createReserve;
    function executePayment(contractHash, payUrl, offeringUrl, onError) {
        if (!installed) {
            logVerbose && console.log("delaying executePayment");
            taler.onPresent(() => {
                executePayment(contractHash, payUrl, offeringUrl);
            });
            return;
        }
        internalExecutePayment(contractHash, payUrl, offeringUrl);
    }
    taler.executePayment = executePayment;
    function internalExecutePayment(contractHash, payUrl, offeringUrl, onError) {
        if (!contractHash) {
            raise_error("protocol-violation", { hint: "contract hash not given" });
        }
        function subst(url, contractHash) {
            url = url.replace("${H_contract}", contractHash);
            url = url.replace("${$}", "$");
            return url;
        }
        /**
         * Handle a failed payment.
         *
         * Try to notify the wallet first, before we show a potentially
         * synchronous error message (such as an alert) or leave the page.
         */
        function handleFailedPayment(r) {
            let timeoutHandle = null;
            function err() {
                raise_error("pay-failed", { status: r.status, response: r.responseText });
            }
            function onResp() {
                if (timeoutHandle != null) {
                    clearTimeout(timeoutHandle);
                    timeoutHandle = null;
                }
                err();
            }
            function onTimeout() {
                timeoutHandle = null;
                err();
            }
            callWallet("taler-payment-failed", { H_contract: contractHash }, onResp);
            timeoutHandle = setTimeout(onTimeout, 200);
        }
        function handleResponse(resp) {
            logVerbose && console.log("handling taler-notify-payment");
            // Payment timeout in ms.
            let timeout_ms = 1000;
            // Current request.
            let r;
            let timeoutHandle = null;
            function sendPay() {
                r = new XMLHttpRequest();
                r.open("post", payUrl);
                r.setRequestHeader("Content-Type", "application/json;charset=UTF-8");
                r.send(JSON.stringify(resp.payment));
                r.onload = function () {
                    if (!r) {
                        return;
                    }
                    switch (r.status) {
                        case 200:
                            callWallet("taler-payment-succeeded", { H_contract: contractHash }, () => {
                                window.location.href = subst(resp.contract.fulfillment_url, resp.H_contract);
                                window.location.reload(true);
                            });
                            break;
                        default:
                            handleFailedPayment(r);
                            break;
                    }
                    r = null;
                    if (timeoutHandle != null) {
                        clearTimeout(timeoutHandle);
                        timeoutHandle = null;
                    }
                };
                function retry() {
                    if (r) {
                        r.abort();
                        r = null;
                    }
                    timeout_ms = Math.min(timeout_ms * 2, 10 * 1000);
                    logVerbose && console.log("sendPay timed out, retrying in ", timeout_ms, "ms");
                    sendPay();
                }
                timeoutHandle = setTimeout(retry, timeout_ms);
            }
            sendPay();
        }
        let args = {
            H_contract: contractHash,
            offering_url: offeringUrl
        };
        callWallet("taler-get-payment", args, handleResponse);
    }
    taler.internalExecutePayment = internalExecutePayment;
    function onPresent(f) {
        presentHandlers.push(f);
    }
    taler.onPresent = onPresent;
    function onAbsent(f) {
        absentHandlers.push(f);
    }
    taler.onAbsent = onAbsent;
    function offerContractFrom(url) {
        if (!installed) {
            logVerbose && console.log("delaying offerContractFrom until GNU Taler wallet is present");
            taler.onPresent(() => {
                offerContractFrom(url);
            });
            return;
        }
        internalOfferContractFrom(url);
    }
    taler.offerContractFrom = offerContractFrom;
    /**
     * OfferRecord a contract to the wallet after
     * downloading it from the given URL.
     */
    function internalOfferContractFrom(url) {
        function handle_contract(contract_wrapper) {
            let args = {
                contract_wrapper: contract_wrapper,
                replace_navigation: true
            };
            callWallet("taler-confirm-contract", args);
        }
        ;
        var contract_request = new XMLHttpRequest();
        logVerbose && console.log("downloading contract from '" + url + "'");
        contract_request.open("GET", url, true);
        contract_request.onload = function (e) {
            if (contract_request.readyState == 4) {
                if (contract_request.status == 200) {
                    logVerbose && console.log("response text:", contract_request.responseText);
                    var contract_wrapper = JSON.parse(contract_request.responseText);
                    if (!contract_wrapper) {
                        console.error("response text was invalid json");
                        let detail = { hint: "invalid json", status: contract_request.status, body: contract_request.responseText };
                        raise_error("contract-failed", detail);
                        return;
                    }
                    handle_contract(contract_wrapper);
                }
                else {
                    let detail = { hint: "contract download failed", status: contract_request.status, body: contract_request.responseText };
                    raise_error("contract-failed", detail);
                    return;
                }
            }
        };
        contract_request.onerror = function (e) {
            let detail = { hint: "contract download failed", status: contract_request.status, body: contract_request.responseText };
            raise_error("contract-failed", detail);
            return;
        };
        contract_request.send();
    }
    taler.internalOfferContractFrom = internalOfferContractFrom;
    function initTaler() {
        function handleProbe() {
            probeExecuted = true;
            if (!installed) {
                logVerbose && console.log("taler install detected");
                installed = true;
                announce();
            }
        }
        function probeTaler() {
            probeExecuted = false;
            var eve = new Event("taler-probe");
            document.dispatchEvent(eve);
        }
        let firstTimeoutCalled = false;
        function onProbeTimeout() {
            if (!probeExecuted) {
                if (installed || !firstTimeoutCalled) {
                    installed = false;
                    firstTimeoutCalled = true;
                    logVerbose && console.log("taler uninstall detected");
                    announce();
                }
            }
            // try again, maybe it'll be installed ...
            probeTaler();
        }
        /**
         * Announce presence/absence and update stylesheets.
         *
         * Only called after document.readyState is at least "interactive".
         */
        function announce() {
            if (!pageLoaded) {
                logVerbose && console.log("page not loaded yet, announcing later");
                return;
            }
            setStyles();
            if (installed) {
                logVerbose && console.log("announcing installed");
                for (var i = 0; i < presentHandlers.length; i++) {
                    presentHandlers[i]();
                }
            }
            else {
                if (firstTimeoutCalled) {
                    logVerbose && console.log("announcing uninstalled");
                    for (var i = 0; i < absentHandlers.length; i++) {
                        absentHandlers[i]();
                    }
                }
                else {
                    logVerbose && console.log("announcing nothing");
                }
            }
        }
        function setStyles() {
            if (!sheet || !sheet.cssRules) {
                return;
            }
            while (sheet.cssRules.length > 0) {
                sheet.deleteRule(0);
            }
            if (installed) {
                sheet.insertRule(".taler-installed-hide { display: none; }", 0);
                sheet.insertRule(".taler-probed-hide { display: none; }", 0);
            }
            else {
                sheet.insertRule(".taler-installed-show { display: none; }", 0);
                if (firstTimeoutCalled) {
                    sheet.insertRule(".taler-probed-hide { display: none; }", 0);
                }
                else {
                    // We're still doing the detection
                    sheet.insertRule(".taler-installed-hide { display: none; }", 0);
                }
            }
        }
        function initStyle() {
            logVerbose && console.log("taking over styles");
            const name = "taler-presence-stylesheet";
            const content = "/* Taler stylesheet controlled by JS */";
            let style = document.getElementById(name);
            if (!style) {
                style = document.createElement("style");
                // Needed by WebKit
                style.appendChild(document.createTextNode(content));
                style.id = name;
                document.head.appendChild(style);
                sheet = style.sheet;
            }
            else {
                // We've taken over the stylesheet now,
                // make it clear by clearing all the rules in it
                // and making it obvious in the DOM.
                if (style.tagName.toLowerCase() === "style") {
                    style.innerText = content;
                }
                if (!style.sheet) {
                    throw Error("taler-presence-stylesheet should be a style sheet (<link> or <style>)");
                }
                sheet = style.sheet;
                while (sheet.cssRules.length > 0) {
                    sheet.deleteRule(0);
                }
            }
        }
        function onPageLoad() {
            pageLoaded = true;
            if (document.readyState == "complete") {
                initStyle();
            }
            else {
                let listener = () => {
                    initStyle();
                    setStyles();
                };
                window.addEventListener("load", listener);
            }
            // We only start the timeout after the page is interactive.
            window.setInterval(onProbeTimeout, 300);
            announce();
        }
        probeTaler();
        document.addEventListener("taler-probe-result", handleProbe, false);
        // Handle the case where the JavaScript is loaded after the page
        // has been loaded for the first time.
        if (document.readyState == "loading") {
            document.addEventListener("DOMContentLoaded", onPageLoad, false);
        }
        else {
            onPageLoad();
        }
    }
    function onPageLoadInExtension() {
        if (document.documentElement.getAttribute("data-taler-nojs")) {
            logVerbose && console.log("doing taler initialization from extension (nojs)");
            initTaler();
        }
    }
    let caught = false;
    try {
        chrome.runtime.sendMessage({ type: "ping" });
    }
    catch (e) {
        caught = true;
    }
    if (caught) {
        logVerbose && console.log("running taler-wallet-lib from page");
        initTaler();
    }
    else {
        logVerbose && console.log("running taler-wallet-lib from extension");
        runningInExtension = true;
        // Wait for even style sheets to load
        if (document.readyState != "complete") {
            window.addEventListener("load", () => onPageLoadInExtension());
        }
        else {
            onPageLoadInExtension();
        }
    }
})(taler || (taler = {}));
// @license-end
//# sourceMappingURL=taler-wallet-lib.js.map