/*
 This file is part of TALER
 (C) 2016 GNUnet e.V.

 TALER is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 TALER is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
System.register([], function (exports_1, context_1) {
    /**
     * Database query abstractions.
     * @module Query
     * @author Florian Dold
     */
    "use strict";
    var __moduleName = context_1 && context_1.id;
    /**
     * Get an unresolved promise together with its extracted resolve / reject
     * function.
     */
    function openPromise() {
        let resolve = null;
        let reject = null;
        const promise = new Promise((res, rej) => {
            resolve = res;
            reject = rej;
        });
        if (!(resolve && reject)) {
            // Never happens, unless JS implementation is broken
            throw Error();
        }
        return { resolve, reject, promise };
    }
    exports_1("openPromise", openPromise);
    var Store, Index, AbortTransaction, QueryStreamBase, QueryStreamFilter, QueryStreamFlatMap, QueryStreamMap, QueryStreamIndexJoin, QueryStreamIndexJoinLeft, QueryStreamKeyJoin, IterQueryStream, QueryRoot;
    return {
        setters: [],
        execute: function () {
            Store = class Store {
                constructor(name, storeParams, validator) {
                    this.name = name;
                    this.validator = validator;
                    this.storeParams = storeParams;
                }
            };
            exports_1("Store", Store);
            Index = class Index {
                constructor(s, indexName, keyPath) {
                    this.storeName = s.name;
                    this.indexName = indexName;
                    this.keyPath = keyPath;
                }
            };
            exports_1("Index", Index);
            exports_1("AbortTransaction", AbortTransaction = Symbol("abort_transaction"));
            QueryStreamBase = class QueryStreamBase {
                constructor(root) {
                    this.root = root;
                }
                then(onfulfilled, onrejected) {
                    return this.root.then(onfulfilled, onrejected);
                }
                flatMap(f) {
                    return new QueryStreamFlatMap(this, f);
                }
                map(f) {
                    return new QueryStreamMap(this, f);
                }
                indexJoin(index, keyFn) {
                    this.root.addStoreAccess(index.storeName, false);
                    return new QueryStreamIndexJoin(this, index.storeName, index.indexName, keyFn);
                }
                indexJoinLeft(index, keyFn) {
                    this.root.addStoreAccess(index.storeName, false);
                    return new QueryStreamIndexJoinLeft(this, index.storeName, index.indexName, keyFn);
                }
                keyJoin(store, keyFn) {
                    this.root.addStoreAccess(store.name, false);
                    return new QueryStreamKeyJoin(this, store.name, keyFn);
                }
                filter(f) {
                    return new QueryStreamFilter(this, f);
                }
                toArray() {
                    let { resolve, promise } = openPromise();
                    let values = [];
                    this.subscribe((isDone, value) => {
                        if (isDone) {
                            resolve(values);
                            return;
                        }
                        values.push(value);
                    });
                    return Promise.resolve()
                        .then(() => this.root.finish())
                        .then(() => promise);
                }
                reduce(f, init) {
                    let { resolve, promise } = openPromise();
                    let acc = init;
                    this.subscribe((isDone, value) => {
                        if (isDone) {
                            resolve(acc);
                            return;
                        }
                        acc = f(value, acc);
                    });
                    return Promise.resolve()
                        .then(() => this.root.finish())
                        .then(() => promise);
                }
            };
            QueryStreamFilter = class QueryStreamFilter extends QueryStreamBase {
                constructor(s, filterFn) {
                    super(s.root);
                    this.s = s;
                    this.filterFn = filterFn;
                }
                subscribe(f) {
                    this.s.subscribe((isDone, value, tx) => {
                        if (isDone) {
                            f(true, undefined, tx);
                            return;
                        }
                        if (this.filterFn(value)) {
                            f(false, value, tx);
                        }
                    });
                }
            };
            QueryStreamFlatMap = class QueryStreamFlatMap extends QueryStreamBase {
                constructor(s, flatMapFn) {
                    super(s.root);
                    this.s = s;
                    this.flatMapFn = flatMapFn;
                }
                subscribe(f) {
                    this.s.subscribe((isDone, value, tx) => {
                        if (isDone) {
                            f(true, undefined, tx);
                            return;
                        }
                        let values = this.flatMapFn(value);
                        for (let v in values) {
                            f(false, value, tx);
                        }
                    });
                }
            };
            QueryStreamMap = class QueryStreamMap extends QueryStreamBase {
                constructor(s, mapFn) {
                    super(s.root);
                    this.s = s;
                    this.mapFn = mapFn;
                }
                subscribe(f) {
                    this.s.subscribe((isDone, value, tx) => {
                        if (isDone) {
                            f(true, undefined, tx);
                            return;
                        }
                        let mappedValue = this.mapFn(value);
                        f(false, mappedValue, tx);
                    });
                }
            };
            QueryStreamIndexJoin = class QueryStreamIndexJoin extends QueryStreamBase {
                constructor(s, storeName, indexName, key) {
                    super(s.root);
                    this.s = s;
                    this.storeName = storeName;
                    this.key = key;
                    this.indexName = indexName;
                }
                subscribe(f) {
                    this.s.subscribe((isDone, value, tx) => {
                        if (isDone) {
                            f(true, undefined, tx);
                            return;
                        }
                        let s = tx.objectStore(this.storeName).index(this.indexName);
                        let req = s.openCursor(IDBKeyRange.only(this.key(value)));
                        req.onsuccess = () => {
                            let cursor = req.result;
                            if (cursor) {
                                f(false, { left: value, right: cursor.value }, tx);
                                cursor.continue();
                            }
                        };
                    });
                }
            };
            QueryStreamIndexJoinLeft = class QueryStreamIndexJoinLeft extends QueryStreamBase {
                constructor(s, storeName, indexName, key) {
                    super(s.root);
                    this.s = s;
                    this.storeName = storeName;
                    this.key = key;
                    this.indexName = indexName;
                }
                subscribe(f) {
                    this.s.subscribe((isDone, value, tx) => {
                        if (isDone) {
                            f(true, undefined, tx);
                            return;
                        }
                        const s = tx.objectStore(this.storeName).index(this.indexName);
                        const req = s.openCursor(IDBKeyRange.only(this.key(value)));
                        let gotMatch = false;
                        req.onsuccess = () => {
                            let cursor = req.result;
                            if (cursor) {
                                gotMatch = true;
                                f(false, { left: value, right: cursor.value }, tx);
                                cursor.continue();
                            }
                            else {
                                if (!gotMatch) {
                                    f(false, { left: value }, tx);
                                }
                            }
                        };
                    });
                }
            };
            QueryStreamKeyJoin = class QueryStreamKeyJoin extends QueryStreamBase {
                constructor(s, storeName, key) {
                    super(s.root);
                    this.s = s;
                    this.storeName = storeName;
                    this.key = key;
                }
                subscribe(f) {
                    this.s.subscribe((isDone, value, tx) => {
                        if (isDone) {
                            f(true, undefined, tx);
                            return;
                        }
                        let s = tx.objectStore(this.storeName);
                        let req = s.openCursor(IDBKeyRange.only(this.key(value)));
                        req.onsuccess = () => {
                            let cursor = req.result;
                            if (cursor) {
                                f(false, { left: value, right: cursor.value }, tx);
                                cursor.continue();
                            }
                            else {
                                f(true, undefined, tx);
                            }
                        };
                    });
                }
            };
            IterQueryStream = class IterQueryStream extends QueryStreamBase {
                constructor(qr, storeName, options) {
                    super(qr);
                    this.options = options;
                    this.storeName = storeName;
                    this.subscribers = [];
                    let doIt = (tx) => {
                        const { indexName = void 0, only = void 0 } = this.options;
                        let s;
                        if (indexName !== void 0) {
                            s = tx.objectStore(this.storeName)
                                .index(this.options.indexName);
                        }
                        else {
                            s = tx.objectStore(this.storeName);
                        }
                        let kr = undefined;
                        if (only !== undefined) {
                            kr = IDBKeyRange.only(this.options.only);
                        }
                        let req = s.openCursor(kr);
                        req.onsuccess = () => {
                            let cursor = req.result;
                            if (cursor) {
                                for (let f of this.subscribers) {
                                    f(false, cursor.value, tx);
                                }
                                cursor.continue();
                            }
                            else {
                                for (let f of this.subscribers) {
                                    f(true, undefined, tx);
                                }
                            }
                        };
                    };
                    this.root.addWork(doIt);
                }
                subscribe(f) {
                    this.subscribers.push(f);
                }
            };
            QueryRoot = class QueryRoot {
                constructor(db) {
                    this.work = [];
                    this.stores = new Set();
                    this.db = db;
                }
                then(onfulfilled, onrejected) {
                    return this.finish().then(onfulfilled, onrejected);
                }
                iter(store) {
                    this.stores.add(store.name);
                    this.scheduleFinish();
                    return new IterQueryStream(this, store.name, {});
                }
                count(store) {
                    const { resolve, promise } = openPromise();
                    const doCount = (tx) => {
                        const s = tx.objectStore(store.name);
                        const req = s.count();
                        req.onsuccess = () => {
                            resolve(req.result);
                        };
                    };
                    this.addWork(doCount, store.name, false);
                    return Promise.resolve()
                        .then(() => this.finish())
                        .then(() => promise);
                }
                deleteIf(store, predicate) {
                    const doDeleteIf = (tx) => {
                        const s = tx.objectStore(store.name);
                        const req = s.openCursor();
                        let n = 0;
                        req.onsuccess = () => {
                            let cursor = req.result;
                            if (cursor) {
                                if (predicate(cursor.value, n++)) {
                                    cursor.delete();
                                }
                                cursor.continue();
                            }
                        };
                    };
                    this.addWork(doDeleteIf, store.name, true);
                    return this;
                }
                iterIndex(index, only) {
                    this.stores.add(index.storeName);
                    this.scheduleFinish();
                    return new IterQueryStream(this, index.storeName, {
                        only,
                        indexName: index.indexName
                    });
                }
                /**
                 * Put an object into the given object store.
                 * Overrides if an existing object with the same key exists
                 * in the store.
                 */
                put(store, val) {
                    let doPut = (tx) => {
                        tx.objectStore(store.name).put(val);
                    };
                    this.scheduleFinish();
                    this.addWork(doPut, store.name, true);
                    return this;
                }
                putWithResult(store, val) {
                    const { resolve, promise } = openPromise();
                    let doPutWithResult = (tx) => {
                        let req = tx.objectStore(store.name).put(val);
                        req.onsuccess = () => {
                            resolve(req.result);
                        };
                        this.scheduleFinish();
                    };
                    this.addWork(doPutWithResult, store.name, true);
                    return Promise.resolve()
                        .then(() => this.finish())
                        .then(() => promise);
                }
                mutate(store, key, f) {
                    let doPut = (tx) => {
                        let reqGet = tx.objectStore(store.name).get(key);
                        reqGet.onsuccess = () => {
                            let r = reqGet.result;
                            let m;
                            try {
                                m = f(r);
                            }
                            catch (e) {
                                if (e == AbortTransaction) {
                                    tx.abort();
                                    return;
                                }
                                throw e;
                            }
                            tx.objectStore(store.name).put(m);
                        };
                    };
                    this.scheduleFinish();
                    this.addWork(doPut, store.name, true);
                    return this;
                }
                /**
                 * Add all object from an iterable to the given object store.
                 * Fails if the object's key is already present
                 * in the object store.
                 */
                putAll(store, iterable) {
                    const doPutAll = (tx) => {
                        for (let obj of iterable) {
                            tx.objectStore(store.name).put(obj);
                        }
                    };
                    this.scheduleFinish();
                    this.addWork(doPutAll, store.name, true);
                    return this;
                }
                /**
                 * Add an object to the given object store.
                 * Fails if the object's key is already present
                 * in the object store.
                 */
                add(store, val) {
                    const doAdd = (tx) => {
                        tx.objectStore(store.name).add(val);
                    };
                    this.scheduleFinish();
                    this.addWork(doAdd, store.name, true);
                    return this;
                }
                /**
                 * Get one object from a store by its key.
                 */
                get(store, key) {
                    if (key === void 0) {
                        throw Error("key must not be undefined");
                    }
                    const { resolve, promise } = openPromise();
                    const doGet = (tx) => {
                        const req = tx.objectStore(store.name).get(key);
                        req.onsuccess = () => {
                            resolve(req.result);
                        };
                    };
                    this.addWork(doGet, store.name, false);
                    return Promise.resolve()
                        .then(() => this.finish())
                        .then(() => promise);
                }
                /**
                 * Get one object from a store by its key.
                 */
                getIndexed(index, key) {
                    if (key === void 0) {
                        throw Error("key must not be undefined");
                    }
                    const { resolve, promise } = openPromise();
                    const doGetIndexed = (tx) => {
                        const req = tx.objectStore(index.storeName)
                            .index(index.indexName)
                            .get(key);
                        req.onsuccess = () => {
                            resolve(req.result);
                        };
                    };
                    this.addWork(doGetIndexed, index.storeName, false);
                    return Promise.resolve()
                        .then(() => this.finish())
                        .then(() => promise);
                }
                scheduleFinish() {
                    if (!this.finishScheduled) {
                        Promise.resolve().then(() => this.finish());
                        this.finishScheduled = true;
                    }
                }
                /**
                 * Finish the query, and start the query in the first place if necessary.
                 */
                finish() {
                    if (this.kickoffPromise) {
                        return this.kickoffPromise;
                    }
                    this.kickoffPromise = new Promise((resolve, reject) => {
                        if (this.work.length == 0) {
                            resolve();
                            return;
                        }
                        const mode = this.hasWrite ? "readwrite" : "readonly";
                        const tx = this.db.transaction(Array.from(this.stores), mode);
                        tx.oncomplete = () => {
                            resolve();
                        };
                        tx.onabort = () => {
                            reject(Error("transaction aborted"));
                        };
                        for (let w of this.work) {
                            w(tx);
                        }
                    });
                    return this.kickoffPromise;
                }
                /**
                 * Delete an object by from the given object store.
                 */
                delete(storeName, key) {
                    const doDelete = (tx) => {
                        tx.objectStore(storeName).delete(key);
                    };
                    this.scheduleFinish();
                    this.addWork(doDelete, storeName, true);
                    return this;
                }
                /**
                 * Low-level function to add a task to the internal work queue.
                 */
                addWork(workFn, storeName, isWrite) {
                    this.work.push(workFn);
                    if (storeName) {
                        this.addStoreAccess(storeName, isWrite);
                    }
                }
                addStoreAccess(storeName, isWrite) {
                    if (storeName) {
                        this.stores.add(storeName);
                    }
                    if (isWrite) {
                        this.hasWrite = true;
                    }
                }
            };
            exports_1("QueryRoot", QueryRoot);
        }
    };
});
//# sourceMappingURL=query.js.map