/*
 This file is part of TALER
 (C) 2016 GNUnet e.V.

 TALER is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 TALER is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
System.register(["src/helpers", "src/wallet", "src/types", "src/renderHtml"], function (exports_1, context_1) {
    /**
     * Popup shown to the user when they click
     * the Taler browser action button.
     *
     * @author Florian Dold
     */
    "use strict";
    var __moduleName = context_1 && context_1.id;
    function onUpdateNotification(f) {
        let port = chrome.runtime.connect({ name: "notifications" });
        let listener = (msg, port) => {
            f();
        };
        port.onMessage.addListener(listener);
        return () => {
            port.onMessage.removeListener(listener);
        };
    }
    function main() {
        console.log("popup main");
        let el = (React.createElement("div", null,
            React.createElement(WalletNavBar, null),
            React.createElement("div", { style: { margin: "1em" } },
                React.createElement(Router, null,
                    React.createElement(WalletBalanceView, { route: "/balance", default: true }),
                    React.createElement(WalletHistory, { route: "/history" }),
                    React.createElement(WalletDebug, { route: "/debug" })))));
        ReactDOM.render(el, document.getElementById("content"));
    }
    exports_1("main", main);
    function Tab(props) {
        let cssClass = "";
        if (props.target == Router.getRoute()) {
            cssClass = "active";
        }
        let onClick = (e) => {
            Router.setRoute(props.target);
            e.preventDefault();
        };
        return (React.createElement("a", { onClick: onClick, href: props.target, className: cssClass }, props.children));
    }
    function ExtensionLink(props) {
        let onClick = (e) => {
            chrome.tabs.create({
                "url": chrome.extension.getURL(props.target)
            });
            e.preventDefault();
        };
        return (React.createElement("a", { onClick: onClick, href: props.target }, props.children));
    }
    function formatHistoryItem(historyItem) {
        const d = historyItem.detail;
        const t = historyItem.timestamp;
        console.log("hist item", historyItem);
        switch (historyItem.type) {
            case "create-reserve":
                return (React.createElement("p", null, i18n.parts `Bank requested reserve (${renderHtml_1.abbrev(d.reservePub)}) for ${renderHtml_1.prettyAmount(d.requestedAmount)}.`));
            case "confirm-reserve": {
                // FIXME: eventually remove compat fix
                let exchange = d.exchangeBaseUrl ? URI(d.exchangeBaseUrl).host() : "??";
                let amount = renderHtml_1.prettyAmount(d.requestedAmount);
                let pub = renderHtml_1.abbrev(d.reservePub);
                return (React.createElement("p", null, i18n.parts `Started to withdraw ${amount} from ${exchange} (${pub}).`));
            }
            case "offer-contract": {
                let link = chrome.extension.getURL("view-contract.html");
                let linkElem = React.createElement("a", { href: link }, renderHtml_1.abbrev(d.contractHash));
                let merchantElem = React.createElement("em", null, renderHtml_1.abbrev(d.merchantName, 15));
                return (React.createElement("p", null, i18n.parts `Merchant ${merchantElem} offered contract ${linkElem}.`));
            }
            case "depleted-reserve": {
                let exchange = d.exchangeBaseUrl ? URI(d.exchangeBaseUrl).host() : "??";
                let amount = renderHtml_1.prettyAmount(d.requestedAmount);
                let pub = renderHtml_1.abbrev(d.reservePub);
                return (React.createElement("p", null, i18n.parts `Withdrew ${amount} from ${exchange} (${pub}).`));
            }
            case "pay": {
                let url = helpers_1.substituteFulfillmentUrl(d.fulfillmentUrl, { H_contract: d.contractHash });
                let merchantElem = React.createElement("em", null, renderHtml_1.abbrev(d.merchantName, 15));
                let fulfillmentLinkElem = React.createElement("a", { href: url, onClick: openTab(url) }, "view product");
                return (React.createElement("p", null, i18n.parts `Paid ${renderHtml_1.prettyAmount(d.amount)} to merchant ${merchantElem}.  (${fulfillmentLinkElem})`));
            }
            default:
                return (React.createElement("p", null,
                    "i18n`Unknown event ($",
                    historyItem.type,
                    ")`"));
        }
    }
    function reload() {
        try {
            chrome.runtime.reload();
            window.close();
        }
        catch (e) {
        }
    }
    function confirmReset() {
        if (confirm("Do you want to IRREVOCABLY DESTROY everything inside your" +
            " wallet and LOSE ALL YOUR COINS?")) {
            chrome.runtime.sendMessage({ type: "reset" });
            window.close();
        }
    }
    function WalletDebug(props) {
        return (React.createElement("div", null,
            React.createElement("p", null, "Debug tools:"),
            React.createElement("button", { onClick: openExtensionPage("/src/popup/popup.html") }, "wallet tab"),
            React.createElement("button", { onClick: openExtensionPage("/src/pages/show-db.html") }, "show db"),
            React.createElement("button", { onClick: openExtensionPage("/src/pages/tree.html") }, "show tree"),
            React.createElement("button", { onClick: openExtensionPage("/src/pages/logs.html") }, "show logs"),
            React.createElement("br", null),
            React.createElement("button", { onClick: confirmReset }, "reset"),
            React.createElement("button", { onClick: reload }, "reload chrome extension")));
    }
    function openExtensionPage(page) {
        return function () {
            chrome.tabs.create({
                "url": chrome.extension.getURL(page)
            });
        };
    }
    function openTab(page) {
        return function () {
            chrome.tabs.create({
                "url": page
            });
        };
    }
    var helpers_1, wallet_1, types_1, renderHtml_1, Router, WalletNavBar, WalletBalanceView, WalletHistory;
    return {
        setters: [
            function (helpers_1_1) {
                helpers_1 = helpers_1_1;
            },
            function (wallet_1_1) {
                wallet_1 = wallet_1_1;
            },
            function (types_1_1) {
                types_1 = types_1_1;
            },
            function (renderHtml_1_1) {
                renderHtml_1 = renderHtml_1_1;
            }
        ],
        execute: function () {
            Router = class Router extends React.Component {
                static setRoute(s) {
                    window.location.hash = s;
                }
                static getRoute() {
                    // Omit the '#' at the beginning
                    return window.location.hash.substring(1);
                }
                static onRoute(f) {
                    Router.routeHandlers.push(f);
                    return () => {
                        let i = Router.routeHandlers.indexOf(f);
                        this.routeHandlers = this.routeHandlers.splice(i, 1);
                    };
                }
                componentWillMount() {
                    console.log("router mounted");
                    window.onhashchange = () => {
                        this.setState({});
                        for (let f of Router.routeHandlers) {
                            f();
                        }
                    };
                }
                componentWillUnmount() {
                    console.log("router unmounted");
                }
                render() {
                    let route = window.location.hash.substring(1);
                    console.log("rendering route", route);
                    let defaultChild = null;
                    let foundChild = null;
                    React.Children.forEach(this.props.children, (child) => {
                        let childProps = child.props;
                        if (!childProps) {
                            return;
                        }
                        if (childProps["default"]) {
                            defaultChild = child;
                        }
                        if (childProps["route"] == route) {
                            foundChild = child;
                        }
                    });
                    let child = foundChild || defaultChild;
                    if (!child) {
                        throw Error("unknown route");
                    }
                    Router.setRoute(child.props["route"]);
                    return React.createElement("div", null, child);
                }
            };
            Router.routeHandlers = [];
            WalletNavBar = class WalletNavBar extends React.Component {
                componentWillMount() {
                    this.cancelSubscription = Router.onRoute(() => {
                        this.setState({});
                    });
                }
                componentWillUnmount() {
                    if (this.cancelSubscription) {
                        this.cancelSubscription();
                    }
                }
                render() {
                    console.log("rendering nav bar");
                    return (React.createElement("div", { className: "nav", id: "header" },
                        React.createElement(Tab, { target: "/balance" }, i18n `Balance`),
                        React.createElement(Tab, { target: "/history" }, i18n `History`),
                        React.createElement(Tab, { target: "/debug" }, i18n `Debug`)));
                }
            };
            WalletBalanceView = class WalletBalanceView extends React.Component {
                constructor() {
                    super(...arguments);
                    this.gotError = false;
                    this.canceler = undefined;
                    this.unmount = false;
                }
                componentWillMount() {
                    this.canceler = onUpdateNotification(() => this.updateBalance());
                    this.updateBalance();
                }
                componentWillUnmount() {
                    console.log("component WalletBalanceView will unmount");
                    if (this.canceler) {
                        this.canceler();
                    }
                    this.unmount = true;
                }
                updateBalance() {
                    chrome.runtime.sendMessage({ type: "balances" }, (resp) => {
                        if (this.unmount) {
                            return;
                        }
                        if (resp.error) {
                            this.gotError = true;
                            console.error("could not retrieve balances", resp);
                            this.setState({});
                            return;
                        }
                        this.gotError = false;
                        console.log("got wallet", resp);
                        this.balance = resp;
                        this.setState({});
                    });
                }
                renderEmpty() {
                    let helpLink = (React.createElement(ExtensionLink, { target: "/src/pages/help/empty-wallet.html" }, i18n `help`));
                    return (React.createElement("div", null,
                        React.createElement(i18n.Translate, null,
                            "You have no balance to show. Need some",
                            " ",
                            React.createElement("span", null, helpLink),
                            " ",
                            "getting started?")));
                }
                formatPending(entry) {
                    let incoming;
                    let payment;
                    console.log("available: ", entry.pendingIncoming ? renderHtml_1.prettyAmount(entry.available) : null);
                    console.log("incoming: ", entry.pendingIncoming ? renderHtml_1.prettyAmount(entry.pendingIncoming) : null);
                    if (types_1.Amounts.isNonZero(entry.pendingIncoming)) {
                        incoming = (React.createElement(i18n.Translate, { wrap: "span" },
                            React.createElement("span", { style: { color: "darkgreen" } },
                                "+",
                                renderHtml_1.prettyAmount(entry.pendingIncoming)),
                            " ",
                            "incoming"));
                    }
                    if (types_1.Amounts.isNonZero(entry.pendingPayment)) {
                        payment = (React.createElement(i18n.Translate, { wrap: "span" },
                            React.createElement("span", { style: { color: "darkblue" } }, renderHtml_1.prettyAmount(entry.pendingPayment)),
                            " ",
                            "being spent"));
                    }
                    let l = [incoming, payment].filter((x) => x !== undefined);
                    if (l.length == 0) {
                        return React.createElement("span", null);
                    }
                    if (l.length == 1) {
                        return React.createElement("span", null,
                            "(",
                            l,
                            ")");
                    }
                    return React.createElement("span", null,
                        "(",
                        l[0],
                        ", ",
                        l[1],
                        ")");
                }
                render() {
                    let wallet = this.balance;
                    if (this.gotError) {
                        return i18n `Error: could not retrieve balance information.`;
                    }
                    if (!wallet) {
                        return React.createElement("span", null);
                    }
                    console.log(wallet);
                    let listing = Object.keys(wallet).map((key) => {
                        let entry = wallet[key];
                        return (React.createElement("p", null,
                            renderHtml_1.prettyAmount(entry.available),
                            " ",
                            this.formatPending(entry)));
                    });
                    if (listing.length > 0) {
                        let link = chrome.extension.getURL("/src/pages/tree.html");
                        let linkElem = React.createElement("a", { href: link, target: "_blank" }, "advanced view");
                        return (React.createElement("div", null,
                            listing,
                            linkElem));
                    }
                    return this.renderEmpty();
                }
            };
            WalletHistory = class WalletHistory extends React.Component {
                constructor() {
                    super(...arguments);
                    this.gotError = false;
                    this.unmounted = false;
                }
                componentWillMount() {
                    this.update();
                    onUpdateNotification(() => this.update());
                }
                componentWillUnmount() {
                    console.log("history component unmounted");
                    this.unmounted = true;
                }
                update() {
                    chrome.runtime.sendMessage({ type: "get-history" }, (resp) => {
                        if (this.unmounted) {
                            return;
                        }
                        console.log("got history response");
                        if (resp.error) {
                            this.gotError = true;
                            console.error("could not retrieve history", resp);
                            this.setState({});
                            return;
                        }
                        this.gotError = false;
                        console.log("got history", resp.history);
                        this.myHistory = resp.history;
                        this.setState({});
                    });
                }
                render() {
                    console.log("rendering history");
                    let history = this.myHistory;
                    if (this.gotError) {
                        return i18n `Error: could not retrieve event history`;
                    }
                    if (!history) {
                        // We're not ready yet
                        return React.createElement("span", null);
                    }
                    let subjectMemo = {};
                    let listing = [];
                    for (let record of history.reverse()) {
                        if (record.subjectId && subjectMemo[record.subjectId]) {
                            continue;
                        }
                        if (record.level != undefined && record.level < wallet_1.HistoryLevel.User) {
                            continue;
                        }
                        subjectMemo[record.subjectId] = true;
                        let item = (React.createElement("div", { className: "historyItem" },
                            React.createElement("div", { className: "historyDate" }, (new Date(record.timestamp)).toString()),
                            formatHistoryItem(record)));
                        listing.push(item);
                    }
                    if (listing.length > 0) {
                        return React.createElement("div", { className: "container" }, listing);
                    }
                    return React.createElement("p", null, i18n `Your wallet has no events recorded.`);
                }
            };
        }
    };
});
//# sourceMappingURL=popup.js.map