/*
 This file is part of TALER
 (C) 2015 GNUnet e.V.

 TALER is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 TALER is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
System.register(["src/emscripten/taler-emscripten-lib"], function (exports_1, context_1) {
    "use strict";
    var __moduleName = context_1 && context_1.id;
    /**
     * Count the UTF-8 characters in a JavaScript string.
     */
    function countUtf8Bytes(str) {
        var s = str.length;
        // JavaScript strings are UTF-16 arrays
        for (let i = str.length - 1; i >= 0; i--) {
            var code = str.charCodeAt(i);
            if (code > 0x7f && code <= 0x7ff) {
                // We need an extra byte in utf-8 here
                s++;
            }
            else if (code > 0x7ff && code <= 0xffff) {
                // We need two extra bytes in utf-8 here
                s += 2;
            }
            // Skip over the other surrogate
            if (code >= 0xDC00 && code <= 0xDFFF) {
                i--;
            }
        }
        return s;
    }
    /**
     * Create a packed arena object from the base32 crockford encoding.
     */
    function fromCrock(s, ctor) {
        let x = new ctor();
        x.alloc();
        x.loadCrock(s);
        return x;
    }
    /**
     * Create a packed arena object from the base32 crockford encoding for objects
     * that have a special decoding function.
     */
    function fromCrockDecoded(s, ctor, decodeFn) {
        let obj = new ctor();
        let buf = ByteArray.fromCrock(s);
        obj.nativePtr = decodeFn(buf.nativePtr, buf.size());
        buf.destroy();
        return obj;
    }
    /**
     * Encode an object using a special encoding function.
     */
    function encode(obj, encodeFn, arena) {
        let ptr = emscAlloc.malloc(PTR_SIZE);
        let len = encodeFn(obj.nativePtr, ptr);
        let res = new ByteArray(len, undefined, arena);
        res.nativePtr = taler_emscripten_lib_1.default.getValue(ptr, '*');
        emsc.free(ptr);
        return res;
    }
    // XXX: This only works up to 54 bit numbers.
    function set64(p, n) {
        for (let i = 0; i < 8; ++i) {
            taler_emscripten_lib_1.default.setValue(p + (7 - i), n & 0xFF, "i8");
            n = Math.floor(n / 256);
        }
    }
    // XXX: This only works up to 54 bit numbers.
    function set32(p, n) {
        for (let i = 0; i < 4; ++i) {
            taler_emscripten_lib_1.default.setValue(p + (3 - i), n & 0xFF, "i8");
            n = Math.floor(n / 256);
        }
    }
    /**
     * Blind a value so it can be blindly signed.
     */
    function rsaBlind(hashCode, blindingKey, pkey, arena) {
        let buf_ptr_out = emscAlloc.malloc(PTR_SIZE);
        let buf_size_out = emscAlloc.malloc(PTR_SIZE);
        let res = emscAlloc.rsa_blind(hashCode.nativePtr, blindingKey.nativePtr, pkey.nativePtr, buf_ptr_out, buf_size_out);
        let buf_ptr = taler_emscripten_lib_1.default.getValue(buf_ptr_out, '*');
        let buf_size = taler_emscripten_lib_1.default.getValue(buf_size_out, '*');
        emsc.free(buf_ptr_out);
        emsc.free(buf_size_out);
        if (res != GNUNET_OK) {
            // malicious key
            return null;
        }
        return new ByteArray(buf_size, buf_ptr, arena);
    }
    exports_1("rsaBlind", rsaBlind);
    /**
     * Sign data using EdDSA.
     */
    function eddsaSign(purpose, priv, a) {
        let sig = new EddsaSignature(a);
        sig.alloc();
        let res = emsc.eddsa_sign(priv.nativePtr, purpose.nativePtr, sig.nativePtr);
        if (res < 1) {
            throw Error("EdDSA signing failed");
        }
        return sig;
    }
    exports_1("eddsaSign", eddsaSign);
    /**
     * Verify EdDSA-signed data.
     */
    function eddsaVerify(purposeNum, verify, sig, pub, a) {
        let r = emsc.eddsa_verify(purposeNum, verify.nativePtr, sig.nativePtr, pub.nativePtr);
        return r === GNUNET_OK;
    }
    exports_1("eddsaVerify", eddsaVerify);
    /**
     * Unblind a blindly signed value.
     */
    function rsaUnblind(sig, bk, pk, a) {
        let x = new RsaSignature(a);
        x.nativePtr = emscAlloc.rsa_unblind(sig.nativePtr, bk.nativePtr, pk.nativePtr);
        return x;
    }
    exports_1("rsaUnblind", rsaUnblind);
    /**
     * Diffie-Hellman operation between an ECDHE private key
     * and an EdDSA public key.
     */
    function ecdhEddsa(priv, pub) {
        let h = new HashCode();
        h.alloc();
        let res = emsc.ecdh_eddsa(priv.nativePtr, pub.nativePtr, h.nativePtr);
        if (res != GNUNET_OK) {
            throw Error("ecdh_eddsa failed");
        }
        return h;
    }
    exports_1("ecdhEddsa", ecdhEddsa);
    /**
     * Derive a fresh coin from the given seed.  Used during refreshing.
     */
    function setupFreshCoin(secretSeed, coinIndex) {
        let priv = new EddsaPrivateKey();
        priv.isWeak = true;
        let blindingKey = new RsaBlindingKeySecret();
        blindingKey.isWeak = true;
        let buf = new ByteArray(priv.size() + blindingKey.size());
        emsc.setup_fresh_coin(secretSeed.nativePtr, coinIndex, buf.nativePtr);
        priv.nativePtr = buf.nativePtr;
        blindingKey.nativePtr = buf.nativePtr + priv.size();
        return { priv, blindingKey };
    }
    exports_1("setupFreshCoin", setupFreshCoin);
    var taler_emscripten_lib_1, PTR_SIZE, GNUNET_OK, GNUNET_YES, GNUNET_NO, GNUNET_SYSERR, getEmsc, emsc, emscAlloc, SignaturePurpose, RandomQuality, HashContext, MallocArenaObject, SimpleArena, SyncArena, arenaStack, Amount, PackedArenaObject, AmountNbo, EddsaPrivateKey, EcdsaPrivateKey, EcdhePrivateKey, EddsaPublicKey, EcdsaPublicKey, EcdhePublicKey, RsaBlindingKeySecret, HashCode, ByteArray, EccSignaturePurpose, SignatureStruct, WithdrawRequestPS, RefreshMeltCoinAffirmationPS, AbsoluteTimeNbo, UInt64, UInt32, DepositRequestPS, DenominationKeyValidityPS, RsaPublicKey, EddsaSignature, RsaSignature;
    return {
        setters: [
            function (taler_emscripten_lib_1_1) {
                taler_emscripten_lib_1 = taler_emscripten_lib_1_1;
            }
        ],
        execute: function () {/*
             This file is part of TALER
             (C) 2015 GNUnet e.V.
            
             TALER is free software; you can redistribute it and/or modify it under the
             terms of the GNU General Public License as published by the Free Software
             Foundation; either version 3, or (at your option) any later version.
            
             TALER is distributed in the hope that it will be useful, but WITHOUT ANY
             WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
             A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
            
             You should have received a copy of the GNU General Public License along with
             TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
             */
            /**
             * Medium-level interface to emscripten-compiled modules used
             * by the wallet.
             *
             * The high-level interface (using WebWorkers) is exposed in src/cryptoApi.ts.
             *
             * @author Florian Dold
             */
            "use strict";
            // Size of a native pointer.
            PTR_SIZE = 4;
            GNUNET_OK = 1;
            GNUNET_YES = 1;
            GNUNET_NO = 0;
            GNUNET_SYSERR = -1;
            /**
             * Get an emscripten-compiled function.
             */
            getEmsc = (name, ret, argTypes) => {
                return (...args) => {
                    return taler_emscripten_lib_1.default.ccall(name, ret, argTypes, args);
                };
            };
            /**
             * Wrapped emscripten functions that do not allocate any memory.
             */
            emsc = {
                free: (ptr) => taler_emscripten_lib_1.default._free(ptr),
                get_value: getEmsc("TALER_WR_get_value", "number", ["number"]),
                get_fraction: getEmsc("TALER_WR_get_fraction", "number", ["number"]),
                get_currency: getEmsc("TALER_WR_get_currency", "string", ["number"]),
                amount_add: getEmsc("TALER_amount_add", "number", ["number", "number", "number"]),
                amount_subtract: getEmsc("TALER_amount_subtract", "number", ["number", "number", "number"]),
                amount_normalize: getEmsc("TALER_amount_normalize", "void", ["number"]),
                amount_get_zero: getEmsc("TALER_amount_get_zero", "number", ["string", "number"]),
                amount_cmp: getEmsc("TALER_amount_cmp", "number", ["number", "number"]),
                amount_hton: getEmsc("TALER_amount_hton", "void", ["number", "number"]),
                amount_ntoh: getEmsc("TALER_amount_ntoh", "void", ["number", "number"]),
                hash: getEmsc("GNUNET_CRYPTO_hash", "void", ["number", "number", "number"]),
                memmove: getEmsc("memmove", "number", ["number", "number", "number"]),
                rsa_public_key_free: getEmsc("GNUNET_CRYPTO_rsa_public_key_free", "void", ["number"]),
                rsa_signature_free: getEmsc("GNUNET_CRYPTO_rsa_signature_free", "void", ["number"]),
                string_to_data: getEmsc("GNUNET_STRINGS_string_to_data", "number", ["number", "number", "number", "number"]),
                eddsa_sign: getEmsc("GNUNET_CRYPTO_eddsa_sign", "number", ["number", "number", "number"]),
                eddsa_verify: getEmsc("GNUNET_CRYPTO_eddsa_verify", "number", ["number", "number", "number", "number"]),
                hash_create_random: getEmsc("GNUNET_CRYPTO_hash_create_random", "void", ["number", "number"]),
                rsa_blinding_key_destroy: getEmsc("GNUNET_CRYPTO_rsa_blinding_key_free", "void", ["number"]),
                random_block: getEmsc("GNUNET_CRYPTO_random_block", "void", ["number", "number", "number"]),
                hash_context_abort: getEmsc("GNUNET_CRYPTO_hash_context_abort", "void", ["number"]),
                hash_context_read: getEmsc("GNUNET_CRYPTO_hash_context_read", "void", ["number", "number", "number"]),
                hash_context_finish: getEmsc("GNUNET_CRYPTO_hash_context_finish", "void", ["number", "number"]),
                ecdh_eddsa: getEmsc("GNUNET_CRYPTO_ecdh_eddsa", "number", ["number", "number", "number"]),
                setup_fresh_coin: getEmsc("TALER_setup_fresh_coin", "void", ["number", "number", "number"]),
            };
            /**
             * Emscripten functions that allocate memory.
             */
            emscAlloc = {
                get_amount: getEmsc("TALER_WRALL_get_amount", "number", ["number", "number", "number", "string"]),
                eddsa_key_create: getEmsc("GNUNET_CRYPTO_eddsa_key_create", "number", []),
                ecdsa_key_create: getEmsc("GNUNET_CRYPTO_ecdsa_key_create", "number", []),
                ecdhe_key_create: getEmsc("GNUNET_CRYPTO_ecdhe_key_create", "number", []),
                eddsa_public_key_from_private: getEmsc("TALER_WRALL_eddsa_public_key_from_private", "number", ["number"]),
                ecdsa_public_key_from_private: getEmsc("TALER_WRALL_ecdsa_public_key_from_private", "number", ["number"]),
                ecdhe_public_key_from_private: getEmsc("TALER_WRALL_ecdhe_public_key_from_private", "number", ["number"]),
                data_to_string_alloc: getEmsc("GNUNET_STRINGS_data_to_string_alloc", "number", ["number", "number"]),
                purpose_create: getEmsc("TALER_WRALL_purpose_create", "number", ["number", "number", "number"]),
                rsa_blind: getEmsc("GNUNET_CRYPTO_rsa_blind", "number", ["number", "number", "number", "number", "number"]),
                rsa_blinding_key_create: getEmsc("GNUNET_CRYPTO_rsa_blinding_key_create", "number", ["number"]),
                rsa_blinding_key_encode: getEmsc("GNUNET_CRYPTO_rsa_blinding_key_encode", "number", ["number", "number"]),
                rsa_signature_encode: getEmsc("GNUNET_CRYPTO_rsa_signature_encode", "number", ["number", "number"]),
                rsa_blinding_key_decode: getEmsc("GNUNET_CRYPTO_rsa_blinding_key_decode", "number", ["number", "number"]),
                rsa_public_key_decode: getEmsc("GNUNET_CRYPTO_rsa_public_key_decode", "number", ["number", "number"]),
                rsa_signature_decode: getEmsc("GNUNET_CRYPTO_rsa_signature_decode", "number", ["number", "number"]),
                rsa_public_key_encode: getEmsc("GNUNET_CRYPTO_rsa_public_key_encode", "number", ["number", "number"]),
                rsa_unblind: getEmsc("GNUNET_CRYPTO_rsa_unblind", "number", ["number", "number", "number"]),
                hash_context_start: getEmsc("GNUNET_CRYPTO_hash_context_start", "number", []),
                malloc: (size) => taler_emscripten_lib_1.default._malloc(size),
            };
            /**
             * Constants for signatures purposes, define what the signatures vouches for.
             */
            (function (SignaturePurpose) {
                SignaturePurpose[SignaturePurpose["RESERVE_WITHDRAW"] = 1200] = "RESERVE_WITHDRAW";
                SignaturePurpose[SignaturePurpose["WALLET_COIN_DEPOSIT"] = 1201] = "WALLET_COIN_DEPOSIT";
                SignaturePurpose[SignaturePurpose["MASTER_DENOMINATION_KEY_VALIDITY"] = 1025] = "MASTER_DENOMINATION_KEY_VALIDITY";
                SignaturePurpose[SignaturePurpose["WALLET_COIN_MELT"] = 1202] = "WALLET_COIN_MELT";
                SignaturePurpose[SignaturePurpose["TEST"] = 4242] = "TEST";
            })(SignaturePurpose || (SignaturePurpose = {}));
            exports_1("SignaturePurpose", SignaturePurpose);
            /**
             * Desired quality levels for random numbers.
             */
            (function (RandomQuality) {
                RandomQuality[RandomQuality["WEAK"] = 0] = "WEAK";
                RandomQuality[RandomQuality["STRONG"] = 1] = "STRONG";
                RandomQuality[RandomQuality["NONCE"] = 2] = "NONCE";
            })(RandomQuality || (RandomQuality = {}));
            exports_1("RandomQuality", RandomQuality);
            /**
             * Context for cummulative hashing.
             */
            HashContext = class HashContext {
                constructor() {
                    this.hashContextPtr = emscAlloc.hash_context_start();
                }
                /**
                 * Add data to be hashed.
                 */
                read(obj) {
                    if (!this.hashContextPtr) {
                        throw Error("assertion failed");
                    }
                    emsc.hash_context_read(this.hashContextPtr, obj.nativePtr, obj.size());
                }
                /**
                 * Finish the hash computation.
                 */
                finish(h) {
                    if (!this.hashContextPtr) {
                        throw Error("assertion failed");
                    }
                    h.alloc();
                    emsc.hash_context_finish(this.hashContextPtr, h.nativePtr);
                }
                /**
                 * Abort hashing without computing the result.
                 */
                destroy() {
                    if (this.hashContextPtr) {
                        emsc.hash_context_abort(this.hashContextPtr);
                    }
                    this.hashContextPtr = undefined;
                }
            };
            exports_1("HashContext", HashContext);
            /**
             * Arena object that points to an allocaed block of memory.
             */
            MallocArenaObject = class MallocArenaObject {
                constructor(arena) {
                    this._nativePtr = undefined;
                    /**
                     * Is this a weak reference to the underlying memory?
                     */
                    this.isWeak = false;
                    if (!arena) {
                        if (arenaStack.length == 0) {
                            throw Error("No arena available");
                        }
                        arena = arenaStack[arenaStack.length - 1];
                    }
                    arena.put(this);
                }
                destroy() {
                    if (this._nativePtr && !this.isWeak) {
                        emsc.free(this.nativePtr);
                        this._nativePtr = undefined;
                    }
                }
                alloc(size) {
                    if (this._nativePtr !== undefined) {
                        throw Error("Double allocation");
                    }
                    this.nativePtr = emscAlloc.malloc(size);
                }
                set nativePtr(v) {
                    if (v === undefined) {
                        throw Error("Native pointer must be a number or null");
                    }
                    this._nativePtr = v;
                }
                get nativePtr() {
                    // We want to allow latent allocation
                    // of native wrappers, but we never want to
                    // pass 'undefined' to emscripten.
                    if (this._nativePtr === undefined) {
                        throw Error("Native pointer not initialized");
                    }
                    return this._nativePtr;
                }
            };
            /**
             * Arena that must be manually destroyed.
             */
            SimpleArena = class SimpleArena {
                constructor() {
                    this.heap = [];
                }
                put(obj) {
                    this.heap.push(obj);
                }
                destroy() {
                    for (let obj of this.heap) {
                        obj.destroy();
                    }
                    this.heap = [];
                }
            };
            /**
             * Arena that destroys all its objects once control has returned to the message
             * loop.
             */
            SyncArena = class SyncArena extends SimpleArena {
                constructor() {
                    super();
                }
                pub(obj) {
                    super.put(obj);
                    if (!this.isScheduled) {
                        this.schedule();
                    }
                    this.heap.push(obj);
                }
                schedule() {
                    this.isScheduled = true;
                    Promise.resolve().then(() => {
                        this.isScheduled = false;
                        this.destroy();
                    });
                }
            };
            arenaStack = [];
            arenaStack.push(new SyncArena());
            /**
             * Representation of monetary value in a given currency.
             */
            Amount = class Amount extends MallocArenaObject {
                constructor(args, arena) {
                    super(arena);
                    if (args) {
                        this.nativePtr = emscAlloc.get_amount(args.value, 0, args.fraction, args.currency);
                    }
                    else {
                        this.nativePtr = emscAlloc.get_amount(0, 0, 0, "");
                    }
                }
                static getZero(currency, a) {
                    let am = new Amount(undefined, a);
                    let r = emsc.amount_get_zero(currency, am.nativePtr);
                    if (r != GNUNET_OK) {
                        throw Error("invalid currency");
                    }
                    return am;
                }
                toNbo(a) {
                    let x = new AmountNbo(a);
                    x.alloc();
                    emsc.amount_hton(x.nativePtr, this.nativePtr);
                    return x;
                }
                fromNbo(nbo) {
                    emsc.amount_ntoh(this.nativePtr, nbo.nativePtr);
                }
                get value() {
                    return emsc.get_value(this.nativePtr);
                }
                get fraction() {
                    return emsc.get_fraction(this.nativePtr);
                }
                get currency() {
                    return emsc.get_currency(this.nativePtr);
                }
                toJson() {
                    return {
                        value: emsc.get_value(this.nativePtr),
                        fraction: emsc.get_fraction(this.nativePtr),
                        currency: emsc.get_currency(this.nativePtr)
                    };
                }
                /**
                 * Add an amount to this amount.
                 */
                add(a) {
                    let res = emsc.amount_add(this.nativePtr, a.nativePtr, this.nativePtr);
                    if (res < 1) {
                        // Overflow
                        return false;
                    }
                    return true;
                }
                /**
                 * Perform saturating subtraction on amounts.
                 */
                sub(a) {
                    // this = this - a
                    let res = emsc.amount_subtract(this.nativePtr, this.nativePtr, a.nativePtr);
                    if (res == 0) {
                        // Underflow
                        return false;
                    }
                    if (res > 0) {
                        return true;
                    }
                    throw Error("Incompatible currencies");
                }
                cmp(a) {
                    // If we don't check this, the c code aborts.
                    if (this.currency !== a.currency) {
                        throw Error(`incomparable currencies (${this.currency} and ${a.currency})`);
                    }
                    return emsc.amount_cmp(this.nativePtr, a.nativePtr);
                }
                normalize() {
                    emsc.amount_normalize(this.nativePtr);
                }
            };
            exports_1("Amount", Amount);
            /**
             * Managed reference to a contiguous block of memory in the Emscripten heap.
             * Can be converted from / to a serialized representation.
             * Should contain only data, not pointers.
             */
            PackedArenaObject = class PackedArenaObject extends MallocArenaObject {
                constructor(a) {
                    super(a);
                }
                randomize(qual = RandomQuality.STRONG) {
                    emsc.random_block(qual, this.nativePtr, this.size());
                }
                toCrock() {
                    var d = emscAlloc.data_to_string_alloc(this.nativePtr, this.size());
                    var s = taler_emscripten_lib_1.default.Pointer_stringify(d);
                    emsc.free(d);
                    return s;
                }
                toJson() {
                    // Per default, the json encoding of
                    // packed arena objects is just the crockford encoding.
                    // Subclasses typically want to override this.
                    return this.toCrock();
                }
                loadCrock(s) {
                    this.alloc();
                    // We need to get the javascript string
                    // to the emscripten heap first.
                    let buf = ByteArray.fromStringWithNull(s);
                    let res = emsc.string_to_data(buf.nativePtr, s.length, this.nativePtr, this.size());
                    buf.destroy();
                    if (res < 1) {
                        throw { error: "wrong encoding" };
                    }
                }
                alloc() {
                    // FIXME: should the client be allowed to call alloc multiple times?
                    if (!this._nativePtr) {
                        this.nativePtr = emscAlloc.malloc(this.size());
                    }
                }
                hash() {
                    var x = new HashCode();
                    x.alloc();
                    emsc.hash(this.nativePtr, this.size(), x.nativePtr);
                    return x;
                }
                hexdump() {
                    let bytes = [];
                    for (let i = 0; i < this.size(); i++) {
                        let b = taler_emscripten_lib_1.default.getValue(this.nativePtr + i, "i8");
                        b = (b + 256) % 256;
                        bytes.push("0".concat(b.toString(16)).slice(-2));
                    }
                    let lines = [];
                    for (let i = 0; i < bytes.length; i += 8) {
                        lines.push(bytes.slice(i, i + 8).join(","));
                    }
                    return lines.join("\n");
                }
            };
            /**
             * Amount, encoded for network transmission.
             */
            AmountNbo = class AmountNbo extends PackedArenaObject {
                size() {
                    return 24;
                }
                toJson() {
                    let a = new SimpleArena();
                    let am = new Amount(undefined, a);
                    am.fromNbo(this);
                    let json = am.toJson();
                    a.destroy();
                    return json;
                }
            };
            exports_1("AmountNbo", AmountNbo);
            /**
             * Private EdDSA key.
             */
            EddsaPrivateKey = class EddsaPrivateKey extends PackedArenaObject {
                static create(a) {
                    let obj = new EddsaPrivateKey(a);
                    obj.nativePtr = emscAlloc.eddsa_key_create();
                    return obj;
                }
                size() {
                    return 32;
                }
                getPublicKey(a) {
                    let obj = new EddsaPublicKey(a);
                    obj.nativePtr = emscAlloc.eddsa_public_key_from_private(this.nativePtr);
                    return obj;
                }
                static fromCrock(s) {
                    return fromCrock(s, this);
                }
            };
            exports_1("EddsaPrivateKey", EddsaPrivateKey);
            EcdsaPrivateKey = class EcdsaPrivateKey extends PackedArenaObject {
                static create(a) {
                    let obj = new EcdsaPrivateKey(a);
                    obj.nativePtr = emscAlloc.ecdsa_key_create();
                    return obj;
                }
                size() {
                    return 32;
                }
                getPublicKey(a) {
                    let obj = new EcdsaPublicKey(a);
                    obj.nativePtr = emscAlloc.ecdsa_public_key_from_private(this.nativePtr);
                    return obj;
                }
                static fromCrock(s) {
                    return fromCrock(s, this);
                }
            };
            exports_1("EcdsaPrivateKey", EcdsaPrivateKey);
            EcdhePrivateKey = class EcdhePrivateKey extends PackedArenaObject {
                static create(a) {
                    let obj = new EcdhePrivateKey(a);
                    obj.nativePtr = emscAlloc.ecdhe_key_create();
                    return obj;
                }
                size() {
                    return 32;
                }
                getPublicKey(a) {
                    let obj = new EcdhePublicKey(a);
                    obj.nativePtr = emscAlloc.ecdhe_public_key_from_private(this.nativePtr);
                    return obj;
                }
                static fromCrock(s) {
                    return fromCrock(s, this);
                }
            };
            exports_1("EcdhePrivateKey", EcdhePrivateKey);
            EddsaPublicKey = class EddsaPublicKey extends PackedArenaObject {
                size() {
                    return 32;
                }
                static fromCrock(s) {
                    return fromCrock(s, this);
                }
            };
            exports_1("EddsaPublicKey", EddsaPublicKey);
            EcdsaPublicKey = class EcdsaPublicKey extends PackedArenaObject {
                size() {
                    return 32;
                }
                static fromCrock(s) {
                    return fromCrock(s, this);
                }
            };
            exports_1("EcdsaPublicKey", EcdsaPublicKey);
            EcdhePublicKey = class EcdhePublicKey extends PackedArenaObject {
                size() {
                    return 32;
                }
                static fromCrock(s) {
                    return fromCrock(s, this);
                }
            };
            exports_1("EcdhePublicKey", EcdhePublicKey);
            RsaBlindingKeySecret = class RsaBlindingKeySecret extends PackedArenaObject {
                size() {
                    return 32;
                }
                /**
                 * Create a random blinding key secret.
                 */
                static create(a) {
                    let o = new RsaBlindingKeySecret(a);
                    o.alloc();
                    o.randomize();
                    return o;
                }
                static fromCrock(s) {
                    return fromCrock(s, this);
                }
            };
            exports_1("RsaBlindingKeySecret", RsaBlindingKeySecret);
            HashCode = class HashCode extends PackedArenaObject {
                size() {
                    return 64;
                }
                static fromCrock(s) {
                    return fromCrock(s, this);
                }
                random(qual = RandomQuality.STRONG) {
                    this.alloc();
                    emsc.hash_create_random(qual, this.nativePtr);
                }
            };
            exports_1("HashCode", HashCode);
            ByteArray = class ByteArray extends PackedArenaObject {
                size() {
                    return this.allocatedSize;
                }
                constructor(desiredSize, init, a) {
                    super(a);
                    if (init === undefined) {
                        this.nativePtr = emscAlloc.malloc(desiredSize);
                    }
                    else {
                        this.nativePtr = init;
                    }
                    this.allocatedSize = desiredSize;
                }
                static fromStringWithoutNull(s, a) {
                    // UTF-8 bytes, including 0-terminator
                    let terminatedByteLength = countUtf8Bytes(s) + 1;
                    let hstr = emscAlloc.malloc(terminatedByteLength);
                    taler_emscripten_lib_1.default.stringToUTF8(s, hstr, terminatedByteLength);
                    return new ByteArray(terminatedByteLength - 1, hstr, a);
                }
                static fromStringWithNull(s, a) {
                    // UTF-8 bytes, including 0-terminator
                    let terminatedByteLength = countUtf8Bytes(s) + 1;
                    let hstr = emscAlloc.malloc(terminatedByteLength);
                    taler_emscripten_lib_1.default.stringToUTF8(s, hstr, terminatedByteLength);
                    return new ByteArray(terminatedByteLength, hstr, a);
                }
                static fromCrock(s, a) {
                    // this one is a bit more complicated than the other fromCrock functions,
                    // since we don't have a fixed size
                    let byteLength = countUtf8Bytes(s);
                    let hstr = emscAlloc.malloc(byteLength + 1);
                    taler_emscripten_lib_1.default.stringToUTF8(s, hstr, byteLength + 1);
                    let decodedLen = Math.floor((byteLength * 5) / 8);
                    let ba = new ByteArray(decodedLen, undefined, a);
                    let res = emsc.string_to_data(hstr, byteLength, ba.nativePtr, decodedLen);
                    emsc.free(hstr);
                    if (res != GNUNET_OK) {
                        throw Error("decoding failed");
                    }
                    return ba;
                }
            };
            exports_1("ByteArray", ByteArray);
            /**
             * Data to sign, together with a header that includes a purpose id
             * and size.
             */
            EccSignaturePurpose = class EccSignaturePurpose extends PackedArenaObject {
                size() {
                    return this.payloadSize + 8;
                }
                constructor(purpose, payload, a) {
                    super(a);
                    this.nativePtr = emscAlloc.purpose_create(purpose, payload.nativePtr, payload.size());
                    this.payloadSize = payload.size();
                }
            };
            exports_1("EccSignaturePurpose", EccSignaturePurpose);
            SignatureStruct = class SignatureStruct {
                constructor(x) {
                    this.members = {};
                    for (let k in x) {
                        this.set(k, x[k]);
                    }
                }
                toPurpose(a) {
                    let totalSize = 0;
                    for (let f of this.fieldTypes()) {
                        let name = f[0];
                        let member = this.members[name];
                        if (!member) {
                            throw Error(`Member ${name} not set`);
                        }
                        totalSize += member.size();
                    }
                    let buf = emscAlloc.malloc(totalSize);
                    let ptr = buf;
                    for (let f of this.fieldTypes()) {
                        let name = f[0];
                        let member = this.members[name];
                        let size = member.size();
                        emsc.memmove(ptr, member.nativePtr, size);
                        ptr += size;
                    }
                    let ba = new ByteArray(totalSize, buf, a);
                    return new EccSignaturePurpose(this.purpose(), ba);
                }
                toJson() {
                    let res = {};
                    for (let f of this.fieldTypes()) {
                        let name = f[0];
                        let member = this.members[name];
                        if (!member) {
                            throw Error(`Member ${name} not set`);
                        }
                        res[name] = member.toJson();
                    }
                    res["purpose"] = this.purpose();
                    return res;
                }
                set(name, value) {
                    let typemap = {};
                    for (let f of this.fieldTypes()) {
                        typemap[f[0]] = f[1];
                    }
                    if (!(name in typemap)) {
                        throw Error(`Key ${name} not found`);
                    }
                    if (!(value instanceof typemap[name])) {
                        throw Error("Wrong type for ${name}");
                    }
                    this.members[name] = value;
                }
            };
            WithdrawRequestPS = class WithdrawRequestPS extends SignatureStruct {
                constructor(w) {
                    super(w);
                }
                purpose() {
                    return SignaturePurpose.RESERVE_WITHDRAW;
                }
                fieldTypes() {
                    return [
                        ["reserve_pub", EddsaPublicKey],
                        ["amount_with_fee", AmountNbo],
                        ["withdraw_fee", AmountNbo],
                        ["h_denomination_pub", HashCode],
                        ["h_coin_envelope", HashCode]
                    ];
                }
            };
            exports_1("WithdrawRequestPS", WithdrawRequestPS);
            RefreshMeltCoinAffirmationPS = class RefreshMeltCoinAffirmationPS extends SignatureStruct {
                constructor(w) {
                    super(w);
                }
                purpose() {
                    return SignaturePurpose.WALLET_COIN_MELT;
                }
                fieldTypes() {
                    return [
                        ["session_hash", HashCode],
                        ["amount_with_fee", AmountNbo],
                        ["melt_fee", AmountNbo],
                        ["coin_pub", EddsaPublicKey]
                    ];
                }
            };
            exports_1("RefreshMeltCoinAffirmationPS", RefreshMeltCoinAffirmationPS);
            AbsoluteTimeNbo = class AbsoluteTimeNbo extends PackedArenaObject {
                static fromTalerString(s) {
                    let x = new AbsoluteTimeNbo();
                    x.alloc();
                    let r = /Date\(([0-9]+)\)/;
                    let m = r.exec(s);
                    if (!m || m.length != 2) {
                        throw Error();
                    }
                    let n = parseInt(m[1]) * 1000000;
                    // XXX: This only works up to 54 bit numbers.
                    set64(x.nativePtr, n);
                    return x;
                }
                size() {
                    return 8;
                }
            };
            exports_1("AbsoluteTimeNbo", AbsoluteTimeNbo);
            UInt64 = class UInt64 extends PackedArenaObject {
                static fromNumber(n) {
                    let x = new UInt64();
                    x.alloc();
                    set64(x.nativePtr, n);
                    return x;
                }
                size() {
                    return 8;
                }
            };
            exports_1("UInt64", UInt64);
            UInt32 = class UInt32 extends PackedArenaObject {
                static fromNumber(n) {
                    let x = new UInt32();
                    x.alloc();
                    set32(x.nativePtr, n);
                    return x;
                }
                size() {
                    return 4;
                }
            };
            exports_1("UInt32", UInt32);
            DepositRequestPS = class DepositRequestPS extends SignatureStruct {
                constructor(w) {
                    super(w);
                }
                purpose() {
                    return SignaturePurpose.WALLET_COIN_DEPOSIT;
                }
                fieldTypes() {
                    return [
                        ["h_contract", HashCode],
                        ["h_wire", HashCode],
                        ["timestamp", AbsoluteTimeNbo],
                        ["refund_deadline", AbsoluteTimeNbo],
                        ["transaction_id", UInt64],
                        ["amount_with_fee", AmountNbo],
                        ["deposit_fee", AmountNbo],
                        ["merchant", EddsaPublicKey],
                        ["coin_pub", EddsaPublicKey],
                    ];
                }
            };
            exports_1("DepositRequestPS", DepositRequestPS);
            DenominationKeyValidityPS = class DenominationKeyValidityPS extends SignatureStruct {
                constructor(w) {
                    super(w);
                }
                purpose() {
                    return SignaturePurpose.MASTER_DENOMINATION_KEY_VALIDITY;
                }
                fieldTypes() {
                    return [
                        ["master", EddsaPublicKey],
                        ["start", AbsoluteTimeNbo],
                        ["expire_withdraw", AbsoluteTimeNbo],
                        ["expire_spend", AbsoluteTimeNbo],
                        ["expire_legal", AbsoluteTimeNbo],
                        ["value", AmountNbo],
                        ["fee_withdraw", AmountNbo],
                        ["fee_deposit", AmountNbo],
                        ["fee_refresh", AmountNbo],
                        ["fee_refund", AmountNbo],
                        ["denom_hash", HashCode]
                    ];
                }
            };
            exports_1("DenominationKeyValidityPS", DenominationKeyValidityPS);
            RsaPublicKey = class RsaPublicKey extends MallocArenaObject {
                static fromCrock(s) {
                    return fromCrockDecoded(s, this, emscAlloc.rsa_public_key_decode);
                }
                toCrock() {
                    return this.encode().toCrock();
                }
                destroy() {
                    emsc.rsa_public_key_free(this.nativePtr);
                    this.nativePtr = 0;
                }
                encode(arena) {
                    return encode(this, emscAlloc.rsa_public_key_encode);
                }
            };
            exports_1("RsaPublicKey", RsaPublicKey);
            EddsaSignature = class EddsaSignature extends PackedArenaObject {
                size() {
                    return 64;
                }
            };
            exports_1("EddsaSignature", EddsaSignature);
            RsaSignature = class RsaSignature extends MallocArenaObject {
                static fromCrock(s, a) {
                    return fromCrockDecoded(s, this, emscAlloc.rsa_signature_decode);
                }
                encode(arena) {
                    return encode(this, emscAlloc.rsa_signature_encode);
                }
                destroy() {
                    emsc.rsa_signature_free(this.nativePtr);
                    this.nativePtr = 0;
                }
            };
            exports_1("RsaSignature", RsaSignature);
        }
    };
});
//# sourceMappingURL=emscriptif.js.map