/*
 This file is part of TALER
 (C) 2016 GNUnet e.V.

 TALER is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 TALER is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
System.register([], function (exports_1, context_1) {
    "use strict";
    var __moduleName = context_1 && context_1.id;
    var Checkable;
    return {
        setters: [],
        execute: function () {
            /**
             * Decorators for type-checking JSON into
             * an object.
             * @module Checkable
             * @author Florian Dold
             */
            (function (Checkable) {
                Checkable.SchemaError = (function SchemaError(message) {
                    this.name = 'SchemaError';
                    this.message = message;
                    this.stack = new Error().stack;
                });
                Checkable.SchemaError.prototype = new Error;
                let chkSym = Symbol("checkable");
                function checkNumber(target, prop, path) {
                    if ((typeof target) !== "number") {
                        throw new Checkable.SchemaError(`expected number for ${path}`);
                    }
                    return target;
                }
                function checkString(target, prop, path) {
                    if (typeof target !== "string") {
                        throw new Checkable.SchemaError(`expected string for ${path}, got ${typeof target} instead`);
                    }
                    return target;
                }
                function checkBoolean(target, prop, path) {
                    if (typeof target !== "boolean") {
                        throw new Checkable.SchemaError(`expected boolean for ${path}, got ${typeof target} instead`);
                    }
                    return target;
                }
                function checkAnyObject(target, prop, path) {
                    if (typeof target !== "object") {
                        throw new Checkable.SchemaError(`expected (any) object for ${path}, got ${typeof target} instead`);
                    }
                    return target;
                }
                function checkAny(target, prop, path) {
                    return target;
                }
                function checkList(target, prop, path) {
                    if (!Array.isArray(target)) {
                        throw new Checkable.SchemaError(`array expected for ${path}, got ${typeof target} instead`);
                    }
                    for (let i = 0; i < target.length; i++) {
                        let v = target[i];
                        prop.elementChecker(v, prop.elementProp, path.concat([i]));
                    }
                    return target;
                }
                function checkOptional(target, prop, path) {
                    console.assert(prop.propertyKey);
                    prop.elementChecker(target, prop.elementProp, path.concat([prop.propertyKey]));
                    return target;
                }
                function checkValue(target, prop, path) {
                    let type = prop.type;
                    if (!type) {
                        throw Error(`assertion failed (prop is ${JSON.stringify(prop)})`);
                    }
                    let v = target;
                    if (!v || typeof v !== "object") {
                        throw new Checkable.SchemaError(`expected object for ${path.join(".")}, got ${typeof v} instead`);
                    }
                    let props = type.prototype[chkSym].props;
                    let remainingPropNames = new Set(Object.getOwnPropertyNames(v));
                    let obj = new type();
                    for (let prop of props) {
                        if (!remainingPropNames.has(prop.propertyKey)) {
                            if (prop.optional) {
                                continue;
                            }
                            throw new Checkable.SchemaError("Property missing: " + prop.propertyKey);
                        }
                        if (!remainingPropNames.delete(prop.propertyKey)) {
                            throw new Checkable.SchemaError("assertion failed");
                        }
                        let propVal = v[prop.propertyKey];
                        obj[prop.propertyKey] = prop.checker(propVal, prop, path.concat([prop.propertyKey]));
                    }
                    if (remainingPropNames.size != 0) {
                        throw new Checkable.SchemaError("superfluous properties " + JSON.stringify(Array.from(remainingPropNames.values())));
                    }
                    return obj;
                }
                function Class(target) {
                    target.checked = (v) => {
                        return checkValue(v, {
                            propertyKey: "(root)",
                            type: target,
                            checker: checkValue
                        }, ["(root)"]);
                    };
                    return target;
                }
                Checkable.Class = Class;
                function ClassWithValidator(target) {
                    target.checked = (v) => {
                        let cv = checkValue(v, {
                            propertyKey: "(root)",
                            type: target,
                            checker: checkValue
                        }, ["(root)"]);
                        let instance = new target();
                        if (typeof instance.validate !== "function") {
                            throw Error("invalid Checkable annotion: validate method required");
                        }
                        // May throw exception
                        instance.validate.call(cv);
                        return cv;
                    };
                    return target;
                }
                Checkable.ClassWithValidator = ClassWithValidator;
                function Value(type) {
                    if (!type) {
                        throw Error("Type does not exist yet (wrong order of definitions?)");
                    }
                    function deco(target, propertyKey) {
                        let chk = mkChk(target);
                        chk.props.push({
                            propertyKey: propertyKey,
                            checker: checkValue,
                            type: type
                        });
                    }
                    return deco;
                }
                Checkable.Value = Value;
                function List(type) {
                    let stub = {};
                    type(stub, "(list-element)");
                    let elementProp = mkChk(stub).props[0];
                    let elementChecker = elementProp.checker;
                    if (!elementChecker) {
                        throw Error("assertion failed");
                    }
                    function deco(target, propertyKey) {
                        let chk = mkChk(target);
                        chk.props.push({
                            elementChecker,
                            elementProp,
                            propertyKey: propertyKey,
                            checker: checkList,
                        });
                    }
                    return deco;
                }
                Checkable.List = List;
                function Optional(type) {
                    let stub = {};
                    type(stub, "(optional-element)");
                    let elementProp = mkChk(stub).props[0];
                    let elementChecker = elementProp.checker;
                    if (!elementChecker) {
                        throw Error("assertion failed");
                    }
                    function deco(target, propertyKey) {
                        let chk = mkChk(target);
                        chk.props.push({
                            elementChecker,
                            elementProp,
                            propertyKey: propertyKey,
                            checker: checkOptional,
                            optional: true,
                        });
                    }
                    return deco;
                }
                Checkable.Optional = Optional;
                function Number(target, propertyKey) {
                    let chk = mkChk(target);
                    chk.props.push({ propertyKey: propertyKey, checker: checkNumber });
                }
                Checkable.Number = Number;
                function AnyObject(target, propertyKey) {
                    let chk = mkChk(target);
                    chk.props.push({
                        propertyKey: propertyKey,
                        checker: checkAnyObject
                    });
                }
                Checkable.AnyObject = AnyObject;
                function Any(target, propertyKey) {
                    let chk = mkChk(target);
                    chk.props.push({
                        propertyKey: propertyKey,
                        checker: checkAny,
                        optional: true
                    });
                }
                Checkable.Any = Any;
                function String(target, propertyKey) {
                    let chk = mkChk(target);
                    chk.props.push({ propertyKey: propertyKey, checker: checkString });
                }
                Checkable.String = String;
                function Boolean(target, propertyKey) {
                    let chk = mkChk(target);
                    chk.props.push({ propertyKey: propertyKey, checker: checkBoolean });
                }
                Checkable.Boolean = Boolean;
                function mkChk(target) {
                    let chk = target[chkSym];
                    if (!chk) {
                        chk = { props: [] };
                        target[chkSym] = chk;
                    }
                    return chk;
                }
            })(Checkable || (Checkable = {}));
            exports_1("Checkable", Checkable);
        }
    };
});
//# sourceMappingURL=checkable.js.map