/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import { Duration, Logger, NotificationType, TalerCorebankApiClient, j2s } from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { makeNoFeeCoinConfig } from "../harness/denomStructures.js";
import {
    BankService,
  ExchangeService,
  FakebankService,
  GlobalTestState,
  MerchantService,
  generateRandomPayto,
  setupDb,
} from "../harness/harness.js";
import {
  applyTimeTravelV2,
  createWalletDaemonWithClient,
  withdrawViaBankV3,
} from "../harness/helpers.js";

const logger = new Logger("test-exchange-timetravel.ts");

/**
 * Test how the wallet handles an expired denomination.
 */
export async function runWalletDenomExpireTest(t: GlobalTestState) {
  // Set up test environment

  const db = await setupDb(t);

  const bank = await BankService.create(t, {
    allowRegistrations: true,
    currency: "TESTKUDOS",
    database: db.connStr,
    httpPort: 8082,
  });

  const exchange = ExchangeService.create(t, {
    name: "testexchange-1",
    currency: "TESTKUDOS",
    httpPort: 8081,
    database: db.connStr,
  });

  const merchant = await MerchantService.create(t, {
    name: "testmerchant-1",
    currency: "TESTKUDOS",
    httpPort: 8083,
    database: db.connStr,
  });

  let receiverName = "Exchange";
  let exchangeBankUsername = "exchange";
  let exchangeBankPassword = "mypw";
  let exchangePaytoUri = generateRandomPayto(exchangeBankUsername);

  await exchange.addBankAccount("1", {
    accountName: exchangeBankUsername,
    accountPassword: exchangeBankPassword,
    wireGatewayApiBaseUrl: new URL("accounts/exchange/taler-wire-gateway/", bank.baseUrl).href,
    accountPaytoUri: exchangePaytoUri,
  });

  bank.setSuggestedExchange(exchange, exchangePaytoUri);

  await bank.start();

  await bank.pingUntilAvailable();

  const bankClient = new TalerCorebankApiClient(bank.corebankApiBaseUrl, {
    auth: {
      username: "admin",
      password: "adminpw",
    },
  });

  await bankClient.registerAccountExtended({
    name: receiverName,
    password: exchangeBankPassword,
    username: exchangeBankUsername,
    is_taler_exchange: true,
    payto_uri: exchangePaytoUri,
  });

  exchange.addCoinConfigList(makeNoFeeCoinConfig("TESTKUDOS"));

  await exchange.start();
  await exchange.pingUntilAvailable();

  merchant.addExchange(exchange);

  await merchant.start();
  await merchant.pingUntilAvailable();

  console.log("merchant started, configuring instances");

  await merchant.addInstanceWithWireAccount({
    id: "default",
    name: "Default Instance",
    paytoUris: [generateRandomPayto("merchant-default")],
  });

  await merchant.addInstanceWithWireAccount({
    id: "minst1",
    name: "minst1",
    paytoUris: [generateRandomPayto("minst1")],
  });

  console.log("setup done!");

  const { walletClient } = await createWalletDaemonWithClient(t, {
    name: "default",
  });

  // Withdraw digital cash into the wallet.

  const wres = await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange,
    amount: "TESTKUDOS:15",
  });
  await wres.withdrawalFinishedCond;

  const denomLossCond = walletClient.waitForNotificationCond((n) => {
    return (
      n.type === NotificationType.TransactionStateTransition &&
      n.transactionId.startsWith("txn:denom-loss:")
    );
  });

  // Travel into the future, the deposit expiration is two years
  // into the future.
  console.log("applying first time travel");
  await applyTimeTravelV2(
    Duration.toMilliseconds(Duration.fromSpec({ days: 800 })),
    {
      walletClient,
      exchange,
      merchant,
    },
  );

  t.logStep("before-wait-denom-loss");

  // Should be detected automatically, as exchange entry is surely outdated.
  await denomLossCond;

  const bal = await walletClient.call(WalletApiOperation.GetBalances, {});
  console.log(`balances: ${j2s(bal)}`);

  const txns = await walletClient.call(WalletApiOperation.GetTransactions, {
    sort: "stable-ascending",
    includeRefreshes: true,
  });
  console.log(`transactions: ${j2s(txns)}`);
}

runWalletDenomExpireTest.suites = ["exchange"];
