/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  ConfirmPayResultType,
  Duration,
  MerchantApiClient,
  NotificationType,
  PreparePayResultType,
  TalerCorebankApiClient,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { makeNoFeeCoinConfig } from "../harness/denomStructures.js";
import {
  BankService,
  ExchangeService,
  GlobalTestState,
  MerchantService,
  generateRandomPayto,
  setupDb,
} from "../harness/harness.js";
import {
  applyTimeTravelV2,
  createWalletDaemonWithClient,
  withdrawViaBankV3,
} from "../harness/helpers.js";

/**
 * Basic time travel test.
 */
export async function runTimetravelAutorefreshTest(t: GlobalTestState) {
  // Set up test environment

  const db = await setupDb(t);

  const bank = await BankService.create(t, {
    allowRegistrations: true,
    currency: "TESTKUDOS",
    database: db.connStr,
    httpPort: 8082,
  });

  const exchange = ExchangeService.create(t, {
    name: "testexchange-1",
    currency: "TESTKUDOS",
    httpPort: 8081,
    database: db.connStr,
  });

  const merchant = await MerchantService.create(t, {
    name: "testmerchant-1",
    currency: "TESTKUDOS",
    httpPort: 8083,
    database: db.connStr,
  });

  let receiverName = "Exchange";
  let exchangeBankUsername = "exchange";
  let exchangeBankPassword = "mypw";
  let exchangePaytoUri = generateRandomPayto(exchangeBankUsername);

  await exchange.addBankAccount("1", {
    accountName: exchangeBankUsername,
    accountPassword: exchangeBankPassword,
    wireGatewayApiBaseUrl: new URL(
      "accounts/exchange/taler-wire-gateway/",
      bank.baseUrl,
    ).href,
    accountPaytoUri: exchangePaytoUri,
  });

  bank.setSuggestedExchange(exchange, exchangePaytoUri);

  await bank.start();

  await bank.pingUntilAvailable();

  const bankClient = new TalerCorebankApiClient(bank.corebankApiBaseUrl, {
    auth: {
      username: "admin",
      password: "adminpw",
    },
  });

  await bankClient.registerAccountExtended({
    name: receiverName,
    password: exchangeBankPassword,
    username: exchangeBankUsername,
    is_taler_exchange: true,
    payto_uri: exchangePaytoUri,
  });

  exchange.addCoinConfigList(makeNoFeeCoinConfig("TESTKUDOS"));

  await exchange.start();
  await exchange.pingUntilAvailable();

  merchant.addExchange(exchange);

  await merchant.start();
  await merchant.pingUntilAvailable();

  await merchant.addInstanceWithWireAccount({
    id: "default",
    name: "Default Instance",
    paytoUris: [generateRandomPayto("merchant-default")],
  });

  await merchant.addInstanceWithWireAccount({
    id: "minst1",
    name: "minst1",
    paytoUris: [generateRandomPayto("minst1")],
  });

  console.log("setup done!");

  const { walletClient } = await createWalletDaemonWithClient(t, {
    name: "w1",
    persistent: true,
  });

  const merchantClient = new MerchantApiClient(merchant.makeInstanceBaseUrl());

  t.logStep("exchangeUpdated1Cond");

  // Withdraw digital cash into the wallet.
  t.logStep("Withdraw digital cash into the wallet.");
  const wres = await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange,
    amount: "TESTKUDOS:15",
  });
  t.logStep("wait");
  await wres.withdrawalFinishedCond;

  const exchangeUpdated1Cond = walletClient.waitForNotificationCond(
    (x) =>
      x.type === NotificationType.ExchangeStateTransition &&
      x.exchangeBaseUrl === exchange.baseUrl,
  );
  t.logStep("waiting tx");
  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});
  {
    const balance = await walletClient.call(WalletApiOperation.GetBalances, {});
    t.assertAmountEquals(balance.balances[0].available, "TESTKUDOS:15");
  }

  // Travel into the future, the deposit expiration is two years
  // into the future.
  t.logStep("applying first time travel");
  await applyTimeTravelV2(
    Duration.toMilliseconds(Duration.fromSpec({ days: 400 })),
    {
      walletClient,
      exchange,
      merchant,
    },
  );

  // The time travel should cause exchanges to update.
  t.logStep("The time travel should cause exchanges to update");
  await exchangeUpdated1Cond;
  t.logStep("exchange updated, waiting for tx");
  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});
  {
    const balance = await walletClient.call(WalletApiOperation.GetBalances, {});
    t.assertAmountEquals(balance.balances[0].available, "TESTKUDOS:15");
  }

  t.logStep("withdrawing second time");
  const wres2 = await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange,
    amount: "TESTKUDOS:20",
  });
  await wres2.withdrawalFinishedCond;

  t.logStep("witdrawn, waiting tx");
  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  {
    const balance = await walletClient.call(WalletApiOperation.GetBalances, {});
    t.assertAmountEquals(balance.balances[0].available, "TESTKUDOS:35");
  }

  const exchangeUpdated2Cond = walletClient.waitForNotificationCond(
    (x) =>
      x.type === NotificationType.ExchangeStateTransition &&
      x.exchangeBaseUrl === exchange.baseUrl,
  );

  // Travel into the future, the deposit expiration is two years
  // into the future.
  t.logStep("applying second time travel");
  await applyTimeTravelV2(
    Duration.toMilliseconds(Duration.fromSpec({ years: 2, months: 6 })),
    {
      walletClient,
      exchange,
      merchant,
    },
  );

  // The time travel should cause exchanges to update.
  t.logStep("The time travel should cause exchanges to update.");
  await exchangeUpdated2Cond;
  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});
  {
    const balance = await walletClient.call(WalletApiOperation.GetBalances, {});
    t.assertAmountEquals(balance.balances[0].available, "TESTKUDOS:35");
  }

  // At this point, the original coins should've been refreshed.
  // It would be too late to refresh them now, as we're past
  // the two year deposit expiration.

  const orderResp = await merchantClient.createOrder({
    order: {
      fulfillment_url: "http://example.com",
      summary: "foo",
      amount: "TESTKUDOS:30",
    },
  });

  const orderStatus = await merchantClient.queryPrivateOrderStatus({
    orderId: orderResp.order_id,
    instance: "default",
  });

  t.assertTrue(orderStatus.order_status === "unpaid");

  const r = await walletClient.call(WalletApiOperation.PreparePayForUri, {
    talerPayUri: orderStatus.taler_pay_uri,
  });

  console.log(r);

  t.assertTrue(r.status === PreparePayResultType.PaymentPossible);

  const cpr = await walletClient.call(WalletApiOperation.ConfirmPay, {
    transactionId: r.transactionId,
  });

  t.assertTrue(cpr.type === ConfirmPayResultType.Done);
}

runTimetravelAutorefreshTest.suites = ["wallet"];
