/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  Duration,
  MerchantApiClient,
  TransactionMajorState,
  TransactionMinorState,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { GlobalTestState } from "../harness/harness.js";
import {
  createSimpleTestkudosEnvironmentV3,
  withdrawViaBankV3,
} from "../harness/helpers.js";

/**
 * Run test for basic, bank-integrated withdrawal.
 */
export async function runRefundAutoTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bankClient, exchange, merchant } =
    await createSimpleTestkudosEnvironmentV3(t);

  const merchantClient = new MerchantApiClient(merchant.makeInstanceBaseUrl());

  // Withdraw digital cash into the wallet.

  const wres = await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange,
    amount: "TESTKUDOS:20",
  });

  await wres.withdrawalFinishedCond;

  // Test case where the auto-refund happens
  {
    // Set up order.
    const orderResp = await merchantClient.createOrder({
      order: {
        summary: "Buy me!",
        amount: "TESTKUDOS:5",
        fulfillment_url: "taler://fulfillment-success/thx",
        auto_refund: {
          d_us: 3000 * 1000,
        },
      },
      refund_delay: Duration.toTalerProtocolDuration(
        Duration.fromSpec({ minutes: 5 }),
      ),
    });

    let orderStatus = await merchantClient.queryPrivateOrderStatus({
      orderId: orderResp.order_id,
    });

    t.assertTrue(orderStatus.order_status === "unpaid");

    // Make wallet pay for the order

    const r1 = await walletClient.call(WalletApiOperation.PreparePayForUri, {
      talerPayUri: orderStatus.taler_pay_uri,
    });

    await walletClient.call(WalletApiOperation.ConfirmPay, {
      transactionId: r1.transactionId,
    });

    // Check if payment was successful.

    orderStatus = await merchantClient.queryPrivateOrderStatus({
      orderId: orderResp.order_id,
    });

    t.assertTrue(orderStatus.order_status === "paid");

    const ref = await merchantClient.giveRefund({
      amount: "TESTKUDOS:5",
      instance: "default",
      justification: "foo",
      orderId: orderResp.order_id,
    });

    console.log(ref);

    // The wallet should now automatically pick up the refund.
    await walletClient.call(
      WalletApiOperation.TestingWaitTransactionsFinal,
      {},
    );

    const transactions = await walletClient.call(
      WalletApiOperation.GetTransactions,
      {
        sort: "stable-ascending",
      },
    );
    console.log(JSON.stringify(transactions, undefined, 2));

    const transactionTypes = transactions.transactions.map((x) => x.type);
    t.assertDeepEqual(transactionTypes, ["withdrawal", "payment", "refund"]);
  }

  // Now test the case where the auto-refund just expires

  {
    // Set up order.
    const orderResp = await merchantClient.createOrder({
      order: {
        summary: "Buy me!",
        amount: "TESTKUDOS:5",
        fulfillment_url: "taler://fulfillment-success/thx",
        auto_refund: {
          d_us: 3000 * 1000,
        },
      },
      refund_delay: Duration.toTalerProtocolDuration(
        Duration.fromSpec({ minutes: 5 }),
      ),
    });

    let orderStatus = await merchantClient.queryPrivateOrderStatus({
      orderId: orderResp.order_id,
    });

    t.assertTrue(orderStatus.order_status === "unpaid");

    // Make wallet pay for the order

    const r1 = await walletClient.call(WalletApiOperation.PreparePayForUri, {
      talerPayUri: orderStatus.taler_pay_uri,
    });

    await walletClient.call(WalletApiOperation.ConfirmPay, {
      transactionId: r1.transactionId,
    });

    // Check if payment was successful.

    orderStatus = await merchantClient.queryPrivateOrderStatus({
      orderId: orderResp.order_id,
    });

    t.assertTrue(orderStatus.order_status === "paid");

    await walletClient.call(WalletApiOperation.TestingWaitTransactionState, {
      transactionId: r1.transactionId,
      txState: {
        major: TransactionMajorState.Pending,
        minor: TransactionMinorState.AutoRefund,
      },
    });
    // Only time-travel the wallet
    await walletClient.call(WalletApiOperation.TestingSetTimetravel, {
      offsetMs: 5000,
    });
    await walletClient.call(WalletApiOperation.TestingWaitTransactionState, {
      transactionId: r1.transactionId,
      txState: {
        major: TransactionMajorState.Done,
      },
    });
  }
}

runRefundAutoTest.suites = ["wallet"];
