/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
import { HttpStatusCode, TalerError, assertUnreachable } from "@gnu-taler/taler-util";
import { Fragment, VNode, h } from "preact";
import { useState } from "preact/hooks";
import { ErrorLoadingMerchant } from "../../../components/ErrorLoadingMerchant.js";
import { Loading } from "../../../components/exception/loading.js";
import { DeleteModal } from "../../../components/modal/index.js";
import { useSessionContext } from "../../../context/session.js";
import { useInstanceDetails } from "../../../hooks/instance.js";
import { LoginPage } from "../../login/index.js";
import { NotFoundPageOrAdminCreate } from "../../notfound/index.js";
import { DetailPage } from "./DetailPage.js";

interface Props {
  onUpdate: () => void;
  onDelete: () => void;
}

export default function Detail({
  onUpdate,
  onDelete,
}: Props): VNode {
  const { state } = useSessionContext();
  const result = useInstanceDetails();
  const [deleting, setDeleting] = useState<boolean>(false);

  // const { deleteInstance } = useInstanceAPI();
  const { lib } = useSessionContext();

  if (!result) return <Loading />
  if (result instanceof TalerError) {
    return <ErrorLoadingMerchant error={result} />
  }
  if (result.type === "fail") {
    switch(result.case) {
      case HttpStatusCode.Unauthorized: {
        return <LoginPage />
      }
      case HttpStatusCode.NotFound: {
        return <NotFoundPageOrAdminCreate />;
      }
      default: {
        assertUnreachable(result)
      }
    }
  }


  return (
    <Fragment>
      <DetailPage
        selected={result.body}
        onUpdate={onUpdate}
        onDelete={() => setDeleting(true)}
      />
      {deleting && (
        <DeleteModal
          element={{ name: result.body.name, id: state.instance }}
          onCancel={() => setDeleting(false)}
          onConfirm={async (): Promise<void> => {
            if (state.status !== "loggedIn") {
              return
            }
            try {
              await lib.instance.deleteCurrentInstance(state.token);
              onDelete();
            } catch (error) {
              //FIXME: show message error
            }
            setDeleting(false);
          }}
        />
      )}
    </Fragment>
  );
}
